/* GNU Midnight Commander -- GNOME edition
 *
 * Directory display routines
 *
 * Copyright (C) 1997 The Free Software Foundation
 *
 * Author: Miguel de Icaza
 */
#include <config.h>
#include <string.h>
#include <stdlib.h>		/* atoi */
#include "fs.h"
#include "x.h"
#include "dir.h"
#include "panel.h"
#include "command.h"
#include "panel.h"		/* current_panel */
#include "command.h"		/* cmdline */
#include "main.h"
#include "color.h"
#include "mouse.h"
#include "layout.h"		/* get_panel_widget */
#include "ext.h"		/* regex_command */
#include "cmd.h"		/* copy_cmd, ren_cmd, delete_cmd, ... */
#include "gscreen.h"
#include "dir.h"

#include "directory.xpm"

/* This is used to initialize our pixmaps */
static int pixmaps_ready;
GdkPixmap *directory_pixmap;
GdkBitmap *directory_mask;

void
repaint_file (WPanel *panel, int file_index, int move, int attr, int isstatus)
{
}

void
show_dir (WPanel *panel)
{
	printf ("show_dir to %s %s\n", panel->cwd, panel->filter ? panel->filter : "<no-filt>");
}

static void
panel_file_list_set_type_bitmap (GtkCList *cl, int row, int column, int color, file_entry *fe)
{
	/* Here, add more icons */
	switch (color){
	case DIRECTORY_COLOR:
		gtk_clist_set_pixmap (cl, row, column, directory_pixmap, directory_mask);
		break;
	}
}

static void
panel_file_list_set_row_colors (GtkCList *cl, int row, int color_pair)
{
	gtk_clist_set_foreground (cl, row, &gmc_color_pairs [color_pair].fore);
	gtk_clist_set_background (cl, row, &gmc_color_pairs [color_pair].back);
}

void
x_fill_panel (WPanel *panel)
{
	g_panel_contents *g = (g_panel_contents *) panel->widget.wdata;
	const int top       = panel->count;
	const int items     = panel->format->items;
	GtkCList *cl        = GTK_CLIST (g->list);
	int i, col, type_col, color;
	char  **texts;

	texts = malloc (sizeof (char *) * items);

	gtk_clist_freeze (GTK_CLIST (cl));
	gtk_clist_clear (GTK_CLIST (cl));

	/* which column holds the type information */
	type_col = -1;
	
	for (i = 0; i < top; i++){
		file_entry *fe = &panel->dir.list [i];
		format_e *format = panel->format;
		
		for (col = 0; format; format = format->next){
			char *text;

			if (!format->use_in_gui)
			    continue;

			if (type_col == -1)
				if (strcmp (format->id, "type") == 0)
					type_col = col;
			
			if (!format->string_fn)
				texts [col] = "";
			else
				texts [col] = (*format->string_fn)(fe, 10);
			col++;
		}
		gtk_clist_append (cl, texts);
		
		color = file_entry_color (fe);
		panel_file_list_set_row_colors (cl, i, color);
		if (type_col != -1)
			panel_file_list_set_type_bitmap (cl, i, type_col, color, fe);
	}
	gtk_clist_thaw (GTK_CLIST (cl));
	free (texts);
}

void
gmc_panel_set_size (int index, int boot)
{
	Widget *w;
	WPanel *p;
	
	w = (Widget *) get_panel_widget (index);
	p = (WPanel *) w;
	w->cols = 40;
	w->lines = 25;
	set_panel_formats (p);
	paint_panel (p);

	if (!boot)
		paint_frame (p);

	x_fill_panel (p);
}

void
x_panel_set_size (int index)
{
	printf ("WARNING: set size called\n");
	gmc_panel_set_size (index, 1);
}

void
x_select_item (WPanel *panel)
{
	printf ("selecting %d\n", panel->selected);
}

void
x_panel_select_item (WPanel *panel, int index, int value)
{
	printf ("%s %d\n", value ? "select" : "unselect", index);
}

void
x_unselect_item (WPanel *panel)
{
	if (panel->active)
		printf ("unselecting %d\n", panel->selected);
}

void
x_filter_changed (WPanel *panel)
{
	show_dir (panel);
}

void
x_adjust_top_file (WPanel *panel)
{
	printf ("Adjusting top file to %d\n", panel->top_file);
}

#define COLUMN_INSET 3
#define CELL_SPACING 1

GtkWidget
panel_file_list_configure_contents (GtkWidget *file_list, WPanel *panel, int main_width, int height)
{
	format_e *format = panel->format;
	int i, used_columns, expandables, items;
	int char_width, usable_pixels, extra_pixels, width;
	int total_columns, extra_columns;
	int expand_space, extra_space, shrink_space;
	int lost_pixels;
	
	/* Pass 1: Count minimum columns,
	 * set field_len to default to the requested_field_len
	 * and compute how much space we lost to the column decorations
	 */
	lost_pixels = used_columns = expandables = 0;
	for (format = panel->format; format; format = format->next){
		format->field_len = format->requested_field_len;
		if (!format->use_in_gui)
			continue;
		
		used_columns += format->field_len;
		items++;
		if (format->expand)
			expandables++;
		lost_pixels += CELL_SPACING + (2 * COLUMN_INSET);
	}

	/* The left scrollbar might take some space from us, use this information */
	if (GTK_WIDGET_VISIBLE (GTK_CLIST (file_list)->vscrollbar)){
		GtkRequisition req;
		
		gtk_widget_size_request (GTK_CLIST (file_list)->vscrollbar, &req);
		lost_pixels += req.width;
	}
	
	char_width = gdk_string_width (file_list->style->font, "W");
	width = main_width - lost_pixels;
	extra_pixels  = width % char_width;
	usable_pixels = width - extra_pixels;
	total_columns = usable_pixels / char_width;
	extra_columns = total_columns - used_columns;
	if (extra_columns > 0){
		expand_space  = extra_columns / expandables;
		extra_space   = extra_columns % expandables;
	} else
		extra_space = expand_space = 0;

	/* If we dont have enough space, shorten the fields */
	if (used_columns > total_columns){
		expand_space = 0;
		shrink_space = (used_columns - total_columns) / items;
	} else
		shrink_space = 0;
	
	gtk_clist_freeze (GTK_CLIST (file_list));
	for (i = 0, format = panel->format; format; format = format->next){
		if (!format->use_in_gui)
			continue;

		format->field_len += (format->expand ? expand_space : 0) - shrink_space;
		gtk_clist_set_column_width (GTK_CLIST (file_list), i, format->field_len * char_width);
		i++;
	}
	gtk_clist_thaw (GTK_CLIST (file_list));
}

void
panel_file_list_size_allocate_hook (GtkWidget *file_list, GtkAllocation *allocation, WPanel *panel)
{
	panel_file_list_configure_contents (file_list, panel, allocation->width, allocation->height);
}

static void
panel_file_list_column_callback (GtkWidget *widget, int col, WPanel *panel)
{
	format_e *format;
	int i;
	
	for (i = 0, format = panel->format; format; format = format->next){
		if (!format->use_in_gui)
			continue;
		if (i == col){
			sortfn *sorting_routine;
			
			sorting_routine = get_sort_fn (format->id);
			if (!sorting_routine)
				return;

			if (sorting_routine == panel->sort_type)
				panel->reverse = !panel->reverse;
			panel->sort_type = sorting_routine;

			do_re_sort (panel);
			return;
		}
		i++;
	}
}

void
panel_create_pixmaps (GtkWidget *parent)
{
	GdkColor color = gtk_widget_get_style (parent)->bg [GTK_STATE_NORMAL];

	pixmaps_ready = 1;
	directory_pixmap = gdk_pixmap_create_from_xpm_d (parent->window, &directory_mask, &color, directory_xpm);
}

void
panel_configure_file_list (WPanel *panel, GtkWidget *file_list)
{
	format_e *format = panel->format;
	GtkCList *cl = GTK_CLIST (file_list);
	int i;

	/* Set sorting callback */
	gtk_signal_connect (GTK_OBJECT (file_list), "click_column",
			    GTK_SIGNAL_FUNC (panel_file_list_column_callback), panel);

	/* Configure the CList */
	gtk_clist_set_selection_mode (cl, GTK_SELECTION_SINGLE);
	gtk_clist_set_policy (cl, GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	
	for (i = 0, format = panel->format; format; format = format->next){
		GtkJustification just;

		if (!format->use_in_gui)
			continue;

		/* Set desired justification */
		if (format->just_mode == J_LEFT)
			just = GTK_JUSTIFY_LEFT;
		else
			just = GTK_JUSTIFY_RIGHT;
		
		gtk_clist_set_column_justification (cl, i, just);
		i++;
	}
}

GtkWidget *
panel_create_file_list (WPanel *panel)
{
	const int items = panel->format->items;
	format_e  *format = panel->format;
	GtkWidget *file_list;
	gchar     **titles;
	int       i;

	titles = g_new (char *, items);
	for (i = 0; i < items; format = format->next)
		if (format->use_in_gui)
			titles [i++] = format->title;

	file_list = gtk_clist_new (items, titles);
	panel_configure_file_list (panel, file_list);
	free (titles);
	gtk_signal_connect (GTK_OBJECT (file_list), "size_allocate",
			    (GtkSignalFunc) panel_file_list_size_allocate_hook,
			    panel);

	return file_list;
}

void
x_create_panel (Dlg_head *h, widget_data parent, WPanel *panel)
{
	g_panel_contents *g = g_new (g_panel_contents, 1);
	GtkWidget *table;

	g->table = gtk_table_new (1, 1, 0);
	gtk_widget_show (g->table);
	
	g->list  = panel_create_file_list (panel);
	gtk_widget_show (g->list);

	gtk_container_add (GTK_CONTAINER (g->table), g->list);
	
	gtk_widget_show (g->table);
	panel->widget.wdata = g;

	/* Now, insert our table in our parent */
	gtk_container_add (GTK_CONTAINER (gtk_object_get_data (GTK_OBJECT (h->wdata), "parent-container")),
			   g->table);
	
	if (!pixmaps_ready){
		if (!GTK_WIDGET_REALIZED (g->list))
			gtk_widget_realize (g->list);
		panel_create_pixmaps (g->list);
	}
}

void
panel_update_cols (Widget *panel, int frame_size)
{
	panel->cols  = 60;
	panel->lines = 20;
}

void
set_hintbar (char *str)
{
	printf ("Setting hint to %s\n", str);
}

char *get_nth_panel_name (int num)
{
    static char buffer [20];
    
    if (!num)
        return "New Left Panel";
    else if (num == 1)
        return "New Right Panel";
    else {
        sprintf (buffer, "%ith Panel", num);
        return buffer;
    }
}

void
paint_frame (WPanel *panel)
{
}
