/*--------------------------------*-C-*---------------------------------*
 * This module is all new by Rob Nation
 *
 * Modifications by mj olesen and Raul Garcia Garcia (rgg@tid.es)
 *----------------------------------------------------------------------*/
#include "rxvt.h"

#include <unistd.h>		/*#define HAVE_UNISTD_H*/
#include <X11/Xlib.h>
#include <X11/cursorfont.h>

#include "screen.h"
#include "xsetup.h"
#include "debug.h"
#include "graphics.h"

/* commands:
 * 'C' = Clear
 * 'F' = Fill
 * 'G' = Geometry
 * 'L' = Line
 * 'P' = Points
 * 'T' = Text
 * 'W' = Window
 */
#ifndef GRX_SCALE
# define GRX_SCALE	10000
#endif

/*----------------------------------------------------------------------*
 * extern variables referenced
 */
extern int fore_color;
/*----------------------------------------------------------------------*
 * extern variables declared here
 */
int graphics_up = 0;

/*----------------------------------------------------------------------*
 * local variables
 */
typedef struct grcommand_t {
   char command;
   int color;
   int ncoords;
   int *coords;
   unsigned char *text;
   struct grcommand_t *next;
} grcommand_t;

typedef struct grwin_t {
   Window win;
   int x, y, w, h;
   int screen;
   grcommand_t *graphics;
   struct grwin_t *prev, *next;
} grwin_t;

static grwin_t *gr_root = NULL;

/*----------------------------------------------------------------------*
 * extern functions referenced
 */

/*----------------------------------------------------------------------*
 * local functions
 */
static void Gr_CreateNewWindow (int nargs, long *args);
static void Gr_ClearWindow (grwin_t *grwin, grcommand_t *data);
static void Gr_DrawText (grwin_t *gr_win, grcommand_t *data);
static void Gr_ReturnGeom (grwin_t *grwin, grcommand_t *data);
static void Gr_FillArea (grwin_t *grwin, grcommand_t *data);
static void Gr_DestroyWindow (grwin_t *grwin);
static void Gr_Redraw (grwin_t *grwin);
static void DoOneCommand (grwin_t *grwin, grcommand_t *data);

/*----------------------------------------------------------------------*/

void
Gr_ButtonReport (char ch, int x, int y)
{
#ifdef RXVT_GRAPHICS
   grwin_t *grwin;

   for (grwin = gr_root; grwin != NULL; grwin = grwin->next)
     if ((x > grwin->x) &&
	 (y > grwin->y) &&
	 (x < grwin->x + grwin->w) &&
	 (y < grwin->y + grwin->h))
     break;

   if (grwin == NULL)
     return;

   x = GRX_SCALE * (x - grwin->x) / grwin->w;
   y = GRX_SCALE * (y - grwin->y) / grwin->h;
   tty_printf ("\033%c%ld;%d;%d;\n", ch, grwin->win, x, y);
#endif
}

void
Gr_do_graphics (char c, int nargs, long *args, unsigned char *text)
{
#ifdef RXVT_GRAPHICS
   static long last_id = 0;
   grwin_t *grwin;
   grcommand_t *newcom, *oldcom;
   long id;
   int i;

   if (c == 'W')
     {
	Gr_CreateNewWindow (nargs, args);
	return;
     }

   id = (nargs > 0) ? args[0] : None;

   if ((c == 'G') && (id == None))
     {
	Gr_ReturnGeom (NULL, NULL);
	return;
     }

   if (((Window)id == 0) && (last_id != 0))
     id = last_id;
   if ((Window)id == None)
     return;

   grwin = gr_root;
   while ((grwin != NULL) && (grwin->win != (Window)id))
     grwin = grwin->next;

   if (grwin == NULL)
     return;

   if (c == 'G')
     {
	Gr_ReturnGeom (grwin, NULL);
	return;
     }

   /* record this new command */
   newcom = MALLOC(sizeof(grcommand_t), "grcommand_t");
   newcom->ncoords = nargs-1;
   newcom->coords = MALLOC(newcom->ncoords*sizeof(long),"coords");
   newcom->next = NULL;
   newcom->command = c;
   newcom->color = fore_color;
   for (i = 0; i < newcom->ncoords; i++)
     newcom->coords[i] = args[i+1];
   if ((c == 'T') && (nargs >= 5))
     {
	newcom->text = MALLOC (args[4] * sizeof(char),"text");
	for (i = 0; i < args[4]; i++)
	  newcom->text[i] = text[i];
     }
   else
     newcom->text = NULL;
   /* If newcomm = fill area, and rectangle is full window, drop
    * all prior commands. */
   if ((newcom->command == 'F') && (grwin) && (grwin->graphics))
     {
	for (i = 0; i < newcom->ncoords; i+=4)
	  {
	     if ((newcom->coords[i] == 0) && (newcom->coords[i+1] == 0) &&
		 (newcom->coords[i+2] = GRX_SCALE) &&
		 (newcom->coords[i+3] = GRX_SCALE))
	       {
		  /* drop previous commands */
		  oldcom = grwin->graphics;
		  while (oldcom->next != NULL)
		    {
		       grcommand_t *t;
		       t = oldcom;
		       oldcom = oldcom->next;
		       free (t);
		    }
		  grwin->graphics = NULL;
	       }
	  }
     }
   /* insert new command into command list */

   oldcom = grwin->graphics;
   if (oldcom == NULL)
     grwin->graphics = newcom;
   else
     {
	while (oldcom->next != NULL)
	  oldcom = oldcom->next;
	oldcom->next = newcom;
     }
   DoOneCommand (grwin, newcom);
#endif
}

#ifdef RXVT_GRAPHICS
static void
Gr_CreateNewWindow (int nargs, long *args)
{
   int x, y, w, h;
   Window win1;
   grwin_t *grwin;
   Cursor cursor;

   if (nargs != 4)
     {
	printf ("CreateWindow: needed 4 args. Got %d\n",nargs);
	return;
     }

   x = args[0]*RxvtWin.pwidth  / GRX_SCALE + MARGIN;
   y = args[1]*RxvtWin.pheight / GRX_SCALE + MARGIN;
   w = args[2]*RxvtWin.pwidth  / GRX_SCALE;
   h = args[3]*RxvtWin.pheight / GRX_SCALE;

   if (w < 0 || h < 0)
     return;

   win1 = XCreateSimpleWindow (Xdisplay, RxvtWin.vt, x, y, w, h,
			       0,
			       pixel_colors [FG_COLOR],
			       pixel_colors [BG_COLOR]);
   cursor = XCreateFontCursor (Xdisplay, XC_left_ptr);
   XDefineCursor (Xdisplay, win1, cursor);
   XMapWindow (Xdisplay, win1);
   XSelectInput (Xdisplay, win1, ExposureMask);
   grwin = MALLOC (sizeof(grwin_t),"grwin");
   grwin->win = win1;
   grwin->x = x;
   grwin->y = y;
   grwin->w = w;
   grwin->h = h;
   grwin->screen = 0;
   grwin->prev = NULL;
   grwin->next = gr_root;
   if (grwin->next)
     grwin->next->prev = grwin;
   gr_root = grwin;
   grwin->graphics = NULL;
   tty_printf ("\033W%ld\n", (long)win1);

   graphics_up++;
}
#endif

#ifdef RXVT_GRAPHICS
static void
Gr_ClearWindow (grwin_t *grwin, grcommand_t *data)
{
   grcommand_t *grcom, *next;

   for (grcom = grwin->graphics; grcom != NULL; grcom = next)
     {
	next = grcom->next;
	free (grcom->coords);
	if (grcom->text != NULL)
	  free (grcom->text);
	free (grcom);
     }
   grwin->graphics = NULL;
   XClearWindow (Xdisplay, grwin->win);
}
#endif

#ifdef RXVT_GRAPHICS
static void
Gr_DrawText (grwin_t *grwin, grcommand_t *data)
{
   int x1, y1, alignment;

   if (data->ncoords >= 4)
     {
	x1 = data->coords[0] * grwin->w / GRX_SCALE;
	y1 = data->coords[1] * grwin->h / GRX_SCALE;
	alignment = data->coords[2];
	if ((alignment & HORIZONTAL_ALIGNMENT) == RIGHT_TEXT)
	  x1 -= XTextWidth (Xfont, data->text,data->coords[3]);
	if ((alignment & HORIZONTAL_ALIGNMENT) == HCENTER_TEXT)
	  x1 -= (XTextWidth (Xfont, data->text,data->coords[3])>>1);

	if ((alignment & VERTICAL_ALIGNMENT) == TOP_TEXT)
	  y1 += Xfont->ascent;
	if ((alignment & VERTICAL_ALIGNMENT) == BOTTOM_TEXT)
	  y1 -= Xfont->descent;
	if ((alignment & VERTICAL_ALIGNMENT) == VCENTER_TEXT)
	  y1 = y1 - Xfont->descent+((Xfont->ascent+Xfont->descent)>>1);
	if ((alignment & VERTICAL_ALIGNMENT) == VCAPS_CENTER_TEXT)
	  y1 += (Xfont->ascent>>1);

	XDrawString (Xdisplay, grwin->win, Xgc,
		     x1, y1,
		     data->text, data->coords[3]);
     }
}
#endif

#ifdef RXVT_GRAPHICS
static void
Gr_ReturnGeom (grwin_t *grwin, grcommand_t *data)
{
   if (grwin)
     tty_printf ("\033G%ld %ld %ld %ld %ld %ld %ld %ld %ld %ld\n",
		 grwin->win,
		 grwin->x, grwin->y, grwin->w, grwin->h,
		 RxvtWin.fwidth,
		 RxvtWin.fheight,
		 GRX_SCALE * RxvtWin.fwidth  / grwin->w,
		 GRX_SCALE * RxvtWin.fheight / grwin->h,
		 Xdepth);
   else				/* rxvt terminal window size */
     tty_printf ("\033G0 0 0 %ld %ld %ld %ld %ld %ld %ld\n",
		 RxvtWin.pwidth - 2*MARGIN,
		 RxvtWin.pheight - 2*MARGIN,
		 RxvtWin.fwidth,
		 RxvtWin.fheight,
		 GRX_SCALE * RxvtWin.fwidth  / (RxvtWin.pwidth  - 2 * MARGIN),
		 GRX_SCALE * RxvtWin.fheight / (RxvtWin.pheight - 2 * MARGIN),
		 Xdepth);
}
#endif

void
Gr_scroll (int count)
{
#ifdef RXVT_GRAPHICS
   static int last_offset = 0;
   grwin_t *grwin, *next;

   if ((count == 0) && (last_offset == RxvtWin.offset))
     {
	return;
     }

   last_offset = RxvtWin.offset;

   for (grwin = gr_root; grwin != NULL; grwin = next)
     {
	next = grwin->next;
	grwin->y -= count*RxvtWin.fheight;
	if (grwin->y + grwin->h < - RxvtWin.scrollback*RxvtWin.fheight)
	  Gr_DestroyWindow (grwin);
	else
	  XMoveWindow (Xdisplay, grwin->win,
		       grwin->x,
		       grwin->y + RxvtWin.offset*RxvtWin.fheight);
     }
#endif
}

void
Gr_ClearScreen (void)
{
#ifdef RXVT_GRAPHICS
   grwin_t *grwin, *next;

   for (grwin = gr_root; grwin != NULL; grwin = next)
     {
	next = grwin->next;
	if ((grwin->screen == 0) && (grwin->y + grwin->h > 0))
	  {
	     if (grwin->y >= 0)
	       Gr_DestroyWindow (grwin);
	     else
	       XResizeWindow (Xdisplay, grwin->win,
			      grwin->w, - grwin->y);
	  }
     }
#endif
}

void
Gr_ChangeScreen (void)
{
#ifdef RXVT_GRAPHICS
   grwin_t *grwin, *next;

   for (grwin = gr_root; grwin != NULL; grwin = next)
     {
	next = grwin->next;
	if (grwin->y + grwin->h > 0)
	  {
	     if (grwin->screen == 1)
	       {
		  XMapWindow (Xdisplay, grwin->win);
		  grwin->screen = 0;
	       }
	     else
	       {
		  XUnmapWindow (Xdisplay, grwin->win);
		  grwin->screen = 1;
	       }
	  }
     }
#endif
}

void
Gr_reset (void)
{
#ifdef RXVT_GRAPHICS
   grwin_t *grwin, *next;

   for (grwin = gr_root; grwin != NULL; grwin = next)
     {
	next = grwin->next;
	Gr_DestroyWindow (grwin);
     }

   graphics_up = 0;
#endif
}

#ifdef RXVT_GRAPHICS
static void
Gr_DestroyWindow (grwin_t *grwin)
{
   grcommand_t *grcom, *next;

   if (grwin == NULL)
     return;

   for (grcom = grwin->graphics; grcom != NULL; grcom = next)
     {
	next = grcom->next;
	free (grcom->coords);
	if (grcom->text != NULL)
	  free (grcom->text);
	free (grcom);
     }

   XDestroyWindow (Xdisplay, grwin->win);
   if (grwin->next != NULL)
     grwin->next->prev = grwin->prev;
   if (grwin->prev != NULL)
     grwin->prev->next = grwin->next;
   else
     gr_root = grwin->next;
   free (grwin);

   graphics_up--;
}
#endif

#ifdef RXVT_GRAPHICS
static void
Gr_Redraw (grwin_t *grwin)
{
   grcommand_t *grcom;
   for (grcom = grwin->graphics; grcom != NULL; grcom = grcom->next)
     DoOneCommand (grwin, grcom);
}
#endif

void
Gr_expose (Window win)
{
#ifdef RXVT_GRAPHICS
   grwin_t *grwin;

   for (grwin = gr_root; grwin != NULL; grwin = grwin->next)
     {
	if (grwin->win == win)
	  {
	     Gr_Redraw (grwin);
	     break;
	  }
     }
#endif
}

#ifdef RXVT_GRAPHICS
static void
DoOneCommand (grwin_t *grwin, grcommand_t *data)
{
   int i, n;
   union {
      XPoint pt [2*250];
      XRectangle rect [250];
   } xdata;

#ifdef COLOR
   XGCValues newgcv;

   if (data->color != 0)
     {
	newgcv.foreground = pixel_colors [data->color];
	XChangeGC (Xdisplay, Xgc, GCForeground, &newgcv);
     }
#endif
   switch (data->command)
     {
      case 'L':
	if (grwin && (data->ncoords > 3))
	  {
	     for (n = i = 0; i < data->ncoords; i += 2, n++)
	       {
		  xdata.pt [n].x = data->coords [i]   * grwin->w / GRX_SCALE;
		  xdata.pt [n].y = data->coords [i+1] * grwin->h / GRX_SCALE;
	       }
	     XDrawLines (Xdisplay,
			 grwin->win, Xgc, xdata.pt, n, CoordModeOrigin);
	  }
	break;

      case 'P':
	if (grwin && (data->ncoords > 3))
	  {
	     for (n = i = 0; i < data->ncoords; i += 2, n++)
	       {
		  xdata.pt [n].x = data->coords [i]   * grwin->w / GRX_SCALE;
		  xdata.pt [n].y = data->coords [i+1] * grwin->h / GRX_SCALE;
	       }
	     XDrawPoints (Xdisplay,
			  grwin->win, Xgc, xdata.pt, n, CoordModeOrigin);
	  }
	break;

      case 'F':
	if ((data->ncoords > 0) && (grwin))
	  {
	     for (n = i = 0; i < data->ncoords; i += 4, n++)
	       {
		  xdata.rect [n].x = data->coords[i]   * grwin->w / GRX_SCALE;
		  xdata.rect [n].y = data->coords[i+1] * grwin->h / GRX_SCALE;
		  xdata.rect [n].width = ((data->coords[i+2]
					   - data->coords[i]+1) *
					  grwin->w / GRX_SCALE);
		  xdata.rect [n].height = ((data->coords[i+3]
					    - data->coords[i+1]+1) *
					   grwin->h / GRX_SCALE);
	       }
	     XFillRectangles (Xdisplay, grwin->win, Xgc, xdata.rect, n);
	  }
	break;
      case 'T':	Gr_DrawText (grwin, data);	break;
      case 'C':	Gr_ClearWindow (grwin, data);
     }
#ifdef COLOR
   if (data->color != 0)
     {
	newgcv.foreground = pixel_colors [FG_COLOR];
	XChangeGC (Xdisplay, Xgc, GCForeground, &newgcv);
     }
#endif
}
#endif

void
Gr_Resize (int w, int h)
{
#ifdef RXVT_GRAPHICS
   grwin_t *grwin;

   for (grwin = gr_root; grwin != NULL; grwin = grwin->next)
     {
	if (RxvtWin.pheight != h)
	  {
	     grwin->y += (RxvtWin.pheight - h);
	     XMoveWindow (Xdisplay, grwin->win,
			  grwin->x,
			  grwin->y + RxvtWin.offset*RxvtWin.fheight);
	  }
	Gr_Redraw (grwin);
     }
#endif
}
/*----------------------- end-of-file (C source) -----------------------*/
