/*=============================================================================

			      HydraCom Version 1.00

			 A sample implementation of the
		   HYDRA Bi-Directional File Transfer Protocol

			     HydraCom was written by
		   Arjen G. Lentz, LENTZ SOFTWARE-DEVELOPMENT
		  COPYRIGHT (C) 1991-1993; ALL RIGHTS RESERVED

		       The HYDRA protocol was designed by
		 Arjen G. Lentz, LENTZ SOFTWARE-DEVELOPMENT and
			     Joaquim H. Homrighausen
		  COPYRIGHT (C) 1991-1993; ALL RIGHTS RESERVED


  Revision history:
  06 Sep 1991 - (AGL) First tryout
  .. ... .... - Internal development
  11 Jan 1993 - HydraCom version 1.00, Hydra revision 001 (01 Dec 1992)


  For complete details of the Hydra and HydraCom licensing restrictions,
  please refer to the license agreements which are published in their entirety
  in HYDRACOM.C and LICENSE.DOC, and also contained in the documentation file
  HYDRACOM.DOC

  Use of this file is subject to the restrictions contained in the Hydra and
  HydraCom licensing agreements. If you do not find the text of this agreement
  in any of the aforementioned files, or if you do not have these files, you
  should immediately contact LENTZ SOFTWARE-DEVELOPMENT and/or Joaquim 
  Homrighausen at one of the addresses listed below. In no event should you
  proceed to use this file without having accepted the terms of the Hydra and
  HydraCom licensing agreements, or such other agreement as you are able to
  reach with LENTZ SOFTWARE-DEVELOMENT and Joaquim Homrighausen.


  Hydra protocol design and HydraCom driver:	     Hydra protocol design:
  Arjen G. Lentz				     Joaquim H. Homrighausen
  LENTZ SOFTWARE-DEVELOPMENT			     389, route d'Arlon
  Langegracht 7B				     L-8011 Strassen
  3811 BT  Amersfoort				     Luxembourg
  The Netherlands
  FidoNet 2:283/512, AINEX-BBS +31-33-633916	     FidoNet 2:270/17
  arjen_lentz@f512.n283.z2.fidonet.org		     joho@ae.lu

  Please feel free to contact us at any time to share your comments about our
  software and/or licensing policies.

=============================================================================*/

#include <sys/time.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>
#include <termios.h>
#include <paths.h>
#include "xutil.h"
#include "ttyio.h"
#include "emsi.h"
#include "session.h"
#include "hydra.h"		/* need 2types.h & hydra.h */

extern int errno;
extern file_list *respond_wazoo(char*);
#ifdef SHOW_SPEED
extern int connect_speed;
#endif

/* HYDRA Some stuff to aid readability of the source and prevent typos ----- */
#define hcrc16(crc,c)  (crc16tab[(	     crc ^ (c)) & 0xff] ^ ((crc >> 8) & 0x00ff))
#define hcrc32(crc,c)  (crc32tab[((byte) crc ^ (c)) & 0xff] ^ ((crc >> 8) & 0x00ffffffL))
#define h_crc16poly	   (0x8408)
#define h_crc32poly	   (0xedb88320L)
#define h_crc16test(crc)   (((crc) == 0xf0b8	 ) ? 1 : 0)
#define h_crc32test(crc)   (((crc) == 0xdebb20e3L) ? 1 : 0)
static	word   *crc16tab;		/* CRC-16 table		     */
static	dword  *crc32tab;		/* CRC-32 table		     */

#define h_uuenc(c)	   (((c) & 0x3f) + '!')
#define h_uudec(c)	   (((c) - '!') & 0x3f)
#define h_long1(buf)	   (*((long *) (buf)))
#define h_long2(buf)	   (*((long *) ((buf) + ((int) sizeof (long)))))
#define h_long3(buf)	   (*((long *) ((buf) + (2 * ((int) sizeof (long))))))
typedef long		   h_timer;
#define h_timer_set(t)	   (time(NULL) + (t))
#define h_timer_running(t) (t != 0L)
#define h_timer_expired(t) (time(NULL) > (t))
#define h_timer_reset()	   (0L)


/* HYDRA's memory ---------------------------------------------------------- */
static	boolean originator;		/* are we the orig side?     */
static	int	batchesdone;		/* No. HYDRA batches done    */
static	boolean hdxlink;		/* hdx link & not orig side  */
static	dword	hydra_options;		/* INIT options hydra_init() */
static	word	timeout;			/* general timeout in secs   */
static	char	abortstr[] = { 24,24,24,24,24,24,24,24,8,8,8,8,8,8,8,8,8,8,0 };
static	char   *hdxmsg	   = "Fallback to one-way xfer";
static	char   *pktprefix  = "";
static	char   *autostr	   = "hydra\r";

static	byte   *txbuf,	       *rxbuf;		/* packet buffers	     */
static	dword	htxoptions, hrxoptions;	     /* HYDRA options (INIT seq)  */
static	char	txpktprefix[H_PKTPREFIX + 1];	/* pkt prefix str they want  */
static	long	htxwindow,	hrxwindow;	/* window size (0=streaming) */
static	h_timer			braindead;	/* braindead timer	     */
static	byte   *txbufin;			/* read data from disk here  */
static	byte	txlastc;			/* last byte put in txbuf    */
static	byte			rxdle;		/* count of received H_DLEs  */
static	byte			rxpktformat;	/* format of pkt receiving   */
static	byte		       *rxbufptr;	/* current position in rxbuf */
static	byte		       *rxbufmax;	/* highwatermark of rxbuf    */
static	char	txfname[256],	 rxfname[256];	/* fname of current files    */
static	char		       *rxpathname;	/* pointer to rx pathname    */
static	long	txftime,	rxftime;	/* file timestamp (UNIX)     */
static	long	txfsize,	rxfsize;	/* file length		     */
static	int	txfd,		rxfd;		/* file handles		     */
static	word			rxpktlen;	/* length of last packet     */
static	word			rxblklen;	/* len of last good data blk */
static	byte	txstate,	rxstate;	/* xmit/recv states	     */
static	long	txpos,		rxpos;		/* current position in files */
static	word	txblklen;			/* length of last block sent */
static	word	txmaxblklen;			/* max block length allowed  */
static	long	txlastack;			/* last dataack received     */
static	long	txstart,	rxstart;	/* time we started this file */
static	long	txoffset,	rxoffset;	/* offset in file we begun   */
static	h_timer txtimer,	rxtimer;	/* retry timers		     */
static	word	txretries,	rxretries;	/* retry counters	     */
static	long			rxlastsync;	/* filepos last sync retry   */
static	long	txsyncid,	rxsyncid;	/* id of last resync	     */
static	word	txgoodneeded;			/* to send before larger blk */
static	word	txgoodbytes;			/* no. sent at this blk size */

char	*frqpathfile = NULL;			/* Hydra's freq file name */
char	ishfreq = 0;				/* semaphore for Hydra's freq */

struct _h_flags {
	char  *str;
	dword  val;
};

static struct _h_flags h_flags[] = {
	{ "XON", HOPT_XONXOFF },
	{ "TLN", HOPT_TELENET },
	{ "CTL", HOPT_CTLCHRS },
	{ "HIC", HOPT_HIGHCTL },
	{ "HI8", HOPT_HIGHBIT },
	{ "BRK", HOPT_CANBRK  },
	{ "ASC", HOPT_CANASC  },
	{ "UUE", HOPT_CANUUE  },
	{ "C32", HOPT_CRC32   },
	{ "DEV", HOPT_DEVICE  },
	{ "FPT", HOPT_FPT     },
	{ NULL , 0x0L	      }
};

#define NAMELEN	 256
#define LINELEN	 512

static char xfer_log[PATHLEN];
static boolean xfer_logged;
static char xfer_pathname[PATHLEN];
static char xfer_real[NAMELEN],
	    xfer_temp[NAMELEN];
static long xfer_fsize,
	    xfer_ftime;

extern		hanged_up;
/*---------------------------------------------------------------------------*/

extern int errno;
static struct termios oldtty;

int sys_init(void)	   /* all system dependent init should be done here */
{
	struct termios tty;

	if (tcgetattr(fdin, &tty) != 0) {
		loginf("Can't get terminal setting: %s",
			strerror(errno));
		endprog(1);
	}
	oldtty = tty;
	cfmakeraw(&tty);
	if (flowflags & 0x02 || flowflags & 0x0b) tty.c_cflag |= CRTSCTS;
	if (parity) tty.c_cflag |= (CS7 | PARENB);
	if (tcsetattr(fdin, TCSAFLUSH, &tty) < 0) {
		loginf("Can't set terminal settings: %s",
			strerror(errno));
		endprog(1);
	}
	if (tcsetattr(fdout, TCSADRAIN, &tty) < 0) {
		loginf("Can't set terminal settings: %s",
			strerror(errno));
		endprog(1);
	}
	return((int)cfgetispeed(&tty));
}

void sys_restore(void)
{
	if (tcsetattr(fdin, TCSAFLUSH, &oldtty) < 0)
		loginf("Can't return terminal settings: %s",
			strerror(errno));
}

#define com_putbyte(x)		PUTCHAR(x)
#define com_getbyte()		GETCHAR(1)
/* #define com_putblock(s,x)	PUT(s,x) */

void com_putblock (byte *s, word len)
{
	int	numsent = 0;
Again:
	while (!hanged_up && (len > 0)) {
		numsent = write(fdout, s, len);
		if (numsent < 1) {
			if (errno == EAGAIN) goto Again;
			debug(11,"com_putblock: EAGAIN");
		}
		else { len -= numsent; s +=numsent; }
	}
}

int com_outfull (void)		
{				
	int queue = 0;

#ifndef NO_COM_OUTFULL
	ioctl(fdout, TIOCOUTQ, &queue);
#endif
	return queue;
}

void com_flush()
{
#ifdef HAS_SGTTY_H
	int	fd;
	fd = FWRITE;
	ioctl(fdout, TIOCFLUSH, &fd);
#else
	tcdrain(fdout);
#endif
	usleep(100000);
}

void com_purge()
{
#ifdef HAS_SGTTY_H
	int	fd;
	fd = FREAD;
	ioctl(fdout, TIOCFLUSH, &fd);
#else
	tcflush(fdin, TCIFLUSH);
#endif
}

void com_dump()
{
#ifdef HAS_SGTTY_H
	int	fd;
	fd = FWRITE;
	ioctl(fdout, TIOCFLUSH, &fd);
#else
	tcflush(fdout, TCOFLUSH);
#endif
}

/*-------------------------------------------------------------------*/
void unique_name (char *pathname)
{
	static char *suffix = ".000";
	register char *p;
	register int   n;

	if (fexist(pathname)) {
	   p = pathname;
	   while (*p && *p!='.') p++;
	   for (n=0; n<4; n++) {
	       if (!*p) {
		  *p	 = suffix[n];
		  *(++p) = '\0';
	       }
	       else
		  p++;
	   }
	   
	   while (fexist(pathname)) {
		 p = pathname + ((int) strlen(pathname)) - 1;
		 if (!isdigit(*p))
		    *p = '0';
		 else {
		    for (n=3; n--;) {
			if (!isdigit(*p)) *p = '0';
			if (++(*p) <= '9') break;
			else		   *p-- = '0';
		    }/*for*/
		 }
	   }/*while(exist)*/
	}/*if(exist)*/
}/*unique_name()*/

/*-------------------------------------------------------------------*/	 
char *xfer_init (char *fname, long fsize, long ftime)
{
	char  linebuf[LINELEN + 1];
	char  bad_real[NAMELEN],
	      bad_temp[NAMELEN],
	      bad_from[40];
	long  bad_fsize,
	      bad_ftime,
	      bad_stime;
	char *p;
	FILE *fp;

	if (single_done)
	   return (NULL);

	strcpy(xfer_real,fname);
	xfer_fsize = fsize;
	xfer_ftime = ftime;

	mergepath(xfer_pathname,download,xfer_real);
	if (fexist(xfer_pathname)) {
	   struct stat f;

	   stat(xfer_pathname,&f);
	   if (xfer_fsize == f.st_size && xfer_ftime == f.st_mtime)
	      return (NULL);				/* already have file */
	}

	mergepath(xfer_log,download,"/tmp/BAD-XFER.LOG");

	if ((fp = fopen(xfer_log,"r")) != NULL) {
	   while (fgets(linebuf,LINELEN,fp)) {
		 sscanf(linebuf,"%s %s %ld %lo %lo %s",
				bad_real, bad_temp, &bad_fsize, &bad_ftime,
				&bad_stime, bad_from);
		 if (!strcmp(xfer_real,bad_real) &&
		     xfer_fsize == bad_fsize && xfer_ftime == bad_ftime) {
		    mergepath(xfer_pathname,download,bad_temp);
		    if (fexist(xfer_pathname)) {
		       fclose(fp);
		       strcpy(xfer_temp,bad_temp);

		       xfer_logged = true;
		       return (xfer_pathname);
		    }
		 }
	   }

	   fclose(fp);
	}

	strcpy(xfer_pathname,download);
	p = xfer_pathname + ((int) strlen(xfer_pathname));
	strcat(xfer_pathname,"/tmp/BAD-XFER.000");
	unique_name(xfer_pathname);
	strcpy(xfer_temp,p);

	xfer_logged = false;
	return (xfer_pathname);
}/*xfer_init()*/

/*-------------------------------------------------------------------*/
boolean xfer_bad (void)
{
	struct stat f;
	FILE *fp;
	int n = 0;
	struct flock fl;
	
	fl.l_type = F_WRLCK;
	fl.l_whence = 0;
	fl.l_start = 0L;
	fl.l_len = 0L;

	if (single_file[0])
	   single_done = true;

	if (xfer_logged)			/* Already a logged bad-xfer */
	   return (true);

	if ((strlen (xfer_real) <=12) || strspn (xfer_real, "0123456789abcdefABCDEF") == 8)
/*	   for (n=0; n<=strlen (xfer_real); n++)
	      *(xfer_real + n) = tolower (*(xfer_real + n )); */	 

	n = ((int) strlen(xfer_real)) - 1;
	if (n > 3 &&
	    (!strcmp(&xfer_real[n-3],".PKT") || !strcmp(&xfer_real[n-3],".REQ"))) {
	   xfer_del();
	   return (false);			/* don't recover .PKT / .REQ */
	}

	stat(xfer_pathname,&f);
	if (noresume || f.st_size < 1024L) {	 /* not allowed/worth saving */
	   xfer_del();
	   return (false);
	}

	if ((fp = fopen(xfer_log,"a")) != NULL) {
	   fprintf(fp,"%s %s %ld %lo %lo %s\n",
		     xfer_real, xfer_temp, xfer_fsize, xfer_ftime,
		     time (NULL), ascfnode(remote->addr, 0x1f));
	   fl.l_pid = getpid();
	   if ( fcntl( fileno(fp), F_SETLK, &fl) != 0) {
		loginf ("$cannot lock local file \"%s\"", xfer_log);
		fclose (fp);
		xfer_del ();
		return (false);
	   }
	   fclose(fp);

	   return (true);			      /* bad-xfer logged now */
	}

	xfer_del();
	return (false);				    /* Couldn't log bad-xfer */
}/*xfer_bad()*/

/*-------------------------------------------------------------------*/
char *xfer_okay (void)
{
	static char new_pathname[PATHLEN];
	char *p;

	strcpy (new_pathname, download);
	p = new_pathname + ((int) strlen (new_pathname)); /* start of fname */

	if (*p != '/' && *p != '\\') { strcat (new_pathname, "/"); p++; }

	if (single_file[0]) {
	   strcat (new_pathname, single_file);	/* add override fname */
	   single_done = true;
	}
	else {
	   strcat (new_pathname, xfer_real);	/* add real fname */
	   unique_name (new_pathname);		/* make it unique */
	}
	rename (xfer_pathname, new_pathname);	/* rename temp to real */
	if (!nostamp && xfer_ftime)
	   setstamp (new_pathname, xfer_ftime); /* set timestamp */

	if (xfer_logged)			/* delete from bad-xfer log */
	   xfer_del();

	if ((strlen (xfer_real) == 12) &&
		(strspn (xfer_real, "0123456789abcdefABCDEF") == 8) &&
		(strcasecmp (xfer_real + 8, ".req") == 0)) {
	   char pidstr[10];

	   memset (pidstr, 0, 10);
	   debug (11, "received wazoo freq file");
	   ishfreq = 1;

	   if (frqpathfile)	free (frqpathfile);

	   sprintf (pidstr, "{%lx}", (unsigned long) getpid ());

	   frqpathfile = xstrcpy (new_pathname);
	   frqpathfile = xstrcat (frqpathfile,pidstr);
	   if (rename (new_pathname, frqpathfile))
	   loginf ("cannote rename %s->%s", new_pathname, frqpathfile);
	}
	else	ishfreq = 0;

	return (strcmp(p,xfer_real) ? p : NULL);	      /* dup rename? */
}

/*-------------------------------------------------------------------*/
void xfer_del (void)
{
	char  new_log[PATHLEN];
	char  linebuf[LINELEN + 1];
	char  bad_real[NAMELEN],
	      bad_temp[NAMELEN],
	      bad_from[40];
	long  bad_fsize,
	      bad_ftime,
	      bad_stime;
	FILE *fp, *new_fp;
	boolean left;

	if (fexist(xfer_pathname))
	   unlink(xfer_pathname);

	if ((fp = fopen(xfer_log, "r")) != NULL) {
	   mergepath(new_log,download,"BAD-XFER.$$$");
	   if ((new_fp = fopen(new_log, "w")) != NULL) {
	      left = false;
	      while (fgets(linebuf,LINELEN,fp)) {
		    sscanf(linebuf,"%s %s %ld %lo %lo %s",
				   bad_real, bad_temp, &bad_fsize, &bad_ftime,
				   &bad_stime, bad_from);
		    if (strcmp(xfer_real,bad_real) ||
			strcmp(xfer_temp,bad_temp) ||
			xfer_fsize != bad_fsize || xfer_ftime != bad_ftime) {
		       fputs(linebuf,new_fp);
		       left = true;
		    }
	      }
	      fclose(fp);
	      fclose(new_fp);
	      unlink(xfer_log);
	      if (left) rename(new_log,xfer_log);
	      else	unlink(new_log);
	   }
	   else
	      fclose(fp);
	}
}/*xfer_del()*/

/******************/
/* end of fmisc.c */
/******************/

/*---------------------------------------------------------------------------*/
static void hydra_msgdev (byte *data, word len)
{	/* text is already NUL terminated by calling func hydra_devrecv() */
	len = len;
	debug(3,"*HMSGDEV: %s",data);
}/*hydra_msgdev()*/

/*---------------------------------------------------------------------------*/
static	word	devtxstate;			/* dev xmit state	     */
static	h_timer devtxtimer;			/* dev xmit retry timer	     */
static	word	devtxretries;			/* dev xmit retry counter    */
static	long	devtxid,	devrxid;	/* id of last devdata pkt    */
static	char	devtxdev[H_FLAGLEN + 1];	/* xmit device ident flag    */
static	byte   *devtxbuf;			/* ptr to usersupplied dbuf  */
static	word	devtxlen;			/* len of data in xmit buf   */

struct _h_dev {
	char  *dev;
	void (*func) (byte *data, word len);
};

static	struct _h_dev h_dev[] = {
	{ "MSG", hydra_msgdev },		/* internal protocol msg     */
	{ "CON", NULL	      },		/* text to console (chat)    */
	{ "PRN", NULL	      },		/* data to printer	     */
	{ "ERR", NULL	      },		/* text to error output	     */
	{ NULL , NULL	      }
};

/*---------------------------------------------------------------------------*/
boolean hydra_devfree (void)
{
	if (devtxstate || !(htxoptions & HOPT_DEVICE) || txstate >= HTX_END)
	   return (false);			/* busy or not allowed	     */
	else
	   return (true);			/* allowed to send a new pkt */
}/*hydra_devfree()*/

/*---------------------------------------------------------------------------*/
boolean hydra_devsend (char *dev, byte *data, word len)
{
	if (!dev || !data || !len || !hydra_devfree())
	   return (false);

	strncpy(devtxdev,dev,H_FLAGLEN);
	devtxdev[H_FLAGLEN] = '\0';
	strupr(devtxdev);
	devtxbuf = data;
	devtxlen = (len > H_MAXBLKLEN) ? H_MAXBLKLEN : len;

	devtxid++;
	devtxtimer   = h_timer_reset();
	devtxretries = 0;
	devtxstate   = HTD_DATA;

	/* special for chat, only prolong life if our side keeps typing! */
	if (chattimer > 0L && !strcmp(devtxdev,"CON") && txstate == HTX_REND)
	   braindead = h_timer_set(H_BRAINDEAD);

	return (true);
}/*hydra_devsend()*/

/*---------------------------------------------------------------------------*/
boolean hydra_devfunc (char *dev, void (*func) (byte *data, word len))
{
	register int i;

	for (i = 0; h_dev[i].dev; i++) {
	    if (!strncasecmp(dev,h_dev[i].dev,H_FLAGLEN)) {
	       h_dev[i].func = func;
	       return (true);
	    }
	}

	return (false);
}/*hydra_devfunc()*/

/*---------------------------------------------------------------------------*/
static void hydra_devrecv (void)
{
	register char *p = (char *) rxbuf;
	register int   i;
	word len = rxpktlen;

	p += (int) sizeof (long);			/* skip the id long  */
	len -= (int) sizeof (long);
	for (i = 0; h_dev[i].dev; i++) {		/* walk through devs */
	    if (!strncmp(p,h_dev[i].dev,H_FLAGLEN)) {
	       if (h_dev[i].func) {
		  len -= ((int) strlen(p)) + 1;		/* sub devstr len    */
		  p += ((int) strlen(p)) + 1;		/* skip devtag	     */
		  p[len] = '\0';			/* NUL terminate     */
		  (*h_dev[i].func)((byte *) p,len);	/* call output func  */
	       }
	       break;
	    }
	}
}/*hydra_devrecv()*/

/*---------------------------------------------------------------------------*/
static void put_flags (char *buf, struct _h_flags flags[], long val)
{
	register char *p;
	register int   i;

	p = buf;
	for (i = 0; flags[i].val; i++) {
	    if (val & flags[i].val) {
	       if (p > buf) *p++ = ',';
	       strcpy(p,flags[i].str);
	       p += H_FLAGLEN;
	    }
	}
	*p = '\0';
}/*put_flags()*/

/*---------------------------------------------------------------------------*/
static dword get_flags (char *buf, struct _h_flags flags[])
{
	register dword	val;
	register char  *p;
	register int	i;

	val = 0x0L;
	for (p = strtok(buf,","); p; p = strtok(NULL,",")) {
	    for (i = 0; flags[i].val; i++) {
		if (!strcmp(p,flags[i].str)) {
		   val |= flags[i].val;
		   break;
		}
	    }
	}

	return (val);
}/*get_flags()*/

/*---------------------------------------------------------------------------*/
static word crc16block (register byte *buf, register word len)
{
	register word crc;

	for (crc = 0xffff; len > 0; len--)
	    crc = hcrc16(crc,*buf++);
	return (crc);
}/*crc16block()*/

/*---------------------------------------------------------------------------*/
static dword crc32block (register byte *buf, register word len)
{
	register dword crc;

	for (crc = 0xffffffffL; len > 0; len--)
	    crc = hcrc32(crc,*buf++);
	return (crc);
}/*crc32block()*/

/*---------------------------------------------------------------------------*/
static byte *put_binbyte (register byte *p, register byte c)
{
	register byte n;

	n = c;
	if (htxoptions & HOPT_HIGHCTL)
	   n &= 0x7f;

	if (n == H_DLE ||
	    ((htxoptions & HOPT_XONXOFF) && (n == XON || n == XOFF)) ||
	    ((htxoptions & HOPT_TELENET) && n == '\r' && txlastc == '@') ||
	    ((htxoptions & HOPT_CTLCHRS) && (n < 32 || n == 127))) {
	   *p++ = H_DLE;
	   c ^= 0x40;
	}

	*p++ = c;
	txlastc = n;

	return (p);
}/*put_binbyte()*/

/*---------------------------------------------------------------------------*/
static void txpkt (register word len, int type)
{
	register byte *in, *out;
	register word  c, n;
	boolean crc32 = false;
	byte	format;
	static char hexdigit[] = "0123456789abcdef";

	txbufin[len++] = type;

	switch (type) {
	       case HPKT_START:
	       case HPKT_INIT:
	       case HPKT_INITACK:
	       case HPKT_END:
	       case HPKT_IDLE:
		    format = HCHR_HEXPKT;
		    break;

	       default:
		    /* COULD do smart format selection depending on data and options! */
		    if (htxoptions & HOPT_HIGHBIT) {
		       if ((htxoptions & HOPT_CTLCHRS) && (htxoptions & HOPT_CANUUE))
			  format = HCHR_UUEPKT;
		       else if (htxoptions & HOPT_CANASC)
			  format = HCHR_ASCPKT;
		       else
			  format = HCHR_HEXPKT;
		    }
		    else
		       format = HCHR_BINPKT;
		    break;
	}

	if (format != HCHR_HEXPKT && (htxoptions & HOPT_CRC32))
	   crc32 = true;

  {
   char *s1, *s2, *s3, *s4;

   debug(11," tx PKT (format='%c'  type='%c'  crc=%d  len=%d)",
	     format, type, crc32 ? 32 : 16, len - 1);

   switch (type) {
	  case HPKT_START:    debug(11,"    <autostr>START");
			      break;
	  case HPKT_INIT:     s1 = ((char *) txbufin) + ((int) strlen((char *) txbufin)) + 1;
			      s2 = s1 + ((int) strlen(s1)) + 1;
			      s3 = s2 + ((int) strlen(s2)) + 1;
			      s4 = s3 + ((int) strlen(s3)) + 1;
			      debug(11,"    INIT (appinfo='%s'	can='%s'  want='%s'  options='%s'  pktprefix='%s')",
				      (char *) txbufin, s1, s2, s3, s4);
			      break;
	  case HPKT_INITACK:  debug(11,"    INITACK");
			      break;
	  case HPKT_FINFO:    debug(11,"    FINFO (%s)",txbufin);
			      break;
	  case HPKT_FINFOACK: if (rxfd >= 0) {
				 if (rxpos > 0L) s1 = "RES";
				 else		 s1 = "BOF";
			      }
			      else if (rxpos == -1L) s1 = "HAVE";
			      else if (rxpos == -2L) s1 = "SKIP";
			      else		     s1 = "EOB";
			      debug(11,"    FINFOACK (pos=%ld %s  rxstate=%d  rxfd=%d)",
				      rxpos,s1,rxstate,rxfd);
			      break;
	  case HPKT_DATA:     debug(11,"    DATA (ofs=%ld  len=%d)",
				      intell(h_long1(txbufin)), len - 5);
			      break;
	  case HPKT_DATAACK:  debug(11,"    DATAACK (ofs=%ld)",
				      intell(h_long1(txbufin)));
			      break;
	  case HPKT_RPOS:     debug(11,"    RPOS (pos=%ld%s  blklen=%ld	 syncid=%ld)",
				      rxpos, rxpos < 0L ? " SKIP" : "",
				      intell(h_long2(txbufin)), rxsyncid);
			      break;
	  case HPKT_EOF:      debug(11,"    EOF (ofs=%ld%s)",
				      txpos, txpos < 0L ? " SKIP" : "");
			      break;
	  case HPKT_EOFACK:   debug(11,"    EOFACK");
			      break;
	  case HPKT_IDLE:     debug(11,"    IDLE");
			      break;
	  case HPKT_END:      debug(11,"    END");
			      break;
	  case HPKT_DEVDATA:  debug(11,"    DEVDATA (id=%ld  dev='%s'  len=%u)",
				      devtxid, devtxdev, devtxlen);
			      break;
	  case HPKT_DEVDACK:  debug(11,"    DEVDACK (id=%ld)",
				      intell(h_long1(rxbuf)));
			      break;
	  default:	      /* This couldn't possibly happen! ;-) */
			      break;
   } /* switch (type) */
  } /* old H_DEBUG */

	if (crc32) {
	   dword crc = ~crc32block(txbufin,len);

	   txbufin[len++] = crc;
	   txbufin[len++] = crc >> 8;
	   txbufin[len++] = crc >> 16;
	   txbufin[len++] = crc >> 24;
	}
	else {
	   word crc = ~crc16block(txbufin,len);

	   txbufin[len++] = crc;
	   txbufin[len++] = crc >> 8;
	}
/*
 * Convert buffer for send.
 */
	in = txbufin;
	out = txbuf;
	txlastc = 0;
	*out++ = H_DLE;
	*out++ = format;

	switch (format) {
	       case HCHR_HEXPKT:
		    for (; len > 0; len--, in++) {
			if (*in & 0x80) {
			   *out++ = '\\';
			   *out++ = hexdigit[((*in) >> 4) & 0x0f];
			   *out++ = hexdigit[(*in) & 0x0f];
			}
			else if (*in < 32 || *in == 127) {
			   *out++ = H_DLE;
			   *out++ = (*in) ^ 0x40;
			}
			else if (*in == '\\') {
			   *out++ = '\\';
			   *out++ = '\\';
			}
			else
			   *out++ = *in;
		    }
		    break;

	       case HCHR_BINPKT:
		    for (; len > 0; len--)
			out = put_binbyte(out,*in++);
		    break;

	       case HCHR_ASCPKT:
		    for (n = c = 0; len > 0; len--) {
			c |= ((*in++) << n);
			out = put_binbyte(out,c & 0x7f);
			c >>= 7;
			if (++n >= 7) {
			   out = put_binbyte(out,c & 0x7f);
			   n = c = 0;
			}
		    }
		    if (n > 0)
		       out = put_binbyte(out,c & 0x7f);
		    break;

	       case HCHR_UUEPKT:
		    for ( ; len >= 3; in += 3, len -= 3) {
			*out++ = h_uuenc(in[0] >> 2);
			*out++ = h_uuenc(((in[0] << 4) & 0x30) | ((in[1] >> 4) & 0x0f));
			*out++ = h_uuenc(((in[1] << 2) & 0x3c) | ((in[2] >> 6) & 0x03));
			*out++ = h_uuenc(in[2] & 0x3f);
		    }
		    if (len > 0) {
		       *out++ = h_uuenc(in[0] >> 2);
		       *out++ = h_uuenc(((in[0] << 4) & 0x30) | ((in[1] >> 4) & 0x0f));
		       if (len == 2)
			  *out++ = h_uuenc((in[1] << 2) & 0x3c);
		    }
		    break;
	} /* switch (format) */

	*out++ = H_DLE;
	*out++ = HCHR_PKTEND;

	if (type != HPKT_DATA && format != HCHR_BINPKT) {
	   *out++ = '\r';
	   *out++ = '\n';
	}
/*
 * Send PacketPrefix to modem.
 */
	for (in = (byte *) txpktprefix; *in; in++) {
	    switch (*in) {
		   case 221: /* transmit break signal for one second */
			     break;
		   case 222: sleep(1);
			     break;
		   case 223: com_putbyte(0);
			     break;
		   default:  com_putbyte(*in);
			     break;
	    }
	}
/*
 * Send buffer to modem.
 */
	com_putblock(txbuf,(word) (out - txbuf));
	usleep(100000);

}/*txpkt()*/

/*---------------------------------------------------------------------------*/
static int rxpkt (void)
{
	register byte *p, *q=NULL;
	register int   c, n, i;

	if (hanged_up)
	   return (H_CARRIER);

	p = rxbufptr;
/*
 * Read 1 byte from modem.
 */
	while ((c = com_getbyte()) >= 0) {
	      if (hrxoptions & HOPT_HIGHBIT)
		 c &= 0x7f;

	      n = c;
	      if (hrxoptions & HOPT_HIGHCTL)
		 n &= 0x7f;
	      if (n != H_DLE &&
		  (((hrxoptions & HOPT_XONXOFF) && (n == XON || n == XOFF)) ||
		   ((hrxoptions & HOPT_CTLCHRS) && (n < 32 || n == 127))))
		 continue;

	      if (rxdle || c == H_DLE) {
		 switch (c) {
			case H_DLE:
			     if (++rxdle >= 5)
				return (H_CANCEL);
			     break;

			case HCHR_PKTEND:
			     rxbufptr = p;

			     switch (rxpktformat) {
				    case HCHR_BINPKT:
					 q = rxbufptr;
					 break;

				    case HCHR_HEXPKT:
					 for (p = q = rxbuf; p < rxbufptr; p++) {
					     if (*p == '\\' && *++p != '\\') {
						i = *p;
						n = *++p;
						if ((i -= '0') > 9) i -= ('a' - ':');
						if ((n -= '0') > 9) n -= ('a' - ':');
						if ((i & ~0x0f) || (n & ~0x0f)) {
						   i = H_NOPKT;
						   break;
						}
						*q++ = (i << 4) | n;
					     }
					     else
						*q++ = *p;
					 }
					 if (p > rxbufptr)
					    c = H_NOPKT;
					 break;

				    case HCHR_ASCPKT:
					 n = i = 0;
					 for (p = q = rxbuf; p < rxbufptr; p++) {
					     i |= ((*p & 0x7f) << n);
					     if ((n += 7) >= 8) {
						*q++ = (byte) (i & 0xff);
						i >>= 8;
						n -= 8;
					     }
					 }
					 break;

				    case HCHR_UUEPKT:
					 n = (int) (rxbufptr - rxbuf);
					 for (p = q = rxbuf; n >= 4; n -= 4, p += 4) {
					     if (p[0] <= ' ' || p[0] >= 'a' ||
						 p[1] <= ' ' || p[1] >= 'a' ||
						 p[2] <= ' ' || p[2] >= 'a') {
						c = H_NOPKT;
						break;
					     }
					     *q++ = (byte) ((h_uudec(p[0]) << 2) | (h_uudec(p[1]) >> 4));
					     *q++ = (byte) ((h_uudec(p[1]) << 4) | (h_uudec(p[2]) >> 2));
					     *q++ = (byte) ((h_uudec(p[2]) << 6) | h_uudec(p[3]));
					 }
					 if (n >= 2) {
					    if (p[0] <= ' ' || p[0] >= 'a') {
					       c = H_NOPKT;
					       break;
					    }
					    *q++ = (byte) ((h_uudec(p[0]) << 2) | (h_uudec(p[1]) >> 4));
					    if (n == 3) {
					       if (p[0] <= ' ' || p[0] >= 'a') {
						  c = H_NOPKT;
						  break;
					       }
					       *q++ = (byte) ((h_uudec(p[1]) << 4) | (h_uudec(p[2]) >> 2));
					    }
					 }
					 break;

				    default:   /* This'd mean internal fluke */
					   debug(11," rx <PKTEND> (pktformat='%c' dec=%d hex=%02x) ??",
						rxpktformat, rxpktformat, rxpktformat);
					 c = H_NOPKT;
					 break;
			     }

			     rxbufptr = NULL;

			     if (c == H_NOPKT)
				break;

			     rxpktlen = (word) (q - rxbuf);
			     if (rxpktformat != HCHR_HEXPKT && (hrxoptions & HOPT_CRC32)) {
				if (rxpktlen < 5) {
				   c = H_NOPKT;
				   break;
				}
				n = h_crc32test(crc32block(rxbuf,rxpktlen));
				rxpktlen -= (int) sizeof (long);  /* remove CRC-32 */
			     }
			     else {
				if (rxpktlen < 3) {
				   c = H_NOPKT;
				   break;
				}
				n = h_crc16test(crc16block(rxbuf,rxpktlen));
				rxpktlen -= (int) sizeof (word);  /* remove CRC-16 */
			     }

			     rxpktlen--;		     /* remove type  */

			     if (n) {
		char *s1, *s2, *s3, *s4;

		debug(11," rx PKT (format='%c'	type='%c'  len=%d)",
			rxpktformat, (int) rxbuf[rxpktlen], rxpktlen);

	switch (rxbuf[rxpktlen]) {
	  case HPKT_START:	debug(11,"    START");
				break;
	  case HPKT_INIT:	s1 = ((char *) rxbuf) + ((int) strlen((char *) rxbuf)) + 1;
				s2 = s1 + ((int) strlen(s1)) + 1;
				s3 = s2 + ((int) strlen(s2)) + 1;
				s4 = s3 + ((int) strlen(s3)) + 1;
				debug(11,"    INIT (appinfo='%s'	can='%s'  want='%s'  options='%s'  pktprefix='%s')",
					(char *) rxbuf, s1, s2, s3, s4);
				break;
	  case HPKT_INITACK:  debug(11,"    INITACK");
			      break;
	  case HPKT_FINFO:    debug(11,"    FINFO ('%s'	 rxstate=%d)",rxbuf,rxstate);
			      break;
	  case HPKT_FINFOACK: debug(11,"    FINFOACK (pos=%ld  txstate=%d  txfd=%d)",
				      intell(h_long1(rxbuf)), txstate, txfd);
			      break;
	  case HPKT_DATA:     debug(11,"    DATA (rxstate=%d  pos=%ld  len=%u)",
				      rxstate, intell(h_long1(rxbuf)),
				      (word) (rxpktlen - ((int) sizeof (long))));
			      break;
	  case HPKT_DATAACK:  debug(11,"    DATAACK (rxstate=%d	 pos=%ld)",
				      rxstate, intell(h_long1(rxbuf)));
			      break;
	  case HPKT_RPOS:     debug(11,"    RPOS (pos=%ld%s  blklen=%u->%ld  syncid=%ld%s  txstate=%d  txfd=%d)",
				      intell(h_long1(rxbuf)),
				      intell(h_long1(rxbuf)) < 0L ? " SKIP" : "",
				      txblklen, intell(h_long2(rxbuf)),
				      intell(h_long3(rxbuf)),
				      intell(h_long3(rxbuf)) == rxsyncid ? " DUP" : "",
				      txstate, txfd);
			      break;
	  case HPKT_EOF:      debug(11,"    EOF (rxstate=%d  pos=%ld%s)",
				      rxstate, intell(h_long1(rxbuf)),
				      intell(h_long1(rxbuf)) < 0L ? " SKIP" : "");
			      break;
	  case HPKT_EOFACK:   debug(11,"    EOFACK (txstate=%d)", txstate);
			      break;
	  case HPKT_IDLE:     debug(11,"    IDLE");
			      break;
	  case HPKT_END:      debug(11,"    END");
			      break;
	  case HPKT_DEVDATA:  s1 = ((char *) rxbuf) + ((int) sizeof (long));
			      debug(11,"    DEVDATA (id=%ld  dev=%s  len=%u",
				      intell(h_long1(rxbuf)), s1,
				      rxpktlen - (((int) sizeof (long)) + ((int) strlen(s1)) + 1));
			      break;
	  case HPKT_DEVDACK:  debug(11,"    DEVDACK (devtxstate=%d  id=%ld)",
				      devtxstate, intell(h_long1(rxbuf)));
			      break;
	  default:	      debug(11,"    Unkown pkttype %d (txstate=%d  rxstate=%d)",
				      (int) rxbuf[rxpktlen], txstate, rxstate);
			      break;
   }
				return ((int) rxbuf[rxpktlen]);
			     }/*goodpkt*/

   debug(11," Bad CRC (format='%c'  type='%c'  len=%d)",
	     rxpktformat, (int) rxbuf[rxpktlen], rxpktlen);
			     break;

			case HCHR_BINPKT: 
			case HCHR_HEXPKT: 
			case HCHR_ASCPKT: 
			case HCHR_UUEPKT:

   debug(11," rx <PKTSTART> (pktformat='%c')",c);

			     rxpktformat = c;
			     p = rxbufptr = rxbuf;
			     rxdle = 0;
			     break;

			default:
			     if (p) {
				if (p < rxbufmax)
				   *p++ = (byte) (c ^ 0x40);
				else {

   debug(11," rx Pkt too long - discarded");

				   p = NULL;
				}
			     }
			     rxdle = 0;
			     break;
		 }
	      }
	      else if (p) {
		 if (p < rxbufmax)
		    *p++ = (byte) c;
		 else {

   debug(11," rx Pkt too long - discarded");

		    p = NULL;
		 }
	      }
	} /* while (com_getbyte()) */

	rxbufptr = p;

	if (h_timer_running(braindead) && h_timer_expired(braindead)) {

   debug(11," rx BrainDead (timer=%08lx	 time=%08lx)",
	   braindead,time(NULL));
	   return (H_BRAINTIME);
	}
	if (h_timer_running(txtimer) && h_timer_expired(txtimer)) {

   debug(11," rx TxTimer (timer=%08lx  time=%08lx)",
	   txtimer,time(NULL));
	   return (H_TXTIME);
	}
	if (h_timer_running(devtxtimer) && h_timer_expired(devtxtimer)) {

   debug(11," rx DevTxTimer (timer=%08lx  time=%08lx)",
	   devtxtimer,time(NULL));
	   return (H_DEVTXTIME);
	}

/*	  sys_idle(); */
	return (H_NOPKT);
}/*rxpkt()*/

#ifdef HAS_SELECT
/*---------------------------------------------------------------------------*/
/*
 *      ,   
 * -.       
 *   ,      .
 */

	word	txlen;
	int	txtype;

static int txrxpkt ()
{
	fd_set		readfds, writefds, exceptfds;
	struct timeval	seltimer;
	int		rc;

	if (hanged_up)
		return (H_CARRIER);

	FD_ZERO(&readfds);
	FD_ZERO(&writefds);
	FD_ZERO(&exceptfds);
	FD_SET(0,&readfds);
	FD_SET(1,&writefds);
	FD_SET(0,&exceptfds);
	FD_SET(1,&exceptfds);
	seltimer.tv_sec = 10;
	seltimer.tv_usec = 0;

	rc = select(2, &readfds, &writefds, &exceptfds, &seltimer);

	if (rc < 0)
	{
		if (hanged_up)
			return (H_CARRIER);
		else
			logerr("$select for HYDRA failed");
		return (H_NOPKT);
	}
	else if (rc == 0)
	{
		logerr("$select HYDRA timeout");
		return (H_NOPKT);
	}
	else /* rc > 0 */
	{
		if ((FD_ISSET(0, &exceptfds)) || (FD_ISSET(1, &exceptfds)))
		{
			logerr("$select STAT_ERROR");
			return (H_NOPKT);
		}
	}
/* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
 *       .
 *
 * + txtype!    ,   .
 * BUG:
 *     -  .
 */
	if ((FD_ISSET(1, &writefds)) && txtype)
	{
/*
 *  .
 */
		txpkt(txlen, txtype);
		txtype = 0;
	}/*writefds*/

	if (FD_ISSET(0, &readfds))
	{
/*
 *     .
 */
		rc = rxpkt();
		return (rc);
	}/*readfds*/
/*
 *   
 */
	return (H_NOPKT);
}/*txrxpkt()*/
#endif /* HAS_SELECT */

/*---------------------------------------------------------------------------*/
static void hydra_status (boolean xmit)
{
	long pos    = xmit ? txpos    : rxpos,
	     fsize  = xmit ? txfsize  : rxfsize;

	if (pos >= fsize)
	   debug(11,"%ld/%ld (EOF)",pos,fsize);
	else {
	   int left = (int) ((((fsize - pos) / 128L) * 1340L) / cur_speed);
	   char *p = "";

	   if (xmit) {
	      if      (txstate == HTX_DATAACK) p = "ACK ";
	      else if (txstate == HTX_XWAIT)   p = "WAIT ";
	   }
	   debug(11,"%ld/%ld (%s%d:%02d min left)",
			pos, fsize, p, left / 60, left % 60);
	}
}/*hydra_status()*/

/*---------------------------------------------------------------------------*/
static void hydra_pct (boolean xmit)
{
	long offset = xmit ? txoffset : rxoffset,
	     fsize  = xmit ? txfsize  : rxfsize,
	     start  = xmit ? txstart  : rxstart,
	     elapsed, bytes, cps, pct;

	elapsed = time(NULL) - start;
	bytes = fsize - offset;
	if (bytes < 1024L || elapsed == 0L)
	   return;
	cps = bytes / elapsed;
	pct = (cps * 1000L) / ((long) cur_speed);
	loginf("+%s-H CPS: %ld (%ld bytes), %d:%02d min.  Eff: %ld%%",
		xmit ? "Sent" : "Rcvd", cps, bytes,
		(int) (elapsed / 60), (int) (elapsed % 60), pct);
}/*hydra_pct()*/

/*---------------------------------------------------------------------------*/
void hydra_badxfer (void)
{
	if (rxfd >= 0) {
	   close(rxfd);		/* close file */
	   rxfd = -1;
	   if (xfer_bad())
	      debug(11,"+HRECV: Bad xfer recovery-info saved");
	   else
	      debug(11,"-HRECV: Bad xfer - file deleted");
	}
}/*hydra_badxfer()*/

/*---------------------------------------------------------------------------*/
void hydra_init (dword want_options)
{
	register word i, j;
	word  crc16;
	dword crc32;

	txbuf		= (byte *)  malloc(H_BUFLEN);
	rxbuf		= (byte *)  malloc(H_BUFLEN);
	crc16tab	= (word *)  malloc(256 * ((int) sizeof (word)));
	crc32tab	= (dword *) malloc(256 * ((int) sizeof (dword)));
	if (!txbuf || !rxbuf || !crc16tab || !crc32tab) {
	 debug(2,"!HYDRA: Can't allocate buffers!");
	 endprog(2);
	}
	txbufin	 = txbuf + ((H_MAXBLKLEN + H_OVERHEAD + 5) * 2);
	rxbufmax = rxbuf + H_MAXPKTLEN;

	for (i = 0; i < 256; i++) {
	 crc16 = i;
	 crc32 = i;
	 for (j = 8; j > 0; j--) {
	  if (crc16 & 1) crc16 = (crc16 >> 1) ^ h_crc16poly;
	  else		crc16 >>= 1;
	   if (crc32 & 1) crc32 = (crc32 >> 1) ^ h_crc32poly;
	   else		crc32 >>= 1;
	 }
	 crc16tab[i] = crc16;
	 crc32tab[i] = crc32;
	}

	batchesdone = 0;

	originator = nooriginator ? false : true;

	if (originator)
		hdxlink = false;
	else if (hdxsession)
		hdxlink = true;

	hydra_options = (want_options & HCAN_OPTIONS) & ~HUNN_OPTIONS;

	timeout = (word) (40960L / cur_speed);
	if	(timeout < H_MINTIMER) timeout = H_MINTIMER;
	else if (timeout > H_MAXTIMER) timeout = H_MAXTIMER;

	txmaxblklen = (cur_speed / 300) * 128;
	if	(txmaxblklen < 256)	    txmaxblklen = 256;
	else if (txmaxblklen > H_MAXBLKLEN) txmaxblklen = H_MAXBLKLEN;

	rxblklen = txblklen = (cur_speed < 2400U) ? 256 : 512;

	txgoodbytes  = 0;
	txgoodneeded = 1024;

	txstate = HTX_DONE;

	hydra_devfunc("CON",rem_chat);

	chatfill  = 0;
	chattimer = -1L;
	lasttimer = 0L;

	debug(1,"+HYDRA session (%s-directional mode)",
		hdxlink ? "Uni" : "Bi");
}/*hydra_init()*/

/*---------------------------------------------------------------------------*/
void hydra_deinit (void)
{
	free(txbuf);
	free(rxbuf);
	free(crc16tab);
	free(crc32tab);
}/*hydra_deinit()*/

/*---------------------------------------------------------------------------*/
int hydra (char *txpathname, char *txalias)
{
	int   res=0;
	int   pkttype;
	char *p=NULL, *q;
	int   i;
	struct stat f;

	/*-------------------------------------------------------------------*/
	if (txstate == HTX_DONE) {
/*
 *  .
 *  .
 */
		txstate = HTX_START;
		debug(11,"*HYDRA txInit");
		htxoptions = HTXI_OPTIONS;
		txpktprefix[0] = '\0';
/*
 *   .
 */
		rxstate = HRX_INIT;
		debug(11,"*HYDRA rxInit");
		hrxoptions = HRXI_OPTIONS;
		rxfd = -1;
		rxdle = 0;
		rxbufptr = NULL;
		rxtimer   = h_timer_reset();

		devtxid    = devrxid = 0L;
		devtxtimer = h_timer_reset();
		devtxstate = HTD_DONE;

		braindead = h_timer_set(H_BRAINDEAD);
	}
	else
		txstate = HTX_FINFO;

	txtimer	  = h_timer_reset();
	txretries = 0;

	/*-------------------------------------------------------------------*/
	if (txpathname) {
		stat(txpathname,&f);
		txfsize = f.st_size;
		txftime = f.st_mtime;

		if ((txfd = dos_open(txpathname,0)) < 0) {
			debug(3,"-HSEND: Unable to open %s",txpathname);
			return (XFER_SKIP);
		}
/*
 *    .
 */
		for (p = txpathname, q = txfname; *p; p++) {
			if (*q = *p, *p == '/' || *p == ':' || *p == '\\')
				q = txfname;
			else q++;
		}
		*q = '\0';

/*	     if (txalias)
		strupr(txalias); */
		txstart  = 0L;
		txsyncid = 0L;
	}
	else {
		txfd = -1;
		strcpy(txfname,"");
	}

	/*-------------------------------------------------------------------*/
	do {
	   /*----------------------------------------------------------------*/
	   switch (devtxstate) {
		  /*---------------------------------------------------------*/
		  case HTD_DATA:
		       if (txstate > HTX_RINIT) {
			  h_long1(txbufin) = intell(devtxid);
			  p = ((char *) txbufin) + ((int) sizeof(long));
			  strcpy(p,devtxdev);
			  p += H_FLAGLEN + 1;
			  memcpy(p,devtxbuf,devtxlen);
/* #ifdef HAS_SELECT
			  txlen = ((int) sizeof (long)) + H_FLAGLEN + 1 + devtxlen;
			  txtype = HPKT_DEVDATA;
#else */
			  txpkt(((int) sizeof (long)) + H_FLAGLEN + 1 + devtxlen,HPKT_DEVDATA);
/* #endif */
			  devtxtimer = h_timer_set((!rxstate && txstate == HTX_REND) ? timeout / 2 : timeout);
			  devtxstate = HTD_DACK;
		       }
		       break;

		  /*---------------------------------------------------------*/
		  default:
		       break;

		  /*---------------------------------------------------------*/
	   }

	   /*----------------------------------------------------------------*/
	   switch (txstate) {
		  /*---------------------------------------------------------*/
		  case HTX_START:
		       com_putblock((byte *) autostr,(int) strlen(autostr));
#ifdef HAS_SELECT
			txlen = 0;
			txtype = HPKT_START;
#else
		       txpkt(0,HPKT_START);
#endif
		       txtimer = h_timer_set(H_START);
		       txstate = HTX_SWAIT;
		       break;

		  /*---------------------------------------------------------*/
		  case HTX_INIT:
		       p = (char *) txbufin;
			sprintf(p,"%08lx%s %s %X",
			H_REVSTAMP, sysop, "ifcico v3.0", PRODCODE);
		       p += ((int) strlen(p)) + 1;/* our app info & HYDRA rev. */
		       put_flags(p,h_flags,HCAN_OPTIONS);    /* what we CAN  */
		       p += ((int) strlen(p)) + 1;
		       put_flags(p,h_flags, hydra_options);	/* what we WANT */
		       p += ((int) strlen(p)) + 1;
		       sprintf(p,"%08lx%08lx",		     /* TxRx windows */
				hydra_txwindow, hydra_rxwindow);
		       p += ((int) strlen(p)) + 1;
		       strcpy(p,pktprefix);	/* pkt prefix string we want */
		       p += ((int) strlen(p)) + 1;

		       htxoptions = HTXI_OPTIONS;
#ifdef HAS_SELECT
			txlen = (word) (((byte *) p) - txbufin);
			txtype = HPKT_INIT;
#else
		       txpkt((word) (((byte *) p) - txbufin), HPKT_INIT);
#endif
		       htxoptions = hrxoptions;
		       txtimer = h_timer_set(timeout / 2);
		       txstate = HTX_INITACK;
		       break;

		  /*---------------------------------------------------------*/
		  case HTX_FINFO:
		       if (txfd >= 0) {
			  if (!txretries) {
			     debug (11, "%s", txfname);
			     if (txalias) {
				debug (11,"  -> %s",txalias);
			     }
			     debug(1,"+Send-H: %s%s%s (%ldb), %d min.",
				     txpathname, txalias ? " -> " : "", txalias ? txalias : "",
				     txfsize, (int) (txfsize * 10L / cur_speed + 27L) / 54L);
			     loginf("+Send-H: %s (%ldb), %d min.",
				     txalias ? txalias : txfname,
				     txfsize, (int) (txfsize * 10L / cur_speed + 27L) / 54L);

/*			       strlwr(txfname);*/
			  }
			  sprintf((char *) txbufin,"%08lx%08lx%08lx%08lx%08lx%s",
				  txftime, txfsize, 0L, 0L, 0L,
				  txalias ? txalias : txfname);
		       }
		       else {
			  if (!txretries) {
			     debug(11,"+HSEND: End of batch");
			  }
			  strcpy((char *) txbufin,txfname);
		       }
#ifdef HAS_SELECT
			txlen = ((int) strlen((char *) txbufin)) + 1;
			txtype = HPKT_FINFO;
#else
		       txpkt(((int) strlen((char *) txbufin)) + 1,HPKT_FINFO);
#endif
		       txtimer = h_timer_set(txretries ? timeout / 2 : timeout);
		       txstate = HTX_FINFOACK;
		       break;

		  /*---------------------------------------------------------*/
		  case HTX_XDATA:
		       if (com_outfull() > txmaxblklen)
			  break;

		       if (txpos < 0L)
			  i = -1;				     /* Skip */
		       else {
			  h_long1(txbufin) = intell(txpos);
			  if ((i = read(txfd, txbufin + ((int) sizeof (long)), txblklen)) < 0) {
			     debug(5,"!HSEND: File read error");
			     close(txfd);	/* close file */
			     txfd = -1;
			     txpos = -2L;			     /* Skip */
			  }
		       }

		       if (i > 0) {
			  txpos += i;
#ifdef HAS_SELECT
				txlen = ((int) sizeof (long)) + i;
				txtype = HPKT_DATA;
#else
			  txpkt(((int) sizeof (long)) + i, HPKT_DATA);
#endif
			  if (txblklen < txmaxblklen &&
			      (txgoodbytes += i) >= txgoodneeded) {
			     txblklen <<= 1;
			     if (txblklen >= txmaxblklen) {
				txblklen = txmaxblklen;
				txgoodneeded = 0;
			     }
			     txgoodbytes = 0;
			  }

			  if (htxwindow && (txpos >= (txlastack + htxwindow))) {
			     txtimer = h_timer_set(txretries ? timeout / 2 : timeout);
			     txstate = HTX_DATAACK;
			  }

			  if (!txstart)
			     txstart = time(NULL);
			  hydra_status(true);
			  break;
		       }

		       /* fallthrough to HTX_EOF */

		  /*---------------------------------------------------------*/
		  case HTX_EOF:
		       h_long1(txbufin) = intell(txpos);
#ifdef HAS_SELECT
			txlen = (int) sizeof (long);
			txtype = HPKT_EOF;
#else
		       txpkt((int) sizeof (long),HPKT_EOF);
#endif
		       txtimer = h_timer_set(txretries ? timeout / 2 : timeout);
		       txstate = HTX_EOFACK;
		       break;

		  /*---------------------------------------------------------*/
		  case HTX_END:
/*     ??? */
#ifdef HAS_SELECT
			txlen = 0;
			txtype = HPKT_END;
#else
		       txpkt(0,HPKT_END);
		       txpkt(0,HPKT_END);
#endif
		       txtimer = h_timer_set(timeout / 2);
		       txstate = HTX_ENDACK;
		       break;

		  /*---------------------------------------------------------*/
		  default:
		       break;

		  /*---------------------------------------------------------*/
	   } /* switch (txstate) */
/*
 *   .
 */
	   /*----------------------------------------------------------------*/
#ifdef HAS_SELECT
	   while (txstate && (pkttype = txrxpkt()) != H_NOPKT) {
#else
	   while (txstate && (pkttype = rxpkt()) != H_NOPKT) {
#endif
		 /*----------------------------------------------------------*/
		 switch (pkttype) {
			/*---------------------------------------------------*/
			case H_CARRIER:
			case H_CANCEL:
			case H_SYSABORT:
			case H_BRAINTIME:
			     switch (pkttype) {
				    case H_CARRIER:   p = "Carrier lost";	   break;
				    case H_CANCEL:    p = "Aborted by other side"; break;
				    case H_SYSABORT:  p = "Aborted by operator";   break;
				    case H_BRAINTIME: p = "Other end died";	   break;
			     }
			     debug(11,"-HYDRA: %s",p);
			     txstate = HTX_DONE;
			     res = XFER_ABORT;
			     break;

			/*---------------------------------------------------*/
			case H_TXTIME:
			     if (txstate == HTX_XWAIT || txstate == HTX_REND) {
#ifdef HAS_SELECT
				txlen = 0;
				txtype = HPKT_IDLE;
#else
				txpkt(0,HPKT_IDLE);
#endif
				txtimer = h_timer_set(H_IDLE);
				break;
			     }

			     if (++txretries > H_RETRIES) {
				debug(11,"-HSEND: Too many errors");
				txstate = HTX_DONE;
				res = XFER_ABORT;
				break;
			     }

			     debug(11,"-HSEND: Timeout - Retry %u",txretries);

			     txtimer = h_timer_reset();
/*
 * Timeout - retry.
 */
			     switch (txstate) {
				    case HTX_SWAIT:    txstate = HTX_START; break;
				    case HTX_INITACK:  txstate = HTX_INIT;  break;
				    case HTX_FINFOACK: txstate = HTX_FINFO; break;
				    case HTX_DATAACK:  txstate = HTX_XDATA; break;
				    case HTX_EOFACK:   txstate = HTX_EOF;   break;
				    case HTX_ENDACK:   txstate = HTX_END;   break;
			     }
			     break;

			/*---------------------------------------------------*/
			case H_DEVTXTIME:
			     if (++devtxretries > H_RETRIES) {
				debug(11,"-HDEVTX: Too many errors");
				txstate = HTX_DONE;
				res = XFER_ABORT;
				break;
			     }

			     debug(11,"-HDEVTX: Timeout - Retry %u",devtxretries);

			     devtxtimer = h_timer_reset();
			     devtxstate = HTD_DATA;
			     break;

			/*---------------------------------------------------*/
			case HPKT_START:
			     if (txstate == HTX_START || txstate == HTX_SWAIT) {
				txtimer = h_timer_reset();
				txretries = 0;
				txstate = HTX_INIT;
				braindead = h_timer_set(H_BRAINDEAD);
			     }
			     break;

			/*---------------------------------------------------*/
			case HPKT_INIT:
			     if (rxstate == HRX_INIT) {
				p = (char *) rxbuf;
				p += ((int) strlen(p)) + 1;
				q = p + ((int) strlen(p)) + 1;
				hrxoptions  = hydra_options | HUNN_OPTIONS;
				hrxoptions |= get_flags(q,h_flags);
				hrxoptions &= get_flags(p,h_flags);
				hrxoptions &= HCAN_OPTIONS;
				if (hrxoptions < (hydra_options & HNEC_OPTIONS)) {
				   debug(5,"!HYDRA: Incompatible on this link");
				   txstate = HTX_DONE;
				   res = XFER_ABORT;
				   break;
				}
				p = q + ((int) strlen(q)) + 1;
				hrxwindow = htxwindow = 0L;
				sscanf(p,"%08lx%08lx", &hrxwindow, &htxwindow);
				if (hrxwindow < 0L) hrxwindow = 0L;
				if (hydra_rxwindow &&
				    (!hrxwindow || hydra_rxwindow < hrxwindow))
				   hrxwindow = hydra_rxwindow;
				if (htxwindow < 0L) htxwindow = 0L;
				if (hydra_txwindow &&
				    (!htxwindow || hydra_txwindow < htxwindow))
				   htxwindow = hydra_txwindow;
				p += ((int) strlen(p)) + 1;
				strncpy(txpktprefix,p,H_PKTPREFIX);
				txpktprefix[H_PKTPREFIX] = '\0';

				if (!batchesdone) {
				   long revstamp;

				   p = (char *) rxbuf;
				   sscanf(p,"%08lx",&revstamp);
				   debug(11,"*HYDRA: Other's HydraRev=%s",
					   h_revdate(revstamp));
				   p += 8;
				   if ((q = strchr(p,',')) != NULL) *q = ' ';
				   if ((q = strchr(p,',')) != NULL) *q = '/';
				   debug(11,"*HYDRA: Other's App.Info '%s'",p);
				   put_flags((char *) rxbuf,h_flags, hrxoptions);
				   debug(11,"*HYDRA: Using link options '%s'",rxbuf);
				   if (htxwindow || hrxwindow)
				      debug(11,"*HYDRA: Window tx=%ld rx=%ld",
						htxwindow, hrxwindow);
				}

				chattimer = (hrxoptions & HOPT_DEVICE) ? 0L : -2L;

				htxoptions = hrxoptions;
				rxstate = HRX_FINFO;
				
				if (hrxoptions & HOPT_DEVICE) {
				   p = "Remote has no chat facility available\r\n";
				   hydra_devsend ("CON", (byte *)p, (int)strlen (p));
				}
			     }

#ifdef HAS_SELECT
				txlen = 0;
				txtype = HPKT_INITACK;
#else
			     txpkt(0,HPKT_INITACK);
#endif
			     break;

			/*---------------------------------------------------*/
			case HPKT_INITACK:
			     if (txstate == HTX_INIT || txstate == HTX_INITACK) {
				braindead = h_timer_set(H_BRAINDEAD);
				txtimer = h_timer_reset();
				txretries = 0;
				txstate = HTX_RINIT;
			     }
			     break;

			/*---------------------------------------------------*/
			case HPKT_FINFO:
			     if (rxstate == HRX_FINFO) {
				braindead = h_timer_set(H_BRAINDEAD);
				if (!rxbuf[0]) {
				   debug(11,"*HRECV: End of batch");
				   rxpos = 0L;
				   rxstate = HRX_DONE;
				   batchesdone++;
				}
				else {
				   long diskfree;

				   rxfsize = rxftime = 0L;
				   rxfname[0] = '\0';
				   sscanf((char *) rxbuf,"%08lx%08lx%*08lx%*08lx%*08lx%s",
					  &rxftime, &rxfsize, rxfname);
/*				     strupr(rxfname);*/
				   debug(11,"Recv-H \"%s\"",rxfname);
				   rxpathname = xfer_init(rxfname,rxfsize,rxftime);
				   diskfree = 10000000;/*freespace(rxpathname);*/
				   if (!rxpathname) {	/* Already have file */
				      if (single_done) {
					 debug(11,"+HRECV: Skipping additional files (file %s)",rxfname);
					 rxpos = -2L;
				      }
				      else {
					 debug(11,"+HRECV: Already have %s",rxfname);
					 rxpos = -1L;
				      }
				   }
				   else if (rxfsize + 10240L > diskfree) {
				      debug(0,"!HRECV: %s not enough diskspace: %ld > %ld",
					      rxfname, rxfsize + 10240L, diskfree);
				      rxpos = -2L;
				   }
				   else {
				      if (fexist(rxpathname)) { /* Resuming? */
					 if ((rxfd = dos_open(rxpathname,1)) < 0) {
					    debug(3,"!HRECV: Unable to re-open %s",rxpathname);
					    rxpos = -2L;
					 }
				      }
				      else if ((rxfd = dos_open(rxpathname,1)) < 0) {
					 debug(3,"!HRECV: Unable to create %s",rxpathname);
					 rxpos = -2L;
				      }

				      if (rxfd >= 0) {
					 debug(1,"+Recv-H: %s (%ldb), %d min.",
						 rxfname, rxfsize,
						 (int) (rxfsize * 10L / cur_speed + 27L) / 54L);
					 loginf("+Recv-H: %s (%ldb), %d min.",
						 rxfname, rxfsize,
						 (int) (rxfsize * 10L / cur_speed + 27L) / 54L);
					 if (lseek(rxfd, 0L, SEEK_END) < 0L) {
					    debug(2,"!HRECV: File seek error");
					    hydra_badxfer();
					    rxpos = -2L;
					 }
					 else {
					    rxoffset = rxpos = lseek(rxfd, 0L, SEEK_CUR);
					    if (rxpos < 0L) {
					       debug(2,"!HRECV: File tell error");
					       hydra_badxfer();
					       rxpos = -2L;
					    }
					    else {
					       rxstart = 0L;
					       rxtimer = h_timer_reset();
					       rxretries = 0;
					       rxlastsync = 0L;
					       rxsyncid = 0L;
					       hydra_status(false);
					       if (rxpos > 0L) {
						  debug(1,"+HRECV: Resuming from offset %ld (%d min. to go)",
							  rxpos, (int) ((rxfsize - rxoffset) * 10L / cur_speed + 27L) / 54L);
						  loginf("+Recv-H: Resuming from offset %ld (%d min. to go)",
							  rxpos, (int) ((rxfsize - rxoffset) * 10L / cur_speed + 27L) / 54L);
					       }
					       rxstate = HRX_DATA;
					    }
					 }
				      }
				   }
				}
			     }
			     else if (rxstate == HRX_DONE)
				rxpos = (!rxbuf[0]) ? 0L : -2L;

			     h_long1(txbufin) = intell(rxpos);
#ifdef HAS_SELECT
				txlen = (int) sizeof (long);
				txtype = HPKT_FINFOACK;
#else
			     txpkt((int) sizeof (long),HPKT_FINFOACK);
#endif
			     break;

			/*---------------------------------------------------*/
			case HPKT_FINFOACK:
			     if (txstate == HTX_FINFO || txstate == HTX_FINFOACK) {
				braindead = h_timer_set(H_BRAINDEAD);
				txretries = 0;
				if (!txfname[0]) {
				   txtimer = h_timer_set(H_IDLE);
				   txstate = HTX_REND;
				}
				else {
				   txtimer = h_timer_reset();
				   txpos = intell(h_long1(rxbuf));
				   if (txpos >= 0L) {
				      txoffset = txpos;
				      txlastack = txpos;
				      hydra_status(true);
				      if (txpos > 0L) {
					 debug(1,"+HSEND: Transmitting from offset %ld (%d min. to go)",
						 txpos, (int) ((txfsize - txoffset) * 10L / cur_speed + 27L) / 54L);
					 loginf("+HSEND: Transmitting from offset %ld (%d min. to go)",
						 txpos, (int) ((txfsize - txoffset) * 10L / cur_speed + 27L) / 54L);
					 if (lseek(txfd, txpos, SEEK_SET) < 0L) {
					    debug(2,"!HSEND: File seek error");
					    close(txfd);	/* close file */
					    txfd = -1;
					    txpos = -2L;
					    txstate = HTX_EOF;
					    break;
					 }
				      }
				      txstate = HTX_XDATA;
				   }
				   else {
				      close(txfd);	/* close file */
				      if (txpos == -1L) {
					 debug(1,"+HSEND: They already have %s",txfname);
					 return (XFER_OK);
				      }
				      else {  /* (txpos < -1L) file NOT sent */
					 debug(1,"+HSEND: Skipping %s",txfname);
					 return (XFER_SKIP);
				      }
				   }
				}
			     }
			     break;

			/*---------------------------------------------------*/
			case HPKT_DATA:
			     if (rxstate == HRX_DATA) {
				if (intell(h_long1(rxbuf)) != rxpos ||
				    intell(h_long1(rxbuf)) < 0L) {
				   if (intell(h_long1(rxbuf)) <= rxlastsync) {
				      rxtimer = h_timer_reset();
				      rxretries = 0;
				   }
				   rxlastsync = intell(h_long1(rxbuf));

				   if (!h_timer_running(rxtimer) ||
				       h_timer_expired(rxtimer)) {
				      if (rxretries > 4) {
					 if (txstate < HTX_REND &&
					     !originator && !hdxlink) {
					    hdxlink = true;
					    rxretries = 0;
					 }
				      }
				      if (++rxretries > H_RETRIES) {
					 debug(11,"-HRECV: Too many errors");
					 txstate = HTX_DONE;
					 res = XFER_ABORT;
					 break;
				      }
				      if (rxretries == 1)
					 rxsyncid++;

				      rxblklen /= 2;
				      i = rxblklen;
				      if      (i <=  64) i =   64;
				      else if (i <= 128) i =  128;
				      else if (i <= 256) i =  256;
				      else if (i <= 512) i =  512;
				      else		 i = 1024;
				      debug(11,"-HRECV: Bad pkt at %ld - Retry %u (newblklen=%u)",
					      rxpos,rxretries,i);
				      h_long1(txbufin) = intell(rxpos);
				      h_long2(txbufin) = intell((long) i);
				      h_long3(txbufin) = intell(rxsyncid);
#ifdef HAS_SELECT
					txlen = 3 * ((int) sizeof(long));
					txtype = HPKT_RPOS;
#else
				      txpkt(3 * ((int) sizeof(long)),HPKT_RPOS);
#endif
				      rxtimer = h_timer_set(timeout);
				   }
				}
				else {
				   braindead = h_timer_set(H_BRAINDEAD);
				   rxpktlen -= (int) sizeof (long);
				   rxblklen = rxpktlen;
				   if (write(rxfd, rxbuf + ((int) sizeof (long)), rxpktlen) < 0) {
				      debug(3,"!HRECV: File write error");
				      hydra_badxfer();
				      rxpos = -2L;
				      rxretries = 1;
				      rxsyncid++;
				      h_long1(txbufin) = intell(rxpos);
				      h_long2(txbufin) = intell(0L);
				      h_long3(txbufin) = intell(rxsyncid);
#ifdef HAS_SELECT
					txlen = 3 * ((int) sizeof(long));
					txtype = HPKT_RPOS;
#else
				      txpkt(3 * ((int) sizeof(long)),HPKT_RPOS);
#endif
				      rxtimer = h_timer_set(timeout);
				      break;
				   }
				   rxretries = 0;
				   rxtimer = h_timer_reset();
				   rxlastsync = rxpos;
				   rxpos += rxpktlen;
				   if (hrxwindow) {
				      h_long1(txbufin) = intell(rxpos);
#ifdef HAS_SELECT
					txlen = (int) sizeof(long);
					txtype = HPKT_DATAACK;
#else
				      txpkt((int) sizeof(long),HPKT_DATAACK);
#endif
				   }
				   if (!rxstart)
				      rxstart = time(NULL) -
						((rxpktlen * 10) / cur_speed);
				   hydra_status(false);
				}/*badpkt*/
			     }/*rxstate==HRX_DATA*/
			     break;

			/*---------------------------------------------------*/
			case HPKT_DATAACK:
			     if (txstate == HTX_XDATA || txstate == HTX_DATAACK ||
				 txstate == HTX_XWAIT ||
				 txstate == HTX_EOF || txstate == HTX_EOFACK) {
				if (htxwindow && intell(h_long1(rxbuf)) > txlastack) {
				   txlastack = intell(h_long1(rxbuf));
				   if (txstate == HTX_DATAACK &&
				       (txpos < (txlastack + htxwindow))) {
				      txstate = HTX_XDATA;
				      txretries = 0;
				      txtimer = h_timer_reset();
				   }
				}
			     }
			     break;

			/*---------------------------------------------------*/
			case HPKT_RPOS:
			     if (txstate == HTX_XDATA || txstate == HTX_DATAACK ||
				 txstate == HTX_XWAIT ||
				 txstate == HTX_EOF || txstate == HTX_EOFACK) {
				if (intell(h_long3(rxbuf)) != txsyncid) {
				   txsyncid = intell(h_long3(rxbuf));
				   txretries = 1;
				   txtimer = h_timer_reset();
				   txpos = intell(h_long1(rxbuf));
				   if (txpos < 0L || txfsize == txpos) {
				      if (txfd >= 0) {
					 debug(11,"+HSEND: Skipping %s",txfname);
					 close(txfd);	/* close file */
					 txfd = -1;
					 txstate = HTX_EOF;
				      }
				      txpos = -2L;
				      break;
				   }

				   if (txblklen > intell(h_long2(rxbuf)))
				      txblklen = (word) intell(h_long2(rxbuf));
				   else
				      txblklen >>= 1;
				   if	   (txblklen <=	 64) txblklen =	  64;
				   else if (txblklen <= 128) txblklen =	 128;
				   else if (txblklen <= 256) txblklen =	 256;
				   else if (txblklen <= 512) txblklen =	 512;
				   else			     txblklen = 1024;
				   txgoodbytes = 0;
				   txgoodneeded += 1024;
				   if (txgoodneeded > 8192)
				      txgoodneeded = 8192;
/*				   txgoodneeded += txmaxblklen * 2;
				   if (txgoodneeded > txmaxblklen * 8)
				   txgoodneeded = txmaxblklen * 8;
*/
				   hydra_status(true);
				   debug(1,"+HSEND: Resending from offset %ld (newblklen=%u)",
					   txpos,txblklen);
				   if (lseek(txfd, txpos, SEEK_SET) < 0L) {
				      debug(3,"!HSEND: File seek error");
				      close(txfd);	/* close file */
				      txfd = -1;
				      txpos = -2L;
				      txstate = HTX_EOF;
				      break;
				   }

				   if (txstate != HTX_XWAIT)
				      txstate = HTX_XDATA;
				}
				else {
				   if (++txretries > H_RETRIES) {
				      debug(11,"-HSEND: Too many errors");
				      txstate = HTX_DONE;
				      res = XFER_ABORT;
				      break;
				   }
				}
			     }
			     break;

			/*---------------------------------------------------*/
			case HPKT_EOF:
			     if (rxstate == HRX_DATA) {
				if (intell(h_long1(rxbuf)) < 0L) {
				   hydra_badxfer();
				   debug(1,"+HRECV: Skipping %s",rxfname);
				   rxstate = HRX_FINFO;
				   braindead = h_timer_set(H_BRAINDEAD);
				}
				else if (intell(h_long1(rxbuf)) != rxpos) {
				   if (intell(h_long1(rxbuf)) <= rxlastsync) {
				      rxtimer = h_timer_reset();
				      rxretries = 0;
				   }
				   rxlastsync = intell(h_long1(rxbuf));

				   if (!h_timer_running(rxtimer) ||
				       h_timer_expired(rxtimer)) {
				      if (++rxretries > H_RETRIES) {
					 debug(11,"-HRECV: Too many errors");
					 txstate = HTX_DONE;
					 res = XFER_ABORT;
					 break;
				      }
				      if (rxretries == 1)
					 rxsyncid++;

				      rxblklen /= 2;
				      i = rxblklen;
				      if      (i <=  64) i =   64;
				      else if (i <= 128) i =  128;
				      else if (i <= 256) i =  256;
				      else if (i <= 512) i =  512;
				      else		 i = 1024;
				      debug(11,"-HRECV: Bad EOF at %ld - Retry %u (newblklen=%u)",
					      rxpos,rxretries,i);
				      h_long1(txbufin) = intell(rxpos);
				      h_long2(txbufin) = intell((long) i);
				      h_long3(txbufin) = intell(rxsyncid);
#ifdef HAS_SELECT
					txlen = 3 * ((int) sizeof(long));
					txtype = HPKT_RPOS;
#else
				      txpkt(3 * ((int) sizeof(long)),HPKT_RPOS);
#endif
				      rxtimer = h_timer_set(timeout);
				   }
				}
				else {
				   rxfsize = rxpos;
				   close(rxfd);	/* close file */
				   rxfd = -1;
				   hydra_pct(false);

				   p = xfer_okay();
				   if (p) {
				      debug(1,"+HRECV: Dup file renamed: %s",p);
				   }

				   hydra_status(false);
				   debug(1,"+Rcvd-H %s",p ? p : rxfname);
				   rxstate = HRX_FINFO;
				   braindead = h_timer_set(H_BRAINDEAD);
				}/*skip/badeof/eof*/
			     }/*rxstate==HRX_DATA*/

			     if (rxstate == HRX_FINFO)
#ifdef HAS_SELECT
				txlen = 0;
				txtype = HPKT_EOFACK;
#else
				txpkt(0,HPKT_EOFACK);
#endif
			     break;

			/*---------------------------------------------------*/
			case HPKT_EOFACK:
			     if (txstate == HTX_EOF || txstate == HTX_EOFACK) {
				braindead = h_timer_set(H_BRAINDEAD);
				if (txfd >= 0) {
				   txfsize = txpos;
				   close(txfd);	/* close file */
				   hydra_pct(true);
				   return (XFER_OK);
				}
				else
				   return (XFER_SKIP);
			     }
			     break;

			/*---------------------------------------------------*/
			case HPKT_IDLE:
			     if (txstate == HTX_XWAIT) {
				hdxlink = false;
				txtimer = h_timer_reset();
				txretries = 0;
				txstate = HTX_XDATA;
			     }
			     else if (txstate >= HTX_FINFO && txstate < HTX_REND)
				braindead = h_timer_set(H_BRAINDEAD);
			     break;

			/*---------------------------------------------------*/
			case HPKT_END:
			     /* special for chat, other side wants to quit */
			     if (chattimer > 0L && txstate == HTX_REND) {
				chattimer = -3L;
				break;
			     }

			     if (txstate == HTX_END || txstate == HTX_ENDACK) {
/*
 * End of sesion - exit.
 */
/* !!!!!!!!!!!!!! BUG ? !!!!!!!!!!!!!!!!!!!!!!!!! */
/*
#ifdef HAS_SELECT
				txlen = 0;
				txtype = HPKT_END;
				(void)txrxpkt();
#else */
				txpkt(0,HPKT_END);
				txpkt(0,HPKT_END);
				txpkt(0,HPKT_END);
/* #endif */
				debug(1,"+HYDRA: Completed");
				txstate = HTX_DONE;
				res = XFER_OK;
			     }
			     break;

			/*---------------------------------------------------*/
			case HPKT_DEVDATA:
			     if (devrxid != intell(h_long1(rxbuf))) {
				hydra_devrecv();
				devrxid = intell(h_long1(rxbuf));
			     }
			     h_long1(txbufin) = intell(devrxid);
#ifdef HAS_SELECT
				txlen = (int) sizeof (long);
				txtype = HPKT_DEVDACK;
#else
			     txpkt((int) sizeof (long),HPKT_DEVDACK);
#endif
			     break;

			/*---------------------------------------------------*/
			case HPKT_DEVDACK:
			     if (devtxstate && (devtxid == intell(h_long1(rxbuf)))) {
				devtxtimer = h_timer_reset();
				devtxstate = HTD_DONE;
			     }
			     break;

			/*---------------------------------------------------*/
			default:  /* unknown packet types: IGNORE, no error! */
			     break;

			/*---------------------------------------------------*/
		 }/*switch(pkttype)*/
/*
 *   txstate    ().
 */
		 /*----------------------------------------------------------*/
		 switch (txstate) {
			/*---------------------------------------------------*/
			case HTX_START:
			case HTX_SWAIT:
			     if (rxstate == HRX_FINFO) {
				txtimer = h_timer_reset();
				txretries = 0;
				txstate = HTX_INIT;
			     }
			     break;

			/*---------------------------------------------------*/
			case HTX_RINIT:
			     if (rxstate == HRX_FINFO) {
				txtimer = h_timer_reset();
				txretries = 0;
				txstate = HTX_FINFO;
			     }
			     break;

			/*---------------------------------------------------*/
			case HTX_XDATA:
			     if (rxstate && hdxlink) {
				debug(11,"*HYDRA: %s",hdxmsg);
				hydra_devsend("MSG",(byte *) hdxmsg,(int) strlen(hdxmsg));

				txtimer = h_timer_set(H_IDLE);
				txstate = HTX_XWAIT;
			     }
			     break;

			/*---------------------------------------------------*/
			case HTX_XWAIT:
			     if (!rxstate) {
				txtimer = h_timer_reset();
				txretries = 0;
				txstate = HTX_XDATA;
			     }
			     break;

			/*---------------------------------------------------*/
			case HTX_REND:
			     if (!rxstate && !devtxstate) {
				/* special for chat, braindead will protect */
				if (chattimer > 0L) break;
				if (chattimer == 0L) chattimer = -3L;

				txtimer = h_timer_reset();
				txretries = 0;
				txstate = HTX_END;
			     }
			     break;

			/*---------------------------------------------------*/
		 }/*switch(txstate)*/
	   }/*while(txstate&&pkttype)*/
	} while (txstate);

	if (txfd >= 0)
	   close(txfd);		/* close file */
	hydra_badxfer();

	if (res == XFER_ABORT) {
	   com_dump();
	   if (!hanged_up) {
	      com_putblock((byte *) abortstr,(int) strlen(abortstr));
/*	      com_flush(); */
	   }
	   com_purge();
	}
	else
	   com_flush();

	if (res == XFER_OK) res = 0;

	return (res);
}/*hydra()*/

/******************/
/* end of hydra.c */
/******************/

void rem_chat (byte *data, word len)
{
	len = len;

	while (*data) {
	      switch (*data) {
		     case '\a':
			  if (!nobell)
			     putc(7,stderr);
			  break;

		     case '\n':
			  putchar('\r');
			  /* fallthrough to default */

		     default:
			  putchar((int) *data);
			  break;
	      }
	      data++;
	}

}/*rem_chat()*/

void mergepath(char *filepath,char *path,char *file)
{
	strcpy(filepath,path);
	strcat(filepath,file);
}/*mergepath()*/


int fexist (char *filename)
{
	struct stat f;

	return ((stat(filename,&f) != -1) ? 1 : 0);
}/*fexist()*/

static char *mon[12] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun",
			 "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

char *h_revdate (long revstamp)
{
	static char  buf[12];
	struct tm   *t;

	t = localtime(&revstamp);
	sprintf(buf, "%02d %s %d",
		     t->tm_mday, mon[t->tm_mon], t->tm_year + 1900);

	return (buf);
}/*h_revdate()*/

void strupr(char *s)
{
	while ((*s++ = toupper(*s)));
}
/*****************/
/* end of misc.c */
/*****************/

void setstamp(char *name,long tim)		    /* Set time/date of file */
{
	struct timeval tm;

	if (tim == 0L)	gettimeofday(&tm, (struct timezone *)NULL);
	else		tm.tv_sec = tim, tm.tv_usec = 0;
	utimes(name, &tm);
}
/******************/
/* end of syspc.c */
/******************/

int dos_open (char *pathname, byte create)
{
	register int access;

	if (create) {
		access = O_RDWR|O_CREAT;
		if (create == 2) access |= O_TRUNC;
	} else	access = O_RDONLY;

	return (open(pathname, access, create ? 0644 : 0));
}/*dos_open()*/


/*********************/
/* end of dos_file.c */
/*********************/
#if 0
extern file_list *respond_wazoo(char*);

extern int hydra_s(void);
extern int hydra_s(void)
{
	int rc=0;
	file_list *tosend=NULL,*respond=NULL;
	char *nonhold_mail;

	debug(1,"start hydra transfer");
/* Hydra init */
	hydra_txwindow = hydra_rxwindow = 8192;
	hydra_init(0);
	if (localoptions & NOHOLD) nonhold_mail=ALL_MAIL;
	else nonhold_mail=NONHOLD_MAIL;
	if (emsi_remote_lcodes & HAT)
	{
		loginf("remote asked to \"hold all traffic\", no send");
		tosend=NULL;
	}
	else tosend=create_filelist(remote,nonhold_mail,0);
/*     -  */
	if ((tosend != NULL) || ((emsi_remote_lcodes & NPU) == 0))
/***------------------------------------------------------------------***/
/*	while ((de=readdir(dp))) if (regexec(regptr, de->d_name)) */
	for (respond=tosend; respond; respond=respond->next)
	 {
	 if (respond->remote)
	  switch (rc=hydra(respond->local, respond->remote))
	  {
	    case XFER_ABORT:
			return (1);

	    case XFER_SKIP:
	    default:
			break;

	    case XFER_OK:
			execute_disposition (respond);
			break;
	  } /* Hydra switch */
	else execute_disposition (respond);
	} /* for */
/****-------------------------------------------------------------------****/
/*
	if ((emsi_local_opts & NRQ) == 0)
		if ((respond=respond_wazoo(NULL)))
*/
			rc=hydra(NULL, NULL); /* respond */ /* end of bath */
	tidy_filelist(tosend,(rc == 0));
	tidy_filelist(respond,0);
/* Hydra deinit */
	hydra_deinit();
	debug(1,"end hydra transfer");
	return rc;
}
#endif

int hydra_proc ()
{
	int rc = 0;
	fa_list *eff_remote, tmpl;
	file_list *tosend = NULL, *freq = NULL;

	extern char *freqname, *frqpathfile, ishfreq;	/* Hydra's freq attributes */

	/* default configuration */
	nobell = false;
	parity = false;
	flowflags = 0;
	hdxsession = nooriginator = false;
	hydra_txwindow = hydra_rxwindow = 0L; 
/* BUG!!!
	hydra_txwindow = hydra_rxwindow = 8192L;
 */
	noresume = nostamp = false;
	hydra_options = 0L;
	download = inbound;

	single_file[0] = '\0';
	single_done = false;
	logfp = NULL;
	pid = getpid ();

	loginf ("start Hydra session");

#ifdef SHOW_SPEED
	cur_speed = connect_speed?connect_speed:19200;
#else
	cur_speed = 19200;	/* sys_init(); */
	loginf ("Connected at %ld", cur_speed);
#endif
/* = cur_speed = (cur_speed) ? cur_speed : 19200L; */
/*	dcdmask = 0x80; */

/*	sys_init (); */
	hydra_init (hydra_options);

	if (emsi_remote_lcodes & NPU) {
	    loginf ("remote requested \"no pickup\", no send");
	    eff_remote = NULL;
	}
	else if (emsi_remote_lcodes & PUP) {
		 loginf ("remote requested \"pickup primary\"");
		 tmpl.addr = remote->addr;
		 tmpl.next = NULL;
		 eff_remote = &tmpl;
		}
	else
	    eff_remote = remote;
	tosend = create_filelist (eff_remote, ALL_MAIL, 0);

	rc = batch_hydra (tosend, NULL);
	rc = hydra (NULL, NULL);

	if (ishfreq && !hanged_up) {
	    if (freqname)	free (freqname);
	    freqname = xstrcpy (frqpathfile);
	    freq = respond_wazoo (NULL);
	    rc = batch_hydra (freq, NULL);
	    ishfreq = 0;
	}

	rc = hydra (NULL, NULL);
	hydra_deinit ();
/*	sys_restore (); */

	tidy_filelist (tosend, (rc == 0));
	return (rc);

}				/* hydra_proc */

int batch_hydra (file_list *lst, char *alias)
{
	int rc = 0, maxrc = 0;
	file_list *tmpf, **tmpfl;
	extern char *freqname, *frqpathfile, ishfreq;

	for (tmpf = lst; tmpf && (maxrc < 2); tmpf = tmpf->next) {
	    if (tmpf->remote) {
	       switch (rc = hydra (tmpf->local, tmpf->remote)) {
		 case XFER_ABORT:	return (1);
		 case XFER_SKIP:
		 case XFER_OK:	rc = abs (rc ? 0 : rc);
				if (rc > maxrc) maxrc = rc;
				if (rc == 0) execute_disposition (tmpf);
				if (ishfreq) { /* && emsi_remote_opts & RH1) */
				    if (freqname)  free (freqname);
				    freqname = xstrcpy (frqpathfile);
				    for (tmpfl = &lst; *tmpfl; tmpfl = &((*tmpfl)->next));
				    *tmpfl = respond_wazoo (NULL);
				    ishfreq = 0;
				}
				continue;
	       }
	    }
	    else if (maxrc == 0) execute_disposition (tmpf);
	}
	if (maxrc < 2)	rc = abs (rc);
	if (rc > maxrc) maxrc = rc;

  return (maxrc < 2) ? 0 : maxrc;
}				/*batch_hydra()*/
