;;; mew-minibuf.el --- Minibuffer input methods for Mew

;; Author:  Kazu Yamamoto <Kazu@Mew.org>
;; Created: Mar 23, 1997
;; Revised: Oct 25, 1997

;;; Code:

(defconst mew-minibuf-version "mew-minibuf.el version 0.08")

(require 'mew)

;; ugh! Hard coding... Setq by yourself if you don't use +draft.
(defvar mew-range-alist 
  '(("^+draft" . "all") ("^+trash" . "all") (".*" . "update"))
  "*alist of folder and default range pair")

(defvar mew-mini-address-map nil)
(defvar mew-mini-folder-map  nil)

(if mew-mini-address-map
    ()
  (setq mew-mini-address-map (make-sparse-keymap))
  (define-key mew-mini-address-map "\t"     'mew-complete-address)
  (define-key mew-mini-address-map " "      'mew-complete-address)
  (define-key mew-mini-address-map "\C-c\t" 'mew-circular-complete-domain)
  (define-key mew-mini-address-map "\r"     'exit-minibuffer)
  (define-key mew-mini-address-map "\n"     'exit-minibuffer)
  (define-key mew-mini-address-map "\C-g"   'abort-recursive-edit)
  (if mew-use-bbdb
      (define-key mew-mini-address-map "\e\t"     'bbdb-complete-name))
  )

(if mew-mini-folder-map
    ()
  (setq mew-mini-folder-map (make-sparse-keymap))
  (define-key mew-mini-folder-map " "    'mew-complete-folder)
  (define-key mew-mini-folder-map "\t"   'mew-complete-folder)
  (define-key mew-mini-folder-map "\r"   'exit-minibuffer)
  (define-key mew-mini-folder-map "\n"   'exit-minibuffer)
  (define-key mew-mini-folder-map "\C-g" 'abort-recursive-edit)
  (define-key mew-mini-folder-map "\M-p" 'previous-history-element)
  (define-key mew-mini-folder-map "\M-n" 'next-history-element)
  )

;;;
;;; Input method
;;;

(defvar mew-input-folder-hist nil)

(defun mew-input-folder (default)
  (let ((folder))
    (setq folder (read-from-minibuffer (format "Folder name (%s): " default)
				       "+"
				       mew-mini-folder-map
				       nil
				       mew-input-folder-hist))
    (directory-file-name 
     (if (or (string= folder "") (string= folder "+"))
 	 default
       folder)
     )
    ))

(defun mew-input-folders (default &optional prompt)
  (let (form folders)
    (if prompt
	(progn
	  (setq form "Folder name : ")
	  (setq default prompt)
	  )
      (setq form (format "Folder name (%s): " default))
      (setq prompt "+")
      )
    (setq folders (read-from-minibuffer form
					prompt
					mew-mini-folder-map
					nil
					mew-input-folder-hist))
    (if (or (string= folders "") (string= folders "+"))
	(setq folders default))
    (mapcar
     (function directory-file-name) 
     (mew-header-split
      (mew-header-syntax folders) ?,))
    ))

(defun mew-input-range (folder &optional range)
  "Return (range erase-update)."
  (mew-decode-syntax-delete)
  ;; for the case when parts are expanded in the bottom of the folder
  (let* ((pair (mew-assoc-match2 folder mew-range-alist 0))
	 (default (or (cdr pair) "update"))) ;; just in case
    (if (and mew-ask-range (not range))
	(setq range
	      (completing-read (format "Range (%s): " default)
			       (mapcar
				(function (lambda (x) (cons x x)))
				mew-input-range-list))))
    (if (or (string= range "") (not range))
	(setq range default))
    (if (not (string-match "^[0-9a-zA-Z]" range))
	(error "Illegal range: %s" range))
    (cond
     ;; range is other than "update"
     ((not (string= range "update"))
      (list range 'erase)) ;; non-update, erase it
     ;; update
     ((get-buffer folder)
      (mew-update-range))
     ;; update but folder doesn't exist in Emacs. 
     (t (list "all" 'update)) ;; no need to erase
     )
    ))

(defun mew-update-range ()
  (save-excursion
    (goto-char (point-max))
    (if (bobp)
	(list "all" 'update) ;; buffer is empty. no need to erase
      (forward-line -1)
      (list 
       (concat
	(int-to-string (1+ (string-to-int (mew-summary-message-number))))
	"-" 
	"last")
       'update) ;; this is update!
      )))

(defun mew-input-address (prompt)
  (read-from-minibuffer (concat prompt " ") "" mew-mini-address-map nil)
  )

(defun mew-input-address2 (prompt default)
  (let ((ret (read-from-minibuffer 
	      (format prompt default) "" mew-mini-address-map nil)))
    (if (string= ret "") default ret)
    ))

(defun mew-input-file-name (&optional prompt default)
  (let ((msg (or prompt "File: "))
	(use-dialog-box nil)
	file)
    (cond
     ((null default)
      (setq file mew-home))
     ((or (string-match (format "^[~%s]" mew-path-separator) default)
        ;; allow drive letter -- "^\\([A-Za-z]:\\|[~%s]\\)"
        (string-match (format "^[A-Za-z]:%s.+" mew-path-separator) default))
      (setq file default))
     (t
      (setq file (concat mew-home default))
      )
     )
    (expand-file-name (read-file-name msg file file))
    ))

(defun mew-input-directory-name (&optional default)
  (let ((dir (expand-file-name (read-file-name "Directory : " default nil t))))
    (if (file-directory-p dir)
	dir
      (message "%s is not directory" dir)
      (sit-for 1)
      (mew-input-directory-name)
      )
    ))

(defun mew-input-string (prompt subdir default)
  (let ((input (read-string (format prompt subdir default) "")))
    (if (string= input "") default input))
  )

(defun mew-input-type (prompt filename default type-list)
  (let ((completion-ignore-case t)
	(type))
    (setq type (completing-read
		(format prompt filename default)
		(mapcar
		 (function (lambda (x) (cons x x)))
		 type-list)
		nil
		t  ;; not require match
		""))
    (if (string= type "") default type)
    ))

(defun mew-input-config (default)
  (let ((config))
    (setq config (completing-read
		  (format "Config value (%s): "
			  (or default mew-config-default))
		  (mapcar
		   (function
		    (lambda (x) (cons x x)))
		   mew-config-list)
		  nil nil nil))
    (if (string= config "")
	(or default mew-config-default)
      config)
    ))

(provide 'mew-minibuf)

;;; Copyright Notice:

;; Copyright (C) 1997, 1998 Mew developing team.
;; All rights reserved.

;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions
;; are met:
;; 
;; 1. Redistributions of source code must retain the above copyright
;;    notice, this list of conditions and the following disclaimer.
;; 2. Redistributions in binary form must reproduce the above copyright
;;    notice, this list of conditions and the following disclaimer in the
;;    documentation and/or other materials provided with the distribution.
;; 3. All advertising materials mentioning features or use of this software
;;    must display the following acknowledgement:
;;       This product includes software developed by 
;;       Mew developing team and its contributors.
;; 4. Neither the name of the team nor the names of its contributors
;;    may be used to endorse or promote products derived from this software
;;    without specific prior written permission.
;; 
;; THIS SOFTWARE IS PROVIDED BY THE TEAM AND CONTRIBUTORS ``AS IS'' AND
;; ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
;; PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE TEAM OR CONTRIBUTORS BE
;; LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
;; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
;; SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
;; BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
;; OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
;; IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; mew-minibuf.el ends here
