/* 
 * ratAppInit.c --
 *
 *	Provides a default version of the Tcl_AppInit procedure for
 *	use in wish and similar Tk-based applications.
 *
 *
 * TkRat software and its included text is Copyright 1996,1997,1998
 * by Martin Forssn
 *
 * The full text of the legal notice is contained in the file called
 * COPYRIGHT, included with this distribution.
 */

#include <pwd.h>
#include <tk.h>
#include "ratFolder.h"
#include "ratPGP.h"
#include <locale.h>

/*
 * Figure out if we are using a version later than 8.0b1
 */
#if TCL_MAJOR_VERSION >= 8 && (TCL_MINOR_VERSION >0 || TCL_RELEASE_LEVEL >=1)
#    define TCL_CREATEOK
#endif

/*
 * The following variable is a special hack that is needed in order for
 * Sun shared libraries to be used for Tcl.
 */

#ifdef NEED_MATHERR
extern int matherr();
int *tclDummyMathPtr = (int *) matherr;
#endif

/*
 * The following structure is used by the RatBgExec command to keep the
 * information about processes running in the background.
 */
typedef struct RatBgInfo {
    Tcl_Interp *interp;
    int numPids;
    int *pidPtr;
    int status;
    char *exitStatus;
    struct RatBgInfo *nextPtr;
} RatBgInfo;

/*
 * How often we should check for dead processes (in milliseconds)
 */
#define DEAD_INTERVAL 200

/*
 * Names of days and months as per rfc822.
 */
char *dayName[] = {"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"};
char *monthName[] = {"jan", "Feb", "Mar", "Apr", "May", "Jun",
		     "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};

/*
 * Information about my mailbox
 */
char *currentHost = NULL;
char *currentMailboxName = NULL;
char *currentPersonalName = NULL;

/*
 * If we have the display or not
 */
static int hasDisplay = 1;

/*
 * True if we shouldn't used the window system
 */
static int noX = 0;

/*
 * Buffer for delayed output
 */
static char ratDelayBuffer[3];

/*
 * Communication with the child
 */
static FILE *toSender = NULL;
static int fromSender;
static int sendSequence = 0;
#ifndef TCL_CREATEOK
static Tcl_File fromSenderFile;
#endif

static Tcl_FileProc RatHandleSender;
static int RatCreateSender(Tcl_Interp *interp);

/*
 * List of sent messages
 */
typedef struct SentMsg {
    int id;
    char *handler;
    struct SentMsg *nextPtr;
} SentMsg;
static SentMsg *sentMsg = NULL;

/*
 * The location of all library files
 */
static char *autoPath = NULL;

/*
 * Local functions
 */
static Tk_TimerProc RatChildHandler;
static Tcl_VarTraceProc RatReject;
static Tcl_AppInitProc RatAppInit;
static Tcl_VarTraceProc RatHostWatcher;
static Tcl_VarTraceProc RatInitCurrent;
static Tcl_CmdProc RatBgExec;
static Tcl_CmdProc RatSend;
static Tcl_CmdProc RatGetEncoding;
static Tcl_CmdProc RatCleanup;
static Tcl_CmdProc RatTildeSubst;
static Tcl_CmdProc RatTime;
static Tcl_CmdProc RatLock;
static Tcl_CmdProc RatIsLocked;
static Tcl_CmdProc RatType;
static Tcl_CmdProc RatDSE;
static Tcl_CmdProc RatExpire;
static Tcl_CmdProc RatLL;
static Tcl_CmdProc RatDbaseCheck;
static Tcl_CmdProc RatMangleNumberCmd;
static Tcl_CmdProc RatEncodingCompatCmd;
static Tcl_CmdProc RatFormatDateCmd;


/*
 *----------------------------------------------------------------------
 *
 * main --
 *
 *	This is the main program for the application.
 *
 * Results:
 *	None: Tk_Main never returns here, so this procedure never
 *	returns either.
 *
 * Side effects:
 *	Whatever the application does.
 *
 *----------------------------------------------------------------------
 */

int
main(int argc, char **argv)
{
    int i, fargc = 0, wantArg, interactive = 0;
    char **fargv, *dir;
    struct stat sbuf;

    setlocale(LC_CTYPE, "");
    setlocale(LC_COLLATE, "");

    fargv = (char**)malloc((argc+1)*sizeof(char*));
    fargv[fargc++] = argv[0];
    for (i=1; i<argc; i++) {
	if (!strcmp("-i", argv[i])) {
	    interactive = 1;
	}
    }
    if (!interactive) {
	fargc++;
    }
    for (i=1, wantArg=0; i<argc; i++) {
	if (!strcmp("-i", argv[i])) {
	    continue;
	}
	if (!strcmp("-noX", argv[i])) {
	    noX = 1;
	    continue;
	}
	if (1 == wantArg) {
	    wantArg = 0;
	} else if ('-' == argv[i][0]) {
	    if (strcmp("-sync", argv[i])) {
		wantArg = 1;
	    }
	} else {
	    autoPath = argv[i];
	    continue;
	}
	fargv[fargc++] = argv[i];
    }
    if ((dir = getenv("RATLIBDIR"))) {
	fargv[1] = (char*)malloc(strlen(dir)+7);
	sprintf(fargv[1], "%s/tkrat", dir);
	autoPath = dir;
    } else {
	if (!autoPath) {
	    if (NULL == (dir = getenv("LIBDIR"))) {
		fprintf(stderr, "Bad installation. I miss the LIBDIR env "
				"variable\n");
		exit(-1);
	    }
	    autoPath = (char*)malloc(strlen(dir)+1);
	    strcpy(autoPath, dir);
	}
	fargv[1] = (char*)malloc(strlen(autoPath)+7);
	sprintf(fargv[1], "%s/tkrat", autoPath);
    }
    if (NULL == getenv("COMPRESS")) {
	fprintf(stderr, "Bad installation. I miss the COMPRESS env variable\n");
	exit(-1);
    }
    if (NULL == getenv("CSUFFIX")) {
	fprintf(stderr, "Bad installation. I miss the CSUFFIX env variable\n");
	exit(-1);
    }
    if (NULL == getenv("CONFIG_DIR")) {
	fprintf(stderr,
		"Bad installation. I miss the CONFIG_DIR env variable\n");
	exit(-1);
    }
    if (NULL == getenv("PGP")) {
	fprintf(stderr, "Bad installation. I miss the PGP env variable\n");
	exit(-1);
    }
    if (!interactive) {
	if (stat(fargv[1], &sbuf)) {
	    fprintf(stderr, "You have installed the files incorrectly. "
			    "I expect to find them in\n");
	    if (dir) {
		fprintf(stderr, "\"%s\" as specified in the RATLIBDIR "
				"environment variable.\n", dir);
	    } else {
		fprintf(stderr, "\"%s\" as specified at build time.\n",
			autoPath);
		fprintf(stderr, "You can use the RATLIBDIR environment "
				"variable to indicate where\n");
		fprintf(stderr, "they really are installed.\n");
	    }
	    exit(1);
	}
    }
    if (noX) {
	Tcl_Main(fargc, fargv, RatAppInit);
    } else {
	Tk_Main(fargc, fargv, RatAppInit);
    }
    return 0;			/* Needed only to prevent compiler warning. */
}

/*
 *----------------------------------------------------------------------
 *
 * RatAppInit --
 *
 *	This procedure performs application-specific initialization.
 *	Most applications, especially those that incorporate additional
 *	packages, will have their own version of this procedure.
 *
 * Results:
 *	Returns a standard Tcl completion code, and leaves an error
 *	message in interp->result if an error occurs.
 *
 * Side effects:
 *	Depends on the startup script.
 *
 *----------------------------------------------------------------------
 */

static int
RatAppInit(Tcl_Interp *interp)
{
    struct passwd *pwPtr;
    Tk_Window main;
    Tcl_DString ds;
    char *s;
#if TCL_MAJOR_VERSION < 8
    char buf[1024];
#endif

    if (Tcl_Init(interp) == TCL_ERROR) {
	return TCL_ERROR;
    }

    if (!noX) {
	main = Tk_MainWindow(interp);
	if (Tk_Init(interp) == TCL_ERROR) {
	    return TCL_ERROR;
	}
    }

    /*
     * Create async handlers.
     */

    /*
     * Initialize some variables
     */
    Tcl_SetVar(interp, "ratSenderSending", "0", TCL_GLOBAL_ONLY);
    Tcl_DStringInit(&ds);
    Tcl_DStringAppendElement(&ds, autoPath);
    s = Tcl_GetVar(interp, "auto_path", TCL_GLOBAL_ONLY);
    Tcl_DStringAppend(&ds, " ", 1);
    Tcl_DStringAppend(&ds, s, strlen(s));
    Tcl_SetVar(interp, "auto_path", Tcl_DStringValue(&ds), TCL_GLOBAL_ONLY);
    Tcl_DStringFree(&ds);

    /*
     * Make sure we know who we are and that we keep track of any changes
     */
    RatInitCurrent(NULL, interp, NULL, NULL, 0);
    Tcl_TraceVar2(interp, "option", "charset", TCL_GLOBAL_ONLY|TCL_TRACE_WRITES,
	    RatInitCurrent, NULL);
    Tcl_TraceVar2(interp, "option", "domain", TCL_GLOBAL_ONLY|TCL_TRACE_WRITES,
	    RatHostWatcher, NULL);
    Tcl_TraceVar2(interp, "option", "masquerade_as",
	    TCL_GLOBAL_ONLY|TCL_TRACE_WRITES, RatHostWatcher, NULL);

    /*
     * Make sure that env(USER) and env(HOME) are set. If not then we
     * initialize them.
     */
    if (!Tcl_GetVar2(interp, "env", "USER", TCL_GLOBAL_ONLY)) {
	pwPtr = getpwuid(getuid());
	Tcl_SetVar2(interp, "env", "USER", pwPtr->pw_name, TCL_GLOBAL_ONLY);
    }
    if (!Tcl_GetVar2(interp, "env", "HOME", TCL_GLOBAL_ONLY)) {
	pwPtr = getpwuid(getuid());
	Tcl_SetVar2(interp, "env", "HOME", pwPtr->pw_dir, TCL_GLOBAL_ONLY);
    }

    /*
     * Call the init procedures for included packages.  Each call should
     * look like this:
     *
     * if (Mod_Init(interp) == TCL_ERROR) {
     *     return TCL_ERROR;
     * }
     *
     * where "Mod" is the name of the module.
     */
    if (RatFolderInit(interp) == TCL_ERROR) {
	return TCL_ERROR;
    }
    if (RatDSNInit(interp) == TCL_ERROR) {
	return TCL_ERROR;
    }

    Tcl_InitHashTable(&aliasTable, TCL_STRING_KEYS);
    Tcl_InitHashTable(&userTable, TCL_STRING_KEYS);

    /*
     * Call Tcl_CreateCommand for application-specific commands, if
     * they weren't already created by the init procedures called above.
     */
    Tcl_CreateCommand(interp, "RatBgExec", RatBgExec, NULL, NULL);
    Tcl_CreateCommand(interp, "RatGenId", RatGenId, NULL, NULL);
    Tcl_CreateCommand(interp, "RatSend", RatSend, NULL, NULL);
    Tcl_CreateCommand(interp, "RatGetEncoding", RatGetEncoding, NULL, NULL);
    Tcl_CreateCommand(interp, "RatCleanup", RatCleanup, NULL, NULL);
    Tcl_CreateCommand(interp, "RatTildeSubst", RatTildeSubst, NULL, NULL);
    Tcl_CreateCommand(interp, "RatTime", RatTime, NULL, NULL);
    Tcl_CreateCommand(interp, "RatLock", RatLock, NULL, NULL);
    Tcl_CreateCommand(interp, "RatIsLocked", RatIsLocked, NULL, NULL);
    Tcl_CreateCommand(interp, "RatHold", RatHold, NULL, NULL);
    Tcl_CreateCommand(interp, "RatAlias", RatAlias, NULL, NULL);
    Tcl_CreateCommand(interp, "RatType", RatType, NULL, NULL);
    Tcl_CreateCommand(interp, "RatDaysSinceExpire", RatDSE, NULL, NULL);
    Tcl_CreateCommand(interp, "RatExpire", RatExpire, NULL, NULL);
    Tcl_CreateCommand(interp, "RatSMTPSupportDSN", RatSMTPSupportDSN,NULL,NULL);
    Tcl_CreateCommand(interp, "RatLL", RatLL, NULL, NULL);
    Tcl_CreateCommand(interp, "RatDbaseCheck", RatDbaseCheck, NULL, NULL);
    Tcl_CreateCommand(interp, "RatSplitAdr", RatSplitAddresses, NULL, NULL);
    Tcl_CreateCommand(interp, "RatMailcapReload", RatMailcapReload, NULL, NULL);
    Tcl_CreateCommand(interp, "RatPGP", RatPGPCmd, NULL, NULL);
    Tcl_CreateCommand(interp, "RatMangleNumber", RatMangleNumberCmd, NULL,NULL);
    Tcl_CreateCommand(interp, "RatEncodingCompat", RatEncodingCompatCmd,
	    NULL, NULL);
    Tcl_CreateCommand(interp, "RatFormatDate", RatFormatDateCmd, NULL, NULL);

    /*
     * Create tcl8 compatibility functions
     */
#if TCL_MAJOR_VERSION < 8
    sprintf(buf, "rename file RatOldFile");
    Tcl_Eval(interp, buf);
    Tcl_CreateCommand(interp, "file", RatFile, NULL, NULL);
#endif	/* TCL_MAJOR_VERSION < 8 */

    /*
     * Specify a user-specific startup file to invoke if the application
     * is run interactively.  Typically the startup file is "~/.apprc"
     * where "app" is the name of the application.  If this line is deleted
     * then no user-specific startup file will be run under any conditions.
     */

    Tcl_SetVar(interp, "tcl_rcFileName", ".tkratrc", TCL_GLOBAL_ONLY);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * RatInitCurrent --
 *
 *	Construct currentHost, currentMailboxName and
 *	currentPersonalName strings. The algorithm for building
 *	currentHost is:
 *	  if option(masqerade_as) is set then
 *	     use the value of option(masqerade_as) as currentHost
 *	  else
 *	      if gethostname() returns a name with a dot in it then
 *	         use it as currentHost
 *	      else
 *	         use the result of gethostname and the value of option(domain)
 *	      endif
 *	  endif
 *
 * Results:
 *      None.
 *
 * Side effects:
 *	The current* global variables are initialized
 *
 *
 *----------------------------------------------------------------------
 */

static char*
RatInitCurrent(ClientData clientData, Tcl_Interp *interp, char *name1,
	       char *name2, int flags)
{
    char *charset = Tcl_GetVar2(interp, "option", "charset", TCL_GLOBAL_ONLY);
    struct passwd *passwdPtr;
    unsigned char buf[1024];

    if (currentHost) {
	free(currentHost);
	free(currentMailboxName);
	free(currentPersonalName);
    }
    if (NULL == (currentHost = Tcl_GetVar2(interp, "option",
	    "masquerade_as", TCL_GLOBAL_ONLY)) || !*currentHost) {
	gethostname(buf, sizeof(buf));
	if (!strchr(buf, '.')) {
	    char *domain = Tcl_GetVar2(interp, "option", "domain",
		    TCL_GLOBAL_ONLY);
	    if (domain && *domain) {
		strcat(buf, ".");
		strcat(buf, domain);
	    }
	}
	currentHost = (char*)buf;
    }
    currentHost = cpystr(currentHost);
    Tcl_SetVar(interp, "ratCurrentHost", currentHost, TCL_GLOBAL_ONLY);
    passwdPtr = getpwuid(getuid());
    currentMailboxName = cpystr(passwdPtr->pw_name);
    strcpy((char*)buf, passwdPtr->pw_gecos);
    if (strchr((char*)buf, ',')) {
	*strchr((char*)buf, ',') = '\0';
    }
    currentPersonalName = (char*)RatEncodeHeaderLine(buf, charset, 0, 1);
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * RatLog --
 *
 *	Sends a log message to the interface
 *
 * Results:
 *      None.
 *
 * Side effects:
 *	The tcl command 'RatLog' will be called
 *
 *
 *----------------------------------------------------------------------
 */

void
RatLog(Tcl_Interp *interp, RatLogLevel level, char *message, int keep)
{
    char *argv = message;
    char *buf, *parsedMsg;
    int levelNumber;

    switch(level) {
    case RAT_BABBLE:	levelNumber = 0; break;
    case RAT_PARSE:	levelNumber = 1; break;
    case RAT_INFO:	levelNumber = 2; break;
    case RAT_WARN:	levelNumber = 3; break;
    case RAT_ERROR:	levelNumber = 4; break;
    case RAT_FATAL:	/* fallthrough */
    default:		levelNumber = 5; break;
    }

    parsedMsg = Tcl_Merge(1, &argv);
    if (hasDisplay) {
	buf = (char*) ckalloc(16 + strlen(parsedMsg)+9);
	sprintf(buf, "RatLog %d %s %s", levelNumber, parsedMsg,
		keep?"explicit":"time");
	if (TCL_OK != Tcl_GlobalEval(interp, buf)) {
	    fprintf(stderr, "Error: '%s'\nWhile executing '%s'\n",
		    interp->result, buf);
	}
	ckfree(buf);
    } else {
	fprintf(stdout, "STATUS %d %s %d", levelNumber, parsedMsg, keep);
	fputc('\0', stdout);
	fflush(stdout);
    }
    ckfree(parsedMsg);
}


/*
 *----------------------------------------------------------------------
 *
 * RatMangleNumber --
 *
 *      Creates a string representation of the given number that is maximum
 *      four characters long. The actual mangling is done in the tcl-proc
 *      ratMangleNumber.
 *
 * Results:
 *      Returns a pointer to a static buffer containg the string
 *	representation of the number.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

char*
RatMangleNumber(int number)
{
    static char buf[32];     /* Scratch area */

    if (number < 1000) {
	sprintf(buf, "%d", number);
    } else if (number < 10000) {
	sprintf(buf, "%.1fk", number/1000.0);
    } else if (number < 1000000) {
	sprintf(buf, "%dk", (number+500)/1000);
    } else if (number < 10000000) {
	sprintf(buf, "%.1fM", number/1000000.0);
    } else {
	sprintf(buf, "%dM", (number+500000)/1000000);
    }
    return buf;
}


/*
 *----------------------------------------------------------------------
 *
 * RatMangleNumberCmd --
 *
 *	See ../doc/interface for a descriptions of arguments and result.
 *
 * Results:
 *      A list of strings to display to the user.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatMangleNumberCmd(ClientData dummy, Tcl_Interp *interp, int argc, char *argv[])
{
    int number;

    if (2 != argc || TCL_OK != Tcl_GetInt(interp, argv[1], &number)) {
	Tcl_AppendResult(interp, "Usage: ", argv[0], " number", (char*) NULL);
	return TCL_ERROR;
    }
    Tcl_SetResult(interp, RatMangleNumber(number), TCL_VOLATILE);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * RatBgExec --
 *
 *	See ../doc/interface
 *
 * Results:
 *      The return value is normally TCL_OK and the result can be found
 *      in interp->result. If something goes wrong TCL_ERROR is returned
 *      and an error message will be left in interp->result.
 *
 * Side effects:
 *      AN entry is added to ratBgInfoPtr.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatBgExec(ClientData dummy, Tcl_Interp *interp, int argc, char *argv[])
{
    static RatBgInfo *ratBgList = NULL;
    RatBgInfo *bgInfoPtr;
    char **pidv;
    Tcl_DString ds;
    int i;

    if (argc != 3) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" exitStatus cmd\"", (char *) NULL);
	return TCL_ERROR;
    }

    bgInfoPtr = (RatBgInfo*)ckalloc(sizeof(*bgInfoPtr));
    bgInfoPtr->interp = interp;
    bgInfoPtr->exitStatus = (char*)cpystr(argv[1]);
    Tcl_DStringInit(&ds);
    Tcl_DStringAppend(&ds, "exec ", 5);
    Tcl_DStringAppend(&ds, argv[2], -1);
    Tcl_DStringAppend(&ds, " &", 2);
    if (TCL_OK != Tcl_Eval(interp, Tcl_DStringValue(&ds))) {
	Tcl_DStringFree(&ds);
	Tcl_SetVar(bgInfoPtr->interp, bgInfoPtr->exitStatus, "-1",
		TCL_GLOBAL_ONLY);
	ckfree(bgInfoPtr);
	return TCL_ERROR;
    }
    Tcl_DStringFree(&ds);
    Tcl_SplitList(interp, interp->result, &bgInfoPtr->numPids, &pidv);
    bgInfoPtr->pidPtr = (int*)ckalloc(bgInfoPtr->numPids*sizeof(int));
    for (i=0; i<bgInfoPtr->numPids; i++) {
	bgInfoPtr->pidPtr[i] = atoi(pidv[i]);
    }
    if (!ratBgList) {
	Tcl_CreateTimerHandler(DEAD_INTERVAL, RatChildHandler, &ratBgList);
    }
    bgInfoPtr->nextPtr = ratBgList;
    ratBgList = bgInfoPtr;
    ckfree(pidv);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RatChildHandler --
 *
 *	This process checks if processes in a pipeline are dead. When
 *	all are dead the corresponding variables are set etc.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Sets variables mentioned in the RatBgInfo structure.
 *
 *
 *----------------------------------------------------------------------
 */

void
RatChildHandler(ClientData clientData)
{
    RatBgInfo *bgInfoPtr, **bgInfoPtrPtr = (RatBgInfo**)clientData;
    int i, allDead, status, result;

    while (*bgInfoPtrPtr) {
	bgInfoPtr = *bgInfoPtrPtr;
	allDead = 1;
	for (i = 0; i < bgInfoPtr->numPids; i++) {
	    if (bgInfoPtr->pidPtr[i]) {
		result = waitpid(bgInfoPtr->pidPtr[i], &status, WNOHANG);
		if ((result == bgInfoPtr->pidPtr[i])
			|| ((result == -1) && (errno == ECHILD))) {
		    bgInfoPtr->pidPtr[i] = 0;
		    if (i == bgInfoPtr->numPids-1) {
			bgInfoPtr->status = WEXITSTATUS(status);
		    }
		} else {
		    allDead = 0;
		}
	    }
	}
	if (allDead) {
	    char buf[36];

	    sprintf(buf, "%d", bgInfoPtr->status);
	    Tcl_SetVar(bgInfoPtr->interp, bgInfoPtr->exitStatus, buf,
		    TCL_GLOBAL_ONLY);
	    *bgInfoPtrPtr = bgInfoPtr->nextPtr;
	    ckfree(bgInfoPtr->pidPtr);
	    ckfree(bgInfoPtr);
	} else {
	    bgInfoPtrPtr = &(*bgInfoPtrPtr)->nextPtr;
	}
    }
    if (*(RatBgInfo**)clientData) {
	Tcl_CreateTimerHandler(DEAD_INTERVAL, RatChildHandler, clientData);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * RatGenId --
 *
 *	See ../doc/interface
 *
 * Results:
 *      The return value is normally TCL_OK and the result can be found
 *      in interp->result. If something goes wrong TCL_ERROR is returned
 *      and an error message will be left in interp->result.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatGenId(ClientData dummy, Tcl_Interp *interp, int argc, char *argv[])
{
    static long lastid = 0;

    long t = time(NULL);
    if (t <= lastid)
        lastid++;
    else
        lastid = t;
    sprintf(interp->result, "%lx.%x", lastid, (int)getpid());
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RatSend --
 *
 *	See ../doc/interface
 *
 *	This checks that we have something that looks like a good message.
 *	The actual sending is done by a subprocess called the sending
 *	process. We communicate with that process via the stdin and stdout
 *	channels. The follwing commands can be sent to the sender:
 *	    SEND id prefix deferred
 *	    QUIT
 *	The sender can send the following commands:
 *	    STATUS level status_text keep
 *	    FAILED id prefix deferred text
 *	    SAVE file save_to to from cc subject
 *	    SENT id deferred
 *	    PGP pgp_specific_data
 *	The server will respond to all PGP commands
 *
 * Results:
 *      The return value is normally TCL_OK and the result can be found
 *      in interp->result. If something goes wrong TCL_ERROR is returned
 *      and an error message will be left in interp->result.
 *
 * Side effects:
 *      A message is sent.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatSend(ClientData dummy, Tcl_Interp *interp, int argc, char *argv[])
{
    char *tmp, *cacheDir, *handler = NULL, buf[1024];
    Tcl_DString tildeBuffer;
    SentMsg **smPtrPtr;

    if (argc != 2 && argc != 3) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" action ?handler?\"", (char *) NULL);
	return TCL_ERROR;
    }

    Tcl_DStringInit(&tildeBuffer);
    cacheDir = Tcl_GetVar2(interp, "option", "send_cache", TCL_GLOBAL_ONLY);
    cacheDir = Tcl_TranslateFileName(interp, cacheDir, &tildeBuffer);

    if (!strcmp("kill", argv[1])) {
	if (toSender) {
	    fprintf(toSender, "QUIT\n");
	    fflush(toSender);
	}
    } else if (!strcmp("init", argv[1]) || !strcmp("sendDeferred", argv[1])) {
	Tcl_DString fileList;
	char **listArgv;
	int listArgc, i, sent;

	Tcl_DStringInit(&fileList);
	if (TCL_OK != RatHoldList(interp, cacheDir, &fileList)
		|| TCL_OK != Tcl_SplitList(interp, Tcl_DStringValue(&fileList),
			&listArgc, &listArgv)) {
	    Tcl_DStringFree(&fileList);
	    goto error;
	}
	Tcl_DStringFree(&fileList);

	if (!strcmp("init", argv[1])) {
	    sprintf(buf, "%d", listArgc);
	    Tcl_SetVar(interp, "ratDeferred", buf, TCL_GLOBAL_ONLY);
	} else {
	    Tcl_DString cmd;

	    if (TCL_OK != RatCreateSender(interp)) {
		free(listArgv);
		goto error;
	    }
	    Tcl_SetVar(interp, "ratSenderSending", "1", TCL_GLOBAL_ONLY);
	    Tcl_DStringInit(&cmd);
	    Tcl_DStringAppendElement(&cmd, "SEND");
	    for (i=0, sent=0; i<listArgc; i++) {
		for (smPtrPtr=&sentMsg;
			*smPtrPtr && strcmp((*smPtrPtr)->handler, listArgv[i]);
			smPtrPtr = &(*smPtrPtr)->nextPtr);
		if (NULL != *smPtrPtr) continue;
		*smPtrPtr =
			(SentMsg*)malloc(sizeof(SentMsg)+strlen(listArgv[i])+1);
		(*smPtrPtr)->id = sendSequence;
		(*smPtrPtr)->handler = (char*)*smPtrPtr+sizeof(SentMsg);
		strcpy((*smPtrPtr)->handler, listArgv[i]);
		(*smPtrPtr)->nextPtr = NULL;
		Tcl_DStringStartSublist(&cmd);
		sprintf(buf, "%d", sendSequence++);
		Tcl_DStringAppendElement(&cmd, buf);
		sprintf(buf, "%s/%s", cacheDir, listArgv[i]);
		Tcl_DStringAppendElement(&cmd, buf);
		Tcl_DStringAppendElement(&cmd, "1");
		Tcl_DStringEndSublist(&cmd);
		sent++;
	    }
	    fprintf(toSender, "%s\n", Tcl_DStringValue(&cmd));
	    fflush(toSender);
	    Tcl_DStringFree(&cmd);
	    sprintf(buf, "%d", sent);
	    Tcl_SetResult(interp, buf, TCL_VOLATILE);
	}
	free(listArgv);

    } else if (argc == 3) {
	/*
	 * The algorithm here is:
	 *  - First we make sure that we got something that at least looks
	 *	  like a letter.
	 *	- Insert the message into the send cache.
	 *	- If the delivery_mode is deferred then we return.
	 *	- If we do not have a child process then we create one.
	 *	- Make the child process send the message.
	 *  - Return.
	 */
	if ((NULL == (tmp = Tcl_GetVar2(interp, argv[2], "to",TCL_GLOBAL_ONLY)))
		|| RatIsEmpty(tmp)) {
	    Tcl_SetResult(interp, "RatSend needs at least the to element",
		    TCL_STATIC);
	    goto error;
	}

	if (TCL_OK != RatHoldInsert(interp, cacheDir, argv[2], "")) {
	    goto error;
	}
	handler = cpystr(interp->result);

	if (!strcmp(argv[1], "direct")) {
	    if (TCL_OK != RatCreateSender(interp)) {
		free(handler);
		goto error;
	    }
	    Tcl_SetVar(interp, "ratSenderSending", "1", TCL_GLOBAL_ONLY);
	    for (smPtrPtr=&sentMsg; *smPtrPtr;smPtrPtr = &(*smPtrPtr)->nextPtr);
	    *smPtrPtr = (SentMsg*)malloc(sizeof(SentMsg)+strlen(handler)+1);
	    (*smPtrPtr)->id = sendSequence;
	    (*smPtrPtr)->handler = (char*)*smPtrPtr+sizeof(SentMsg);
	    strcpy((*smPtrPtr)->handler, handler);
	    (*smPtrPtr)->nextPtr = NULL;
	    fprintf(toSender, "SEND {%d %s 0}\n", sendSequence++, handler);
	    fflush(toSender);
	} else {
	    tmp = Tcl_GetVar(interp, "ratDeferred", TCL_GLOBAL_ONLY);
	    sprintf(buf, "%d", atoi(tmp)+1);
	    Tcl_SetVar(interp, "ratDeferred", buf, TCL_GLOBAL_ONLY);
	}
	free(handler);
    }

    Tcl_DStringFree(&tildeBuffer);
    return TCL_OK;

error:
    Tcl_DStringFree(&tildeBuffer);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * RatCreateSender --
 *
 *	Create the sender subprocess (if not already running).
 *
 * Results:
 *	A standard tcl result.
 *
 * Side effects:
 *      A new process may be created.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatCreateSender(Tcl_Interp *interp)
{
    int toPipe[2], fromPipe[2], senderPid;

    if (toSender) {
	return TCL_OK;
    }

    /*
     * Create the sender subprocess and create a handler on the from pipe.
     */
    pipe(toPipe);
    pipe(fromPipe);
    if (0 == (senderPid = fork())) {
	close(toPipe[1]);
	close(fromPipe[0]);
	dup2(toPipe[0], 0);
	dup2(fromPipe[1], 1);
	fcntl(0, F_SETFD, 0);
	fcntl(1, F_SETFD, 0);
	hasDisplay = 0;
	RatSender(interp);
	/* notreached */
    }
    if (-1 == senderPid) {
	Tcl_SetResult(interp, "Failed to fork sender process", TCL_STATIC);
	return TCL_ERROR;
    }
    close(toPipe[0]);
    close(fromPipe[1]);
    toSender = fdopen(toPipe[1], "w");
    fromSender = fromPipe[0];
#ifndef TCL_CREATEOK
    fromSenderFile = Tcl_GetFile((ClientData)fromSender, TCL_UNIX_FD);
    Tcl_CreateFileHandler(fromSenderFile, TCL_READABLE, RatHandleSender,
	    (ClientData)interp);
#else
    Tcl_CreateFileHandler(fromSender, TCL_READABLE, RatHandleSender,
	    (ClientData)interp);
#endif
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RatHandleSender --
 *
 *	Handle events from the sender process.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *      Whatever the sender dictates.
 *
 *
 *----------------------------------------------------------------------
 */

static void
RatHandleSender(ClientData clientData, int mask)
{
    Tcl_Interp *interp = (Tcl_Interp*)clientData;
    char buf[1024], **argv, **destArgv, *msg, *tmp;
    int i, argc, destArgc, fd, deferred = 0, id = 0, flags;
    Tcl_DString cmd;
    struct stat sbuf;
    SentMsg *smPtr, **smPtrPtr;

    i = 0;
    do {
	if (1 != read(fromSender, &buf[i], 1)) {
	    Tcl_SetVar(interp, "ratSenderSending", "0", TCL_GLOBAL_ONLY);
#ifndef TCL_CREATEOK
	    Tcl_DeleteFileHandler(fromSenderFile);
#else
	    Tcl_DeleteFileHandler(fromSender);
#endif
	    fclose(toSender);
	    toSender = NULL;
	    return;
	}
    } while (buf[i++] != '\0');

    Tcl_SplitList(interp, buf, &argc, &argv);

    if (!strcmp(argv[0], "STATUS")) {
	RatLog(interp, atoi(argv[1]), argv[2], atoi(argv[3]));

    } else if (!strcmp(argv[0], "FAILED")) {
	id = atoi(argv[1]);
	deferred = atoi(argv[3]);
	RatLog(interp, RAT_ERROR, argv[4], 0);
	if (TCL_OK != RatHoldExtract(interp, argv[2], NULL, NULL)) {
	    return;
	}
	if (TCL_OK != Tcl_VarEval(interp, "ComposeExtracted ", interp->result,
		NULL)) {
	    RatLog(interp, RAT_ERROR, interp->result, 0);
	}

    } else if (!strcmp(argv[0], "SAVE")) {
	Tcl_SplitList(interp, argv[2], &destArgc, &destArgv);
	(void)stat(argv[1], &sbuf);
	msg = (char*)malloc(sbuf.st_size);
	fd = open(argv[1], O_RDONLY);
	read(fd, msg, sbuf.st_size);
	close(fd);

	if (!strcmp(destArgv[1], "file")) {
	    Tcl_Channel channel;
	    int perm;
	    struct tm *tmPtr;
	    time_t now;

	    if (5 == destArgc) {
		Tcl_GetInt(interp, destArgv[4], &perm);
	    } else {
		Tcl_GetInt(interp, Tcl_GetVar2(interp, "option", "permissions",
			TCL_GLOBAL_ONLY), &perm);
	    }
	    channel = Tcl_OpenFileChannel(interp,destArgv[3],"a", perm);
	    if (NULL != channel) {
		now = time(NULL);
		tmPtr = gmtime(&now);
		sprintf(buf, "From %s@%s %s %s %2d %02d:%02d GMT 19%02d\n",
			currentMailboxName, currentHost,
			dayName[tmPtr->tm_wday], monthName[tmPtr->tm_mon],
			tmPtr->tm_mday, tmPtr->tm_hour, tmPtr->tm_min,
			tmPtr->tm_year);
		strcat(buf, "Status: RO\n");
		Tcl_Write(channel, buf, strlen(buf));
		Tcl_Write(channel, msg, sbuf.st_size);
		Tcl_Close(interp, channel);
	    } else {
		sprintf(buf, "%s: %s", "Failed to save copy of outgoing"
			"message", Tcl_PosixError(interp));
		RatLog(interp, RAT_ERROR, buf, 0);
	    }
	} else if (!strcmp(destArgv[1], "dbase")) {
	    struct tm *tmPtr;
	    time_t now;

	    now = time(NULL);
	    tmPtr = gmtime(&now);
	    sprintf(buf, "From %s@%s %s %s %2d %02d:%02d GMT 19%02d\n",
		    currentMailboxName, currentHost,
		    dayName[tmPtr->tm_wday], monthName[tmPtr->tm_mon],
		    tmPtr->tm_mday, tmPtr->tm_hour, tmPtr->tm_min,
		    tmPtr->tm_year);

	    if (TCL_OK != RatDbInsert(interp, argv[3], argv[4], argv[5],
		    argv[6], time(NULL), "RO", destArgv[3], atoi(destArgv[5]),
		    destArgv[4], buf, msg, sbuf.st_size)) {
		sprintf(buf, "%s: %s", "Failed to save copy of outgoing"
			"message", interp->result);
		RatLog(interp, RAT_ERROR, buf, 0);
	    }
	} else if (!strcmp(destArgv[1], "imap")) {
	    AppendToIMAP(interp, destArgv[3], destArgv[4], argv[7],
			 argv[8], msg, sbuf.st_size);
	} else if (!strcmp(destArgv[1], "mh")) {
	    RatLog(interp, RAT_ERROR, "Unfortunately I can not save outgoing "
		    "mail in mh-folders yet.", 0);
	} else {
	    RatLog(interp, RAT_ERROR,
		    "Internal error: illegal save type in RatHandleSender", 0);
	}
	unlink(argv[1]);
	free(msg);
	free(destArgv);
    } else if (!strcmp(argv[0], "PGP")) {
	if (!strcmp("getpass", argv[1])) {
	    tmp = RatPGPPhrase(interp);
	    if (tmp) {
		Tcl_ScanElement(tmp, &flags);
		Tcl_ConvertElement(tmp, buf, flags);
		fprintf(toSender, "PGP PHRASE %s\n", buf);
		memset(buf, '\0', strlen(buf));
		memset(tmp, '\0', strlen(tmp));
		free(tmp);
	    } else {
		fprintf(toSender, "PGP NOPHRASE\n");
	    }
	    fflush(toSender);
	} else if (!strcmp("error", argv[1])) {
	    ClearPGPPass(NULL);
	    Tcl_DStringInit(&cmd);
	    Tcl_DStringAppend(&cmd, "RatPGPError", -1);
	    Tcl_DStringAppendElement(&cmd, argv[2]);
	    if (TCL_OK != Tcl_Eval(interp, Tcl_DStringValue(&cmd))) {
		fprintf(toSender, "PGP ABORT\n");
	    } else {
		fprintf(toSender, "PGP %s\n", interp->result);
	    }
	    fflush(toSender);
	    Tcl_DStringFree(&cmd);
	}
    } else if (!strcmp(argv[0], "SENT")) {
	id = atoi(argv[1]);
	deferred = atoi(argv[2]);
    }
    if (!strcmp(argv[0], "SENT") || !strcmp(argv[0], "FAILED")) {
	if (deferred) {
	    tmp = Tcl_GetVar(interp, "ratDeferred", TCL_GLOBAL_ONLY);
	    sprintf(buf, "%d", atoi(tmp)-1);
	    Tcl_SetVar(interp, "ratDeferred", buf, TCL_GLOBAL_ONLY);
	}
	if (id == sendSequence-1) {
	    Tcl_SetVar(interp, "ratSenderSending", "0", TCL_GLOBAL_ONLY);
	}
	for (smPtrPtr=&sentMsg; *smPtrPtr && (*smPtrPtr)->id != id;
		smPtrPtr = &(*smPtrPtr)->nextPtr);
	if (*smPtrPtr) {
	    smPtr = *smPtrPtr;
	    *smPtrPtr = smPtr->nextPtr;
	    free(smPtr);
	}
    }
    free(argv);
}

/*
 *----------------------------------------------------------------------
 *
 * RatGetEncoding --
 *
 *	See ../doc/interface
 *
 * Results:
 *      The return value is normally TCL_OK and the result can be found
 *      in interp->result. If something goes wrong TCL_ERROR is returned
 *      and an error message will be left in interp->result.
 *
 * Side effects:
 *      The file passed as argument is read.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatGetEncoding(ClientData dummy, Tcl_Interp *interp, int argc, char *argv[])
{
    FILE *fp;
    int seen8bit = 0;
    int seenZero = 0;
    int c;

    if (argc != 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" filename\"", (char *) NULL);
	return TCL_ERROR;
    }

    if (NULL == (fp = fopen(argv[1], "r"))) {
	RatLog(interp, RAT_ERROR, "Failed to open file", 0);
	Tcl_SetResult(interp, "binary", TCL_STATIC);
	return TCL_OK;
    }

    while (c = getc(fp), !feof(fp)) {
	if (0 == c) {
	    seenZero = 1;
	    break;
	} else if (0x80 & c) {
	    seen8bit = 1;
	}
    }
    if (seenZero) {
	Tcl_SetResult(interp, "binary", TCL_STATIC);
    } else if (seen8bit) {
	Tcl_SetResult(interp, "8bit", TCL_STATIC);
    } else {
	Tcl_SetResult(interp, "7bit", TCL_STATIC);
    }

    fclose(fp);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RatCleanup --
 *
 *	See ../doc/interface
 *
 * Results:
 *      The return value is always TCL_OK.
 *
 * Side effects:
 *      The database is closed.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatCleanup(ClientData dummy, Tcl_Interp *interp, int argc, char *argv[])
{
    RatDbClose();
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RatTildeSubst --
 *
 *	See ../doc/interface
 *
 * Results:
 *      A standard tcl result.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatTildeSubst(ClientData dummy, Tcl_Interp *interp, int argc, char *argv[])
{
    Tcl_DString buffer;
    char *expandedName;

    if (argc != 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" filename\"", (char *) NULL);
	return TCL_ERROR;
    }

    expandedName = Tcl_TranslateFileName(interp, argv[1], &buffer);
    Tcl_SetResult(interp, expandedName, TCL_VOLATILE);
    Tcl_DStringFree(&buffer);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RatTime --
 *
 *	See ../doc/interface
 *
 * Results:
 *      A standard tcl result.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatTime(ClientData dummy, Tcl_Interp *interp, int argc, char *argv[])
{
    time_t goal;

    if (argc > 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" [+days]\"", (char *) NULL);
	return TCL_ERROR;
    }

    goal = time(NULL);
    if (argc == 2) {
	goal += atoi(&argv[1][1])*24*60*60;
    }
    sprintf(interp->result, "%d", (int)goal);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RatSearch --
 *
 *	Does a case insensitive search of a string.
 *
 * Results:
 *      Returns 1 if the searchFor string is found in the searchIn string
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatSearch(char *searchFor, char *searchIn)
{
    static char *buf = NULL;	/* Buffer used to hold lowercase version */
    static int bufLength = 0;	/* Length of static buffer */
    int i, j, lengthFor, lengthIn;

    for (i=0; searchFor[i]; i++) {
	if (i >= bufLength) {
	    bufLength += 16;
	    buf = REALLOC(buf, bufLength);
	}
	if (isupper(searchFor[i])) {
	    buf[i] = tolower(searchFor[i]);
	} else {
	    buf[i] = searchFor[i];
	}
    }
    buf[i] = '\0';
    lengthFor = i;
    lengthIn = strlen(searchIn);
    for (i = 0; i <= lengthIn-lengthFor; i++) {
	j = 0;
	while (((isupper(searchIn[i+j]) && buf[j] == tolower(searchIn[i+j])) ||
		    buf[j] == searchIn[i+j]) && buf[j]) {
	    j++;
	}
	if (!buf[j]) {
	    return 1;
	}
    }
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * RatLock --
 *
 *	See ../doc/interface
 *
 * Results:
 *      A standard tcl result.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatLock(ClientData dummy, Tcl_Interp *interp, int argc, char *argv[])
{
    char *value, *correct;
    int i;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" variable ...\"", (char *) NULL);
	return TCL_ERROR;
    }

    for (i=1; i<argc;i++) {
	value = Tcl_GetVar(interp, argv[i], TCL_GLOBAL_ONLY);
	if (value) {
	    correct = cpystr(value);
	} else {
	    correct = cpystr("");
	}
	Tcl_TraceVar(interp, argv[i], 
		TCL_GLOBAL_ONLY | TCL_TRACE_WRITES | TCL_TRACE_UNSETS,
		RatReject, (ClientData)correct);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RatReject --
 *
 *	See ../doc/interface
 *
 * Results:
 *      A standard tcl result.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

static char*
RatReject(ClientData clientData, Tcl_Interp *interp, char *name1,
	char *name2, int flags)
{
    char *correct = (char*)clientData;

    if (flags & TCL_INTERP_DESTROYED) {
	ckfree(correct);
	return NULL;
    }
    if (flags & TCL_TRACE_DESTROYED) {
	Tcl_TraceVar2(interp, name1, name2,
		TCL_GLOBAL_ONLY | TCL_TRACE_WRITES | TCL_TRACE_UNSETS,
		RatReject, (ClientData)correct);
    }
    return "Variable is locked";
}

/*
 *----------------------------------------------------------------------
 *
 * RatIsLocked --
 *
 *	See ../doc/interface
 *
 * Results:
 *      A standard tcl result.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatIsLocked(ClientData dummy, Tcl_Interp *interp, int argc, char *argv[])
{
    if (argc != 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" variable\"", (char *) NULL);
	return TCL_ERROR;
    }
    if (Tcl_VarTraceInfo(interp, argv[1], TCL_GLOBAL_ONLY, RatReject, NULL)) {
	Tcl_SetResult(interp, "1", TCL_STATIC);
    } else {
	Tcl_SetResult(interp, "0", TCL_STATIC);
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RatType --
 *
 *	See ../doc/interface for a descriptions of arguments and result.
 *
 *	The algorithm is to first determine if the file exists and its
 *	encoding, then run the file command on it and try to match the
 *	result agains the typetable. If we don't find any match the type
 *	defaults to application/octet-stream.
 *
 * Results:
 *      A standard tcl result.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatType(ClientData dummy, Tcl_Interp *interp, int argc, char *argv[])
{
    int listArgc, elemArgc;
    char *cmdArgv[2], buf[1024], **listArgv, **elemArgv;
    char *encodingName, *fileType;
    RatEncoding encoding;
    Tcl_Channel channel;
    unsigned char c;
    int length, i;
    FILE *fp;

    if (argc != 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" filename\"", (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Determine encoding
     */
    if (NULL == (fp = fopen(argv[1], "r"))) {
	Tcl_AppendResult(interp, "error opening file \"", argv[1],
		"\": ", Tcl_PosixError(interp), (char *) NULL);
	return TCL_ERROR;
    }
    encoding = RAT_7BIT;
    length = 0;
    while (c = (unsigned char)getc(fp), !feof(fp)) {
	if ('\0' == c) {
	    encoding = RAT_BINARY;
	    break;
	}
	if ('\n' == c) {
	    length = 0;
	} else {
	    if (++length == 1024) {
		encoding = RAT_BINARY;
		break;
	    }
	}
	if (c & 0x80) {
	    encoding = RAT_8BIT;
	}
    }
    fclose(fp);
    switch(encoding) {
    case RAT_7BIT:   encodingName = "7bit";   break;
    case RAT_8BIT:   encodingName = "8bit";   break;
    case RAT_BINARY: encodingName = "binary"; break;
    case RAT_BASE64:	/* fallthrough */
    case RAT_QP:	/* fallthrough */
    case RAT_UNKOWN:	/* fallthrough */
    default:		/* fallthrough */
	fprintf(stderr, "This can't happen RatType() %d\n", encoding);
	encodingName = NULL;
	break;
    }

    /*
     * Run the "file" command.
     */
    cmdArgv[0] = "file";
    cmdArgv[1] = argv[1];
    if (!(channel = Tcl_OpenCommandChannel(interp, 2, cmdArgv, TCL_STDOUT))) {
	return TCL_ERROR;
    }
    length = Tcl_Read(channel, buf, sizeof(buf)-1);
    buf[length] = '\0';
    Tcl_Close(interp, channel);
    fileType = strchr(buf, ':')+1;
    Tcl_SplitList(interp, Tcl_GetVar2(interp, "option", "typetable",
	    TCL_GLOBAL_ONLY), &listArgc, &listArgv);
    for (i=0; i<listArgc; i++) {
	Tcl_SplitList(interp, listArgv[i], &elemArgc, &elemArgv);
	if (Tcl_StringMatch(fileType, elemArgv[0])) {
	    Tcl_ResetResult(interp);
	    sprintf(interp->result, "%s %s", elemArgv[1], encodingName);
	    ckfree(elemArgv);
	    break;
	}
	ckfree(elemArgv);
    }
    ckfree(listArgv);
    if (i == listArgc) {
	Tcl_ResetResult(interp);
	sprintf(interp->result, "application/octet-stream %s", encodingName);
    }

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RatTclPuts --
 *
 *	A version of the unix puts which converts CRLF to the local
 *	newline convention.
 *
 * Results:
 *      Always returns 1L.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

long
RatTclPuts(void *stream_x, char *string)
{
    Tcl_Channel channel = (Tcl_Channel)stream_x;
    char *p;

    for(p = string; *p; p++) {
	if (*p=='\015' && *(p+1)=='\012') {
	    if (-1 == Tcl_Write(channel, "\n", 1)) {
		return 0;
	    }
	    p++;
	} else {
	    if (-1 == Tcl_Write(channel, p, 1)) {
		return 0;
	    }
	}
    }
    return(1L);                                 /* T for c-client */
}

/*
 *----------------------------------------------------------------------
 *
 * RatStringPuts --
 *
 *	A version of the unix puts which converts CRLF to the local
 *	newline convention, and instead of storing into a file we
 *	append the data to an Tcl_DString.
 *
 * Results:
 *      Always returns 1L.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

long
RatStringPuts(void *stream_x, char *string)
{
    Tcl_DString *dsPtr = (Tcl_DString*)stream_x;
    char *p;

    for (p = string; *p; p++) {
      if (*p=='\015' && *(p+1)=='\012') {
	  Tcl_DStringAppend(dsPtr, "\n", 1);
	  p++;
      } else {
	  Tcl_DStringAppend(dsPtr, p, 1);
      }
    }

    return(1L);                                 /* T for c-client */
}

/*
 *----------------------------------------------------------------------
 *
 * RatInitDelaySoutr --
 *
 *	Initializes the buffer for RatDelaySoutr by clearing it.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Modifies the ratDelayBuffer array.
 *
 *
 *----------------------------------------------------------------------
 */
void
RatInitDelaySoutr(void)
{
    ratDelayBuffer[0] = '\0';
    ratDelayBuffer[1] = '\0';
    ratDelayBuffer[2] = '\0';
}

/*
 *----------------------------------------------------------------------
 *
 * RatDelaySoutr --
 *
 *	A output function to use with rfc822_output that writes to
 *	a file destriptor. This function is special in this that it
 *	always delay writing the last two characters. This allows one to
 *	filter final newlines.
 *
 * Results:
 *      Always returns 1L.
 *
 * Side effects:
 *      Modifies the ratDelayBuffer array.
 *
 *
 *----------------------------------------------------------------------
 */

long
RatDelaySoutr(void *stream_x, char *string)
{
    int len1, len2;
    len1 = strlen(ratDelayBuffer);
    len2 = strlen(string);

    if (len1+len2 <= 2) {
	strcat(ratDelayBuffer, string);
	return 1;
    }
    write((int)stream_x, ratDelayBuffer, len1);
    write((int)stream_x, string, len2-2);
    ratDelayBuffer[0] = string[len2-2];
    ratDelayBuffer[1] = string[len2-1];
    return 1;
}


/*
 *----------------------------------------------------------------------
 *
 * RatParseMsg --
 *
 *	Parses the message given as argument into an MESSAGE structure.
 *	The data at message is used in place so it may not be freed
 *	before the MESSAGE structure is freed.
 *
 * Results:
 *      Returns a pointer to a newly allocated MESSAGE structure
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

MESSAGE*
RatParseMsg(Tcl_Interp *interp, char *message)
{
    int length;		/* Length of header */
    int bodyOffset = 0;	/* Offset of body from start of header */
    MESSAGE *msgPtr;	/* Pointer to message to return */
    STRING bodyString;	/* Body data */

    for (length = 0; message[length]; length++) {
	if (message[length] == '\n' && message[length+1] == '\n') {
	    length++;
	    bodyOffset = length+1;
	    break;
	}
	if (message[length]=='\r' && message[length+1]=='\n'
		&& message[length+2]=='\r' && message[length+3]=='\n') {
	    length += 2;
	    bodyOffset = length+2;
	    break;
	}
    }
    msgPtr = (MESSAGE*)ckalloc(sizeof(MESSAGE));
    msgPtr->text.text.data = message;
    msgPtr->text.text.size = strlen(message);
    msgPtr->text.offset = bodyOffset;
    INIT(&bodyString, mail_string, (void*) (message+bodyOffset),
	    strlen(message)-bodyOffset);
    rfc822_parse_msg(&msgPtr->env, &msgPtr->body, message, length, &bodyString,
	    currentHost, NIL);
    if (msgPtr->body->type != TYPEMULTIPART 
	    && msgPtr->body->type != TYPEMESSAGE) {
	msgPtr->body->contents.text.data = cpystr(message+bodyOffset);
	msgPtr->body->contents.text.size = strlen(message+bodyOffset);
    }
    return msgPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * RatDSE --
 *
 *	See ../doc/interface for a descriptions of arguments and result.
 *
 * Results:
 *      A standard tcl result and the requested number is left in the
 *	result string.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatDSE(ClientData dummy, Tcl_Interp *interp, int argc, char *argv[])
{
    sprintf(interp->result, "%d", RatDbDaysSinceExpire(interp));
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RatExpire --
 *
 *	See ../doc/interface for a descriptions of arguments and result.
 *
 * Results:
 *      A standard tcl result.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatExpire(ClientData dummy, Tcl_Interp *interp, int argc, char *argv[])
{
    if (argc != 3) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" inbox backupDir\"", (char *) NULL);
	return TCL_ERROR;
    }
    return RatDbExpire(interp, argv[1], argv[2]);
}

/*
 *----------------------------------------------------------------------
 *
 * RatIsEmpty --
 *
 *	Check if a string contains anything else than whitespace.
 *
 * Results:
 *	Returns null if the string contains other chars than whitespace.
 *	Otherwise non-null is returned.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatIsEmpty (char *string)
{
    while (string && *string && isspace(*string)) {
	string++;
    }
    if (string && *string) {
	return 0;
    }
    return 1;
}

/*
 *----------------------------------------------------------------------
 *
 * RatEncodingCompatCmd --
 *
 *	See ../doc/interface for a descriptions of arguments and result.
 *
 * Results:
 *	True if the encodings are compatibe otherwise false.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatEncodingCompatCmd(ClientData dummy, Tcl_Interp *interp, int argc,
	char *argv[])
{
    if (3 != argc) {
	Tcl_AppendResult(interp, "Usage: ", argv[0], " wanted avail",
		(char*) NULL);
	return TCL_ERROR;
    }
    if (RatEncodingCompat(interp, argv[1], argv[2])) {
	Tcl_SetResult(interp, "1", TCL_STATIC);
    } else {
	Tcl_SetResult(interp, "0", TCL_STATIC);
    }
    return TCL_OK;
}



/*
 *----------------------------------------------------------------------
 *
 * RatEncodingCompat --
 *
 *	See ../doc/interface for a descriptions of arguments and result.
 *
 * Results:
 *	True if the encodings are compatibe otherwise false.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatEncodingCompat(Tcl_Interp *interp, char *wanted, char *avail)
{
    if (!strcasecmp(wanted, avail)) {
	return 1;
    }
    if (!strcasecmp("us-ascii", wanted) && !strncasecmp("iso-8859-", avail, 9)){
	return 1;
    }
    return 0;
}


/*
 *----------------------------------------------------------------------
 *
 * RatLindex --
 *
 *	Get a specific entry of a list.
 *
 * Results:
 *	A pointer to a static area which contains the requested item.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

char*
RatLindex(Tcl_Interp *interp, char *list, int index)
{
    static char *item = NULL;
    static int itemsize = 0;
    char **argv;
    int argc;

    Tcl_SplitList(interp, list, &argc, &argv);

    if (index >= argc) {
	ckfree(argv);
	return NULL;
    }
    if (itemsize < (int)(strlen(argv[index])+1)) {
	itemsize = strlen(argv[index])+1;
	item = (char*)REALLOC(item, itemsize);
    }
    strcpy(item, argv[index]);
    ckfree(argv);
    return item;
}

/*
 *----------------------------------------------------------------------
 *
 * RatLL --
 *
 *	See ../doc/interface for a descriptions of arguments and result.
 *
 * Results:
 *      The length of the given line.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatLL(ClientData dummy, Tcl_Interp *interp, int argc, char *argv[])
{
    char *cPtr;
    int l;

    if (2 != argc) {
	Tcl_AppendResult(interp, "Usage: ", argv[0], " line", (char*) NULL);
	return TCL_ERROR;
    }

    for (l=0, cPtr = argv[1]; *cPtr; cPtr++) {
	if ('\t' == *cPtr) {
	    l += 8-l%8;
	} else {
	    l++;
	}
    }
    sprintf(interp->result, "%d", l);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * RatDbaseCheck --
 *
 *	See ../doc/interface for a descriptions of arguments and result.
 *
 * Results:
 *      A list of strings to display to the user.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatDbaseCheck(ClientData dummy, Tcl_Interp *interp, int argc, char *argv[])
{
    int fix;

    if (2 != argc || TCL_OK != Tcl_GetBoolean(interp, argv[1], &fix)) {
	Tcl_AppendResult(interp, "Usage: ", argv[0], " fix", (char*) NULL);
	return TCL_ERROR;
    }
    return RatDbCheck(interp, fix);
}


/*
 *----------------------------------------------------------------------
 *
 * RatHostWatcher --
 *
 *	A trace function that gets called when the user modifies any of
 *	the hostname options.
 *
 * Results:
 *      NULL.
 *
 * Side effects:
 *      The RatInitCurrent() function will be called.
 *
 *
 *----------------------------------------------------------------------
 */

static char*
RatHostWatcher(ClientData clientData, Tcl_Interp *interp, char *name1,
	       char *name2, int flags)
{
    RatInitCurrent(NULL, interp, NULL, NULL, 0);
    Tcl_Eval(interp, "CheckHostName");
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * RatFormatDateCmd --
 *
 *	See ../doc/interface for a descriptions of arguments and result.
 *
 * Results:
 *      A list of strings to display to the user.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatFormatDateCmd(ClientData dummy, Tcl_Interp *interp, int argc, char *argv[])
{
    if (7 != argc) {
	Tcl_AppendResult(interp, "Usage: ", argv[0], \
		" year month day hour min sec", (char*) NULL);
	return TCL_ERROR;
    }
    Tcl_SetResult(interp, RatFormatDate(interp, atoi(argv[2])-1, atoi(argv[3])),
	    TCL_VOLATILE);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * RatFormatDate --
 *
 *	Print the data in a short format.
 *
 * Results:
 *      A pointer to a static area.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

char*
RatFormatDate(Tcl_Interp *interp, int month, int day)
{
    static char buf[8];
    static char *months[12];
    static int initialized = 0;

    if (!initialized) {
	int i, argc;
	char **argv;

	Tcl_SplitList(interp,
		Tcl_GetVar2(interp, "t", "months", TCL_GLOBAL_ONLY),
		&argc, &argv);
	for (i=0; i<12; i++) {
	    months[i] = argv[i];
	}
	initialized = 1;
    }

    sprintf(buf, "%2d %s", day, months[month]);
    return buf;
}

/*
 *----------------------------------------------------------------------
 *
 * RatGetTimeZone --
 *
 *	Determines the current timezone.  The method varies wildly
 *	between different platform implementations, so its hidden in
 *	this function.
 *
 *	This function is shamelessy stolen from tcl8.0p2
 *
 * Results:
 *	The return value is the local time zone, measured in
 *	minutes away from GMT (-ve for east, +ve for west).
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
RatGetTimeZone(unsigned long currentTime)
{
    /*
     * Determine how a timezone is obtained from "struct tm".  If there is no
     * time zone in this struct (very lame) then use the timezone variable.
     * This is done in a way to make the timezone variable the method of last
     * resort, as some systems have it in addition to a field in "struct tm".
     * The gettimeofday system call can also be used to determine the time
     * zone.
     */
    
#if defined(HAVE_TM_TZADJ)
#   define TCL_GOT_TIMEZONE
    time_t      curTime = (time_t) currentTime;
    struct tm  *timeDataPtr = localtime(&curTime);
    int         timeZone;

    timeZone = timeDataPtr->tm_tzadj  / 60;
    if (timeDataPtr->tm_isdst) {
        timeZone += 60;
    }
    
    return timeZone;
#endif

#if defined(HAVE_TM_GMTOFF) && !defined (TCL_GOT_TIMEZONE)
#   define TCL_GOT_TIMEZONE
    time_t     curTime = (time_t) currentTime;
    struct tm *timeDataPtr = localtime(&curTime);
    int        timeZone;

    timeZone = -(timeDataPtr->tm_gmtoff / 60);
    if (timeDataPtr->tm_isdst) {
        timeZone += 60;
    }
    
    return timeZone;
#endif

#if defined(USE_DELTA_FOR_TZ)
#define TCL_GOT_TIMEZONE 1
    /*
     * This hack replaces using global var timezone or gettimeofday
     * in situations where they are buggy such as on AIX when libbsd.a
     * is linked in.
     */

    int timeZone;
    time_t tt;
    struct tm *stm;
    tt = 849268800L;      /*    1996-11-29 12:00:00  GMT */
    stm = localtime(&tt); /* eg 1996-11-29  6:00:00  CST6CDT */
    /* The calculation below assumes a max of +12 or -12 hours from GMT */
    timeZone = (12 - stm->tm_hour)*60 + (0 - stm->tm_min);
    return timeZone;  /* eg +360 for CST6CDT */
#endif

    /*
     * Must prefer timezone variable over gettimeofday, as gettimeofday does
     * not return timezone information on many systems that have moved this
     * information outside of the kernel.
     */
    
#if defined(HAVE_TIMEZONE_VAR) && !defined (TCL_GOT_TIMEZONE)
#   define TCL_GOT_TIMEZONE
    static int setTZ = 0;
    int        timeZone;

    if (!setTZ) {
        tzset();
        setTZ = 1;
    }

    /*
     * Note: this is not a typo in "timezone" below!  See tzset
     * documentation for details.
     */

    timeZone = timezone / 60;

    return timeZone;
#endif

#if !defined(NO_GETTOD) && !defined (TCL_GOT_TIMEZONE)
#   define TCL_GOT_TIMEZONE
    struct timeval  tv;
    struct timezone tz;
    int timeZone;

    gettimeofday(&tv, &tz);
    timeZone = tz.tz_minuteswest;
    if (tz.tz_dsttime) {
        timeZone += 60;
    }
    
    return timeZone;
#endif

#ifndef TCL_GOT_TIMEZONE
    /*
     * Cause compile error, we don't know how to get timezone.
     */
    error: autoconf did not figure out how to determine the timezone. 
#endif

}
