/*
 * ratFolder.c --
 *
 *      This file contains basic support code for the folder commands. Each
 *      folder type is created using an unique command. This command returns
 *      a folder handler, which when invoked calls the RatFolderCmd()
 *      procedure with a pointer to a RatFolderInfo structure as clientData.
 *
 * TkRat software and its included text is Copyright 1996,1997,1998
 * by Martin Forssn
 *
 * The full text of the legal notices is contained in the file called
 * COPYRIGHT, included with this distribution.
 */

#include "ratFolder.h"

/*
 * This structure is used to hold the data while sorting
 */
typedef struct SortData {
    char *subject;
    char *sender;
    time_t date;
    struct SortData *nextPtr;
} SortData;

/*
 * The different sort methods
 */
typedef enum {SORT_NONE, SORT_SUBJECT, SORT_SUBJDATE,
	      SORT_SENDER, SORT_SENDERDATE, SORT_DATE} SortOrder;

/*
 * The number of folders opened. This is used when making
 * the folder entities.
 */
static int numFolders = 0;

/*
 * Global variables used to controll the sorting functions
 */
static SortData *baseSortDataPtr;


static int RatFolderOpen(ClientData clientData,
	Tcl_Interp *interp, int argc, char *argv[]);
static int RatFolderCmd(ClientData clientData,
	Tcl_Interp *interp, int argc, char *argv[]);
static void RatFolderSort(Tcl_Interp *interp, RatFolderInfo *infoPtr);
static int RatFolderSortCompareDate(const void *arg1, const void *arg2);
static int RatFolderSortCompareSubject(const void *arg1, const void *arg2);
static int RatFolderSortCompareSender(const void *arg1, const void *arg2);
static char *RatFolderSubjCopy (char *subject);


/*
 *----------------------------------------------------------------------
 *
 * RatFolderInit --
 *
 *      Initializes the folder commands.
 *
 * Results:
 *      The return value is normally TCL_OK; if something goes wrong
 *	TCL_ERROR is returned and an error message will be left in
 *	interp->result.
 *
 * Side effects:
 *	The folder creation commands are created in interp. And the
 *	Std folder is initialized.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatFolderInit(Tcl_Interp *interp)
{
    RatInitMessages();
    if (TCL_OK != RatStdFolderInit(interp)) {
	return TCL_ERROR;
    }
    if (TCL_OK != RatDbFolderInit(interp)) {
	return TCL_ERROR;
    }
    Tcl_CreateCommand(interp, "RatOpenFolder", RatFolderOpen,
	    (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateCommand(interp, "RatParseExp", RatParseExpCmd, NULL, NULL);
    Tcl_CreateCommand(interp, "RatGetExp", RatGetExpCmd, NULL, NULL);
    Tcl_CreateCommand(interp, "RatFreeExp", RatFreeExpCmd, NULL, NULL);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * RatFolderOpen --
 *
 *      See the INTERFACE specification
 *
 * Results:
 *      The return value is normally TCL_OK and a foilder handle is left
 *	in interp->result; if something goes wrong TCL_ERROR is returned
 *	and an error message will be left in interp->result.
 *
 * Side effects:
 *	The folder creation commands are created in interp.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatFolderOpen(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
    RatFolderInfo *infoPtr;
    int i;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" prot [args]\"", (char *) NULL);
	return TCL_ERROR;
    }
    if (!strcmp(argv[1], "std")) {
	infoPtr = RatStdFolderCreate(interp, argc, argv);
    } else if (!strcmp(argv[1], "dbase")) {
	infoPtr = RatDbFolderCreate(interp, argc, argv);
    } else {
	Tcl_SetResult(interp,
		"Wrong protocol. Only 'std' and 'dbase' are supported",
		TCL_STATIC);
	return TCL_ERROR;
    }
    if (NULL == infoPtr) {
	return TCL_ERROR;
    }
    infoPtr->cmdName = ckalloc(16);
    infoPtr->allocated = infoPtr->number;
    infoPtr->msgCmdPtr = (char **) ckalloc(infoPtr->allocated*sizeof(char*));
    infoPtr->privatePtr = (ClientData**)ckalloc(
	    infoPtr->allocated*sizeof(ClientData));
    for (i=0; i<infoPtr->allocated; i++) {
	infoPtr->msgCmdPtr[i] = (char *) NULL;
	infoPtr->privatePtr[i] = (ClientData*) NULL;
    }
    infoPtr->presentationOrder = (int*) ckalloc(infoPtr->allocated*sizeof(int));
    infoPtr->hidden = (int*) ckalloc(infoPtr->allocated*sizeof(int));
    RatFolderSort(interp, infoPtr);
    sprintf(infoPtr->cmdName, "RatFolder%d", numFolders++);
    Tcl_CreateCommand(interp, infoPtr->cmdName, RatFolderCmd,
    	    (ClientData) infoPtr, (Tcl_CmdDeleteProc *) NULL);
    Tcl_SetResult(interp, infoPtr->cmdName, TCL_VOLATILE);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * RatFolderCmd --
 *
 *      Main folder entity procedure. This procedure implements the
 *	folder commands mentioned in ../doc/interface. In order to make
 *	this a tad easier it uses the procedures defined in the
 *	RatFolderInfo structure :-)
 *
 * Results:
 *      Depends on the input :-)
 *
 * Side effects:
 *	The specified folder may be modified.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatFolderCmd(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
    RatFolderInfo *infoPtr = (RatFolderInfo*) clientData;
    int length;
    char c;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" option ?arg?\"", (char *) NULL);
	return TCL_ERROR;
    }
    c = argv[1][0];
    length = strlen(argv[1]);
    if ((c == 'u') && (strncmp(argv[1], "update", length) == 0)
	    && (length > 1)) {
	int i, number, expunge, oldNumber, oldVisible, delta;

	if (argc != 3 || TCL_OK != Tcl_GetBoolean(interp, argv[2], &expunge)) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " update expunge\"", (char *) NULL);
	    return TCL_ERROR;
	}
	oldVisible = infoPtr->visible;
	oldNumber = infoPtr->number;
	number = (*infoPtr->updateProc)(infoPtr, interp, expunge);
	if (number < 0) {
	    RatLog(interp, RAT_WARN, interp->result, 0);
	    number = 0;
	} else {
	    delta = number - oldNumber;
	    infoPtr->number = number;
	    if (delta <0) {
		delta = 0;
	    }
	    if (infoPtr->number > infoPtr->allocated) {
		infoPtr->allocated = infoPtr->number;
		infoPtr->msgCmdPtr = (char **) REALLOC(infoPtr->msgCmdPtr,
			infoPtr->allocated*sizeof(char*));
		infoPtr->privatePtr = (ClientData**)REALLOC(infoPtr->privatePtr,
			infoPtr->allocated*sizeof(ClientData*));
		infoPtr->presentationOrder = (int *) REALLOC(
			infoPtr->presentationOrder,
			infoPtr->allocated*sizeof(int));
		infoPtr->hidden = (int *) REALLOC(infoPtr->hidden,
			infoPtr->allocated*sizeof(int));
	    }
	    for (i=infoPtr->number-delta; i<infoPtr->number; i++) {
		infoPtr->msgCmdPtr[i] = (char *) NULL;
		infoPtr->privatePtr[i] = (ClientData*) NULL;
	    }
	    RatFolderSort(interp, infoPtr);
	}
	delta = infoPtr->visible - oldVisible;
	Tcl_ResetResult(interp);
	sprintf(interp->result, "%d", (delta>0 ? delta : 0));
	return TCL_OK;

    } if ((c == 'c') && (strncmp(argv[1], "close", length) == 0)
	    && (length > 2)) {
	int i, ret;
	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " close\"", (char *) NULL);
	    return TCL_ERROR;
	}
	ckfree(infoPtr->name);
	for(i=0; i < infoPtr->number; i++) {
	    if (NULL != infoPtr->msgCmdPtr[i]) {
		(void)RatMessageDelete(interp, infoPtr->msgCmdPtr[i]);
	    }
	}
	ret = (*infoPtr->closeProc)(infoPtr, interp);
	(void)Tcl_DeleteCommand(interp, argv[0]);
	ckfree(infoPtr->cmdName);
	ckfree(infoPtr->msgCmdPtr);
	ckfree(infoPtr->privatePtr);
	ckfree(infoPtr->presentationOrder);
	ckfree(infoPtr->hidden);
	ckfree(infoPtr);
	return ret;

    } if ((c == 's') && (strncmp(argv[1], "setName", length) == 0)
	    && (length > 3)) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " setName name\"", (char *) NULL);
	    return TCL_ERROR;
	}
	ckfree(infoPtr->name);
	infoPtr->name = (char *) ckalloc(strlen(argv[2])+1);
	strcpy(infoPtr->name, argv[2]);
	return TCL_OK;

    } if ((c == 'i') && (strncmp(argv[1], "info", length) == 0)
	    && (length > 2)) {
	int infoArgc;
	char *infoArgv[3];
	char numberBuf[16], sizeBuf[16];
	char *list;

	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " info\"", (char *) NULL);
	    return TCL_ERROR;
	}
	sprintf(numberBuf, "%d", infoPtr->visible);
	sprintf(sizeBuf, "%d", infoPtr->size);
	infoArgv[0] = infoPtr->name;
	infoArgv[1] = numberBuf;
	infoArgv[2] = sizeBuf;
	infoArgc = 3;
	list = Tcl_Merge(infoArgc, infoArgv);
	Tcl_SetResult(interp, list, TCL_DYNAMIC);
	return TCL_OK;

    } if ((c == 'l') && (strncmp(argv[1], "list", length) == 0)
	    && (length > 1)) {
	ListExpression *exprPtr;
	Tcl_DString result;
	int i;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " list format\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (NULL == (exprPtr = RatParseList(argv[2]))) {
	    Tcl_SetResult(interp, "Illegal list format", TCL_STATIC);
	    return TCL_ERROR;
	}

	Tcl_DStringInit(&result);
	for (i=0; i < infoPtr->visible; i++) {
	    Tcl_DStringAppendElement(&result, RatDoList(interp, exprPtr,
		    infoPtr->infoProc, (ClientData)infoPtr, i));
	}
	RatFreeListExpression(exprPtr);
	Tcl_DStringResult(interp, &result);
	return TCL_OK;

    } if ((c == 'g') && (strcmp(argv[1], "get") == 0)) {
	int index;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " get index\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (TCL_OK != Tcl_GetInt(interp, argv[2], &index)) {
	    return TCL_ERROR;
	}
	if (index < 0 || index >= infoPtr->visible) {
	    Tcl_SetResult(interp, "Index is out of bounds", TCL_STATIC);
	    return TCL_ERROR;
	}
	if (NULL == infoPtr->msgCmdPtr[infoPtr->presentationOrder[index]]) {
	    infoPtr->msgCmdPtr[infoPtr->presentationOrder[index]] =
		    (*infoPtr->createProc)(infoPtr, interp,
		    infoPtr->presentationOrder[index]);
	}
	Tcl_SetResult(interp,
		infoPtr->msgCmdPtr[infoPtr->presentationOrder[index]],
		TCL_VOLATILE);
	return TCL_OK;

    } if ((c == 's') && (strncmp(argv[1], "setFlag", length) == 0)
	    && (length > 3)) {
	int index, value;
	RatFlag flag;

	if (argc != 5) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " setFlag index flag value\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (TCL_OK != Tcl_GetInt(interp, argv[2], &index)) {
	    return TCL_ERROR;
	}
	if (index < 0 || index >= infoPtr->visible) {
	    Tcl_SetResult(interp, "Index is out of bounds", TCL_STATIC);
	    return TCL_ERROR;
	}
	if (0 == strcmp(argv[3], "seen")) {
	    flag = RAT_SEEN;
	} else if (0 == strcmp(argv[3], "deleted")) {
	    flag = RAT_DELETED;
	} else if (0 == strcmp(argv[3], "flagged")) {
	    flag = RAT_FLAGGED;
	} else if (0 == strcmp(argv[3], "answered")) {
	    flag = RAT_ANSWERED;
	} else {
	    Tcl_SetResult(interp, "Unkown flag; should be one of seen, deleted, flagged, answered",
		     TCL_STATIC);
	    return TCL_ERROR;
	}
	if (TCL_OK != Tcl_GetBoolean(interp, argv[4], &value)) {
	    return TCL_ERROR;
	}

	(*infoPtr->setFlagProc)(infoPtr, interp,
		infoPtr->presentationOrder[index], flag, value);
	return TCL_OK;

    } if ((c == 'g') && (strncmp(argv[1], "getFlag", length) == 0)
	    && (length > 3)) {
	int index;
	RatFlag flag;

	if (argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " getFlag index flag\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (TCL_OK != Tcl_GetInt(interp, argv[2], &index)) {
	    return TCL_ERROR;
	}
	if (index < 0 || index >= infoPtr->visible) {
	    Tcl_SetResult(interp, "Index is out of bounds", TCL_STATIC);
	    return TCL_ERROR;
	}
	if (0 == strcmp(argv[3], "seen")) {
	    flag = RAT_SEEN;
	} else if (0 == strcmp(argv[3], "deleted")) {
	    flag = RAT_DELETED;
	} else if (0 == strcmp(argv[3], "flagged")) {
	    flag = RAT_FLAGGED;
	} else if (0 == strcmp(argv[3], "answered")) {
	    flag = RAT_ANSWERED;
	} else {
	    Tcl_SetResult(interp, "Unkown flag; should be one of seen, deleted, flagged, answered",
		     TCL_STATIC);
	    return TCL_ERROR;
	}

	Tcl_ResetResult(interp);
	sprintf(interp->result, "%d",
		(*infoPtr->getFlagProc)(infoPtr, interp,
		infoPtr->presentationOrder[index], flag));
	return TCL_OK;

    } if ((c == 'f') && (strncmp(argv[1], "flagged", length) == 0)
	    && (length > 1)) {
	RatFlag flag;
	char buf[64];
	int i;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " flagged flag\"", (char *) NULL);
	    return TCL_ERROR;
	}
	if (0 == strcmp(argv[2], "seen")) {
	    flag = RAT_SEEN;
	} else if (0 == strcmp(argv[2], "deleted")) {
	    flag = RAT_DELETED;
	} else if (0 == strcmp(argv[2], "flagged")) {
	    flag = RAT_FLAGGED;
	} else if (0 == strcmp(argv[2], "answered")) {
	    flag = RAT_ANSWERED;
	} else {
	    Tcl_SetResult(interp, "Unkown flag; should be one of seen, deleted, flagged, answered",
		     TCL_STATIC);
	    return TCL_ERROR;
	}

	Tcl_ResetResult(interp);
	for (i=0; i<infoPtr->visible; i++) {
	    if ((*infoPtr->getFlagProc)(infoPtr, interp,
		    infoPtr->presentationOrder[i], flag)) {
		sprintf(buf, "%d", i);
		Tcl_AppendElement(interp, buf);
	    }
	}
	return TCL_OK;

    } if ((c == 'i') && (strncmp(argv[1], "insert", length) == 0)
	    && (length > 2)) {
	Tcl_CmdInfo cmdInfo;
	int i, result;

	if (argc < 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " insert message ?message ...?\"", (char *) NULL);
	    return TCL_ERROR;
	}

	if (argc-2 > infoPtr->allocated-infoPtr->number) {
	    infoPtr->allocated += argc;
	    infoPtr->msgCmdPtr = (char**)REALLOC(infoPtr->msgCmdPtr,
	    	    infoPtr->allocated*sizeof(char*));
	    infoPtr->privatePtr = (ClientData**)REALLOC(infoPtr->privatePtr,
	    	    infoPtr->allocated*sizeof(ClientData*));
	    infoPtr->presentationOrder = (int*)REALLOC(
		    infoPtr->presentationOrder, infoPtr->allocated*sizeof(int));
	}
	for(i=2; i<argc; i++) {
	    if (0 == Tcl_GetCommandInfo(interp, argv[i], &cmdInfo)
		    || NULL == cmdInfo.clientData) {
		Tcl_AppendResult(interp, "error \"", argv[i],
			"\" is not a valid message command", (char *) NULL);
		return TCL_ERROR;
	    }
	}
	result = (*infoPtr->insertProc)(infoPtr, interp, argc-2, &argv[2]);
	RatFolderSort(interp, infoPtr);
	return result;

    } if ((c == 't') && (strncmp(argv[1], "type", length) == 0)
	    && (length > 2)) {
	Tcl_SetResult(interp, infoPtr->type, TCL_STATIC);
	return TCL_OK;

    } if ((c == 'f') && (strncmp(argv[1], "find", length) == 0)
	    && (length > 1)) {
	int msgNo, i;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " find message\"", (char *) NULL);
	    return TCL_ERROR;
	}

	strcpy(interp->result, "-1");
	for (msgNo=0; msgNo < infoPtr->number; msgNo++) {
	    if (infoPtr->msgCmdPtr[msgNo]
	    	    && !strcmp(infoPtr->msgCmdPtr[msgNo], argv[2])) {
		for (i=0; msgNo != infoPtr->presentationOrder[i]; i++);
	        sprintf(interp->result, "%d", i);
		break;
	    }
	}
	return TCL_OK;

    } if ((c == 'm') && (strncmp(argv[1], "match", length) == 0)
	    && (length > 1)) {
	Tcl_DString ds;
	int i, expId;
	char buf[8];

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " match expId\"", (char *) NULL);
	    return TCL_ERROR;
	}

	Tcl_DStringInit(&ds);
	expId = atoi(argv[2]);

	for (i=0; i<infoPtr->number; i++) {
	    if (RatExpMatch(interp, expId, infoPtr->infoProc,
		    (ClientData)infoPtr, i)) {
		sprintf(buf, "%d", i);
		Tcl_DStringAppendElement(&ds, buf);
	    }
	}
	Tcl_DStringResult(interp, &ds);
	return TCL_OK;

    } else {
	Tcl_AppendResult(interp, "bad option \"", argv[1],
		"\": must be update, close, setName, info, list, ",
		"get, setFlag, getFlag, flagged, expunge or insert",
		(char *) NULL);
	return TCL_ERROR;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * RatFolderSort --
 *
 *      Sorts the folder according to the users wishes. The user may
 *	communicates their will via the folder_sort variable. Currently
 *	The following methods are implemented:
 *	  subjectonly		- Alphabetically on subject
 *	  sender		- Alphabetically on sender name
 *	  folder		- Sorts in native folder order
 *	  reverseFolder		- The reverse of the above
 *	  date			- By date sent
 *	  reverseDate		- By reverse date sent
 *	  subject		- Group messages with the same subject
 *				  and sort the groups by the earliest date
 *				  in each group.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The presentation order member of the RatFolderInfo structure
 *	is initialized. The size of the folder is updated.
 *
 *
 *----------------------------------------------------------------------
 */

static void
RatFolderSort(Tcl_Interp *interp, RatFolderInfo *infoPtr)
{
    char *sortMethod=Tcl_GetVar2(interp,"option","folder_sort",TCL_GLOBAL_ONLY);
    int i, j, k, snarf, numParm, seen, *tmpPtr,
	*p=infoPtr->presentationOrder,
	reverse=0, needDate=0, needSubject=0, needSender=0,
	*uniqList, uniqListUsed, *subList, *lengthList;
    SortOrder order = SORT_NONE;
    SortData *dataPtr, *dPtr;
    char *s, **pArgv;

    Tcl_GetBoolean(interp, Tcl_GetVar2(interp, "option", "dsn_snarf_reports",
	    TCL_GLOBAL_ONLY), &snarf);

    if (!strcmp(sortMethod, "subject")) {
	order = SORT_SUBJDATE;
	needDate = 1;
	needSubject = 1;
    } else if (!strcmp(sortMethod, "sender")) {
	order = SORT_SENDERDATE;
	needDate = 1;
	needSender = 1;
    } else if (!strcmp(sortMethod, "subjectonly")) {
	order = SORT_SUBJECT;
	needSubject = 1;
    } else if (!strcmp(sortMethod, "senderonly")) {
	order = SORT_SENDER;
	needSender = 1;
    } else if (!strcmp(sortMethod, "date")) {
	needDate = 1;
	order = SORT_DATE;
    } else if (!strcmp(sortMethod, "folder")) {
	order = SORT_NONE;
    } else if (!strcmp(sortMethod, "reverseFolder")) {
	order = SORT_NONE;
	reverse = 1;
    } else if (!strcmp(sortMethod, "reverseDate")) {
	needDate = 1;
	order = SORT_DATE;
	reverse = 1;
    }

    dataPtr = (SortData*)malloc(infoPtr->number*sizeof(*dataPtr));
    infoPtr->size = 0;
    for (i=0; i<infoPtr->number; i++) {
	infoPtr->hidden[i] = 0;
	infoPtr->presentationOrder[i] = i;
	s = (*infoPtr->infoProc)(interp, (ClientData)infoPtr,RAT_FOLDER_SIZE,i);
	if (s) {
	    infoPtr->size += atoi(s);
	}
	s = (*infoPtr->infoProc)(interp, (ClientData)infoPtr,RAT_FOLDER_TYPE,i);
	if (s && !strcasecmp(s, "multipart/report")) {
	    s = (*infoPtr->infoProc)(interp, (ClientData)infoPtr,
		RAT_FOLDER_PARAMETERS, i);
	    if (!s) {
		continue;
	    }
	    Tcl_SplitList(interp, s, &numParm, &pArgv);
	    for (j=0; j<numParm; j++) {
		if (!strcasecmp(pArgv[j], "report-type delivery-status")) {
		    /*
		     * This is a DSN; call the proper authorities
		     */
		    seen=(*infoPtr->getFlagProc)(infoPtr, interp, i, RAT_SEEN);
		    if (!infoPtr->msgCmdPtr[i]) {
			infoPtr->msgCmdPtr[i] =
				(*infoPtr->createProc)(infoPtr, interp, i);
		    }
		    if (RatDSNHandle(interp, infoPtr->msgCmdPtr[i]) && snarf) {
			infoPtr->hidden[i] = 1;
			(*infoPtr->setFlagProc)(infoPtr,interp,i,RAT_DELETED,1);
		    }
		    (*infoPtr->setFlagProc)(infoPtr,interp,i,RAT_SEEN,seen);
		    break;
		}
	    }
	    ckfree(pArgv);
	}
	if (needSubject) {
	    dataPtr[i].subject = RatFolderSubjCopy(
		    (*infoPtr->infoProc)(interp, (ClientData)infoPtr,
		    RAT_FOLDER_SUBJECT, i));
	    lcase(dataPtr[i].subject);
	}
	if (needSender) {
	    dataPtr[i].sender = cpystr((*infoPtr->infoProc)(interp,
		    (ClientData)infoPtr, RAT_FOLDER_NAME, i));
	    lcase(dataPtr[i].sender);
	}
	if (needDate) {
	    dataPtr[i].date = atoi((*infoPtr->infoProc)(interp,
			(ClientData)infoPtr, RAT_FOLDER_DATE_N, i));
	}
    }

    baseSortDataPtr = dataPtr;
    switch (order) {
    case SORT_NONE:
	for (i=0; i<infoPtr->number; i++) {
	    p[i] = i;
	}
	break;
    case SORT_SUBJDATE:
    case SORT_SENDERDATE:
	/*
	 * This algorithm is complicated:
	 * - First we build a list of unique subjects in uniqList. Each entry
	 *   in this list contains the index of the first message with this
	 *   subject. The messages are linked with the nextPtr field in
	 *   the SortData structs.
	 * - Then we sort each found subject. This is done by placing the
	 *   indexes of the messages in subList. And sort that. When it
	 *   is sorted we rebuild the subject chains via the nextPtr;
	 * - After that we sort the first message in each subject. This is done
	 *   by reusing the uniqList. We replace each entry in it with a pointer
	 *   to the first entry in the set. Actually we do this in the preceding
	 *   step. Then we sort this list.
	 * - Finally we build to result array.
	 */
	uniqList = (int*)malloc(2*infoPtr->number*sizeof(*uniqList));
	subList = &uniqList[infoPtr->number];
	lengthList = &uniqList[2*infoPtr->number];
	for (i=uniqListUsed=0; i<infoPtr->number; i++) {
	    for (j=0; j<uniqListUsed; j++) {
		if (order == SORT_SUBJDATE ? !strcmp(dataPtr[i].subject,
			dataPtr[uniqList[j]].subject) : !strcmp(
			dataPtr[i].sender, dataPtr[uniqList[j]].sender)) {
		    dataPtr[i].nextPtr = dataPtr[uniqList[j]].nextPtr;
		    dataPtr[uniqList[j]].nextPtr = &dataPtr[i];
		    break;
		}
	    }
	    if (j == uniqListUsed) {
		dataPtr[i].nextPtr = NULL;
		uniqList[uniqListUsed++] = i;
	    }
	}
	for (i=0; i<uniqListUsed; i++) {
	    if (NULL != dataPtr[uniqList[i]].nextPtr) {
		for (j = 0, dPtr = &dataPtr[uniqList[i]]; dPtr;
			dPtr = dPtr->nextPtr) {
		    subList[j++] = dPtr-dataPtr;
		}
		qsort((void*)subList, j, sizeof(int), RatFolderSortCompareDate);
		for (k=0; k<j-1; k++) {
		    dataPtr[subList[k]].nextPtr = &dataPtr[subList[k+1]];
		}
		dataPtr[subList[k]].nextPtr = NULL;
		uniqList[i] = subList[0];
	    }
	}
	qsort((void*)uniqList, uniqListUsed, sizeof(int),
		RatFolderSortCompareDate);
	for (i=k=0; i<uniqListUsed; i++) {
	    for (dPtr = &dataPtr[uniqList[i]]; dPtr; dPtr = dPtr->nextPtr) {
		p[k++] = dPtr-baseSortDataPtr;
	    }
	}
	free(uniqList);
	break;
    case SORT_SENDER:
	for (i=0; i<infoPtr->number; i++) {
	    p[i] = i;
	}
	qsort((void*)p, infoPtr->number,sizeof(int),RatFolderSortCompareSender);
	break;
    case SORT_SUBJECT:
	for (i=0; i<infoPtr->number; i++) {
	    p[i] = i;
	}
	qsort((void*)p,infoPtr->number,sizeof(int),RatFolderSortCompareSubject);
	break;
    case SORT_DATE:
	for (i=0; i<infoPtr->number; i++) {
	    p[i] = i;
	}
	qsort((void*)p, infoPtr->number, sizeof(int), RatFolderSortCompareDate);
	break;
    }

    if (reverse) {
	tmpPtr = (int*)malloc(infoPtr->number*sizeof(int));
	for (i=infoPtr->number-1, j=0; i >= 0; i--) {
	    if (!infoPtr->hidden[p[i]]) {
		tmpPtr[j++] = p[i];
	    }
	}
	memcpy(p, tmpPtr, j*sizeof(int));
	free(tmpPtr);
    } else {
	for (i=j=0; i < infoPtr->number; i++) {
	    if (!infoPtr->hidden[p[i]]) {
		p[j++] = p[i];
	    }
	}
    }
    infoPtr->visible = j;

    /*
     * Cleanup dataPtr
     */
    for (i=0; i<infoPtr->number; i++) {
	if (!infoPtr->hidden[i]) {
	    if (needSubject) {
		free(dataPtr[i].subject);
	    }
	    if (needSender) {
		free(dataPtr[i].sender);
	    }
	}
    }
    free(dataPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * RatFolderSortCompar* --
 *
 *	This is the comparison functions used by RatFolderSort. They
 *	expect to get pointers to integers as argumens. The integers
 *	pointed at are indexes into a list of SortData structs which can be
 *	found at the address in baseSortDataPtr.
 *
 * Results:
 *	An integers describing the order of the compared objects.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatFolderSortCompareDate(const void *arg1, const void *arg2)
{
    return baseSortDataPtr[*((int*)arg1)].date
	    - baseSortDataPtr[*((int*)arg2)].date;
}

static int
RatFolderSortCompareSubject(const void *arg1, const void *arg2)
{
    return strcmp(baseSortDataPtr[*((int*)arg1)].subject,
		  baseSortDataPtr[*((int*)arg2)].subject);
}

static int
RatFolderSortCompareSender(const void *arg1, const void *arg2)
{
    return strcmp(baseSortDataPtr[*((int*)arg1)].sender,
		  baseSortDataPtr[*((int*)arg2)].sender);
}


/*
 *----------------------------------------------------------------------
 *
 * RatFolderSubjCopy --
 *
 * Copy a subject line and remove certain constructs (the re:).
 *
 * Results:
 *	Pointer to a new copy of the subject. It is the callers
 *	responsibility to free this string later.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

static char*
RatFolderSubjCopy(char *subject)
{
    char *s = subject, *ret;
    int len;

    if (s) {
	/*
	 * We first try to find the start of the actual text (i.e. without any
	 * leading Re:'s and whitespaces. Then we find how long the text is
	 * (ignore trailing whitespaces).
	 */
	while (*s) {
	    while (*s && isspace(*s)) s++;
	    if (!strncasecmp(s, "re", 2) && (':' == s[2] || isspace(s[2]))) {
		s += 2;
		if (*s == ':') {
		    s++;
		}
	    } else {
		break;
	    }
	}
	len = strlen(s);
	while (len && isspace(s[len-1])) len--;

	ret = (char*)malloc(len+1);
	strncpy(ret, s, len);
	ret[len] = '\0';
    } else {
	ret = (char*)malloc(1);
	ret[0] = '\0';
    }
    return ret;
}

/*
 *----------------------------------------------------------------------
 *
 * RatGetMsgInfo --
 *
 * Gets info from message structure and formats it somewhat. None of the
 * informations items needs both the messagePtr and the eltPtr. The following
 * table describes which needs which:
 *
 *  Info			msgPtr	envPtr	bodyPtr	eltPtr	size
 *  RAT_FOLDER_SUBJECT		-	needed	-	-	-
 *  RAT_FOLDER_NAME		needed	needed	-	-	-
 *  RAT_FOLDER_MAIL_REAL	-	needed	-	-	-
 *  RAT_FOLDER_MAIL		needed	needed	-	-	-
 *  RAT_FOLDER_NAME_RECIPIENT	needed	needed	-	-	-
 *  RAT_FOLDER_MAIL_RECIPIENT	needed	needed	-	-	-
 *  RAT_FOLDER_SIZE		-	-	-	-	needed
 *  RAT_FOLDER_SIZE_F		-	-	-	-	needed
 *  RAT_FOLDER_DATE_F		-	needed	-	needed	-
 *  RAT_FOLDER_DATE_N		-	needed	-	needed	-
 *  RAT_FOLDER_STATUS		    [not supported ]
 *  RAT_FOLDER_TYPE		-	-	needed	-	-
 *  RAT_FOLDER_PARAMETERS	-	-	needed	-	-
 *  RAT_FOLDER_FLAGS		-	-	-	needed	-
 *  RAT_FOLDER_INDEX		    [not supported ]
 *
 * Results:
 *	A pointer to a string which is valid at least until the next call
 *	to this structure.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

char *
RatGetMsgInfo(Tcl_Interp *interp, RatFolderInfoType type, MessageInfo *msgPtr,
	ENVELOPE *envPtr, BODY *bodyPtr, MESSAGECACHE *eltPtr, int size)
{
    static Tcl_DString ds;
    static int initialized = 0;
    time_t time, zonediff;
    MESSAGECACHE dateElt, *dateEltPtr;
    PARAMETER *parmPtr;
    ADDRESS *adrPtr;
    struct tm tm;

    if (!initialized) {
	Tcl_DStringInit(&ds);
	initialized = 1;
    }

    switch (type) {
	case RAT_FOLDER_SUBJECT:
	    return envPtr->subject;
	case RAT_FOLDER_MAIL_REAL:
	    Tcl_DStringSetLength(&ds, 0);
	    for (adrPtr = envPtr->from; adrPtr; adrPtr = adrPtr->next) {
		if (adrPtr->mailbox && adrPtr->host) {
		    break;
		}
	    }
	    if (!adrPtr) {
		return "";
	    }
	    Tcl_DStringAppend(&ds, RatAddressMail(adrPtr), -1);
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_NAME:
	    if (!envPtr->from) {
		return "";
	    }
	    if (RAT_ISME_YES == msgPtr->fromMe
		    || (RAT_ISME_UNKOWN == msgPtr->fromMe
			&& RatAddressIsMe(interp, envPtr->from, 1))) {
		msgPtr->fromMe = RAT_ISME_YES;
		if (envPtr->to && envPtr->to->personal) {
		    Tcl_DStringSetLength(&ds, 0);
		    Tcl_DStringAppend(&ds,
			    Tcl_GetVar2(interp, "t", "to", TCL_GLOBAL_ONLY),-1);
		    Tcl_DStringAppend(&ds, ": ", 2);
		    Tcl_DStringAppend(&ds, envPtr->to->personal, -1);
		    return Tcl_DStringValue(&ds);
		}
	    } else {
		msgPtr->fromMe = RAT_ISME_NO;
		if (envPtr->from->personal) {
		    return envPtr->from->personal;
		}
	    }
	    /* fallthrough */
	case RAT_FOLDER_MAIL:
	    Tcl_DStringSetLength(&ds, 0);
	    if (RAT_ISME_YES == msgPtr->fromMe
		    || (RAT_ISME_UNKOWN == msgPtr->fromMe
			&& RatAddressIsMe(interp, envPtr->from, 1))) {
		msgPtr->fromMe = RAT_ISME_YES;
		adrPtr = envPtr->to;
		Tcl_DStringAppend(&ds,
			Tcl_GetVar2(interp, "t", "to", TCL_GLOBAL_ONLY),-1);
		Tcl_DStringAppend(&ds, ": ", 2);
	    } else {
		msgPtr->fromMe = RAT_ISME_NO;
		adrPtr = envPtr->from;
	    }
	    for (; adrPtr; adrPtr = adrPtr->next) {
		if (adrPtr->mailbox && adrPtr->host) {
		    break;
		}
	    }
	    if (!adrPtr) {
		return "";
	    }
	    Tcl_DStringAppend(&ds, RatAddressMail(adrPtr), -1);
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_NAME_RECIPIENT:
	    if (!envPtr->to) {
		return "";
	    }
	    if (RAT_ISME_YES == msgPtr->toMe
		    || (RAT_ISME_UNKOWN == msgPtr->toMe
			&& RatAddressIsMe(interp, envPtr->to, 1))) {
		msgPtr->toMe = RAT_ISME_YES;
		if (envPtr->from->personal) {
		    Tcl_DStringSetLength(&ds, 0);
		    Tcl_DStringAppend(&ds,
			    Tcl_GetVar2(interp, "t","from",TCL_GLOBAL_ONLY),-1);
		    Tcl_DStringAppend(&ds, ": ", 2);
		    Tcl_DStringAppend(&ds, envPtr->from->personal, -1);
		    return Tcl_DStringValue(&ds);
		}
	    } else {
		msgPtr->toMe = RAT_ISME_NO;
		if (envPtr->to->personal) {
		    return envPtr->to->personal;
		}
	    }
	    /* fallthrough */
	case RAT_FOLDER_MAIL_RECIPIENT:
	    Tcl_DStringSetLength(&ds, 0);
	    if (RAT_ISME_YES == msgPtr->toMe
		    || (RAT_ISME_UNKOWN == msgPtr->toMe
			&& RatAddressIsMe(interp, envPtr->to, 1))) {
		msgPtr->toMe = RAT_ISME_YES;
		adrPtr = envPtr->from;
		Tcl_DStringAppend(&ds,
			Tcl_GetVar2(interp, "t", "from", TCL_GLOBAL_ONLY),-1);
		Tcl_DStringAppend(&ds, ": ", 2);
	    } else {
		msgPtr->toMe = RAT_ISME_NO;
		adrPtr = envPtr->to;
	    }
	    for (; adrPtr; adrPtr = adrPtr->next) {
		if (adrPtr->mailbox && adrPtr->host) {
		    break;
		}
	    }
	    if (!adrPtr) {
		return "";
	    }
	    Tcl_DStringAppend(&ds, RatAddressMail(adrPtr), -1);
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_SIZE:
	    Tcl_DStringSetLength(&ds, 256);
	    sprintf(Tcl_DStringValue(&ds), "%d", size);
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_SIZE_F:
	    return RatMangleNumber(size);
	case RAT_FOLDER_DATE_F:
	    if (envPtr->date && T == mail_parse_date(&dateElt, envPtr->date)) {
		dateEltPtr = &dateElt;
	    } else {
		dateEltPtr = eltPtr;
	    }
	    return RatFormatDate(interp, dateEltPtr->month-1, dateEltPtr->day);
	case RAT_FOLDER_DATE_N:
	    if (envPtr->date && T == mail_parse_date(&dateElt, envPtr->date)) {
		dateEltPtr = &dateElt;
	    } else {
		dateEltPtr = eltPtr;
	    }
	    Tcl_DStringSetLength(&ds, 256);
	    tm.tm_sec = dateEltPtr->seconds;
	    tm.tm_min = dateEltPtr->minutes;
	    tm.tm_hour = dateEltPtr->hours;
	    tm.tm_mday = dateEltPtr->day;
	    tm.tm_mon = dateEltPtr->month - 1;
	    tm.tm_year = dateEltPtr->year+70;
	    tm.tm_wday = 0;
	    tm.tm_yday = 0;
	    tm.tm_isdst = -1;
	    time = mktime(&tm);
	    zonediff = (dateEltPtr->zhours*60+dateEltPtr->zminutes)*60;
	    if (!dateEltPtr->zoccident) {
		zonediff *= -1;
	    }
	    time += zonediff;
	    sprintf(Tcl_DStringValue(&ds), "%ld", time);
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_DATE_IMAP4:
	    if (envPtr->date && T == mail_parse_date(&dateElt, envPtr->date)) {
		dateEltPtr = &dateElt;
	    } else {
		dateEltPtr = eltPtr;
	    }
	    Tcl_DStringSetLength(&ds, 256);
	    return mail_date(Tcl_DStringValue(&ds), dateEltPtr);
	case RAT_FOLDER_TYPE:
	    Tcl_DStringSetLength(&ds, strlen(bodyPtr->subtype)+2+
		    strlen(body_types[bodyPtr->type]));
	    sprintf(Tcl_DStringValue(&ds), "%s/%s",
		    body_types[bodyPtr->type],
		    bodyPtr->subtype);
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_PARAMETERS:
	    Tcl_DStringSetLength(&ds, 0);
	    for (parmPtr = bodyPtr->parameter; parmPtr;
		    parmPtr = parmPtr->next) {
		Tcl_DStringStartSublist(&ds);
		Tcl_DStringAppendElement(&ds, parmPtr->attribute);
		Tcl_DStringAppendElement(&ds, parmPtr->value);
		Tcl_DStringEndSublist(&ds);
	    }
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_TO:
	    Tcl_DStringSetLength(&ds, 2048);
	    *(Tcl_DStringValue(&ds)) = '\0';
	    rfc822_write_address(Tcl_DStringValue(&ds), envPtr->to);
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_FROM:
	    Tcl_DStringSetLength(&ds, 2048);
	    *(Tcl_DStringValue(&ds)) = '\0';
	    rfc822_write_address(Tcl_DStringValue(&ds), envPtr->from);
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_SENDER:
	    Tcl_DStringSetLength(&ds, 2048);
	    *(Tcl_DStringValue(&ds)) = '\0';
	    rfc822_write_address(Tcl_DStringValue(&ds), envPtr->sender);
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_CC:
	    Tcl_DStringSetLength(&ds, 2048);
	    *(Tcl_DStringValue(&ds)) = '\0';
	    rfc822_write_address(Tcl_DStringValue(&ds), envPtr->cc);
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_REPLY_TO:
	    Tcl_DStringSetLength(&ds, 2048);
	    *(Tcl_DStringValue(&ds)) = '\0';
	    rfc822_write_address(Tcl_DStringValue(&ds), envPtr->reply_to);
	    return Tcl_DStringValue(&ds);
	case RAT_FOLDER_FLAGS:
	    Tcl_DStringSetLength(&ds, 0);
	    if (eltPtr->seen) {
		Tcl_DStringAppend(&ds, " " RAT_SEEN_STR, -1);
	    }
	    if (eltPtr->deleted) {
		Tcl_DStringAppend(&ds, " " RAT_DELETED_STR, -1);
	    }
	    if (eltPtr->flagged) {
		Tcl_DStringAppend(&ds, " " RAT_FLAGGED_STR, -1);
	    }
	    if (eltPtr->answered) {
		Tcl_DStringAppend(&ds, " " RAT_ANSWERED_STR, -1);
	    }
	    if (Tcl_DStringLength(&ds)) {
		return Tcl_DStringValue(&ds)+1;
	    } else {
		return "";
	    }
	case RAT_FOLDER_STATUS:	/*fallthrough */
	case RAT_FOLDER_INDEX:	/*fallthrough */
	case RAT_FOLDER_END:
	    return NULL;
    }
    return NULL;
}
