# font.tcl --
#
# Handles cataloguing and selection of fonts
#
#
#  TkRat software and its included text is Copyright 1996,1997,1998
#  by Martin Forssn
#
#  The full text of the legal notice is contained in the file called
#  COPYRIGHT, included with this distribution.


# AddFont --
#
# Add a font to the list of known fonts.
#
# Arguments:
# encoding   - The encoding of this font
# size       - The size difference from "normal"
# attributes - The attributes
# name	     - The name of the font

proc AddFont {encoding size attributes name} {
    global fontList fontEncoding
    set encoding [string tolower $encoding]
    set def [list $encoding $size $attributes]
    set fontList($def) $name
    set fontEncoding($encoding) 1

    # Add us-ascii version if iso-8859 font
    if [regexp iso-8859- $encoding] {
	set def [list us-ascii $size $attributes]
	set fontList($def) $name
	set fontEncoding(us-ascii) 1
	return 1
    }
}


# RemoveFonts --
#
# Remove slected fonts from the list of known fonts.
# We currently ignore the problem that some encodings may become unsupported
#
# Arguments:
# name - Name of fonts to remove (may be regexp)

proc RemoveFonts {name} {
    global fontList

    foreach f [array names fontList] {
	if [regexp -- $name $fontList($f)] {
	    unset FontList($f)
	}
    }
}


# GetFont --
#
# Get the font that best matches the requirements
#
# Arguments:
# encoding   - The encoding of this font
# size       - The size difference from "normal"
# attributes - The attributes

proc GetFont {encoding size attributes} {
    global fontList option

    if [info exists fontList([list $encoding $size $attributes])] {
	return $fontList([list $encoding $size $attributes])
    }

    # The ultimate result
    set name {}
    set equivalence 100

    incr size $option(fontsize)

    foreach font [array names fontList] {
	# Check encoding
	if ![RatEncodingCompat $encoding [lindex $font 0]] {
	    continue
	}
	# calculate equivalence rating
	set e [expr 4*abs([lindex $font 1]-$size)]
	set attr [lindex $font 2]
	foreach a $attributes {
	    set i [lsearch -exact $attr $a]
	    if { -1 == $i} {
		incr e 2
	    } else {
		set attr [lreplace $attr $i $i]
	    }
	}
	incr e [llength $attr]
	# Check if good
	if {$e < $equivalence} {
	    set equivalence $e
	    set name $fontList($font)
	    if {0 == $e} {
		return $name
	    }
	}
    }
    return $name
}


# RatSelectEncoding --
#
# Select which encoding to express some characters in
#
# Arguments:
# encoding - The current encoding of the characters

proc RatSelectEncoding {encoding} {
    global fontEncoding option

    # First we try to find it
    if [info exists fontEncoding($encoding)] {
	return $encoding
    }

    # Then we try to find a compatible one
    foreach e [array names fontEncoding] {
	if [RatEncodingCompat $encoding $e] {
	    return $e
	}
    }

    # Finally we fall back to our default encoding
    return $option(charset)
}
