#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>

#include <atalk/adouble.h>

/* byte-range locks. ad_lock is used by afp_bytelock and afp_openfork
 * to establish locks. both ad_lock and ad_tmplock take 0, 0, 0 to
 * signify locking of the entire file. */
int ad_lock(struct adouble *ad, const u_int32_t eid, const int type,
	    const off_t off, const int endflag, const off_t len)
{
  struct flock lock;
  int fd;
  
  lock.l_start = off;
  lock.l_type = type;
  lock.l_whence = (endflag) ? SEEK_END : SEEK_SET;
  lock.l_len = len;

  if (eid == ADEID_DFORK) 
    fd = ad_dfileno(ad);
  else {
    fd = ad_hfileno(ad);
    lock.l_start += ad_getentrylen(ad, eid);
  }

  /* okay, lock it. */
  if (fcntl(fd, F_SETLK, &lock) < 0)
    return -1;
  else if ((type != ADLOCK_CLR) && endflag) { /* return l_start */
    fcntl(fd, F_GETLK, &lock);
    return lock.l_start;
  }

  return off;
}


/* ad_tmplock is used by afpd to lock actual read/write operations. 
 * it saves the current lock state before attempting to lock to prevent
 * mixups. */
int ad_tmplock(struct adouble *ad, const u_int32_t eid, const int type,
	       off_t off, const int end, const off_t len)
{
  struct flock lock, *savelock;
  int fd;

  if (eid == ADEID_DFORK) {
    if (end && (off = ad_size(ad, eid) - off) < 0)
      return -1;
    fd = ad_dfileno(ad);
    savelock = &ad_dlock(ad);

 } else {
    if (end)
      off = ad_getentrylen(ad, eid) - off;
    fd = ad_hfileno(ad);
    savelock = &ad_hlock(ad);
  }

  lock.l_whence = SEEK_SET;
  lock.l_start = off;
  lock.l_len = len;

  /* unlock the current lock. there's a race condition here. */
  if (type == ADLOCK_CLR) {
    lock.l_type = F_UNLCK;
    fcntl(fd, F_SETLK, &lock); /* unlock */
    return (savelock->l_type == F_UNLCK) ? 0 : fcntl(fd, F_SETLK, savelock); 

  } else {
    lock.l_type = F_WRLCK; /* save all locks */

    /* get any lock that might encompass what we want to lock */
    if (fcntl(fd, F_GETLK, &lock) < 0)
      return -1;

    memcpy(savelock, &lock, sizeof(lock));
    
    /* now try locking the requested part */
    lock.l_type = type;
    lock.l_start = off;
    lock.l_len = len;
    if (fcntl(fd, F_SETLK, &lock) < 0)
      return -1;
  }

  return 0;
}
