/*
 * routine.c
 *
 * general use routines
 * Copyright (c) 1988, 89, 90, 91, 92, 93 Miguel Santana
 * Copyright (c) 1995, 96, 97, 98 Akim Demaille, Miguel Santana
 * $Id: routines.c,v 1.42 1998/03/03 15:51:21 demaille Exp $
 */

/*
 * This file is part of a2ps.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "a2ps.h"
#include "routines.h"
#include "stpncpy.h"
#include "msg.h"
#include "backupfile.h"

/*
 * If the file exists, and the backup machinery is on,
 * make the backup (i.e. rename the file etc.)
 */
void
xbackup_file (const char * filename)
{
  char * backup_name;
  struct stat filestat;

  /* If not backup required, or file is /dev/null, do nothing */
  if ((backup_type == none) || strequ (filename, "/dev/null"))
    return ;

  if (stat (filename, &filestat))
    {
      if (errno == ENOENT)
	/* the file does not exist: return */
	return ;
      else
	/* Another kind of error occured: exit */
	error (1, errno, _("cannot get informations on file `%s'"),
	       filename);
    }

  /* The file exists: make a backup */
  backup_name = xfind_backup_file_name (filename);
  if (rename (filename, backup_name))
    error (1, errno, _("cannot rename file `%s' as `%s'"),
		       filename, backup_name);
  FREE (backup_name);
}


/*
 * Easy call to find_backup_file_name
 */
char *
xfind_backup_file_name (const char * file)
{
  char * res;

  res = find_backup_file_name (file);
  if (!res)
    error (1, 0, _("Memory exhausted"));

  return res;
}

/*
 * Convert a list of string of valid chars to an yes/no array
 */
void
string_to_array (uchar arr[256], const uchar * string)
{
  int c;

  for (c = 0 ; c < 256 ; c++)
    arr [c] = FALSE;
  for ( /* nothing */ ; *string ; string ++)
    arr [*string] = TRUE;
}


/*
 * Concatenation of a char. No malloc is done.
 */
void
ustrccat (uchar * string, uchar c)
{
  int len = strlen((char *)string);
  *(string+len) = c;
  *(string+len+1) = '\0';
}

/*
 * return TRUE iff there are no upper case chars
 */
int 
is_strlower (const uchar * string)
{
  for (/* skip */; *string != NUL; string++)
    if (isupper(*string))
      return FALSE;
  return TRUE;
}

/*
 * Change the value of STRING into lower case
 */
static inline char *
_strnlower (char * string, size_t len)
{
  size_t i;
  for (i=0 ; i < len ; i++)
    string [i] = tolower (string [i]);
  return string;
}

char *
strnlower (char * string, size_t len)
{
  return _strnlower (string, len);
}

char *
strlower (char * string)
{
  return _strnlower (string, strlen (string));
}

/*
 * Count the number of occurrence of C in S
 */
int
strcnt (uchar * s, uchar c)
{
  int res;
  for (res = 0 ; *s ; s++)
    if (*s == c)
      res++;
  return res;
}

/*
 * Extract a substring for START, of LENGTH, making sure to
 * set the trailing '\0' (return pos of \0)
 */
char *
strsub (char * dest, const char * string, int start, int length)
{
  char * end = stpncpy (dest, string + start, length);
  *end = '\0';
  return end;
}

/*
 * fopen, but exits on failure
 */
static inline FILE *
_xfopen (const char * filename, const char * rights, const char * format)
{
  FILE * res;

  message (msg_file,
	   (stderr, "Opening file `%s' for `%s'\n", filename, rights));
  res = fopen (filename, rights);
  if (!res)
    error (1, errno, format, filename);
  return res;
}

FILE *
xfopen (const char * filename, const char * rights, const char * format)
{
  return _xfopen (filename, rights, format);
}

FILE *
xrfopen (const char * filename)
{
  return _xfopen (filename, "r", _("cannot open file `%s'"));
}

FILE *
xwfopen (const char * filename)
{
  return _xfopen (filename, "w", _("cannot create file `%s'"));
}



/*
 * Like popen, but exist upon failure
 */
static inline FILE *
_xpopen (const char * filename, const char * rights, const char * format)
{
  FILE * res;

  message (msg_file,
	   (stderr, "Opening pipe `%s' for `%s'\n", filename, rights));
  res = popen (filename, rights);
  if (!res)
    error (1, errno, format, filename);
  return res;
}

FILE *
xpopen (const char * filename, const char * rights, const char * format)
{
  return _xpopen (filename, rights, format);
}

FILE *
xrpopen (const char * filename)
{
  return _xpopen (filename, "r", _("cannot open a pipe on `%s'"));
}

FILE *
xwpopen (const char * filename)
{
  return _xpopen (filename, "w", _("cannot open a pipe on `%s'"));
}

/*
 * Copy the content of IN into OUT
 */
static inline void
_streams_copy (FILE * in, FILE * out)
{
  size_t read_length;
  char buf [BUFSIZ];

  while ((read_length = fread (buf, sizeof (char), sizeof (buf), in)))
    fwrite (buf, sizeof (char), read_length, out);
}

void
streams_copy (FILE * in, FILE * out)
{
  _streams_copy (in, out);
}



/*
 * Dump the content of a file onto STREAM, and remove.
 * Used when honoring a subcontract.
 */
void
dump_file_content_and_unlink (FILE * stream, const char * filename)
{
  FILE * fp;

  message (msg_tool | msg_file, (stderr, "Dumping file `%s'\n", filename));

  fp = xrfopen (filename);
  _streams_copy (fp, stream);
  fclose (fp);

  /* Don't complain if you can't unlink.  Who cares of a tmp file? */
  unlink (filename);
}

