;; $Id: dbcommon.dsl 1.9 1998/02/22 21:15:43 nwalsh Exp $
;;
;; This file is part of the Modular DocBook Stylesheet distribution.
;; See ../README or http://www.berkshire.net/~norm/dsssl/
;;
;; This file contains general functions common to both print and HTML
;; versions of the DocBook stylesheets.
;;

;; If **ANY** change is made to this file, you _MUST_ alter the
;; following definition:

(define %docbook-common-version%
  "Modular DocBook Stylesheet Common Functions version 1.06")

;; === element lists ====================================================

(define book-element-list 
  (list "BOOK"))

(define division-element-list
  (list "PART"))

(define component-element-list
  (list "PREFACE" "CHAPTER" "APPENDIX" 
	"ARTICLE"
	"GLOSSARY" "BIBLIOGRAPHY" "INDEX"
	"REFERENCE" "REFENTRY"
	"BOOK")) ;; just in case nothing else matches...

(define major-component-element-list
  (list "PREFACE" "CHAPTER" "APPENDIX" 
	"ARTICLE"
	"GLOSSARY" "BIBLIOGRAPHY" "INDEX"
	"REFERENCE" "REFENTRY"
	"PART"
	"BOOK")) ;; just in case nothing else matches...

(define section-element-list
  (list "SECT1" "SECT2" "SECT3" "SECT4" "SECT5"
	"SIMPLESECT"
	"REFSECT1" "REFSECT2" "REFSECT3"))

(define block-element-list
  (list "EXAMPLE" "FIGURE" "TABLE" "EQUATION" "PROCEDURE"))

(define outer-parent-list
  (list "TOC" "LOT" "APPENDIX" "CHAPTER" "PART" "PREFACE" "REFERENCE"
	"BIBLIOGRAPHY" "GLOSSARY" "INDEX" "SETINDEX"
	"SECT1" "SECT2" "SECT3" "SECT4" "SECT5" "SIMPLESECT"
	"PARTINTRO" "BIBLIODIV" "GLOSSDIV" "INDEXDIV"
	"REFENTRY" "REFSECT1" "REFSECT2" "REFSECT3"
	"MSGTEXT" "MSGEXPLAN"))

(define list-list
  (list "ORDEREDLIST" "ITEMIZEDLIST" "VARIABLELIST" "SEGMENTEDLIST"
        "SIMPLELIST" "CALLOUTLIST" "STEP"))

;; === automatic TOC ====================================================

;; Returns #t if nd should appear in the auto TOC
(define (appears-in-auto-toc? nd)
  (if (or (equal? (gi nd) "REFSECT1")
	  (have-ancestor? "REFSECT1" nd))
      #f
      #t))

;; # return elements of nl for which appears-in-auto-toc? is #t
(define (toc-list-filter nodelist)
  (let loop ((toclist (empty-node-list)) (nl nodelist))
    (if (node-list-empty? nl)
	toclist
	(if (appears-in-auto-toc? (node-list-first nl))
	    (loop (node-list toclist (node-list-first nl))
		  (node-list-rest nl))
	    (loop toclist (node-list-rest nl))))))
  
;; === common ===========================================================

(define (INLIST?)
  (has-ancestor-member? (current-node) list-list))

(define (INBLOCK?)
  (has-ancestor-member? (current-node) 
			'("EXAMPLE" "INFORMALEXAMPLE"
			  "FIGURE" "INFORMALFIGURE"
			  "EQUATION" "INFORMALEQUATION"
			  "FUNCSYNOPSIS"
			  "PROGRAMLISTINGCO"
			  "SCREENCO"
			  "GRAPHICCO")))

(define (PARNUM)
  (child-number (parent (current-node))))

(define (NESTEDFNUM n fmt)
  (if (number? n)
      (format-number n fmt)
      #f))

(define (FNUM n) (NESTEDFNUM n "1"))

(define (book-start?)
  ;; Returns #t if the current-node is in the first division or 
  ;; component of a book.
  (let ((book (ancestor "BOOK"))
	(nd   (ancestor-member 
	       (current-node) 
	       (append component-element-list division-element-list))))
    (let loop ((ch (children book)))
      (if (node-list-empty? ch)
	  #f
	  (if (member (gi (node-list-first ch)) 
		      (append component-element-list division-element-list))
	      (node-list=? (node-list-first ch) nd)
	      (loop (node-list-rest ch)))))))

(define (first-chapter?)
  ;; Returns #t if the current-node is in the first chapter of a book
  (let* ((book (ancestor "BOOK"))
	 (nd   (ancestor-member 
		(current-node) 
		(append component-element-list division-element-list)))
	 (bookch (children book))
	 (bookcomp (expand-children bookch '("PART"))))
    (let loop ((nl bookcomp))
      (if (node-list-empty? nl)
	  #f
	  (if (equal? (gi (node-list-first nl)) "CHAPTER")
	      (if (node-list=? (node-list-first nl) nd)
		  #t
		  #f)
	      (loop (node-list-rest nl)))))))

;; === bibliographic ====================================================

(define (author-string #!optional (author (current-node)))
  ;; Return a formatted string representation of the contents of:
  ;; AUTHOR:
  ;;   Handles *only* Honorific, FirstName, SurName, and Lineage.
  ;;   Handles *only* the first of each.
  ;;   Format is "Honorific. FirstName SurName, Lineage"
  ;; CORPAUTHOR:
  ;;   returns (data corpauthor)
  (let* ((h_nl (select-elements (descendants author) "HONORIFIC"))
	 (f_nl (select-elements (descendants author) "FIRSTNAME"))
	 (s_nl (select-elements (descendants author) "SURNAME"))
	 (l_nl (select-elements (descendants author) "LINEAGE"))
	 (has_h (not (node-list-empty? h_nl)))
	 (has_f (not (node-list-empty? f_nl)))
	 (has_s (not (node-list-empty? s_nl)))
	 (has_l (not (node-list-empty? l_nl))))
    (if (or (equal? (gi author) "AUTHOR")
	    (equal? (gi author) "EDITOR")
	    (equal? (gi author) "OTHERCREDIT"))
	(string-append
	 (if has_h (string-append (data (node-list-first h_nl)) ".") "")
	 (if has_f (string-append 
		    (if has_h " " "") 
		    (data (node-list-first f_nl))) "")
	 (if has_s (string-append 
		    (if (or has_h has_f) " " "") 
		    (data (node-list-first s_nl))) "")
	 (if has_l (string-append ", " (data (node-list-first l_nl))) ""))
	(data author))))

(define (author-list-string #!optional (author (current-node)))
  ;; Return a formatted string representation of the contents of AUTHOR
  ;; *including appropriate punctuation* if the AUTHOR occurs in a list
  ;; of AUTHORs in an AUTHORGROUP:
  ;;
  ;;   John Doe
  ;; or
  ;;   John Doe and Jane Doe
  ;; or
  ;;   John Doe, Jane Doe, and A. Nonymous
  ;;

  (let* ((author-node-list (select-elements
			    (descendants 
			     (ancestor "AUTHORGROUP")) "AUTHOR"))
	 (corpauthor-node-list (select-elements
				(descendants 
				 (ancestor "AUTHORGROUP")) "CORPAUTHOR"))
	 (othercredit-node-list (select-elements
				 (descendants 
				  (ancestor "AUTHORGROUP")) "OTHERCREDIT"))
	 (author-count (if (have-ancestor? "AUTHORGROUP" author)
			   (+ (node-list-length author-node-list)
			      (node-list-length corpauthor-node-list)
			      (node-list-length othercredit-node-list))
			   1)))
    (string-append
     (if (and (> author-count 1)
	      (last-sibling? author))
	 (string-append %gentext-and% " ")
	 "")
     (author-string)
     (if (and (> author-count 2)
	      (not (last-sibling? author)))
	 ", "
	 (if (and (> author-count 1)
		  (not (last-sibling? author)))		  
	     " "
	     "")))))

;; === procedures =======================================================

(define ($hierarch-number-format$ depth)
  (case (modulo depth 5)
    ((1) "1")
    ((2) "a")
    ((3) "i")
    ((4) "A")
    (else "I")))

(define ($hierarch-number$ nd seperator)
  (if (equal? (gi nd) "STEP")
      (string-append
       (format-number
	(child-number nd) 
	($hierarch-number-format$ ($proc-step-depth$ nd)))
       seperator)
      ""))

(define ($proc-step-depth$ nd)
  (let loop ((step nd) (depth 0))
    (if (equal? (gi step) "PROCEDURE")
	depth
	(loop (parent step) 
	      (if (equal? (gi step) "STEP")
		  (+ depth 1)
		  depth)))))

(define ($proc-step-number$ nd)
  (let* ((step (if (equal? (gi nd) "STEP") nd (parent nd)))
	 (str ($hierarch-number$ step "")))
    (string-append str (gentext-label-title-sep "STEP"))))

(define ($proc-step-xref-number$ nd)
  (let loop ((step nd) (str "") (first #t))
    (if (equal? (gi step) "PROCEDURE")
	str
	(loop (parent step) 
	      (if (equal? (gi step) "STEP")
		  (string-append 
		   ($hierarch-number$ step
				      (if first
					  ""
					  (gentext-intra-label-sep "STEP")))
		   str)
		  str)
	      (if (equal? (gi step) "STEP")
		  #f
		  first)))))

;; === sections =========================================================

(define (SECTLEVEL #!optional (sect (current-node)))
  (if (equal? (gi sect) "SIMPLESECT")
      ;; SimpleSect is special, it should be level "n+1", where "n" is
      ;; the level of the numbered section that contains it.  If it is
      ;; the *first* sectioning element in a chapter, make it 
      ;; %default-simplesect-level%
      (cond
       ((have-ancestor? "SECT5") 6)
       ((have-ancestor? "SECT4") 5)
       ((have-ancestor? "SECT3") 4)
       ((have-ancestor? "SECT2") 3)
       ((have-ancestor? "SECT1") 2)
       ((have-ancestor? "REFSECT3") 5)
       ((have-ancestor? "REFSECT2") 4)
       ((have-ancestor? "REFSECT1") 3)
       (else %default-simplesect-level%))
      (cond
       ((equal? (gi sect) "SECT5") 5)
       ((equal? (gi sect) "SECT4") 4)
       ((equal? (gi sect) "SECT3") 3)
       ((equal? (gi sect) "SECT2") 2)
       ((equal? (gi sect) "SECT1") 1)
       ((equal? (gi sect) "REFSECT3") 4)
       ((equal? (gi sect) "REFSECT2") 3)
       ((equal? (gi sect) "REFSECT1") 2)
       ((equal? (gi sect) "REFERENCE") 1)
       (else 1))))

;; === synopsis =========================================================

;; The following definitions match those given in the reference
;; documentation for DocBook V3.0
(define	%arg-choice-opt-open-str% "[")
(define	%arg-choice-opt-close-str% "]")
(define	%arg-choice-req-open-str% "{")
(define	%arg-choice-req-close-str% "}")
(define	%arg-choice-plain-open-str% " ")
(define	%arg-choice-plain-close-str% " ")
(define	%arg-choice-def-open-str% "[")
(define	%arg-choice-def-close-str% "]")
(define	%arg-rep-repeat-str% "...")
(define	%arg-rep-norepeat-str% "")
(define	%arg-rep-def-str% "")
(define %arg-or-sep% " | ")
(define %cmdsynopsis-hanging-indent% 4pi)

;; === linking ==========================================================

;; From the DocBook V3.0 Reference entry for element XREF:
;;
;; Description
;;
;;   Cross reference link to another part of the document. XRef is empty,
;;   and has common, Linkend, and Endterm attributes.
;;
;;   Processing Expectations
;;
;;   XRef must have a Linkend, but the Endterm is optional. If it is used,
;;   the content of the element it points to is displayed as the text of
;;   the cross reference; if it is absent, the XRefLabel of the
;;   cross-referenced object is displayed.
;;
;; If neither the ENDTERM nor the XREFLABEL is present, then the cross
;; reference text is taken from the (gentext-xref-strings) function
;; in the localization file, like this
;; 
;; A cross reference to an element, the target, begins with the
;; text returned by (gentext-xref-strings (gi target)).  Within
;; that text, the following substitutions are made:
;; 
;; %p is replaced by the number of the page on which target occurs
;; %[x]g is replaced by the (gentext-element-name)
;; %[x]n is replaced by the label
;; %[x]t is replaced by the title
;; 
;; Where "x" is either absent, in which case the target is used, or
;; one of the following:
;; 
;; b = the ancestral book
;; c = the ancestral component
;; d = the ancestral division
;; k = the ancestral block
;; s = the ancestral section
;; 
;; So, %cn is replaced by the label (number) of the chapter that
;; contains target.  %st is replaced by the title of the section
;; that contains target, %g is replaced by the
;; (gentext-element-name) of the target, etc.
;; 
;; What elements constitute a book, component, division, block, and
;; section are defined by the lists *-element-list.
;; 
;; As if this wasn't confusing enough, _one_ additional level of
;; indirection is available.  If one of the special symbols, #b,
;; #c, #d, #k, or #s occurs in the (gentext-xref-strings) text,
;; then it will be replaced by the appropriate
;; (gentext-xref-strings-indirect) text depending on whether or not
;; the target and the reference occur in the same element.
;; 
;; Here's a concrete example:
;; 
;; Given
;; 
;; (define (gentext-xref-strings-indirect key)
;;   (case key
;;     (("k") '(" in %kg %kn" ""))
;;     (("s") '(" in %cg %cn" " in this %cg"))))
;; 
;; and
;; 
;; (define (gentext-xref-strings giname)
;;   (case giname
;;     (("STEP") "step %n#k")
;;     (("SECT1") "the section called %t#s")))
;; 
;; A cross reference to a step in the same procedure as the reference
;; will use the string "step %n" as the gentext-xref-strings text.
;; A cross reference to a step in another procedure will use the
;; string "step %n in %kg %kn".
;; 
;; So, a reference from step 5 to step 3 in the same procedure will
;; appear as "step 3".  A reference to step 6 in the third
;; procedure in the fourth chapter from some other place will
;; appear as "step 6 in procedure 4.3".
;; 
;; Likewise a reference to another section in the current chapter
;; will appear like this "the section called target-title in this
;; Chapter", and a reference to a section in an appendix will
;; appear like this "the section called target-title in Appendix
;; B".
;; 
;; ======================================================================

(define (auto-xref target)
  (let* ((cont-blok (ancestor-member target block-element-list))
	 (cont-sect (ancestor-member target section-element-list))
	 (cont-comp (ancestor-member target component-element-list))
	 (cont-divn (ancestor-member target division-element-list))
 	 (cont-book (ancestor-member target book-element-list))
	 (substitute (list
		      (list "%bg" (element-gi-sosofo cont-book))
		      (list "%bn" (element-label-sosofo cont-book))
		      (list "%bt" (element-title-xref-sosofo cont-book))
		      (list "%cg" (element-gi-sosofo cont-comp))
		      (list "%cn" (element-label-sosofo cont-comp))
		      (list "%ct" (element-title-xref-sosofo cont-comp))
		      (list "%dg" (element-gi-sosofo cont-divn))
		      (list "%dn" (element-label-sosofo cont-divn))
		      (list "%dt" (element-title-xref-sosofo cont-divn))
		      (list "%g"  (element-gi-sosofo target))
		      (list "%kg" (element-gi-sosofo cont-blok))
		      (list "%kn" (element-label-sosofo cont-blok))
		      (list "%kt" (element-title-xref-sosofo cont-blok))
		      (list "%n"  (element-label-sosofo target))
		      (list "%p"  (element-page-number-sosofo target))
		      (list "%sg" (element-gi-sosofo cont-sect))
		      (list "%sn" (element-label-sosofo cont-sect))
		      (list "%st" (element-title-xref-sosofo cont-sect))
		      (list "%t"  (element-title-xref-sosofo target))))
	 (text        (subst-xref-strings-indirect 
		       (gentext-xref-strings (gi target))
		       (current-node)
		       target))
	 (tlist   (match-split-list text (assoc-objs substitute))))
    (make sequence
      (string-list-sosofo tlist substitute))))

;; ======================================================================

(define (subst-xref-strings-indirect string reference referent)
  (let* ((rnce-blok (ancestor-member reference block-element-list))
	 (rnce-sect (ancestor-member reference section-element-list))
	 (rnce-comp (ancestor-member reference component-element-list))
	 (rnce-divn (ancestor-member reference division-element-list))
 	 (rnce-book (ancestor-member reference book-element-list))

	 (rent-blok (ancestor-member referent block-element-list))
	 (rent-sect (ancestor-member referent section-element-list))
	 (rent-comp (ancestor-member referent component-element-list))
	 (rent-divn (ancestor-member referent division-element-list))
 	 (rent-book (ancestor-member referent book-element-list))

	 (title     (element-title-xref-sosofo referent))

	 (b (if (node-list-empty? rent-book)
		""
		(if (node-list=? rnce-book rent-book)
		    (car (cdr (gentext-xref-strings-indirect "b")))
		    (car (gentext-xref-strings-indirect "b")))))

	 (c (if (node-list-empty? rent-comp)
		""
		(if (node-list=? rnce-comp rent-comp)
		    (car (cdr (gentext-xref-strings-indirect "c")))
		    (car (gentext-xref-strings-indirect "c")))))

	 (d (if (node-list-empty? rent-divn)
		""
		(if (node-list=? rnce-divn rent-divn)
		    (car (cdr (gentext-xref-strings-indirect "d")))
		    (car (gentext-xref-strings-indirect "d")))))

	 (k (if (node-list-empty? rent-blok)
		""
		(if (node-list=? rnce-blok rent-blok)
		    (car (cdr (gentext-xref-strings-indirect "k")))
		    (car (gentext-xref-strings-indirect "k")))))

	 (s (if (node-list-empty? rent-sect)
		""
		(if (node-list=? rnce-sect rent-sect)
		    (car (cdr (gentext-xref-strings-indirect "s")))
		    (car (gentext-xref-strings-indirect "s"))))))

    (string-replace-list string 
			 (list "#b" b "#c" c "#d" d "#k" k "#s" s))))

;; ======================================================================

(define (set-autolabel nd #!optional (force-label? #f))
  "")

(define (book-autolabel nd #!optional (force-label? #f))
  "")

(define (part-autolabel nd #!optional (force-label? #f))
  (format-number (child-number nd) "I"))

(define (dedication-autolabel nd #!optional (force-label? #f))
  "")

(define (preface-autolabel nd #!optional (force-label? #f))
  "")

(define (article-autolabel nd #!optional (force-label? #f))
  "")

(define (chapter-autolabel nd #!optional (force-label? #f))
  (if (or force-label? %chapter-autolabel%)
      (format-number (element-number nd) "1")
      ""))

(define (appendix-autolabel nd #!optional (force-label? #f))
  (if (or force-label? %chapter-autolabel%)
      (format-number (element-number nd) "A")
      ""))

(define (bibliography-autolabel nd #!optional (force-label? #f))
  "")

(define (bibliodiv-autolabel nd #!optional (force-label? #f))
  "")

(define (glossary-autolabel nd #!optional (force-label? #f))
  "")

(define (glossdiv-autolabel nd #!optional (force-label? #f))
  "")

(define (index-autolabel nd #!optional (force-label? #f))
  "")

(define (reference-autolabel nd #!optional (force-label? #f))
  (format-number (child-number nd) "1"))

(define (refentry-autolabel nd #!optional (force-label? #f))
  (let* ((isep (gentext-intra-label-sep (gi nd)))
	 (refnamediv (select-elements (children nd) "REFNAMEDIV"))
	 (refd       (select-elements (children refnamediv) "REFDESCRIPTOR"))
	 (refnames   (select-elements (children refnamediv) "REFNAME")))
    ""))
;;    (if (node-list-empty? refd)
;;	(if (node-list-empty? refnames)
;;	    ""
;;	    (data (node-list-first refnames)))
;;	(data refd))))

(define (section-autolabel nd #!optional (force-label? #f))
  (let* ((isep (gentext-intra-label-sep (gi nd)))
	 (haschn (not (node-list-empty? (ancestor "CHAPTER" nd))))
	 (hasapn (not (node-list-empty? (ancestor "APPENDIX" nd))))
	 (prefix (cond
		  (haschn (string-append 
			   (element-label (ancestor "CHAPTER" nd)) isep))
		  (hasapn (string-append 
			   (element-label (ancestor "APPENDIX" nd)) isep))
		  (else ""))))
    (if (or force-label? %section-autolabel%)
	(case (gi nd)
	  (("SECT1") (string-append prefix 
				    (format-number (child-number nd) "1")))
	  (("SECT2") (string-append (element-label 
				     (ancestor "SECT1" nd) force-label?)
				    isep 
				    (format-number (child-number nd) "1")))
	  (("SECT3") (string-append (element-label 
				     (ancestor "SECT2" nd) force-label?)
				    isep 
				    (format-number (child-number nd) "1")))
	  (("SECT4") (string-append (element-label 
				     (ancestor "SECT3" nd) force-label?)
				    isep 
				    (format-number (child-number nd) "1")))
	  (("SECT5") (string-append (element-label 
				     (ancestor "SECT4" nd) force-label?)
				    isep 
				    (format-number (child-number nd) "1")))
	  (else (string-append (gi nd) " IS NOT A SECTION!")))
	"")))
  
(define (refsection-autolabel nd #!optional (force-label? #f))
  (let* ((isep (gentext-intra-label-sep (gi nd)))
	 (haschn (not (node-list-empty? (ancestor "CHAPTER" nd))))
	 (hasapn (not (node-list-empty? (ancestor "APPENDIX" nd))))
	 (prefix (string-append
		  (element-label (ancestor "REFENTRY" nd) force-label?)
		  isep)))
    (if (or force-label? %section-autolabel%)
	(case (gi nd)
	  (("REFSECT1") (string-append prefix 
				       (format-number (child-number nd) "1")))
	  (("REFSECT2") (string-append (element-label 
					(ancestor "REFSECT1" nd) force-label?)
				       isep 
				       (format-number (child-number nd) "1")))
	  (("REFSECT3") (string-append (element-label 
					(ancestor "REFSECT2" nd) force-label?)
				       isep 
				       (format-number (child-number nd) "1")))
	  (else (string-append (gi nd) " IS NOT A SECTION!")))
	"")))

(define (step-autolabel nd #!optional (force-label? #f))
  ($proc-step-xref-number$ nd))

(define (sidebar-autolabel nd #!optional (force-label? #f))
  "")

(define (legalnotice-autolabel nd #!optional (force-label? #f))
  "")

(define (block-autolabel nd #!optional (force-label? #f))
  (let* ((chn (element-label (ancestor "CHAPTER" nd)))
	 (apn (element-label (ancestor "APPENDIX" nd)))
	 (rfn (element-label (ancestor "REFENTRY" nd)))
	 (bkn (format-number (component-child-number nd) "1")))
    (if (equal? chn "")
	(if (equal? apn "")
	    (if (equal? rfn "")
		bkn
		(string-append rfn (gentext-intra-label-sep (gi nd)) bkn))
	    (string-append apn (gentext-intra-label-sep (gi nd)) bkn))  
	(string-append chn (gentext-intra-label-sep (gi nd)) bkn))))

;; For all elements, if a LABEL attribute is present, that is the label
;; that they get.  Otherwise:
;; BOOK gets the Book volume, by book-autolabel
;; PREFACE gets "", by preface-autolabel
;; CHAPTER gets the Chapter number, by chapter-autolabel
;; APPENDIX gets the Appendix letter, by appendix-autolabel
;; REFERENCE gets "", by reference-autolabel
;; REFENTRY gets "", by refentry-autolabel
;; SECT* gets the nested section number (e.g., 1.3.5), by section-autolabel
;; REFSECT* gets the nested section number, by refsection-autolabel
;; everything else gets numbered by block-autolabel
;;
(define (element-label #!optional (nd (current-node)) (force-label? #f))
  (if (node-list-empty? nd)
      ""
      (let ((label (attribute-string "LABEL" nd)))
	(if label
	    label
	    (case (gi nd)
	      ;; Use a seperately defined assoc list?
	      (("APPENDIX") (appendix-autolabel nd force-label?))
	      (("ARTICLE") (article-autolabel nd force-label?))
	      (("BIBLIODIV") (bibliodiv-autolabel nd force-label?))
	      (("BIBLIOGRAPHY") (bibliography-autolabel nd force-label?))
	      (("BOOK") (book-autolabel nd force-label?))
	      (("CHAPTER") (chapter-autolabel nd force-label?))
	      (("DEDICATION") (dedication-autolabel nd force-label?))
	      (("GLOSSARY") (glossary-autolabel nd force-label?))
	      (("GLOSSDIV") (glossdiv-autolabel nd force-label?))
	      (("INDEX") (index-autolabel nd force-label?))
	      (("LEGALNOTICE") (legalnotice-autolabel nd force-label?))
	      (("PART") (part-autolabel nd force-label?))
	      (("PREFACE") (preface-autolabel nd force-label?))
	      (("REFENTRY") (refentry-autolabel nd force-label?))
	      (("REFERENCE") (reference-autolabel nd force-label?))
	      (("REFSECT1") (refsection-autolabel nd force-label?))
	      (("REFSECT2") (refsection-autolabel nd force-label?))
	      (("REFSECT3") (refsection-autolabel nd force-label?))
	      (("SECT1") (section-autolabel nd force-label?))
	      (("SECT2") (section-autolabel nd force-label?))
	      (("SECT3") (section-autolabel nd force-label?))
	      (("SECT4") (section-autolabel nd force-label?))
	      (("SECT5") (section-autolabel nd force-label?))
	      (("SET") (set-autolabel nd force-label?))
	      (("SIDEBAR") (sidebar-autolabel nd force-label?))
	      (("SIMPLESECT") "")
	      (("STEP") (step-autolabel nd force-label?))
	      (else (block-autolabel nd force-label?)))))))

;; ======================================================================

;; Returns the element label as a sosofo
;;
(define (element-label-sosofo nd #!optional (force-label? #f))
  (if (string=? (element-label nd force-label?) "")
      (empty-sosofo)
      (make sequence
	(literal (element-label nd force-label?)))))

;; ======================================================================

(define (set-title-sosofo nd)
  (let* ((setinfo (select-elements (children nd) "SETINFO"))
	 (sititles (select-elements  
		    (expand-children (children setinfo) 
				     '("BOOKBIBLIO" 
				       "BIBLIOMISC"
				       "BIBLIOSET"))
		    "TITLE"))
	 (settitles (select-elements (children nd) "TITLE"))
	 (titles   (if (node-list-empty? settitles)
		       sititles
		       settitles)))
    (if (node-list-empty? titles)
	(empty-sosofo)
	(with-mode xref-title-mode
	  (process-node-list (node-list-first titles))))))

(define (book-title-sosofo nd)
  (let* ((bookinfo (select-elements (children nd) "BOOKINFO"))
	 (bititles (select-elements  
		    (expand-children (children bookinfo) 
				     '("BOOKBIBLIO" 
				       "BIBLIOMISC"
				       "BIBLIOSET"))
		    "TITLE"))
	 (chtitles (select-elements (children nd) "TITLE"))
	 (titles   (if (node-list-empty? chtitles)
		       bititles
		       chtitles)))
    (if (node-list-empty? titles)
	(empty-sosofo)
	(with-mode xref-title-mode
	  (process-node-list (node-list-first titles))))))

(define (part-title-sosofo nd)
  (let* ((docinfo  (select-elements (children nd) "DOCINFO"))
	 (dititles (select-elements  
		    (expand-children (children docinfo) 
				     '("BOOKBIBLIO" 
				       "BIBLIOMISC"
				       "BIBLIOSET"))
		    "TITLE"))
	 (chtitles (select-elements (children nd) "TITLE"))
	 (titles   (if (node-list-empty? chtitles)
		       dititles
		       chtitles)))
    (if (node-list-empty? titles)
	(empty-sosofo)
	(with-mode xref-title-mode
	  (process-node-list (node-list-first titles))))))

(define (article-title-sosofo nd)
  (let* ((artheader (select-elements (children nd) "ARTHEADER"))
	 (titles (select-elements (children artheader) "TITLE")))
    (if (node-list-empty? titles)
	(empty-sosofo)
	(with-mode xref-title-mode
	  (process-node-list (node-list-first titles))))))

(define (preface-title-sosofo nd)
  (let* ((docinfo  (select-elements (children nd) "DOCINFO"))
	 (dititles (select-elements  
		    (expand-children (children docinfo) 
				     '("BOOKBIBLIO" 
				       "BIBLIOMISC"
				       "BIBLIOSET"))
		    "TITLE"))
	 (chtitles (select-elements (children nd) "TITLE"))
	 (titles   (if (node-list-empty? chtitles)
		       dititles
		       chtitles)))
    (if (node-list-empty? titles)
	(empty-sosofo)
	(with-mode xref-title-mode
	  (process-node-list (node-list-first titles))))))

(define (chapter-title-sosofo nd)
  (let* ((docinfo  (select-elements (children nd) "DOCINFO"))
	 (dititles (select-elements  
		    (expand-children (children docinfo) 
				     '("BOOKBIBLIO" 
				       "BIBLIOMISC"
				       "BIBLIOSET"))
		    "TITLE"))
	 (chtitles (select-elements (children nd) "TITLE"))
	 (titles   (if (node-list-empty? chtitles)
		       dititles
		       chtitles)))
    (if (node-list-empty? titles)
	(empty-sosofo)
	(with-mode xref-title-mode
	  (process-node-list (node-list-first titles))))))

(define (appendix-title-sosofo nd)
  (let* ((docinfo  (select-elements (children nd) "DOCINFO"))
	 (dititles (select-elements  
		    (expand-children (children docinfo) 
				     '("BOOKBIBLIO" 
				       "BIBLIOMISC"
				       "BIBLIOSET"))
		    "TITLE"))
	 (chtitles (select-elements (children nd) "TITLE"))
	 (titles   (if (node-list-empty? chtitles)
		       dititles
		       chtitles)))
    (if (node-list-empty? titles)
	(empty-sosofo)
	(with-mode xref-title-mode
	  (process-node-list (node-list-first titles))))))

(define (reference-title-sosofo nd)
  (let* ((docinfo  (select-elements (children nd) "DOCINFO"))
	 (dititles (select-elements  
		    (expand-children (children docinfo) 
				     '("BOOKBIBLIO" 
				       "BIBLIOMISC"
				       "BIBLIOSET"))
		    "TITLE"))
	 (chtitles (select-elements (children nd) "TITLE"))
	 (titles   (if (node-list-empty? chtitles)
		       dititles
		       chtitles)))
    (if (node-list-empty? titles)
	(empty-sosofo)
	(with-mode xref-title-mode
	  (process-node-list (node-list-first titles))))))

;; Returns either the REFENTRYTITLE or the first REFNAME.
;;
(define (refentry-title-sosofo nd)
  (let* ((refmeta (select-elements (descendants nd) "REFMETA"))
	 (refttl  (select-elements (descendants refmeta) "REFENTRYTITLE"))
	 (refndiv (select-elements (descendants nd) "REFNAMEDIV"))
	 (refname (select-elements (descendants refndiv) "REFNAME")))
    (if (node-list-empty? refttl)
	(if (node-list-empty? refname)
	    (empty-sosofo)
	    (with-mode xref-title-mode
	      (process-node-list (node-list-first refname))))
	(with-mode xref-title-mode
	  (process-node-list (node-list-first refttl))))))

(define (optional-title-sosofo nd)
  (let* ((docinfo  (select-elements (children nd) "DOCINFO"))
	 (dititles (select-elements (children docinfo) "TITLE"))
	 (chtitles (select-elements (children nd) "TITLE"))
	 (titles   (if (node-list-empty? chtitles)
		       dititles
		       chtitles)))
    (if (node-list-empty? titles)
	(literal (gentext-element-name (gi nd)))
	(with-mode xref-title-mode
	  (process-node-list (node-list-first titles))))))

(define (glossary-title-sosofo nd)
  (optional-title-sosofo nd))

(define (bibliography-title-sosofo nd)
  (optional-title-sosofo nd))

(define (index-title-sosofo nd)
  (optional-title-sosofo nd))

(define (dedication-title-sosofo nd)
  (optional-title-sosofo nd))

(define (section-title-sosofo nd)
  (let* ((info     (select-elements (children nd) 
				    '("SECT1INFO" "SECT2INFO" "SECT3INFO"
						  "SECT4INFO" "SECT5INFO")))
	 (ititles  (select-elements (children info) "TITLE"))
	 (ctitles  (select-elements (children nd) "TITLE"))
	 (titles   (if (node-list-empty? ctitles)
		       ititles
		       ctitles)))
    (if (node-list-empty? titles)
	(empty-sosofo)
	(with-mode xref-title-mode
	  (process-node-list (node-list-first titles))))))

(define (refsection-title-sosofo nd)
  (let* ((info     (select-elements (children nd) 
				    '("REFSECT1INFO" "REFSECT2INFO" 
						     "REFSECT3INFO")))
	 (ititles  (select-elements (children info) "TITLE"))
	 (ctitles  (select-elements (children nd) "TITLE"))
	 (titles   (if (node-list-empty? ctitles)
		       ititles
		       ctitles)))
    (if (node-list-empty? titles)
	(empty-sosofo)
	(with-mode xref-title-mode
	  (process-node-list (node-list-first titles))))))

(define (block-title-sosofo nd)
  (let ((titles (select-elements (children nd) "TITLE")))
    (if (node-list-empty? titles)
	(empty-sosofo)
	(with-mode xref-title-mode
	  (process-node-list (node-list-first titles))))))

;; Returns the title of the element as a sosofo.
;;
(define (element-title-sosofo #!optional (nd (current-node)))
  (if (node-list-empty? nd)
      (empty-sosofo)
      (case (gi nd)
	;; Use a seperately defined assoc list?
	(("APPENDIX") (appendix-title-sosofo nd))
	(("ARTICLE") (article-title-sosofo nd))
	(("BIBLIOGRAPHY") (bibliography-title-sosofo nd))
	(("BOOK") (book-title-sosofo nd))
	(("CHAPTER") (chapter-title-sosofo nd))
	(("DEDICATION") (dedication-title-sosofo nd))
	(("GLOSSARY") (glossary-title-sosofo nd))
	(("INDEX") (index-title-sosofo nd))
	(("PART") (part-title-sosofo nd))
	(("PREFACE") (preface-title-sosofo nd))
	(("REFENTRY") (refentry-title-sosofo nd))
	(("REFERENCE") (reference-title-sosofo nd))
	(("REFSECT1") (refsection-title-sosofo nd))
	(("REFSECT2") (refsection-title-sosofo nd))
	(("REFSECT3") (refsection-title-sosofo nd))
	(("SECT1") (section-title-sosofo nd))
	(("SECT2") (section-title-sosofo nd))
	(("SECT3") (section-title-sosofo nd))
	(("SECT4") (section-title-sosofo nd))
	(("SECT5") (section-title-sosofo nd))
	(("SET") (set-title-sosofo nd))
	(else (block-title-sosofo nd)))))

;; ======================================================================

(define (set-title nd)
  (let* ((setinfo (select-elements (children nd) "SETINFO"))
	 (titles (node-list 
		  (if setinfo
		      (select-elements (children setinfo) "TITLE")
		      (empty-sosofo))
		  (select-elements (children nd) "TITLE"))))
    (if (node-list-empty? titles)
	""
	(data (node-list-first titles)))))

(define (book-title nd)
  (let* ((bookinfo (select-elements (children nd) "BOOKINFO"))
	 (titles (node-list 
		  (if bookinfo
		      (select-elements (children bookinfo) "TITLE")
		      (empty-sosofo))
		  (select-elements (children nd) "TITLE"))))
    (if (node-list-empty? titles)
	""
	(data (node-list-first titles)))))

(define (part-title nd)
  (let ((titles (select-elements (children nd) "TITLE")))
    (if (node-list-empty? titles)
	""
	(data (node-list-first titles)))))

(define (article-title nd)
  (let* ((artheader (select-elements (children nd) "ARTHEADER"))
	 (titles (select-elements (children artheader) "TITLE")))
    (if (node-list-empty? titles)
	""
	(data (node-list-first titles)))))

(define (preface-title nd)
  (let ((titles (select-elements (children nd) "TITLE")))
    (if (node-list-empty? titles)
	""
	(data (node-list-first titles)))))

(define (chapter-title nd)
  (let ((titles (select-elements (children nd) "TITLE")))
    (if (node-list-empty? titles)
	""
	(data (node-list-first titles)))))

(define (appendix-title nd)
  (let ((titles (select-elements (children nd) "TITLE")))
    (if (node-list-empty? titles)
	""
	(data (node-list-first titles)))))

(define (reference-title nd)
  (let ((titles (select-elements (children nd) "TITLE")))
    (if (node-list-empty? titles)
	""
	(data (node-list-first titles)))))

;; Returns either the REFENTRYTITLE or the first REFNAME.
;;
(define (refentry-title nd)
  (let* ((refmeta (select-elements (descendants nd) "REFMETA"))
	 (refttl  (select-elements (descendants refmeta) "REFENTRYTITLE"))
	 (refndiv (select-elements (descendants nd) "REFNAMEDIV"))
	 (refname (select-elements (descendants refndiv) "REFNAME")))
    (if (node-list-empty? refttl)
	(if (node-list-empty? refname)
	    ""
	    (data (node-list-first refname)))
	(data (node-list-first refttl)))))

(define (bibliography-title nd)
  (let ((titles (select-elements (children nd) "TITLE")))
    (if (node-list-empty? titles)
	(gentext-element-name "BIBLIOGRAPHY")
	(data (node-list-first titles)))))

(define (glossary-title nd)
  (let ((titles (select-elements (children nd) "TITLE")))
    (if (node-list-empty? titles)
	(gentext-element-name "GLOSSARY")
	(data (node-list-first titles)))))

(define (index-title nd)
  (let ((titles (select-elements (children nd) "TITLE")))
    (if (node-list-empty? titles)
	(gentext-element-name "INDEX")
	(data (node-list-first titles)))))

(define (dedication-title nd)
  (let ((titles (select-elements (children nd) "TITLE")))
    (if (node-list-empty? titles)
	(gentext-element-name "DEDICATION")
	(data (node-list-first titles)))))

(define (section-title nd)
  (let ((titles (select-elements (children nd) "TITLE")))
    (if (node-list-empty? titles)
	""
	(data (node-list-first titles)))))

(define (refsection-title nd)
  (let ((titles (select-elements (children nd) "TITLE")))
    (if (node-list-empty? titles)
	""
	(data (node-list-first titles)))))

(define (block-title nd)
  (let ((titles (select-elements (children nd) "TITLE")))
    (if (node-list-empty? titles)
	""
	(data (node-list-first titles)))))

;; Returns the data of the title of the element
(define (element-title nd)
  (if (node-list-empty? nd)
      ""
      (case (gi nd)
	;; Use a seperately defined assoc list?
	(("APPENDIX") (appendix-title nd))
	(("ARTICLE") (article-title nd))
	(("BIBLIOGRAPHY") (bibliography-title nd))
	(("BOOK") (book-title nd))
	(("CHAPTER") (chapter-title nd))
	(("DEDICATION") (dedication-title nd))
	(("GLOSSARY") (glossary-title nd))
	(("INDEX") (index-title nd))
	(("PART") (part-title nd))
	(("PREFACE") (preface-title nd))
	(("REFENTRY") (refentry-title nd))
	(("REFERENCE") (reference-title nd))
	(("REFSECT1") (refsection-title nd))
	(("REFSECT2") (refsection-title nd))
	(("REFSECT3") (refsection-title nd))
	(("SECT1") (section-title nd))
	(("SECT2") (section-title nd))
	(("SECT3") (section-title nd))
	(("SECT4") (section-title nd))
	(("SECT5") (section-title nd))
	(("SET") (set-title nd))
	(else (block-title nd)))))

;; ======================================================================
;; Returns the element gi as a sosofo
;;
(define (element-gi-sosofo nd)
  (if (node-list-empty? nd)
      (empty-sosofo)
      (make sequence
	(literal (gentext-element-name (gi nd))))))

;; ======================================================================
