/*
 * dproc.c - Linux process access functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dproc.c,v 1.10 98/03/06 08:23:08 abe Exp $";
#endif

#include "lsof.h"

#if	defined(DEBIAN_LINUX_LSOF)
#include <sys/utsname.h>
#define	DEBIAN_SYSTEM_MAP	"/boot/System.map-"
#endif	/* defined(DEBIAN_LINUX_LSOF) */


_PROTOTYPE(static void get_kernel_access,(void));
_PROTOTYPE(static int hash_ksym,(char *nm));
_PROTOTYPE(static char *hasparmhash,(char *nm));
_PROTOTYPE(static int nlist_private,(char *fnm, struct NLIST_TYPE *nl));
_PROTOTYPE(static void process_mapmem,(struct inode *x, struct vm_area_struct *m));


/*
 * Local static values
 */

static int Coff = 0;			/* kernel loader format is COFF */
static KA_T Kta;			/* kernel task[] address */
static caddr_t *Nc = NULL;		/* node cache */
static MALLOC_S Nn = 0;			/* number of Nc[] entries allocated */
static char *MapNm[] = {		/* possible system map file paths */
	(char *)NULL,			/* place marker for an optional path */
	"/System.map",
	"/boot/System.map",
	"/zSystem.map",
	"/boot/zSystem.map",
	"/usr/src/linux/System.map",
	"/usr/src/linux/zSystem.map"
};
#define	MAPNMLEN	(sizeof(MapNm) / sizeof(char *))


/*
 * Local definitions
 */

#define	KSHASHSZ	256		/* kernel symbol hash table size */


/*
 * gather_proc_info() -- gather process information
 */

void
gather_proc_info()
{
	struct file *f;
	struct file fs;
	int nt;
	struct inode *ip;
	MALLOC_S len;
	int ppid = 0;
	short pss, sf;
	struct task_struct *t, ts;
	static struct task_struct **tt = NULL;
	uid_t uid;

#if	defined(HASHASHPID)
	int hlm;
	struct task_struct *thp;
	int ttlm = PIDHASH_SZ;
#else	/* !defined(HASHASHPID) */
	int ttlm = NR_TASKS;
#endif	/* defined(HASHASHPID) */

#if	LINUXV>1321
	struct files_struct fis;
	struct fs_struct fss;
	struct mm_struct mms;
#endif	/* LINUXV>1321 */

#if	defined(HASPPID)
	struct task_struct pt;
#endif	/* defined(HASPPID) */

#if	defined(HASDENTRY)
	struct dentry de;
#endif	/* defined(HASDENTRY) */
	

/*
 * Allocate space for task structure pointers and read them.
 */
	len = (MALLOC_S)(ttlm * sizeof(struct task_struct *));
	if (tt == (struct task_struct **)NULL) {
	    if ((tt = (struct task_struct **)malloc(len)) == NULL) {
		(void) fprintf(stderr,
		    "%s: no space for %d task structure pointers\n", Pn, ttlm);
		Exit(1);
	    }
	}
	if (kread(Kta, (char *)tt, (int)len)) {
	    (void) fprintf(stderr,
		"%s: can't read task structure pointers from %#lx\n", Pn, Kta);
	    Exit(1);
	}

#if	defined(HASNCACHE)
/*
 * Load the kernel name cache.
 */
	(void) ncache_load();
#endif	/* defined(HASNCACHE) */

/*
 * Examine task structures and their associated information.
 */
	for (nt = 0, t = &ts; nt < ttlm; nt++) {

#if	defined(HASHASHPID)
	/*
	 * If the task table has a hashed pointer array, follow each of
	 * its chains to its end.
	 */
	    for (hlm = 0, thp = tt[nt]; thp && hlm < NR_TASKS; hlm++) {
		if (kread((KA_T)thp, (char *)t, sizeof(ts)))
		    break;
		Tsp = thp;
		thp = t->pidhash_next;
#else	/* !defined(HASHASHPID) */
		if ((Tsp = tt[nt]) == (struct task_struct *)NULL
		||   kread((KA_T)Tsp, (char *)t, sizeof(ts)))
		    continue;
#endif	/* defined(HASHASHPID) */

		if (t->state == TASK_ZOMBIE)
		    continue;
		if (is_proc_excl((pid_t)t->pid, (pid_t)t->pgrp,
		    (UID_ARG)t->uid, &pss, &sf))
			continue;

#if	defined(HASPPID)
	    /*
	     * Get the parent task for its PID.
	     */
		if (Fppid || (Ffield && FieldSel[LSOF_FID_PPID].st)) {
		    if (!t->p_pptr
		    ||  kread((KA_T)t->p_pptr, (char *)&pt, sizeof(pt)))
			ppid = 0;
		    else
			ppid = (int)pt.pid;
		}
#endif	/* !defined(HASPPID) */

	    /*
	     * Allocate a local process structure.
	     */
		if (is_cmd_excl(t->comm, &pss, &sf))
		    continue;
		alloc_lproc(t->pid, t->pgrp, ppid, (UID_ARG)t->uid, t->comm,
		    (int)pss, (int)sf);
		Plf = (struct lfile *)NULL;

#if	LINUXV>1321
	    /*
	     * Read file system structure for Linux version 1.3.22 and above.
	     */
		if (t->fs == (struct fs_struct *)NULL
		||  kread((KA_T)t->fs, (char *)&fss, sizeof(fss)))
		    continue;
		
#endif	/* LINUXV>1321 */

	    /*
	     * Save current working directory information.
	     */
		Cfd = -1;
		Cfp = (struct file *)NULL;

#if	LINUXV<=1321
		if ((ip = t->PWD))
#else	/* LINUXV>1321 */
# if	defined(HASDENTRY)
		if (fss.pwd
		&&  kread((KA_T)fss.pwd, (char *)&de, sizeof(de)) == 0
		&&  (ip = de.d_inode))
# else	/* !defined(HASDENTRY) */
		if ((ip = fss.pwd))
# endif	/* defined(HASDENTRY) */
#endif	/* LINUXV<=1321 */

		{
		    alloc_lfile(CWD, -1);
		    process_inode(ip, (struct file_operations *)NULL);
		    if (Lf->sf)
			link_lfile();
		}
	    /*
	     * Save root directory information.
	     */

#if	LINUXV<=1321
		if ((ip = t->ROOT))
#else	/* LINUXV>1321 */
# if	defined(HASDENTRY)
		if (fss.root
		&&  kread((KA_T)fss.root, (char *)&de, sizeof(de)) == 0
		&&  (ip = de.d_inode))
# else	/* !defined(HASDENTRY) */
		if ((ip = fss.root))
# endif	/* defined(HASDENTRY) */
#endif	/* LINUXV<=1321 */

		{
		    alloc_lfile(RTD, -1);
		    process_inode(ip, (struct file_operations *)NULL);
		    if (Lf->sf)
			link_lfile();
		}
	    /*
	     * Save information on the executable file and mapped memory.
	     */

#if	LINUXV<1147
		if (t->executable || t->mmap)
		    process_mapmem(t->executable, t->mmap);
#else	/* LINUXV>=1147 */
# if	LINUXV<=1321
		if (t->mm->mmap)
		    process_mapmem(NULL, t->mm->mmap);
# else	/* LINUXV>1321 */
		if (t->mm
		&&  kread((KA_T)t->mm, (char *)&mms, sizeof(mms)) == 0
		&&  mms.mmap)
		    process_mapmem(NULL, mms.mmap);
# endif	/* LINUXV<=1321 */
#endif	/* LINUXV<1147 */

	    /*
	     * Save information on file descriptors.
	     */

#if	LINUXV>1321
		if (t->files == (struct files_struct *)NULL
		||  kread((KA_T)t->files, (char *)&fis, sizeof(fis)))
		    continue;
#endif	/* LINUXV>1321 */

		for (Cfd = 0, f = &fs; Cfd < NR_OPEN; Cfd++) {

#if	LINUXV<1147
		    if (!(Cfp = t->filp[Cfd]))
#else	/* LINUXV>=1147 */
# if	LINUXV<=1321
		    if (!(Cfp = t->files->fd[Cfd]))
# else	/* LINUXV>1321 */
		    if (!(Cfp = fis.fd[Cfd]))
# endif	/* LINUXV<=1321 */
#endif	/* LINUXV<1147 */

			continue;
		    if (kread((KA_T)Cfp, (char *)f, sizeof(fs)))
			continue;
		    alloc_lfile(NULL, Cfd);
		    process_file((KA_T)f);
		    if (Lf->sf)
			link_lfile();
		}
	    /*
	     * Examine results.
	     */
		if (examine_lproc())
		    return;

#if	defined(HASHASHPID)
	    }
#endif	/* defined(HASHASHPID) */

	}
}


/*
 * get_kernel_access() - get access to kernel memory
 */

static void
get_kernel_access()
{

#if	defined(WILLDROPGID)
/*
 * If kernel memory isn't coming from KMEM, drop setgid permission
 * before attempting to open the (Memory) file.
 */
	if (Memory)
		(void) dropgid();
#else	/* !defined(WILLDROPGID) */
/*
 * See if the non-KMEM memory file is readable.
 */
	if (Memory && !is_readable(Memory, 1))
		Exit(1);
#endif	/* defined(WILLDROPGID) */

/*
 * Open kernel memory access.
 */
	if ((Kd = open(Memory ? Memory : KMEM, O_RDONLY, 0)) < 0) {
		(void) fprintf(stderr, "%s: can't open %s: %s\n", Pn,
			Memory ? Memory : KMEM, strerror(errno));
		Exit(1);
	}

#if	defined(WILLDROPGID)
/*
 * Drop setgid permission, if necessary.
 */
	if (!Memory)
		(void) dropgid();
#else	/* !defined(WILLDROPGID) */
/*
 * See if the name list file is readable.
 */
	if (Nmlst && !is_readable(Nmlst, 1))
		Exit(1);
#endif	/* defined(WILLDROPGID) */

/*
 * Access kernel symbols.
 */
	(void) build_Nl(Drive_Nl);
	if (!Nmlst) {
	    if (!(Nmlst = get_nlist_path(1))) {
		(void) fprintf(stderr,
		    "%s: can't determine system map file path\n", Pn);
		Exit(1);
	    }
	}
	if (nlist_private(Nmlst, Nl) < 0) {
		(void) fprintf(stderr,
		    "%s: can't read kernel name list from %s\n", Pn, Nmlst);
		Exit(1);
	}
/*
 * Make sure important kernel symbols are defined.
 */
	if (get_Nl_value("task", Drive_Nl, (unsigned long *)&Kta) < 0 || !Kta)
	{
		(void) fprintf(stderr,
		    "%s: no %stask kernel definition\n",
		    Pn, Coff ? "_" : "");
		Exit(1);
	}
}


/*
 * get_nlist_path() - get kernel name list path
 */

char *
get_nlist_path(ap)
	int ap;				/* on success, return an allocated path
					 * string pointer if 1; return a
					 * constant character pointer if 0;
					 * return NULL if failure */
{
	int i;
	struct stat sb;

#if	defined(DEBIAN_LINUX_LSOF)
	char *cp = (char *)NULL;
	MALLOC_S len;
	struct utsname un;
/*
 * Add Debian Linux system map file name to MapNm[].
 */
	if (!MapNm[0]) {
	    if (uname(&un) != 0) {
		(void) fprintf(stderr, "%s: uname(2): %s\n",
		    Pn, strerror(errno));
		Exit(1);
	    }
	    len = strlen(DEBIAN_SYSTEM_MAP) + strlen(un.release) + 1;
	    if (!(cp = (char *)malloc(len))) {
		(void) fprintf(stderr,
		    "%s: can't allocate %d bytes for: %s%s\n",
		    Pn, len, DEBIAN_SYSTEM_MAP, un.release);
		Exit(1);
	    }
	    (void) sprintf(cp, "%s%s", DEBIAN_SYSTEM_MAP, un.release);
	    MapNm[0] = cp;
	}
#endif	/* defined(DEBIAN_LINUX_LSOF) */

	for (i = 0; i < MAPNMLEN; i++) {
	    if (!MapNm[i])
		continue;
	    if (stat(MapNm[i], &sb) == 0) {
		if (!ap)
		    return("");
		return(MapNm[i]);
	    }
	}
	return((char *)NULL);
}


/*
 * hash_ksym() - hash kernel symbol by name
 */

static int
hash_ksym(nm)
	char *nm;
{
	int i, j;

	for (i = j = 0; *nm; nm++) {
		i ^= (int)*nm << j;
		j ^= 1;
	}
	return(i % KSHASHSZ);
}


/*
 * hasparmhash() - does symbol name have parameter hash
 *
 * Note: this code identifies names that terminate with a module version
 * number of the form _R<8 character hex address> by returning the character
 * address of the version number (i.e., of the '_').
 */

static char *
hasparmhash(nm)
	char *nm;			/* pointer to symbol name */
{
	char *cp, *rv;
	int n;

	if ((cp = strrchr(nm, '_')) == NULL || *(cp + 1) != 'R')
		return(NULL);
	rv = cp;
	for (cp += 2, n = 0; n < 8; cp++, n++) {
		if ((*cp >= '0' && *cp <= '9')
		||  (*cp >= 'a' && *cp <= 'f')
		||  (*cp >= 'A' && *cp <= 'F'))
			continue;
		return(NULL);
	}
	return((*cp == '\0') ? rv : NULL);
}


/*
 * initialize() - perform all initialization
 */

void
initialize()
{
	size_t sz;
	unsigned long hi, st;

	get_kernel_access();
}


/*
 * kread() - read from kernel memory
 *
 * return: 0 = success
 *	   1 = read failed (perhaps some data was read, but not all)
 *	  -1 = seek to kernel memory address failed
 */

int
kread(addr, buf, len)
	KA_T addr;			/* kernel memory address */
	char *buf;			/* buffer to receive data */
	READLEN_T len;			/* length to read */
{
	int br;

	if (lseek(Kd, (off_t)addr, SEEK_SET) == (off_t)-1)
		return(-1);
	br = read(Kd, buf, len);
	return((br == (int)len) ? 0 : 1);
}


/*
 * nlist_private() - private get name list
 */

static int
nlist_private(fnm, nl)
	char *fnm;			/* name of file to search */
	struct NLIST_TYPE *nl;		/* names to locate */
{
	unsigned long addr;
	char buf[128];
	char *cp, *eol, *nm;
	int elf = 0;
	int i, j, mm, nf, nk, nn;
	FILE *nfs;
	struct ksym {
		char *name;
		unsigned long addr;
		struct ksym *next;
	} **kh, *kp, *kpn;
	struct NLIST_TYPE *np;

#if	defined(HAS_QUERY_MODULE)
	void *qb;
	int qbl;
	size_t qmr;
	struct module_symbol *qp;
#else	/* !defined(HAS_QUERY_MODULE) */
	int ik;
	struct kernel_sym *ksa = NULL;
	struct kernel_sym *ks;
#endif	/* defined(HAS_QUERY_MODULE) */

	if ((nfs = fopen(fnm, "r")) == NULL)
		return(-1);

#if	defined(HAS_QUERY_MODULE)
/*
 * Use the query_module syscall to get kernel symbol names.  This
 * involves determining the number of symbols before reading them.
 */
	for (qb = (void *)NULL, qbl = 0;;) {
	    if (query_module((char *)NULL, QM_SYMBOLS, qb, qbl, &qmr)) {
		if (errno != ENOSPC) {
		    if (!Fwarn) {
			if (errno = ENOSYS) {
			    (void) fprintf(stderr,
				"%s: query_module unimplemented\n", Pn);
			    (void) fprintf(stderr,
				"      CONFIG_MODULES not defined in autoconf.h?\n");
			} else {
			    (void) fprintf(stderr,
				"%s: query_module error: %s\n",
				Pn, strerror(errno));
			}
			(void) fprintf(stderr,
			    "%s: WARNING: unable to verify symbols in %s\n",
			    Pn, fnm);
		    }
		    qbl = 0;
		    if (qb) {
			(void) free((FREE_P *)qb);
			qb = (void *)NULL;
		    }
		    break;
		}
		qbl = qmr;
		if (!qb)
		    qb = (void *)malloc(qbl + 1);
		else
		    qb = (void *)realloc(qb, qbl + 1);
		if (!qb) {
		    (void) fprintf(stderr,
			"%s: no space (%d) for query_module buffer\n",
			Pn, qbl + 1);
		    Exit(1);
		}
		continue;
	    }
	    if (qbl)
		*((char *)(qb + qbl)) = '\0';
	    break;
	}
	if (qbl && qmr) {

	/*
	 * Allocate hash table space.
	 */
	    if ((kh = (struct ksym **)calloc((MALLOC_S)sizeof(struct ksym),
		      KSHASHSZ))
	    == NULL) {
		(void) fprintf(stderr,
		    "%s: no space for %d byte kernel symbol hash table\n",
		    Pn, (KSHASHSZ * sizeof(struct ksym)));
		Exit(1);
	    }
	/*
	 * Scan the query_module buffer and store its addresses, hashed
	 * by their names for later comparison to the same values in
	 * /[z]System.map.
	 *
	 * Look for the symbol "_system_utsname" or "system_utsname" to
	 * determine the loader format of the kernel: it's * COFF if the
	 * symbol has a leading `_'; ELF if it does not.
	 */
	    for (i = 0, qp = (struct module_symbol *)qb; i < qmr; i++, qp++) {
		if ((j = (int)qp->name) >= qbl)
		    continue;
		cp = (char *)(qb + j);
		if (!*cp || *cp == '#')
		    continue;
		if ((kp = (struct ksym *)malloc((MALLOC_S)sizeof(struct ksym)))
		== NULL) {
		    (void) fprintf(stderr,
			"%s: no space for kernel symbol structure: %s\n",
			Pn, nm);
		    Exit(1);
		}
		if ((nm = hasparmhash(cp)) != (char *)NULL)
		    *nm = '\0';	/* chop off module version number */
		if ((kp->name = (char *)malloc((MALLOC_S)(strlen(cp) + 1)))
		== (char *)NULL) {
		    (void) fprintf(stderr,
			"%s: no space for kernel symbol name: %s\n", Pn, nm);
		    Exit(1);
		}
		j = hash_ksym(cp);
		(void) strcpy(kp->name, cp);
		if (strcmp(cp, "_system_utsname") == 0)
		    Coff = 1;
		else if (strcmp(cp, "system_utsname") == 0)
		    elf = 1;
		kp->addr = qp->value;
		kp->next = kh[j];
		kh[j] = kp;
	    }
	    (void) free((FREE_P *)qb);
	} else
	    kh = (struct ksym **)NULL;
#else	/* !defined(HAS_QUERY_MODULE) */
/*
 * Read the kernel symbols via get_kernel_syms().
 */
	if ((nk = get_kernel_syms(NULL)) < 0) {
	    if (!Fwarn) {
		if (errno == ENOSYS) {
		    (void) fprintf(stderr,
			"%s: WARNING: get_kernel_syms() unimplemented\n", Pn);
		    (void) fprintf(stderr,
		        "      CONFIG_MODULES not defined in autoconf.h?\n");
		} else {
		    (void) fprintf(stderr,
			"%s: get_kernel_syms() error: %s\n",
			Pn, strerror(errno));
		}
		(void) fprintf(stderr,
		    "%s: WARNING: unable to verify symbols in %s\n", Pn, fnm);
	    }
	} else {
	    i = nk * sizeof(struct kernel_sym);
	    if ((ksa = (struct kernel_sym *)malloc((MALLOC_S)i)) == NULL) {
		(void) fprintf(stderr, "%s: no space for kernel symbols\n", Pn);
		Exit(1);
	    }
	}
	if (nk > 0) {
	    if (get_kernel_syms(ksa) < 0) {
		(void) fprintf(stderr, "%s: get_kernel_syms: %s\n",
		    Pn, strerror(errno));
		Exit(1);
	    }
	/*
	 * Allocate hash table space.
	 */
	    if ((kh = (struct ksym **)calloc((MALLOC_S)sizeof(struct ksym),
		      KSHASHSZ))
	    == NULL) {
		(void) fprintf(stderr,
		    "%s: no space for %d byte kernel symbol hash table\n",
		    Pn, (KSHASHSZ * sizeof(struct ksym)));
		Exit(1);
	    }
	/*
	 * Scan the kernel symbol table and store their addresses, hashed
	 * by their names for later comparison to the same values in
	 * /[z]System.map.
	 *
	 * Look for the symbol "_system_utsname" or "system_utsname" to
	 * determine the loader format of the kernel: it's * COFF if the
	 * symbol has a leading `_'; ELF if it does not.
	 *
	 * Symbols whose names begin with `#' are module markers.  Non-kernel
	 * module markers have the module name after the `#'; kernel module
	 * markers have a `\0' after the `#'.
	 */
	    for (i = 0, ik = 1, ks = ksa; i < nk; i++, ks++) {
		if (ks->name[0] == '#') {
		    ik = !ks->name[1] ? 1 : 0;
		    continue;
		}
		if (!ik)
		    continue;
		if ((kp = (struct ksym *)malloc((MALLOC_S)sizeof(struct ksym)))
		== NULL) {
		    (void) fprintf(stderr,
			"%s: no space for kernel symbol structure: %s\n",
			Pn, nm);
		    Exit(1);
		}
		if ((nm = hasparmhash(ks->name)) != (char *)NULL)
		    *nm = '\0';	/* chop off module version number */
		if ((kp->name = (char *)malloc((MALLOC_S)(strlen(ks->name)+1)))
		== (char *)NULL) {
		    (void) fprintf(stderr,
			"%s: no space for kernel symbol name: %s\n", Pn, nm);
		    Exit(1);
		}
		j = hash_ksym(ks->name);
		(void) strcpy(kp->name, ks->name);
		if (strcmp(ks->name, "_system_utsname") == 0)
		    Coff = 1;
		else if (strcmp(ks->name, "system_utsname") == 0)
		    elf = 1;
		kp->addr = ks->value;
		kp->next = kh[j];
		kh[j] = kp;
	    }
	    (void) free((FREE_P *)ksa);
	} else
	    kh = (struct ksym **)NULL;
#endif	/* defined(HAS_QUERY_MODULE) */

/*
 * Complain if we don't know the kernel binary's format, COFF or ELF, or if
 * we have conflicting evidence.  In either case, set the default format.
 */
	if ((!Coff && !elf) || (Coff && elf)) {

#if	defined(KERN_LD_ELF)
		Coff = 0;
		elf = 1;
#else	/* !defined(KERN_LD_ELF) */
		Coff = 1;
		elf = 0;
#endif	/* defined(KERN_LD_ELF) */

	    if (!Fwarn) {
		(void) fprintf(stderr, "%s: WARNING: uncertain kernel", Pn);
		(void) fprintf(stderr, " loader format; assuming %s.\n",

#if	defined(KERN_LD_ELF)
			"ELF"
#else	/* !defined(KERN_LD_ELF) */
			"COFF"
#endif	/* defined(KERN_LD_ELF) */

			);

	    }
	}
/*
 * Read the lines of the name list file.  Look for the symbols defined
 * in Nl[].  Look for any symbols also defined from the get_kernel_syms()
 * call and complain if their addresses don't match.  Count the symbols
 * in Nl[] and quit when addresses have been located for all of them.
 */
	for (nn = 0, np = nl; np->NL_NAME; nn++, np++)
		;
	mm = nf = 0;
	while (nf < nn && fgets(buf, sizeof(buf), nfs) != NULL) {
	    if ((eol = strchr(buf, '\n')) == NULL)
		continue;
	    if ((cp = strchr(buf, ' ')) == NULL)
		continue;
	    nm = cp + 3;
	    *eol = '\0';
	    addr = strtoul(buf, &cp, 16);
	    if (kh) {

	    /*
	     * Look for name in the kernel symbol hash table; if it's there,
	     * check the address; complain in detail about the first mismatch;
	     * count mismatches.
	     */
		for (kp = kh[hash_ksym(nm)]; kp; kp = kp->next) {
		    if (strcmp(kp->name, nm) == 0)
			break;
		}
		if (kp && addr != kp->addr) {
		    if (++mm == 1 && !Fwarn) {
			(void) fprintf(stderr,
			    "%s: kernel symbol address mismatch: %s\n",
			    Pn, nm);
			(void) fprintf(stderr,
			    "      get_kernel_syms() value is %#x;", kp->addr);
			(void) fprintf(stderr, " %s value is %#x.\n",
			    fnm, addr);
		    }
		}
	    }
	/*
	 * Search for symbol in Nl[].
	 */
	    for (np = nl; np->NL_NAME && np->NL_NAME[0]; np++) {
		cp = Coff ? np->NL_NAME : (np->NL_NAME + 1);
		if (strcmp(nm, cp) != 0)
		    continue;
		if (np->n_type) {

		/*
		 * Warn about a duplicate.
		 */
		    if (!Fwarn)
			(void) fprintf(stderr,
			    "%s: WARNING: %s: ambiguous symbol: %s\n",
			    Pn, fnm, cp);
		    continue;
		}
	    /*
	     * Save address; set type to 1, signifying an address has been
	     * located.  Count the number of addresses located.
	     */
		np->n_value = addr;
		np->n_type = 1;
		nf++;
		break;
	    }
	}
	(void) fclose(nfs);
/*
 * Complete the kernel symbol mismatch message if there were additional
 * mismatches.  Quit if there were any mismatches.
 */
	if (mm) {
	    if (!Fwarn) {
		if (mm > 1) {
		    (void) fprintf(stderr,
			"      There %s %d additional mismatch%s.\n",
			(mm == 2) ? "was" : "were",
			mm - 1,
			(mm == 2) ? "" : "es");
		 }
		 (void) fprintf(stderr,
			"      %s and the booted kernel may not be a",
			fnm);
		 (void) fprintf(stderr, " matched set.\n");
	    }
	    Exit(1);
	}
/*
 * Free the hashed kernel name space.
 */
	if (kh) {
	    for (i = 0; i < KSHASHSZ; i++) {
		if ((kp = kh[i]) == NULL)
		    continue;
		while (kp) {
		    kpn = kp->next;
		    if (kp->name)
			(void) free((FREE_P *)kp->name);
		    (void) free((MALLOC_S *)kp);
		    kp = kpn;
		}
	    }
	    (void) free((FREE_P *)kh);
	}
	return(1);
}


/*
 * process_mapmem() - process mapped memory
 */

static void
process_mapmem(x, m)
	struct inode *x;
	struct vm_area_struct *m;
{
	int f;
	int i, j, lm;
	MALLOC_S ma;
	struct inode *ia;
	char *ty;
	struct vm_area_struct v;

#if	defined(HASDENTRY)
	struct dentry de;
#endif	/* defined(HASDENTRY) */

	for (f = 1, i = lm = 0; m; lm++) {

	/*
	 * Avoid infinite loop.
	 */
		if (lm > 1000) {
			if (!Fwarn)
				(void) fprintf(stderr,
					"%s: too many memory maps, PID %d\n",
					Pn, Lp->pid);
				return;
		}
		if (f) {
		
		/*
		 * The first time through the loop, use the executable inode
		 * address.
		 */
			ty = "txt";
			f = 0;
			if ((ia = x) == (struct inode *)NULL)
				continue;
		} else {

		/*
		 * After the first time through the loop, use the inode
		 * addresses in the memory map.
		 */
			if (kread((KA_T)m, (char *)&v, sizeof(v)))
				break;
			m = v.vm_next;

#if	defined(HASDENTRY)
			if (!v.vm_dentry
			||  kread((KA_T)v.vm_dentry, (char *)&de, sizeof(de))
			||  !(ia = de.d_inode))
#else	/* !defined(HASDENTRY) */
			if (!(ia = v.vm_inode))
#endif	/* defined(HASDENTRY) */

				continue;
			ty = "mem";
		}
	/*
	 * Skip duplicate inode addresses.
	 */
		for (j = 0; j < i; j++) {
			if (Nc[j] == (caddr_t)ia)
				break;
		}
		if (j < i)
			continue;
	/*
	 * Cache the inode address for duplicate checking.
	 */
		if (i >= Nn) {
			Nn += 10;
			ma = (MALLOC_S)(Nn * sizeof(caddr_t));
			if (Nc)
				Nc = (caddr_t *)realloc(Nc, ma);
			else
				Nc = (caddr_t *)malloc(ma);
			if (!Nc) {
			    (void) fprintf(stderr,
				"%s: no space for memmap pointers, PID %d\n",
				Pn, Lp->pid);
			    Exit(1);
			}
		}
		Nc[i++] = (caddr_t)ia;
	/*
	 * Save the inode information.
	 */
		alloc_lfile(ty, -1);
		process_inode(ia, (struct file_operations *)NULL);
		if (Lf->sf)
			link_lfile();
	}
}


#if	defined(HASNCACHE)


#include "dialects/linux/kmem/include/kncache.h"


/*
 * Device and inode cache values.
 */

# if	defined(HASDEVKNC) || defined(HASINOKNC) || defined(HASNFSKNC)
struct l_nch {				/* local cache entry structure */

#  if	defined(HASINOKNC)
	unsigned long na;		/* inode address */
	unsigned long kpda;		/* kernel parent dentry address */
	struct l_nch *pa;		/* parent local address */
	int rp;				/* parent is file system root */
#  else	/* !defined(HASINOKNC) */
	dev_t dev;			/* device */
	unsigned long dir;		/* parent directory inode number */
	unsigned long ino;		/* inode number */
	struct l_nch *pa;		/* parent Nc address */
#  endif	/* defined(HASINOKNC) */

	char *nm;			/* name, including '\0' terminator */
	unsigned char nl;		/* name length */
	unsigned char dup;		/* duplicate entry status */
};

static int Asz = 0;			/* cache allocated size */
static int Csz = 0;			/* cache current size */
static int Hm = 0;			/* cache hash mask */
static struct l_nch **Hp = (struct l_nch **)NULL;
					/* cache hash pointers */
static struct l_nch *Nmc = (struct l_nch *)NULL;
static int Nh = 0;			/* cache hash pointer count */
					/* cache */

#  if	defined(HASINOKNC)
_PROTOTYPE(static struct l_nch *ncache_addr,(unsigned long na));
_PROTOTYPE(static int ncache_isroot,(unsigned long na));
#  else	/* !defined(HASINOHNC) */
_PROTOTYPE(static struct l_nch *ncache_addr,(dev_t *d, unsigned long i));
#  endif	/* defined(HASINOKNC) */

_PROTOTYPE(static void ncache_ent_nm,(struct l_nch *lc, char *nm, int len));

#  if	LINUXV<=1321
#define	DEV	dev
#  else	/* LINUXV>1321 */
#define	DEV	dc_dev
#  endif	/* LINUXV<=1321 */

#  if	defined(HASINOKNC)
#define hash(na)	Hp+((((int)(na)>>2)*31415)&Hm)
#  else	/* !defined(HASINOHNC) */
#define hash(d, i)	Hp+((((int)(d + i)>>2)*31415)&Hm)
#  endif	/* defined(HASINOKNC) */


/*
 * ncache_addr() - look up a node's local address
 */

static struct l_nch *

#  if	defined(HASINOKNC)
ncache_addr(na)
	unsigned long na;		/* node address */
#  else	/* !defined(HASINOHNC) */
ncache_addr(d, i)
	dev_t *d;			/* device number */
	unsigned long i;		/* inode number */
#  endif	/* defined(HASINOKNC) */

{
	struct l_nch **hp;

#  if	defined(HASINOKNC)
	for (hp = hash(na); *hp; hp++) {
	    if ((*hp)->na == na)
		return(*hp);
	}
#  else	/* !defined(HASINOHNC) */
	for (hp = hash(*d, i); *hp; hp++) {
	    if ((*hp)->dev == *d && (*hp)->ino == i)
		return(*hp);
	}
#  endif	/* defined(HASINOKNC) */

	return((struct l_nch *)NULL);
}


/*
 * ncache_ent_nm() - enter name in l_nch entry
 */

static void
ncache_ent_nm(lc, nm, len)
	struct l_nch *lc;		/* entry receiving name */
	char *nm;			/* name address */
	int len;			/* name length */
{
	char *na;

	if (!(na = (char *)malloc(len + 1))) {
	    (void) fprintf(stderr, "%s: no space (%d) for cache name\n",
		Pn, len + 1);
	    Exit(1);
	}
	(void) strncpy(na, nm, len);
	na[len] = '\0';
	lc->nm = na;
	lc->nl = len;
}
# endif	/* defined(HASDEVKNC) || defined(HASINOKNC) || defined(HASNFSKNC) */


# if	defined(HASINOKNC)
/*
 * ncache_isroot() - is the node a file system root
 */

static int
ncache_isroot(na)
	unsigned long na;			/* node address */
{
	struct inode i;
	int j;
	struct mounts *mtp;
	static int nca = 0;
	static unsigned long *nc = (unsigned long *)NULL;
	static int ncu = 0;
	struct stat *sb;
/*
 * Search the list of already-cached root inode addresses.
 */
	for (j = 0; j < ncu; j++) {
	    if (nc[j] == na)
		return(1);
	}
/*
 * Read the inode for its device and inode number.
 *
 * Search the mount table for a match on device and inode numbers.
 */
	if (kread((KA_T)na, (char *)&i, sizeof(i)))
	    return(0);
	for (mtp = Mtab; mtp; mtp = mtp->next) {
	    if (!mtp->dir || !mtp->inode)
		continue;
	    if (i.i_dev == mtp->dev && i.i_ino == mtp->inode)
		break;
	}
	if (!mtp)
	    return(0);
/*
 * Cache the root inode address, allocating and expanding the cache,
 * as necessary.
 */
	if (ncu >= nca) {
	    nca += 10;
	    j = nca * sizeof(unsigned long *);
	    if (nc)
		nc = (unsigned long *)realloc(nc, j);
	    else
		nc = (unsigned long *)malloc(j);
	    if (!nc) {
		(void) fprintf(stderr, "%s: no room (%d) for root node cache\n",
		    Pn, j);
		Exit(1);
	    }
	    nca += 10;
	}
	nc[ncu++] = na;
	return(1);
}
# endif	/* defined(HASINOKNC) */


/*
 * ncache_load() -- load kernel's name cache
 */

void
ncache_load()
{

# if	defined(HASDEVKNC) || defined(HASINOKNC) || defined(HASNFSKNC)
	static char *c = (char *)NULL;
	static unsigned long cv[3];
	static MALLOC_S cl;
	int f, i, len, nx;
	struct l_nch **hp, *lc;

#  if	defined(HASDEVKNC)
	static int dl = DCACHE_SIZE * sizeof(struct dir_cache_entry);
	struct dir_cache_entry *ld;
#  endif	/* defined(HASDEVKNC) */

#  if	defined(HASINOKNC)
	struct dentry de;
	struct list_head *kl, *ll, lh, *lp;
	struct inode ib;
	KA_T ka;
	static char *nb = (char *)NULL;
	static int nbl = 0;
#  endif	/* defined(HASINOKNC) */

#  if	defined(HASNFSKNC)
	struct nfs_lookup_cache_entry *nd;
	static nl = NFS_LOOKUP_CACHE_SIZE*sizeof(struct nfs_lookup_cache_entry);
#  endif	/* defined(HASNFSKNC) */

	if (!Fncache)
	    return;
	if (Asz == 0) {

	/*
	 * Do startup (first-time) functions.
	 */
	    cl = 0;
	    cv[0] = cv[1] = cv[2] = (unsigned long)0;

#  if	defined(HASDEVKNC)
	/*
	 * Decode device cache parameters.
	 */
	    if (get_Nl_value("lv1c", Drive_Nl, &cv[0]) >= 0 && cv[0])
		Asz = DCACHE_SIZE;
	    if (get_Nl_value("lv2c", Drive_Nl, &cv[1]) >= 0 && cv[1])
		Asz += DCACHE_SIZE;
	    if (Asz)
		cl = DCACHE_SIZE * sizeof(struct dir_cache_entry);
	    else
		dl = 0;
#  endif	/* defined(HASDEVKNC) */

#  if	defined(HASINOKNC)
	/*
	 * Decode inode address cache parameters.
	 */
	    if (get_Nl_value("dehash", Drive_Nl, &cv[0]) >= 0 && cv[0]) {
		Asz = DCACHE_SIZE;
		cl = D_HASHSIZE * sizeof(struct dentry);
	    } else
		cl = 0;
#  endif	/* defined(HASINOKNC) */

#  if	defined(HASNFSKNC)
	/*
	 * Define NFS cache parameters.
	 */
	    if (get_Nl_value("nlkupc", Drive_Nl, &cv[2]) >= 0 && cv[2]) {
		i = NFS_LOOKUP_CACHE_SIZE;
		Asz += i;
		if ((i * sizeof(struct nfs_lookup_cache_entry)) > cl)
		    cl = i * sizeof(struct nfs_lookup_cache_entry);
	    } else
		nl = 0;
#  endif	/* defined(HASNFSKNC) */

	    if (Asz == 0)
		return;
	/*
	 * Allocate space for the local cache.
	 */
	    len = Asz * sizeof(struct l_nch);
	    if (!(Nmc = (struct l_nch *)malloc((MALLOC_S)len))) {
		(void) fprintf(stderr,
		    "%s: no space for %d byte local name cache\n", Pn, len);
		Exit(1);
	    }
	/*
	 * Allocate a buffer to read the kernel caches.
	 */
	    if ((c = (char *)malloc(cl)) == (char *)NULL) {
		(void) fprintf(stderr,
		    "%s: no space for %d byte kernel name cache copy\n",
		    Pn, cl);
		Exit(1);
	    }
	} else {

	/*
	 * Do setup for repeat calls.
	 */
	    if (Hp) {
		for (i = 0; i < (Nh + Csz); i++) {
		    if ((lc = Hp[i]) && lc->nm) {
			(void) free((FREE_P *)lc->nm);
			lc->nm = (char *)NULL;
		    }
		}
		(void) free((FREE_P *)Hp);
		Hp = NULL;
	    }
	}
	Csz = 0;
	lc = Nmc;

#  if	defined(HASDEVKNC)
/*
 * Read each device name cache and add its entries to the local name cache.
 */
	if (dl) {
	    for (nx = 0; nx < 2; nx++) {
		if (!cv[nx])
		    continue;
		if (kread((KA_T)cv[nx], c, dl))
		    continue;
		for (i = 0, ld = (struct dir_cache_entry *)c;
		     i < DCACHE_SIZE;
		     i++, ld++)
		{
		    if (!ld->DEV || !ld->dir || !ld->ino)
			continue;
		    if ((len = (int)ld->name_len) < 1 || len > DCACHE_NAME_LEN)
			continue;
		    if (len < 3 && ld->name[0] == '.') {
			if (len == 1 || (len == 2 && ld->name[1] == '.'))
			    continue;
		    }
		    lc->dev = ld->DEV;
		    lc->dir = ld->dir;
		    lc->ino = ld->ino;
		    (void) ncache_ent_nm(lc, ld->name, len);
		    lc->pa = (struct l_nch *)NULL;
		    lc->dup = 0;
		    lc++;
		    Csz++;
	        }
	    }
	}
#  endif	/* defined(HASDEVKNC) */

#  if	defined(HASINOKNC)
/*
 * Read inode address cache and add its entries to the local name cache.
 */
	if (!Asz)
	    return;
	if (kread((KA_T)(kl = (struct list_head *)cv[0]), c, cl))
	    return;
	ll = (struct list_head *)c;
	for (i = 0; i < D_HASHSIZE; i++, kl++, ll++) {
	    for (lp = ll->next; lp && (lp != kl);) {
		ka = (KA_T)list_entry(lp, struct dentry, d_hash);
		if (kread(ka, (char *)&de, sizeof(de)))
		    break;
		lp = de.d_hash.next;
		if (!de.d_inode || !de.d_name.name || (len = de.d_name.len) < 1
		||  len > MAXPATHLEN)
		    continue;
	    /*
	     * Allocate space for and read the name characters.
	     */
		if (len >= nbl) {
		    nbl = len + 1;
		    if (nb)
			nb = (char *)realloc(nb, nbl);
		    else
			nb = (char *)malloc(nbl);
		    if (!nb) {
			(void) fprintf(stderr,
			    "%s: can't allocate name buffer (%d)\n", Pn, nbl);
			Exit(1);
		    }
		}
		if (kread((KA_T)de.d_name.name, nb, len))
		    continue;
		nb[len] = '\0';
		if (len < 3 && nb[0] == '.') {
		    if (len == 1 || (len == 2 && nb[1] == '.'))
			 continue;
		}
		if (Csz >= Asz) {

		/*
		 * Increase local cache space allocation.
		 */
		    Asz += DCACHE_SIZE;
		    nx = Asz * sizeof(struct l_nch);
		    if (!(Nmc = (struct l_nch *)realloc(Nmc, nx))) {
		      (void) fprintf(stderr,
			  "%s: no realloc space (%d) for local name cache\n",
			  Pn, nx);
		      Exit(1);
		    }
		    lc = &Nmc[Csz];
		}
	    /*
	     * Create a local cache entry.
	     */
		lc->na = (unsigned long)de.d_inode;
		(void) ncache_ent_nm(lc, nb, len);
		lc->rp = 0;
		lc->kpda = (unsigned long)de.d_parent;
		lc->pa = (struct l_nch *)NULL;
		lc++;
		Csz++;
	    }
	}
#  endif	/* defined(HASINOHNC) */

#  if	defined(HASNFSKNC)
/*
 * Read NFS name cache and add its entries to the local cache.
 */
	if (nl && cv[2] && kread((KA_T)cv[2], c, nl) == 0) {
	    for (i = 0, nd = (struct nfs_lookup_cache_entry *)c;
	      i < NFS_LOOKUP_CACHE_SIZE;
	      i++, nd++)
	    {
		if (!nd->dev || !nd->inode)
		    continue;
		nd->filename[NFS_MAXNAMLEN] = '\0';
		if ((len = strlen(nd->filename)) < 1)
		    continue;
		if (len < 3 && nd->filename[0] == '.') {
		    if (len == 1 || nd->filename[1] == '.')
			continue;
		}
		lc->dev = (dev_t)nd->dev;
		lc->dir = (unsigned long)nd->inode;
		lc->ino = (unsigned long)nd->fattr.fileid;
		(void) ncache_ent_nm(lc, nd->filename, len);
		lc->pa = (struct l_nch *)NULL;
		lc->dup = 0;
		lc++;
		Csz++;
	    }
	}
#  endif	/* defined(HASNFSKNC) */

/*
 * Reduce memory usage, as required.
 */
	if (Csz < 1) {
	    Csz = 0;
	    if (!RptTm) {
		(void) free((FREE_P *)Nmc);
		Nmc = (struct l_nch *)NULL;
	    }
	    return;
	}
	if (Csz < Asz && !RptTm) {
	    len = Csz * sizeof(struct l_nch);
	    if ((Nmc = (struct l_nch *)realloc(Nmc, len))
	    == (struct l_nch *)NULL)
	    {
		(void) fprintf(stderr, "%s: can't realloc local name cache\n",
		    Pn);
		Exit(1);
	    }
	    Asz = Csz;
	}
/*
 * Build a hash table to locate cache entries.
 */
	for (Nh = 1; Nh < Csz; Nh <<= 1)
	    ;
	Nh <<= 1;
	Hm = Nh - 1;
	if ((Hp = (struct l_nch **)calloc(Nh + Csz, sizeof(struct l_nch *)))
	== (struct l_nch **)NULL) {
	    if (!Fwarn)
		(void) fprintf(stderr,
		    "%s: no space for %d name cache hash pointers\n",
		    Pn, Nh + Csz);
	    Exit(1);
	}
/*
 * Enter cache pointers in the hash table.  Look for entries with the
 * same keys that have different names.
 */
	for (i = 0, lc = Nmc; i < Csz; i++, lc++) {

# if	defined(HASINOKNC)
	    for (f = 0, hp = hash(lc->na); *hp; hp++)
# else	/* !defined(HASINOHNC) */
	    for (f = 0, hp = hash(lc->dev, lc->ino); *hp; hp++)
# endif	/* defined(HASINOKNC) */

	    {

# if	defined(HASINOKNC)
		if ((*hp)->na == lc->na)
# else	/* !defined(HASINOHNC) */
		if ((*hp)->dev == lc->dev && (*hp)->ino == lc->ino)
# endif	/* defined(HASINOKNC) */

		{
		    if (strcmp((*hp)->nm, lc->nm) == 0)
			f = 1;
		    else {
			f = 2;	/* same key, different names */
			break;
		    }
		}
	    }
	    if (!f)
		*hp = lc;
	    else if (f == 2) {

	    /*
	     * Since entries with the same key located, mark entries with
	     * those keys as duplicates.
	     */

# if	defined(HASINOKNC)
		for (hp = hash(lc->na); *hp; hp++)
# else	/* !defined(HASINOHNC) */
		for (hp = hash(lc->dev, lc->ino); *hp; hp++)
# endif	/* defined(HASINOKNC) */

		{

# if	defined(HASINOKNC)
		    if((*hp)->na == lc->na)
# else	/* !defined(HASINOHNC) */
		    if ((*hp)->dev == lc->dev && (*hp)->ino == lc->ino)
# endif	/* defined(HASINOKNC) */

			(*hp)->dup = 1;
		}
	    }
	}
/*
 * Make a final pass through the local cache and convert parent directory
 * keys to local name cache pointers.
 */
	for (i = 0, lc = Nmc; i < Csz; i++, lc++) {

# if	defined(HASINOKNC)
	    if (lc->kpda
	    && !kread((KA_T)lc->kpda, (char *)&de, sizeof(de))
	    && de.d_inode) {
		if (!(lc->pa = ncache_addr((unsigned long)de.d_inode))) {
		    lc->rp = ncache_isroot((unsigned long)de.d_inode);
		}
	    }
# else	/* !defined(HASINOKNC) */
	    if (lc->dir)
		lc->pa = ncache_addr(&lc->dev, lc->dir);
# endif	/* defined(HASINOKNC) */

	}
# else	/* !defined(HASDEVKNC) && !defined(HASINOKNC) && !defined(HASNFSKNC) */
	if (!Fwarn)
	    (void) fprintf(stderr,
		"%s: WARNING: kncache.h defines no kernel name caches.\n", Pn);
# endif	/* defined(HASDEVKNC) || defined(HASINOKNC) || defined(HASNFSKNC) */

}


/*
 * ncache_lookup() - look up a node's name in the kernel's name caches
 */

char *
ncache_lookup(buf, blen, fp)
	char *buf;			/* receiving name buffer */
	int blen;			/* receiving buffer length */
	int *fp;			/* full path reply */
{

# if	defined(HASDEVKNC) || defined(HASINOKNC) || defined(HASNFSKNC)
	char *cp = buf;
	struct l_nch *lc;
	int nl, rlen;

	*cp = '\0';
	*fp = 0;
/*
 * If the entry has an inode number that matches the inode number of the
 * file system mount point, return an empty path reply.  That tells the
 * caller to print the file system mount point name only.
 */
	if (Lf->inp_ty == 1 && Lf->fs_ino && Lf->inode == Lf->fs_ino)
	    return(cp);
/*
 * Look up the name cache entry for the key.
 */

#  if	defined(HASINOKNC)
	if (!Lf->na)
#  else	/* !defined(HASINOHNC) */
	if (!Lf->dev_def || Lf->inp_ty != 1)
#  endif	/* defined(HASINOKNC) */

		return(NULL);
	if (Csz

#  if	defined(HASINOKNC)
	&&  (lc = ncache_addr(Lf->na))
#  else	/* !defined(HASINOHNC) */
	&&  (lc = ncache_addr(&Lf->dev, Lf->inode))
#  endif	/* defined(HASINOKNC) */

	&&  !lc->dup)
	{
	    if ((nl = (int)lc->nl) > (blen - 1))
		return(*cp ? cp : NULL);
	    cp = buf + blen - nl - 1;
	    rlen = blen - nl - 1;
	    (void) strcpy(cp, lc->nm);

#  if	defined(HASINOKNC)
	    if (!lc->pa && lc->rp) {
		*fp = 1;
		return(*cp ? cp : NULL);
	    }
#  endif	/* defined(HASINOKNC) */

	/*
	 * Look up the name cache entries that are parents of the
	 * key.  Quit when:
	 *
	 *	there's no parent;
	 *	the parent is a duplicate cache entry;
	 *	the name is too large to fit in the receiving buffer.
	 */
	    for (;;) {
		if (!lc->pa) {

#  if	defined(HASINOKNC)
		    if (ncache_isroot(lc->na))
#  else	/* !defined(HASINOHNC) */
		    if (lc->dir && Lf->fs_ino
		    &&  lc->dir == Lf->fs_ino)
#  endif	/* defined(HASINOKNC) */

			*fp = 1;
			break;
		}
		lc = lc->pa;
		if (lc->dup)
		    break;

#  if	defined(HASINOKNC)
		if (ncache_isroot(lc->na))
#  else	/* !defined(HASINOHNC) */
		if (lc->ino && Lf->fs_ino
		&&  lc->ino == Lf->fs_ino)
#  endif	/* defined(HASINOKNC) */

		{
		    *fp = 1;
		    break;
		}
		if (((nl = (int)lc->nl) + 1) > rlen)
		    break;
		*(cp - 1) = '/';
		cp--;
		rlen--;
		(void) strncpy((cp - nl), lc->nm, nl);
		cp -= nl;
		rlen -= nl;

# if	defined(HASINOKNC)
		if (lc->rp && !lc->pa) {
		    *fp = 1;
		    break;
		}
# endif	/* defined(HASINOKNC) */

	    }
	    return(*cp ? cp : NULL);
	}
# endif	/* defined(HASDEVKNC) || defined(HASINOKNC) || defined(HASNFSKNC) */

	return(NULL);
}
#endif	/* defined(HASNCACHE) */
