/*
  File: tleds.c
  (X11) netTrafficLEDS - Copyright (C) 1997  Jouni.Lohikoski@iki.fi
  This can be run either on VT or in X.
  <URL:http://www.iki.fi/Jouni.Lohikoski/tleds.html> for more info and
  the latest version.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

---
Makefile:
all:	xtleds tleds
xtleds:	tleds.c
	gcc -O3 -Wall -o xtleds tleds.c -I /usr/X11R6/include/  
		-L /usr/X11R6/lib/ -lX11
tleds:	tleds.c
	gcc -DNO_X_SUPPORT -O3 -Wall -o tleds tleds.c 
---	
This has to be fixed somehow still, XFree needs "XkbDisable" on
v3.2 XFRee, which is no good. Put following two lines in your XF86Config if
you use this from X. "Xleds 2 3" is needed always when using X with tleds.
XF86Config:
	XkbDisable 	# v3.1 Xfree doesn't need this, v3.3 ?
	Xleds 2 3
*/
#define VERSION	"1.01"
#define MYNAME	"tleds"

/* If you don't want X stuff. */
#ifdef NO_X_SUPPORT
#define REMOVE_X_CODE 1
#else
#define REMOVE_X_CODE 0
#endif

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <paths.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>
#include <signal.h>
#if (! REMOVE_X_CODE)
#include <X11/Xlib.h>
#else
#define LedModeOff         0
#define LedModeOn          1
#endif
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <linux/kd.h>
#include <sys/ioctl.h>
#include <assert.h>

#ifndef TRUE
#define TRUE 1
#define FALSE 0
#endif
#define MAXLEN		255
#define KEYBOARDDEVICE	"/dev/console"
#define NETDEVFILENAME	"/proc/net/dev"
#define TERMINATESTR	"Program (and child) terminated.\n"
#define FIELDCOUNT 	12
#define DEEPSLEEP	10
#define DEFPPPDELAY 	200
#define DEFETHDELAY	100
#define NUMLOCKLED	2
#define SCROLLLOCKLED	3
typedef enum {CLEAR = 0, SET = 1, TOGGLE = 2} LedMode;

void		handle_my_argvs (char** interfaceName, int* sleeptime,
				int argc, char** argv);
void		usage (char* name);
int		get_sleeptime (int isDefinedByUser, char* interfaceName);
void		create_pid_file (pid_t pid, const char* name);
pid_t		get_old_pid ();
void		parent_wants_me_dead (int);
void		my_signal_handler (int);
void		my_exit ();
void		led (int what, LedMode mode);
char*		find_device_line (char* buffer, char* netDeviceName);
char**		split_on_blank (char* line);
void 		report_traffic (char** list);
inline void	clear_led (int what) { led(what, CLEAR); }
inline void	set_led (int what) { led(what, SET); }
inline void	toggle_led (int what) { led(what, TOGGLE); }
inline void	my_sleep (struct timeval sleeptimeval);

static const char	devFileName[] = NETDEVFILENAME;
static char		pidFileName[30] = ""; /* 30 should be enough */
static char		rootPidFileName[30] = "";
#if (! REMOVE_X_CODE)
static Display 		*myDisplay = NULL;
#else
static char		*myDisplay = NULL;
#endif
static int		keyboardDevice = 0;
static int		opt_b = FALSE, opt_d = FALSE, opt_h = FALSE,
			opt_k = FALSE, opt_q = FALSE, opt_v = FALSE;

int	main (int argc, char* argv[])
{
	char*	interfaceName;
	char	buffer[MAXLEN], oleds;
	char*	tmpPointer;
	char**	list;
	pid_t	pid, pid2;
	int	sleeptime;
	struct timeval sleeptimeval;
	
	interfaceName = NULL;
	sleeptime = 0;
	handle_my_argvs(&interfaceName, &sleeptime, argc, argv);
#ifdef DEBUG
	opt_b = TRUE;	/* We are debugging so don't go to the background */
#endif
	if (opt_v && !opt_q)
		fprintf(stderr,
		    "%s version %s, GPL (c) 1997 Jouni.Lohikoski@iki.fi\n",
		    MYNAME, VERSION);
	strcpy(pidFileName, _PATH_TMP);
	strcpy(rootPidFileName, _PATH_VARRUN);
	strcat(pidFileName, MYNAME); /* Was argv[0]. Probs coz/if path. */
	strcat(rootPidFileName, MYNAME);
	strcat(pidFileName, ".pid");
	strcat(rootPidFileName, ".pid");
	if (opt_k) {
		if (! (pid = get_old_pid())) {
			if (!opt_q) {
				fprintf(stderr,
					"Couldn't find what to kill.\n");
				perror(pidFileName);
			}
			exit(1);
		}
		kill(pid, SIGUSR1);
		if (!opt_q) 
			printf("One moment...(3 secs)...\n");
		sleep(3);
		if ((pid2 = get_old_pid())) {
			fprintf(stderr,
			   "PID: %d - Hmm...not sure if I succeeded in kill.\n",
			   pid2);
			exit(1);
		}
		if (! opt_q)
			printf("Killed. (The old PID was %d)\n", pid);
		return 0;
	}
	if (opt_h) {
		usage(argv[0]);
		return 0;
	}
	if (! opt_q) {
		printf("Setting keyboard LEDs based on %s %s %s %s\n",
		    "changes of Receive/Transmit\npackets of", interfaceName,
		    "in", devFileName);
		printf("Delay between updates is %d milliseconds.\n",
		    sleeptime);
	}
	if (! find_device_line(buffer, interfaceName) && !opt_q) {
		printf(
		    "There is currently no such interface as %s in %s.\n%s\n",
		    interfaceName, devFileName,
		    "Maybe later there will be. Kill me (-k) if ya want.");
	}
	
	if(! opt_b) {
		if (-1 == (pid = fork())) {
			perror("fork");
			return 1;
		}
	} else {
		pid = getpid();
	}
	if (pid) {
		create_pid_file(pid, argv[0]);
		if (! opt_q)
			printf("Running in %sground. Pid: %ld\n",
				(opt_b ? "fore" : "back"),
				(long)pid);
		if (! opt_b)
			exit(0);
	}
	if (atexit(my_exit)) {
		perror("atexit() failed");
		return 1;
	}
	if (! opt_b) {
		signal(SIGUSR1, parent_wants_me_dead);
	}
	signal(SIGHUP, SIG_IGN);
	signal(SIGTERM, my_signal_handler);
	signal(SIGINT, my_signal_handler);
	signal(SIGQUIT, my_signal_handler);
	signal(SIGTSTP, my_signal_handler); 
	signal(SIGUSR2, SIG_IGN);
	signal(SIGPIPE, my_signal_handler);
	if (! geteuid()) {	/* We are running as EUID root - CONSOLE */
		if (-1 == (keyboardDevice = open(KEYBOARDDEVICE, O_RDONLY))) {
			perror(KEYBOARDDEVICE);
			fprintf(stderr, TERMINATESTR);
			exit(1);
		}
	} else {		/* EUID not root */
#if (! REMOVE_X_CODE)
		if (! (myDisplay = XOpenDisplay(NULL))		/* X  */
	  		&& ioctl(0, KDGETLED, &oleds) ) { 	/* VT */
			perror("Can't open X DISPLAY on the current host.");
#else
		if (ioctl(0, KDGETLED, &oleds) ) {
			perror ("KDGETLED");
			fprintf(stderr,
				"Error reading current led setting.\n%s\n",
				"Maybe stdin is not a VT?");
#endif
	
			fprintf(stderr, TERMINATESTR);
			exit (1);
		}
	}
	sleeptimeval.tv_sec = (int)((long)sleeptime * 1000L) / 1000000L;
	sleeptimeval.tv_usec = (int)((long)sleeptime * 1000L) % 1000000L;
	
	/* The main loop */
	while (1) {
		if ((tmpPointer = find_device_line(buffer, interfaceName))) {
			list = split_on_blank(tmpPointer);
			report_traffic(list);
			my_sleep(sleeptimeval);
		} else {
			sleep(DEEPSLEEP);
		}
	}
	return 0;
}

char*	find_device_line (char* buffer, char* netDeviceName)
{
	static long	fileOffset = 0L; 
	register FILE*	devFile;
	
	if (! (devFile = fopen(devFileName, "r")) ) {
		perror(devFileName);
		exit(1);
	}
	/* Skip two lines. (the header) */
	/* Two choices how to do this. Didn't find any differences in speed. */
#if 0
	fgets(buffer, MAXLEN, devFile);
	fgets(buffer, MAXLEN, devFile);
#else
	if (fileOffset) {
		fseek(devFile, fileOffset, SEEK_SET);
	} else {
		fgets(buffer, MAXLEN, devFile);
		fileOffset += (long)strlen(buffer);
		fgets(buffer, MAXLEN, devFile);
		fileOffset += (long)strlen(buffer);
	}
#endif

	while ( fgets(buffer, MAXLEN, devFile) ) {
		while(isblank(*buffer))
			buffer++;
		if (buffer == strstr(buffer, netDeviceName)) {
			fclose(devFile);
			return buffer;
		}
	}
	fclose(devFile);
	return NULL;
}

void	my_sleep (struct timeval sleeptimevalptr)
{
	select(0, NULL, NULL, NULL, &sleeptimevalptr);
}
	
char**	split_on_blank (char* line)
{
/*
Has a "bug". If tried to monitor network traffic e.g. on "dummy".
Look at the end of this file for example /proc/net/dev listing.
*/

	static char* 	list[FIELDCOUNT] = {};
	register int 	i;

	i = 0;
	goto middle;	/* speed(?) hack */
	for (; i < FIELDCOUNT; i++) {
		while (isblank(*line))
			line++;
	middle:
		list[i] = line;
		while (! isblank(*line) && *line != ':' && *line != '\n')
			line++;
		*(line++) = '\0';
	}
	return list;
}
	
void	report_traffic (char** list)
{
	static long	formerReceived = 0L;
	static long	formerTransmitted = 0L;
	register long	received, transmitted;

	received = atol(list[1]);
	transmitted = atol(list[6]);
	
	if (received != formerReceived) {
		set_led(NUMLOCKLED);
		formerReceived = received;
	} else {
		clear_led(NUMLOCKLED);
	}
	
	if (transmitted != formerTransmitted) {
		set_led(SCROLLLOCKLED);
		formerTransmitted = transmitted;
	} else {
		clear_led(SCROLLLOCKLED);
	}
}

void	led (int led, LedMode mode)
{
	char	ledVal;
#if (! REMOVE_X_CODE)
	XKeyboardControl values;
#else
	struct {
		int	led_mode;
		int	led;
	} values;
#endif
#ifdef DEBUG
	printf("led(%d, %d)\n", led, (int)mode);
#endif
	switch (mode) {
		case SET:
			values.led_mode = LedModeOn;
			break;
		case CLEAR:
			values.led_mode = LedModeOff;
			break;
		case TOGGLE:
			values.led_mode = LedModeOn; /* So doesn't toggle */
	}
	values.led = led;
	if (myDisplay) {
#if (! REMOVE_X_CODE)
		XChangeKeyboardControl(myDisplay, KBLed | KBLedMode, &values);
		XSync(myDisplay, FALSE);
#endif
	} else {
		if (ioctl(keyboardDevice, KDGETLED, &ledVal)) {
			perror("KDGETLED");
			exit(1);
		}
		switch (led) {
			case SCROLLLOCKLED:
				if (mode == SET)
					ledVal |= LED_SCR;
				else
					ledVal &= ~LED_SCR;
				break;
			case NUMLOCKLED:
				if (mode == SET)
					ledVal |= LED_NUM;
				else
					ledVal &= ~LED_NUM;
				break;
			default:
				perror("wrong led-value");
				exit(1);
		}
		if (ioctl(keyboardDevice, KDSETLED, ledVal)) {
			perror ("KDSETLED");
			exit(1);
		}
	}
}

void	parent_wants_me_dead (int x)
{
	exit(x);
}

void	my_signal_handler (int x)
{
	exit(x);
}

void	my_exit()
{
	char	oleds;
	if (opt_b && ! opt_q)
		printf("Bye-Bye !\n");
	if(getpid() == get_old_pid())
		unlink(pidFileName);
	if (! ioctl(keyboardDevice, KDGETLED, &oleds)) { /* VT */
		clear_led(NUMLOCKLED);
		clear_led(SCROLLLOCKLED);
	}
	if (keyboardDevice) {	/* EUID root - CONSOLE */
		close(keyboardDevice);
		unlink(rootPidFileName);
	}
	if (myDisplay) {
#if (! REMOVE_X_CODE)
		XCloseDisplay(myDisplay);	/* X */
#endif
	}
}

void	create_pid_file (pid_t pid, const char* name)
{
	FILE*		pidFile;
	pid_t		oldPid;
	char		procFileName[80];
	char		*tmpPidFileName;
	char		pidString[11]; /* "length" of UINT_MAX */
	struct stat	status;
	int		isAnother;
	
	if (geteuid())
		tmpPidFileName = pidFileName;
	else
		tmpPidFileName = rootPidFileName;  /* root */
	/*
	We check if there already is the *.pid file and if maybe the
	process' (child) which created it is dead, so we could try to fix.
	*/
	isAnother = FALSE;
	oldPid = (pid_t) 0;
	if (! stat(pidFileName, &status)
	    || ! stat(rootPidFileName, &status)) { 
		if ((oldPid = get_old_pid())) {
			strcpy(procFileName, "/proc/");
			sprintf(pidString, "%ld", (long)oldPid);
			strcat(procFileName, pidString);
			strcat(procFileName, "/environ");			
			if(! stat(procFileName, &status)) { /* The old proc. */
				isAnother = TRUE;
			} else {
			/* The old process was not alive, so we try to fix. */
				if (unlink(pidFileName)) {
					perror(pidFileName);
					kill(pid, SIGUSR1);
					exit(1);
				}
				if (! geteuid()) {	/* root */
					if (unlink(rootPidFileName)) {
						perror(rootPidFileName);
						kill(pid, SIGUSR1);
						exit(1);
					}
				}
			}
		} else {
			isAnother = TRUE;
		}				
	}
	
	if (isAnother) {
		if (oldPid)
			fprintf(stderr, "(The old PID %ld) ", (long)oldPid);
		fprintf(stderr, "%s %s runnning.\n%s %s %s\n",
			"\nSorry, can't run. There might be another",
			name, "If not, try: rm", pidFileName, rootPidFileName);
		kill(pid, SIGUSR1);
		exit(1);
	}
		
	if( !(pidFile = fopen(tmpPidFileName, "w"))) {
		perror(tmpPidFileName);
		kill(pid, SIGUSR1);
		exit(1);
	}
	fprintf(pidFile, "%ld\n", (long)pid);
	fclose(pidFile);
	if (chmod(tmpPidFileName, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH)) {
		perror(tmpPidFileName);
		exit(1);
	}
	if (! geteuid()) {  /* EUID root */
		if (symlink(tmpPidFileName, pidFileName)) {
			perror(pidFileName);
			exit(1);
		}
	}
}

pid_t	get_old_pid ()
{
	FILE*	pidFile;
	long	returnValue;

	if (! (pidFile = fopen(pidFileName, "r"))) {
		if (! (pidFile = fopen(rootPidFileName, "r")))
			return (pid_t)0L;
	}
	fscanf(pidFile, "%ld", &returnValue);
	return (pid_t)returnValue;
}

void	handle_my_argvs (char** interfaceName, int* sleeptime,
			int argc, char* argv[])
{
	int	c;

	while(EOF != (c = getopt(argc, argv, "bd:hkqv"))) {
		switch (c) {
			case 'b':
				opt_b = TRUE;
				break;
			case 'd':
				opt_d = TRUE;
				*sleeptime
					= get_sleeptime(TRUE, NULL);
				break;
			case 'h':
				opt_h = TRUE;
				break;
			case 'k':
				opt_k = TRUE;
				break;
			case 'q':
				opt_q = TRUE;
				break;
			case 'v':
				opt_v = TRUE;
				break;
			default:
				opt_h = TRUE;
				/* assert(0); */
		}
	}
	*interfaceName = argv[optind];
	if (! *interfaceName || ! (*interfaceName)[0]) {
		opt_h = TRUE; /* We may also have opt_k so we won't get h. */
		return;
	}
	if (! *sleeptime)
		*sleeptime = get_sleeptime(FALSE, *interfaceName);
}

int	get_sleeptime (int isDefinedByUser, char* interfaceName)
{
	int	returnValue;
	
	if (isDefinedByUser) {
		returnValue = atol(optarg);
		if (returnValue < 0 || returnValue > 10000) {
			opt_h = TRUE;  /* Illegal value. */
			return 0;
		}
		return returnValue;
	} else {
	/* Ok, we have to figure ourselves what would be good update delay. */
		if (interfaceName == strstr(interfaceName, "eth"))
			returnValue = DEFETHDELAY;
		else
			returnValue = DEFPPPDELAY;
		return returnValue;
	}
}

void	usage (char* name)
{
	fprintf(stderr,
	    "Usage: %s [-bhkqv] [-d <update_delay>] <interface_name>\n",
	    name);
	fprintf(stderr, "Example: %s -d 300 ppp0\n", name);
	fprintf(stderr, "Options:\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n",
		"\t-b\tDon't go to the background.",
		"\t-d N\tSet update delay.",
		"\t\tN must be between 1 and 10000 (milliseconds)",
		"\t-h\tHelp. (this)",
		"\t-k\tKill (old) xtleds running.",
		"\t-q\tBe quiet.",
		"\t-v\tPrint version information.",
		"\t\t(`cat /proc/net/dev` to see your interfaces.)");
}

/*
If the format of /proc/net/dev is changed this program will prolly not work,
because the optimizations in e.g. find_device_line()

$ cat /proc/net/dev
Inter-|   Receive                  |  Transmit
 face |packets errs drop fifo frame|packets errs drop fifo colls carrier
     lo:      2    0    0    0    0        2    0    0    0     0    0
   eth0:   3154    0    0    0    0     2553    0    0    0     0    0
  dummy: No statistics available.
   ppp0:  26619    0    0    0    0    42230    0    0    0     0    0
$
*/   
