/* -*-Mode: C;-*-
 * XDELTA - RCS replacement and delta generator
 * Copyright (C) 1997  Josh MacDonald
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: xd-gdbm.c 1.2 Mon, 13 Oct 1997 19:19:20 -0700 jmacd $
 */

#include "xdelta.h"
#include "gdbm.h"

static XdFile*  xd_open (gchar* name, gint flags);
static gchar*   xd_fetch (XdFile *xd, gchar* key);
static datum    xd_fetch_n (XdFile *xd, gchar* key);
static gboolean xd_insert (XdFile *xd, gchar* key, gchar* data);
static gboolean xd_insert_n (XdFile *xd, gchar* key, gchar* data, gint data_len);

static XdFile*
xd_open (gchar* name, gint flags)
{
  GDBM_FILE dbf;
  gchar* versions_str;
  XdFile *xd;

  dbf = gdbm_open (name, 0, flags, 0666, NULL);

  if (!dbf)
    return NULL;

  xd = g_new (XdFile, 1);

  xd->dbf = dbf;

  versions_str = xd_fetch (xd, "xdelta-versions");

  if (!versions_str)
    goto bail;

  xd->versions = strtol (versions_str, NULL, 10);

  return xd;

bail:
  gdbm_close (dbf);
  return NULL;
}

XdFile*
xd_open_read (gchar* name)
{
  return xd_open (name, GDBM_READER | GDBM_FAST);
}

XdFile*
xd_open_write (gchar* name)
{
  return xd_open (name, GDBM_WRITER | GDBM_FAST);
}

void
xd_close (XdFile* xd)
{
  gdbm_close (xd->dbf);
  g_free (xd);
}

XdFile*
xd_create (gchar* name)
{
  GDBM_FILE dbf;
  XdFile *xd;

  dbf = gdbm_open (name, 0, GDBM_WRCREAT | GDBM_FAST, 0666, NULL);

  if (!dbf)
    return NULL;

  xd = g_new (XdFile, 1);

  xd->dbf = dbf;
  xd->versions = 0;

  if (!xd_insert (xd, "xdelta-versions", "0"))
    {
      g_free (xd);
      goto bail;
    }

  return xd;
bail:
  gdbm_close (dbf);

  return NULL;
}

static gboolean
xd_insert (XdFile *xd, gchar* key, gchar* data)
{
  datum keyd, datad;

  keyd.dptr = key;
  keyd.dsize = strlen (key);

  datad.dptr = data;
  datad.dsize = strlen (data);

  return gdbm_store (xd->dbf, keyd, datad, GDBM_REPLACE) == 0;
}

static gboolean
xd_insert_n (XdFile *xd, gchar* key, gchar* data, gint data_len)
{
  datum keyd, datad;

  keyd.dptr = key;
  keyd.dsize = strlen (key);

  datad.dptr = data;
  datad.dsize = data_len;

  return gdbm_store (xd->dbf, keyd, datad, GDBM_REPLACE) == 0;
}

static gchar*
xd_fetch (XdFile *xd, gchar* key)
{
  datum keyd, content;
  gchar* str;

  keyd.dptr = key;
  keyd.dsize = strlen (key);

  content = gdbm_fetch (xd->dbf, keyd);

  if (!content.dptr)
    return NULL;

  str = g_realloc (content.dptr, content.dsize + 1);

  str[content.dsize] = 0;

  return str;
}

static datum
xd_fetch_n (XdFile *xd, gchar* key)
{
  datum keyd;

  keyd.dptr = key;
  keyd.dsize = strlen (key);

  return gdbm_fetch (xd->dbf, keyd);
}

gchar*
xd_get_date_str (XdFile* xd, gint ver)
{
  gchar req[64];
  gchar *date_str;
  time_t  date_time_t;

  sprintf (req, "version-%d-date", ver);

  if (!(date_str = xd_fetch (xd, req)))
    return NULL;

  date_time_t = strtol (date_str, NULL, 16);

  free (date_str);

  return time_t_to_rfc822 (date_time_t);
}

gchar*
xd_get_md5_str  (XdFile* xd, gint ver)
{
  gchar req[64];
  datum md5;
  static gchar formatted_md5[33];
  gint i;

  sprintf (req, "version-%d-md5", ver);

  if (!(md5 = xd_fetch_n (xd, req)).dptr)
    return NULL;

  for (i=0;i<16;i+=1)
    sprintf (formatted_md5 + i*2, "%02x", (0xff) & md5.dptr[i]);

  free (md5.dptr);

  return formatted_md5;
}

gchar*
xd_get_md5_raw  (XdFile* xd, gint ver)
{
  gchar req[64];
  datum md5;
  static char md5_buf[16];

  sprintf (req, "version-%d-md5", ver);

  if (!(md5 = xd_fetch_n (xd, req)).dptr)
    return NULL;

  memcpy (md5_buf, md5.dptr, 16);

  g_free (md5.dptr);

  return md5_buf;
}

gint
xd_get_len (XdFile* xd, gint ver)
{
  gchar req[64];
  guint8 *len_str;
  gint len;

  sprintf (req, "version-%d-len", ver);

  if (!(len_str = xd_fetch (xd, req)))
    return -1;

  len = strtol (len_str, NULL, 16);

  free (len_str);

  return len;
}

datum
xd_get_latest (XdFile* xd)
{
  return xd_fetch_n (xd, "latest-version");
}

gboolean xd_set_date (XdFile* xd, gint ver, time_t t)
{
  gchar req[64];
  gchar buf[64];

  sprintf (req, "version-%d-date", ver);

  sprintf (buf, "%lx", (gulong)t);

  return xd_insert (xd, req, buf);
}

gboolean xd_set_md5  (XdFile* xd, gint ver, gchar raw_md5[16])
{
  gchar req[64];

  sprintf (req, "version-%d-md5", ver);

  return xd_insert_n (xd, req, raw_md5, 16);
}

gboolean xd_set_len  (XdFile* xd, gint ver, gint len)
{
  gchar req[64];
  gchar buf[64];

  sprintf (req, "version-%d-len", ver);
  sprintf (buf, "%lx", (gulong)len);

  return xd_insert (xd, req, buf);
}

gboolean xd_set_latest (XdFile* xd, guint8* seg, gint len)
{
  return xd_insert_n (xd, "latest-version", seg, len);
}

gboolean xd_set_versions (XdFile* xd, gint versions)
{
  gchar buf[64];

  sprintf (buf, "%d", versions);

  xd->versions = versions;

  return xd_insert (xd, "xdelta-versions", buf);
}

datum
xd_get_delta (XdFile* xd, gint delta)
{
  gchar req[64];

  sprintf (req, "delta-%d", delta);

  return xd_fetch_n (xd, req);
}

gboolean
xd_append_delta (XdFile* xd, guint8* seg, gint len)
{
  gchar req[64];

  sprintf (req, "delta-%d", xd->versions-1);

  return xd_insert_n (xd, req, seg, len);
}

gint
xd_checkin (XdFile* xd, gchar* file)
{
  MappedFile* map;
  guint8 raw_md5[16];
  time_t ci_time;

  map = map_file (file);

  if (!map)
    {
      g_print ("xdelta: open %s failed: %s\n", file, strerror (errno));
      return 2;
    }

  g_print ("xdelta: checking in version %d\n", xd->versions);

  ci_time = get_utc_time_t ();
  md5_buffer (map->seg, map->len, raw_md5);

  if (!xd_set_md5 (xd, xd->versions, raw_md5))
    goto bail;

  if (!xd_set_date (xd, xd->versions, ci_time))
    goto bail;

  if (!xd_set_len (xd, xd->versions, map->len))
    goto bail;

  if (xd->versions > 0)
    {
      datum old_last;
      datum new_delta;
      MatchQuery *query;
      FromSegment *from_segs[MAX_PAST_DELTAS*2];
      gint from_seg_count = 0, i;

      old_last = xd_get_latest (xd);

      if (!old_last.dptr)
	goto bail;

      for (i = MAX (0, xd->versions - 1 - MAX_PAST_DELTAS); i < xd->versions - 1; i += 1)
	{
	  datum dat;

	  dat = xd_get_delta (xd, i);

	  if (!dat.dptr)
	    goto bail;

#ifdef DEBUG_XD
	  g_print ("*** checkin %d using delta %d\n", xd->versions, i);
#endif

	  from_segs[from_seg_count++] = from_segment_new (dat.dptr, dat.dsize);
	}

      from_segs [from_seg_count++] = from_segment_new (map->seg, map->len);

      query = match_query_new (NULL, old_last.dptr, old_last.dsize, 4, from_segs, from_seg_count);

      xdelta (query);

      new_delta = xdelta_to_bytes (query);

      if (!new_delta.dptr)
	goto bail;

      if (!xd_append_delta (xd, new_delta.dptr, new_delta.dsize))
	goto bail;
    }

  if (!xd_set_latest (xd, map->seg, map->len))
    goto bail;

  if (!xd_set_versions (xd, xd->versions + 1))
    goto bail;

  return 0;

bail:
  g_print ("xdelta: checkin failed: %s\n", gdbm_strerror (gdbm_errno));

  return 2;
}

gint
xd_checkout (XdFile* xd, gchar* file, gint ver)
{
  FILE* out;
  gint version;
  datum co;
  gint co_version;
  gboolean verify = TRUE;

  out = fopen (file, FOPEN_WRITE_ARG);

  if (!out)
    {
      g_print ("xdelta: open %s failed: %s\n", file, strerror (errno));
      return 2;
    }

  if (ver >= 0)
    version = ver;
  else
    version = xd->versions - 1;

  if (version >= xd->versions)
    {
      g_print ("xdelta: illegal version\n");
      goto bail;
    }

  g_print ("xdelta: checking out version %d\n", version);

  co = xd_get_latest (xd);

  if (!co.dptr)
    goto bail;

  co_version = xd->versions - 1;

  if (version != co_version)
    {
      datum *from_segs = g_new0 (datum, xd->versions);
      datum nco;
      gint i;

      while (version != co_version)
	{
	  datum delta = xd_get_delta (xd, co_version - 1);
	  gint nsegs = xdelta_bytes_seg_count (delta);

	  if (nsegs < 0)
	    goto bail;

	  assert (!from_segs [co_version - 1].dptr);

	  from_segs[co_version - 1] = co;

	  for (i=co_version - 2; i >= 0 && i >= co_version - nsegs; i -= 1)
	    {
#ifdef DEBUG_XD
	      g_print ("*** checkout %d using delta %d\n", co_version, i);
#endif

	      if (from_segs[i].dptr)
		continue;
	      else
		{
		  datum odelta = xd_get_delta (xd, i);

		  if (!odelta.dptr)
		    goto bail;

		  from_segs [i] = odelta;
		}
	    }

	  nco = xpatch (delta, from_segs + co_version - nsegs, nsegs);

	  if (!nco.dptr)
	    goto bail;

	  g_free (co.dptr);
	  from_segs[co_version - 1].dptr = NULL;

	  co = nco;
	  co_version -= 1;
	}

      for (i = 0; i < xd->versions; i += 1)
	if (from_segs[i].dptr)
	  g_free (from_segs[i].dptr);

      g_free (from_segs);
    }

  if (verify)
    {
      gchar md5[16], *correct;

      md5_buffer (co.dptr, co.dsize, md5);

      correct = xd_get_md5_raw (xd, co_version);

      if (memcmp (correct, md5, 16) != 0)
	{
	  g_print ("xdelta: incorrect checksum on checkout\n");
	  goto bail;
	}
    }

  if (fwrite (co.dptr, co.dsize, 1, out) != 1 || fclose (out))
    {
      g_print ("xdelta: write %s failed: %s\n", file, strerror (errno));
      goto bail;
    }

  return 0;

bail:
  g_print ("xdelta: checkout failed: %s\n", gdbm_strerror (gdbm_errno));

  return 2;
}
