/*======================================================================

    A Flash Translation Layer memory card driver

    This driver implements a disk-like block device driver with an
    apparent block size of 512 bytes for flash memory cards.

    Written by David Hinds, dhinds@allegro.stanford.edu
    
======================================================================*/

#include "config.h"
#include "k_compat.h"

/* #define PSYCHO_DEBUG */

#ifdef MODULE
#define init_ftl_cs init_module
#endif

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/ptrace.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/major.h>
#include <linux/fs.h>
#include <linux/ioctl.h>
#include <asm/io.h>
#include <asm/system.h>
#include <asm/segment.h>
#include <stdarg.h>

#include "version.h"
#include "cs_types.h"
#include "cs.h"
#include "bulkmem.h"
#include "cistpl.h"
#include "ds.h"
#include "ftl.h"

/* Major device # for FTL device */
static int major_dev = 0;

/* Funky stuff for setting up a block device */
#define MAJOR_NR		major_dev
#define DEVICE_NAME		"FTL"
#define DEVICE_REQUEST		do_ftl_request
#define DEVICE_ON(device)
#define DEVICE_OFF(device)

#define DEVICE_NR(minor)	((minor)>>3)
#define REGION_NR(minor)	((minor)&7)
#define MINOR_NR(dev,rgn)	(((dev)<<3)+(rgn))

#include BLK_DEV_HDR

#ifdef PCMCIA_DEBUG
static int pc_debug = PCMCIA_DEBUG;
static char *version =
"ftl_cs.c 1.18 1996/05/28 15:16:02 (David Hinds)";
#endif

static int shuffle_freq = 50;

/*====================================================================*/

/* Maximum number of separate memory devices we'll allow */
#define MAX_DEV		8

/* Maximum number of outstanding erase requests per socket */
#define MAX_ERASE	8

/* Sector size -- shouldn't need to change */
#define SECTOR_SIZE	512

static void ftl_config(dev_link_t *link);
static void ftl_release(u_long arg);
static int ftl_event(event_t event, int priority,
		     event_callback_args_t *args);

static dev_link_t *ftl_attach(void);
static void ftl_detach(dev_link_t *);

/* Block IO request handler stuff */
static void do_ftl_request(void);

/* Each memory region corresponds to a minor device */
typedef struct partition_t {
    dev_node_t		dev;
    u_long		state;
    u_long		*VirtualBlockMap;
    u_long		*VirtualPageMap;
    u_long		FreeTotal;
    struct eun_info_t {
	u_long			Offset;
	u_long			EraseCount;
	u_long			Free;
	u_long			Deleted;
    } *EUNInfo;
    struct xfer_info_t {
	u_long			Offset;
	u_long			EraseCount;
	u_short			state;
    } *XferInfo;
    u_short		bam_index;
    u_long		*bam_cache;
    u_short		DataUnits;
    u_long		BlocksPerUnit;
    erase_unit_header_t	header;
    region_info_t	region;
    memory_handle_t	handle;
    int			open;
} partition_t;

/* Partition state flags */
#define FTL_FORMATTED	0x01

/* Transfer unit states */
#define XFER_UNKNOWN	0x00
#define XFER_ERASING	0x01
#define XFER_ERASED	0x02
#define XFER_PREPARED	0x03
#define XFER_FAILED	0x04

typedef struct ftl_dev_t {
    eraseq_handle_t	eraseq_handle;
    eraseq_entry_t	eraseq[MAX_ERASE];
    struct wait_queue	*erase_pending;
    partition_t		minor[2*CISTPL_MAX_DEVICES];
} ftl_dev_t;

static dev_info_t dev_info = "ftl_cs";
static dev_link_t *dev_table[MAX_DEV] = { NULL, /* ... */ };

static int ftl_blocksizes[MINOR_NR(MAX_DEV, 0)] =
{ 0, /* ... */ };

/*====================================================================*/

static int ftl_ioctl(struct inode *inode, struct file *file,
		     u_int cmd, u_long arg);
static int ftl_open(struct inode *inode, struct file *file);
static void ftl_close(struct inode *inode, struct file *file);

static struct file_operations ftl_blk_fops = {
    NULL,		/* lseek */
    block_read,		/* read */
    block_write,	/* write */
    NULL,		/* readdir */
    NULL,		/* select */
    ftl_ioctl,		/* ioctl */
    NULL,		/* mmap */
    ftl_open,		/* open */
    ftl_close,		/* release */
    block_fsync		/* fsync */
};

/*====================================================================*/

static void cs_error(int func, int ret)
{
    CardServices(ReportError, dev_info, (void *)func, (void *)ret);
}

/*======================================================================

    ftl_attach() creates an "instance" of the driver, allocating
    local data structures for one device.  The device is registered
    with Card Services.

======================================================================*/

static dev_link_t *ftl_attach(void)
{
    client_reg_t client_reg;
    dev_link_t *link;
    ftl_dev_t *dev;
    eraseq_hdr_t eraseq_hdr;
    int i, ret;
    
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(KERN_DEBUG "ftl_cs: ftl_attach()\n");
#endif

    for (i = 0; i < MAX_DEV; i++)
	if (dev_table[i] == NULL) break;
    if (i == MAX_DEV) {
	printk(KERN_NOTICE "ftl_cs: no devices available\n");
	return NULL;
    }
    
    /* Create new memory card device */
    link = kmalloc(sizeof(struct dev_link_t), GFP_KERNEL);
    dev_table[i] = link;
    memset(link, 0, sizeof(struct dev_link_t));
    link->release.function = &ftl_release;
    link->release.data = (u_long)link;
    
    dev = kmalloc(sizeof(struct ftl_dev_t), GFP_KERNEL);
    memset(dev, 0, sizeof(ftl_dev_t));
    link->priv = dev;

    /* Register with Card Services */
    client_reg.dev_info = &dev_info;
    client_reg.Attributes = INFO_IO_CLIENT | INFO_CARD_SHARE;
    client_reg.EventMask =
	CS_EVENT_RESET_PHYSICAL | CS_EVENT_CARD_RESET |
	CS_EVENT_CARD_INSERTION | CS_EVENT_CARD_REMOVAL |
	CS_EVENT_PM_SUSPEND | CS_EVENT_PM_RESUME;
    client_reg.event_handler = &ftl_event;
    client_reg.Version = 0x0210;
    client_reg.event_callback_args.client_data = link;
    ret = CardServices(RegisterClient, &link->handle, &client_reg);
    if (ret != CS_SUCCESS) {
	cs_error(RegisterClient, ret);
	ftl_detach(link);
	return NULL;
    }

    for (i = 0; i < MAX_ERASE; i++)
	dev->eraseq[i].State = ERASE_IDLE;
    eraseq_hdr.QueueEntryCnt = MAX_ERASE;
    eraseq_hdr.QueueEntryArray = dev->eraseq;
    dev->eraseq_handle = (void *)link->handle;
    ret = CardServices(RegisterEraseQueue, &dev->eraseq_handle, &eraseq_hdr);
    if (ret != CS_SUCCESS) {
	cs_error(RegisterEraseQueue, ret);
	dev->eraseq_handle = NULL;
	ftl_detach(link);
	return NULL;
    }
    
    return link;
} /* ftl_attach */

/*======================================================================

    This deletes a driver "instance".  The device is de-registered
    with Card Services.  If it has been released, all local data
    structures are freed.  Otherwise, the structures will be freed
    when the device is released.

======================================================================*/

static void ftl_detach(dev_link_t *link)
{
    ftl_dev_t *dev;
    int i;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(KERN_DEBUG "ftl_cs: ftl_detach(0x%p)\n", link);
#endif
    
    /* Locate device structure */
    for (i = 0; i < MAX_DEV; i++)
	if (dev_table[i] == link) break;
    if (i == MAX_DEV)
	return;

    if (link->state & DEV_CONFIG) {
	ftl_release((u_long)link);
	if (link->state & DEV_STALE_CONFIG) {
	    link->state |= DEV_STALE_LINK;
	    return;
	}
    }

    dev = (ftl_dev_t *)link->priv;
    if (dev->eraseq_handle)
	CardServices(DeregisterEraseQueue, dev->eraseq_handle);
    if (link->handle)
	CardServices(DeregisterClient, link->handle);
    
    /* Unlink device structure, free bits */
    dev_table[i] = NULL;
    kfree_s(dev, sizeof(struct ftl_dev_t));
    kfree_s(link, sizeof(struct dev_link_t));
    
} /* ftl_detach */

/*======================================================================

    ftl_config() is scheduled to run after a CARD_INSERTION event
    is received, to configure the PCMCIA socket, and to make the
    ethernet device available to the system.
    
======================================================================*/

static void ftl_config(dev_link_t *link)
{
    ftl_dev_t *dev;
    partition_t *minor;
    region_info_t region;
    dev_node_t **tail;
    int i, ret, nr;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(KERN_DEBUG "ftl_cs: ftl_config(0x%p)\n", link);
#endif

    /* Configure card */
    link->state |= DEV_CONFIG;

    for (i = 0; i < MAX_DEV; i++)
	if (dev_table[i] == link) break;
    dev = (ftl_dev_t *)link->priv;
    tail = &link->dev;
    minor = dev->minor;
    nr = 0;
    region.Attributes = REGION_TYPE_CM;
    ret = CardServices(GetFirstRegion, link->handle, &region);
    while (ret == CS_SUCCESS) {
	minor->region = region;
	sprintf(minor->dev.dev_name, "ftl%dc%d", i, nr);
	minor->dev.major = major_dev;
	minor->dev.minor = MINOR_NR(i, nr);
	*tail = &minor->dev; tail = &minor->dev.next;
	minor++; nr++;
	ret = CardServices(GetNextRegion, link->handle, &region);
    }
    *tail = NULL;
    
    link->state &= ~DEV_CONFIG_PENDING;
    
    if (nr == 0)
	printk(KERN_NOTICE "ftl_cs: no regions found!\n");
    else {
	printk(KERN_INFO "ftl_cs: ftl%d:", i);
	minor = dev->minor;
	for (i = 0; i < nr; i++) {
	    if (minor[i].region.RegionSize & 0xfffff)
		printk(" %ld kb", minor[i].region.RegionSize >> 10);
	    else
		printk(" %ld mb", minor[i].region.RegionSize >> 20);
	}
	printk("\n");
    }
    
} /* ftl_config */

/*======================================================================

    After a card is removed, ftl_release() will unregister the 
    device, and release the PCMCIA configuration.  If the device is
    still open, this will be postponed until it is closed.
    
======================================================================*/

static void ftl_release(u_long arg)
{
    dev_link_t *link = (dev_link_t *)arg;
    int i;
    
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(KERN_DEBUG "ftl_cs: ftl_release(0x%p)\n", link);
#endif

    for (i = 0; i < MAX_DEV; i++)
	if (dev_table[i] == link) break;
    if (link->open) {
#ifdef PCMCIA_DEBUG
	if (pc_debug)
	    printk(KERN_DEBUG "ftl_cs: release postponed, 'ftl%d'"
		   " still open\n", i);
#endif
	link->state |= DEV_STALE_CONFIG;
	return;
    }

    link->dev = NULL;
    if (link->win)
	CardServices(ReleaseWindow, link->win);
    link->state &= ~DEV_CONFIG;
    
    if (link->state & DEV_STALE_LINK)
	ftl_detach(link);
    
} /* ftl_release */

/*======================================================================

    The card status event handler.  Mostly, this schedules other
    stuff to run after an event is received.
    
======================================================================*/

static void save_status(eraseq_entry_t *erase);

static int ftl_event(event_t event, int priority,
		     event_callback_args_t *args)
{
    dev_link_t *link = args->client_data;
    ftl_dev_t *dev;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(KERN_DEBUG "ftl_cs: ftl_event()\n");
#endif
    
    switch (event) {
#ifdef PCMCIA_DEBUG
    case CS_EVENT_REGISTRATION_COMPLETE:
	if (pc_debug)
	    printk(KERN_DEBUG "ftl_cs: registration complete\n");
	break;
#endif
    case CS_EVENT_CARD_REMOVAL:
	link->state &= ~DEV_PRESENT;
	if (link->state & DEV_CONFIG) {
	    link->release.expires = RUN_AT(HZ/20);
	    add_timer(&link->release);
	}
	break;
    case CS_EVENT_CARD_INSERTION:
	link->state |= DEV_PRESENT | DEV_CONFIG_PENDING;
	ftl_config(link);
	break;
    case CS_EVENT_ERASE_COMPLETE:
	save_status((eraseq_entry_t *)(args->info));
	dev = (ftl_dev_t *)link->priv;
	wake_up(&dev->erase_pending);
	break;
    case CS_EVENT_PM_SUSPEND:
	link->state |= DEV_SUSPEND;
	/* Fall through... */
    case CS_EVENT_RESET_PHYSICAL:
	/* get_lock(link); */
	break;
    case CS_EVENT_PM_RESUME:
	link->state &= ~DEV_SUSPEND;
	/* Fall through... */
    case CS_EVENT_CARD_RESET:
	/* free_lock(link); */
	break;
    }
    return 0;
} /* ftl_event */

/*======================================================================

    Scan_partition() checks to see if a memory region contains an FTL
    partition.  It reads all the erase unit headers, builds the erase
    unit map, and then builds the virtual page map.
    
======================================================================*/

static int scan_partition(partition_t *part)
{
    erase_unit_header_t header;
    mem_op_t req;
    u_short xvalid, xtrans, i;
    u_long blocks, j;
    int hdr_ok, ret;

    /* Search first megabyte for a valid FTL header */
    req.Attributes = MEM_OP_BUFFER_KERNEL;
    req.Count = sizeof(header);
    for (req.Offset = 0;
	 req.Offset < 0x100000;
	 req.Offset += part->region.BlockSize) {
	ret = CardServices(ReadMemory, part->handle, &req, &header);
	if (ret != CS_SUCCESS) {
	    cs_error(ReadMemory, ret);
	    return -1;
	}
	if (strcmp(header.DataOrgTuple+3, "FTL100") == 0) break;
    }
    if (req.Offset == 0x100000) {
	printk(KERN_NOTICE "ftl_cs: FTL header not found.\n");
	return -1;
    }
    if ((header.NumEraseUnits > 65536) || (header.BlockSize != 9) ||
	(header.EraseUnitSize < 10) || (header.EraseUnitSize > 31) ||
	(header.NumTransferUnits >= header.NumEraseUnits)) {
	printk(KERN_NOTICE "ftl_cs: FTL header corrupt!\n");
	return -1;
    }
    part->header = header;

    /* Set up erase unit maps */
    part->DataUnits = header.NumEraseUnits - header.NumTransferUnits;
    part->EUNInfo = kmalloc(part->DataUnits * sizeof(struct eun_info_t),
			    GFP_KERNEL);
    for (i = 0; i < part->DataUnits; i++)
	part->EUNInfo[i].Offset = 0xffffffff;
    part->XferInfo =
	kmalloc(header.NumTransferUnits * sizeof(struct xfer_info_t),
		GFP_KERNEL);

    xvalid = xtrans = 0;
    for (i = 0; i < part->header.NumEraseUnits; i++) {
	req.Offset = ((i + part->header.FirstPhysicalEUN)
		      << part->header.EraseUnitSize);
	ret = CardServices(ReadMemory, part->handle, &req, &header);
	if (ret != CS_SUCCESS) {
	    cs_error(ReadMemory, ret);
	    return -1;
	}
	/* Is this a transfer partition? */
	hdr_ok = (strcmp(header.DataOrgTuple+3, "FTL100") == 0);
	if (hdr_ok && (header.LogicalEUN < part->DataUnits) &&
	    (part->EUNInfo[header.LogicalEUN].Offset == 0xffffffff)) {
	    part->EUNInfo[header.LogicalEUN].Offset = req.Offset;
	    part->EUNInfo[header.LogicalEUN].EraseCount =
		header.EraseCount;
	    xvalid++;
	}
	else {
	    if (xtrans == part->header.NumTransferUnits) {
		printk(KERN_NOTICE "ftl_cs: format error: too many "
		       "transfer units!\n");
		return -1;
	    }
	    if (hdr_ok && (header.LogicalEUN == 0xffff)) {
		part->XferInfo[xtrans].state = XFER_PREPARED;
		part->XferInfo[xtrans].EraseCount = header.EraseCount;
	    }
	    else {
		part->XferInfo[xtrans].state = XFER_UNKNOWN;
		/* Pick anything reasonable for the erase count */
		part->XferInfo[xtrans].EraseCount =
		    part->header.EraseCount;
	    }
	    part->XferInfo[xtrans].Offset = req.Offset;
	    xtrans++;
	}
    }
    /* Check for format trouble */
    header = part->header;
    if ((xtrans != header.NumTransferUnits) ||
	(xvalid+xtrans != header.NumEraseUnits)) {
	printk(KERN_NOTICE "ftl_cs: format error: erase units "
	       "don't add up!\n");
	return -1;
    }
    
    /* Set up virtual page map */
    blocks = header.FormattedSize >> header.BlockSize;
    part->VirtualBlockMap =
	kmalloc(blocks * sizeof(u_long), GFP_KERNEL);
    memset(part->VirtualBlockMap, 0xff, blocks * sizeof(u_long));
    part->BlocksPerUnit = (1 << header.EraseUnitSize) >> header.BlockSize;
    req.Count = part->BlocksPerUnit * sizeof(u_long);

    part->bam_cache = kmalloc(part->BlocksPerUnit * sizeof(u_long),
			      GFP_KERNEL);
    part->bam_index = 0xffff;
    part->FreeTotal = 0;
    for (i = 0; i < part->DataUnits; i++) {
	part->EUNInfo[i].Free = 0;
	part->EUNInfo[i].Deleted = 0;
	req.Offset = part->EUNInfo[i].Offset + header.BAMOffset;
	ret = CardServices(ReadMemory, part->handle, &req,
			   part->bam_cache);
	if (ret != CS_SUCCESS) {
	    cs_error(ReadMemory, ret);
	    return -1;
	}
	for (j = 0; j < part->BlocksPerUnit; j++) {
	    if (BLOCK_FREE(part->bam_cache[j])) {
		part->EUNInfo[i].Free++;
		part->FreeTotal++;
	    }
	    else if ((BLOCK_TYPE(part->bam_cache[j]) == BLOCK_DATA) &&
		     (BLOCK_NUMBER(part->bam_cache[j]) < blocks))
		part->VirtualBlockMap[BLOCK_NUMBER(part->bam_cache[j])] =
		    (i << header.EraseUnitSize) + (j << header.BlockSize);
	    else if (BLOCK_DELETED(part->bam_cache[j]))
		part->EUNInfo[i].Deleted++;
	}
    }
    
    return 0;
    
} /* scan_partition */

/*======================================================================

    Erase_xfer() schedules an asynchronous erase operation for a
    transfer unit.
    
======================================================================*/

static int erase_xfer(ftl_dev_t *dev, partition_t *part,
		      u_short xfernum)
{
    int i, ret;
    struct xfer_info_t *xfer;

    xfer = &part->XferInfo[xfernum];
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(KERN_DEBUG "ftl_cs: erasing xfer unit at 0x%lx\n",
	       xfer->Offset);
#endif
    xfer->state = XFER_ERASING;
    /* Is there a free erase slot? */
    for (;;) {
	for (i = 0; i < MAX_ERASE; i++)
	    if (!ERASE_IN_PROGRESS(dev->eraseq[i].State)) break;
	if (i < MAX_ERASE) break;
#ifdef PCMCIA_DEBUG
	if (pc_debug)
	    printk(KERN_DEBUG "ftl_cs: erase queue is full\n");
#endif
	sleep_on(&dev->erase_pending);
    }

    /* Queue the request */
    dev->eraseq[i].State = ERASE_QUEUED;
    dev->eraseq[i].Handle = part->handle;
    dev->eraseq[i].Offset = xfer->Offset;
    dev->eraseq[i].Size = part->region.BlockSize;
    dev->eraseq[i].Optional = part;
    ret = CardServices(CheckEraseQueue, dev->eraseq_handle);
    if (ret != CS_SUCCESS) {
	cs_error(CheckEraseQueue, ret);
	return -EIO;
    }
    xfer->EraseCount++;
    return ret;
} /* erase_xfer */

/*======================================================================

    Prepare_xfer() takes a freshly erased transfer unit and gives
    it an appropriate header.
    
======================================================================*/

static void save_status(eraseq_entry_t *erase)
{
    partition_t *part;
    struct xfer_info_t *xfer;
    int i;
    
    /* Look up the transfer unit */
    part = (partition_t *)(erase->Optional);
    for (i = 0; i < part->header.NumTransferUnits; i++)
	if (part->XferInfo[i].Offset == erase->Offset) break;
    if (i == part->header.NumTransferUnits) {
	printk(KERN_NOTICE "ftl_cs: internal error: "
	       "erase lookup failed!\n");
	return;
    }
    xfer = &part->XferInfo[i];
    if (erase->State == ERASE_PASSED)
	xfer->state = XFER_ERASED;
    else {
	xfer->state = XFER_FAILED;
	printk(KERN_NOTICE "ftl_cs: erase failed: state = %d\n",
	       erase->State);
    }
}

static void prepare_xfer(partition_t *part, int i)
{
    erase_unit_header_t header;
    mem_op_t req;
    struct xfer_info_t *xfer;
    int nbam, ret;
    u_long ctl;

    xfer = &part->XferInfo[i];
    xfer->state = XFER_FAILED;
    
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(KERN_DEBUG "ftl_cs: preparing xfer unit at 0x%lx\n",
	       xfer->Offset);
#endif

    /* Write the transfer unit header */
    header = part->header;
    header.LogicalEUN = 0xffff;
    header.EraseCount = xfer->EraseCount;
    req.Attributes = MEM_OP_BUFFER_KERNEL;
    req.Count = sizeof(header);
    req.Offset = xfer->Offset;
    ret = CardServices(WriteMemory, part->handle, &req, &header);
    if (ret != CS_SUCCESS) {
	cs_error(WriteMemory, ret);
	return;
    }

    /* Write the BAM stub */
    nbam = (part->BlocksPerUnit * sizeof(u_long) +
	    part->header.BAMOffset + SECTOR_SIZE - 1) / SECTOR_SIZE;
    req.Offset = xfer->Offset + part->header.BAMOffset;
    req.Count = sizeof(u_long);
    ctl = BLOCK_CONTROL;
    for (i = 0; i < nbam; i++, req.Offset += sizeof(u_long)) {
	ret = CardServices(WriteMemory, part->handle, &req, &ctl);
	if (ret != CS_SUCCESS) {
	    cs_error(WriteMemory, ret);
	    return;
	}
    }
    xfer->state = XFER_PREPARED;
    
} /* prepare_xfer */

/*======================================================================

    Copy_erase_unit() takes a full erase block and a transfer unit,
    copies everything to the transfer unit, then swaps the block
    pointers.

    All data blocks are copied to the corresponding blocks in the
    target unit, so the virtual block map does not need to be
    updated.
    
======================================================================*/

static int copy_erase_unit(partition_t *part, u_short srcunit,
			   u_short xferunit)
{
    u_char buf[SECTOR_SIZE];
    struct eun_info_t *eun;
    struct xfer_info_t *xfer;
    mem_op_t req;
    u_long src, dest, free, i;
    u_short unit;
    int ret;

    eun = &part->EUNInfo[srcunit];
    xfer = &part->XferInfo[xferunit];
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(KERN_DEBUG "ftl_cs: copying block 0x%lx to 0x%lx\n",
	       eun->Offset, xfer->Offset);
#endif
	
    req.Attributes = MEM_OP_BUFFER_KERNEL;
    
    /* Read current BAM */
    if (part->bam_index != srcunit) {
	req.Offset = eun->Offset + part->header.BAMOffset;
	req.Count = part->BlocksPerUnit * sizeof(u_long);
	ret = CardServices(ReadMemory, part->handle, &req, part->bam_cache);
	/* mark the cache bad, in case we get an error later */
	part->bam_index = 0xffff;
	if (ret != CS_SUCCESS) goto read_error;
    }
    
    /* Write the LogicalEUN for the transfer unit */
    xfer->state = XFER_UNKNOWN;
    req.Count = sizeof(u_short);
    req.Offset = xfer->Offset + 20; /* Bad! */
    unit = 0x7fff;
    ret = CardServices(WriteMemory, part->handle, &req, &unit);
    if (ret != CS_SUCCESS) goto write_error;
    
    /* Copy all data blocks from source unit to transfer unit */
    src = eun->Offset; dest = xfer->Offset;
    req.Count = SECTOR_SIZE;
    free = 0;
    ret = 0;
    for (i = 0; i < part->BlocksPerUnit; i++) {
	switch (BLOCK_TYPE(part->bam_cache[i])) {
	case BLOCK_CONTROL:
	    /* This gets updated later */
	    break;
	case BLOCK_DATA:
	case BLOCK_REPLACEMENT:
	    req.Offset = src;
	    ret = CardServices(ReadMemory, part->handle, &req, buf);
	    if (ret != CS_SUCCESS) goto read_error;
	    req.Offset = dest;
	    ret = CardServices(WriteMemory, part->handle, &req, buf);
	    if (ret != CS_SUCCESS) goto write_error;
	    break;
	default:
	    /* All other blocks must be free */
	    part->bam_cache[i] = 0xffffffff;
	    free++;
	    break;
	}
	src += SECTOR_SIZE;
	dest += SECTOR_SIZE;
    }

    /* Write the BAM to the transfer unit */
    req.Offset = xfer->Offset + part->header.BAMOffset;
    ret = CardServices(WriteMemory, part->handle, &req, part->bam_cache);
    if (ret != CS_SUCCESS) goto write_error;
    
    /* All clear? Then update the LogicalEUN again */
    req.Offset = xfer->Offset + 20; /* Bad! */
    req.Count = sizeof(u_short);
    ret = CardServices(WriteMemory, part->handle, &req, &srcunit);
    if (ret != CS_SUCCESS) goto write_error;

    /* Update the maps and usage stats*/
    i = xfer->EraseCount;
    xfer->EraseCount = eun->EraseCount;
    eun->EraseCount = i;
    i = xfer->Offset;
    xfer->Offset = eun->Offset;
    eun->Offset = i;
    part->FreeTotal -= eun->Free;
    part->FreeTotal += free;
    eun->Free = free;
    eun->Deleted = 0;
    
    /* Now, the cache should be valid for the new block */
    part->bam_index = srcunit;

    return CS_SUCCESS;
    
read_error:
    cs_error(ReadMemory, ret);
    return ret;
    
write_error:
    cs_error(WriteMemory, ret);
    return ret;
} /* copy_erase_unit */

/*======================================================================

    reclaim_block() picks a full erase unit and a transfer unit and
    then calls copy_erase_unit() to copy one to the other.  Then, it
    schedules an erase on the expired block.

    What's a good way to decide which transfer unit and which erase
    unit to use?  Beats me.  My way is to always pick the transfer
    unit with the fewest erases, and usually pick the data unit with
    the most deleted blocks.  But with a small probability, pick the
    oldest data unit instead.  This means that we generally postpone
    the next reclaimation as long as possible, but shuffle static
    stuff around a bit for wear leveling.
    
======================================================================*/

static int reclaim_block(ftl_dev_t *dev, partition_t *part)
{
    u_short i, eun, xfer;
    u_long best;
    int queued, ret;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(KERN_DEBUG "ftl_cs: reclaiming space...\n");
#endif
	
    /* Pick the least erased transfer unit */
    best = 0xffffffff; xfer = 0xffff;
    do {
	queued = 0;
	for (i = 0; i < part->header.NumTransferUnits; i++) {
	    if (part->XferInfo[i].state == XFER_UNKNOWN)
		erase_xfer(dev, part, i);
	    if (part->XferInfo[i].state == XFER_ERASING)
		queued = 1;
	    else if (part->XferInfo[i].state == XFER_ERASED)
		prepare_xfer(part, i);
	    if ((part->XferInfo[i].state == XFER_PREPARED) &&
		(part->XferInfo[i].EraseCount <= best)) {
		    best = part->XferInfo[i].EraseCount;
		    xfer = i;
		}
	}
	if (xfer == 0xffff) {
	    if (queued) {
#ifdef PCMCIA_DEBUG
		if (pc_debug)
		    printk(KERN_DEBUG "ftl_cs: waiting for transfer "
			   "unit to be prepared...\n");
#endif
		sleep_on(&dev->erase_pending);
	    }
	    else {
		static int ne = 0;
		if (++ne < 5)
		    printk(KERN_NOTICE "ftl_cs: reclaim failed: no "
			   "suitable transfer units!\n");
		return CS_GENERAL_FAILURE;
	    }
	}
    } while (xfer == 0xffff);

    eun = 0;
    if ((jiffies % shuffle_freq) == 0) {
#ifdef PCMCIA_DEBUG
	if (pc_debug)
	    printk(KERN_DEBUG "ftl_cs: recycling freshest block...\n");
#endif
	best = 0xffffffff;
	for (i = 0; i < part->DataUnits; i++)
	    if (part->EUNInfo[i].EraseCount <= best) {
		best = part->EUNInfo[i].EraseCount;
		eun = i;
	    }
    }
    else {
	best = 0;
	for (i = 0; i < part->DataUnits; i++)
	    if (part->EUNInfo[i].Deleted >= best) {
		best = part->EUNInfo[i].Deleted;
		eun = i;
	    }
	if (best == 0) {
	    static int ne = 0;
	    if (++ne < 5)
		printk(KERN_NOTICE "ftl_cs: reclaim failed: "
		       "no free blocks!\n");
	    return CS_GENERAL_FAILURE;
	}
    }
    ret = copy_erase_unit(part, eun, xfer);
    if (ret == CS_SUCCESS)
	erase_xfer(dev, part, xfer);
    else
	printk(KERN_NOTICE "ftl_cs: copy_erase_unit failed!\n");
    return ret;
} /* reclaim_block */

/*======================================================================

    Find_free() searches for a free block.  If necessary, it updates
    the BAM cache for the erase unit containing the free block.  It
    returns the block index -- the erase unit is just the currently
    cached unit.  If there are no free blocks, it returns 0 -- this
    is never a valid data block because it contains the header.
    
======================================================================*/

#ifdef PSYCHO_DEBUG
static void dump_lists(partition_t *part)
{
    int i;
    printk(KERN_DEBUG "ftl_cs: Free total = %d\n", part->FreeTotal);
    for (i = 0; i < part->DataUnits; i++)
	printk(KERN_DEBUG "ftl_cs:   unit %d: %d phys, %d free, "
	       "%d deleted\n", i,
	       part->EUNInfo[i].Offset >> part->header.EraseUnitSize,
	       part->EUNInfo[i].Free, part->EUNInfo[i].Deleted);
}
#endif

static u_long find_free(partition_t *part)
{
    u_short stop, eun;
    u_long blk;
    mem_op_t req;
    int ret;
    
    /* Find an erase unit with some free space */
    stop = (part->bam_index == 0xffff) ? 0 : part->bam_index;
    eun = stop;
    do {
	if (part->EUNInfo[eun].Free != 0) break;
	/* Wrap around at end of table */
	if (++eun == part->DataUnits) eun = 0;
    } while (eun != stop);

    if (part->EUNInfo[eun].Free == 0)
	return 0;
    
    /* Is this unit's BAM cached? */
    if (eun != part->bam_index) {
	/* Invalidate cache */
	part->bam_index = 0xffff;
	req.Attributes = MEM_OP_BUFFER_KERNEL;
	req.Count = part->BlocksPerUnit * sizeof(u_long);
	req.Offset = part->EUNInfo[eun].Offset + part->header.BAMOffset;
	ret = CardServices(ReadMemory, part->handle, &req,
			   part->bam_cache);
	if (ret != CS_SUCCESS) {
	    cs_error(ReadMemory, ret);
	    return 0;
	}
	part->bam_index = eun;
    }

    /* Find a free block */
    for (blk = 0; blk < part->BlocksPerUnit; blk++)
	if (BLOCK_FREE(part->bam_cache[blk])) break;
    if (blk == part->BlocksPerUnit) {
#ifdef PSYCHO_DEBUG
	static int ne = 0;
	if (++ne == 1)
	    dump_lists(part);
#endif
	printk(KERN_NOTICE "ftl_cs: bad free list!\n");
	return 0;
    }
#ifdef PCMCIA_DEBUG
    if (pc_debug > 1)
	printk(KERN_DEBUG "ftl_cs: found free block at %ld in %d\n",
	       blk, eun);
#endif
    return blk;
    
} /* find_free */

/*======================================================================

    This gets a memory handle for the region corresponding to the
    minor device number.
    
======================================================================*/

static int ftl_open(struct inode *inode, struct file *file)
{
    int minor = MINOR(inode->i_rdev);
    dev_link_t *link;
    ftl_dev_t *dev;
    partition_t *partition;
    open_mem_t open;
    int ret;
    
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(KERN_DEBUG "ftl_cs: ftl_open(%d)\n", minor);
#endif

    link = dev_table[DEVICE_NR(minor)];
    if (!DEV_OK(link))
	return -ENODEV;
    
    dev = (ftl_dev_t *)link->priv;
    partition = &dev->minor[REGION_NR(minor)];
    if (partition->region.RegionSize == 0)
	return -ENODEV;
    if (partition->handle == NULL) {
	partition->handle = (memory_handle_t)link->handle;
	open.Attributes = partition->region.Attributes;
	open.Offset = partition->region.CardOffset;
	ret = CardServices(OpenMemory, &partition->handle, &open);
	if (ret != CS_SUCCESS)
	    return -ENODEV;
	if (scan_partition(partition) == 0) {
	    partition->state = FTL_FORMATTED;
	    printk(KERN_INFO "ftl_cs: opening %ld kb FTL partition\n",
		   partition->header.FormattedSize >> 10);
	}
	else {
	    CardServices(CloseMemory, partition->handle);
	    printk(KERN_NOTICE "ftl_cs: FTL partition is invalid.\n");
	    return -ENODEV;
	}
    }
    
    partition->open++;
    link->open++;
    MOD_INC_USE_COUNT;
    return 0;
} /* ftl_open */

/*====================================================================*/

static void ftl_close(struct inode *inode, struct file *file)
{
    dev_link_t *link;
    int minor = MINOR(inode->i_rdev);
    ftl_dev_t *dev;
    partition_t *part;
    int i;
    
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(KERN_DEBUG "ftl_cs: ftl_close(%d)\n", minor);
#endif

    /* Flush all writes */
    sync_dev(inode->i_rdev);

    link = dev_table[DEVICE_NR(minor)];
    dev = (ftl_dev_t *)link->priv;
    part = &dev->minor[REGION_NR(minor)];
    
    /* Wait for any pending erase operations to complete */
    for (i = 0; i < part->header.NumTransferUnits; i++) {
	if (part->XferInfo[i].state == XFER_ERASING)
	    sleep_on(&dev->erase_pending);
	if (part->XferInfo[i].state == XFER_ERASED)
	    prepare_xfer(part, i);
    }
    
    link->open--;
    part->open--;
    if (part->open == 0) {
	CardServices(CloseMemory, part->handle);
	part->handle = NULL;
	if (part->VirtualBlockMap) {
	    kfree(part->VirtualBlockMap);
	    part->VirtualBlockMap = NULL;
	}
	if (part->VirtualPageMap) {
	    kfree(part->VirtualPageMap);
	    part->VirtualPageMap = NULL;
	}
	if (part->EUNInfo) {
	    kfree(part->EUNInfo);
	    part->EUNInfo = NULL;
	}
	if (part->XferInfo) {
	    kfree(part->XferInfo);
	    part->XferInfo = NULL;
	}
	if (part->bam_cache) {
	    kfree(part->bam_cache);
	    part->bam_cache = NULL;
	}
    }
    
    MOD_DEC_USE_COUNT;

} /* ftl_close */

/*======================================================================

    Read a series of sectors from an FTL partition.
    
======================================================================*/

static int ftl_read(partition_t *part, caddr_t buffer,
		    u_long sector, u_long nblocks)
{
    mem_op_t req;
    u_long log_addr, bsize, i;
    int ret;
    
#ifdef PCMCIA_DEBUG
    if (pc_debug > 1)
	printk(KERN_DEBUG "ftl_cs: ftl_read(0x%p, 0x%lx, %ld)\n",
	       part->handle, sector, nblocks);
#endif
    if (!(part->state & FTL_FORMATTED)) {
	printk(KERN_NOTICE "ftl_cs: bad partition\n");
	return -EIO;
    }
    bsize = part->region.BlockSize;
    req.Attributes = MEM_OP_BUFFER_KERNEL;
    req.Count = SECTOR_SIZE;
    for (i = 0; i < nblocks; i++) {
	if (((sector+i) * SECTOR_SIZE) >= part->header.FormattedSize) {
	    printk(KERN_NOTICE "ftl_cs: bad read offset\n");
	    return -EIO;
	}
	log_addr = part->VirtualBlockMap[sector+i];
	if (log_addr == 0xffffffff)
	    memset(buffer, 0, SECTOR_SIZE);
	else {
	    req.Offset = (part->EUNInfo[log_addr / bsize].Offset
			  + (log_addr % bsize));
	    ret = CardServices(ReadMemory, part->handle, &req, buffer);
	    if (ret != CS_SUCCESS) {
		cs_error(ReadMemory, ret);
		return -EIO;
	    }
	}
	buffer += SECTOR_SIZE;
    }
    return 0;
} /* ftl_read */

/*======================================================================

    Write a series of sectors to an FTL partition
    
======================================================================*/

static int set_bam_entry(partition_t *part, u_long log_addr,
			 u_long virt_addr)
{
    mem_op_t req;
    u_long bsize, blk;
#ifdef PSYCHO_DEBUG
    u_long old_addr;
#endif
    u_short eun;
    int ret;
    
#ifdef PCMCIA_DEBUG
    if (pc_debug > 1)
	printk(KERN_DEBUG "ftl_cs: set_bam_entry(0x%p, 0x%lx, 0x%lx)\n",
	       part->handle, log_addr, virt_addr);
#endif
    bsize = part->region.BlockSize;
    eun = log_addr / bsize;
    blk = (log_addr % bsize) / SECTOR_SIZE;
    req.Attributes = MEM_OP_BUFFER_KERNEL;
    req.Count = sizeof(u_long);
    req.Offset = (part->EUNInfo[eun].Offset + blk * sizeof(u_long) +
		  part->header.BAMOffset);
    
#ifdef PSYCHO_DEBUG
    CardServices(ReadMemory, part->handle, &req, &old_addr);
    if (((virt_addr == 0xfffffffe) && !BLOCK_FREE(old_addr)) ||
	((virt_addr == 0) && (BLOCK_TYPE(old_addr) != BLOCK_DATA)) ||
	(!BLOCK_DELETED(virt_addr) && (old_addr != 0xfffffffe))) {
	static int ne = 0;
	if (++ne < 5) {
	    printk(KERN_NOTICE "ftl_cs: set_bam_entry() inconsistency!\n");
	    printk(KERN_NOTICE "ftl_cs:   log_addr = 0x%lx, old = 0x"
		   "%lx, new = 0x%lx\n", log_addr, old_addr, virt_addr);
	}
	return CS_GENERAL_FAILURE;
    }
#endif
    if (part->bam_index == eun) {
#ifdef PSYCHO_DEBUG
	if (part->bam_cache[blk] != old_addr) {
	    static int ne = 0;
	    if (++ne < 5) {
		printk(KERN_NOTICE "ftl_cs: set_bam_entry() "
		       "inconsistency!\n");
		printk(KERN_NOTICE "ftl_cs:   log_addr = 0x%lx, cache"
		       " = 0x%lx, card = 0x%lx\n",
		       part->bam_cache[blk], old_addr);
	    }
	    return CS_GENERAL_FAILURE;
	}
#endif
	part->bam_cache[blk] = virt_addr;
    }

    ret = CardServices(WriteMemory, part->handle, &req, &virt_addr);
#ifdef PSYCHO_DEBUG
    if (ret != CS_SUCCESS) {
	printk(KERN_NOTICE "ftl_cs: set_bam_entry() failed!\n");
	printk(KERN_NOTICE "ftl_cs:   log_addr = 0x%lx, old = 0x%lx,"
	       " new = 0x%lx\n", log_addr, old_addr, virt_addr);
	cs_error(WriteMemory, ret);
    }
#endif
    return ret;
} /* set_bam_entry */

static int ftl_write(ftl_dev_t *dev, partition_t *part, caddr_t buffer,
		     u_long sector, u_long nblocks)
{
    mem_op_t req;
    u_long bsize, log_addr, virt_addr, old_addr, blk, i;
    int ret;

#ifdef PCMCIA_DEBUG
    if (pc_debug > 1)
	printk(KERN_DEBUG "ftl_cs: ftl_write(0x%p, %ld, %ld)\n",
	       part->handle, sector, nblocks);
#endif
    if (!(part->state & FTL_FORMATTED)) {
	printk(KERN_NOTICE "ftl_cs: bad partition\n");
	return -EIO;
    }
    /* See if we need to reclaim space, before we start */
    while (part->FreeTotal < nblocks) {
	ret = reclaim_block(dev, part);
	if (ret != CS_SUCCESS)
	    return ret;
    }
    
    bsize = part->region.BlockSize;
    req.Attributes = MEM_OP_BUFFER_KERNEL;
    req.Count = SECTOR_SIZE;
    virt_addr = sector * SECTOR_SIZE | BLOCK_DATA;
    for (i = 0; i < nblocks; i++) {
	if (virt_addr >= part->header.FormattedSize) {
	    printk(KERN_NOTICE "ftl_cs: bad write offset\n");
	    return -EIO;
	}

	/* Grab a free block */
	blk = find_free(part);
	if (blk == 0) {
	    static int ne = 0;
	    if (++ne < 5)
		printk(KERN_NOTICE "ftl_cs: internal error: "
		       "no free blocks!\n");
	    return -ENOSPC;
	}

	/* Tag the BAM entry, and write the new block */
	log_addr = part->bam_index * bsize + blk * SECTOR_SIZE;
	part->EUNInfo[part->bam_index].Free--;
	part->FreeTotal--;
	if (set_bam_entry(part, log_addr, 0xfffffffe))
	    return -EIO;
	part->EUNInfo[part->bam_index].Deleted++;
	req.Offset = (part->EUNInfo[part->bam_index].Offset +
		      blk * SECTOR_SIZE);
	ret = CardServices(WriteMemory, part->handle, &req, buffer);
	if (ret != CS_SUCCESS) {
	    printk(KERN_NOTICE "ftl_cs: block write failed!\n");
	    printk(KERN_NOTICE "ftl_cs:   log_addr = 0x%lx, virt_addr"
		   " = 0x%lx, Offset = 0x%lx\n", log_addr, virt_addr,
		   req.Offset);
	    cs_error(WriteMemory, ret);
	    return -EIO;
	}
	
	/* Only delete the old entry when the new entry is ready */
	old_addr = part->VirtualBlockMap[sector+i];
	if (old_addr != 0xffffffff) {
	    part->VirtualBlockMap[sector+i] = 0xffffffff;
	    part->EUNInfo[old_addr/bsize].Deleted++;
	    if (set_bam_entry(part, old_addr, 0) != CS_SUCCESS)
		return -EIO;
	}

	/* Finally, set up the new pointers */
	if (set_bam_entry(part, log_addr, virt_addr))
	    return -EIO;
	part->VirtualBlockMap[sector+i] = log_addr;
	part->EUNInfo[part->bam_index].Deleted--;
	
	buffer += SECTOR_SIZE;
	virt_addr += SECTOR_SIZE;
    }
    return 0;
} /* ftl_write */

/*======================================================================

    IOCTL calls for getting device parameters.

======================================================================*/

static int ftl_ioctl(struct inode *inode, struct file *file,
		     u_int cmd, u_long arg)
{
    int minor = MINOR(inode->i_rdev);
    dev_link_t *link;
    ftl_dev_t *dev;
    partition_t *partition;
    int ret;

    link = dev_table[DEVICE_NR(minor)];
    
    if (!DEV_OK(link))
	return -ENODEV;
    dev = (ftl_dev_t *)link->priv;
    partition = &dev->minor[REGION_NR(minor)];
    switch (cmd) {
    case BLKGETSIZE:
	if (!(partition->state & FTL_FORMATTED))
	    ret = -EIO;
	else {
	    ret = verify_area(VERIFY_WRITE, (long *)arg, sizeof(long));
	    if (ret) break;
	    put_fs_long(partition->header.FormattedSize/SECTOR_SIZE,
			(long *)arg);
	}
	break;
    default:
	ret = -EINVAL;
    }
    return ret;
} /* ftl_ioctl */

/*======================================================================

    Handler for block device requests

======================================================================*/

static void do_ftl_request(void)
{
    int ret, minor;
    dev_link_t *link;
    ftl_dev_t *dev;
    partition_t *part;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(KERN_DEBUG "ftl_cs: starting do_ftl_request()\n");
#endif
    
    do {
	sti();
	INIT_REQUEST;

	minor = MINOR(DEVICE(CURRENT));
	
	link = dev_table[DEVICE_NR(minor)];
	dev = (ftl_dev_t *)link->priv;
	part = &dev->minor[REGION_NR(minor)];
	ret = 0;
	
	switch (CURRENT->cmd) {
	    
	case READ:
	    ret = ftl_read(part, CURRENT->buffer, CURRENT->sector,
			   CURRENT->current_nr_sectors);
	    break;
	    
	case WRITE:
	    ret = ftl_write(dev, part, CURRENT->buffer, CURRENT->sector,
			    CURRENT->current_nr_sectors);
	    break;
	    
	default:
	    panic("ftl_cs: unknown block command!\n");
	    
	}
	if (ret == 0)
	    end_request(1);
	else {
	    static int ne = 0;
	    if (++ne < 5)
		end_request(0);
	    else
		end_request(1);
	}
    } while (1);
} /* do_ftl_request */

/*====================================================================*/

int init_ftl_cs(void)
{
    servinfo_t serv;
    int i;
    
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(KERN_INFO "%s\n", version);
#endif
    
    CardServices(GetCardServicesInfo, &serv);
    if (serv.Revision != CS_RELEASE_CODE) {
	printk(KERN_NOTICE "ftl_cs: Card Services release "
	       "does not match!\n");
	return -1;
    }
    
    register_pcmcia_driver(&dev_info, &ftl_attach, &ftl_detach);

    major_dev = register_blkdev(0, "ftl", &ftl_blk_fops);
    if (major_dev == 0)
	printk(KERN_NOTICE "ftl_cs: unable to grab major "
	       "device number!\n");

    blk_dev[major_dev].request_fn = DEVICE_REQUEST;
    for (i = 0; i < MINOR_NR(MAX_DEV, 0); i++)
	ftl_blocksizes[i] = 1024;
    blksize_size[major_dev] = ftl_blocksizes;
    
    return 0;
}

#ifdef MODULE
void cleanup_module(void)
{
    int i;
    dev_link_t *link;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(KERN_DEBUG "ftl_cs: unloading\n");
#endif
    unregister_pcmcia_driver(&dev_info);
    if (major_dev != 0)
	unregister_blkdev(major_dev, "ftl");
    for (i = 0; i < MAX_DEV; i++) {
	link = dev_table[i];
	if (link) {
	    if (link->state & DEV_CONFIG)
		ftl_release((u_long)link);
	    ftl_detach(link);
	}
    }
}
#endif /* MODULE */

