/*
 *	This code is derived from Anthony D. Joseph's code and all the changes here
 *	are also under the original copyright below.
 *
 *	This code supports version 2.00 of WaveLAN/PCMCIA cards (2.4GHz), and
 *	can work on Linux 1.3.55 with support of David Hinds' PCMCIA Card Services
 *	version 2.8.8. 
 *
 *	Joe Finney (joe@comp.lancs.ac.uk) at Lancaster University in UK added critical 
 *	code in the routine to initialize the Modem Management Controller.
 *
 *	I am sorry that I have changed the original feature of Anthony's code. I didn't
 *	mean to show the difference but to understand the code better (basically I follow
 *	David's skeleton.c).
 *
 *	Hopefully, this code can be within all the subsequent versions of David's 
 *	PCMCIA Card Services. Thanks to David Hinds.
 *
 *	Thanks to Alan Cox and Bruce Janson for their advice.
 *
 *				-- Yunzhou Li (scip4166@nus.sg)
 */

/*
 * AT&T GIS (formerly NCR) WaveLAN PCMCIA card: An Ethernet-like radio
 *   transceiver controlled by an Intel 82593 coprocessor.
 *
 *   A non-shared memory PCMCIA ethernet driver for linux
 *
 * ISA version modified to support PCMCIA by Anthony Joseph (adj@lcs.mit.edu)
 *
 * /usr/src/pc/modules/RCS/wavelan_cs.c,v 1.6 1996/08/29 04:41:46 root Exp
 *
 ****************************************************************************
 *   Copyright 1995
 *   Anthony D. Joseph
 *   Massachusetts Institute of Technology
 *
 *   Permission to use, copy, modify, and distribute this program
 *   for any purpose and without fee is hereby granted, provided
 *   that this copyright and permission notice appear on all copies
 *   and supporting documentation, the name of M.I.T. not be used
 *   in advertising or publicity pertaining to distribution of the
 *   program without specific prior permission, and notice be given
 *   in supporting documentation that copying and distribution is
 *   by permission of M.I.T.  M.I.T. makes no representations about
 *   the suitability of this software for any purpose.  It is pro-
 *   vided "as is" without express or implied warranty.         
 ****************************************************************************
 *
 * Pre-RCS revisions:
 *     0.10  29-Sep-94 Initial version
 *     0.20  15-Nov-94 Fixed frame reception problem(incorrect min frame size)
 *     0.25  15-Nov-94 Turned off src addr insertion and fixed skbuf generation
 *     0.30  17-Nov-94 Cleaned up code and comments
 *     0.40  19-Nov-94 Fixed sk_buff kfree problem during transmission
 *
 * RCS revisions:
 * wavelan_cs.c,v
 * Revision 1.6  1996/08/29 04:41:46  root
 * *** empty log message ***
 *
 * Revision 1.5  1996/08/21 15:01:59  root
 * *** empty log message ***
 *
 * Revision 1.4  1996/08/15 03:40:01  root
 * *** empty log message ***
 *
 * Revision 1.3  1996/08/06 03:14:02  root
 * *** empty log message ***
 *
 * Revision 1.2  1996/07/24 16:33:18  root
 * Fixed to build against 1.2.X kernels.
 *
 * Revision 1.1  1996/07/17 15:23:51  root
 * Initial revision
 *
 * Revision 1.0  1995/04/24  05:42:35  adj
 * Alpha release
 *
 * Revision 0.70  1995/04/16  00:24:45  adj
 * Added support for wavelan major device
 * Additional code cleanup
 *
 * Revision 0.60  1995/04/07  23:24:25  adj
 * Major code rewrite and cleanup for 1.2.3 kernel and 2.5.3 pcmcia support
 *
 * Revision 0.50  1995/04/05  15:19:16  adj
 * Initial alpha release
 *
 *
 * Debugging support using PCMCIA_DEBUG:
 *     undefined - No extra output, no extra overhead
 *     0 - No extra output
 *     1 - Show important procedure entry/exit
 *
 * Debugging support using WAVELAN_DEBUG:
 *     undefined - No extra output, no extra overhead
 *     0 - No extra output
 *     1 - Show less important procedure entry/exit
 *     2 - Show configuration/initialization process
 *     3 - Show detailed configuration/initialization information
 *     4 - Track interrupts and packet headers
 *     5 - Dump incoming/outgoing packets
 *
 * Credits:
 *     Special thanks to Jan Hoogendoorn of AT&T GIS Utrecht for
 *       providing extremely useful information about WaveLAN PCMCIA hardware
 *
 *     This driver is based upon several other drivers, in particular:
 *       David Hinds' Linux driver for the PCMCIA 3c589 ethernet adapter
 *       Bruce Janson's Linux driver for the AT-bus WaveLAN adapter
 *	 Anders Klemets' PCMCIA WaveLAN adapter driver
 *       Robert Morris' BSDI driver for the PCMCIA WaveLAN adapter
 *
 * Additional Credits:
 *
 *     This software was developed under Linux 1.2.3
 *     (Slackware 2.0 distribution).
 *
 *     It is based on other device drivers and information either written
 *     or supplied by:
 *	 James Ashton (jaa101@syseng.anu.edu.au),
 *	 Ajay Bakre (bakre@paul.rutgers.edu),
 *	 Donald Becker (becker@super.org),
 *	 Loeke Brederveld (Loeke.Brederveld@Utrecht.NCR.com),
 *	 Allan Creighton (allanc@cs.su.oz.au),
 *	 Matthew Geier (matthew@cs.su.oz.au),
 *	 Remo di Giovanni (remo@cs.su.oz.au),
 *	 Mark Hagan (mhagan@wtcpost.daytonoh.NCR.COM),
 *	 Jan Hoogendoorn (JHOOGEND@wcnd.utrecht.attgis.com),
 *       Bruce Janson (bruce@cs.su.oz.au)
 *	 Anders Klemets (klemets@paul.rutgers.edu),
 *	 Marc Meertens (Marc.Meertens@Utrecht.NCR.com),
 *	 Robert Morris (rtm@das.harvard.edu),
 *	 Ian Parkin (ian@cs.su.oz.au),
 *	 John Rosenberg (johnr@cs.su.oz.au),
 *	 George Rossi (george@phm.gov.au),
 *	 Arthur Scott (arthur@cs.su.oz.au),
 *	 Peter Storey,
 *	 Girish Welling (welling@paul.rutgers.edu),
 *
 */

#include "config.h"
#include "k_compat.h"

#ifdef MODULE
#define init_wavelan_cs init_module
#endif

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/ptrace.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/interrupt.h>
#include <linux/in.h>
#include <linux/delay.h>
#include <asm/io.h>
#include <asm/system.h>
#include <asm/bitops.h>

#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/if_arp.h>
#include <linux/ioport.h>
#include <linux/fcntl.h>

#include "cs_types.h"
#include "cs.h"
#include "cistpl.h"
#include "cisreg.h"
#include "ds.h"
#include "version.h"

/*
 * Provoke a run-time error if the compiler has padded device register
 * structures. Does NOT handle reordering of components.
 */
#define	STRUCT_CHECK	1
#include "wavelan.h"

#define SNRCOUNT 20
#define MAX_MULTI_ADDR 64
#define IGNORE_NORMAL_XMIT_ERRS

#define WAVELAN_DEV(link)  ((device *)(link)->priv)
#define WAVELAN_LP(link)   ((net_local *)((device *)(link)->priv)->priv)

#ifdef PCMCIA_DEBUG
static int pc_debug = PCMCIA_DEBUG;
#else
static int pc_debug = 0;
#endif
static const char *version = "wavelan_cs.c,v 1.6 1996/08/29 04:41:46 root Exp\n";

/*====================================================================*/

/* Parameters that can be set with 'insmod' */

static int xmt_watch = 1;	/* Reporting of transmission errors */
static int gathersnr = 1;	/* Controls gathering of per-packet snr data */
static int mem_speed = 0;	/* Shared memory speed, in ns */
#ifdef RECORD_SNR
static int recordsnr = 0;
#endif /* RECORD_SNR */

/* Bit map of interrupts to choose from */
/* This means pick from 15, 14, 12, 11, 10, 9, 7, 5, 4 and 3 */
static u_long irq_mask = 0xdeb8;

/*====================================================================*/

#ifdef WAVELAN_DEBUG
static u_int wavelan_debug = WAVELAN_DEBUG;
#endif	/* WAVELAN_DEBUG */

static dev_info_t dev_info = "wavelan_cs";
static dev_link_t *dev_list = NULL;
static int major_dev = -1;

typedef struct device		device;
typedef struct enet_statistics	en_stats;
typedef struct net_local	net_local;
typedef struct timer_list	timer_list;

struct net_local
{
  dev_node_t 		node;
  en_stats	        stats;          /* Ethernet interface statistics */
  u_char	        nwid[2];        /* WaveLAN network ID */
  int   		stop;		/* Current i82593 Stop Hit Register */
  int   		rfp;		/* Last DMA machine receive pointer */
  int                   overrunning;	/* Receiver overrun flag */
  int			nresets;	/* Number of hw resets */
  int			correct_nwid;	/* Num rcv'd pkts w/ correct nwid */
  int			wrong_nwid;	/* Num rcv'd pkts w/ incorrect nwid */
  unsigned int		full_promiscuous;
  u_char		promiscuous;	/* Promiscuous mode */
  u_char		multicast;	/* Multicast mode */
  u_char 		cmd_wait;	/* Pending i82593 command flag */
  u_char         	status;		/* Current i82593 status */
  int			snrrcvctr;	/* Rcv packet counter for SNR use */
  int			siglvl[SNRCOUNT];
  int			sillvl[SNRCOUNT];
  int			sigqual[SNRCOUNT];
  u_int 		snr[5];
#ifdef	IF_CNTRS
  struct wl_cntrs       cntrs;		/* WaveLAN-specific counters */
#endif	IF_CNTRS
  psa_t			psa;		/* Copy of card's PSA */
};

/*====================================================================*/

static void wavelan_config(dev_link_t *link);
static void wavelan_release(u_long arg);
static int wavelan_event(event_t event, int priority,
			 event_callback_args_t *args);

static dev_link_t *wavelan_attach(void);
static void wavelan_detach(dev_link_t *link);

static int wavelan_open(struct inode *inode, struct file *file);
static int wavelan_read(struct inode *inode, struct file *file, 
			char *buf, int count);

static void wv_interrupt(device *dev);
static void wavelan_interrupt IRQ(int irq, void *dev_id, struct pt_regs *regs);

static struct file_operations wv_fops = {
  NULL,		/* lseek */
  wavelan_read,	/* read */
  NULL,		/* write */
  NULL,		/* readdir */
  NULL,		/* select */
  NULL,		/* ioctl */
  NULL,		/* mmap */
  wavelan_open,	/* open */
  NULL,		/* release */
  NULL		/* fsync */
};

/*====================================================================*/

static void cs_error(int func, int ret)
{
    CardServices(ReportError, dev_info, (void *)func, (void *)ret);
}

/*======================================================================

  Read from card's Host Adaptor Status Register.
    
======================================================================*/

static inline u_char hasr_read(u_short base)
{
  return(inb(HASR(base)));
} /* hasr_read */

/*======================================================================

  Write to card's Host Adapter Command Register.
    
======================================================================*/

static inline void hacr_write(u_short base, u_char hacr)
{
  outb(hacr, HACR(base));
} /* hacr_write */

/*======================================================================

  Write to card's Host Adapter Command Register. Include a delay for
  those times when it is needed.
    
======================================================================*/

static inline void hacr_write_slow(u_short base, u_char hacr)
{
  hacr_write(base, hacr);
  /* delay might only be needed sometimes */
  udelay(1000L);
} /* hacr_write_slow */

/*======================================================================

  Routine to write bytes to the Modem Management Controller.
    
======================================================================*/

static void mmc_write(u_short base, u_char o, u_char *b, int n)
{
  while (n-- > 0 ) {
    while (inb(HASR(base)) & HASR_MMI_BUSY) ;	/* Wait for MMC to go idle */
    outb((u_char)((o << 1) | MMR_MMI_WR), MMR(base));
    outb((u_char)(*b), MMD(base));
    b++;
    o++;
  }
} /* mmc_write */

/*======================================================================

  Routine to read bytes from the Modem Management Controller.
  The implementation is complicated by a lack of address lines,
  which prevents decoding of the low-order bit.
    
======================================================================*/

static void mmc_read(u_short base, u_char o, u_char *b, int n)
{
  while (n-- > 0) {
    while (inb(HASR(base)) & HASR_MMI_BUSY) ;	/* Wait for MMC to go idle */
    outb(o << 1, MMR(base));			/* Set the read address */
    o++;

    outb(0, MMD(base));				/* Required dummy write */
    while (inb(HASR(base)) & HASR_MMI_BUSY) ;	/* Wait for MMC to go idle */
    *b++ = (u_char)(inb(MMD(base)));		/* Now do the actual read */
  }
} /* mmc_read */

/*======================================================================

  Read the Paramter Storage Area from the WaveLAN card's memory
    
======================================================================*/

static void psa_read(device *dev, psa_t *psa)
{
    u_char *ptr = ((u_char *)dev->mem_start) + PSA_ADDR;
    u_char *b = (u_char *) psa;
    int n = sizeof(psa_t);

    while (n-- > 0) {
    	*b++ = *ptr;
    	/* Due to a lack of address decode pins, the WaveLAN PCMCIA card
       	   only supports reading even memory addresses. That means the
           increment here MUST be two. */
    	ptr += 2;
    }
} /* psa_read */

/*======================================================================

  Wrapper for disabling interrupts.
    
======================================================================*/

static inline unsigned long wv_splhi(void)
{
  unsigned long flags;

  save_flags(flags);
  cli();
  return(flags);
} /* wv_splhi */

/*======================================================================

  Wrapper for re-enabling interrupts.
    
======================================================================*/

static inline void wv_splx(unsigned long flags)
{
  restore_flags(flags);
} /* wv_splx */

/*======================================================================

  Get the current ethernet statistics. This may be called with the
  card open or closed.
    
======================================================================*/

static en_stats	*wv_get_stats(device *dev)
{
  net_local *lp = (net_local *)dev->priv;
  return(&lp->stats);
} /* wv_get_stats */

/*======================================================================

  Calculates the current Signal-to-Noise Ration (SNR). Currently,
  ignores sigqual.

======================================================================*/

static void wv_getsnr(device *dev, u_char siglvl, u_char sillvl,
			   u_char sigqual)
{
  int s, snrindex = 0;
  net_local *lp = (net_local *)dev->priv;

  if ((s = (siglvl & 0x3f)) > 0) {
    s -= (sillvl & 0x3f) - 0x2d;
    if (s >= 0x38)     snrindex = 4;
    else if(s >= 0x2a) snrindex = 3;
    else if(s >= 0x1c) snrindex = 2;
    else if(s >= 0x0e) snrindex = 1;
    else if(s >= 0)    snrindex = 0;
    else printk("wavelan_cs: bogus snr level\n");
    lp->snr[snrindex]++;
  }
#if RECORD_SNR
  if (recordsnr) {
    lp->siglvl[lp->snrrcvctr] = siglvl;
    lp->sillvl[lp->snrrcvctr] = sillvl;
    lp->sigqual[lp->snrrcvctr++] = sigqual;
    if (lp->snrrcvctr > SNRCOUNT) {
      char buf[256];
      int i, sgl, sil, sgq;
      lp->snrrcvctr = 0;
      for (i = sgl = sil = sgq = 0; i < SNRCOUNT; i++) {
	sgl += lp->siglvl[i];
	sil += lp->sillvl[i];
	sgq += lp->sigqual[i];
      }
      printk("sgl %d sil %d sgq %d snr0 %d snr1 %d snr2 %d snr3 %d snr4 %d\n",
	     sgl/SNRCOUNT, sil/SNRCOUNT, sgq/SNRCOUNT, lp->snr[0],
	     lp->snr[1], lp->snr[2], lp->snr[3], lp->snr[4]);
    }
  }
#endif /* RECORD_SNR */
} /* wv_getsnr */
  
#if STRUCT_CHECK == 1
/*======================================================================

  Sanity routine to verify the sizes of the various WaveLAN interface
  structures.
    
======================================================================*/

static char *wv_structuct_check(void)
{
#define	SC(t,s,n)	if (sizeof(t) != s) return(n);
  SC(psa_t, PSA_SIZE, "psa_t");
  SC(mmw_t, MMW_SIZE, "mmw_t");
  SC(mmr_t, MMR_SIZE, "mmr_t");
#undef	SC
  return((char *)0);
} /* wv_structuct_check */
#endif	/* STRUCT_CHECK == 1 */

/*======================================================================

  Provide useful WaveLAN-specific device data. Accessed through the
  the wavelan major device.

======================================================================*/

static char *sprintf_stats(char *buffer, device *dev)
{
  u_char v;
  mmr_t	m;
  u_short base = dev->base_addr;
  u_char correct_nwid_h;
  u_char correct_nwid_l;
  u_char wrong_nwid_h;
  u_char wrong_nwid_l;
  net_local *lp = (net_local *)dev->priv;

  v = (u_char)1;
  mmc_write(dev->base_addr, mmwoff(0, mmw_freeze), &v, sizeof(v));

  mmc_read(base, mmroff(0, mmr_correct_nwid_h), &correct_nwid_h,
	   sizeof(correct_nwid_h));
  mmc_read(base, mmroff(0, mmr_correct_nwid_l), &correct_nwid_l,
	   sizeof(correct_nwid_l));
  mmc_read(base, mmroff(0, mmr_wrong_nwid_h), &wrong_nwid_h,
	   sizeof(wrong_nwid_h));
  mmc_read(base, mmroff(0, mmr_wrong_nwid_l), &wrong_nwid_l,
	   sizeof(wrong_nwid_l));
  mmc_read(base, 0, (u_char *)&m, sizeof(m));

  v = (u_char)0;
  mmc_write(dev->base_addr, mmwoff(0, mmw_freeze), &v, sizeof(v));

#ifdef 0
  lp->correct_nwid += (m.mmr_correct_nwid_h << 8) | m.mmr_correct_nwid_l;
  lp->wrong_nwid += (m.mmr_wrong_nwid_h << 8) | m.mmr_wrong_nwid_l;
#endif

  return(buffer +
    sprintf(buffer,
	    "%6s:%s%s%s%s %6d %6d    %2d    %2d  %d:%2d %5d %5d %5d %5d %5d\n",
	    dev->name,
	    (m.mmr_dce_status & MMR_DCE_STATUS_ENERG_DET) ? "E" : " ",
	    (m.mmr_dce_status & MMR_DCE_STATUS_LOOPT_IND) ? "L" : " ",
	    (m.mmr_dce_status & MMR_DCE_STATUS_XMTITR_IND) ? "T" : " ",
	    (m.mmr_dce_status & MMR_DCE_STATUS_JBR_EXPIRED) ? "J" : " ",
	    lp->correct_nwid, lp->wrong_nwid,
	    m.mmr_signal_lvl & MMR_LEVEL_MASK,
	    m.mmr_silence_lvl & MMR_LEVEL_MASK,
	    (m.mmr_sgnl_qual & 0x80) == 0x80,  m.mmr_sgnl_qual & 0xF,
	    lp->snr[0], lp->snr[1], lp->snr[2], lp->snr[3], lp->snr[4]));
} /* sprintf_stats */

/*======================================================================

  Procedure for getting information for use by the WaveLAN information
  device.

======================================================================*/

static int wv_get_info(char *buffer)
{
    char *pos = buffer;
    unsigned long x;

    pos += sprintf(pos,"%s","Iface | dce  +nwid  -nwid level slnce  qual  snr0  snr1  snr2  snr3  snr4\n");

    x = wv_splhi();
    if (dev_list != NULL ) {
    	struct dev_link_t *link=dev_list;
    	do {
      	    pos = sprintf_stats(pos, WAVELAN_DEV(link));
    	}
    	while ((link = link->next) != NULL);
    }
    wv_splx(x);
    return(pos - buffer);
} /* wv_get_info */

/*====================================================================*/

#ifdef WAVELAN_DEBUG
/*======================================================================

  Print the formatted contents of the Parameter Storage Area.
    
======================================================================*/
#ifdef 0
static void wv_psa_show(psa_t *p)
{
  printk("wavelan psa contents:");
  printk("psa_io_base_addr_1: 0x%02x,", p->psa_io_base_addr_1);
  printk("psa_io_base_addr_2: 0x%02x,", p->psa_io_base_addr_2);
  printk("psa_io_base_addr_3: 0x%02x,", p->psa_io_base_addr_3);
  printk("psa_io_base_addr_4: 0x%02x,", p->psa_io_base_addr_4);
  printk("psa_rem_boot_addr_1: 0x%02x,", p->psa_rem_boot_addr_1);
  printk("psa_rem_boot_addr_2: 0x%02x,", p->psa_rem_boot_addr_2);
  printk("psa_rem_boot_addr_3: 0x%02x,", p->psa_rem_boot_addr_3);
  printk("psa_holi_params: 0x%02x,", p->psa_holi_params);
  printk("psa_int_req_no: %d,", p->psa_int_req_no);
  printk("psa_univ_mac_addr[]: %02x:%02x:%02x:%02x:%02x:%02x,",
	 p->psa_univ_mac_addr[0],
	 p->psa_univ_mac_addr[1],
	 p->psa_univ_mac_addr[2],
	 p->psa_univ_mac_addr[3],
	 p->psa_univ_mac_addr[4],
	 p->psa_univ_mac_addr[5]);
  printk("psa_local_mac_addr[]: %02x:%02x:%02x:%02x:%02x:%02x,",
	 p->psa_local_mac_addr[0],
	 p->psa_local_mac_addr[1],
	 p->psa_local_mac_addr[2],
	 p->psa_local_mac_addr[3],
	 p->psa_local_mac_addr[4],
	 p->psa_local_mac_addr[5]);
  printk("psa_univ_local_sel: %d,", p->psa_univ_local_sel);
  printk("psa_comp_number: %d,", p->psa_comp_number);
  printk("psa_thr_pre_set: 0x%02x,", p->psa_thr_pre_set);
  printk("psa_feature_select/decay_prm: 0x%02x,", p->psa_feature_select);
  printk("psa_subband/decay_update_prm: %d,", p->psa_subband);
  printk("psa_quality_thr: 0x%02x,", p->psa_quality_thr);
  printk("psa_mod_delay: 0x%02x,", p->psa_mod_delay);
  printk("psa_nwid: 0x%02x%02x,", p->psa_nwid[0], p->psa_nwid[1]);
  printk("psa_nwid_select: %d,", p->psa_nwid_select);
  printk("psa_encryption_select: %d,", p->psa_encryption_select);
  printk("psa_encryption_key[]: %02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x,",
	 p->psa_encryption_key[0],
	 p->psa_encryption_key[1],
	 p->psa_encryption_key[2],
	 p->psa_encryption_key[3],
	 p->psa_encryption_key[4],
	 p->psa_encryption_key[5],
	 p->psa_encryption_key[6],
	 p->psa_encryption_key[7]);
  printk("psa_databus_width: %d,", p->psa_databus_width);
  printk("psa_call_code/auto_squelch: 0x%02x,", p->psa_call_code[0]);
  printk("psa_conf_status: %d,", p->psa_conf_status);
  printk("psa_crc: 0x%02x%02x,", p->psa_crc[0], p->psa_crc[1]);
  printk("psa_crc_status: 0x%02x,", p->psa_crc_status);
  printk("\n");
} /* wavelan_psa_show */
#endif /* 0 */

/*======================================================================

  Print the formatted status of the Modem Management Controller.
    
======================================================================*/

static void wv_mmc_show(device *dev)
{
  u_char v;
  u_short base = dev->base_addr;
  net_local *lp = (net_local *)dev->priv;
  mmr_t	m;
  char *p;
  int low, high;

  if(hasr_read(base) & HASR_NO_CLK) {
    printk("wavelan_cs: wv_mmc_show: modem not connected\n");
    return;
  }

  p = (char *)&m;
  low = ((char *) &(m.mmr_correct_nwid_l)) - p;
  high = (p+sizeof(m)) - ((char *) &(m.mmr_thr_pre_set));

  v = (u_char) 1;
  mmc_write(dev->base_addr, mmwoff(0, mmw_freeze), &v, sizeof(v));
  mmc_read(base, 0, (u_char *)&m, low);
  mmc_read(base, low+1, (u_char *)&(m.mmr_correct_nwid_h), sizeof(u_char));
  mmc_read(base, low, (u_char *)&(m.mmr_correct_nwid_l), sizeof(u_char));
  mmc_read(base, low+3, (u_char *)&(m.mmr_wrong_nwid_h), sizeof(u_char));
  mmc_read(base, low+2, (u_char *)&(m.mmr_wrong_nwid_l), sizeof(u_char));
  mmc_read(base, low+4, (u_char *)&(m.mmr_thr_pre_set), high);

  v = (u_char) 0;
  mmc_write(dev->base_addr, mmwoff(0, mmw_freeze), &v, sizeof(v));

  lp->correct_nwid += (m.mmr_correct_nwid_h << 8) | m.mmr_correct_nwid_l;
  lp->wrong_nwid += (m.mmr_wrong_nwid_h << 8) | m.mmr_wrong_nwid_l;

  printk("mmr:");
  printk(" dce_status: 0x%x [%s%s%s%s]",
	 m.mmr_dce_status & 0x0F,
	 (m.mmr_dce_status & MMR_DCE_STATUS_ENERG_DET) ? "energy detected,":"",
	 (m.mmr_dce_status & MMR_DCE_STATUS_LOOPT_IND) ?
	 "loop test indicated," : "",
	 (m.mmr_dce_status & MMR_DCE_STATUS_XMTITR_IND) ? "transmitter on,"
	 : "",
	 (m.mmr_dce_status & MMR_DCE_STATUS_JBR_EXPIRED) ?
	 "jabber timer expired," : "");
  printk(" correct_nwid: %d",m.mmr_correct_nwid_h << 8 | m.mmr_correct_nwid_l);
  printk(" wrong_nwid: %d", (m.mmr_wrong_nwid_h << 8) | m.mmr_wrong_nwid_l);
  printk(" thr_pre_set: 0x%x [current signal %s]", m.mmr_thr_pre_set & 0x03F,
	 (m.mmr_thr_pre_set & 0x80) ? "above" : "below");
  printk(" signal_lvl: %d [%s]", m.mmr_signal_lvl & 0x03F,
	 (m.mmr_signal_lvl & 0x80) ? "new msg" : "no new msg");
  printk(" silence_lvl: %d [%s]", m.mmr_silence_lvl & 0x03F,
	 (m.mmr_silence_lvl & 0x80) ? "update done" : "no new update");
  printk(" sgnl_qual: 0x%x [%s]", m.mmr_sgnl_qual & 0x0F,
	 (m.mmr_sgnl_qual & 0x80) ? "Antenna 1" : "Antenna 0");
  printk(" netw_id_l: %x", m.mmr_netw_id_l);
  printk("\n");
} /* wv_mmc_show */

/*======================================================================

  Print the formatted status of the i82593's receive unit.
    
======================================================================*/

static void wv_ru_show(device *dev)
{
  net_local *lp = (net_local *)dev->priv;

  printk("ru: rfp %d stop %d", lp->rfp, lp->stop);
  /*
   * Not implemented yet...
   */
  printk("\n");
} /* wv_ru_show */

/*======================================================================

  Print the formatted status of the WaveLAN PCMCIA device driver.
    
======================================================================*/

static void wv_dev_show(device *dev)
{
  printk("dev:");
  printk(" start=%d,", dev->start);
  printk(" tbusy=%ld,", dev->tbusy);
  printk(" interrupt=%d,", dev->interrupt);
  printk(" trans_start=%ld,", dev->trans_start);
  printk(" flags=0x%x,", dev->flags);
  printk("\n");
} /* wv_dev_show */

/*======================================================================

  Print the formatted status of the WaveLAN PCMCIA device driver's
  private information.
    
======================================================================*/

static void wv_local_show(device *dev)
{
  net_local *lp;

  lp = (net_local *)dev->priv;

  printk("local:");
  /*
   * Not implemented yet...
   */
  printk("\n");
} /* wv_local_show */
#endif	/* WAVELAN_DEBUG */

/*======================================================================

  Routine to synchronously send a command to the i82593 chip. 
  Must be called with interrupts enabled.
    
======================================================================*/

static int wv_82593_cmd(device *dev, char *str, int cmd, int result)
{
    u_short base = dev->base_addr;
    net_local *lp = (net_local *)dev->priv;
    int status, spin;
    unsigned long opri;

#ifdef WAVELAN_DEBUG
    if (wavelan_debug > 3)
    	printk("wv_82593_cmd()\n");
#endif

    /* Spin until the chip finishes executing its current command (if any) */
    do {
    	opri = wv_splhi();
    	outb(OP0_NOP | CR0_STATUS_3, LCCR(base));
    	status = inb(LCSR(base));
    	wv_splx(opri);
    } while ((status & SR3_EXEC_STATE_MASK) != SR3_EXEC_IDLE);

    lp->cmd_wait = TRUE;	/* Signal that we are waiting for command completion */
    outb(cmd, LCCR(base));	/* Send the command */
    if(result == SR0_NO_RESULT) {	/* Return immediately, if the command
					   doesn't return a result */
    	lp->cmd_wait = FALSE;
    	return(TRUE);
    }

    /* Busy wait while the LAN controller executes the command.
     * Interrupts had better be enabled (or it will be a long wait).
     *  (We could enable just the WaveLAN's IRQ...)
     */
    for(spin = 0; (spin < 10000000) && lp->cmd_wait; spin++)
    	;

    if(lp->cmd_wait){
    	outb(OP0_NOP, LCCR(base));
    	status = inb(LCSR(base));
    	if(status & 0x80){
      	    wv_interrupt(dev);
    	} 
	else {
      	    lp->cmd_wait = 0; /* XXX */
      	    printk("wv_82593_cmd: %s timeout, status0 0x%02x\n", str, status);
      	    /* We probably should reset the controller here */
      	    return(FALSE);
    	}
    }

    /* Check the return code provided by the interrupt handler against
       the expected return code provided by the caller */
    if((lp->status & SR0_EVENT_MASK) != result) {
    	printk("wv_82593_cmd: %s failed, status0 = 0x%x\n", str, lp->status);
    	return(FALSE);
    }
#ifdef WAVELAN_DEBUG
    if (wavelan_debug > 3)
    	printk("wv_82593_cmd: %s succeeded, status0 = 0x%x\n",str, lp->status);
#endif
    return(TRUE);
} /* wv_82593_cmd */

/*======================================================================

  This routine does a 593 op-code number 7, and obtains the diagnose
  status for the WaveLAN.
    
======================================================================*/

static int wv_diag(device *dev)
{
    dev_link_t *link;

    for (link = dev_list; link; link = link->next)
    	if (link->priv == dev) break;
    if (!DEV_OK(link))
    	return(-ENODEV);

#ifdef WAVELAN_DEBUG
    if (wavelan_debug > 0)
    	printk("wavelan_cs: entered wv_diag()\n");
#endif
    if (wv_82593_cmd(dev, "wv_diag(): diagnose", OP0_DIAGNOSE,
		  SR0_DIAGNOSE_PASSED))
    	return(TRUE);
    printk("wavelan_cs: i82593 Self Test failed!\n");
    return(FALSE);
} /* wv_diag */

/*======================================================================

  This routine does a standard config of the WaveLAN card.
    
======================================================================*/

static int wv_hw_config(device *dev, u_short base)
{
    struct i82593_conf_block cfblk;
    net_local *lp = (net_local *)dev->priv;

#ifdef WAVELAN_DEBUG
    if (wavelan_debug > 0)
    	printk("wv_hw_config(), size %d\n", sizeof(struct i82593_conf_block));
#endif

    memset(&cfblk, 0x00, sizeof(struct i82593_conf_block));
    cfblk.d6mod = FALSE;  	/* Run in i82593 advanced mode */
    cfblk.fifo_limit = 6;         /* = 48 bytes rx and tx fifo thresholds */
    cfblk.forgnesi = FALSE;       /* 0=82C501, 1=AMD7992B compatibility */
    cfblk.fifo_32 = 0;
    cfblk.throttle_enb = TRUE;
    cfblk.contin = TRUE;          /* enable continuous mode */
    cfblk.cntrxint = FALSE;       /* enable continuous mode receive interrupts */
    cfblk.addr_len = WAVELAN_ADDR_SIZE;
    cfblk.acloc = TRUE;           /* Disable source addr insertion by i82593 */
    cfblk.preamb_len = 2;         /* 7 byte preamble */
    cfblk.loopback = FALSE;
    cfblk.lin_prio = 0;   	/* conform to 802.3 backoff algoritm */
    cfblk.exp_prio = 0;	        /* conform to 802.3 backoff algoritm */
    cfblk.bof_met = 0;	        /* conform to 802.3 backoff algoritm */
    cfblk.ifrm_spc = 6;	        /* 96 bit times interframe spacing */
    cfblk.slottim_low = 0x10 & 0x7;	/* 512 bit times slot time */
    cfblk.slottim_hi = 0x10 >> 3;
    cfblk.max_retr = 15;	
    cfblk.prmisc = ((lp->promiscuous) ? TRUE: FALSE);	/* Promiscuous mode */
    cfblk.bc_dis = FALSE;         /* Enable broadcast reception */
    cfblk.crs_1 = TRUE;		/* Transmit without carrier sense */
    cfblk.nocrc_ins = FALSE;	/* i82593 generates CRC */	
    cfblk.crc_1632 = FALSE;	/* 32-bit Autodin-II CRC */
    cfblk.crs_cdt = FALSE;	/* CD not to be interpreted as CS */
    cfblk.cs_filter = 0;  	/* CS is recognized immediately */
    cfblk.crs_src = FALSE;	/* External carrier sense */
    cfblk.cd_filter = 0;  	/* CD is recognized immediately */
    cfblk.min_fr_len = ETH_ZLEN >> 2;     /* Minimum frame length 64 bytes */
    cfblk.lng_typ = FALSE;	/* Length field > 1500 = type field */
    cfblk.lng_fld = TRUE; 	/* Disable 802.3 length field check */
    cfblk.rxcrc_xf = TRUE;	/* Don't transfer CRC to memory */
    cfblk.artx = TRUE;		/* Disable automatic retransmission */
    cfblk.sarec = TRUE;		/* Disable source addr trig of CD */
    cfblk.tx_jabber = TRUE;	/* Disable jabber jam sequence */
    cfblk.hash_1 = FALSE; 	/* Use bits 0-5 in mc address hash */
    cfblk.lbpkpol = TRUE; 	/* Loopback pin active high */
    cfblk.fdx = FALSE;		/* Disable full duplex operation */
    cfblk.dummy_6 = 0x3f; 	/* all ones */
    cfblk.mult_ia = FALSE;	/* No multiple individual addresses */
    cfblk.dis_bof = FALSE;	/* Disable the backoff algorithm ?! */
    cfblk.dummy_1 = TRUE; 	/* set to 1 */
    cfblk.tx_ifs_retrig = 3;	/* Hmm... Disabled */
#ifdef MULTICAST_ALL
    cfblk.mc_all = (lp->multicast ? TRUE: FALSE);	/* Allow all multicasts */
#else
    cfblk.mc_all = FALSE;		/* No multicast all mode */
#endif
    cfblk.rcv_mon = 0;		/* Monitor mode disabled */
    cfblk.frag_acpt = TRUE;	/* Do not accept fragments */
    cfblk.tstrttrs = FALSE;	/* No start transmission threshold */
    cfblk.fretx = TRUE;		/* FIFO automatic retransmission */
    cfblk.syncrqs = TRUE; 	/* Synchronous DRQ deassertion... */
    cfblk.sttlen = TRUE;  	/* 6 byte status registers */
    cfblk.rx_eop = TRUE;  	/* Signal EOP on packet reception */
    cfblk.tx_eop = TRUE;  	/* Signal EOP on packet transmission */
    cfblk.rbuf_size = RX_SIZE>>11;	/* Set receive buffer size */
    cfblk.rcvstop = TRUE; 	/* Enable Receive Stop Register */

#ifdef WAVELAN_DEBUG
    if (wavelan_debug > 2) {
    	u_char *c = (u_char *) &cfblk;
    	int i;
    	printk("wavelan_cs: config block:");
    	for(i = 0; i < sizeof(struct i82593_conf_block); i++,c++){
      	    if((i % 16) == 0) printk("\n");
      	    printk("%02x ", *c);
    	}
    	printk("\n");
    }
#endif

    outb(TX_BASE & 0xff, PIORL(base));
    outb(((TX_BASE >> 8) & PIORH_MASK) | PIORH_SEL_TX, PIORH(base));
    outb(sizeof(struct i82593_conf_block) & 0xff, PIOP(base));    /* lsb */
    outb(sizeof(struct i82593_conf_block) >> 8, PIOP(base));	/* msb */
    outsb(PIOP(base), (char *) &cfblk, sizeof(struct i82593_conf_block));
    /* reset transmit DMA pointer */
    hacr_write_slow(base, HACR_PWR_STAT | HACR_TX_DMA_RESET);
    hacr_write(base, HACR_DEFAULT);
    if(!wv_82593_cmd(dev, "wv_hw_config(): configure", OP0_CONFIGURE,
		  SR0_CONFIGURE_DONE))
    	return(FALSE);

    /* Initialize adapter's ethernet MAC address */
    outb(TX_BASE & 0xff, PIORL(base));
    outb(((TX_BASE >> 8) & PIORH_MASK) | PIORH_SEL_TX, PIORH(base));
    outb(WAVELAN_ADDR_SIZE, PIOP(base));	/* byte count lsb */
    outb(0, PIOP(base));			/* byte count msb */
    outsb(PIOP(base), &dev->dev_addr[0], WAVELAN_ADDR_SIZE);
    /* reset transmit DMA pointer */
    hacr_write_slow(base, HACR_PWR_STAT | HACR_TX_DMA_RESET);
    hacr_write(base, HACR_DEFAULT);
    if(!wv_82593_cmd(dev, "wv_hw_config(): ia-setup", OP0_IA_SETUP,
		  SR0_IA_SETUP_DONE))
    	return(FALSE);

    if (lp->multicast) {	/* This code still needs to be tested */
	int i;
    	int addrs_len = WAVELAN_ADDR_SIZE * dev->mc_count;

    	/* Initialize adapter's ethernet multicast addresses */
    	outb(TX_BASE & 0xff, PIORL(base));
    	outb(((TX_BASE >> 8) & PIORH_MASK) | PIORH_SEL_TX, PIORH(base));
    	outb(addrs_len & 0xff, PIOP(base));	/* byte count lsb */
    	outb((addrs_len >> 8), PIOP(base));	/* byte count msb */
	for ( i=0; i<dev->mc_count; i++ )
    	    outsb(PIOP(base), &(dev->mc_list->dmi_addr), dev->mc_list->dmi_addrlen);
    	/* reset transmit DMA pointer */
    	hacr_write_slow(base, HACR_PWR_STAT | HACR_TX_DMA_RESET);
    	hacr_write(base, HACR_DEFAULT);
    	if(!wv_82593_cmd(dev, "wv_hw_config(): mc-setup", OP0_MC_SETUP,
		    SR0_MC_SETUP_DONE))
      	    return(FALSE);
    }

#ifdef WAVELAN_DEBUG
    if (wavelan_debug > 0)
    	printk("wv_hw_config complete\n");
#endif
    return(TRUE);
} /* wv_hw_config */

/*======================================================================

  Read the Access Configuration Register, perform a software reset, and
  then re-enable the card's software.
    
======================================================================*/

static void wv_sw_reset(device *dev)
{
    int i;
    conf_reg_t reg = { 0, CS_READ, CISREG_COR, 0 };
    dev_link_t *link;

#ifdef WAVELAN_DEBUG
    if (wavelan_debug > 0)
    	printk("wv_sw_reset\n");
#endif

    for (link = dev_list; link; link = link->next)
    	if (link->priv == dev) break;
    if (!DEV_OK(link)) {
    	printk("wavelan_cs: wv_sw_reset(): No device found\n");
    	return;
    }

    i = CardServices(AccessConfigurationRegister, link->handle, &reg);
    if (i != CS_SUCCESS) {
    	cs_error(AccessConfigurationRegister, i);
    	return;
    }
      
#ifdef WAVELAN_DEBUG
    if (wavelan_debug > 1)
    	printk("wavelan_cs: Config reg is 0x%x\n", (u_int) reg.Value);
#endif
      
    reg.Action = CS_WRITE;
    reg.Value = reg.Value | COR_SW_RESET;
    i = CardServices(AccessConfigurationRegister, link->handle, &reg);
    if (i != CS_SUCCESS) {
    	cs_error(AccessConfigurationRegister, i);
    	return;
    }
      
    reg.Action = CS_WRITE;
    reg.Value = COR_LEVEL_IRQ | COR_CONFIG;
    i = CardServices(AccessConfigurationRegister, link->handle, &reg);
    if (i != CS_SUCCESS) {
    	cs_error(AccessConfigurationRegister, i);
    	return;
    }
} /* wv_sw_reset */

/*======================================================================

  Routine to initialize the Modem Management Controller.
    
======================================================================*/

static void wv_mmc_init(device *dev)
{
    u_short base = dev->base_addr;
    net_local *lp = (net_local *)dev->priv;
    mmw_t	m;
    unsigned char tmp; 	/* added by Joe Finney */
    int configured;

    memset(&m, 0x00, sizeof(m));

#ifdef USE_PSA_CONFIG
    /*
     * For now we use the persistent PSA information as little as
     * possible, thereby allowing us to return to the same known
     * state during a hardware reset.
     */
    configured = lp->psa.psa_conf_status & 1;
#else
    configured = 0;
#endif
	
    /*
     * Set default modem control parameters.
     * See NCR document 407-0024326 Rev. A.
     */
    m.mmw_jabber_enable = 0x01;
    m.mmw_anten_sel = MMW_ANTEN_SEL_ALG_EN;
    m.mmw_ifs = 0x20;
    m.mmw_mod_delay = 0x04;
    m.mmw_jam_time = 0x38;

    m.mmw_encr_enable = 0;
    m.mmw_des_io_invert = 0;
    m.mmw_freeze = 0;
    m.mmw_decay_prm = 0;
    m.mmw_decay_updat_prm = 0;

    if (configured) {
    	/*
    	 * Use configuration defaults from parameter storage area.
    	 */
    	if (lp->psa.psa_nwid_select & 1)
      	    m.mmw_loopt_sel = 0x00;
    	else
      	    m.mmw_loopt_sel = MMW_LOOPT_SEL_UNDEFINED;  /* disable nw id check */

    	m.mmw_thr_pre_set = lp->psa.psa_thr_pre_set & 0x3F;
    	m.mmw_quality_thr = lp->psa.psa_quality_thr & 0x0F;
    } 
    else {
    	if (lp->promiscuous)
      	    m.mmw_loopt_sel = MMW_LOOPT_SEL_UNDEFINED;
    	else
      	    m.mmw_loopt_sel = 0x00;	/* For an unconfig. card, should really use
  				       MMW_LOOPT_SEL_UNDEFINED to get all NWIDs */
    	if (lp->psa.psa_comp_number == 4)
      	    m.mmw_thr_pre_set = 0x03;   /* PCMCIA */
    	else {
      	    /*
      	     * 0x04 for AT,
      	     * 0x01 for MCA.
      	     */
      	    if (lp->psa.psa_comp_number & 1)
		m.mmw_thr_pre_set = 0x01;
      	    else
		m.mmw_thr_pre_set = 0x04;
    	}
    	m.mmw_quality_thr = 0x03;
    }

    m.mmw_netw_id_l = lp->nwid[1];
    m.mmw_netw_id_h = lp->nwid[0];
  
    if(hasr_read(base) & HASR_NO_CLK) {
    	printk("wavelan_cs: wv_mmc_init: modem not connected\n");
    	return;
    }
    mmc_write(base, 0, (u_char *)&m, sizeof(m));

    /* This is the code needed to start the modems receive unit
       properly on version 2.00 frequency select cards; 
       Thanks, Joe! */

    udelay(1000L);
    tmp=0x0F;
    mmc_write(base,0x21,&tmp,1);
    tmp=0x0E;
    mmc_write(base,0x20,&tmp,1);
    udelay(10000L);
    tmp=0x61;
    mmc_write(base,0x21,&tmp,1);
    tmp=0x0E;
    mmc_write(base,0x20,&tmp,1);
    udelay(10000L);
    /* 2.00 patch. Regards, Joe. */

} /* wv_mmc_init */

/*======================================================================

  Routine to gracefully turn off reception, and wait for any commands
  to complete.
    
======================================================================*/

static void wv_graceful_shutdown(device *dev)
{
    u_short base = dev->base_addr;
    unsigned long opri;
    int status;
  
    /* First, send the LAN controller a stop receive command */
    wv_82593_cmd(dev, "wv_graceful_shutdown(): stop-rcv", OP0_STOP_RCV,
	      SR0_NO_RESULT);
    /* Then, spin until the receive unit goes idle */
    do {
    	opri = wv_splhi();
    	outb(OP0_NOP | CR0_STATUS_3, LCCR(base));
    	status = inb(LCSR(base));
    	wv_splx(opri);
    } while((status & SR3_RCV_STATE_MASK) != SR3_RCV_IDLE);
		       
    /* Now, spin until the chip finishes executing its current command */
    do {
    	opri = wv_splhi();
    	outb(OP0_NOP | CR0_STATUS_3, LCCR(base));
    	status = inb(LCSR(base));
    	wv_splx(opri);
    } while ((status & SR3_EXEC_STATE_MASK) != SR3_EXEC_IDLE);
} /* wv_graceful_shutdown */

/*======================================================================

  This routine starts the receive unit running.  First, it checks if
  the card is actually ready. Then the card is instructed to receive
  packets again.
    
======================================================================*/

static void wv_ru_start(device *dev)
{
    u_short base = dev->base_addr;
    net_local *lp = (net_local *)dev->priv;

#ifdef WAVELAN_DEBUG
    if (wavelan_debug > 1)
    	printk("wavelan_cs: entered wv_ru_start()\n");
#endif

    /*
     * We need to start from a quiescent state. To do so, we could check
     * if the card is already running, but instead we just try to shut
     * it down. First, we disable reception (in case it was already enabled).
     */
    wv_graceful_shutdown(dev);

    /* Now we know that no command is being executed. */

    /* Set the receive frame pointer and stop pointer */
    lp->rfp = 0;
    outb(OP0_SWIT_TO_PORT_1 | CR0_CHNL, LCCR(base));

    /* Reset ring management.  This sets the receive frame pointer to 1 */
    outb(OP1_RESET_RING_MNGMT, LCCR(base));

#if 0
    /* XXX the i82593 manual page 6-4 seems to indicate that the stop register
       should be set as below */
    /* outb(CR1_STOP_REG_UPDATE|((RX_SIZE - 0x40)>> RX_SIZE_SHIFT),LCCR(base));*/
#elif 0
    /* but I set it 0 instead */
    lp->stop = 0;
#else
    /* but I set it to 3 bytes per packet less than 8K */
    lp->stop = (0 + RX_SIZE - ((RX_SIZE / 64) * 3)) % RX_SIZE;
#endif
    outb(CR1_STOP_REG_UPDATE | (lp->stop >> RX_SIZE_SHIFT), LCCR(base));
    outb(OP1_INT_ENABLE, LCCR(base));
    outb(OP1_SWIT_TO_PORT_0, LCCR(base));

    /* Reset receive DMA pointer */
    outb(HACR_PWR_STAT | HACR_RX_DMA_RESET, HACR(base));
    udelay(1000L);
    outb(HACR_PWR_STAT, HACR(base));
    udelay(1000L);

    /* Receive DMA on channel 1 */
    wv_82593_cmd(dev, "wv_ru_start(): rcv-enable",
	      CR0_CHNL | OP0_RCV_ENABLE, SR0_NO_RESULT);

#ifdef WAVELAN_DEBUG
    if (wavelan_debug > 4) {
    	int status, opri, i = 0;
      	/* spin until the chip starts receiving */
    	do {
       	    opri = wv_splhi();
      	    outb(OP0_NOP | CR0_STATUS_3, LCCR(base));
      	    status = inb(LCSR(base));
      	    wv_splx(opri);
      	    if (i++ > 10000) break;
    	} while (((status & SR3_RCV_STATE_MASK) != SR3_RCV_ACTIVE) &&
	     ((status & SR3_RCV_STATE_MASK) != SR3_RCV_READY));
    	printk("rcv status is 0x%x [i:%d]\n", (status & SR3_RCV_STATE_MASK), i);
    }
    if (wavelan_debug > 1)
    	printk("wavelan_cs: leaving wv_ru_start()\n");
#endif
} /* wv_ru_start */

/*======================================================================

  Performs the following actions:
  	1. A software reset (using wv_sw_reset)
	2. A power reset
	3. Reset the LAN controller
	4. Initialize the radio modem (using wv_mmc_init)
	5. Configure LAN controller
	6. Perform a diagnostic on the LAN controller
	7. Start the LAN controller's receive unit
    
======================================================================*/

static int wv_hw_reset(device *dev)
{
    u_short base;
    dev_link_t *link;
    net_local *lp;

#ifdef WAVELAN_DEBUG
    if (wavelan_debug > 0)
    	printk("wv_hw_reset\n");
#endif

    for (link = dev_list; link; link = link->next)
    	if (link->priv == dev) break;
    if (!DEV_OK(link)) {
    	printk("wavelan_cs: wv_hw_reset(): No device found\n");
    	return(-1);
    }

    wv_sw_reset(dev);
    lp = (net_local *)dev->priv;
    lp->nresets++;

    base = link->io.BasePort1;
    hacr_write_slow(base, HACR_RESET);
    hacr_write(base, HACR_DEFAULT);

    if(hasr_read(base) & HASR_NO_CLK) {
    	printk("wavelan_cs: modem not connected\n");
    	return(-1);
    }

    outb(OP0_RESET, LCCR(base));	/* reset the LAN controller */
    udelay(1000L);
    wv_mmc_init(dev);        /* initialize modem */

    if (wv_hw_config(dev, base) == FALSE)
    	return(-1);

    if (wv_diag(dev) == FALSE)
    	return(-1);

    /* 
     * insert code for loopback test here
     */
    wv_ru_start(dev);	/* start receive unit */
    return(0);
} /* wv_hw_reset */

/*======================================================================

  Routine to read len bytes from the i82593's ring buffer, starting at
  chip address addr. The results read from the chip are stored in buf.
  The return value is the address to use for next the call.
    
======================================================================*/

static inline int read_ringbuf(device *dev, int addr, char *buf, int len)
{
    u_short base = dev->base_addr;
    int ring_ptr = addr;
    int chunk_len;
    char *buf_ptr = buf;

    /* If buf is NULL, just increment the ring buffer pointer */
    if (buf == NULL)
    	return((ring_ptr - RX_BASE + len) % RX_SIZE + RX_BASE);
    while (len > 0) {
    	/* Position the Program I/O Register at the ring buffer pointer */
    	outb(ring_ptr & 0xff, PIORL(base));
    	outb(((ring_ptr >> 8) & PIORH_MASK), PIORH(base));
    	/* First, determine how much we can read without wrapping around the
           ring buffer */
    	if ((addr + len) < (RX_BASE + RX_SIZE))
      	    chunk_len = len;
    	else
      	    chunk_len = RX_BASE + RX_SIZE - addr;
    	insb(PIOP(base), buf_ptr, chunk_len);
    	buf_ptr += chunk_len;
    	len -= chunk_len;
    	ring_ptr = (ring_ptr - RX_BASE + chunk_len) % RX_SIZE + RX_BASE;
    }
    return(ring_ptr);
} /* read_ringbuf */

/*======================================================================

  Calculate the starting address of the frame pointed to by the receive
  frame pointer.
    
======================================================================*/

static inline int wv_start_of_frame(device *dev, int rfp)
{
    u_short base = dev->base_addr;
    int rp, len;

    rp = (rfp - 5 + RX_SIZE) % RX_SIZE;
    outb(rp & 0xff, PIORL(base));
    outb(((rp >> 8) & PIORH_MASK), PIORH(base));
    len = inb(PIOP(base));
    len |= inb(PIOP(base)) << 8;

    if (len > 1600) {		/* Sanity check on size */
    	printk("wavelan_cs: Received frame too large, rfp %d rp %d len 0x%x\n",
	     rfp, rp, len);
    	return(-1);
    }
  
#ifdef PARANOID
    {	/* For the truly paranoid, this routine checks that the length
	   computed above, really matches the length in i82593's ring
	   buffer. */
    	int len1, len_ptr;
    	u_char c;
    
    	len_ptr = rp;
    	len_ptr = read_ringbuf(dev, len_ptr, &c, 1);
    	len1 = c;
    	len_ptr = read_ringbuf(dev, len_ptr, &c, 1);
    	len1 |= c << 8;
    	if (len != len1 || len > 1600) {
      	    printk("wavelan_cs: oops, rfp %d rp %d len 0x%x len1 0x%x\n", rfp, rp,
	     	len, len1);
      	    return(-1);
    	}
    }
#endif /* PARANOID */
    return((rp - len + RX_SIZE) % RX_SIZE);
} /* wv_start_of_frame */

/*======================================================================

  This routine fills in the appropriate registers and memory
  locations on the WaveLAN card and starts the card off on
  the transmit.
    
======================================================================*/

static void wv_packet_write(device *dev, void *buf, short length, 
					 struct ethhdr *eh)
{
  u_short base = dev->base_addr;
  unsigned long x;
  int clen = length;
  register u_short xmtdata_p, xmtdata_base = TX_BASE;

  /* First, we copy out the data; then, we output the length */

  x = wv_splhi();
  xmtdata_p = xmtdata_base + 2;		/* Skip the length field for now */
  outb(xmtdata_p & 0xff, PIORL(base));
  outb(((xmtdata_p >> 8) & PIORH_MASK) | PIORH_SEL_TX, PIORH(base));

  outsb(PIOP(base), buf, length);	/* Send the data */
  
#if 0	/* We set the length greater than ETH_ZLEN in the caller */
  if (clen < ETH_ZLEN) {		/* Pad out to minimum packet length */
    for (; clen < ETH_ZLEN; ++clen)
      outb(0, PIOP(base));
  }
#endif

  outb(OP0_NOP, PIOP(base));		/* Indicate end of transmit chain */
#ifdef WAVELAN_DEBUG
  if (wavelan_debug > 3)
    printk("CLEN = %d\n", clen);
#endif 

  /* Now, write the length of data buffer */
  outb(xmtdata_base & 0xff, PIORL(base));
  outb(((xmtdata_base >> 8) & PIORH_MASK) | PIORH_SEL_TX, PIORH(base));
  outb(clen & 0xff, PIOP(base));	/* lsb */
  outb(clen >> 8, PIOP(base));  	/* msb */
#ifdef	IF_CNTRS
  clen += 4 		/* Count the crc added by the i82593 */;
  lp->cntrs.pkt_eout[log_2(clen)]++;
  if (clen < 128)  lp->cntrs.pkt_lout[clen>>3]++;
#endif	IF_CNTRS
  
  /* Reset the transmit DMA pointer */
  hacr_write_slow(base, HACR_PWR_STAT | HACR_TX_DMA_RESET);
  hacr_write(base, HACR_DEFAULT);
  /* Send the transmit command */
  wv_82593_cmd(dev, "wv_packet_write(): transmit", OP0_TRANSMIT,
	      SR0_NO_RESULT);

  dev->trans_start = jiffies;	/* Record the initial transmit time */
  wv_splx(x); 

#ifdef WAVELAN_DEBUG
  if (wavelan_debug > 3) {
    u_char *a;
    a = (u_char *)buf;
    printk("%s: tx: dest %02x:%02x:%02x:%02x:%02x:%02x, length %d\n",
	   dev->name,
	   a[0], a[1], a[2], a[3], a[4], a[5],
	   length);
    printk("%s: tx: src %02x:%02x:%02x:%02x:%02x:%02x, type 0x%02x%02x\n",
	   dev->name,
	   a[6], a[7], a[8], a[9], a[10], a[11],
	   a[12], a[13]);
  }
#endif
} /* wv_packet_write */

static int wv_packet_xmit(struct sk_buff *skb, device *dev)
{
#ifdef WAVELAN_DEBUG
  if (wavelan_debug > 3)
    printk("wavelan_cs: wv_packet_xmit\n");
#endif

  if (dev->tbusy) {
    /*
     * If we get here, some higher level has decided we are broken.
     * The ISA/MC WaveLAN driver uses a kernel timer to detect timeouts.
     * That approach doesn't work for us, since the timer could expire
     * AFTER the card had been removed (not a good thing --- tends to
     * hang everything). So instead, we rely upon the ordinary method for
     * detecting a timeout.
     */
    int tickssofar;
#ifdef 0
    int timeout = 3000;
    int opri, status;
    u_short base = dev->base_addr;

    printk("wv_packet_xmit: device busy\n");
    /* Spin until the chip finished executing any current command, since
       we may have to do a hardware reset */
    do {
      opri = wv_splhi();
      outb(OP0_NOP | CR0_STATUS_3, LCCR(base));
      status = inb(LCSR(base));
      wv_splx(opri);
      timeout--;
    } while (timeout && ((status & SR3_EXEC_STATE_MASK) != SR3_EXEC_IDLE));

    if (!timeout) {
#endif
      tickssofar = jiffies - dev->trans_start;
#ifdef WAVELAN_DEBUG
      if (wavelan_debug) printk("wavelan_cs: transmitter timeout\n");
#endif
       if (tickssofar < 15)
         return(1);
#ifdef WAVELAN_DEBUG
      if (wavelan_debug > 2) {
	wv_ru_show(dev);
	wv_dev_show(dev);
	wv_local_show(dev);
      }
#endif
      printk("%s: transmit timed out -- resetting card.\n", dev->name);
      dev->trans_start = jiffies;
      dev->tbusy = 0;
      wv_hw_reset(dev);
#ifdef 0
    } 
#endif
  }

  /*
   * If some higher layer thinks we've missed
   * a tx-done interrupt we are passed NULL.
   * Caution: dev_tint() handles the cli()/sti() itself.
   */
  if (skb == (struct sk_buff *)0) {
    printk("wv_packet_xmit: skb=NULL\n");
    dev_tint(dev);
    return(0);
  }

  if (skb->len <= 0)
  {
    printk("wv_packet_xmit: skb->len<=0\n");
    return(0);
  }
    
#ifdef WAVELAN_DEBUG
      if (wavelan_debug > 3)
	printk("%s: wv_packet_xmit(length = %ld) called\n", dev->name,
	       skb->len);
#endif

  /*
   * For ethernet, fill in the header.
   */
  skb->arp = 1;

  /*
   * Block a timer-based transmit from overlapping a previous transmit.
   */
  if (set_bit(0, (void *)&dev->tbusy) != 0)
    printk("%s: Transmitter access conflict.\n", dev->name);
  else {
    short length;
    u_char *buf;

    if (skb->next) printk("skb has next\n");
    length = (ETH_ZLEN < skb->len) ? skb->len : ETH_ZLEN;
    buf = skb->data;
    
    wv_packet_write(dev, buf, length,(struct ethhdr *) &(skb->h));
  }

  dev_kfree_skb(skb, FREE_WRITE);
  return(0);
} /* wv_packet_xmit */

/*======================================================================

  This routine does the actual copy of data (including the ethernet
  header structure) from the WaveLAN card to an sk_buff chain that
  will be passed up to the network interface layer. NOTE: We
  currently don't handle trailer protocols (neither does the rest of
  the network interface), so if that is needed, it will (at least in
  part) be added here.  The contents of the receive ring buffer are
  copied to a message chain that is then passed to the kernel.

  Note: if any errors occur, the packet is "dropped on the floor"
    
======================================================================*/

static void wv_packet_read(device *dev, int fd_p, int sksize)
{
  net_local *lp = (net_local *)dev->priv;
  struct sk_buff *skb;

#ifdef WAVELAN_DEBUG
  if (wavelan_debug > 4)
    printk("wavelan_cs: entered wv_packet_read() fd_p 0x%x len %d\n",
	   fd_p, sksize);
#endif

  lp->stats.rx_packets++;

  if ((skb = ALLOC_SKB(sksize)) == (struct sk_buff *)0) {
    printk("%s: could not alloc_skb(%d, GFP_ATOMIC).\n", dev->name, sksize);
    lp->stats.rx_dropped++;
    /*
     * Not only do we want to return here, but we also need to drop the
     * packet on the floor to clear the interrupt.
     */
    return;
  } else {
    skb->dev = dev;
    
#define BLOCK_INPUT(buf, len) \
    fd_p = read_ringbuf(dev, fd_p, (char *)buf, len)

    GET_PACKET(dev, skb, sksize);

#ifdef WAVELAN_DEBUG
   if (wavelan_debug > 3) {
     u_char *a;
     a = (u_char *)(skb->data);
     printk("%s: rx: dest %02x:%02x:%02x:%02x:%02x:%02x,\n",
	    dev->name,
	    a[0], a[1], a[2], a[3], a[4], a[5]);
     printk("%s: rx: src %02x:%02x:%02x:%02x:%02x:%02x, type 0x%02x%02x, len %d\n",
	    dev->name,
	    a[6], a[7], a[8], a[9], a[10], a[11],
	    a[12],a[13],
	    sksize);

     
     if (wavelan_debug > 4) {
       int i;
       int maxi;
       printk("%s: len=%d, data=\"", dev->name, sksize);
       if ((maxi = sksize) > 16)
	 maxi = 16;
       for (i = 0; i < maxi; i++) {
	 u_char   c;
	 c = skb->data[i];
	 if (c >= ' ' && c <= '~')
	   printk(" %c", skb->data[i]);
	 else
	   printk("%02x", skb->data[i]);
       }
       if (maxi < sksize)
	 printk("..");
       printk("\"\n\n");
     }
   }
#endif

#ifdef	IF_CNTRS
    lp->cntrs.pkt_ein[log_2(sksize)]++;
    if (sksize < 128) lp->cntrs.pkt_lin[sksize>>3]++;

    if (eh.h_proto == ETHERTYPE_ARP) {
      lp->cntrs.pkt_arp++;
      if (pkt_narp) {
	lp->cntrs.pkt_ein[log_2(sksize)]--;
	if (sksize < 128) lp->cntrs.pkt_lin[sksize>>3]--;
      }
    }
#endif	IF_CNTRS
    if(gathersnr) {
      u_char siglvl, sillvl, sigqual;
      /* skip status and len fields */
      fd_p = read_ringbuf(dev, fd_p, NULL, 4);
      /* read signal level, silence level and signal quality bytes */
      fd_p = read_ringbuf(dev, fd_p, &siglvl, 1);
      fd_p = read_ringbuf(dev, fd_p, &sillvl, 1);
      fd_p = read_ringbuf(dev, fd_p, &sigqual, 1);
#ifdef WAVELAN_DEBUG
     if (wavelan_debug > 4)
       printk("wavelan_cs: Signal level %d/63  Silence level %d/63 signal quality 0x%x/0xf\n",
	      siglvl & 0x03F,sillvl & 0x03F,sigqual & 0x0F);
#endif
      wv_getsnr(dev,siglvl,sillvl,sigqual);
    }
  
    /*
     * Hand the packet to the Network Module
     */
    netif_rx(skb);
    return;
  }
} /* wv_packet_read */


/*======================================================================

  This routine is called by the interrupt handler to initiate a packet
  transfer from the card to the network interface layer above this
  driver.  This routine checks if a buffer has been successfully
  received by the WaveLAN card.  If so, the routine wv_packet_read is
  called to do the actual transfer of the card's data including the
  ethernet header into a packet consisting of an sk_buff chain.
    
======================================================================*/

static void wv_packet_rcv(device *dev)
{
  u_short base = dev->base_addr;
  net_local *lp = (net_local *)dev->priv;
  int newrfp, rp, len, f_start, status;
  int i593_rfp, stat_ptr;
  u_char c;
  int sanity;

#ifdef WAVELAN_DEBUG
  if (wavelan_debug > 3)
    printk("wavelan_cs: entered wv_packet_rcv()\n");
#endif

  /* Get the new receive frame pointer from the i82593 chip */
  outb(CR0_STATUS_2 | OP0_NOP, LCCR(base));
  i593_rfp = inb(LCSR(base));
  i593_rfp |= inb(LCSR(base)) << 8;
  i593_rfp %= RX_SIZE;

  /* Get the new receive frame pointer from the WaveLAN card.
   * It is 3 bytes more than the increment of the i82593 receive
   * frame pointer, for each packet. This is because it includes the
   * 3 roaming bytes added by the mmc.
   */
  newrfp = inb(RPLL(base));
  newrfp |= inb(RPLH(base)) << 8;
  newrfp %= RX_SIZE;

/*  printk("wavelan_cs: i593_rfp %d stop %d newrfp %d lp->rfp %d\n",
	 i593_rfp, lp->stop, newrfp, lp->rfp); */

  if (lp->overrunning || newrfp == lp->rfp)
    printk("wavelan_cs: odd RFPs:  i593_rfp %d stop %d newrfp %d lp->rfp %d\n",
	   i593_rfp, lp->stop, newrfp, lp->rfp);

  while(newrfp != lp->rfp) {
    rp = newrfp;
    /* Find the first frame by skipping backwards over the frames */
    sanity = 0;
    while ((f_start = wv_start_of_frame(dev,rp)) != lp->rfp){
      if(f_start == -1 || sanity++ > 200){
	printk("wavelan_cs: cannot find start of frame ");
	printk(" i593_rfp %d stop %d newrfp %d lp->rfp %d\n",
	       i593_rfp, lp->stop, newrfp, lp->rfp);
	return;
      }
      rp = f_start;
    }

    stat_ptr = (rp - 7 + RX_SIZE) % RX_SIZE;
    stat_ptr = read_ringbuf(dev, stat_ptr, &c, 1);
    status = c;
    stat_ptr = read_ringbuf(dev, stat_ptr, &c, 1);
    status |= c << 8;
    stat_ptr = read_ringbuf(dev, stat_ptr, &c, 1);
    len = c;
    stat_ptr = read_ringbuf(dev, stat_ptr, &c, 1);
    len |= c << 8;

    if(!(status & RX_RCV_OK)) {
      if(status & RX_NO_SFD) lp->stats.rx_frame_errors++;
      if(status & RX_CRC_ERR) lp->stats.rx_crc_errors++;
      if(status & RX_OVRRUN) lp->stats.rx_over_errors++;

      printk("wavelan_cs: packet not received ok, status = 0x%x\n", status);
      lp->stats.rx_errors++;
    } else {
      wv_packet_read(dev, f_start, len - 2);
    }
    lp->rfp = rp;
  }

  /*
   * Update the frame stop register, but set it to less than
   * the full 8K to allow space for 3 bytes of signal strength
   * per packet.
   */
  lp->stop = (i593_rfp + RX_SIZE - ((RX_SIZE / 64) * 3)) % RX_SIZE;
  outb(OP0_SWIT_TO_PORT_1 | CR0_CHNL, LCCR(base));
  outb(CR1_STOP_REG_UPDATE | (lp->stop >> RX_SIZE_SHIFT), LCCR(base));
  outb(OP1_SWIT_TO_PORT_0, LCCR(base));
} /* wv_packet_rcv */

/*======================================================================

  Set or clear the multicast filter for this adaptor.
  num_addrs == -1	Promiscuous mode, receive all packets
  num_addrs == 0	Normal mode, clear multicast list
  num_addrs > 0		Multicast mode, receive normal and MC packets,
			and do best-effort filtering.
			
  As a side effect this routine must also reinitialize the card. This
  is done using the hardware reset code, which requires the interrupt
  code. This can only be done after interrupts have been enabled (as
  indicated by the link->open flag).
    
======================================================================*/

#ifdef NEW_MULTICAST
static void wv_set_multicast_list(device *dev)
{
  net_local *lp;
  dev_link_t *link;

  for (link = dev_list; link; link = link->next)
    if (link->priv == dev) break;
  if (!DEV_OK(link))
    return;
  lp = (net_local *)dev->priv;
  
#ifdef WAVELAN_DEBUG
  if (wavelan_debug > 1)
    printk("%s: setting Rx mode to %d addresses (link->open %ld).\n",
	   dev->name, dev->mc_count, link->open);
#endif

  if(dev->flags&IFF_PROMISC)
  {
    /*
     * Enable promiscuous mode: receive all packets.
     */
    if (!lp->promiscuous || !lp->multicast) {
      lp->promiscuous = lp->multicast = 1;
      if (link->open)	/* Only reset hw if we've installed int handler */
        wv_hw_reset(dev);
    }
  }
  else if((dev->flags&IFF_ALLMULTI)||dev->mc_list) 
  {
  }
  else
  {
    /*
     * Switch to normal mode: disable promiscuous mode and 
     * clear the multicast list.
     */
    if (lp->promiscuous || lp->multicast) {
      lp->promiscuous = lp->multicast = 0;
      if (link->open)	/* Only reset hw if we've installed int handler */
        wv_hw_reset(dev);
    }
  }
} /* wv_set_multicast_list */
#else
static void wv_set_multicast_list(device *dev, int num_addrs, void *addrs)
{
  net_local *lp;
  dev_link_t *link;

  for (link = dev_list; link; link = link->next)
    if (link->priv == dev) break;
  if (!DEV_OK(link))
    return;
  lp = (net_local *)dev->priv;
  
#ifdef WAVELAN_DEBUG
  if (wavelan_debug > 1)
    printk("%s: setting Rx mode to %d addresses (link->open %ld).\n",
	   dev->name, dev->mc_count, link->open);
#endif

  if (num_addrs < 0)
  {
    /*
     * Enable promiscuous mode: receive all packets.
     */
    if (!lp->promiscuous || !lp->multicast) {
      lp->promiscuous = lp->multicast = 1;
      if (link->open)	/* Only reset hw if we've installed int handler */
        wv_hw_reset(dev);
    }
  }
  else if (num_addrs > 0)
  {
  }
  else
  {
    /*
     * Switch to normal mode: disable promiscuous mode and 
     * clear the multicast list.
     */
    if (lp->promiscuous || lp->multicast) {
      lp->promiscuous = lp->multicast = 0;
      if (link->open)	/* Only reset hw if we've installed int handler */
        wv_hw_reset(dev);
    }
  }
} /* wv_set_multicast_list */
#endif

/*======================================================================

  File operation function for opening WaveLAN information device.

======================================================================*/

static int wavelan_open(struct inode *inode, struct file *file)
{
#ifdef PCMCIA_DEBUG
    if (pc_debug)
    	printk("wavelan_open()\n");
#endif
    /* Only allow read access */
    if ((file->f_flags & O_ACCMODE) != O_RDONLY) {
    	return(-EACCES);
    } 
    else
    	return(0);
} /* wavelan_open */

/*======================================================================

  File operation function for reading from WaveLAN information device.

======================================================================*/

static int wavelan_read(struct inode *inode, struct file *file, char *buf,
		   int count)
{
  char *page;
  int length;
  int end;

#ifdef PCMCIA_DEBUG
  if (pc_debug)
    printk("wavelan_read()\n");
#endif
    
  if (count < 0) return(-EINVAL);
  if (!(page = (char*) __get_free_page(GFP_KERNEL))) return(-ENOMEM);

  length = wv_get_info(page);
  if (file->f_pos >= length) {
    free_page((unsigned long) page);
    return(0);
  }
  if (count + file->f_pos > length)
    count = length - file->f_pos;
  end = count + file->f_pos;
  memcpy_tofs(buf, page + file->f_pos, count);
  free_page((unsigned long) page);
  file->f_pos = end;
  return(count);
} /* wavelan_read */

#ifdef 0
/*======================================================================

  Dumps the current i82593 receive buffer to the console.

======================================================================*/

static void wavelan_dump(device *dev)
{
  u_short base = dev->base_addr;
  int i, c;

  /* disable receiver so we can use channel 1 */
  outb(OP0_RCV_DISABLE, LCCR(base));

  /* reset receive DMA pointer */
  hacr_write_slow(base, HACR_PWR_STAT | HACR_RX_DMA_RESET);
  hacr_write(base, HACR_DEFAULT);

  /* dump into receive buffer */
  wv_82593_cmd(dev, "wavelan_dump(): dump", CR0_CHNL|OP0_DUMP, SR0_DUMP_DONE);

  /* set read pointer to start of receive buffer */
  outb(0, PIORL(base));
  outb(0, PIORH(base));

  printk("wavelan_cs: dump:\n");
  printk("     00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F");
  for(i = 0; i < 73; i++){
    if((i % 16) == 0) {
      printk("\n0x%02x:", i);
      if (!i) {
	printk("   ");
	continue;
      }
    }
    c = inb(PIOP(base));
    printk("%02x ", c);
  }
  printk("\n");

  /* enable the receiver again */
  wv_ru_start(dev);
}
#endif

/*======================================================================

  We never need to do anything when a wavelan device is "initialized"
  by the net software, because we only register already-found cards.
    
======================================================================*/

static int wv_init(struct device *dev)
{
    return(0);
} /* wavelan_init */

/*======================================================================

  Configure and start up the WaveLAN PCMCIA adaptor.
    
======================================================================*/

static int wv_start(struct device *dev)
{
    dev_link_t *link;
    int i, r;
    unsigned long x;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
    	printk("wv_start('%s')\n", dev->name);
#endif

    for (link = dev_list; link; link = link->next)
    	if (link->priv == dev) break;
    if (!DEV_OK(link))
    	return(-ENODEV);
    
    link->open++;
    MOD_INC_USE_COUNT;
    
    irq2dev_map[dev->irq] = dev;

    dev->tbusy = 0;   

    x = wv_splhi();
    if ((r = wv_hw_reset(dev)) != -1)
    {
	dev->interrupt = 0;
	dev->start = 1;
    }
    wv_splx(x);

    if (r == -1) 
    {
#ifdef SA_SHIRQ
    	free_irq(dev->irq,NULL);
#else
    	free_irq(dev->irq);
#endif
    	irq2dev_map[dev->irq] = (device *)0;
    	return(-EAGAIN);
    }

#ifdef WAVELAN_DEBUG
    if (wavelan_debug > 1)
    	wv_mmc_show(dev);
#endif

    /* Set up character device for user mode clients */
    i = register_chrdev(0, "wavelan", &wv_fops);
    if (i == -EBUSY) {
    	printk("wavelan_cs: unable to find a free device #\n");
    } 
    else
    	major_dev = i;
    
    return(0);
} /* wv_start */

/*======================================================================

  Shutdown the WaveLAN PCMCIA adaptor.
    
======================================================================*/

static int wv_stop(struct device *dev)
{
    dev_link_t *link;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
    	printk("wv_stop('%s')\n", dev->name);
#endif

    for (link = dev_list; link; link = link->next)
    	if (link->priv == dev) break;
    if (link == NULL)
    	return(-ENODEV);

    /* If the device isn't open, then nothing to do */
    if (!link->open)
	return 0;
    
    dev->tbusy = 1;

    /*
     * We always physically use the IRQ line, so we don't do free_irq().
     * We do remove ourselves from the map.
     */
    irq2dev_map[dev->irq] = 0;

    link->open--;
    dev->start = 0;
    if (link->state & DEV_STALE_CONFIG)
     	wavelan_release((u_long)link);

    MOD_DEC_USE_COUNT;
    
    return(0);
} /* wv_stop */


/*======================================================================

    wavelan_attach() creates an "instance" of the driver, allocating
    local data structures for one device.  The device is registered
    with Card Services.

    The dev_link structure is initialized, but we don't actually
    configure the card at this point -- we wait until we receive a
    card insertion event.
    
======================================================================*/

static dev_link_t *wavelan_attach(void)
{
    client_reg_t client_reg;
    dev_link_t *link;
    struct device *dev;
    net_local *lp;
    int ret;
    
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("wavelan_attach()\n");
#endif

    /* Initialize the dev_link_t structure */
    link = kmalloc(sizeof(struct dev_link_t), GFP_KERNEL);
    memset(link, 0, sizeof(struct dev_link_t));
    link->release.function = &wavelan_release;
    link->release.data = (u_long)link;

    /* The io structure describes IO port mapping */
    link->io.NumPorts1 = 8;
    link->io.Attributes1 = IO_DATA_PATH_WIDTH_8;
    link->io.IOAddrLines = 3;

    /* Interrupt setup */
    link->irq.Attributes = IRQ_TYPE_EXCLUSIVE | IRQ_HANDLE_PRESENT;
    link->irq.IRQInfo1 = IRQ_INFO2_VALID|IRQ_LEVEL_ID;
    link->irq.IRQInfo2 = irq_mask;
    link->irq.Handler = wavelan_interrupt;
    
    /* General socket configuration */
    link->conf.Attributes = CONF_ENABLE_IRQ;
    link->conf.Vcc = 50;
    link->conf.IntType = INT_MEMORY_AND_IO;

    dev = kmalloc(sizeof(struct device), GFP_KERNEL);
    memset(dev, 0, sizeof(struct device));

    /* Make up a wavelan-specific data structure. */
    dev->priv = (net_local *)kmalloc(sizeof(net_local), GFP_KERNEL);
    memset(dev->priv, 0x00, sizeof(net_local));

    /* ???? */
    lp = dev->priv;
    lp->cmd_wait = FALSE;
    lp->status = FALSE;
    lp->snrrcvctr = 0;

    /* wavelan specific entries */
    dev->hard_start_xmit = &wv_packet_xmit;
    dev->get_stats = &wv_get_stats;
    dev->set_multicast_list = &wv_set_multicast_list;
    ether_setup(dev);
    dev->name = ((net_local *)dev->priv)->node.dev_name;
    dev->init = &wv_init;
    dev->open = &wv_start;
    dev->stop = &wv_stop;
    dev->tbusy = 1;
    link->priv = dev;

    /* Register with Card Services */
    link->next = dev_list;
    dev_list = link;
    client_reg.dev_info = &dev_info;
    client_reg.Attributes = INFO_IO_CLIENT | INFO_CARD_SHARE;
    client_reg.EventMask = 
	CS_EVENT_REGISTRATION_COMPLETE |
	CS_EVENT_CARD_INSERTION | CS_EVENT_CARD_REMOVAL |
	CS_EVENT_RESET_PHYSICAL | CS_EVENT_CARD_RESET |
	CS_EVENT_PM_SUSPEND | CS_EVENT_PM_RESUME;
    client_reg.event_handler = &wavelan_event;
    client_reg.Version = 0x0210;
    client_reg.event_callback_args.client_data = link;
    ret = CardServices(RegisterClient, &link->handle, &client_reg);
    if (ret != 0) {
	cs_error(RegisterClient, ret);
	wavelan_detach(link);
	return NULL;
    }

    return link;
} /* wavelan_attach */

/*======================================================================

    This deletes a driver "instance".  The device is de-registered
    with Card Services.  If it has been released, all local data
    structures are freed.  Otherwise, the structures will be freed
    when the device is released.

======================================================================*/

static void wavelan_detach(dev_link_t *link)
{
    dev_link_t **linkp;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("wavelan_detach(0x%p)\n", link);
#endif
    
    /* Locate device structure */
    for (linkp = &dev_list; *linkp; linkp = &(*linkp)->next)
	if (*linkp == link) break;
    if (*linkp == NULL)
	return;

    /*
       If the device is currently configured and active, we won't
       actually delete it yet.  Instead, it is marked so that when
       the release() function is called, that will trigger a proper
       detach().
    */
    if (link->state & DEV_CONFIG) {
#ifdef PCMCIA_DEBUG
	printk("wavelan_cs: detach postponed, '%s' still locked\n",
	       link->dev->dev_name);
#endif
    	wavelan_release((u_long)link);
    	if (link->state & DEV_STALE_CONFIG) {
      	    link->state |= DEV_STALE_LINK;
      	    return;
      	}
    }

    /* Break the link with Card Services */
    if (link->handle)
	CardServices(DeregisterClient, link->handle);
    
    /* Unlink device structure, free pieces */
    *linkp = link->next;
    if (link->priv) {
    	struct device *dev = link->priv;
    	if (dev->priv)
            kfree_s(dev->priv, sizeof(net_local));
    	kfree_s(link->priv, sizeof(struct device));
    }
    kfree_s(link, sizeof(struct dev_link_t));
    
} /* wavelan_detach */

/*======================================================================

    wavelan_config() is scheduled to run after a CARD_INSERTION event
    is received, to configure the PCMCIA socket, and to make the
    ethernet device available to the system.
    
======================================================================*/

static void wavelan_config(dev_link_t *link)
{
    client_handle_t handle;
    tuple_t tuple;
    cisparse_t parse;
    struct device *dev;
    int i, j;
    u_char buf[64];
    win_req_t req;
    memreq_t mem;
    short base;
    net_local *lp;
    
    handle = link->handle;
    dev = link->priv;
    lp = (net_local *)dev->priv;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("wavelan_config(0x%p)\n", link);
#endif

    /*
       This reads the card's CONFIG tuple to find its configuration
       registers.
    */
    do {
	tuple.Attributes = 0;
	tuple.DesiredTuple = CISTPL_CONFIG;
	i = CardServices(GetFirstTuple, handle, &tuple);
	if (i != CS_SUCCESS) break;
	tuple.TupleData = (cisdata_t *)buf;
	tuple.TupleDataMax = 64;
	tuple.TupleOffset = 0;
	i = CardServices(GetTupleData, handle, &tuple);
	if (i != CS_SUCCESS) break;
	i = CardServices(ParseTuple, handle, &tuple, &parse);
	if (i != CS_SUCCESS) break;
	link->conf.ConfigBase = parse.config.base;
	link->conf.Present = parse.config.rmask[0];
    } while (0);
    if (i != CS_SUCCESS) {
	cs_error(ParseTuple, i);
	link->state &= ~DEV_CONFIG_PENDING;
	return;
    }
    
    /* Configure card */
    link->state |= DEV_CONFIG;

    do {
	
	i = CardServices(RequestIO, link->handle, &link->io);
	if (i != CS_SUCCESS) {
	    cs_error(RequestIO, i);
	    break;
	}
	
	/*
	   Now allocate an interrupt line.  Note that this does not
	   actually assign a handler to the interrupt.
	*/
	i = CardServices(RequestIRQ, link->handle, &link->irq);
	if (i != CS_SUCCESS) {
	    cs_error(RequestIRQ, i);
	    break;
	}
	
	/*
	   This actually configures the PCMCIA socket -- setting up
	   the I/O windows and the interrupt mapping.
	*/
	link->conf.ConfigIndex = 1;
	i = CardServices(RequestConfiguration, link->handle, &link->conf);
	if (i != CS_SUCCESS) {
	    cs_error(RequestConfiguration, i);
	    break;
	}

	/*
	   Allocate a 4K memory window.  Note that the dev_link_t
	   structure provides space for one window handle -- if your
	   device needs several windows, you'll need to keep track of
	   the handles in your private data structure, link->priv.
	*/
	req.Attributes = WIN_DATA_WIDTH_8|WIN_MEMORY_TYPE_AM|WIN_ENABLE;
	req.Base = NULL;
	req.Size = 0x1000;
	req.AccessSpeed = mem_speed;
	link->win = (window_handle_t)link->handle;
	i = CardServices(RequestWindow, &link->win, &req);
    	if (i != CS_SUCCESS) {
	    cs_error(RequestWindow, i);
	    break;
	}

      	dev->rmem_start = dev->mem_start = (u_long)req.Base;
      	dev->rmem_end = dev->mem_end = dev->mem_start + req.Size;

	mem.CardOffset = 0; mem.Page = 0;
	i = CardServices(MapMemPage, link->win, &mem);
    	if (i != CS_SUCCESS) {
	    cs_error(MapMemPage, i);
	    break;
	}
#ifdef WAVELAN_DEBUG
    	if (wavelan_debug > 1) {
      	    printk("wavelan_config: dev->memstart 0x%x\n", (u_int) dev->mem_start);
            printk("wavelan_config: before request_irq\n");
    	}
#endif
    	dev->irq = link->irq.AssignedIRQ;
    	dev->base_addr = link->io.BasePort1;
    	dev->tbusy = 0;
#ifdef WAVELAN_DEBUG
    	if (wavelan_debug > 1)
      	    printk("wavelan_config: IRQ %d IOPORT 0x%x (before register_netdev)\n",
	     	dev->irq, (unsigned int)dev->base_addr);
#endif
    	i = register_netdev(dev);
    	if (i != 0) {
      	    printk("wavelan_cs: register_netdev() failed\n");
      	    break;
    	}
    } while (0);

    link->state &= ~DEV_CONFIG_PENDING;
    /* If any step failed, release any partially configured state */
    if (i != 0) {
	wavelan_release((u_long)link);
	return;
    }

    link->dev = &((net_local *)dev->priv)->node;
    
    /* WaveLAN specific configuration */
#if STRUCT_CHECK == 1
    if (wv_structuct_check() != (char *)0) {
    	printk("%s: structure/compiler botch: \"%s\"\n", dev->name,
	     wv_structuct_check());
    	return;
    }
#endif	/* STRUCT_CHECK == 1 */

    wv_sw_reset(dev);
    psa_read(dev, &(lp->psa));

    /*
     * Check the first three octets of the MAC addr for the manufacturer's code.
     *  (Note: For non-AT&T GIS PCMCIA cards, see wavelan.h)
     */ 
    if ((lp->psa.psa_univ_mac_addr[0] != SA_ADDR0) ||
        (lp->psa.psa_univ_mac_addr[1] != SA_ADDR1) ||
        (lp->psa.psa_univ_mac_addr[2] != SA_ADDR2))
    	return;

    lp->nwid[0] = lp->psa.psa_nwid[0];
    lp->nwid[1] = lp->psa.psa_nwid[1];
    base = link->io.BasePort1;
    printk("%s: WaveLAN at %#x,", dev->name, base);
    memcpy(&dev->dev_addr[0], &lp->psa.psa_univ_mac_addr[0], WAVELAN_ADDR_SIZE);
    for (i = 0; i < WAVELAN_ADDR_SIZE; i++)
    	printk("%s%02x", (i == 0) ? " " : ":", dev->dev_addr[i]);
    printk(", IRQ %d", dev->irq);
    printk(", nwid 0x%02x-%02x", lp->nwid[0], lp->nwid[1]);
    printk(", PC");
    switch (lp->psa.psa_comp_number) {
    case PSA_COMP_PC_AT_915:
    case PSA_COMP_PC_AT_2400:
    	printk("-AT");
    	break;
    case PSA_COMP_PC_MC_915:
    case PSA_COMP_PC_MC_2400:
    	printk("-MC");
    	break;
    case PSA_COMP_PCMCIA_915:
    	printk("MCIA");
     	break;
    default:
    	printk("???");
    }
    printk(", ");
    switch (lp->psa.psa_subband) {
    case PSA_SUBBAND_915:
    	printk("915");
      	break;
    case PSA_SUBBAND_2425:
    	printk("2425");
    	break;
    case PSA_SUBBAND_2460:
    	printk("2460");
    	break;
    case PSA_SUBBAND_2484:
    	printk("2484");
    	break;
    case PSA_SUBBAND_2430_5:
    	printk("2430.5");
    	break;
    default:
    	printk("???");
    }
    printk(" MHz (%s)\n", 
	 lp->psa.psa_conf_status == 0x01 ? "configured" : "unconfigured");

    printk(version);
} /* wavelan_config */

/*======================================================================

  After a card is removed, wavelan_release() will unregister the net
  device, and release the PCMCIA configuration.  If the device is
  still open, this will be postponed until it is closed.
    
======================================================================*/

static void wavelan_release(u_long arg)
{
    dev_link_t *link = (dev_link_t *)arg;
    struct device *dev = link->priv;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk("wavelan_release(0x%p)\n", link);
#endif

    /*
       If the device is currently in use, we won't release until it
       is actually closed.
    */
    if (link->open) {
	if (pc_debug)
	    printk("wavelan_cs: release postponed, '%s' still open\n",
		   link->dev->dev_name);
	link->state |= DEV_STALE_CONFIG;
	return;
    }

    /* Unlink the device chain */
    if (link->dev != '\0')
    	unregister_netdev(dev);
    link->dev = NULL;
    
    /* Don't bother checking to see if these succeed or not */
    CardServices(ReleaseWindow, link->win);
    CardServices(ReleaseConfiguration, link->handle);
    CardServices(ReleaseIO, link->handle, &link->io);
    CardServices(ReleaseIRQ, link->handle, &link->irq);
    if (dev->irq != 0)
    	irq2dev_map[dev->irq] = NULL;

    link->state &= ~DEV_CONFIG;
    if (link->state & DEV_STALE_LINK)
	wavelan_detach(link);
    
} /* wavelan_release */

/*======================================================================

    The card status event handler.  Mostly, this schedules other
    stuff to run after an event is received.  A CARD_REMOVAL event
    also sets some flags to discourage the net drivers from trying
    to talk to the card any more.
    
======================================================================*/

static int wavelan_event(event_t event, int priority,
			 event_callback_args_t *args)
{
    dev_link_t *link = args->client_data;
    struct device *dev = link->priv;

#ifdef PCMCIA_DEBUG
    if (pc_debug)
    	printk("wavelan_event(): %s\n",
	   ((event == CS_EVENT_REGISTRATION_COMPLETE)?"registration complete" :
	    ((event == CS_EVENT_CARD_REMOVAL) ? "card removal" :
	     ((event == CS_EVENT_CARD_INSERTION) ? "card insertion" :
	      ((event == CS_EVENT_PM_SUSPEND) ? "pm suspend" :
	       ((event == CS_EVENT_RESET_PHYSICAL) ? "physical reset" :
		((event == CS_EVENT_PM_RESUME) ? "pm resume" :
		 ((event == CS_EVENT_CARD_RESET) ? "card reset" :
		  "unknown"))))))));
#endif
    
    switch (event) {
    case CS_EVENT_REGISTRATION_COMPLETE:
#ifdef PCMCIA_DEBUG
	if (pc_debug)
	    printk("wavelan_cs: registration complete\n");
#endif
	break;
    case CS_EVENT_CARD_REMOVAL:
	link->state &= ~DEV_PRESENT;
	if (link->state & DEV_CONFIG) {
      	    dev->tbusy = 1; dev->start = 0;
            link->release.expires = RUN_AT(5);
      	    add_timer(&link->release);
	}
	break;
    case CS_EVENT_CARD_INSERTION:
	link->state |= DEV_PRESENT | DEV_CONFIG_PENDING;
	wavelan_config(link);
	break;
    case CS_EVENT_PM_SUSPEND:
	link->state |= DEV_SUSPEND;
	/* Fall through... */
    case CS_EVENT_RESET_PHYSICAL:
    	if (link->state & DEV_CONFIG) {
      	    if (link->open) {
		dev->tbusy = 1; dev->start = 0;
      	    }
      	    CardServices(ReleaseConfiguration, link->handle);
    	}
	break;
    case CS_EVENT_PM_RESUME:
	link->state &= ~DEV_SUSPEND;
	/* Fall through... */
    case CS_EVENT_CARD_RESET:
	if (link->state & DEV_CONFIG) {
      	    CardServices(RequestConfiguration, link->handle, &link->conf);
      	    if (link->open) {
		wv_hw_reset(dev);
		dev->tbusy = 0; dev->start = 1;
      	    }
	}
	break;
    }
    return 0;
} /* wavelan_event */

/*======================================================================

    This function is the interrupt handler for the WaveLAN card. This
    routine will be called whenever: 
  	1. A packet is received.
	2. A packet has successfully been transfered and the unit is
	   ready to transmit another packet.
	3. A command has completed execution.
    
======================================================================*/

static void wv_interrupt(device *dev)
{
    net_local *lp = (net_local *)dev->priv;
    u_short base = dev->base_addr;
    int status0;
    u_int tx_status;
	
    if (dev->interrupt)
    	printk("%s: Re-entering the interrupt handler.\n", dev->name);
    dev->interrupt = 1;
#ifdef WAVELAN_DEBUG
    if (wavelan_debug > 3) printk("wavelan_cs: interrupt received\n");
#endif

    while (1) {
    	outb(CR0_STATUS_0 | OP0_NOP, LCCR(base));
    	status0 = inb(LCSR(base));
#ifdef WAVELAN_DEBUG
        if (wavelan_debug > 3) {
	    printk("status0 0x%x [%s => 0x%x]", status0, 
	       (status0&SR0_INTERRUPT)?"int":"no int",status0&~SR0_INTERRUPT);
	    if (status0&SR0_INTERRUPT) {
	  	printk(" [%s => %d]\n", (status0 & SR0_CHNL) ? "chnl" :
		     ((status0 & SR0_EXECUTION) ? "cmd" :
		      ((status0 & SR0_RECEPTION) ? "recv" : "unknown")),
		     (status0 & SR0_EVENT_MASK));
	    } 
	    else
	  	printk("\n");
      	}
#endif
    	/* Return if no actual interrupt from i82593 */
    	if(!(status0 & SR0_INTERRUPT)) {
      	    dev->interrupt = 0;
      	    return;
    	}
	/* This should tell us if the card isn't really there */
	if (!dev->start ||
	    (status0 & SR0_EVENT_MASK == SR0_DIAGNOSE_FAILED)) {
#ifdef WAVELAN_DEBUG
	    if (wavelan_debug > 1)
		printk("%s: interrupt from dead card\n", dev->name);
#endif
	    dev->interrupt = 0;
	    return;
	}
    	lp->status = status0;	/* Save current status (for commands) */

	/* ================ RECEIVING PACKET =====================*/
    	if (status0 & SR0_RECEPTION) {
#ifdef WAVELAN_DEBUG
      	    if (wavelan_debug > 3) printk("wavelan_cs: rcv\n");
#endif
      	    if((status0 & SR0_EVENT_MASK) == SR0_STOP_REG_HIT) {
	    	printk("wavelan_cs: receive buffer overflow\n");
	    	lp->stats.rx_over_errors++;
	    	lp->overrunning = 1;
      	    }
      	    wv_packet_rcv(dev);
      	    lp->overrunning = 0;

      	    if (status0 & SR0_EXECUTION)
	    	printk("wavelan_cs: interrupt is both rx and tx, status0 = %x\n",
	       		status0);


      	    outb(CR0_INT_ACK | OP0_NOP, LCCR(base));	/* Acknowledge the interrupt */
      	    continue;
    	} 
    	else if (!(status0 & SR0_EXECUTION)) {
      	    printk("wavelan_cs: interrupt is neither rx or tx, status0 = %x\n",
	     		status0);

      	    outb(CR0_INT_ACK | OP0_NOP, LCCR(base));	/* Acknowledge the interrupt */
      	    break;
    	}
    	if (lp->cmd_wait) {
#ifdef WAVELAN_DEBUG
      	    if (wavelan_debug > 3) printk("wavelan_cs: cmd completion\n");
#endif
      	    /* We are waiting for a command to complete */
      	    lp->cmd_wait = FALSE;			/* Signal command completion */
      	    outb(CR0_INT_ACK | OP0_NOP, LCCR(base));	/* Acknowledge the interrupt */
      	    continue;
    	}
    	if((status0 & SR0_EVENT_MASK) == SR0_TRANSMIT_DONE ||
           (status0 & SR0_EVENT_MASK) == SR0_RETRANSMIT_DONE) {
#ifdef WAVELAN_DEBUG
      	    if (wavelan_debug > 3) printk("wavelan_cs: xmit done\n");
#endif
      	    tx_status = inb(LCSR(base));
      	    tx_status |= (inb(LCSR(base)) << 8);
#ifdef WAVELAN_DEBUG
      	    if (wavelan_debug > 3) {
		u_int rcv_bytes;
		u_char status3;
		rcv_bytes = inb(LCSR(base));
		rcv_bytes |= (inb(LCSR(base)) << 8);
		status3 = inb(LCSR(base));
		printk("tx_status 0x%02x rcv_bytes 0x%02x status3 0x%x\n",
	       		tx_status, rcv_bytes, (u_int) status3);
      	    }
#endif
      	    if (!(tx_status & TX_OK)) {
		if (tx_status & TX_FRTL) {
	  	    if (xmt_watch) 
	  	        printk("wavelan_cs: frame too long\n");
	  	    lp->stats.tx_errors++;
		}
	    if (tx_status & TX_UND_RUN) {
	  	if (xmt_watch) 
	  	    printk("wavelan_csd: DMA underrun\n");
	  	lp->stats.tx_errors++;
	  	lp->stats.tx_aborted_errors++;
	    }
	    if (tx_status & TX_LOST_CTS) {
	  	if (xmt_watch) 
	  	    printk("wavelan: no CTS\n");
	  	lp->stats.tx_errors++;
	  	lp->stats.tx_carrier_errors++;
	    }
	    if (tx_status & TX_LOST_CRS) {
	  	/* Ignore no carrier messages, they always happen! */
#ifndef IGNORE_NORMAL_XMIT_ERRS
	  	if (xmt_watch) 
	  	    printk("wavelan_cs: no carrier\n");
	      	lp->stats.tx_errors++;
	  	lp->stats.tx_carrier_errors++;
#endif IGNORE_NORMAL_XMIT_ERRS
	    }
	    /* Ignore late collisions since they're more likely to happen
	     * here (the WaveLAN design prevents the LAN controller from
	     * receiving while it is transmitting). We take action only when
	     * the maximum retransmit attempts is exceeded.
	     */
	    if (tx_status & TX_COLL) {
	  	if (tx_status & TX_MAX_COL) {
	    	    if (xmt_watch) 
	    	    	printk("wavelan_cs: channel congestion\n");
	    	    if (!(tx_status & TX_NCOL_MASK)) {
	      		lp->stats.collisions += 0x10;
	    	    }
	    	    lp->stats.tx_errors++;
	     	}
	  	lp->stats.collisions++;
	    }
      	}
        if(tx_status & TX_DEFER) {
#ifndef IGNORE_NORMAL_XMIT_ERRS
	    if (xmt_watch) 
	    	printk("wavelan_cs: channel jammed\n");
#endif IGNORE_NORMAL_XMIT_ERRS
/*		lp->stats.collisions++; */
      	    }
      	    if (tx_status & TX_HRT_BEAT) {
 		if (xmt_watch) 
 		    printk("wavelan_cs: heart beat\n");
		lp->stats.tx_heartbeat_errors++;
      	    }
      	    lp->stats.collisions += (tx_status & TX_NCOL_MASK);
      	    lp->stats.tx_packets++;
       	    dev->tbusy = FALSE;
      	    mark_bh(NET_BH);
      	    outb(CR0_INT_ACK | OP0_NOP, LCCR(base));	/* Acknowledge the interrupt */
    	} 
    	else {
      	    printk("wavelan_cs: unknown interrupt, status0 = %02x\n", status0);
      	    outb(CR0_INT_ACK | OP0_NOP, LCCR(base));	/* Acknowledge the interrupt */
    	}
    }
    dev->interrupt = FALSE;
#if 0
    dev->tbusy = FALSE;
#endif
} /* wv_interrupt */

static void wavelan_interrupt IRQ(int irq, void *dev_id, struct pt_regs *regs)
{
    device *dev;
	
    if ((dev = (device *)(irq2dev_map[irq])) != (device *)0)
    	wv_interrupt(dev);
    else
    	printk("wavelan_interrupt(): irq %d for unknown device.\n", irq);
} /* wavelan_interrupt */

/*====================================================================*/

int init_wavelan_cs(void)
{
    servinfo_t serv;
#ifdef PCMCIA_DEBUG
    if (pc_debug)
	printk(version);
#endif
    CardServices(GetCardServicesInfo, &serv);
    if (serv.Revision != CS_RELEASE_CODE) {
	printk("wavelan_cs: Card Services release does not match!\n");
	return -1;
    }
    register_pcmcia_driver(&dev_info, &wavelan_attach, &wavelan_detach);
    return 0;
}

#ifdef MODULE
void cleanup_module(void)
{
    printk("wavelan_cs: unloading\n");
    if (major_dev != -1) 
    	unregister_chrdev(major_dev, "wavelan");
    unregister_pcmcia_driver(&dev_info);
    while (dev_list != NULL)
	wavelan_detach(dev_list);
}
#endif /* MODULE */

