/*
 * This module is all original code 
 * by Rob Nation 
 * Copyright 1993, Robert Nation
 *     You may use this code for any purpose, as long as the original
 *     copyright remains in the source code and all documentation
 ****************************************************************************/

/***********************************************************************
 *
 * code for moving windows
 *
 ***********************************************************************/
#include "mwm.h"
#include <X11/keysym.h>

extern XEvent   Event;
extern int      menuFromFrameOrWindowOrTitlebar;
Bool            NeedToResizeToo;

/****************************************************************************
 *
 * Start a window move operation
 *
 ****************************************************************************/
void
move_window(XEvent * eventp, Window w, MwmWindow * tmp_win, int context,
	    int val1, int val2, int val1_unit, int val2_unit)
{
    int             FinalX, FinalY;

    /* gotta have a window */
    if (tmp_win == NULL)
	return;

    w = tmp_win->frame;
    if (tmp_win->flags & ICONIFIED) {
	if (tmp_win->icon_pixmap_w != None) {
	    XUnmapWindow(dpy, tmp_win->icon_w);
	    w = tmp_win->icon_pixmap_w;
	}
	else
	    w = tmp_win->icon_w;
    }

    if ((val1 != 0) || (val2 != 0)) {
	FinalX = val1 * val1_unit / 100;
	FinalY = val2 * val2_unit / 100;
    }
    else
	InteractiveMove(&w, tmp_win, &FinalX, &FinalY, eventp);

    if (w == tmp_win->frame) {
	SetupFrame(tmp_win, FinalX, FinalY,
		   tmp_win->frame_width, tmp_win->frame_height, FALSE);
    }
    else {			/* icon window */
	tmp_win->flags |= ICON_MOVED;
	tmp_win->icon_x_loc = FinalX;
	tmp_win->icon_xl_loc = FinalX -
	    (tmp_win->icon_w_width - tmp_win->icon_p_width) / 2;
	tmp_win->icon_y_loc = FinalY;
	XMoveWindow(dpy, tmp_win->icon_w,
		    tmp_win->icon_xl_loc, FinalY + tmp_win->icon_p_height);
	if (tmp_win->icon_pixmap_w != None) {
	    XMapWindow(dpy, tmp_win->icon_w);
	    XMoveWindow(dpy, tmp_win->icon_pixmap_w, tmp_win->icon_x_loc, FinalY);
	    XMapWindow(dpy, w);
	}

    }

    RedrawPager();

    return;
}



/****************************************************************************
 *
 * Move the rubberband around, return with the new window location
 *
 ****************************************************************************/
void
moveLoop(MwmWindow * tmp_win, int XOffset, int YOffset, int Width,
	 int Height, int *FinalX, int *FinalY, Bool opaque_move,
	 Bool AddWindow)
{
    Bool            finished = False;
    Bool            done;
    int             xl, yt, delta_x, delta_y;
    unsigned int    pagerwidth, pagerheight;
    int             ww, wh;
    int             wx, wy;
    int             MaxH, MaxW;
    int             last_x = -10000, last_y = -10000;

    XQueryPointer(dpy, Scr.root_win, &JunkRoot, &JunkChild, &xl, &yt,
		  &JunkX, &JunkY, &JunkMask);
    xl += XOffset;
    yt += YOffset;

    if (!opaque_move && AddWindow)
	MoveOutline(Scr.root_win, xl, yt, Width, Height);

    DisplayPosition(tmp_win, xl + Scr.Vx, yt + Scr.Vy, True);

    while (!finished) {
	/* block until there is an interesting event */
	XMaskEvent(dpy, ButtonPressMask | ButtonReleaseMask | KeyPressMask |
	       PointerMotionMask | ButtonMotionMask | ExposureMask, &Event);
	StashEventTime(&Event);

	/* discard any extra motion events before a logical release */
	if (Event.type == MotionNotify) {
	    while (XCheckMaskEvent(dpy, PointerMotionMask | ButtonMotionMask |
				 ButtonPressMask | ButtonRelease, &Event)) {
		StashEventTime(&Event);
		if (Event.type == ButtonRelease)
		    break;
	    }
	}

	done = FALSE;
	/* Handle a limited number of key press events to allow mouseless
	 * operation */
	if (Event.type == KeyPress)
	    Keyboard_shortcuts(&Event, ButtonRelease);
	switch (Event.type) {
	case KeyPress:
	    done = TRUE;
	    break;
	case ButtonPress:
	    XAllowEvents(dpy, ReplayPointer, CurrentTime);
	    if (Event.xbutton.button == 1 && Event.xbutton.state & ShiftMask) {
		NeedToResizeToo = True;
		/* Fallthrough to button-release */
	    }
	    else {
		done = 1;
		break;
	    }
	case ButtonRelease:
	    if (!opaque_move)
		MoveOutline(Scr.root_win, 0, 0, 0, 0);
	    xl = Event.xmotion.x_root + XOffset;
	    yt = Event.xmotion.y_root + YOffset;

	    /* Resist moving windows over the edge of the screen! */
	    if (((xl + Width) >= Scr.d_width) &&
		((xl + Width) < Scr.d_width + Scr.MoveResistance))
		xl = Scr.d_width - Width - tmp_win->bw;
	    if ((xl <= 0) && (xl > -Scr.MoveResistance))
		xl = 0;
	    if (((yt + Height) >= Scr.d_height) &&
		((yt + Height) < Scr.d_height + Scr.MoveResistance))
		yt = Scr.d_height - Height - tmp_win->bw;
	    if ((yt <= 0) && (yt > -Scr.MoveResistance))
		yt = 0;

	    *FinalX = xl;
	    *FinalY = yt;

	    done = TRUE;
	    finished = TRUE;
	    break;

	case MotionNotify:
	    /* update location of the pager_view window */
	    if ((Scr.MwmPager != NULL) &&
		(xl < Scr.MwmPager->frame_x + Scr.MwmPager->frame_width) &&
		(xl + Width > Scr.MwmPager->frame_x) &&
		(yt < Scr.MwmPager->frame_y + Scr.MwmPager->frame_height) &&
		(yt + Height > Scr.MwmPager->frame_y) && (!opaque_move))
		MoveOutline(Scr.root_win, 0, 0, 0, 0);
	    xl = Event.xmotion.x_root;
	    yt = Event.xmotion.y_root;
	    HandlePaging(Scr.d_width, Scr.d_height, &xl, &yt,
			 &delta_x, &delta_y, False);
	    /* redraw the rubberband */
	    xl += XOffset;
	    yt += YOffset;

	    /* Resist moving windows over the edge of the screen! */
	    if (((xl + Width) >= Scr.d_width) &&
		((xl + Width) < Scr.d_width + Scr.MoveResistance))
		xl = Scr.d_width - Width - tmp_win->bw;
	    if ((xl <= 0) && (xl > -Scr.MoveResistance))
		xl = 0;
	    if (((yt + Height) >= Scr.d_height) &&
		((yt + Height) < Scr.d_height + Scr.MoveResistance))
		yt = Scr.d_height - Height - tmp_win->bw;
	    if ((yt <= 0) && (yt > -Scr.MoveResistance))
		yt = 0;

	    if (Scr.MwmPager) {
		pagerwidth = Scr.MwmPager->frame_width -
		    2 * Scr.MwmPager->boundary_width;
		pagerheight = Scr.MwmPager->frame_height -
		    Scr.MwmPager->title_height - 2 * Scr.MwmPager->boundary_width;

		MaxW = Scr.VxMax + Scr.d_width;
		MaxH = Scr.VyMax + Scr.d_height;

		if (!(tmp_win->flags & STICKY) &&
		    (!(tmp_win->flags & ICONIFIED) ||
		     (!(tmp_win->flags & SUPPRESSICON))) &&
		    (!(tmp_win->flags & ICONIFIED) || (!(Scr.flags & StickyIcons)))) {
		    /* show the actual window */
		    wx = (xl + Scr.Vx) * (int) pagerwidth / MaxW;
		    wy = (yt + Scr.Vy) * (int) pagerheight / MaxH;
		    if ((last_x - wx >= 2) || (last_x - wx <= -2) ||
			(last_y - wy >= 2) || (last_y - wy <= -2)) {
			ww = Width * (int) pagerwidth / MaxW;
			wh = Height * (int) pagerheight / MaxH;
			if (ww < 2)
			    ww = 2;
			if (wh < 2)
			    wh = 2;
			XMoveResizeWindow(dpy,
				       tmp_win->pager_view, wx, wy, ww, wh);
			last_x = wx;
			last_y = wy;
		    }
		}
	    }

	    if (!opaque_move)
		MoveOutline(Scr.root_win, xl, yt, Width, Height);
	    else {
		if (tmp_win->flags & ICONIFIED) {
		    tmp_win->icon_x_loc = xl;
		    tmp_win->icon_xl_loc = xl -
			(tmp_win->icon_w_width - tmp_win->icon_p_width) / 2;
		    tmp_win->icon_y_loc = yt;
		    if (tmp_win->icon_pixmap_w != None)
			XMoveWindow(dpy, tmp_win->icon_pixmap_w,
				    tmp_win->icon_x_loc, yt);
		    else if (tmp_win->icon_w != None)
			XMoveWindow(dpy, tmp_win->icon_w, tmp_win->icon_xl_loc,
				    yt + tmp_win->icon_p_height);

		}
		else
		    XMoveWindow(dpy, tmp_win->frame, xl, yt);
	    }
	    DisplayPosition(tmp_win, xl + Scr.Vx, yt + Scr.Vy, False);
	    done = TRUE;
	    break;

	default:
	    break;
	}
	if (!done) {
	    if (!opaque_move)
		MoveOutline(Scr.root_win, 0, 0, 0, 0);
	    DispatchEvent();
	    if (!opaque_move)
		MoveOutline(Scr.root_win, xl, yt, Width, Height);
	}
    }
}

/***********************************************************************
 *
 *  Procedure:
 *      DisplayPosition - display the position in the dimensions window
 *
 *  Inputs:
 *      tmp_win - the current mwm window
 *      x, y    - position of the window
 *
 ************************************************************************/

void
DisplayPosition(MwmWindow * tmp_win, int x, int y, int Init)
{
    char            str[100];
    int             offset;

    (void) sprintf(str, " %+-4d %+-4d ", x, y);
    if (Init) {
	XSetForeground(dpy, Scr.top_GC,
		 Mwm.components[MWM_FEEDBACK].top_shadow_color);
	XSetForeground(dpy, Scr.bot_GC,
		 Mwm.components[MWM_FEEDBACK].bottom_shadow_color);
	XClearWindow(dpy, Scr.size_win);
	if (Scr.d_depth >= 2)
	    RelieveWindow(tmp_win, Scr.size_win, 0, 0,
			  Scr.SizeStringWidth + SIZE_HINDENT * 2,
			  Scr.StdFont.height + SIZE_VINDENT * 2,
			  Scr.top_GC, Scr.bot_GC, FULL_HILITE);
    }
    else {
	XClearArea(dpy, Scr.size_win, SIZE_HINDENT, SIZE_VINDENT, Scr.SizeStringWidth,
		   Scr.StdFont.height, False);
    }

    offset = (Scr.SizeStringWidth + SIZE_HINDENT * 2
	      - XTextWidth(Scr.StdFont.font, str, strlen(str))) / 2;
    XDrawString(dpy, Scr.size_win, Scr.feedback_GC,
		offset,
		Scr.StdFont.font->ascent + SIZE_VINDENT,
		str, strlen(str));
}


/****************************************************************************
 *
 * For menus, move, and resize operations, we can effect keyboard 
 * shortcuts by warping the pointer.
 *
 ****************************************************************************/
void
Keyboard_shortcuts(XEvent * Event, int ReturnEvent)
{
    int             x, y, x_root, y_root;
    int             move_size, x_move, y_move;
    KeySym          keysym;

    /* Pick the size of the cursor movement */
    move_size = Scr.EntryHeight;
    if (Event->xkey.state & ControlMask)
	move_size = 1;
    if (Event->xkey.state & ShiftMask)
	move_size = 100;

    keysym = XLookupKeysym(&Event->xkey, 0);

    x_move = 0;
    y_move = 0;
    switch (keysym) {
    case XK_Up:
    case XK_k:
    case XK_p:
	y_move = -move_size;
	break;
    case XK_Down:
    case XK_n:
    case XK_j:
	y_move = move_size;
	break;
    case XK_Left:
    case XK_b:
    case XK_h:
	x_move = -move_size;
	break;
    case XK_Right:
    case XK_f:
    case XK_l:
	x_move = move_size;
	break;
    case XK_Return:
    case XK_space:
	/* beat up the event */
	Event->type = ReturnEvent;
	break;
    default:
	break;
    }
    XQueryPointer(dpy, Scr.root_win, &JunkRoot, &Event->xany.window,
		  &x_root, &y_root, &x, &y, &JunkMask);

    if ((x_move != 0) || (y_move != 0)) {
	/* beat up the event */
	XWarpPointer(dpy, None, Scr.root_win, 0, 0, 0, 0, x_root + x_move,
		     y_root + y_move);

	/* beat up the event */
	Event->type = MotionNotify;
	Event->xkey.x += x_move;
	Event->xkey.y += y_move;
	Event->xkey.x_root += x_move;
	Event->xkey.y_root += y_move;
    }
}


void
InteractiveMove(Window * win, MwmWindow * tmp_win, int *FinalX, int *FinalY, XEvent * eventp)
{
    extern int      Stashed_X, Stashed_Y;
    int             origDragX, origDragY, DragX, DragY, DragWidth, DragHeight;
    int             XOffset, YOffset;
    Window          w;
    Bool            opaque_move = False;
    extern Bool     pagerOn;

    InstallRootColormap();
    if (menuFromFrameOrWindowOrTitlebar) {
	/* warp the pointer to the cursor position from before menu appeared */
	XWarpPointer(dpy, None, Scr.root_win, 0, 0, 0, 0, Stashed_X, Stashed_Y);
	XFlush(dpy);
    }


    DragX = eventp->xbutton.x_root;
    DragY = eventp->xbutton.y_root;
    /* If this is left commented out, then the move starts from the button press 
     * location instead of the current location, which seems to be an
     * improvement */
    /*  XQueryPointer(dpy, Scr.root_win, &JunkRoot, &JunkChild,
       &DragX, &DragY,    &JunkX, &JunkY, &JunkMask); */

    if (!GrabEm(MOVE_CURS)) {
	XBell(dpy, Scr.screen);
	return;
    }

    pagerOn = False;

    w = tmp_win->frame;

    if (tmp_win->flags & ICONIFIED)
	if (tmp_win->icon_pixmap_w != None) {
	    XUnmapWindow(dpy, tmp_win->icon_w);
	    w = tmp_win->icon_pixmap_w;
	}
	else
	    w = tmp_win->icon_w;

    *win = w;

    XGetGeometry(dpy, w, &JunkRoot, &origDragX, &origDragY,
		 (unsigned int *) &DragWidth, (unsigned int *) &DragHeight,
		 &JunkBW, &JunkDepth);

    if (DragWidth * DragHeight <
	(Scr.OpaqueSize * Scr.d_width * Scr.d_height) / 100)
	opaque_move = True;
    else
	XGrabServer(dpy);

    if ((!opaque_move) && (tmp_win->flags & ICONIFIED))
	XUnmapWindow(dpy, w);

    DragWidth += JunkBW;
    DragHeight += JunkBW;
    XOffset = origDragX - DragX;
    YOffset = origDragY - DragY;
    XMapRaised(dpy, Scr.size_win);
    moveLoop(tmp_win, XOffset, YOffset, DragWidth, DragHeight, FinalX, FinalY,
	     opaque_move, False);

    XUnmapWindow(dpy, Scr.size_win);
    UninstallRootColormap();

    if (!opaque_move)
	XUngrabServer(dpy);
    UngrabEm();


    pagerOn = True;
}
