/**
 *
 * $Id: GeoUtils.c,v 1.18 1996/04/23 17:40:22 miers Exp $
 *
 * Copyright (C) 1995 Free Software Foundation, Inc.
 *
 * This file is part of the GNU LessTif Library.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **/

static char rcsid[] = "$Id: GeoUtils.c,v 1.18 1996/04/23 17:40:22 miers Exp $";

#include <Xm/XmP.h>
#include <Xm/DebugUtil.h>
#include <Xm/SeparatorP.h>
#include <Xm/SeparatoGP.h>
#include <stdio.h>

#ifdef GEO_DEBUG

static void
dump_box(XmKidGeometry box)
{
    if (!box)
	return;
    while (box->kid) {
	printf("    KID %s REQ %08x X %-5d Y %-5d W %-5d H %-5d B %-5d\n",
		XtName(box->kid), box->box.request_mode, box->box.x, box->box.y,
		box->box.width, box->box.height, box->box.border_width);
	box++;
    }
}

static void
dump_layout(XmGeoRowLayout rows, XmKidGeometry boxes)
{
    if (!rows)
	return;
    --rows;
    do {
	rows++;
	printf("ROW: %08x\n", (unsigned int)rows);
	printf("  end: %d fixup: %08x even_width: %d even_height: %d\n",
		rows->end, (unsigned int)rows->fix_up,
		rows->even_width, rows->even_height);
	printf("  min_height: %d stretch_height: %d uniform_border %d\n",
		rows->min_height, rows->stretch_height, rows->uniform_border);
	printf("  border: %d fill_mode: %d fit_mode: %d sticky_end: %d\n",
		rows->border, rows->fill_mode, rows->fit_mode, rows->sticky_end);
	printf("  space_above: %d space_end: %d space_between: %d\n",
		rows->space_above, rows->space_end, rows->space_between);
	printf("  max_box_height: %d boxes_width: %d fill_width %d\n",
		rows->max_box_height, rows->boxes_width, rows->fill_width);
	printf("  box_count: %d\n", rows->box_count);
	dump_box(boxes);
	boxes += rows->box_count + 1;
    } while (!rows->end);
}

static void
dump_matrix(XmGeoMatrix geo) {
    printf("MATRIX: composite: %08x instigator %08x boxes %08x\n",
	(unsigned int)geo->composite, (unsigned int)geo->instigator,
	(unsigned int)geo->boxes);
    printf("  layouts: %08x margin_w: %d margin_h: %d stretch_boxes: %d\n",
	(unsigned int)geo->layouts, geo->margin_w, geo->margin_h,
	geo->stretch_boxes);
    printf("  uniform_border: %d border: %d max_major: %d boxed_minor: %d\n",
	geo->uniform_border, geo->border, geo->max_major, geo->boxed_minor);
    printf("  fill_minor: %d width: %d height: %d set: %08x\n",
	geo->fill_minor, geo->width, geo->height, (unsigned int)geo->set_except);
    printf("  almost: %08x no_geo: %08x extension %08x destruct: %08x\n",
	(unsigned int)geo->almost_except, (unsigned int)geo->no_geo_request,
	(unsigned int)geo->extension, (unsigned int)geo->ext_destructor);
    printf("  arrange: %08x major: %d\n",
	(unsigned int)geo->arrange_boxes, geo->major_order);
    dump_layout((XmGeoRowLayout)geo->layouts, geo->boxes);
}
#endif

void
FitBoxesProportional(XmKidGeometry boxes, Dimension box_count,
		     Dimension box_wd, short fill_wd)
{
    int tmp, curx;

    XdbDebug(__FILE__, NULL,
	     "FitBoxesProportional: box_count: %d box_wd: %d fill_wd: %d\n",
	     box_count, box_wd, fill_wd);

    if (box_wd > box_count) {
	curx = 0;

	while (boxes->kid) {
	    tmp  = boxes->box.width + boxes->box.border_width * 2;
	    tmp *= fill_wd;
	    tmp /= box_wd;

	    if (tmp > boxes->box.width)
		boxes->box.width = 1;
	    else
		boxes->box.width -= tmp;

	    boxes->box.x += curx;
	    curx -= tmp;
	    boxes++;
	}
    }
    else {
	tmp = -fill_wd;
	if (tmp > box_count)
	    tmp /= box_count;
	else
	    tmp = 1;
	curx = 0;
	while (boxes->kid) {
	    boxes->box.width = tmp;
	    boxes->box.x = curx;
	    curx += tmp;
	    boxes++;
	}
    }
}

void
FitBoxesAveraging(XmKidGeometry boxes, Dimension box_count,
		  Dimension box_wd, Dimension fill_wd)
{
    Widget *tbox;
    int i;

    XdbDebug(__FILE__, NULL,
	     "FitBoxesAveraging: box_count: %d box_wd: %d fill_wd: %d\n",
	     box_count, box_wd, fill_wd);

#if 0
    tbox = (Widget)XtMalloc(box_count * sizeof(int));
    for (i = 0; i < box_count; i++) {
#endif
}

void
SegmentFill(XmKidGeometry boxes, int boxcount, XmGeoRowLayout layout,
	    Position x, Dimension width, Dimension margin,
	    Position curx, Position end, Position start,
	    Dimension space_between)
{
    XmKidGeometry kend;
    Widget tkid;

    kend = boxes + boxcount;
    tkid = boxes->kid;
    boxes->kid = NULL;

    /* stuff */

#if 0
    if () {
	_XmGeoCalcFill();
    }
    else {
	FitBoxesProportional();
    }
#endif
    boxes->kid = tkid;
}

Dimension
_XmGeoLayoutSimple(XmKidGeometry boxes, XmGeoRowLayout layout,
		   Position x, Position y,
		   Dimension end, Dimension start, Dimension space_between)
{
    Dimension bw, bh;

    XdbDebug(__FILE__, NULL, "_XmGeoLayoutSimple: x,y %d %d\n", x, y);

    x += start;

    while (boxes->kid) {
	bh = boxes->box.height + boxes->box.border_width * 2;
	boxes->box.x = x;
	if (bh == layout->max_box_height)
	    boxes->box.y = y;
	else
	    boxes->box.y = y + (layout->max_box_height - bh) / 2;

	x += boxes->box.width + (boxes->box.border_width * 2) + space_between;

	boxes++;
    }
    if (layout->sticky_end) {
	boxes--;
	bw = boxes->box.border_width * 2 + boxes->box.width;
	end -= bw;
	if (end < boxes->box.x)
	    boxes->box.x = end;
    }

    return y + layout->max_box_height;
}

Dimension
_XmGeoLayoutWrap(XmKidGeometry boxes, XmGeoRowLayout layout,
		 Position x, Position y,
		 Dimension start, Dimension space_between, Dimension end,
		 Dimension width, Dimension margin)
{
    Dimension curx, hincr, boxcount;
    XmKidGeometry curbox;

    XdbDebug(__FILE__, NULL, "_XmGeoLayoutWrap\n");

    end += margin;
    curx = x + start;
    curbox = boxes;
    boxcount = 0;

    while (curbox->kid) {
	hincr = curbox->box.width + curbox->box.border_width * 2;
	if (curx + hincr > end && boxcount) {
	    SegmentFill(boxes, boxcount, layout,
			x, width, margin, curx, end, start, space_between);
	    boxes = curbox;
	    y += layout->max_box_height;
	    hincr = curbox->box.width + curbox->box.border_width * 2;
	    curx = x + start;
	    boxcount = 0;
	}
	if (curx + hincr > end) {
	    if (curx + hincr - end < curbox->box.width &&
		curx + hincr - end != 0)
		curbox->box.width -= curx + hincr - end;
	    else
		curbox->box.width = boxcount;
	    curbox++;
	    continue;
	}
	curx += hincr;
	curbox->box.x = curx;
	curbox->box.height += 2 * curbox->box.border_width;
	if (curbox->box.height == layout->max_box_height)
	    boxes->box.y = y;
	else
	    boxes->box.y += (layout->max_box_height - curbox->box.height) / 2;

	curx += space_between;
	curbox++;
	boxcount++;
    }

    if (boxcount)
	SegmentFill(boxes, boxcount, layout,
		    x, width, margin, curx, end, start, space_between);

    if (layout->sticky_end) {
	curbox--;
	curx = boxes->box.border_width * 2 + boxes->box.width;
	end -= curx;
	if (end < curbox->box.x)
	    curbox->box.x = end;
    }

    return y;
}

void
_XmGeoCalcFill(Dimension width, Dimension margin, Dimension box_count,
	       Dimension space_end, Dimension space_between,
	       Dimension *start, Dimension *tween)
{
    Dimension tmp, tmp2;

    XdbDebug(__FILE__, NULL, "_XmGeoCalcFill\n: %d %d %d %d %d %d %d",
	     width, margin, box_count, space_end, space_between,
	     *start, *tween);

    if (space_end == 0) {
	if (margin != 1) {
	    if (space_between == 0)
		space_between = box_count - 1;
	}
	space_end = 1;
    }
    tmp = space_between;
    space_between = space_end;
    tmp2 = tmp * (box_count - 1);
    tmp2 += space_between * 2;
    space_end = tmp2;

    *start = (width * space_between) / space_end;
    if (*start < margin) {
	space_between *= 2;
	if (space_between < space_end) {
	    space_end -= space_between;
	}
	margin *= 2;
	if (margin < width) {
	    width -= margin;
	    *start = margin / 2;
	}
    }
    *tween = (tmp * width) / space_end;
}

/*
 * called from arrange boxes
 */
Position
_XmGeoArrangeList(XmKidGeometry boxes, XmGeoRowLayout layout,
		  Position x, Position y,
		  Dimension width, Dimension margin)
{
    Dimension start, box_wd, end, tween;
    short fill_wd, adjust;

    XdbDebug(__FILE__, NULL, "_XmGeoArrangeList: x: %d y: %d wd: %d\n",
	     x, y, width);

    box_wd = layout->boxes_width;
    fill_wd = layout->boxes_width + layout->fill_width + 2 * margin;
    adjust = fill_wd - width;

    if (layout->space_above > margin)
	start = layout->space_above;
    else
	start = margin;

    end = width + x - margin;

    tween = layout->space_between;

    XdbDebug(__FILE__, NULL, "_XmGeoArrangeList: fill_wd: %d width: %d\n",
	     fill_wd, width);

    if (fill_wd > width && layout->fit_mode == XmGEO_WRAP) {
	    return _XmGeoLayoutWrap(boxes, layout, x, y, 
				    start, tween, end,
				    width, margin);
    }
    else if (fill_wd > width) {
	if (layout->fit_mode == XmGEO_AVERAGING) {
	    FitBoxesAveraging(boxes, layout->box_count, box_wd, adjust);
	}
	else {
	    FitBoxesProportional(boxes, layout->box_count, box_wd, adjust);
	}
    }
    else if (fill_wd != width) {
	if (layout->fill_mode == XmGEO_CENTER) {
	    fill_wd = layout->fill_width + margin * 2 + width - fill_wd;
	    _XmGeoCalcFill(fill_wd,
			   margin, layout->box_count,
			   layout->space_end, tween,
			   &start, &tween);
	}
	else {
	    FitBoxesProportional(boxes, layout->box_count, box_wd, adjust);
	}
    }
    return _XmGeoLayoutSimple(boxes, layout,
			      x, y, end, start, tween);
}

/*
 * also called from arrange boxes
 * WARNING: this function is not 100% correct
 */
Dimension
_XmGeoStretchVertical(XmGeoMatrix geoSpec, Dimension height, Dimension maxh)
{
    register XmGeoRowLayout layout;
    register XmKidGeometry boxPtr;
    int stretch = 0;
    short fill = 0, fill_used;

    XdbDebug(__FILE__, NULL, "_XmGeoStretchVertical\n");

    layout = &(geoSpec->layouts->row);

    if (maxh - height < 0) {

	fill = 0;

	while (!layout->end) {

	    if (layout->stretch_height &&
		layout->min_height > layout->max_box_height) {

		fill += layout->max_box_height - layout->min_height;
		stretch++;
	    }

	    layout++;
	}

	if (maxh + fill < height)
		fill /= -stretch;
	else
		fill = 0;
    }
    else {

	fill = 0;

	while (!layout->end) {
	    if (layout->stretch_height)
		stretch++;
	    layout++;
	}
	fill = (int)maxh - (int)height;
	fill /= stretch;
    }

    layout = &(geoSpec->layouts->row);
    boxPtr = geoSpec->boxes;

    fill_used = 0;
    while (!layout->end) {
	if (layout->stretch_height) {
	
	    while (boxPtr->kid) {
		boxPtr->box.y += fill_used;
		boxPtr->box.height += fill;
		boxPtr++;
	    }

	    boxPtr++;
	    fill_used += fill;
	}
	else
	    boxPtr += layout->box_count + 1;

	layout++;
    }

    return height + fill_used;
}

/*
 * also called from arrange boxes
 * WARNING: this function is not 100% correct
 */
Dimension
_XmGeoFillVertical(XmGeoMatrix geoSpec, Dimension height, Dimension maxh)
{
    register XmGeoRowLayout layout;
    register XmKidGeometry boxPtr;
    int stretch = 0;
    short fill = 0, fill_used;

    XdbDebug(__FILE__, NULL, "_XmGeoFillVertical: ht: %d maxh: %d\n",
		height, maxh);

    layout = &(geoSpec->layouts->row);

    while (!layout->end) {
	stretch++;
	layout++;
    }

    fill = ((int)maxh - (int)height) / stretch;

    layout = &(geoSpec->layouts->row);
    boxPtr = geoSpec->boxes;

    fill_used = 0;
    while (!layout->end) {
	while (boxPtr->kid) {
	    boxPtr->box.y += fill_used;
	    boxPtr++;
	}

	boxPtr++;
	fill_used += fill;

	layout++;
    }

    return height + fill_used;
}
 
/*
 * called from ArrangeBoxes
 * calls _XmGeoBoxesSameWidth, _XmGeoBoxesSameHeight
 */
void 
_XmGeoAdjustBoxes(XmGeoMatrix geoSpec) 
{
    Dimension border;
    register XmKidGeometry boxPtr;
    register XmGeoRowLayout layoutPtr;

    XdbDebug(__FILE__, NULL, "_XmGeoAdjustBoxes\n");

    layoutPtr = &(geoSpec->layouts->row);
    boxPtr = geoSpec->boxes;

    while (!layoutPtr->end) {

	border = 0;

	if (layoutPtr->even_width != 0)
	    _XmGeoBoxesSameWidth(boxPtr, layoutPtr->even_width);

	if (layoutPtr->even_height != 0)
	    _XmGeoBoxesSameHeight(boxPtr, layoutPtr->even_height);

	if (geoSpec->uniform_border)
	    border = geoSpec->border;
	else if (layoutPtr->uniform_border)
	    border = layoutPtr->border;

	while (boxPtr->kid != NULL) {
	    boxPtr->box.border_width = border;
	    boxPtr++;
	}
	boxPtr++;

	layoutPtr++;
    }
}

/*
 * called from HandleChangeManaged (third)
 * calls AdjustBoxes
 */
void
_XmGeoArrangeBoxes(XmGeoMatrix geoSpec,
		   Position x, Position y,
		   Dimension *pW, Dimension *pH)
{
    XmGeoArrangeProc arrange;
    register XmGeoRowLayout layoutPtr;
    register XmKidGeometry boxPtr;
    Dimension wd;
    Position oy;

    XdbDebug(__FILE__, NULL, "_XmGeoArrangeBoxes x: %d y: %d\n", x, y);

    oy = y;

    arrange = geoSpec->arrange_boxes;
    if (arrange != NULL /* && arrange doesn't equal something weird */)
	(arrange)(geoSpec, x, y, pW, pH);

    _XmGeoAdjustBoxes(geoSpec);

    _XmGeoGetDimensions(geoSpec);

    layoutPtr = &(geoSpec->layouts->row);
    boxPtr = geoSpec->boxes;

    if (geoSpec->margin_h < layoutPtr->space_above)
	y += layoutPtr->space_above;
    else
	y += geoSpec->margin_h;

    wd = 2 * geoSpec->margin_w + geoSpec->max_major;
    if (*pW)
	wd = *pW;

    while (!layoutPtr->end) {

	y = _XmGeoArrangeList(boxPtr, layoutPtr, x, y, wd, geoSpec->margin_w);
	boxPtr += layoutPtr->box_count + 1;

	XdbDebug(__FILE__, NULL, "_XmGeoArrangeBoxes(2): y: %d\n", y);

	layoutPtr++;
	y += layoutPtr->space_above;
    }

    XdbDebug(__FILE__, NULL, "_XmGeoArrangeBoxes(3): y: %d\n", y);

    if (layoutPtr->space_above < geoSpec->margin_h)
	y += geoSpec->margin_h - layoutPtr->space_above;

    y -= oy;

    if (*pH != 0 && *pH != y) {
	if (geoSpec->stretch_boxes)
	    y = _XmGeoStretchVertical(geoSpec, y, *pH);
	else
	    y = _XmGeoFillVertical(geoSpec, y, *pH);
    }

    geoSpec->width = wd;
    if (*pW < wd)
	*pW = wd;

    geoSpec->height = y;
    if (*pH < y)
	*pH = y;
}

/*
 * called from AdjustBoxes
 */
Dimension 
_XmGeoBoxesSameWidth(XmKidGeometry rowPtr,
		     Dimension width) 
{
    Dimension wd = width;
    XmKidGeometry boxes = rowPtr;

    XdbDebug(__FILE__, NULL, "_XmGeoBoxesSameWidth: %d\n", width);

    if (width == 0)
	return 0;
    else if (width >= 1) {
	while (boxes->kid != NULL) {
	    if (boxes->box.width > wd)
		wd = boxes->box.width;
	    boxes++;
	} 
    }

    boxes = rowPtr;
    while (boxes->kid != NULL) {
	boxes->box.width = wd;
	boxes++;
    }
    return wd;
}

/*
 * also called from AdjustBoxes
 */
Dimension 
_XmGeoBoxesSameHeight(XmKidGeometry rowPtr,
		      Dimension height) 
{
    Dimension ht = height;
    XmKidGeometry boxes = rowPtr;

    XdbDebug(__FILE__, NULL, "_XmGeoBoxesSameHeight: %d\n", height);

    if (height == 0)
	return 0;
    else if (height >= 1) {
	while (boxes->kid != NULL) {
	    if (boxes->box.height > ht)
		ht = boxes->box.height;
	    boxes++;
	} 
    }
    boxes = rowPtr;
    while (boxes->kid != NULL) {
	boxes->box.height = ht;
	boxes++;
    }
    return ht;
}

void 
_XmGeoClearRectObjAreas(RectObj r,
                        XWindowChanges *old) 
{
    XdbDebug(__FILE__, NULL, "_XmGeoClearRectObjAreas\n");
}

/*
 * called from XmRCGetKidGeo
 */
int 
_XmGeoCount_kids(CompositeWidget c) 
{
    XdbDebug(__FILE__, NULL, "_XmGeoCount_kids\n");

    return 0;
}

/*
 * called from ArrangeBoxes (second)
 */
void 
_XmGeoGetDimensions(XmGeoMatrix geoSpec) 
{
    register XmKidGeometry boxPtr;
    register XmGeoRowLayout layoutPtr;
    Dimension curw = 0, curh = 0, fillh = 0, tmpw, tmph, roww, rowh;
    int bcount;

    XdbDebug(__FILE__, NULL, "_XmGeoGetDimensions\n");

    layoutPtr = &(geoSpec->layouts->row);
    boxPtr = geoSpec->boxes;

    if (layoutPtr->space_above > geoSpec->margin_h)
	fillh = layoutPtr->space_above - geoSpec->margin_h;

    geoSpec->stretch_boxes = False;

    while (!layoutPtr->end) {

	bcount = 0;
	roww = rowh = 0;
	while (boxPtr->kid) {

	    tmph = boxPtr->box.border_width * 2 + boxPtr->box.height;
	    tmpw = boxPtr->box.border_width * 2 + boxPtr->box.width;

	    if (tmph > rowh)
		rowh = tmph;
	    roww += tmpw;
	    boxPtr++;
	    bcount++;
	}

	layoutPtr->box_count = bcount;
	layoutPtr->boxes_width = roww;
	layoutPtr->max_box_height = rowh;

	if (layoutPtr->border != 0) {
	    if (layoutPtr->fit_mode != XmGEO_WRAP) {
		layoutPtr->stretch_height = 0;
		geoSpec->stretch_boxes = True;
	    }
	}

	if (layoutPtr->space_end > geoSpec->margin_w)
	    tmpw = (layoutPtr->space_end - geoSpec->margin_w) * 2;
	else
	    tmpw = 0;
	tmpw += (bcount - 1) * layoutPtr->space_between;

	layoutPtr->fill_width = tmpw;

	roww += tmpw;

	if (curw < roww)
	    curw = roww;

	layoutPtr++;
	curh += rowh;
	if (!layoutPtr->end)
	    fillh += layoutPtr->space_above;
	boxPtr++;
    }

    if (layoutPtr->space_above > geoSpec->margin_h)
	fillh -= layoutPtr->space_above - geoSpec->margin_h;

    geoSpec->max_major = curw;
    geoSpec->boxed_minor = curh;
    geoSpec->fill_minor = fillh;
}

XmKidGeometry 
_XmGetKidGeo(Widget wid,
	     Widget instigator,
	     XtWidgetGeometry *request,
	     int uniform_border,
	     Dimension border,
	     int uniform_width_margins,
	     int uniform_height_margins,
	     Widget help,
	     int geo_type) 
{
    XdbDebug(__FILE__, NULL, "_XmGetKidGeo\n");

    return NULL;
}

/*
 * called from GeoMatrixGet.  Called once for each kid in the Matrix.
 * also called from XmRCGetKidGeo
 */
void 
_XmGeoLoadValues(Widget wid,
		 int geoType,
		 Widget instigator,
		 XtWidgetGeometry *request,
		 XtWidgetGeometry *geoResult) 
{
    XtWidgetGeometry dummy;

    XdbDebug(__FILE__, NULL, "_XmGeoLoadValues\n");

    if (wid != instigator) {
	request = &dummy;
	if (geoType == XmGET_ACTUAL_SIZE || geoType != XmGET_PREFERRED_SIZE)
	    dummy.request_mode = 0;
	else
	    XtQueryGeometry(wid, NULL, request);
    }
    if (request->request_mode & CWX)
	geoResult->x = XtX(wid);
    if (request->request_mode & CWY)
	geoResult->y = XtY(wid);
    if (request->request_mode & CWWidth)
	geoResult->width = XtWidth(wid);
    if (request->request_mode & CWHeight)
	geoResult->height = XtHeight(wid);
    if (request->request_mode & CWBorderWidth)
	geoResult->border_width = wid->core.border_width;

    geoResult->request_mode = CWX|CWY|CWWidth|CWHeight|CWBorderWidth;

    XdbDebug(__FILE__, NULL,
	     "_XmGeoLoadValues: x: %d y: %d wd: %d ht: %d bdr: %d\n",
	     geoResult->x, geoResult->y, geoResult->width, geoResult->height,
	     geoResult->border_width);
}

/*
 * class method calls this to allocate the matrix record (first).
 * class method is called from BB HandleChangeManaged.
 */
XmGeoMatrix 
_XmGeoMatrixAlloc(unsigned int numRows,
		  unsigned int numBoxes,
		  unsigned int extSize) 
{
    XmGeoMatrix mat;

    XdbDebug(__FILE__, NULL, "XmGeoMatrixAlloc called with %d %d %d\n",
	     numRows, numBoxes, extSize);

    mat = (XmGeoMatrix)XtCalloc(1, sizeof(XmGeoMatrixRec));
    mat->extension = XtMalloc(extSize);
    mat->layouts = (XmGeoMajorLayout)XtCalloc(numRows + 1,
					      sizeof(XmGeoMajorLayoutRec));
    mat->boxes = (XmKidGeometry)XtCalloc(numRows + numBoxes + 1,
					 sizeof(XmKidGeometryRec));

    return mat;
}

/*
 * called last in HandleChangeManaged (fifth)
 */
void 
_XmGeoMatrixFree(XmGeoMatrix geo_spec) 
{
    XdbDebug(__FILE__, NULL, "_XmGeoMatrixFree\n");

    if (geo_spec->ext_destructor)
	(*geo_spec->ext_destructor)(geo_spec->extension);
    if (geo_spec->layouts)
	XtFree((char *)geo_spec->layouts);
    if (geo_spec->boxes)
	XtFree((char *)geo_spec->boxes);
    XtFree((char *)geo_spec);
}

/*
 * also called from HandleChangeManaged, after the matrix is created (second).
 * calls GeoLoadValues
 */
void 
_XmGeoMatrixGet(XmGeoMatrix geoSpec,
		int geoType) 
{
    register XmGeoRowLayout layoutPtr;
    register XmKidGeometry boxPtr;

    XdbDebug(__FILE__, NULL, "_XmGeoMatrixGet\n");

    layoutPtr = &(geoSpec->layouts->row);
    boxPtr = geoSpec->boxes;

    while (!layoutPtr->end) {
	if (boxPtr->kid == NULL) {
	    boxPtr++;
	    layoutPtr++;
	    continue;
	}
	_XmGeoLoadValues(boxPtr->kid, XmGET_PREFERRED_SIZE, geoSpec->instigator,
			 NULL, &boxPtr->box);
	boxPtr++;
    }
}

/*
 * called from HandleChangeManaged (fourth)
 */
void 
_XmGeoMatrixSet(XmGeoMatrix geoSpec) 
{
    register XmKidGeometry boxPtr;
    register XmGeoRowLayout layoutPtr;
    int bcount;
    
    XdbDebug(__FILE__, NULL, "_XmGeoMatrixSet\n");

    if (geoSpec->set_except != NULL) {
	if ((geoSpec->set_except)(geoSpec))
	    return;
    }

    boxPtr = geoSpec->boxes;
    layoutPtr = &(geoSpec->layouts->row);

    while (!layoutPtr->end) {

	if (layoutPtr->fix_up)
	    (layoutPtr->fix_up)(geoSpec, 3 /* MAGIC */,
				(XmGeoMajorLayout)layoutPtr, boxPtr);

	bcount = layoutPtr->box_count;
	layoutPtr++;
	boxPtr += bcount + 1;
    }

    boxPtr = geoSpec->boxes;
    layoutPtr = &(geoSpec->layouts->row);

    while (!layoutPtr->end) {

	_XmSetKidGeo(boxPtr, geoSpec->instigator);

	bcount = layoutPtr->box_count;
	layoutPtr++;
	boxPtr += bcount + 1;
    }

    boxPtr = geoSpec->boxes;
    layoutPtr = &(geoSpec->layouts->row);

    while (!layoutPtr->end) {

	if (layoutPtr->fix_up)
	    (layoutPtr->fix_up)(geoSpec, 3 /* MAGIC */,
				(XmGeoMajorLayout)layoutPtr, boxPtr);

	bcount = layoutPtr->box_count;
	layoutPtr++;
	boxPtr += bcount + 1;
    }
}

Boolean 
_XmGeoReplyYes(Widget wid,
	       XtWidgetGeometry *desired,
	       XtWidgetGeometry *response) 
{
    XdbDebug(__FILE__, NULL, "_XmGeoReplyYes\n");

    return False;
}

/*
 * Called from geoMatrixCreate, apparently to fill in init values for
 * KidGeo
 */
Boolean 
_XmGeoSetupKid(XmKidGeometry geo,
	       Widget kidWid) 
{
    XdbDebug(__FILE__, NULL, "_XmGeoSetupKid\n");

    if (!kidWid)
	return False;
    if (!XtIsRectObj(kidWid))
	return False;
    if (!XtIsManaged(kidWid))
	return False;

    geo->kid = kidWid;

    return True;
}

Boolean 
_XmGeometryEqual(Widget wid,
		 XtWidgetGeometry *geoA,
		 XtWidgetGeometry *geoB) 
{
    XdbDebug(__FILE__, NULL, "_XmGeometryEqual\n");

    return False;
}

/*
 * resize method??
 */
void 
_XmHandleSizeUpdate(Widget wid,
		    unsigned char policy,
		    XmGeoCreateProc createMatrix)
{
    XdbDebug(__FILE__, NULL, "_XmHandleSizeUpdate\n");
}

/*
 * query geom method??
 */
XtGeometryResult 
_XmHandleQueryGeometry(Widget wid,
		       XtWidgetGeometry *intended,
		       XtWidgetGeometry *desired,
		       unsigned char policy,
		       XmGeoCreateProc createMatrix) 
{
    XdbDebug(__FILE__, NULL, "_XmHandleQueryGeometry\n");

    return XtGeometryNo;
}

/*
 * parent geometry manager??
 */
XtGeometryResult 
_XmHandleGeometryManager(Widget wid,
			 Widget instigator,
			 XtWidgetGeometry *desired,
			 XtWidgetGeometry *allowed,
			 unsigned char policy,
			 XmGeoMatrix *cachePtr,
			 XmGeoCreateProc createMatrix) 
{
    XdbDebug(__FILE__, NULL, "_XmHandleGeometryManager\n");

    return XtGeometryNo;
}

/*
 * called from the child to the parent??
 */
XtGeometryResult 
_XmMakeGeometryRequest(Widget w,
		       XtWidgetGeometry *geom) 
{
    XdbDebug(__FILE__, NULL, "_XmMakeGeometryRequest\n");

    return XtGeometryNo;
}

/*
 * XmGeoSegmentFixUpProc
 */
void 
_XmMenuBarFix(XmGeoMatrix geoSpec,
	      int action,
	      XmGeoMajorLayout layoutPtr,
	      XmKidGeometry rowPtr) 
{
    XdbDebug(__FILE__, NULL, "_XmMenuBarFix\n");
}

/*
 * XmGeoSegmentFixUpProc
 */
void 
_XmSeparatorFix(XmGeoMatrix geoSpec,
		int action,
		XmGeoMajorLayout layoutPtr,
		XmKidGeometry rowPtr) 
{
    XdbDebug(__FILE__, NULL, "_XmSeparatorFix: action: %d\n", action);

    if (action == 3) {
	while (rowPtr->kid) {

	    if (XmIsSeparator(rowPtr->kid) && XtIsManaged(rowPtr->kid)) {
		if (SEP_Orientation(rowPtr->kid) == XmHORIZONTAL) {
		    rowPtr->box.x -= geoSpec->margin_w;
		    rowPtr->box.width += geoSpec->margin_w * 2;
		}
		else if (SEP_Orientation(rowPtr->kid) == XmVERTICAL) {
		    rowPtr->box.y -= geoSpec->margin_h;
		    rowPtr->box.height += geoSpec->margin_h * 2;
		}
	    }
	    else if (XmIsSeparatorGadget(rowPtr->kid) &&
		     XtIsManaged(rowPtr->kid)) {
		if (SEPG_Orientation(rowPtr->kid) == XmHORIZONTAL) {
		    rowPtr->box.x -= geoSpec->margin_w;
		    rowPtr->box.width += geoSpec->margin_w * 2;
		}
		else if (SEPG_Orientation(rowPtr->kid) == XmVERTICAL) {
		    rowPtr->box.y -= geoSpec->margin_h;
		    rowPtr->box.height += geoSpec->margin_h * 2;
		}
	    }

	    rowPtr++;
	}
    }
    else if (action == 2) {
	while (rowPtr->kid) {

	    if (XmIsSeparator(rowPtr->kid) && XtIsManaged(rowPtr->kid)) {
		if (SEP_Orientation(rowPtr->kid) == XmHORIZONTAL) {
		    rowPtr->box.x += geoSpec->margin_w;
		    rowPtr->box.width -= geoSpec->margin_w * 2;
		}
		else if (SEP_Orientation(rowPtr->kid) == XmVERTICAL) {
		    rowPtr->box.y += geoSpec->margin_h;
		    rowPtr->box.height -= geoSpec->margin_h * 2;
		}
	    }
	    else if (XmIsSeparatorGadget(rowPtr->kid) &&
		     XtIsManaged(rowPtr->kid)) {
		if (SEPG_Orientation(rowPtr->kid) == XmHORIZONTAL) {
		    rowPtr->box.x += geoSpec->margin_w;
		    rowPtr->box.width -= geoSpec->margin_w * 2;
		}
		else if (SEPG_Orientation(rowPtr->kid) == XmVERTICAL) {
		    rowPtr->box.y += geoSpec->margin_h;
		    rowPtr->box.height -= geoSpec->margin_h * 2;
		}
	    }

	    rowPtr++;
	}
    }
}

/* 
 * called from GeoMatrixSet
 * calls _XmConfigureObject
 */
void 
_XmSetKidGeo(XmKidGeometry kg,
	     Widget instigator) 
{
    XdbDebug(__FILE__, NULL, "_XmSetKidGeo: %s : x,y: %d %d w,h: %d %d b: %d\n",
		XtName(kg->kid),
		kg->box.x, kg->box.y, kg->box.width, kg->box.height,
		kg->box.border_width);

    while (kg->kid != NULL) {
	if (kg->kid == instigator) {
	    XtX(instigator) = kg->box.x;
	    XtY(instigator) = kg->box.y;
	    XtWidth(instigator) = kg->box.width;
	    XtHeight(instigator) = kg->box.height;
	    instigator->core.border_width = kg->box.border_width;
	}
	else
	    _XmConfigureObject(kg->kid,
			       kg->box.x, kg->box.y,
			       kg->box.width, kg->box.height,
			       kg->box.border_width);
	kg++;
    }
}

/*
 * internal functions I need to figure out
 */
void
_XmGMReplyToQueryGeometry()
{
    XdbDebug(__FILE__, NULL, "_XmGMReplyToQueryGeometry\n");
}

void
_XmGMEnforceMargin()
{
    XdbDebug(__FILE__, NULL, "_XmGMEnforceMargin\n");
}

void
_XmGMCalcSize(Widget w, Dimension margin_w, Dimension margin_h,
	      Dimension *retw, Dimension *reth)
{
    Widget child;
    int i;
    Dimension curx = 0, cury = 0;

    XdbDebug(__FILE__, NULL, "_XmGMCalcSize\n");

    *retw = *reth = 0;
    for (i = 0; i < MGR_NumChildren(w); i++) {
	child = MGR_Children(w)[i];
	if (!XtIsRectObj(child))
	    continue;
	if (!XtIsManaged(child))
	    continue;
	curx = child->core.x + child->core.width +
		2 * child->core.border_width;
	cury = child->core.y + child->core.height +
		2 * child->core.border_width;
	if (curx > *retw)
	    *retw = curx;
	if (cury > *reth)
	    *reth = cury;
    }

    *retw += MGR_ShadowThickness(w) + margin_w;
    *reth += MGR_ShadowThickness(w) + margin_h;

    if (!*retw)
	*retw = 10;
    if (!*reth)
	*reth = 10;
}

void
_XmGMDoLayout(Widget w, Dimension margin_w, Dimension margin_h,
	      unsigned char resize_policy, short adjust)
{
    unsigned char huh = 0x0c;
    Dimension retw, reth;

    XdbDebug(__FILE__, NULL, "_XmGMDoLayout\n");

    if (adjust)
	huh |= 0x80;

    _XmGMCalcSize(w, margin_w, margin_h, &retw, &reth);

    XtMakeResizeRequest(w, retw, reth, &retw, &reth);
}

void
_XmGMHandleGeometryManager()
{
    XdbDebug(__FILE__, NULL, "_XmGMHandleGeometryManager\n");
}

void
_XmGMHandleQueryGeometry()
{
    XdbDebug(__FILE__, NULL, "_XmGMHandleQueryGeometry\n");
}

Boolean
_XmGMOverlap()
{
    XdbDebug(__FILE__, NULL, "_XmGMOverlap\n");

    return False;
}

