/**
 *
 * $Id: List.c,v 1.31 1996/05/02 07:11:02 u27113 Exp $
 *
 * Copyright (C) 1995 Free Software Foundation, Inc.
 *
 * This file is part of the GNU LessTif Library.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **/

static char rcsid[] = "$Id: List.c,v 1.31 1996/05/02 07:11:02 u27113 Exp $";

#include <LTconfig.h>
#include <X11/Xfuncs.h>
#include <Xm/XmP.h>
#include <Xm/AtomMgr.h>
#include <Xm/DebugUtil.h>
#include <Xm/DragC.h>
#include <Xm/ListP.h>
#include <Xm/ScrollBar.h>
#include <Xm/ScrolledW.h>
#include <stdio.h>

/* Forward Declarations */

static void class_initialize();
static void class_part_initialize(WidgetClass widget_class);
static void destroy (Widget w);
static void resize (Widget w);
static void expose (Widget w, XEvent *event, Region region);
static void initialize (Widget request, Widget new, ArgList args, Cardinal *num_args);
static Boolean set_values (Widget current, Widget request, Widget new, ArgList args, Cardinal *num_args);
static XtGeometryResult query_geometry(Widget w, XtWidgetGeometry *proposed, XtWidgetGeometry *answer);

static void list_border_highlight(Widget w);
static void list_border_unhighlight(Widget w);

/* prototypes for drag-drop */
static Boolean drag_selected_proc(Widget w,
				  Atom *selection,
				  Atom *target,
				  Atom *type_return,
				  XtPointer *value_return,
				  unsigned long *length_return,
				  int *format_return);
static Boolean drag_unselected_proc(Widget w,
				    Atom *selection,
				    Atom *target,
				    Atom *type_return,
				    XtPointer *value_return,
				    unsigned long *length_return,
				    int *format_return);
static void drag_drop_finish(Widget w, 
			     XtPointer client_data, 
			     XtPointer call_data);

/*
 * Resources for the list class
 */
#define Offset(field) XtOffsetOf(XmListRec, list.field)
#define PrimOffset(field) XtOffsetOf(XmListRec, primitive.field)
static XtResource resources[] = {
    {
	XmNlistSpacing, XmCListSpacing, XmRVerticalDimension,
	sizeof(Dimension), Offset(ItemSpacing),
	XmRImmediate, (XtPointer)0
    },
    {
	XmNlistMarginWidth, XmCListMarginWidth, XmRHorizontalDimension,
	sizeof(Dimension), Offset(margin_width),
	XmRImmediate, (XtPointer)0
    },
    {
	XmNlistMarginHeight, XmCListMarginHeight, XmRVerticalDimension,
	sizeof(Dimension), Offset(margin_height),
	XmRImmediate, (XtPointer)0
    },
    {
	XmNfontList, XmCFontList, XmRFontList,
	sizeof(XmFontList), Offset(font),
	XmRImmediate, (XtPointer)NULL
    },
    {
	XmNstringDirection, XmCStringDirection, XmRStringDirection,
	sizeof(XmStringDirection), Offset(StrDir),
	XmRImmediate, (XtPointer)((XmStringDirection)XmUNSPECIFIED)
	/* this was STRING_L_TO_R */
    },
    {
	XmNitems, XmCItems, XmRXmStringTable,
	sizeof(XmStringTable), Offset(items),
	XmRStringTable, (XtPointer)NULL
    },
    {
	XmNitemCount, XmCItemCount, XmRInt,
	sizeof(int), Offset(itemCount),
	XmRImmediate, (XtPointer)0
    },
    {
	XmNselectedItems, XmCSelectedItems, XmRXmStringTable,
	sizeof(XmStringTable), Offset(selectedItems),
	XmRStringTable, (XtPointer)NULL
    },
    {
	XmNselectedItemCount, XmCSelectedItemCount, XmRInt,
	sizeof(int), Offset(selectedItemCount),
	XmRImmediate, (XtPointer)0
    },
    {
	XmNvisibleItemCount, XmCVisibleItemCount, XmRInt,
	sizeof(int), Offset(visibleItemCount),
	XmRImmediate, (XtPointer)0
	/* this is necessary to trigger the stuff in _XmListSetGeometry */
    },
    {
	XmNtopItemPosition, XmCTopItemPosition, XmRTopItemPosition,
	sizeof(int), Offset(top_position),
	XmRImmediate, (XtPointer)0
    },
    {
	XmNselectionPolicy, XmCSelectionPolicy, XmRSelectionPolicy,
	sizeof(unsigned char), Offset(SelectionPolicy),
	XmRImmediate, (XtPointer)XmBROWSE_SELECT
    },
    {
	XmNlistSizePolicy, XmCListSizePolicy, XmRListSizePolicy,
	sizeof(unsigned char), Offset(SizePolicy),
	XmRImmediate, (XtPointer)XmVARIABLE
    },
    {
	XmNscrollBarDisplayPolicy, XmCScrollBarDisplayPolicy, XmRScrollBarDisplayPolicy,
	sizeof(unsigned char), Offset(ScrollBarDisplayPolicy),
	XmRCallProc, (XtPointer)NULL
	/* this was XmRImmediate, (XtPointer)XmAS_NEEDED */
    },
    {
	XmNautomaticSelection, XmCAutomaticSelection, XmRBoolean,
	sizeof(Boolean), Offset(AutoSelect),
	XmRImmediate, (XtPointer)False
    },
    {
	XmNdoubleClickInterval, XmCDoubleClickInterval, XmRInt,
	sizeof(int), Offset(ClickInterval),
	XmRImmediate, (XtPointer)XmUNSPECIFIED
	/* was XmRCallProc, (XtPointer)_XmDoubelClickIntervalDefault */
    },
    {
	XmNsingleSelectionCallback, XmCCallback, XmRCallback,
	sizeof(XtCallbackList), Offset(SingleCallback),
	XmRCallback, (XtPointer)NULL
    },
    {
	XmNmultipleSelectionCallback, XmCCallback, XmRCallback,
	sizeof(XtCallbackList), Offset(MultipleCallback),
	XmRCallback, (XtPointer)NULL
    },
    {
	XmNextendedSelectionCallback, XmCCallback, XmRCallback,
	sizeof(XtCallbackList), Offset(ExtendCallback),
	XmRCallback, (XtPointer)NULL
    },
    {
	XmNbrowseSelectionCallback, XmCCallback, XmRCallback, 
	sizeof(XtCallbackList), Offset(BrowseCallback),
	XmRCallback, (XtPointer)NULL
    },
    {
	XmNdefaultActionCallback, XmCCallback, XmRCallback,
	sizeof(XtCallbackList), Offset(DefaultCallback),
	XmRCallback, (XtPointer)NULL
    },
    {
	XmNhorizontalScrollBar, XmCHorizontalScrollBar, XmRWidget,
	sizeof(Widget), Offset(hScrollBar),
	XmRImmediate, (XtPointer)NULL
    },
    {
	XmNverticalScrollBar, XmCVerticalScrollBar, XmRWidget,
	sizeof(Widget), Offset(vScrollBar),
	XmRImmediate, (XtPointer)NULL
    },
    {
	XmNnavigationType, XmCNavigationType, XmRNavigationType,
	sizeof(XmNavigationType), XtOffsetOf(XmListRec, primitive.navigation_type),
	XmRImmediate, (XtPointer)XmTAB_GROUP
    }
};

static XmSyntheticResource syn_resources[] = {
    {
	XmNlistSpacing,
	sizeof(Dimension), Offset(ItemSpacing),
	_XmFromVerticalPixels, _XmToVerticalPixels
    },
    {
	XmNmarginWidth,
	sizeof(Dimension), Offset(margin_width),
	_XmFromHorizontalPixels, _XmToHorizontalPixels
    },
    {
	XmNmarginHeight,
	sizeof(Dimension), Offset(margin_height),
	_XmFromVerticalPixels, _XmToVerticalPixels
    },
    {
	XmNtopItemPosition,
	sizeof(int), Offset(top_position),
	NULL /* FIXME */, NULL /* FIXME */
    },
  };
  
static void ListAddMode(Widget, XEvent*, String*, Cardinal*);
static void ListBeginData(Widget, XEvent*, String*, Cardinal*);
static void ListBeginDataExtend(Widget, XEvent*, String*, Cardinal*);
static void ListBeginExtend(Widget, XEvent*, String*, Cardinal*);
static void ListBeginLine(Widget, XEvent*, String*, Cardinal*);
static void ListBeginSelect(Widget, XEvent*, String*, Cardinal*);
static void ListBeginToggle(Widget, XEvent*, String*, Cardinal*);
static void ListButtonMotion(Widget, XEvent*, String*, Cardinal*);
static void ListCopyToClipboard(Widget, XEvent*, String*, Cardinal*);
static void ListDefaultAction(Widget, XEvent*, String*, Cardinal*);
static void ListEndData(Widget, XEvent*, String*, Cardinal*);
static void ListEndDataExtend(Widget, XEvent*, String*, Cardinal*);
static void ListEndExtend(Widget, XEvent*, String*, Cardinal*);
static void ListEndLine(Widget, XEvent*, String*, Cardinal*);
static void ListEndSelect(Widget, XEvent*, String*, Cardinal*);
static void ListEndToggle(Widget, XEvent*, String*, Cardinal*);
static void ListEnter(Widget, XEvent*, String*, Cardinal*);
static void ListExtendNextItem(Widget, XEvent*, String*, Cardinal*);
static void ListExtendPrevItem(Widget, XEvent*, String*, Cardinal*);
static void ListFocusIn(Widget, XEvent*, String*, Cardinal*);
static void ListFocusOut(Widget, XEvent*, String*, Cardinal*);
static void ListKbdActivate(Widget, XEvent*, String*, Cardinal*);
static void ListKbdBeginExtend(Widget, XEvent*, String*, Cardinal*);
static void ListKbdBeginSelect(Widget, XEvent*, String*, Cardinal*);
static void ListKbdCancel(Widget, XEvent*, String*, Cardinal*);
static void ListKbdDeselectAll(Widget, XEvent*, String*, Cardinal*);
static void ListKbdEndExtend(Widget, XEvent*, String*, Cardinal*);
static void ListKbdEndSelect(Widget, XEvent*, String*, Cardinal*);
static void ListKbdSelectAll(Widget, XEvent*, String*, Cardinal*);
static void ListLeftChar(Widget, XEvent*, String*, Cardinal*);
static void ListLeave(Widget, XEvent*, String*, Cardinal*);
static void ListLeftPage(Widget, XEvent*, String*, Cardinal*);
static void ListNextItem(Widget, XEvent*, String*, Cardinal*);
static void ListNextPage(Widget, XEvent*, String*, Cardinal*);
static void ListPrevItem(Widget, XEvent*, String*, Cardinal*);
static void ListPrevPage(Widget, XEvent*, String*, Cardinal*);
static void ListProcessDrag(Widget, XEvent*, String*, Cardinal*);
static void ListRightChar(Widget, XEvent*, String*, Cardinal*);
static void ListRightPage(Widget, XEvent*, String*, Cardinal*);

char _XmList_ListXlations1[] = 
   "s c <Key>osfBeginLine:ListBeginDataExtend()\n\
    :c <Key>osfBeginLine:ListBeginData()\n\
    :<Key>osfBeginLine:ListBeginLine()\n\
    s c <Key>osfEndLine:ListEndDataExtend()\n\
    :c <Key>osfEndLine:ListEndData()\n\
    :<Key>osfEndLine:ListEndLine()\n\
    :<Key>osfPageLeft:ListLeftPage()\n\
    :c <Key>osfPageUp:ListLeftPage()\n\
    :<Key>osfPageUp:ListPrevPage()\n\
    :<Key>osfPageRight:ListRightPage()\n\
    :c <Key>osfPageDown:ListRightPage()\n\
    :<Key>osfPageDown:ListNextPage()\n\
    :s <KeyDown>osfSelect:ListKbdBeginExtend()\n\
    :s <KeyUp>osfSelect:ListKbdEndExtend()\n\
    :<KeyDown>osfSelect:ListKbdBeginSelect()\n\
    :<KeyUp>osfSelect:ListKbdEndSelect()\n\
    :<Key>osfActivate:ListKbdActivate()\n\
    :<Key>osfAddMode:ListAddMode()\n\
    :<Key>osfHelp:PrimitiveHelp()\n\
    :<Key>osfCancel:ListKbdCancel()";

char _XmList_ListXlations2[] = 
   "<Unmap>:PrimitiveUnmap()\n\
    <Enter>:ListEnter()\n\
    <Leave>:ListLeave()\n\
    <FocusIn>:ListFocusIn()\n\
    <FocusOut>:ListFocusOut()\n\
    Button1<Motion>:ListButtonMotion()\n\
    s ~m ~a <Btn1Down>:ListBeginExtend()\n\
    s ~m ~a <Btn1Up>:ListEndExtend()\n\
    c ~s ~m ~a <Btn1Down>:ListBeginToggle()\n\
    c ~s ~m ~a <Btn1Up>:ListEndToggle()\n\
    ~s ~c ~m ~a <Btn1Down>:ListBeginSelect()\n\
    ~s ~c ~m ~a <Btn1Up>:ListEndSelect()\n\
    <Btn2Down>:ListProcessDrag()\n\
    :c <Key>osfLeft:ListLeftPage()\n\
    :<Key>osfLeft:ListLeftChar()\n\
    :c <Key>osfRight:ListRightPage()\n\
    :<Key>osfRight:ListRightChar()\n\
    s <Key>osfUp:ListExtendPrevItem()\n\
    :<Key>osfUp:ListPrevItem()\n\
    s <Key>osfDown:ListExtendNextItem()\n\
    :<Key>osfDown:ListNextItem()\n\
    :c <Key>osfInsert:ListCopyToClipboard()\n\
    :<Key>osfCopy:ListCopyToClipboard()\n\
    ~s c ~m ~a <Key>slash:ListKbdSelectAll()\n\
    ~s c ~m ~a <Key>backslash:ListKbdDeselectAll()\n\
    s ~m ~a <Key>Tab:PrimitivePrevTabGroup()\n\
    ~m ~a <Key>Tab:PrimitiveNextTabGroup()\n\
    ~s ~m ~a <Key>Return:ListKbdActivate()\n\
    ~s ~m ~a <KeyDown>space:ListKbdBeginSelect()\n\
    ~s ~m ~a <KeyUp>space:ListKbdEndSelect()\n\
    s ~m ~a <KeyDown>space:ListKbdBeginExtend()\n\
    s ~m ~a <KeyUp>space:ListKbdEndExtend()";

static XtActionsRec actions[] = {
     {"ListAddMode", ListAddMode}, 
     {"ListBeginData", ListBeginData}, 
     {"ListBeginDataExtend", ListBeginDataExtend}, 
     {"ListBeginExtend", ListBeginExtend}, 
     {"ListBeginLine", ListBeginLine}, 
     {"ListBeginSelect", ListBeginSelect}, 
     {"ListBeginToggle", ListBeginToggle}, 
     {"ListButtonMotion", ListButtonMotion}, 
     {"ListCopyToClipboard", ListCopyToClipboard}, 
     {"ListEndData", ListEndData}, 
     {"ListEndDataExtend", ListEndDataExtend}, 
     {"ListEndExtend", ListEndExtend}, 
     {"ListEndLine", ListEndLine}, 
     {"ListEndSelect", ListEndSelect}, 
     {"ListEndToggle", ListEndToggle}, 
     {"ListExtendNextItem", ListExtendNextItem}, 
     {"ListExtendPrevItem", ListExtendPrevItem}, 
     {"ListKbdActivate", ListKbdActivate}, 
     {"ListKbdBeginExtend", ListKbdBeginExtend}, 
     {"ListKbdBeginSelect", ListKbdBeginSelect}, 
     {"ListKbdCancel", ListKbdCancel}, 
     {"ListKbdDeselectAll", ListKbdDeselectAll}, 
     {"ListKbdEndExtend", ListKbdEndExtend}, 
     {"ListKbdEndSelect", ListKbdEndSelect}, 
     {"ListKbdSelectAll", ListKbdSelectAll}, 
     {"ListLeftChar", ListLeftChar}, 
     {"ListLeftPage", ListLeftPage}, 
     {"ListNextItem", ListNextItem}, 
     {"ListNextPage", ListNextPage}, 
     {"ListPrevItem", ListPrevItem}, 
     {"ListPrevPage", ListPrevPage}, 
     {"ListProcessDrag", ListProcessDrag}, 
     {"ListRightChar", ListRightChar}, 
     {"ListRightPage", ListRightPage}, 
     {"ListEnter", ListEnter},
     {"ListLeave", ListLeave},
     {"ListFocusIn", ListFocusIn}, 
     {"ListFocusOut", ListFocusOut}, 
};

static XmBaseClassExtRec _XmListCoreClassExtRec = {
    /* next_extension            */ NULL,
    /* record_type               */ NULLQUARK,                             
    /* version                   */ XmBaseClassExtVersion,
    /* size                      */ sizeof(XmBaseClassExtRec),
    /* initialize_prehook        */ NULL, /* FIXME */
    /* set_values_prehook        */ NULL, /* FIXME */
    /* initialize_posthook       */ NULL, /* FIXME */
    /* set_values_posthook       */ NULL, /* FIXME */
    /* secondary_object_class    */ NULL, /* FIXME */
    /* secondary_object_create   */ NULL, /* FIXME */
    /* get_secondary_resources   */ NULL, /* FIXME */
    /* fast_subclass             */ { 0 }, /* FIXME */
    /* get_values_prehook        */ NULL, /* FIXME */
    /* get_values_posthook       */ NULL, /* FIXME */
    /* class_part_init_prehook   */ NULL,
    /* class_part_init_posthook  */ NULL,
    /* ext_resources             */ NULL,
    /* compiled_ext_resources    */ NULL,
    /* num_ext_resources         */ 0,
    /* use_sub_resources         */ FALSE,
    /* widget_navigable          */ XmInheritWidgetNavigable,
    /* focus_change              */ XmInheritFocusChange,
    /* wrapper_data              */ NULL
};

XmPrimitiveClassExtRec _XmListPrimClassExtRec = {
    /* next_extension      */ NULL,
    /* record_type         */ NULLQUARK,
    /* version             */ XmPrimitiveClassExtVersion,
    /* record_size         */ sizeof(XmPrimitiveClassExtRec),
    /* widget_baseline     */ NULL, /* FIX ME */
    /* widget_display_rect */ NULL, /* FIX ME */
    /* widget_margins      */ NULL  /* FIX ME */
};

XmListClassRec xmListClassRec = {
    /* Core class part */
    {
      /* superclass            */ (WidgetClass) &xmPrimitiveClassRec,
      /* class_name            */ "XmList",
      /* widget_size           */ sizeof(XmListRec),
      /* class_initialize      */ class_initialize,
      /* class_part_initialize */ class_part_initialize,
      /* class_inited          */ FALSE,
      /* initialize            */ initialize,
      /* initialize_hook       */ NULL,
      /* realize               */ XtInheritRealize,
      /* actions               */ actions,
      /* num_actions           */ XtNumber(actions),
      /* resources             */ resources,
      /* num_resources         */ XtNumber(resources),
      /* xrm_class             */ NULLQUARK,
      /* compress_motion       */ TRUE,
      /* compress_exposure     */ XtExposeCompressMaximal,
      /* compress_enterleave   */ TRUE,
      /* visible_interest      */ FALSE,
      /* destroy               */ destroy,
      /* resize                */ resize,
      /* expose                */ expose,
      /* set_values            */ set_values,
      /* set_values_hook       */ NULL,
      /* set_values_almost     */ XtInheritSetValuesAlmost,
      /* get_values_hook       */ NULL,
      /* accept_focus          */ NULL,
      /* version               */ XtVersion,
      /* callback offsets      */ NULL,
      /* tm_table              */ _XmList_ListXlations1,
      /* query_geometry        */ query_geometry,
      /* display_accelerator   */ NULL,
      /* extension             */ (XtPointer)&_XmListCoreClassExtRec
    },
    /* Primitive Class part */
    {
	/* border_highlight      */ list_border_highlight,
       	/* border_unhighlight    */ list_border_unhighlight,
       	/* translations          */ XtInheritTranslations,
       	/* arm_and_activate_proc */ NULL,
       	/* synthetic resources   */ syn_resources, 
        /* num syn res           */ XtNumber(syn_resources),
	/* extension             */ (XtPointer)&_XmListPrimClassExtRec
    },
    /* List Class part */
    {
      /* extension */ NULL
    }
};

WidgetClass xmListWidgetClass = (WidgetClass)&xmListClassRec;

/* local functions */

static void
_XmListAddItemUnselected(Widget w, 
			 XmString item, 
			 int position)
{
    int i;

    if (position < 0 || position > List_ItemCount(w))
    {
	XdbDebug(__FILE__, w, "XmListAddItemUnselected (itemCount %d) : illegal position %d\n",
		List_ItemCount(w), position);
	return;
    }

    List_Items(w) = (XmStringTable)XtRealloc((XtPointer)List_Items(w),
						 (List_ItemCount(w) + 1) *
						 sizeof(XmString*));

    List_InternalList(w) = (ElementPtr*)XtRealloc((XtPointer)List_InternalList(w),
						  (List_ItemCount(w) + 1) *
						  sizeof(ElementPtr));

    if (position == 0) 
    { 
	/* Append */
	List_Items(w)[List_ItemCount(w)] = XmStringCopy(item);
	List_InternalList(w)[List_ItemCount(w)] = (ElementPtr)XtMalloc(sizeof(Element));
	List_InternalList(w)[List_ItemCount(w)]->name = _XmStringCreate(item);
	List_InternalList(w)[List_ItemCount(w)]->selected = 
	  List_InternalList(w)[List_ItemCount(w)]->last_selected =
	  List_InternalList(w)[List_ItemCount(w)]->LastTimeDrawn = False;

	List_InternalList(w)[List_ItemCount(w)]->NumLines = XmStringLineCount(item);
	List_InternalList(w)[List_ItemCount(w)]->length = XmStringLength(item);
	XmStringExtent(List_Font(w), item,
		       &List_InternalList(w)[List_ItemCount(w)]->width,
		       &List_InternalList(w)[List_ItemCount(w)]->height);

	if (List_ItemCount(w) > 1)
	  List_InternalList(w)[List_ItemCount(w)]->CumHeight = 
	    (List_InternalList(w)[List_ItemCount(w) - 1]->CumHeight + 
	     List_InternalList(w)[List_ItemCount(w)]->height);
	else
	  List_InternalList(w)[List_ItemCount(w)]->CumHeight = 
	    List_InternalList(w)[List_ItemCount(w)]->height;
    }
    else
    {
	Dimension width, height;

	XmStringExtent(List_Font(w), item,
		       &width, &height);

	/* Insert */
	for (i=List_ItemCount(w); i>0 && i >= position; i--)
	    List_Items(w)[i] = List_Items(w)[i-1];
	
	List_Items(w)[i] = XmStringCopy(item);

	for (i=List_ItemCount(w); i>0 && i >= position; i--)
	{
	    List_InternalList(w)[i-1]->LastTimeDrawn = False;
	    List_InternalList(w)[i-1]->height += height;
	    List_InternalList(w)[i] = List_InternalList(w)[i-1];
	}

	List_InternalList(w)[position - 1] = (ElementPtr)XtMalloc(sizeof(Element));
	List_InternalList(w)[position - 1]->name = _XmStringCreate(item);
	List_InternalList(w)[position - 1]->selected = 
	  List_InternalList(w)[position - 1]->last_selected =
	  List_InternalList(w)[position - 1]->LastTimeDrawn = False;

	List_InternalList(w)[position - 1]->NumLines = XmStringLineCount(item);
	List_InternalList(w)[position - 1]->length = XmStringLength(item);
	List_InternalList(w)[position - 1]->height = height;
	List_InternalList(w)[position - 1]->width = width;

	if (position > 1)
	  List_InternalList(w)[position - 1]->CumHeight = 
	    (List_InternalList(w)[position - 2]->CumHeight + 
	     List_InternalList(w)[position - 1]->height);
	else
	  List_InternalList(w)[position - 1]->CumHeight = 
	    List_InternalList(w)[position - 1]->height;
    }


    List_ItemCount(w)++;
}

static void
_XmListAddItemsUnselected(Widget w, 
			  XmString *items, 
			  int item_count, 
			  int position)
{
    int i, n;

    List_ItemCount(w) += item_count;

    List_Items(w) = (XmStringTable)XtRealloc((XtPointer)List_Items(w),
						 List_ItemCount(w) *
						 sizeof(XmString*));

    if (position == 0) 
    { 
	/* Append */

	for (i=List_ItemCount(w) - item_count, n=0; n<item_count; i++, n++)
	    List_Items(w)[i] = items[n];

	return;
    }
    else
    {
	/* Insert */

	for (i=List_ItemCount(w)-1; i>0 && i >= position + item_count; i--) 
	    List_Items(w)[i] = List_Items(w)[i-item_count];

	for (i=position-1, n=0; n<item_count; n++, i++)
	    List_Items(w)[i] = items[n];
    }
}

/**
 *
 * _XmListInstallItems(Widget w)
 *
 * Creates a new items and InternalList instance variable for the widget.
 *
 **/
static void
_XmListInstallItems(Widget w)
{
    if (List_ItemCount(w) != 0) 
    {
        int i;
        XmString *new_items = (XmString*)XtMalloc(List_ItemCount(w) 
					          * sizeof(XmString));
        ElementPtr *new_elements = (ElementPtr*)XtMalloc(List_ItemCount(w) 
						         * sizeof(ElementPtr));
        Dimension cum_height = Prim_ShadowThickness(w);
        Dimension max_height_so_far, max_width_so_far;
        Dimension highlight = (List_HighlightThickness(w) 
			       ? List_HighlightThickness(w) : 1);
    
        /* used in initializing max_height_so_far/max_width_so_far */
        XmString x = XmStringCreateSimple(" "); 
    
        XmStringExtent(List_Font(w), x, &max_width_so_far, &max_height_so_far);
        XmStringFree(x);
    
        for (i=0; i < List_ItemCount(w); i++)
        {
	    new_items[i] = XmStringCopy(List_Items(w)[i]);
    
	    new_elements[i] = (ElementPtr)XtMalloc(sizeof(Element));
    	
	    new_elements[i]->name = _XmStringCreate(new_items[i]);
	    new_elements[i]->selected = 
	        new_elements[i]->last_selected = 
	        new_elements[i]->LastTimeDrawn = False;
    	
	    new_elements[i]->NumLines = XmStringLineCount(new_items[i]);
	    new_elements[i]->length = XmStringLength(new_items[i]);
	    XmStringExtent(List_Font(w), new_items[i], 
		           &new_elements[i]->width, &new_elements[i]->height);
    	
	    cum_height += (new_elements[i]->height 
		           + 2 * highlight 
		           + List_ItemSpacing(w));
    	
	    new_elements[i]->CumHeight = cum_height;
    	
	    if (max_height_so_far < new_elements[i]->height)
	        max_height_so_far = new_elements[i]->height;
    	
	    if (max_width_so_far < new_elements[i]->width)
	        max_width_so_far = new_elements[i]->width;
        }
    
        List_Items(w) = new_items;
        List_InternalList(w) = new_elements;
        List_SelectedIndices(w) = (int *)XtMalloc(sizeof(int) 
					          * List_ItemCount(w));
        List_SelectedItems(w) = (XmString *)XtMalloc(sizeof(XmString) 
						     * List_ItemCount(w));
        List_SelectedItemCount(w) = 0;
        List_MaxItemHeight(w) = max_height_so_far;
        List_MaxWidth(w) = max_width_so_far;
    }
    else 
    {
        List_Items(w) = NULL;
        List_InternalList(w) = NULL;
        List_SelectedIndices(w) = NULL;
        List_SelectedItems(w) = NULL;
        List_SelectedItemCount(w) = 0;
        List_MaxItemHeight(w) = 0;
        List_MaxWidth(w) = 0;
    }
}

static Boolean
_XmListDeselectPos(Widget w, 
		   int position)
{
    int i, j;
 
    if (position < 0 || position > List_ItemCount(w))
      return False;

    if (List_SelectionPolicy(w) == XmSINGLE_SELECT
        || List_SelectionPolicy(w) == XmBROWSE_SELECT)
    {
        if (List_LastItem(w) == position + 1)
        {
            List_InternalList(w)[position]->selected = False;
            List_InternalList(w)[position]->LastTimeDrawn = False;
            List_LastItem(w) = 0;
            return True;
        }
    }
    else  /* XmMULTIPLE_SELECT or XmEXTENDED_SELECT */
    {
        if (List_SelectedItemCount(w) == 0)
          return False;

        List_InternalList(w)[position]->selected = False;
        List_InternalList(w)[position]->LastTimeDrawn = False;

        for (i=0; i < List_SelectedItemCount(w); i++)
	    if (position == List_SelectedIndices(w)[i])
            {
                for (j=i; j < List_SelectedItemCount(w) - 1; j++)
                    List_SelectedIndices(w)[j] = List_SelectedIndices(w)[j+1];
    
	        List_SelectedItemCount(w)--;
    
                List_SelectedIndices(w) = (int*)XtRealloc((XtPointer)List_SelectedIndices(w),
                                                              List_SelectedItemCount(w) * sizeof(int));
                break;
            }

    }
    return True;
}

static void
_XmListDeletePos(Widget w, int position)
{
    int i;

    if (position < 1 || position > List_ItemCount(w))
	return;
    
    XmStringFree(List_Items(w)[position]);

    XtFree((XtPointer)List_InternalList(w)[position]);

    for (i=position; i < List_ItemCount(w) - 1; i++)
    {
        List_Items(w)[i] = List_Items(w)[i+1];
        List_InternalList(w)[i] = List_InternalList(w)[i+1];
    }
    
    --List_ItemCount(w);

    List_Items(w) = (XmStringTable)XtRealloc((XtPointer)List_Items(w),
					     List_ItemCount(w) 
                                             * sizeof(XmString*));

    List_InternalList(w) = (ElementPtr*)XtRealloc((XtPointer)List_InternalList(w),
						  List_ItemCount(w)
						  * sizeof(ElementPtr));
}

static Boolean
_XmListDeleteItem(Widget w, XmString item)
{
    int i;
    
    for (i = 0; i < List_ItemCount(w); i++)
	if (XmStringCompare(item, List_Items(w)[i])) 
        {
	    _XmListDeletePos(w,i);
	    return True;
	}
    
    return False;
}

static void
_XmListFreeItems(Widget w)
{
    int i;

    for (i = 0; i < List_ItemCount(w); i++)
    {
	XtFree((XtPointer)List_Items(w)[i]);
        XtFree((XtPointer)List_InternalList(w)[i]);
    }

    XtFree((XtPointer)List_Items(w));
    XtFree((XtPointer)List_InternalList(w));

    for (i = 0; i < List_SelectedItemCount(w); i++)
	XtFree((XtPointer)List_SelectedItems(w)[i]);

    XtFree((XtPointer)List_SelectedItems(w));
    XtFree((XtPointer)List_SelectedIndices(w));
}

static void
_XmListInvokeCallbacks(Widget w, 
		       XEvent *event,
		       Boolean default_action)
{
    XmListCallbackStruct call_data;
    XtCallbackList callbacks = NULL;
    static int *selected_item_positions;
    static XmString *selected_items;

    XdbDebug(__FILE__, w, "_XmListInvokeCallbacks()\n");

    /* information that is always present in callbacks */
    call_data.event = event;
    call_data.item = List_Items(w)[List_LastItem(w)-1];
    call_data.item_position = List_LastItem(w);
    call_data.item_length = XmStringLength(call_data.item);

    if (default_action) /* defaultActionCallback */
    {
	call_data.reason = XmCR_DEFAULT_ACTION;
	callbacks = List_DefaultCallback(w);
    }
    else /* one of the selection policy callbacks */
    {
	switch (List_SelectionPolicy(w))
	{
	case XmBROWSE_SELECT:
	    call_data.reason = XmCR_BROWSE_SELECT;
	    callbacks = List_BrowseCallback(w);
	    break;
	case XmSINGLE_SELECT:
	    call_data.reason = XmCR_SINGLE_SELECT;
	    callbacks = List_SingleCallback(w);
	    break;
	case XmMULTIPLE_SELECT:
	    call_data.reason = XmCR_MULTIPLE_SELECT;
	    callbacks = List_MultipleCallback(w);
	    break;
	case XmEXTENDED_SELECT:
	    call_data.reason = XmCR_EXTENDED_SELECT;
	    callbacks = List_ExtendCallback(w);
	    break;
	}
    }
    
    if (call_data.reason == XmCR_EXTENDED_SELECT
	|| call_data.reason == XmCR_MULTIPLE_SELECT)
    {
	int i;
	
	selected_item_positions = 
	    (int*)XtRealloc((XtPointer)selected_item_positions,
			    List_SelectedItemCount(w)
			    * sizeof(int));
	
	selected_items = (XmString*)XtRealloc((XtPointer)selected_items,
					      List_SelectedItemCount(w) 
					      * sizeof(XmString));
	
	for (i=0; i < List_SelectedItemCount(w); i++)
	    selected_items[i] = XmStringCopy(List_SelectedItems(w)[i]);
	
	bcopy(List_SelectedIndices(w), 
	      selected_item_positions,
	      List_SelectedItemCount(w) * sizeof(int));
	
	call_data.selected_items = selected_items;
	call_data.selected_item_positions = selected_item_positions;

	if (call_data.reason == XmCR_EXTENDED_SELECT)
	    call_data.selection_type = List_SelectionType(w);
    }

    XtCallCallbackList(w,
		       callbacks,
		       (XtPointer)&call_data);
    
    if (call_data.reason == XmCR_EXTENDED_SELECT
	|| call_data.reason == XmCR_MULTIPLE_SELECT)
    {
	int i;
	
	for (i=0; i < List_SelectedItemCount(w); i++)
	    XmStringFree(selected_items[i]);
    }
}

/* set the redraw_all_visible to True if all the items visible should be
   redrawn.  If you're changing just one item, set this to false,
   but make sure you set the LastTimeDrawn in that item's InternalList entry
   to False */
static void
_XmListRedraw(Widget w,
	      Boolean redraw_all_visible)
{
    Position x, y;
    int i;
    Dimension highlight = (List_HighlightThickness(w) 
			   ? List_HighlightThickness(w) : 1);
    Dimension fill_width = (XtWidth(w) 
                            - 2 * (Prim_ShadowThickness(w)
                                   + List_HighlightThickness(w)));

    XdbDebug(__FILE__, w, "XmListRedraw()\n");

    if (!XtIsRealized(w))
	return;

    _XmDrawShadows(XtDisplay(w),
		   XtWindow(w),
		   Prim_TopShadowGC(w),
		   Prim_BottomShadowGC(w),
		   0,0,
		   XtWidth(w),
		   XtHeight(w),
		   Prim_ShadowThickness(w), XmSHADOW_IN);
		   
    x = (List_MarginWidth(w) 
	 + Prim_ShadowThickness(w) 
	 + List_HighlightThickness(w)
	 + 1); /* why is this + 1??? */
    y = (List_MarginHeight(w) 
	 + Prim_ShadowThickness(w) 
	 + List_HighlightThickness(w)
	 + 1); /* why is this + 1??? */

    for (i = List_TopPosition(w) - 1; 
	 (i < List_ItemCount(w))
	 && (i < List_VisibleItemCount(w) + List_TopPosition(w) - 1);
	 i++) 
      {

	  if (i < 0)
	      continue;

	  if (List_InternalList(w)[i]->LastTimeDrawn == False
	      || redraw_all_visible == True) 

          {
              XdbDebug(__FILE__, w, "  XmListRedraw() redrawing item %d...yes\n");
              
              /* we don't want to continually redraw this item... */	     
              List_InternalList(w)[i]->LastTimeDrawn = True;
 
	      XFillRectangle(XtDisplay(w), XtWindow(w),
			     (List_InternalList(w)[i]->selected
			      ? List_NormalGC(w) : List_InverseGC(w)),
			     x, y,
                             fill_width, 
			     /*List_MaxWidth(w) - 1, */
			     List_MaxItemHeight(w));
	       
	      XmStringDrawImage(XtDisplay(w), 
			        XtWindow(w), 
			        List_Font(w),
			        List_InternalList(w)[i]->name, 
			        (List_InternalList(w)[i]->selected
			         ? List_InverseGC(w): List_NormalGC(w)),
			        x - List_XOrigin(w), y,
			        List_InternalList(w)[i]->width, 
			        XmALIGNMENT_BEGINNING,
			        List_StrDir(w), 
			        NULL);
	  }
          else
              XdbDebug(__FILE__, w, "  XmListRedraw() redrawing item %d...no\n");

	  y += (List_MaxItemHeight(w)
		+ 2 * highlight 
		+ List_ItemSpacing(w));
      }

    XSetClipMask(XtDisplay(w), List_InverseGC(w), None);
    XSetClipMask(XtDisplay(w), List_NormalGC(w), None);
}

static void
_XmListRecalcItems(Widget w)
{
    Dimension available_height;
    int nitems;
    Dimension highlight = (List_HighlightThickness(w) 
			   ? List_HighlightThickness(w) : 1);

    available_height = (XtHeight(w) 
			- 2 * (highlight 
			       + List_MarginHeight(w) 
			       + Prim_ShadowThickness(w)));

    nitems = (available_height 
	      / (List_MaxItemHeight(w) +
		 List_ItemSpacing(w) +
		 highlight));
    
    List_VisibleItemCount(w) = nitems > 0 ? nitems : 1;
    
    if (List_VisibleItemCount(w) > List_ItemCount(w))
    {
	List_VisibleItemCount(w) = List_ItemCount(w);
	if (List_TopPosition(w) > 1)
	    List_TopPosition(w) = 1;
    }

    XdbDebug(__FILE__, w, "XmListRecalcItems => VisibleItemCount %d\n", List_VisibleItemCount(w));
}

static void
_XmListSelectPos(Widget w, int position)
{
    int i;

    if (position < 0 || position >= List_ItemCount(w))
	return;

    if (List_InternalList(w)[position]->selected == True)
        return;
    else
    {
        List_InternalList(w)[position]->selected = True;
        List_InternalList(w)[position]->LastTimeDrawn = False;
    }

    if (List_SelectionPolicy(w) == XmSINGLE_SELECT
	|| List_SelectionPolicy(w) == XmBROWSE_SELECT)
    {
	List_LastItem(w) = position + 1;
    }
    else 
    {
	for (i=0; i < List_SelectedItemCount(w); i++)
	    if (position == List_SelectedIndices(w)[i])
		return; /* it's already in the list.  Don't add it again */
	
	List_SelectedItems(w)[List_SelectedItemCount(w)] = List_Items(w)[position];
	List_SelectedIndices(w)[List_SelectedItemCount(w)] = position;
	
	List_SelectedItemCount(w)++;
    }
}

static void
_XmListSelectPosIfMatch(Widget w, int position)
{
    int i;
    
    if(List_InternalList(w)[position]->selected)
	return;

    for(i=0; i<List_SelectedItemCount(w); ++i)
	if(XmStringCompare(List_Items(w)[position],
			   List_SelectedItems(w)[i])) 
	{
	    _XmListSelectPos(w, position);
	    break;
	}
}

static void
_XmListSetVertScrollBar(Widget w)
{
    if(!List_IsScrolledList(w) || !XtIsManaged(List_VSB(w)))
	return;

    List_Vorigin(w) = List_TopPosition(w);
    List_Vextent(w) = List_VisibleItemCount(w);
    List_Vmin(w) = 1;

    if(List_ItemCount(w) < List_VisibleItemCount(w)) 
	List_Vmax(w) = List_VisibleItemCount(w);
    else
	List_Vmax(w) = List_ItemCount(w);
    
    XtVaSetValues((Widget)List_VSB(w),
		  XmNmaximum, List_Vmax(w),
		  XmNminimum, List_Vmin(w),
		  XmNsliderSize, List_Vextent(w),
		  XmNvalue, List_Vorigin(w),
		  XmNpageIncrement, List_Vextent(w),
		  NULL);
}

static void
_XmListCalculateInitialGeometry(Widget new, Widget request)
{
    int nitems;
    Dimension highlight = List_HighlightThickness(new) ? List_HighlightThickness(new) : 1;
    Dimension available_height;

    if (XtHeight(request) == 0 || XtWidth(request) == 0)
    {
	if (List_VisibleItemCount(new) == 0)
	    List_VisibleItemCount(new) = 1;

	XtWidth(new) = List_MaxWidth(new) + 2 * (highlight + List_MarginWidth(new) + Prim_ShadowThickness(new));

	XtHeight(new) = ((List_MaxItemHeight(new) + 2 * highlight) 
			 * List_VisibleItemCount(new) 
			 + List_ItemSpacing(new) * (List_VisibleItemCount(new) - 1)
			 + highlight 
			 + 2 * (List_MarginHeight(new) + Prim_ShadowThickness(new)));
    }
    else
    {
	/* we have been given a height && width.  This overrides
	   the XmNvisibleItemCount resource */

	if (XtHeight(new) > List_ItemSpacing(new) + 2 * (highlight + List_MarginHeight(new) + Prim_ShadowThickness(new)))
	    available_height = (XtHeight(new) - List_ItemSpacing(new) -
				2 * (highlight + List_MarginHeight(new) + Prim_ShadowThickness(new)));
	else
	    available_height = 0;
	
	nitems = available_height / (List_MaxItemHeight(new) +
				     List_ItemSpacing(new));
	
	List_VisibleItemCount(new) = nitems > 0 ? nitems : 1;	
    }

    XdbDebug(__FILE__, new, "_XmListCalculateInitialGeometry => VisibleItemCount %d\n",
	List_VisibleItemCount(new));
}


static void
_XmListSetGeometry(Widget w)
{
    XtGeometryResult result;
    Dimension width, height, desired_width, desired_height;
    int i;
    Dimension highlight = List_HighlightThickness(w) ? List_HighlightThickness(w) : 1;

    List_MaxWidth(w) = 0;

    for(i=0; i < List_ItemCount(w); ++i) 
    {
	XmStringExtent(List_Font(w), List_Items(w)[i], &width, &height);

	if( width > List_MaxWidth(w) )
	    List_MaxWidth(w) = width;
    }

    desired_width = List_MaxWidth(w) + 2*(Prim_ShadowThickness(w) +
					  List_MarginWidth(w) + highlight);

    if(List_VisibleItemCount(w) == 0)
    {
	Dimension available_height;
	int nitems;

	if (XtHeight(w) > List_ItemSpacing(w) + 2 * (highlight + List_MarginHeight(w) + Prim_ShadowThickness(w)))
	    available_height = (XtHeight(w) - List_ItemSpacing(w) -
				2 * (highlight + List_MarginHeight(w) + Prim_ShadowThickness(w)));
	else
	    available_height = 0;
	
	nitems = available_height / (List_MaxItemHeight(w) +
				     List_ItemSpacing(w));

	List_VisibleItemCount(w) = nitems > 0 ? nitems : 1;
	XdbDebug(__FILE__, w, "XmListSetGeometry: VisibleItemCount %d\n", List_VisibleItemCount(w));
    }

    desired_height = ((List_MaxItemHeight(w) + highlight) *
		      List_VisibleItemCount(w) + 2 * (List_MarginHeight(w) + Prim_ShadowThickness(w) + highlight)
		      + (List_VisibleItemCount(w) - 1) * List_ItemSpacing(w)) ;
	
    switch(List_SizePolicy(w)) {
    case XmVARIABLE:
	if(XtWidth(w) > desired_width)
	    desired_width = XtWidth(w);
	break;
    case XmCONSTANT:
	desired_width = XtWidth(w);
	break;
    case XmRESIZE_IF_POSSIBLE:
    default:
	break;
    }
    
    switch(List_SBDisplayPolicy(w)) {
    case XmSTATIC:
	desired_height = XtHeight(w);
	break;
    case XmAS_NEEDED:
	if(List_ItemCount(w) <= List_VisibleItemCount(w) && List_IsScrolledList(w)
	   && XtIsManaged(List_VSB(w)))
	    XtUnmanageChild((Widget)List_VSB(w));
	else if (List_ItemCount(w) > List_VisibleItemCount(w) &&
		 List_IsScrolledList(w) && !XtIsManaged(List_VSB(w)))
	    XtManageChild((Widget)List_VSB(w));
    default:
	break;
    }

    result = XtMakeResizeRequest((Widget)w,desired_width,desired_height,
				 &width,&height);

    switch(result) {
      case XtGeometryNo:
	return;
      case XtGeometryYes:
	XtWidth(w) = width;
	XtHeight(w) = height;

	if(List_SizePolicy(w) != XmCONSTANT && List_IsScrolledList(w) &&
	   XtIsManaged(List_HSB(w)))
	    XtUnmanageChild((Widget)List_HSB(w));
	break;
      case XtGeometryAlmost:
	if(List_IsScrolledList(w)) {
	    if(width < desired_width && !XtIsManaged(List_HSB(w)))
		XtManageChild((Widget)List_HSB(w));
	    if(height < desired_height) {
		Dimension available_height = XtHeight(w) +
		    List_ItemSpacing(w) - highlight +
		    2 * (List_MarginHeight(w) + Prim_ShadowThickness(w));
		int nitems = available_height / (List_MaxItemHeight(w) + highlight +
						 List_ItemSpacing(w));
		List_VisibleItemCount(w) = nitems > 0 ? nitems : 1;

		XdbDebug(__FILE__, w, "visibleItemCount => %d (line %d)\n", List_VisibleItemCount(w), __LINE__);
		desired_height = List_VisibleItemCount(w) *
		    (List_MaxItemHeight(w) + highlight + List_ItemSpacing(w)) -
		    List_ItemSpacing(w) + highlight +
		    2 * (List_MarginHeight(w) + Prim_ShadowThickness(w));

		(void)XtMakeResizeRequest((Widget)w,width,desired_height,
					  NULL,&height);
		
		if(!XtIsManaged(List_VSB(w)))
		    XtManageChild((Widget)List_VSB(w));
	    }
	}
	(void)XtMakeResizeRequest((Widget)w,width,height,NULL,NULL);
	break;
      default:
	XtWarning("_XmListSetGeometry: Strange response from XtMakeResizeRequest.\n");
	break;
    }
/* This is not necessary ...
    XtWidth(w) = width;
    XtHeight(w) = height;
 ... it is already done by XtMakeResizeRequest */

    if(List_IsScrolledList(w)) {
	if(XtIsManaged(List_HSB(w))) {
	    XtVaSetValues((Widget)List_HSB(w),/**appetite**/ NULL);
	}
	_XmListSetVertScrollBar(w);
    }
}

static void
_XmListSetItem(Widget w, XEvent *event, int position)
{
    if(List_SelectionPolicy(w) == XmBROWSE_SELECT ) 
    {
	if (List_SelectedItemCount(w))
	    _XmListDeselectPos(w, List_SelectedIndices(w)[0]);

	_XmListSelectPos(w, position - 1);

    } 
    else if(List_SelectionPolicy(w) == XmEXTENDED_SELECT) 
    {
	if(!List_AddMode(w))
	    XmListDeselectAllItems(w);
	_XmListSelectPos(w, position - 1);
    }

    List_LastHLItem(w) = position;
    _XmListRedraw(w, True);

#if 0
    if(List_AutoSelect(w) && !List_AddMode(w))
	_XmListInvokeCallbacks(w,_XmCR_SELECT,event,XmINITIAL);
#endif
}


/* internal callbacks */


static void
_XmListHorizontalScrollBarCallback(Widget w,
				   XtPointer client_data, 
				   XtPointer call_data)

{
    XmScrollBarCallbackStruct *cbs = (XmScrollBarCallbackStruct*) call_data;
    Widget l = (Widget)client_data;

    List_XOrigin(l) = cbs->value;

    _XmListRedraw(l, True);
}

static void
_XmListVerticalScrollBarCallback(Widget w,
				 XtPointer client_data, 
				 XtPointer call_data)
{
    XmScrollBarCallbackStruct *cbs = (XmScrollBarCallbackStruct*) call_data;
    Widget l = (Widget)client_data;

    List_TopPosition(l) = cbs->value;

    _XmListRedraw(l, True);
}


/* widget methods */

static void
class_initialize()
{
    _XmListCoreClassExtRec.record_type = XmQmotif;
}

static void
class_part_initialize(WidgetClass widget_class)
{
    _XmFastSubclassInit(widget_class, XmLIST_BIT);
}

static void
destroy(Widget w)
{
    XdbDebug(__FILE__, w, "Detroy()\n");

    _XmListFreeItems(w);

    XtReleaseGC(w, List_NormalGC(w));
    XtReleaseGC(w, List_HighlightGC(w));
}

static void
resize(Widget w)
{
/*    _XmListRecalcItems(w);*/

    XmListSetHorizPos(w, List_Horigin(w));
}

static void
expose(Widget w, XEvent *event, Region region)
{
#define superclass (&xmPrimitiveClassRec)
    (*superclass->core_class.expose)(w, event, region);
#undef superclass

    _XmListRecalcItems(w);
    _XmListSetVertScrollBar(w);
    _XmListRedraw(w, True);
}

static void
CreateNormalGC(Widget w)
{
    XGCValues values;

    values.foreground = Prim_Foreground(w);
    values.background = XtBackground(w);

    List_NormalGC(w) = XtGetGC(w, GCForeground | GCBackground, &values);
}

static void
CreateHighlightGC(Widget w)
{
    unsigned long mask = GCForeground | GCBackground;
    XGCValues values;

    values.foreground = Prim_HighlightColor(w);
    values.background = XtBackground(w);

    if (List_AddMode(w))
    {
#if 0
/* URK!  DashTile isn't initialized! */
	mask |= GCTile | GCFillStyle;
	
	values.tile = List_DashTile(w);
	values.fill_style = FillTiled;
#endif
    }

    List_HighlightGC(w) = XtGetGC(w, mask, &values);
}

static void
CreateInsensitiveGC(Widget w)
{
    XGCValues values;

    values.foreground = Prim_HighlightColor(w);
    values.background = XtBackground(w);
    values.fill_style = FillStippled;

    List_HighlightGC(w) = XtGetGC(w, GCForeground | GCBackground | GCFillStyle, &values);
}

static void
CreateInverseGC(Widget w)
{
    XGCValues values;

    values.foreground = XtBackground(w);
    values.background = Prim_HighlightColor(w);

    List_InverseGC(w) = XtGetGC(w, GCForeground | GCBackground, &values);    
}

static void
initialize(Widget request, 
	   Widget new, 
	   ArgList args, 
	   Cardinal *num_args)
{
    XtTranslations trans = XtParseTranslationTable(_XmList_ListXlations2);

    XdbDebug(__FILE__, new, "initialize()\n");

    XtOverrideTranslations(new, trans);

    if(List_Font(new) == (XmFontList)XmUNSPECIFIED || List_Font(new) == NULL)  
	List_Font(new) = _XmGetDefaultFontList(new, XmTEXT_FONTLIST);

    List_AddMode(new) = False;

    List_LastHLItem(new) = 0;  /* initialize to a non-legal values */
    List_LastItem(new) = 0; 
    List_CurrentKbdItem(new) = 0;

    if (List_TopPosition(new) == 0)
	List_TopPosition(new) = 1;

    if (List_ItemSpacing(new) == 0)
	List_ItemSpacing(new) = 3;

    List_HighlightThickness(new) = Prim_HighlightThickness(new);

    CreateNormalGC(new);
    CreateInsensitiveGC(new);
    CreateInverseGC(new);
    CreateHighlightGC(new);

    if (XmIsScrolledWindow(XtParent(new)))
	List_Mom(new) = (XmScrolledWindowWidget)XtParent(new);
    else
	List_Mom(new) = NULL;

    if(List_IsScrolledList(new))
    {
	char *name = XtMalloc(strlen(XtName(new))+4);

	strcpy(name,XtName(new));
	strcat(name,"HSB");
	List_HSB(new) = (XmScrollBarWidget)XtVaCreateManagedWidget(name, 
								   xmScrollBarWidgetClass, 
								   (Widget)List_Mom(new),
								   XmNorientation, XmHORIZONTAL, 
								   NULL);
	XtAddCallback((Widget)List_HSB(new), XmNdecrementCallback,
		      _XmListHorizontalScrollBarCallback, 
		      (XtPointer)new);
	XtAddCallback((Widget)List_HSB(new), XmNdragCallback,
		      _XmListHorizontalScrollBarCallback,
		      (XtPointer)new);
	XtAddCallback((Widget)List_HSB(new), XmNincrementCallback,
		      _XmListHorizontalScrollBarCallback,
		      (XtPointer)new);
	XtAddCallback((Widget)List_HSB(new), XmNpageDecrementCallback,
		      _XmListHorizontalScrollBarCallback,
		      (XtPointer)new);
	XtAddCallback((Widget)List_HSB(new), XmNpageIncrementCallback,
		      _XmListHorizontalScrollBarCallback,
		      (XtPointer)new);
	XtAddCallback((Widget)List_HSB(new), XmNtoBottomCallback,
		      _XmListHorizontalScrollBarCallback,
		      (XtPointer)new);
	XtAddCallback((Widget)List_HSB(new), XmNtoTopCallback,
		      _XmListHorizontalScrollBarCallback,
		      (XtPointer)new);
	XtAddCallback((Widget)List_HSB(new), XmNvalueChangedCallback,
		      _XmListHorizontalScrollBarCallback,
		      (XtPointer)new);

	strcpy(name,XtName(new));
	strcat(name,"VSB");
	List_VSB(new) = (XmScrollBarWidget)XtVaCreateManagedWidget(name,
								   xmScrollBarWidgetClass,
								   XtParent(new),
								   NULL);

	XtAddCallback((Widget)List_VSB(new), XmNdecrementCallback,
		      _XmListVerticalScrollBarCallback,
		      (XtPointer)new);
	XtAddCallback((Widget)List_VSB(new), XmNdragCallback,
		      _XmListVerticalScrollBarCallback,
		      (XtPointer)new);
	XtAddCallback((Widget)List_VSB(new), XmNincrementCallback,
		      _XmListVerticalScrollBarCallback,
		      (XtPointer)new);
	XtAddCallback((Widget)List_VSB(new), XmNpageDecrementCallback,
		      _XmListVerticalScrollBarCallback,
		      (XtPointer)new);
	XtAddCallback((Widget)List_VSB(new), XmNpageIncrementCallback,
		      _XmListVerticalScrollBarCallback,
		      (XtPointer)new);
	XtAddCallback((Widget)List_VSB(new), XmNtoBottomCallback,
		      _XmListVerticalScrollBarCallback,
		      (XtPointer)new);
	XtAddCallback((Widget)List_VSB(new), XmNtoTopCallback,
		      _XmListVerticalScrollBarCallback,
		      (XtPointer)new);
	XtAddCallback((Widget)List_VSB(new), XmNvalueChangedCallback,
		      _XmListVerticalScrollBarCallback,
		      (XtPointer)new);

	XtVaSetValues((Widget)List_Mom(new),
		      XmNscrollBarDisplayPolicy, XmSTATIC,
		      XmNscrollingPolicy, XmAPPLICATION_DEFINED,
		      XmNvisualPolicy, XmVARIABLE,
		      /* Instead of call to XmScrolledWindowSetAreas() */
		      XmNworkWindow, new, 
		      XmNhorizontalScrollBar, List_HSB(new),
		      XmNverticalScrollBar, List_VSB(new),
		      NULL);
	XtFree(name);
    }

    List_XOrigin(new) = 0; /* initialize x coordinate */
    
    List_Horigin(new) = 0; 
    List_Hmin(new) = 0;
    List_Vorigin(new) = 0; 
    List_Vmin(new) = 0;
    
    _XmListInstallItems(new);

    _XmListCalculateInitialGeometry(new, request);
}

static Boolean
set_values(Widget old, 
	   Widget request, 
	   Widget new, 
	   ArgList args, 
	   Cardinal *num_args)
{
    Boolean need_refresh = False;
    int i;

    XdbDebug(__FILE__, new, "XmList set_values\n");

    if (List_SizePolicy(old) != List_SizePolicy(new)) 
    {
	_XmWarning(new, "Cannot change XmNlistSizePolicy after initialization.\n");

	List_SizePolicy(new) = List_SizePolicy(old);
    }

    if (List_Font(old) != List_Font(new))
    {
	XmFontListFree(List_Font(old));
	_XmListSetGeometry(new);
	need_refresh = True;
    }

    if (List_ItemCount(old) != List_ItemCount(new) &&
	List_Items(old) == List_Items(new)) 
    {
	List_ItemCount(new) = List_ItemCount(old);
	List_Items(new) = List_Items(old);
	_XmListSetGeometry(new);
	_XmWarning(new, "XmNitemCount and XmNitems must be set by the same XtSetValues.\n");
    } 
    else if (List_ItemCount(old) != List_ItemCount(new) ||
	     List_Items(old) != List_Items(new)) 
    {
	_XmListInstallItems(new); 
	_XmListFreeItems(old);
	_XmListSetGeometry(new);
	need_refresh = True;
    }

    if(List_MarginHeight(old) != List_MarginHeight(new) || 
       List_MarginWidth(old) != List_MarginWidth(new) ||
       List_ItemSpacing(old) != List_ItemSpacing(new) ||
       List_SBDisplayPolicy(old) != List_SBDisplayPolicy(new) ||
       List_StrDir(old) != List_StrDir(new) ||
       List_TopPosition(old) != List_TopPosition(new)) {
	_XmListSetGeometry(new);
	need_refresh = True;
    }

    if(List_VisibleItemCount(old) != List_VisibleItemCount(new)) {
	_XmListSetGeometry(new);
	need_refresh = True;
    }

    if(List_SelectedItemCount(old) != List_SelectedItemCount(new) &&
       List_SelectedItems(old) == List_SelectedItems(new)) {
	List_SelectedItemCount(new) = List_SelectedItemCount(old);
	List_SelectedItems(new) = List_SelectedItems(old);
	_XmWarning(new, "XmNselectedItemCount and XmNselectedItems must be set by the same XtSetValues.\n");
    } else if(List_SelectedItemCount(old) != List_SelectedItemCount(new) ||
	      List_SelectedItems(old) != List_SelectedItems(new)) {	      
	List_SelectedItems(new) =
	    (XmStringTable)XtMalloc((List_SelectedItemCount(new) ?
				     List_SelectedItemCount(new) : 1) *
				    sizeof(XmString));
	for(i = 0; i < List_SelectedItemCount(new); ++i )
	    List_SelectedItems(new)[i] =
		XmStringCopy(List_SelectedItems(old)[i]);
	for( i = 0; i < List_ItemCount(new); ++i ) {
#if 0
	    new->list.selected[i] = False;
#endif
	    _XmListSelectPosIfMatch(new,i);
	}
	need_refresh = True;
    }

    if(List_SelectionPolicy(old) != List_SelectionPolicy(new))
    {
	switch(List_SelectionPolicy(new)) 
	{
	case XmBROWSE_SELECT:
	case XmEXTENDED_SELECT:
	    List_AddMode(new) = False;
	    break;
	case XmSINGLE_SELECT:
	case XmMULTIPLE_SELECT:
	    List_AddMode(new) = True;
	    break;
	default:
	    _XmWarning(new, "Invalid selectionPolicy.\n");
	}
	/* create the new highlight gc */
	XtReleaseGC(new, List_HighlightGC(new));
	CreateHighlightGC(new);
    }

    return need_refresh;
}


/* Actions */

static void
ListAddMode(Widget w, 
	    XEvent *event, 
	    String *params, 
	    Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListAddMode action\n");

    if(List_SelectionPolicy(w)==XmEXTENDED_SELECT)
	List_AddMode(w) = !List_AddMode(w);
}

static void
ListBeginData(Widget w, 
	      XEvent *event, 
	      String *params, 
	      Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListBeginData action\n");

    List_TopPosition(w) = 1;
    
    if(!List_AddMode(w)) 
    {
	XmListDeselectAllItems(w);
	_XmListSelectPos(w,0);

	/* send out callbacks */
	List_SelectionType(w) = XmINITIAL;

	_XmListInvokeCallbacks(w, event, False);
    }
}

static void
ListBeginDataExtend(Widget w, 
		    XEvent *event, 
		    String *params, 
		    Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListBeginDataExtend action\n");

    if (List_ItemCount(w) == 0)
	return;

    if (List_SelectionPolicy(w) == XmMULTIPLE_SELECT)
    {
	List_KbdSelection(w) = True;
	if (List_CurrentKbdItem(w))
	    List_InternalList(w)[List_CurrentKbdItem(w)-1]->LastTimeDrawn = False;
	List_CurrentKbdItem(w) = 1;
	List_InternalList(w)[List_CurrentKbdItem(w)-1]->LastTimeDrawn = False;
        _XmListRecalcItems(w);
	_XmListRedraw(w, False);
    }
    else if (List_SelectionPolicy(w) == XmEXTENDED_SELECT)
    {
	/* FIX ME */
    }
}

static void
ListBeginExtend(Widget w, 
		XEvent *event, 
		String *params, 
		Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListBeginExtend action\n");

    if (List_ItemCount(w) == 0)
	return;

    if (List_SelectionPolicy(w) == XmEXTENDED_SELECT)
    {
	/* FIX ME */

	if (List_AutoSelect(w))
	{
	    List_SelectionType(w) = XmINITIAL;
	    _XmListInvokeCallbacks(w, event, False);
	}
    }
}

static void
ListBeginLine(Widget w, 
	      XEvent *event, 
	      String *params, 
	      Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListBeginList action\n");

    XmListSetHorizPos(w, List_Hmin(w));
}

static void
ListBeginSelect(Widget w, 
		XEvent *event, 
		String *params, 
		Cardinal *num_params)
{
    XButtonEvent *bevent = (XButtonEvent *) event;
    int position;

    XdbDebug(__FILE__, w, "ListBeginSelect action\n");

    if (List_ItemCount(w) == 0)
	return;

    position = XmListYToPos(w, bevent->y + List_Vorigin(w));

    XdbDebug(__FILE__, w, "Selected position %d\n", position);

    switch(List_SelectionPolicy(w)) 
    {
    case XmBROWSE_SELECT:
	if (List_LastItem(w))
	    _XmListDeselectPos(w, List_LastItem(w) - 1);
	_XmListSelectPos(w, position - 1);
	break;
    case XmSINGLE_SELECT:
	if (List_InternalList(w)[position - 1]->selected)
	    _XmListDeselectPos(w, position - 1);
	else 
	{
            /* if something was previously selected,
               deselect it */
	    if (List_LastItem(w) != 0)
                _XmListDeselectPos(w, List_LastItem(w) - 1);
	    
	    _XmListSelectPos(w, position - 1);
	}
	break;
    case XmMULTIPLE_SELECT:
	if (List_InternalList(w)[position - 1]->selected)
	    _XmListDeselectPos(w, position - 1);
	else
	    _XmListSelectPos(w, position - 1);
    case XmEXTENDED_SELECT:
	XmListDeselectAllItems(w);

        _XmListSelectPos(w, position - 1);
        /* FIX ME ? */
	break;
    }

    List_LastHLItem(w) = position;

    _XmListRecalcItems(w);
    _XmListRedraw(w, False);

#if 0
    if(List_AutoSelect(w))
	_XmListInvokeCallbacks(w, _XmCR_SELECT, event, XmINITIAL);
#endif
}

static void
ListBeginToggle(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListBeginToggle()\n");

    if (List_ItemCount(w) == 0)
	return;

    if (List_SelectionPolicy(w) == XmEXTENDED_SELECT)
    {
      /* FIX ME */

      if (List_AutoSelect(w))
	_XmListInvokeCallbacks(w, event, False);
    }
}

static void
ListButtonMotion(Widget w, 
                 XEvent *event, 
                 String *params, 
                 Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListButtonMotion() action\n");

    if (List_SelectionPolicy(w) == XmBROWSE_SELECT)
    {
	XButtonEvent *bevent = (XButtonEvent *)event;
	int new_item;

	new_item = XmListYToPos(w, bevent->y + List_Vorigin(w));

	/* List_LastItem was set in the ListBeginSelect action */

        if (List_LastItem(w) != new_item)
	{
            _XmListDeselectPos(w, List_LastItem(w) - 1);
	    
	    _XmListSelectPos(w, new_item - 1);
	    
	    _XmListRecalcItems(w);
	    _XmListRedraw(w, False);

	    /* if automaticselect is True, call XmNbrowseSelectionCallback */

            if (List_AutoSelect(w))
	        _XmListInvokeCallbacks(w, event, False);
	}
    }
}

static void
ListCopyToClipboard(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListCopyToClipboard()\n");

    if (List_ItemCount(w) == 0 || List_SelectedItemCount(w) == 0)
	return;

    /* FIX ME */
}

static void
ListDefaultAction(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListDefaultAction()\n");

#if 0
    _XmListInvokeCallbacks(w, XmCR_DEFAULT_ACTION, event, XmINITIAL);
#endif
}

static void
ListEndData(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListEndData()\n");

    List_TopPosition(w) = List_ItemCount(w) - List_VisibleItemCount(w);
    
    if (List_TopPosition(w) < 1) 
	List_TopPosition(w) = 1;

    if(!List_AddMode(w)) 
    {
	XmListDeselectAllItems(w);
	_XmListSelectPos(w, List_ItemCount(w) - 1);
#if 0
	_XmListInvokeCallbacks(w,_XmCR_SELECT,event,XmINITIAL);
#endif
    }
}

static void
ListEndDataExtend(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListEndDataExtend()\n");

    if (List_ItemCount(w) == 0)
	return;

    if (List_SelectionPolicy(w) == XmMULTIPLE_SELECT)
    {
	List_KbdSelection(w) = True;
	if (List_CurrentKbdItem(w))
	    List_InternalList(w)[List_CurrentKbdItem(w)-1]->LastTimeDrawn = False;
	List_KbdSelection(w) = List_ItemCount(w);
	List_InternalList(w)[List_CurrentKbdItem(w)-1]->LastTimeDrawn = False;
        _XmListRecalcItems(w);
	_XmListRedraw(w, False);
    }
    else if (List_SelectionPolicy(w) == XmEXTENDED_SELECT)
    {
	/* FIX ME */
    }
}

static void
ListEndExtend(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListEndExtend()\n");

    if (List_ItemCount(w) == 0)
	return;

    if (List_SelectionPolicy(w) == XmEXTENDED_SELECT)
    {
	/* FIX ME */
    }
}

static void
ListEndLine(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListEndLine()\n");

    XmListSetHorizPos(w, List_Hmax(w));
}

static void
ListEndSelect(Widget w, 
              XEvent *event, 
              String *params, 
              Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListEndSelect() action\n");

    if (List_ItemCount(w) == 0)
	return;

    List_KbdSelection(w) = True;
    if (List_CurrentKbdItem(w))
	List_InternalList(w)[List_CurrentKbdItem(w)-1]->LastTimeDrawn = False;
    List_CurrentKbdItem(w) = List_LastHLItem(w);
    List_InternalList(w)[List_CurrentKbdItem(w)-1]->LastTimeDrawn = False;
    _XmListRecalcItems(w);
    _XmListRedraw(w, False);
   
    if ((List_SelectionPolicy(w) == XmSINGLE_SELECT
         || List_SelectionPolicy(w) == XmMULTIPLE_SELECT)
        || ((List_SelectionPolicy(w) == XmBROWSE_SELECT 
            || List_SelectionPolicy(w) == XmEXTENDED_SELECT)
	   && List_AutoSelect(w) == False))
    {
        _XmListInvokeCallbacks(w, event, False);
    }
}

static void
ListEndToggle(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListEndToggle()\n");

    if (List_ItemCount(w) == 0)
	return;

    if (List_SelectionPolicy(w) == XmEXTENDED_SELECT)
    {
	/* FIX ME */
    }
}

static void
ListExtendNextItem(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListExtendNextItem()\n");

    if (List_ItemCount(w) == 0)
	return;

    if (List_SelectionPolicy(w) == XmEXTENDED_SELECT)
    {
	/* FIX ME */
    }
}

static void
ListExtendPrevItem(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListExtendPrevItem()\n");

    if (List_ItemCount(w) == 0)
	return;

    if (List_SelectionPolicy(w) == XmEXTENDED_SELECT)
    {
	/* FIX ME */
    }
}

static void
ListFocusIn(Widget w, 
	    XEvent *event, 
	    String *params, 
	    Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListFocusIn() action\n");

    XtCallActionProc(w, "PrimitiveFocusIn", event, params, *num_params);
}

static void
ListFocusOut(Widget w, 
	     XEvent *event, 
	     String *params, 
	     Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListFocusOut() action\n");

    XtCallActionProc(w, "PrimitiveFocusOut", event, params, *num_params);
}

static void
ListKbdActivate(Widget w, 
		XEvent *event, 
		String *params, 
		Cardinal *num_params)
{

    XdbDebug(__FILE__, w, "ListKbdActivate() action\n");

    _XmListInvokeCallbacks(w, event, True);
}

static void
ListKbdBeginExtend(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListKbdBeginExtend()\n");

    /**starving**/
}

static void
ListKbdBeginSelect(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListKbdBeginSelect()\n");

    /**starving**/
}

static void
ListKbdCancel(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListKbdCancel()\n");

    /**starving**/
}

static void
ListKbdDeselectAll(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListKbdDeselectAll()\n");

    /**starving**/
}

static void
ListKbdEndExtend(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListKbdEndExtend()\n");

    /**starving**/
}

static void
ListKbdEndSelect(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListKbdEndSelect()\n");

    /**starving**/
}

static void
ListKbdSelectAll(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListKbdSelectAll()\n");

    /**starving**/
}

static void
ListLeftChar(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListLeftChar()\n");

    XmListSetHorizPos(w, List_Horigin(w) - List_CharWidth(w));
}

static void
ListLeftPage(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListLeftPage()\n");

    /**starving**/
}

static void
ListNextItem(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    int position;

    XdbDebug(__FILE__, w, "ListNextItem()\n");

    position = List_LastHLItem(w) + 1;

    if(position >= List_ItemCount(w))
	return;

    if(position > List_TopPosition(w) + List_VisibleItemCount(w)) {
	++List_TopPosition(w);
	_XmListSetVertScrollBar(w);
    }

    _XmListSetItem(w, event, position);
}

static void
ListNextPage(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    int position;

    XdbDebug(__FILE__, w, "ListNextPage()\n");

    position = List_TopPosition(w) + List_VisibleItemCount(w);

    if(position < List_ItemCount(w) - List_VisibleItemCount(w))
	List_TopPosition(w) = position;
    else {
        List_TopPosition(w) = List_ItemCount(w) -
	    List_VisibleItemCount(w);
	if(position < List_ItemCount(w))
	    position = List_TopPosition(w);
	else
	    position = List_ItemCount(w) - 1;
    }

    _XmListSetVertScrollBar(w);
    _XmListSetItem(w, event, position);
}

static void
ListPrevItem(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    int position;

    XdbDebug(__FILE__, w, "ListPrevItem()\n");

    position = List_LastHLItem(w) - 1;

    if(position < 0)
	return;

    if(position < List_TopPosition(w)) {
	List_TopPosition(w) = position;
	_XmListSetVertScrollBar(w);
    }

    _XmListSetItem(w, event, position);
}

static void
ListPrevPage(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    int position;
    
    XdbDebug(__FILE__, w, "ListPrevPage()\n");

    position = List_TopPosition(w) - List_VisibleItemCount(w);

    if(position < 0)
	position = 0;

    if(position < List_TopPosition(w)) {
	List_TopPosition(w) = position;
	_XmListSetVertScrollBar(w);
    }

    _XmListSetItem(w, event, position);
}

static void
ListProcessDrag(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    /* CHECK TO SEE IF THE EXPORT TARGET IS REALLY COMPOUND_TEXT -- FIX ME ! */
    int position;
    Atom export_target;
    Arg args[10];
    int n = 0;
    Widget dc;
    
    XdbDebug(__FILE__, w, "ListProcessDrag()\n");

    /* If the list is empty, they couldn't really have dragged from it,
       now could they? */
    if (List_ItemCount(w) == 0)
	return;

    /* set up some stuff */
    export_target = XmInternAtom(XtDisplay(w),
				 "COMPOUND_TEXT", 
				 False);

    XtSetArg(args[n], XmNexportTargets, &export_target); n++;
    XtSetArg(args[n], XmNnumExportTargets, 1); n++;
    XtSetArg(args[n], XmNdragOperations, XmDROP_COPY); n++;
    XtSetArg(args[n], XmNclientData, w); n++;

    /* determine the item that they dragged from */
    position = XmListYToPos(w, event->xbutton.y + List_Vorigin(w));
    
    if (position > List_ItemCount(w))
	position = List_ItemCount(w);

    /* now decide whether the item was selected or not.  If it was, we want
       to transfer all the selected items.  If not, we only want to transfer
       the item they dragged from.

       position is +1 since XmListYToPos can return 0 */

    if (XmListPosSelected(w, position + 1))
    {
	/* the position was selected.  drag all the selected positions */
	XtSetArg(args[n], XmNconvertProc, drag_selected_proc); n++;
    }
    else
    {
	/* the position wasn't selected.  drag only the one item */
	XtSetArg(args[n], XmNconvertProc, drag_unselected_proc); n++;
    }
    dc = XmDragStart(w, event, args, n);

    XtAddCallback (dc, XmNdragDropFinishCallback, drag_drop_finish, NULL);
}

static void
ListRightChar(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListRightChar()\n");

    XmListSetHorizPos(w, List_Horigin(w) + List_CharWidth(w));
}

static void
ListRightPage(Widget w, XEvent *event, String *params, Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "ListRightPage()\n");


    /**starving**/
}


Widget
XmCreateList(Widget parent, char *name, Arg *arglist, Cardinal argCount)
{
    return XtCreateWidget(name, xmListWidgetClass, parent, arglist, argCount);
}

Widget
XmCreateScrolledList(Widget parent, char *name, Arg *arglist, Cardinal argCount)
{
    Widget sw;
    char *sname;
    int i;
    Arg *al;
    
    sname = XtMalloc(strlen(name)+3);
    strcpy(sname, name);
    strcat(sname, "SW");

    al = (Arg *)XtCalloc(argCount + 4, sizeof(Arg));
    for (i=0; i<argCount; i++)
    {
      al[i].name = arglist[i].name;
      al[i].value = arglist[i].value;
    }

    XtSetArg(al[i], XmNscrollingPolicy, XmAPPLICATION_DEFINED); i++;
    XtSetArg(al[i], XmNvisualPolicy, XmVARIABLE); i++;
    XtSetArg(al[i], XmNscrollBarDisplayPolicy, XmSTATIC); i++;
    XtSetArg(al[i], XmNshadowThickness, 0); i++;

    sw = XtCreateManagedWidget(sname, xmScrolledWindowWidgetClass, parent, al, i);
    XtFree((XtPointer)al);

    return XtCreateWidget(name, xmListWidgetClass, sw, arglist, argCount);
}

void
XmListAddItem(Widget w, 
	      XmString item, 
	      int position)
{
    XdbDebug(__FILE__, w, "XmListAddItem()\n");

    if (position < 0)
      /* do some warning type stuff here... */
      position = 0;

    _XmListAddItemUnselected(w, item, position);
    _XmListSelectPosIfMatch(w, position);
    _XmListRecalcItems(w);
    _XmListSetVertScrollBar(w);
    _XmListRedraw(w, True);
}

void
XmListAddItems(Widget w, 
	       XmString *items, 
	       int item_count, 
	       int position)
{
    int i;
    
    XdbDebug(__FILE__, w, "XmListAddItem\n");

    if(!position--)
	position = List_ItemCount(w) - 1;
    
    _XmListAddItemsUnselected(w, items, item_count, position);
    for(i=0; i<item_count; ++i)
      _XmListSelectPosIfMatch(w, position+i);
    _XmListRecalcItems(w);
    _XmListRedraw(w, True);
}

void
XmListAddItemUnselected(Widget w, XmString item, int position)
{
    XdbDebug(__FILE__, w, "XmListAddItemUnselected()\n");

    _XmListAddItemUnselected(w, item, position);
    _XmListSetGeometry(w);
    _XmListRecalcItems(w);
    _XmListRedraw(w, True);
}

void
XmListAddItemsUnselected(Widget w, XmString *items, int item_count, int position)
{
    XdbDebug(__FILE__, w, "XmListAddItemsUnselected()\n");

    _XmListAddItemsUnselected(w, items, item_count, position);
    _XmListSetGeometry(w);
    _XmListRecalcItems(w);
    _XmListRedraw(w, True);
}

void
XmListDeleteAllItems(Widget w)
{
    int i;
    
    XdbDebug(__FILE__, w, "XmListDeleteAllItems()\n");
    
    for( i = 0; i < List_ItemCount(w); ++i )
	XmStringFree(List_Items(w)[i]);

    List_ItemCount(w) = List_SelectedItemCount(w) = 0;
    
    _XmListSetGeometry(w);
    _XmListRecalcItems(w);
    _XmListRedraw(w, True);
}

void
XmListDeleteItem(Widget w, XmString item)
{
    XdbDebug(__FILE__, w, "XmListDeleteItem()\n");

    if(_XmListDeleteItem(w, item)) {
	_XmListSetGeometry(w);
        _XmListRecalcItems(w);
	_XmListRedraw(w, True);
    } else
	XtWarning("XmListDeleteItem: item not found in list.\n");
}

void
XmListDeleteItems(Widget w, XmString *items, int item_count)
{
    Boolean need_refresh = False;
    int i;
    
    XdbDebug(__FILE__, w, "XmListDeleteItems()\n");
    
    for( i = 0; i < item_count; ++i )
	need_refresh |= _XmListDeleteItem(w, items[i]);

    if( need_refresh ) {
	_XmListSetGeometry(w);
        _XmListRecalcItems(w);
	_XmListRedraw(w, True);
    }
}

void
XmListDeleteItemsPos(Widget w, int item_count, int position)
{
    int i = 0;
    
    XdbDebug(__FILE__, w, "XmListDeleteItemsPos()\n");
    
    if( position > List_ItemCount(w) ) {
	XtWarning("XmDeleteItemsPos: position exceeds itemCount.\n");
	return;
    }

    if(!position--)
	position = List_ItemCount(w) - 1;

    while(i++ < item_count && i < List_ItemCount(w) )
	_XmListDeletePos(w, position);
    
    _XmListSetGeometry(w);
    _XmListRecalcItems(w);
    _XmListRedraw(w, True);
}

void
XmListDeletePos(Widget w, int position)
{
    XdbDebug(__FILE__, w, "XmListDeletePos()\n");
    
    if( position > List_ItemCount(w) ) {
	XtWarning("XmDeletePos: position exceeds itemCount.\n");
	return;
    }

    if(!position--)
	position = List_ItemCount(w) - 1;

    _XmListDeletePos(w, position);
    _XmListSetGeometry(w);
    _XmListRecalcItems(w);
    _XmListRedraw(w, True);
}

void
XmListDeletePositions(Widget w, int *position_list, int position_count)
{
    int i, j;
    
    XdbDebug(__FILE__, w, "XmListDeletePositions()\n");
    
    for(i=0; i<position_count; ++i)
	position_list[i] = (position_list[i] ? position_list[i] :
			    List_ItemCount(w)) - 1;

    for(i=List_ItemCount(w); i>=0; --i) {
	for(j=0; j<position_count; ++j)
	    if(position_list[j]==i) {
		_XmListDeletePos(w, i);
		break;
	    }
    }
    
    _XmListSetGeometry(w);
    _XmListRecalcItems(w);
    _XmListRedraw(w, True);
}

void
XmListDeselectAllItems(Widget w)
{
    int i;
    
    XdbDebug(__FILE__, w, "XmListDeselectAllItems()\n");
  
    for (i = 0; i < List_ItemCount(w); i++)
        if (List_InternalList(w)[i]->selected)
        {
            List_InternalList(w)[i]->selected = False;
            List_InternalList(w)[i]->LastTimeDrawn = False;
        } 

    for (i = 0; i < List_SelectedItemCount(w); i++ )
	XmStringFree(List_SelectedItems(w)[i]);

    List_SelectedItemCount(w) = 0;

    _XmListRecalcItems(w);
    _XmListRedraw(w, False);
}

void
XmListDeselectItem(Widget w, XmString item)
{
    int i;
    
    XdbDebug(__FILE__, w, "XmListDeselectItem()\n");
   
    for (i = 0; i < List_ItemCount(w); i++)
	if (XmStringCompare(item, List_InternalList(w)[i]->name)) 
            XmListDeselectPos(w, i+1);
}

void
XmListDeselectPos(Widget w, int position)
{
    XdbDebug(__FILE__, w, "XmListDeselectPos()\n");

    if(!position--)
	position = List_ItemCount(w) - 1;

    if(_XmListDeselectPos(w, position)) 
    {
        _XmListRecalcItems(w);
	_XmListRedraw(w, False);
    } 
    else
	XtWarning("XmListDeselectPos: item not found in selectedItems.\n");
}

int
XmListGetKbdItemPos(Widget w)
{
    XdbDebug(__FILE__, w, "XmListGetKbdItemPos()\n");
    
    return List_LastHLItem(w);
}


Boolean
XmListGetMatchPos(Widget w, XmString item, int **position_list, int *position_count)
{
    int i;

    XdbDebug(__FILE__, w, "XmListGetMatchPos()\n");
    
    *position_count = 0;

    for(i=0; i<List_ItemCount(w); ++i)
	if(XmStringCompare(item, List_Items(w)[i])) {
	    if(!(*position_count)++)
		*position_list = (int*)XtMalloc(sizeof(int));
	    else
		*position_list = (int*)XtRealloc((XtPointer)*position_list,
						 *position_count*sizeof(int));
	    (*position_list)[*position_count-1] = i+1;
	}

    return *position_count ? True : False;
}

Boolean
XmListGetSelectedPos(Widget w, int **position_list, int *position_count)
{
#if	0
    int i;
#endif

    XdbDebug(__FILE__, w, "XmListGetSelectPos()\n");
    
    *position_count = 0;

#if 0
    for(i=0; i<List_ItemCount(w); ++i)
	if(w->list.selected[i])
	    ++(*position_count);
#endif

    if(!*position_count)
	return False;

    *position_list = (int*)XtMalloc(*position_count*sizeof(int));
    *position_count = 0;

#if 0
    for(i=0; i<List_ItemCount(w); ++i)
	if(w->list.selected[i])
	    (*position_list)[(*position_count)++] = i+1;
#endif
   
    return True;
}

Boolean
XmListItemExists(Widget w, XmString item)
{
    int i;

    XdbDebug(__FILE__, w, "XmListItemExists()\n");
    
    for(i=0; i<List_ItemCount(w); ++i)
	if(XmStringCompare(item, List_Items(w)[i]))
	    return True;

    return False;
}

int
XmListItemPos(Widget w, XmString item)
{
    int i;

    XdbDebug(__FILE__, w, "XmListItemPos()\n");
    
    for(i=0; i<List_ItemCount(w); ++i)
	if(XmStringCompare(item, List_Items(w)[i]))
	    return i+1;

    return 0;
}

Boolean
XmListPosSelected(Widget w, int position)
{
    XdbDebug(__FILE__, w, "XmListPosSelected()\n");

    if( position < 0 || position > List_ItemCount(w) )
	return False;

    if(!position--)
	position = List_ItemCount(w) - 1;

#if 0
    return w->list.selected[position];
#else
    return False;
#endif
}

Boolean
XmListPosToBounds(Widget w, 
		  int position, 
		  Position *x, 
		  Position *y,	
		  Dimension *width,
		  Dimension *height)
{
    Dimension highlight = (List_HighlightThickness(w) 
			   ? List_HighlightThickness(w) 
			   : 1);

    XdbDebug(__FILE__, w, "XmListPosToBounds()\n");

    if (!position)
	position = List_ItemCount(w);

    if (position < List_TopPosition(w) 
	|| position >= List_TopPosition(w) + List_VisibleItemCount(w))
	return False;

    if (x)
	*x = (List_MarginWidth(w) 
	      + Prim_ShadowThickness(w) 
	      + highlight);

    if (width)
	*width = List_InternalList(w)[position - 1]->width;

    if (height)
	*height = List_InternalList(w)[position - 1]->height;

    if (y)
	*y = ((Prim_ShadowThickness(w) 
	       + List_MarginHeight(w) 
	       + highlight)
	      + (List_InternalList(w)[position - 1]->CumHeight 
		 - List_InternalList(w)[position - 1]->height
		 - List_Vorigin(w)));
	     
    return True;
}

void
XmListReplaceItems(Widget w, XmString *old_items, int item_count, XmString *new_items)
{
#if	0
    int i, j;
#endif

    XdbDebug(__FILE__, w, "XmListReplaceItems()\n");

#if 0
    for(i=0; i<List_ItemCount(w); ++i)
	for(j=0; j<List_ItemCount(w); ++j)
	    if(XmStringCompare(old_items[i], List_Items(w)[j])) {
		need_refresh = True;
		(void)_XmListDeselectPos(w, j);
		XmStringFree(List_Items(w)[j]);
		List_Items(w)[j] = XmStringCopy(new_items[i]);
		_XmListSelectPosIfMatch(w, j);
	    }
    
    if(need_refresh)
    {
        _XmListRecalcItems(w);
	_XmListRedraw(w, True);
    }
#endif
}

void
XmListReplaceItemsPos(Widget w, XmString *new_items, int item_count, int position)
{
#if	0
    int i, j = (position ? position : List_ItemCount(w)) - 1;
#endif

    XdbDebug(__FILE__, w, "XmListReplaceItemsPos()\n");

#if 0
    for(i=0; i<item_count && j<List_ItemCount(w); ++i) 
    {
	(void)_XmListDeselectPos(w, j);
	XmStringFree(List_Items(w)[j]);
	List_Items(w)[j] = XmStringCopy(new_items[i]);
	_XmListSelectPosIfMatch(w, j);
	j++;
    }

    _XmListRecalcItems(w);
    _XmListRedraw(w, True);
#endif
}

void
XmListReplaceItemsPosUnselected(Widget w, XmString *new_items, int item_count, int position)
{
    int i;

    XdbDebug(__FILE__, w, "XmListReplaceItemsPosUnselected()\n");

    if(!position--)
	position = List_ItemCount(w) - 1;

    for(i=0; i<item_count && position<List_ItemCount(w); ++i) {
	(void)_XmListDeselectPos(w, position);
	XmStringFree(List_Items(w)[position]);
	List_Items(w)[position] = XmStringCopy(new_items[i]);
    }
  
    _XmListRecalcItems(w);
    _XmListRedraw(w, True);
}

void
XmListReplaceItemsUnselected(Widget w, XmString *old_items, int item_count, XmString *new_items)
{
    Boolean need_refresh = False;
    int i, j;
    
    XdbDebug(__FILE__, w, "XmListReplaceItemsUnselected()\n");
    
    for(i=0; i<item_count; ++i)
	for(j=0; j<List_ItemCount(w); ++j)
	    if(XmStringCompare(old_items[i], List_Items(w)[j])) {
		need_refresh = True;
		(void)_XmListDeselectPos(w, j);
		XmStringFree(List_Items(w)[j]);
		List_Items(w)[j] = XmStringCopy(new_items[i]);
	    }
    
    if(need_refresh)
    {
        _XmListRecalcItems(w);
	_XmListRedraw(w, True);
    }
}

void
XmListReplacePositions(Widget w, int *position_list, XmString *item_list, int item_count)
{
    int i, j;

    XdbDebug(__FILE__, w, "XmListReplacePosition()\n");
    
    for(i=0; i<item_count; ++i ) {
	j = (position_list[i] ? position_list[i] : List_ItemCount(w)) - 1;
	(void)_XmListDeselectPos(w, j);
	List_Items(w)[j] = XmStringCopy(item_list[i]);
	_XmListSelectPosIfMatch(w, j);
    }

    _XmListRecalcItems(w);
    _XmListRedraw(w, True);
}

void
XmListSelectItem(Widget w, XmString item, Boolean notify)
{
    int i;

    XdbDebug(__FILE__, w, "XmListSelectItem()\n");

    for(i=0; i<List_ItemCount(w); ++i )
	if(XmStringCompare(item, List_Items(w)[i])) {
	    XmListSelectPos(w,i+1,notify);
	    return;
	}
}

void
XmListSelectPos(Widget w, int position, Boolean notify)
{
    XdbDebug(__FILE__, w, "XmListSelectPos()\n");

    if(!position--)
	position = List_ItemCount(w) - 1;

    _XmListSelectPos(w, position);
    _XmListRecalcItems(w);
    _XmListRedraw(w, True);

    if( notify ) {
	XAnyEvent *event = (XAnyEvent*)XtMalloc(sizeof(XAnyEvent));
	event->type = 0;
	event->serial = 0;
	event->send_event = False;
	event->display = XtDisplay(w);
	event->window = XtWindow(w);

#if 0	
	_XmListInvokeCallbacks(w,_XmCR_SELECT,(XEvent*)event,XmINITIAL);
#endif

	XtFree((XtPointer)event);
    }
}

void
XmListSetAddMode(Widget w, Boolean mode)
{
    XdbDebug(__FILE__, w, "XmListSetAddMode()\n");

    if(List_SelectionPolicy(w) == XmEXTENDED_SELECT)
	List_AddMode(w) = mode;
}

void
XmListSetBottomItem(Widget w, XmString item)
{
    Boolean need_refresh = False;
    int i;
    
    XdbDebug(__FILE__, w, "XmListSetBottomItem()\n");
    
    for(i=0; i<List_SelectedItemCount(w); ++i)
	if(XmStringCompare(item, List_Items(w)[i])) {
	    List_TopPosition(w) = i - List_VisibleItemCount(w) + 1;
	    need_refresh = True;
	    break;
	}

    if(need_refresh)
    {
        _XmListRecalcItems(w);
	_XmListRedraw(w, True);
    }
}

void
XmListSetBottomPos(Widget w, int position)
{
    XdbDebug(__FILE__, w, "XmListSetBottomPos()\n");
    
    if( !position )
	position = List_ItemCount(w) - 1;

    List_TopPosition(w) = position - List_VisibleItemCount(w);

    _XmListRecalcItems(w);
    _XmListRedraw(w, True);
}

void
XmListSetHorizPos(Widget w, int position)
{
    XdbDebug(__FILE__, w, "XmListSetHorizPos()      List_MaxWidth() = %d\n", List_MaxWidth(w));
    
    if(!List_IsScrolledList(w) || !XtIsManaged(List_HSB(w)))
	return;
   
    List_Horigin(w) = position;
    List_Hmax(w) = (List_MaxWidth(w) 
                    - XtWidth(w) 
                    - 2 * (Prim_ShadowThickness(w)
                           + List_HighlightThickness(w)));
    List_Hmin(w) = 0;
    List_Hextent(w) = (XtWidth(w) 
                       - 2 * (Prim_ShadowThickness(w)
                              + List_HighlightThickness(w)));

    XtVaSetValues((Widget)List_HSB(w),
		  XmNmaximum, List_Hmax(w),
		  XmNminimum, List_Hmin(w),
		  XmNsliderSize, List_Hextent(w),
		  XmNvalue, List_Horigin(w),
		  NULL);
}

void
XmListSetItem(Widget w, XmString item)
{
    Boolean need_refresh = False;
    int i;

    XdbDebug(__FILE__, w, "XmListSetItem()\n");
    
    for(i=0; i<List_SelectedItemCount(w); ++i)
	if(XmStringCompare(item, List_Items(w)[i])) {
	    List_TopPosition(w) = i;
	    need_refresh = True;
	    break;
	}

    if(need_refresh)
    {
        _XmListRecalcItems(w);
	_XmListRedraw(w, True);
    }
}

Boolean
XmListSetKbdItemPos(Widget w, int position)
{
    XdbDebug(__FILE__, w, "XmListSetKbdItemPos()\n");

    if( position < 0 || position >= List_ItemCount(w) ||
       !List_ItemCount(w) )
	return False;
    
    if( !position-- )
	position = List_ItemCount(w) - 1;

    List_LastHLItem(w) = position;

    _XmListRecalcItems(w);
    _XmListRedraw(w, True);

    return True;
}

void
XmListSetPos(Widget w, int position)
{
    XdbDebug(__FILE__, w, "XmListSetPos()\n");
    
    if( !position )
	position = List_ItemCount(w) - 1;

    List_TopPosition(w) = position;

    _XmListRecalcItems(w);
    _XmListRedraw(w, True);
}

void
XmListUpdateSelectedList(Widget w)
{
#if	0
    int i;
#endif

    XdbDebug(__FILE__, w, "XmListUpdateSelectedList()\n");
    
    List_SelectedItemCount(w) = 0;
#if 0
    for(i=0; i<List_ItemCount(w); ++i)
	if(w->list.selected[i])
	    ++List_SelectedItemCount(w);
#endif
    List_SelectedItems(w) =
	(XmStringTable)XtRealloc((XtPointer)List_SelectedItems(w),
				 List_SelectedItemCount(w) *
				 sizeof(XmString));

    List_SelectedItemCount(w) = 0;
#if 0
    for(i=0; i<List_ItemCount(w); ++i)
	if(w->list.selected[i])
	    List_SelectedItems(w)[List_SelectedItemCount(w)++] =
		XmStringCopy(List_Items(w)[i]);
#endif
    _XmListRecalcItems(w);
    _XmListRedraw(w, True);
}

int
XmListYToPos(Widget w, Position y)
{
    int pos;
    Dimension highlight = (List_HighlightThickness(w) 
                           ? List_HighlightThickness(w) 
                           : 1);

    XdbDebug(__FILE__, w, "XmListYToPos()\n");

    if (List_ItemCount(w) == 0)
	return 0;

    for (pos = List_TopPosition(w) - 1; 
         pos < List_TopPosition(w) + List_VisibleItemCount(w) - 1; 
         pos ++)
    {
	if (y < (List_InternalList(w)[pos]->CumHeight 
                 + highlight 
                 + List_ItemSpacing(w)))
	    return pos + 1;
    }

    return List_VisibleItemCount(w);
}

/* FIX ME
 * This routine determines the preferred size of List.
 * It is currently wrong because it assumes that all visible list items
 * are of the same size. This is false because items are XmStrings which
 * may display in different fonts.
 *
 * We could also wonder about the approach to take :
 *	- use the items that are currently visible to determine our size
 * or
 *	- make sure we can always display XmNvisibleItemCount of the largest items.
 * The former seems correct...
 */
static XtGeometryResult
query_geometry(Widget w, XtWidgetGeometry *proposed, XtWidgetGeometry *answer)
{
    Dimension highlight = List_HighlightThickness(w) ? List_HighlightThickness(w) : 1;

    if (List_ItemCount(w) == 0 || List_VisibleItemCount(w) == 0) {
	answer->request_mode = CWWidth | CWHeight;
	answer->width = 100;
	answer->height = 100;	/* FIX ME should be based on fontList */
    } else {
	answer->request_mode = CWWidth | CWHeight;
	answer->width = (List_MaxWidth(w) + 2 * (List_MarginWidth(w) 
		+ List_HighlightThickness(w) + Prim_ShadowThickness(w)));

	answer->height = ((List_MaxItemHeight(w) + highlight) 
		* List_VisibleItemCount(w) 
		+ List_ItemSpacing(w) * (List_VisibleItemCount(w) - 1)
		+ highlight 
		+ 2 * (List_MarginHeight(w) + Prim_ShadowThickness(w)));
    }

    XdbDebug(__FILE__, w, "query_geometry  => W %d H %d (items %d visible %d)\n",
		answer->width, answer->height,
		List_ItemCount(w), List_VisibleItemCount(w));

    if (proposed->width >= answer->width && proposed->height >= answer->height) 
	return XtGeometryYes;
    else if (answer->width == XtWidth(w) && answer->height == XtHeight(w)) {
	if (answer)
	    answer->request_mode = 0;
	return XtGeometryNo;
    } else 
	return XtGeometryAlmost;    
}

static void
list_border_highlight(Widget w)
{ 
    Dimension highlight = (List_HighlightThickness(w) 
			   ? List_HighlightThickness(w) 
			   : 1);

    XdbDebug(__FILE__, w, "list_border_highlight()\n");

    if (List_LastHLItem(w) > (List_TopPosition(w)
			      + List_VisibleItemCount(w))
	|| List_LastHLItem(w) < List_TopPosition(w))
    {
	int i;

	for (i=0; i < highlight; i++)
	    XDrawRectangle(XtDisplay(w),
			   XtWindow(w),
			   List_HighlightGC(w),
			   Prim_ShadowThickness(w) + i,
			   Prim_ShadowThickness(w) + i,
			   (XtWidth(w) 
			    - 1 
			    - List_MarginWidth(w) 
			    - 2 * Prim_ShadowThickness(w) 
			    - i),
			   (XtHeight(w) 
			    - 1 
			    - List_MarginHeight(w) 
			    - 2 * Prim_ShadowThickness(w) 
			    - i));

    }
    else
    {
	
    }
}


static void
list_border_unhighlight(Widget w)
{
    Dimension highlight = (List_HighlightThickness(w) 
			   ? List_HighlightThickness(w) 
			   : 1);

    XdbDebug(__FILE__, w, "list_border_unhighlight()\n");

    if (List_LastHLItem(w) > (List_TopPosition(w) 
			      + List_VisibleItemCount(w))
	|| List_LastHLItem(w) < List_TopPosition(w)) 
    {
	int i;

	for (i=0; i < highlight; i++)
	    XDrawRectangle(XtDisplay(w),
			   XtWindow(w),
			   List_HighlightGC(w),
			   Prim_ShadowThickness(w) + i,
			   Prim_ShadowThickness(w) + i,
			   (XtWidth(w) 
			    - 1 
			    - List_MarginWidth(w) 
			    - 2 * Prim_ShadowThickness(w) 
			    - i), 
			   (XtHeight(w) 
			    - 1 
			    - List_MarginHeight(w) 
			    - 2 * Prim_ShadowThickness(w) 
			    - i));
    }
    else
    {
	
    }
}

static void 
ListLeave(Widget w, 
	  XEvent *event, 
	  String *params, 
	  Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "XmListLeave()\n");

    XtCallActionProc(w, "PrimitiveLeave", event, params, *num_params);    
}

static void 
ListEnter(Widget w, 
	  XEvent *event, 
	  String *params, 
	  Cardinal *num_params)
{
    XdbDebug(__FILE__, w, "XmListEnter()\n");

    XtCallActionProc(w, "PrimitiveEnter", event, params, *num_params);    
}

static Boolean
drag_selected_proc(Widget w, 
		   Atom *selection, 
		   Atom *target, 
		   Atom *type_return, 
		   XtPointer *value_return, 
		   unsigned long *length_return, 
		   int *format_return)
{
    Atom COMPOUND_TEXT;
    Atom MOTIF_DROP;
    
    COMPOUND_TEXT = XmInternAtom(XtDisplay(w), "COMPOUND_TEXT", False);
    MOTIF_DROP = XmInternAtom(XtDisplay(w), "_MOTIF_DROP", False);
    
    if (*selection != MOTIF_DROP)
	return False;
    
    XdbDebug(__FILE__, w, "We're dealing with a motif drop\n");

    return False;
}

static Boolean
drag_unselected_proc(Widget w, 
		     Atom *selection, 
		     Atom *target, 
		     Atom *type_return, 
		     XtPointer *value_return, 
		     unsigned long *length_return, 
		     int *format_return)
{
    Atom COMPOUND_TEXT;
    Atom MOTIF_DROP;

    COMPOUND_TEXT = XmInternAtom(XtDisplay(w), "COMPOUND_TEXT", False);
    MOTIF_DROP = XmInternAtom(XtDisplay(w), "_MOTIF_DROP", False);

    if (*selection != MOTIF_DROP)
	return False;

    XdbDebug(__FILE__, w, "We're dealing with a motif drop\n");

    return False;
}

static void 
drag_drop_finish(Widget w, 
			XtPointer client_data, 
			XtPointer call_data)
{
    Widget source_icon = NULL;

    XtVaGetValues(w, XmNsourceCursorIcon, &source_icon, NULL);

    if (source_icon)
	XtDestroyWidget(source_icon);
}
