/**
 *
 * $Id: Text.c,v 1.20 1996/04/23 17:40:42 miers Exp $
 *
 * Copyright (C) 1995 Free Software Foundation, Inc.
 *
 * This file is part of the GNU LessTif Library.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **/

 static char rcsid[] = "$Id: Text.c,v 1.20 1996/04/23 17:40:42 miers Exp $";

#define	DO_MOSAIC_HACK
#define	I18N

#include <LTconfig.h>
#include <Xm/XmP.h>
#include <Xm/DebugUtil.h>
#include <Xm/TextP.h>
#include <Xm/ScrolledW.h>
#include <Xm/XmI.h>
#include <stdio.h>
#include <limits.h> /* for INT_MAX */
#ifdef HAVE_STRING_H
#include <string.h>
#define ANSI_STRING
#else
#include <strings.h>
#endif
#include <stdlib.h>

#ifdef	DO_MOSAIC_HACK
#include <Xm/TextF.h>
#endif

/* Forward Declarations */

static char * _delete(char *s,int pos, int len);
static Boolean _inword(char s);
static void _DrawString(XmTextWidget tw);
static void _DrawCursor(XmTextWidget tw);
static void class_initialize();
static void class_part_initialize(WidgetClass class);
static void initialize(Widget request, Widget new, ArgList args, Cardinal *num_args);
static void destroy(Widget w);
static void realize(Widget w, XtValueMask *value_mask, XSetWindowAttributes *attributes);
static void expose(Widget w, XEvent *event, Region region);
static XtGeometryResult query_geometry(Widget w, XtWidgetGeometry *proposed, XtWidgetGeometry *answer);
static void _XmTextSetEditable(Widget w, Boolean e);

static Boolean set_values(Widget current, Widget request, Widget new, ArgList args, Cardinal *num_args);

#define Offset(_name) XtOffsetOf(XmTextRec, text._name)

/* Resources for the Text  class */
static XtResource resources[] = { 
    {
	XmNsource, XmCSource, XmRPointer,
	sizeof(XmTextSource), Offset(source),
	XmRPointer, (XtPointer)NULL /* FIXME: Motif has a wacky value here */
    },
    {
	XmNactivateCallback, XmCCallback, XmRCallback,
	sizeof(XtCallbackList), Offset(activate_callback),
	XmRCallback, (XtPointer)NULL
    },
    {
	XmNfocusCallback, XmCCallback, XmRCallback,
	sizeof(XtCallbackList), Offset(focus_callback),
	XmRCallback, (XtPointer)NULL
    },
    {
	XmNlosingFocusCallback, XmCCallback, XmRCallback,
	sizeof(XtCallbackList), Offset(losing_focus_callback),
	XmRCallback, (XtPointer)NULL
    },
    {
	XmNvalueChangedCallback, XmCCallback, XmRCallback,
	sizeof(XtCallbackList), Offset(value_changed_callback),
	XmRCallback, (XtPointer)NULL
    },
    {
	XmNmodifyVerifyCallback, XmCCallback, XmRCallback,
	sizeof(XtCallbackList), Offset(modify_verify_callback),
	XmRCallback, (XtPointer)NULL
    },
    {
	XmNmodifyVerifyCallbackWcs, XmCCallback, XmRCallback,
	sizeof(XtCallbackList), Offset(wcs_modify_verify_callback),
	XmRCallback, (XtPointer)NULL
    },
    {
	XmNmotionVerifyCallback, XmCCallback, XmRCallback,
	sizeof(XtCallbackList), Offset(motion_verify_callback),
	XmRCallback, (XtPointer)NULL
    },
    {
	XmNgainPrimaryCallback, XmCCallback, XmRCallback,
	sizeof(XtCallbackList), Offset(gain_primary_callback),
	XmRCallback, (XtPointer)NULL
    },
    {
	XmNlosePrimaryCallback, XmCCallback, XmRCallback,
	sizeof(XtCallbackList), Offset(lose_primary_callback),
	XmRCallback, (XtPointer)NULL
    },
    {
	XmNvalue, XmCValue, XmRString,
	sizeof(String), Offset(value),
	XmRString, (XtPointer)NULL /* FIXME: Motif has a wacky value here */
    },
    {
	XmNvalueWcs, XmCValueWcs, XmRValueWcs,
	sizeof(wchar_t*), Offset(wc_value),
	XmRString, (XtPointer)NULL
    },
    {
	XmNmaxLength, XmCMaxLength, XmRInt,
	sizeof(int), Offset(max_length),
	XmRImmediate, (XtPointer)INT_MAX
    },
    {
	XmNmarginHeight, XmCMarginHeight, XmRVerticalDimension,
	sizeof(Dimension), Offset(margin_height),
	XmRImmediate, (XtPointer)5
    },
    {
	XmNmarginWidth, XmCMarginWidth, XmRHorizontalDimension,
	sizeof(Dimension), Offset(margin_width),
	XmRImmediate, (XtPointer)5
    },
    {
        XmNoutputCreate, XmCOutputCreate, XmRFunction,
	sizeof(OutputCreateProc), Offset(output_create),
        XmRFunction, (XtPointer)_XmTextOutputCreate
    },
    {
        XmNinputCreate, XmCInputCreate, XmRFunction,
	sizeof(InputCreateProc), Offset(input_create),
        XmRFunction, (XtPointer)_XmTextInputCreate
    },
    {
	XmNtopCharacter, XmCTopCharacter, XmRTextPosition,
	sizeof(XmTextPosition), Offset(top_character),
	XmRImmediate, (XtPointer) 0
    },
    {
	XmNcursorPosition, XmCCursorPosition, XmRTextPosition,
	sizeof(XmTextPosition), Offset(cursor_position),
	XmRImmediate, (XtPointer)0
    },
    {
	XmNeditMode, XmCEditMode, XmREditMode,
	sizeof(int), Offset(edit_mode),
	XmRImmediate, (XtPointer) XmSINGLE_LINE_EDIT
    },
    {
	XmNautoShowCursorPosition, XmCAutoShowCursorPosition, XmRBoolean,
	sizeof(Boolean), Offset(auto_show_cursor_position),
	XmRImmediate, (XtPointer)True
    },
    {
	XmNeditable, XmCEditable, XmRBoolean,
	sizeof(Boolean), Offset(editable),
	XmRImmediate, (XtPointer)True
    },
    {
	XmNverifyBell, XmCVerifyBell, XmRBoolean,
	sizeof(Boolean), Offset(verify_bell),
	XtRImmediate, (XtPointer)((unsigned char)XmUNSPECIFIED)
    },
    {
	XmNnavigationType, XmCNavigationType, XmRNavigationType,
	sizeof(XmNavigationType), XtOffsetOf(XmTextRec, primitive.navigation_type),
	XmRImmediate, (XtPointer)XmTAB_GROUP
    }
};

static XmSyntheticResource syn_resources[] = {
    {
	XmNmarginWidth,
	sizeof(Dimension), Offset(margin_width),
	_XmFromHorizontalPixels, _XmToHorizontalPixels
    },
    {
	XmNmarginHeight,
	sizeof(Dimension), Offset(margin_height),
	_XmFromVerticalPixels, _XmToVerticalPixels
    }
};

static void activate();
static void backward_character();
static void backward_paragraph();
static void backward_word();
static void beep();
static void beginning_of_file();
static void beginning_of_line();
static void clear_selection();
static void copy_clipboard();
static void copy_primary();
static void copy_to();
static void cut_clipboard();
static void cut_primary();
static void delete_next_character();
static void delete_previous_character();
static void delete_next_word(Widget, XEvent *, String *, Cardinal *);
static void delete_previous_word(Widget, XEvent *, String *, Cardinal *);
static void delete_selection();
static void delete_to_end_of_line();
static void delete_to_start_of_line();
static void delete_all();
static void do_quick_action();
static void end_of_file();
static void end_of_line();
static void extend_adjust();
static void extend_end();
static void extend_start();
static void forward_character();
static void forward_paragraph();
static void forward_word();
static void grab_focus();
static void focus_in(Widget, XEvent *, String *, Cardinal*);
static void focus_out(Widget, XEvent *, String *, Cardinal*);
static void Help();
static void insert_string();
static void key_select();
static void kill_next_character();
static void kill_next_word();
static void kill_previous_character();
static void kill_previous_word();
static void kill_selection();
static void kill_to_end_of_line();
static void kill_to_start_of_line();
static void move_destination();
static void move_to();
static void newline();
static void newline_and_backup();
static void newline_and_indent();
static void next_line();
static void next_page();
static void next_tab_group();
static void page_left();
static void page_right();
static void paste_clipboard();
static void prev_tab_group();
static void previous_line();
static void previous_page();
static void process_bdrag();
static void process_cancel();
static void process_down();
static void process_up();
static void process_home();
static void process_return();
static void process_shift_down();
static void process_shift_up();
static void process_tab();
static void quick_copy_set();
static void quick_cut_set();
static void redraw_display();
static void scroll_one_line_down();
static void scroll_one_line_up();
static void secondary_adjust();
static void secondary_notify();
static void secondary_start();
static void select_adjust();
static void select_all();
static void select_end();
static void select_start();
static void self_insert();
static void set_anchor();
static void set_insertion_point();
static void set_selection_hint();
static void toggle_add_mode();
static void traverse_home();
static void traverse_next();
static void traverse_prev();
static void unkill();


static char defaultTranslations[] =
   "Ctrl<Key>A:         beginning-of-line()\n"\
   "Ctrl<Key>B:         backward-character()\n"\
   "Ctrl<Key>C:         paste-clipboard()\n"\
   "Ctrl<Key>D:         delete-next-character()\n"\
   "Ctrl<Key>E:		end-of-line()\n"\
   "Ctrl<Key>F:         forward-character()\n"\
   "Ctrl<Key>H:         delete-previous-character()\n"\
   "Ctrl<Key>J:		newline-and-indent()\n"\
   "Ctrl<Key>K:         kill-to-end-of-line()\n"\
   "Ctrl<Key>L:         redraw-display()\n"\
   "Ctrl<Key>M:		newline()\n"\
   "Ctrl<Key>N:		next-line()\n"\
   "Ctrl<Key>O:		newline-and-backup()\n"\
   "Ctrl<Key>P:		previous-line()\n"\
   "Ctrl<Key>V:		next-page()\n"\
   "Ctrl<Key>W:         kill-selection()\n"\
   "Meta<Key>B:         backward-word()\n"\
   "Meta<Key>F:         forward-word()\n"\
   "Meta<Key>V:         previous-page()\n"\
   ":Meta<Key>d:	delete-next-word()\n"\
   ":Meta<Key>h:	delete-previous-word()\n"\
   "<Key>Left:          backward-character()\n"\
   "<Key>Right:         forward-character()\n"\
   "<Key>Up:		previous-line()\n"\
   "<Key>Down:		next-line()\n"\
   "<Key>End:           end-of-line()\n"\
   "<Key>Home:          beginning-of-line()\n"\
   "<Key>Return:        process-return()\n"\
   "<Key>BackSpace:     delete-previous-character()\n"\
   "<Key>Delete:        delete-next-character()\n"\
   "<Key>:              self-insert()\n"\
   "<EnterWindow>:	enter-window()\n"\
   "<LeaveWindow>:	leave-window()\n"\
   "<Key>osfActivate:	activate()\n"\
   "<FocusIn>:		focus-in()\n"\
   "<FocusOut>:		focus-out()\n"\
   "<Btn1Down>:		select-start()\n"\
   "<Btn1Motion>:	extend-adjust()\n"\
   "<Btn1Up>:		select-end()\n"\
   "<Btn2Down>:		copy-primary()\n";

/* action table table */

static XtActionsRec actions[] = {
{"activate",activate},
{"backward-character",backward_character},
{"backward-paragraph",backward_paragraph},
{"backward-word",backward_word},
{"beep",beep},
{"beginning-of-file",beginning_of_file},
{"beginning-of-line",beginning_of_line},
{"clear-selection",clear_selection},
{"copy-clipboard",copy_clipboard},
{"copy-primary",copy_primary},
{"copy-to",copy_to},
{"cut-clipboard",cut_clipboard},
{"cut-primary",cut_primary},
{"delete-next-character",delete_next_character},
{"delete-previous-character",delete_previous_character},
{"delete-next-word",delete_next_word},
{"delete-previous-word",delete_previous_word},
{"delete-selection",delete_selection},
{"delete-to-end-of-line",delete_to_end_of_line},
{"delete-to-start-of-line",delete_to_start_of_line},
{"delete-all",delete_all},
{"do-quick-action",do_quick_action},
{"end-of-file",end_of_file},
{"end-of-line",end_of_line},
{"extend-adjust",extend_adjust},
{"extend-end",extend_end},
{"extend-start",extend_start},
{"forward-character",forward_character},
{"forward-paragraph",forward_paragraph},
{"forward-word",forward_word},
{"focus-in", focus_in},
{"focus-out", focus_out},
{"grab-focus",grab_focus},
{"Help",Help},
{"insert-string",insert_string},
{"key-select",key_select},
{"kill-next-character",kill_next_character},
{"kill-next-word",kill_next_word},
{"kill-previous-character",kill_previous_character},
{"kill-previous-word",kill_previous_word},
{"kill-selection",kill_selection},
{"kill-to-end-of-line",kill_to_end_of_line},
{"kill-to-start-of-line",kill_to_start_of_line},
{"move-destination",move_destination},
{"move-to",move_to},
{"newline",newline},
{"newline-and-backup",newline_and_backup},
{"newline-and-indent",newline_and_indent},
{"next-line",next_line},
{"next-page",next_page},
{"next-tab-group",next_tab_group},
{"page-left",page_left},
{"page-right",page_right},
{"paste-clipboard",paste_clipboard},
{"prev-tab-group",prev_tab_group},
{"previous-line",previous_line},
{"previous-page",previous_page},
{"process-bdrag",process_bdrag},
{"process-cancel",process_cancel},
{"process-down",process_down},
{"process-up",process_up},
{"process-home",process_home},
{"process-return",process_return},
{"process-shift_down",process_shift_down},
{"process-shift-up",process_shift_up},
{"process-tab",process_tab},
{"quick-copy-set",quick_copy_set},
{"quick-cut-set",quick_cut_set},
{"redraw-display",redraw_display},
{"scroll-one-line-down",scroll_one_line_down},
{"scroll-one-line-up",scroll_one_line_up},
{"secondary-adjust",secondary_adjust},
{"secondary-notify",secondary_notify},
{"secondary-start",secondary_start},
{"select-adjust",select_adjust},
{"select-all",select_all},
{"select-end",select_end},
{"select-start",select_start},
{"self-insert",self_insert},
{"set-anchor",set_anchor},
{"set-insertion-point",set_insertion_point},
{"set-selection-hint",set_selection_hint},
{"toggle-add-mode",toggle_add_mode},
{"traverse-home",traverse_home},
{"traverse-next",traverse_next},
{"traverse-prev",traverse_prev},
{"unkill",unkill},
};

static XmBaseClassExtRec _XmTextCoreClassExtRec = {
    /* next_extension            */ NULL,
    /* record_type               */ NULLQUARK,                             
    /* version                   */ XmBaseClassExtVersion,
    /* size                      */ sizeof(XmBaseClassExtRec),
    /* initialize_prehook        */ NULL, /* FIXME */
    /* set_values_prehook        */ NULL, /* FIXME */
    /* initialize_posthook       */ NULL, /* FIXME */
    /* set_values_posthook       */ NULL, /* FIXME */
    /* secondary_object_class    */ NULL, /* FIXME */
    /* secondary_object_create   */ NULL, /* FIXME */
    /* get_secondary_resources   */ NULL, /* FIXME */
    /* fast_subclass             */ { 0 }, /* FIXME */
    /* get_values_prehook        */ NULL, /* FIXME */
    /* get_values_posthook       */ NULL, /* FIXME */
    /* class_part_init_prehook   */ NULL,
    /* class_part_init_posthook  */ NULL,
    /* ext_resources             */ NULL,
    /* compiled_ext_resources    */ NULL,
    /* num_ext_resources         */ 0,
    /* use_sub_resources         */ FALSE,
    /* widget_navigable          */ XmInheritWidgetNavigable,
    /* focus_change              */ XmInheritFocusChange,
    /* wrapper_data              */ NULL
};

XmPrimitiveClassExtRec _XmTextPrimClassExtRec = {
    /* next_extension      */ NULL,
    /* record_type         */ NULLQUARK,
    /* version             */ XmPrimitiveClassExtVersion,
    /* record_size         */ sizeof(XmPrimitiveClassExtRec),
    /* widget_baseline     */ NULL, /* FIX ME */
    /* widget_display_rect */ NULL, /* FIX ME */
    /* widget_margins      */ NULL  /* FIX ME */
};

XmTextClassRec xmTextClassRec = {
    /* Core class part */
    {
	/* superclass            */ (WidgetClass) &xmPrimitiveClassRec,
        /* class_name            */ "XmText",
	/* widget_size           */ sizeof(XmTextRec),
	/* class_initialize      */ class_initialize,
	/* class_part_initialize */ class_part_initialize,
	/* class_inited          */ FALSE,
	/* initialize            */ initialize,
	/* initialize_hook       */ NULL,
	/* realize               */ realize,
	/* actions               */ actions,
	/* num_actions           */ XtNumber(actions),
	/* resources             */ resources,
	/* num_resources         */ XtNumber(resources),
	/* xrm_class             */ NULLQUARK,
	/* compress_motion       */ TRUE,
	/* compress_exposure     */ XtExposeCompressMaximal,
	/* compress_enterleave   */ TRUE,
	/* visible_interest      */ FALSE,
	/* destroy               */ destroy,
	/* resize                */ NULL,
	/* expose                */ expose,
	/* set_values            */ set_values,
	/* set_values_hook       */ NULL,
	/* set_values_almost     */ XtInheritSetValuesAlmost,
	/* get_values_hook       */ NULL,
	/* accept_focus          */ NULL,
	/* version               */ XtVersion,
	/* callback offsets      */ NULL,
	/* tm_table              */ defaultTranslations,
	/* query_geometry        */ query_geometry,
	/* display_accelerator   */ XtInheritDisplayAccelerator,
	/* extension             */ (XtPointer)&_XmTextCoreClassExtRec
    },
    /* Primitive Class part */
    {
	/* border_highlight      */ XmInheritBorderHighlight,
       	/* border_unhighlight    */ XmInheritBorderUnhighlight,
       	/* translations          */ NULL,
       	/* arm_and_activate_proc */ XmInheritArmAndActivate,
       	/* synthetic resources   */ syn_resources, 
        /* num syn res           */ XtNumber(syn_resources),
	/* extension             */ (XtPointer)&_XmTextPrimClassExtRec,
    },
    /* Text Class part */
    {
	/* extension */ NULL
    }
};

WidgetClass xmTextWidgetClass = (WidgetClass)&xmTextClassRec;

extern Boolean _XmCvtStringToXmString(Display *, XrmValue *, Cardinal *, XrmValue *, XrmValue *, XtPointer *);
extern XmFontList _XmFontListCreateDefault(Display *);

static void
class_initialize()
{
    _XmTextCoreClassExtRec.record_type = XmQmotif;
}

static void
class_part_initialize(WidgetClass widget_class)
{
    _XmFastSubclassInit(widget_class, XmTEXT_BIT);
}

static void
initialize(Widget request,
	   Widget new,
	   ArgList args,
	   Cardinal *num_args)
{
    XmTextWidget tw=(XmTextWidget) new;
    Dimension width, height;
    XFontStruct *fs;   
    XmFontListEntry entry;

    if (Text_Value(tw) == (char *)XmUNSPECIFIED || Text_Value(tw) == NULL) {
	XdbDebug(__FILE__, (Widget)tw,
		 "XmText %s initialize: XmUNSPECIFIED value -> 0x%X\n",
		 XtName(new), Text_Value(tw));
	Text_Value(tw)=XtNewString("");
    }

    tw->text.fontList = _XmGetDefaultFontList(new, XmTEXT_FONTLIST);

    entry = _XmFontListEntryFromTag(tw->text.fontList, XmFONTLIST_DEFAULT_TAG);
    fs = (XFontStruct*)entry->font;

    height=fs->ascent;

    width = 20 * (fs->max_bounds.rbearing - fs->min_bounds.lbearing);
    XtWidth(tw) = (2 * Prim_ShadowThickness(tw) + 
		   2 *Text_MarginWidth(tw) +
		   width);
    
    XtHeight(tw) = (2 * Prim_ShadowThickness(tw) +
		    2 * Text_MarginHeight(tw) +
		    height);

    tw->text.length = strlen(Text_Value(tw));

    tw->text.myGC = XtAllocateGC(new,0,0,NULL,0,0);
}

static void
destroy(Widget w)
{
    XtReleaseGC(w,((XmTextWidget)w)->text.myGC);
}

#define	CHECK_CLASS(w)	XmTextCheckClass(w, __FILE__, __LINE__)

static void
XmTextCheckClass(Widget w, char *f, int l)
{
    if (! XtIsSubclass(w, xmTextWidgetClass)) {
	XdbDebug(__FILE__, w, "Don't take LessTif for a fool !\n");
	XdbDebug(__FILE__, w,
		 "XmTextField != XmText. Widget %s misused at %s %d\n",
		 XtName(w), f, l);
    }
}

static Boolean
set_values(Widget old,
	   Widget request,
	   Widget new,
	   ArgList args,
	   Cardinal *num_args)
{
  Boolean	refresh_needed = False;
  XmTextWidget	nw = (XmTextWidget) new;
  XmTextWidget	ow = (XmTextWidget) old;

  XdbDebug(__FILE__, new, "XmText %s SetValues\n", XtName(new));
  
  if (Text_Value(nw) != Text_Value(ow)) {
      XtFree(Text_Value(ow));
      Text_Value(nw) = XtNewString(Text_Value(nw) ? Text_Value(nw) : "");
      nw->text.length = strlen(Text_Value(nw));
      refresh_needed = True;
  }
  
  if (Text_Editable(nw) != Text_Editable(ow)) {
	_XmTextSetEditable(new, Text_Editable(nw));
	refresh_needed = True;
  }
  /* FIX ME - lots more cases */

  return refresh_needed;
}

static void
realize(Widget w, XtValueMask *value_mask, XSetWindowAttributes *attributes)
{
    XmTextWidget tw=(XmTextWidget) w;

#define superclass (&xmPrimitiveClassRec)
    (*superclass->core_class.realize)(w, value_mask, attributes);
#undef superclass

    XSetForeground(XtDisplay(tw),tw->text.myGC,XtBackground(tw));

    _XmTextSetEditable(w, Text_Editable(w));

/* Hack to give this thing a size until there is a real output object */
    if (XtWidth(w) == 0 || XtHeight(w) == 0) {
	XtWidth(w) = 200;
	XtHeight(w) = 100;
    }
}


static void
expose(Widget w,
       XEvent *event,
       Region region)
{
    XmTextWidget tw = (XmTextWidget) w;

    if (!XtIsRealized(w))
	return;

    XFillRectangle(XtDisplay(tw),XtWindow(tw),tw->text.myGC,
		  Prim_ShadowThickness(tw), Prim_ShadowThickness(tw),
		  XtWidth(tw) -2* Prim_ShadowThickness(tw),
		  XtHeight(tw) -2* Prim_ShadowThickness(tw));

    XSetForeground(XtDisplay(tw),
		   tw->text.myGC,
		   Prim_Foreground(tw));

    _DrawString(tw); 
    _DrawCursor(tw); 

    _XmDrawShadows(XtDisplay(w),
		   XtWindow(w),
		   Prim_TopShadowGC(tw),
		   Prim_BottomShadowGC(tw),
		   0,0,
		   XtWidth(tw), XtHeight(tw),
		   Prim_ShadowThickness(tw),
		   XmSHADOW_IN);

    XSetForeground(XtDisplay(tw), tw->text.myGC, XtBackground(tw));

#define superclass (&xmPrimitiveClassRec)
    (*superclass->core_class.expose)(w, event, region);
#undef superclass
}




static void _DrawString(XmTextWidget tw)
{
    int String_X,String_Y;
    XFontStruct *fs;
    XmFontListEntry entry;
    int start=Text_TopCharacter(tw);
    int end=Text_TopCharacter(tw);
    int row;
    int image=0;
    unsigned long foreground;
    XGCValues gcvals;
    int oldStringX;

    if (Text_CursorPosition(tw) > tw->text.length)
	Text_CursorPosition(tw) = tw->text.length;

    entry = _XmFontListEntryFromTag(tw->text.fontList, XmFONTLIST_DEFAULT_TAG);
    fs = (XFontStruct*)entry->font;

    String_X = Prim_ShadowThickness(tw) + Text_MarginWidth(tw);
    oldStringX = String_X; 
    String_Y = fs->ascent + Text_MarginHeight(tw) + Prim_ShadowThickness(tw);

    if(Text_CursorPosition(tw)==0)
    {
	tw->text.cursorXPosition = Prim_ShadowThickness(tw) +
				   Text_MarginWidth(tw);
	tw->text.cursorYPosition=String_Y; 
	tw->text.cursorYTop=String_Y - fs->ascent;
    }
    if(Text_Value(tw)[end]=='\n')
    {
	start++;
	String_Y += fs->ascent + fs->descent;
    }

    while (start<tw->text.length) {
	do {
	    end++;
	    if (tw->text.selectbegin == end) {
		XDrawString(XtDisplay(tw),
			    XtWindow(tw),
			    tw->text.myGC,
			    String_X, String_Y,
			    Text_Value(tw) + start, end-start);
		XGetGCValues(XtDisplay(tw),
			     tw->text.myGC,
			     GCForeground|GCBackground,
			     &gcvals);
		foreground = gcvals.foreground;
		gcvals.foreground = gcvals.background;
		gcvals.background = foreground;
/* 		XChangeGC(XtDisplay(tw),tw->text.myGC,GCForeground|GCBackground, */
/* 			  &gcvals); */
		String_X = XTextWidth(fs,
				      Text_Value(tw) + start,
				      end-start) + String_X;
		start = end;
		image = 1;
	    }
	    if(tw->text.selectend == end) {
		XGetGCValues(XtDisplay(tw),
			     tw->text.myGC,
			     GCForeground|GCBackground,
			     &gcvals);
		XDrawImageString(XtDisplay(tw),
				 XtWindow(tw),
				 tw->text.myGC,
				 String_X, String_Y,
				 Text_Value(tw) + start, end - start);
		foreground = gcvals.foreground;
		gcvals.foreground = gcvals.background;
		gcvals.background = foreground;
/* 		XChangeGC(XtDisplay(tw),tw->text.myGC,GCForeground|GCBackground, */
/* 			  &gcvals); */
		image = 0;
		String_X=XTextWidth(fs,
				    Text_Value(tw) + start,
				    end-start) + String_X;
		start = end;
	    }
	    if (Text_CursorPosition(tw) == end) {
		tw->text.cursorXPosition = XTextWidth(fs,
						      Text_Value(tw) + start,
						      end - start) + String_X;
		tw->text.cursorYPosition = String_Y; 
		tw->text.cursorYTop = String_Y - fs->ascent;
	    }
	} while ((end < tw->text.length) && (Text_Value(tw)[end] != '\n'));
	end++;
	if (image)
	    XDrawImageString(XtDisplay(tw),
			     XtWindow(tw),
			     tw->text.myGC, 
			     String_X, String_Y,
			     Text_Value(tw) + start, end - start - 1); 
	else	
	    XDrawString(XtDisplay(tw),
			XtWindow(tw),
			tw->text.myGC, 
			String_X, String_Y,
			Text_Value(tw) + start, end - start - 1); 
	start = end; 
	String_Y += fs->ascent + fs->descent;
	if (Text_Value(tw)[end] == '\n') {
	    start++;
	    String_Y += fs->ascent + fs->descent;
	}
	if(Text_CursorPosition(tw) == end)
	{
	    tw->text.cursorXPosition = Prim_ShadowThickness(tw) +
				       Text_MarginWidth(tw);
	    tw->text.cursorYPosition = String_Y; 
	    tw->text.cursorYTop = String_Y - fs->ascent;
	}
	row++;
	String_X = oldStringX;
    }
}



static void _DrawCursor(XmTextWidget tw)
{
    int i;

    if (tw->text.cursorYTop > XtHeight(tw)) {
	for(i = 0; (Text_Value(tw)[i] != '\n') && (i < tw->text.length); i++)
	    ;
	i++;
	Text_TopCharacter(tw) = i;
    }
    else {
	XDrawLine(XtDisplay(tw),
		  XtWindow(tw),
		  tw->text.myGC,
		  tw->text.cursorXPosition, tw->text.cursorYTop,
		  tw->text.cursorXPosition, tw->text.cursorYPosition);
    }
}



static XtGeometryResult 
query_geometry(Widget w, XtWidgetGeometry *proposed, XtWidgetGeometry *answer)
{
    XmTextWidget tw = (XmTextWidget)w;
    XtWidgetGeometry a;

#define Wants(x)   (proposed->request_mode & x)

    if (proposed->request_mode != 0) {	/* NULL case should not yet end here ! */
	if ((! (Wants(CWWidth))) && (! Wants(CWHeight))) {
		/* If they don't ask width/height, let them have whatever they like */
	    if (answer)
		*answer = *proposed;
	    return XtGeometryYes;
	}
    }

    a.request_mode = CWWidth | CWHeight;
    a.width = XtWidth(tw);
    a.height = XtHeight(tw);

    if (answer)
	*answer = a;

    if (proposed->request_mode == 0)	/* NULL proposed -> return Width+Height */
	return XtGeometryAlmost;

    if (proposed->width >= answer->width && proposed->height >= answer->height) 
	return XtGeometryYes;
    else if (answer->width == XtWidth(tw) && answer->height == XtHeight(tw)) {
	if (answer)
	    answer->request_mode = 0;
	return XtGeometryNo;
    } else 
	return XtGeometryAlmost;
}

Widget 
XmCreateText(Widget parent,
	     char *name,
	     Arg *arglist,
	     Cardinal argCount)
{
    return XtCreateWidget(name, xmTextWidgetClass, parent, arglist, argCount);
}

Widget 
XmCreateScrolledText(Widget parent,
		     char *name,
		     Arg *arglist,
		     Cardinal argcount)
{
    Widget	sw;
    char	*sname;
    Arg		a;

    sname = XtMalloc(strlen(name)+4);
    strcpy(sname, name);
    strcat(sname, "_SW");

    sw = XtCreateWidget(sname, xmScrolledWindowWidgetClass, parent, arglist, argcount);

/* No merge operation: we're sure we can use SetValues on this without harm. */
    XtSetArg(a, XmNscrollingPolicy, XmAPPLICATION_DEFINED);
    XtSetValues(sw, &a, 1);

    XtManageChild(sw);

    return XtCreateWidget(name, xmTextWidgetClass, sw, arglist, argcount);
}

void
XmTextClearSelection(Widget widget, Time time)
{  /* DO ME */
}

Boolean
XmTextCopy(Widget widget, Time time)
/* DO ME */{
    return False;
}

Boolean
XmTextCut(Widget widget, Time time)
/* DO ME */{
    return False;
}

Boolean
XmTextFindString(Widget widget,
                 XmTextPosition start,
                 char *string,
                 XmTextDirection direction,
                 XmTextPosition *position)
{
    return False;
}

int
XmTextGetBaseline(Widget widget)
{
    XmTextWidget tw=(XmTextWidget) widget;
    XFontStruct *fs;
    XmFontListEntry entry;

    entry = _XmFontListEntryFromTag(tw->text.fontList, XmFONTLIST_DEFAULT_TAG);
    fs = (XFontStruct*)entry->font;

    return(fs->ascent + Text_MarginHeight(tw) + Prim_ShadowThickness(tw));
}


XmTextPosition
XmTextGetCursorPosition(Widget widget)
{
    return(Text_CursorPosition(widget));
}


Boolean
XmTextGetEditable(Widget widget)
{
    return(Text_Editable(widget));
}

XmTextPosition
XmTextGetInsertionPosition(Widget widget)
{
    return(Text_CursorPosition(widget));
}

XmTextPosition
XmTextGetLastPosition(Widget widget)
{
    return(Text_LastPosition(widget));
}

int
XmTextGetMaxLength(Widget widget)
{
    return(Text_MaxLength(widget));
}

char *XmTextGetSelection(Widget widget)
/* DO ME */{
    return NULL;
}

Boolean
XmTextGetSelectionPosition(Widget widget, XmTextPosition *left, XmTextPosition *right)
/* DO ME */{
    return False;
}

wchar_t *XmTextGetSelectionWcs(Widget widget)
/* DO ME */{
    return NULL;
}


char *XmTextGetString(Widget widget)
{
    return(XtNewString(Text_Value(widget)));
}

wchar_t *XmTextGetStringWcs(Widget widget)
{
    return (Text_WcsValue(widget));
}

/*
XmTextGetSubstring()

Gets a substring of the text 
*/
int
XmTextGetSubstring(Widget widget, XmTextPosition start, int num_chars, int buffer_size, char *buffer)
{
    int len;
    int retval=XmCOPY_SUCCEEDED;

    if(num_chars>buffer_size)
	len=num_chars;
    else
    {
	len=buffer_size-1;
	retval=XmCOPY_TRUNCATED;
    }
    strncpy(buffer,((XmTextWidget) widget)->text.value+start,len);
    buffer[len]='\0';
    
    return(retval);
}

int
XmTextGetSubstringWcs(Widget widget, XmTextPosition start, int num_chars, int buffer_size, wchar_t *buffer)
/* DO ME */{
    return 0;
}

XmTextPosition
XmTextGetTopCharacter(Widget widget)
{
  return (Text_TopCharacter(widget));
}

void
XmTextInsert(Widget w, XmTextPosition position, char *string)
{
    XmTextWidget	tw = (XmTextWidget) w;
    char		*newval;
    int			l;

    CHECK_CLASS(w);

    if (string == NULL)
	return;
    if (position < 0 || position > tw->text.length)
	return;

    l = strlen(string);
    newval = XtMalloc(l + tw->text.length + 1);

    strncpy(newval, Text_Value(tw), position);
    newval[position] = '\0';	/* on Linux, strncpy() doesn't add a 0 byte */
    strcat(newval,string);

    if (position < tw->text.length)
	strcat(newval, Text_Value(tw) + position);

/* cut off the string at the max length, if we have one */
    if(l + tw->text.length > Text_MaxLength(tw))
   	newval[Text_MaxLength(tw)]='\0';

    XtFree(Text_Value(tw));

    Text_Value(tw) = newval;
    tw->text.length += l;

    expose(w, NULL, (Region) NULL);
}

void
_XmTextInsert(Widget w, XmTextPosition position, char *string, XEvent *evp)
{
    XmTextWidget		tw = (XmTextWidget) w;
    int				l;
    XmTextVerifyCallbackStruct	cbs;
    XmTextBlockRec		rec;

    CHECK_CLASS(w);

    if (string == NULL)
	return;
    if (position < 0 || position > tw->text.length)
	return;

    l = strlen(string);
/*
 * Do ModifyVerifyCallback, but since this is a very frequent operation,
 * first check whether there's something in the callback list
 */
    cbs.doit = True;

    if (XtHasCallbacks(w, XmNmodifyVerifyCallback) == XtCallbackHasSome) {
	cbs.event = evp;
	cbs.reason = XmCR_MODIFYING_TEXT_VALUE;
	cbs.currInsert = Text_CursorPosition(tw);
	if (position < Text_CursorPosition(tw))
	    cbs.newInsert = Text_CursorPosition(tw);
	else
	    cbs.newInsert = Text_CursorPosition(tw) + l;
	cbs.startPos = cbs.endPos = position;
	cbs.text = &rec;

	rec.ptr = string;
	rec.length = l;
	rec.format = XmFMT_8_BIT;

	XtCallCallbackList(w, tw->text.modify_verify_callback, (XtPointer)&cbs);
    }

    if (cbs.doit)
	XmTextInsert(w, position, string);
}

void
XmTextInsertWcs(Widget widget, XmTextPosition position, wchar_t *wcstring)
/* DO ME */{
}

Boolean
XmTextPaste(Widget widget)
/* DO ME */{
    return(False);
}

Boolean
XmTextPosToXY(Widget widget, XmTextPosition position, Position *x, Position *y)
{
    XmTextWidget me=(XmTextWidget) widget;
    int i;
    int start=0;
    int line=0;
    XFontStruct *fs;
    XmFontListEntry entry;

    entry = _XmFontListEntryFromTag(me->text.fontList, XmFONTLIST_DEFAULT_TAG);
    fs = (XFontStruct*)entry->font;

    for(i=0;i<me->text.length;i++)
    {
	if(i=='\n')
	{
	    line++;
	    start=i;
	}
	if(i==position)
	{
	    *x= (fs->ascent + fs->descent) * line + fs->ascent +
	        Prim_ShadowThickness(me) + Text_MarginHeight(me);
	    *y= XTextWidth(fs, Text_Value(me) + start, i - start)+
                Prim_ShadowThickness(me) + Text_MarginWidth(me);
	    return(True);
	}
    }
    return(False);

}

Boolean
XmTextRemove(Widget widget)
/* DO ME */{
    return(False);
}

void
XmTextReplace(Widget widget, XmTextPosition from_pos, XmTextPosition to_pos, char *value)
{
/* DO ME */}

void
XmTextReplaceWcs(Widget widget, XmTextPosition from_pos, XmTextPosition to_pos, wchar_t *wcstring)
{
/* DO ME */}

void
XmTextSetAddMode(Widget widget, Boolean state)
{
/* DO ME */}

void
XmTextSetCursorPosition(Widget widget, XmTextPosition position)
{
    Text_CursorPosition(widget) = position;
    expose(widget, NULL, (Region) NULL);
}

static void
_XmTextSetEditable(Widget w, Boolean e)
{
    if (! XtIsRealized(w))
	return;

#ifdef	I18N
    if (e) {	/* becomes editable */
	Arg	args[10];
	int	nargs;

	XmImRegister(w, 0);

	nargs = 0;
	XtSetArg(args[nargs], XmNbackground, XtBackground(w)); nargs++;
	XtSetArg(args[nargs], XmNforeground, Prim_Foreground(w)); nargs++;
	XmImSetValues(w, args, nargs);
  } else {	/* Becomes un-editable */
	XmImUnregister(w);
  }
#endif
}

void
XmTextSetEditable(Widget w, Boolean editable)
{
    if (! XmIsText(w))
	return;

    if (Text_Editable(w) != editable)
	_XmTextSetEditable(w, editable);

    Text_Editable(w) = editable;
}

void
XmTextSetHighlight(Widget widget, XmTextPosition left, XmTextPosition right, XmHighlightMode mode)
{
/* DO ME */}

void
XmTextSetInsertionPosition(Widget widget, XmTextPosition position)
{
    CHECK_CLASS(widget);

    Text_CursorPosition(widget) = position;
    expose(widget, NULL, (Region) NULL);
}

void
XmTextSetMaxLength(Widget widget, int max_length)
{
    XmTextWidget tw=(XmTextWidget) widget;

    CHECK_CLASS(widget);

    Text_MaxLength(tw)=max_length;

    if(tw->text.length>max_length)
    {
	tw->text.length=max_length;
	Text_Value(tw)[max_length]='\0';
    }
}

void
XmTextSetSelection(Widget widget, XmTextPosition first, XmTextPosition last, Time time)
/* DO ME */{
}

void
XmTextSetString(Widget widget, char *value)
{
    XmTextWidget tw=(XmTextWidget) widget;

    CHECK_CLASS(widget);

#ifdef	DO_MOSAIC_HACK
/* This is a hack for Mosaic to work ! */
    if (XtIsSubclass(widget, xmTextFieldWidgetClass))
	XmTextFieldSetString(widget, value);
    else
#endif
    {
	XtFree(Text_Value(tw));

	Text_Value(tw)=XtNewString(value);
	tw->text.length = strlen(value);

	Text_CursorPosition(tw) = 0;
    }
}

void
XmTextDisableRedisplay(Widget w)
{
	XdbDebug(__FILE__, w, "XmTextDisableRedisplay\n");
}

void
XmTextEnableRedisplay(Widget w)
{
	XdbDebug(__FILE__, w, "XmTextEnableRedisplay\n");
}


XmTextSource
XmTextGetSource(Widget w)
{
	XdbDebug(__FILE__, w, "XmTextGetSource\n");

    return NULL;
}

void
XmTextSetSource(Widget w, XmTextSource s, XmTextPosition top, XmTextPosition curs)
{
	XdbDebug(__FILE__, w, "XmTextSetSource\n");
}

void
XmTextScroll(Widget w, int n)
{
	XdbDebug(__FILE__, w, "XmTextScroll\n");
}

void
XmTextSetStringWcs(Widget widget, wchar_t *wcstring)
/* DO ME */{
}

void
XmTextSetTopCharacter(Widget widget, XmTextPosition top_character)
{
    CHECK_CLASS(widget);
}

void
XmTextShowPosition(Widget widget, XmTextPosition position)
/* DO ME */{
}

XmTextPosition
XmTextXYToPos(Widget widget, Position x, Position y)
{
    XmTextWidget me=(XmTextWidget) widget;
    int i;
    XFontStruct *fs;
    XmFontListEntry entry;

    entry = _XmFontListEntryFromTag(me->text.fontList, XmFONTLIST_DEFAULT_TAG);
    fs = (XFontStruct*)entry->font;

    y -= Prim_ShadowThickness(me) + Text_MarginHeight(me);
    y /= (fs->ascent + fs->descent);
    for (i = Text_TopCharacter(me); (i < me->text.length) && (y > 0); i++) {
	if (Text_Value(me)[i] == '\n')
	    y--;
    }
    if (y > 0)
	return 0;
    x -= Prim_ShadowThickness(me) + Text_MarginWidth(me);
    while ((x > 0) && (i < me->text.length)) {
	i++;
	x-=XTextWidth(fs,me->text.value+i,1);
    }
    i--;
    if(i < 0)
	i = 0;
    if (x > 0)
	return 0;
    return(i);

}


/*-------------------------------------------------------
action routines
-------------------------------------------------------*/

static void
activate(Widget w,XEvent *ev,String *params, Cardinal *num_params)
{
  XmTextWidget tw = (XmTextWidget) w;
  XmAnyCallbackStruct cb;
  
  if (Text_EditMode(tw) != XmSINGLE_LINE_EDIT)
	return;

  cb.reason = XmCR_ACTIVATE;
  cb.event = ev;

  XtCallCallbackList(w, tw->text.activate_callback, (XtPointer)&cb);
  if(XmIsManager(XtParent(w))){
      /* FIXME: Send event to manager. How do we do this ? XSendEvent ? */
  }
}

static void backward_character(Widget w,XEvent *ev,String *params,
                        Cardinal *num_params)
{
    XmTextWidget me=(XmTextWidget) w;
    if(Text_CursorPosition(me)>0)
        Text_CursorPosition(me)--;
    expose(w,NULL,(Region)NULL);
}

static void backward_paragraph(Widget w,XEvent *ev,String *params,
                        Cardinal *num_params)
{
    XmTextWidget me=(XmTextWidget) w;
    while((Text_CursorPosition(me)>0)&&
	  (me->text.value[Text_CursorPosition(me)]!='\n')&&
	  (me->text.value[Text_CursorPosition(me)-1]!='\n'))
	  Text_CursorPosition(me)--;
    while((Text_CursorPosition(me)>0)&&
	  (me->text.value[Text_CursorPosition(me)]=='\n'))
	  Text_CursorPosition(me)--;
    expose(w,NULL,(Region)NULL);
}

static void backward_word(Widget w,XEvent *ev, String *params, 
                          Cardinal *num_params)
{
    XmTextWidget me=(XmTextWidget) w;
    
    while((Text_CursorPosition(me)>0) &&
	 (_inword(me->text.value[Text_CursorPosition(me)])))
        Text_CursorPosition(me)--;
    while((Text_CursorPosition(me)>0) &&
	 (!_inword(me->text.value[Text_CursorPosition(me)])))
        Text_CursorPosition(me)--;
    expose(w,NULL,(Region)NULL);
}

static void beep()
/* DO ME */{
}

static void beginning_of_file(Widget w,XEvent *ev,String *params,
                        Cardinal *num_params)
{
    XmTextWidget me=(XmTextWidget) w;
    Text_CursorPosition(me)=0;
    expose(w,NULL,(Region)NULL);
}

static void beginning_of_line(Widget w,XEvent *ev,String *params,
                        Cardinal *num_params)
{
    XmTextWidget me=(XmTextWidget) w;
    while(Text_CursorPosition(me) >= 0 && 
	  me->text.value[Text_CursorPosition(me)]!='\n')
	Text_CursorPosition(me)--;
    Text_CursorPosition(me)++;
    expose(w,NULL,(Region)NULL);
}

static void clear_selection()
/* DO ME */{
}

static void copy_clipboard()
/* DO ME */{
}

static void copy_primary()
/* DO ME */{
}

static void copy_to()
/* DO ME */{
}

static void cut_clipboard()
/* DO ME */{
}

static void cut_primary()
/* DO ME */{
}

static void delete_next_character(Widget w,XEvent *ev,String *params,
                        Cardinal *num_params)
{
    XmTextWidget me=(XmTextWidget) w;

    if(!Text_Editable(me))  /* if we can't edit, just return */
	return;
    if(Text_CursorPosition(me)<me->text.length)
    {
	me->text.value=_delete(me->text.value,
				     Text_CursorPosition(me),1);
   	me->text.length--; 
        expose(w,NULL,(Region)NULL);
    }
}

static void delete_previous_character(Widget w,XEvent *ev,String *params,
                        Cardinal *num_params)
{
    XmTextWidget me=(XmTextWidget) w;

    if(!Text_Editable(me))  /* if we can't edit, just return */
	return;
    if(Text_CursorPosition(me)>0)
    {
	me->text.value=_delete(me->text.value,
				   Text_CursorPosition(me)-1,1);
        backward_character(w,ev,params,num_params);
        me->text.length--;
        expose(w,NULL,(Region)NULL);
    }
}

static void delete_next_word(Widget w, XEvent *ev, String *params, Cardinal *num_params)
/* DO ME */{

}

static void delete_previous_word(Widget w, XEvent *ev, String *params, Cardinal *num_params)
/* DO ME */{
}
static void delete_selection()
/* DO ME */{
}

static void delete_to_end_of_line( Widget w,XEvent *ev,String *params,
                        Cardinal *num_params)
{
  XmTextWidget me=(XmTextWidget) w;
  if(!Text_Editable(me))  /* if we can't edit, just return */
    return;
  while(Text_CursorPosition(me) < me->text.length &&  
	me->text.value[Text_CursorPosition(me)]!='\n'){
    me->text.value=_delete(me->text.value,
			   Text_CursorPosition(me),1); 
    me->text.length--;
  }
  expose(w,NULL,(Region)NULL);
}

static void delete_to_start_of_line( Widget w,XEvent *ev,String *params,
                        Cardinal *num_params)
{
    

}

static void delete_all(Widget w,XEvent *ev,String *params,
                        Cardinal *num_params)
{
    XmTextWidget me=(XmTextWidget) w;
    
    if(!Text_Editable(me))  /* if we can't edit, just return */
	return;
   
    me->text.length=0;
    Text_CursorPosition(me)=0;
    expose(w,NULL,(Region)NULL);
}

static void do_quick_action()
/* DO ME */{
}

static void end_of_line(Widget w,XEvent *ev,String *params,
                        Cardinal *num_params)
{
    XmTextWidget me=(XmTextWidget) w;
    while(Text_CursorPosition(me) <= me->text.length && 
	  me->text.value[Text_CursorPosition(me)]!='\n')
	Text_CursorPosition(me)++;
    expose(w,NULL,(Region)NULL);
}

static void end_of_file(Widget w,XEvent *ev,String *params,
                        Cardinal *num_params)
{
    XmTextWidget me=(XmTextWidget) w;
    Text_CursorPosition(me)=me->text.length;
    expose(w,NULL,(Region)NULL);
}


static void extend_adjust(Widget w, XEvent *ev, String *params,
			    Cardinal *num_params)
{
    XmTextWidget me=(XmTextWidget) w;

    if(ev->type!=MotionNotify)
	return;

    Text_CursorPosition(me)=XmTextXYToPos(w,ev->xmotion.x,ev->xmotion.y);

    if(Text_CursorPosition(me)<me->text.selectbegin)
    {
	me->text.selectend=me->text.selectbegin;
	me->text.selectbegin=Text_CursorPosition(me);
    }
    else
	me->text.selectend=Text_CursorPosition(me);
    if(me->text.selectend>=me->text.length)
	me->text.selectend=me->text.length-1;

    expose(w, NULL, (Region) NULL);

}


static void extend_end()
/* DO ME */{
}

static void extend_start()
/* DO ME */{
}

static void forward_character( Widget w,XEvent *ev,String *params,
                        Cardinal *num_params)
{
    XmTextWidget me=(XmTextWidget) w;
    if(Text_CursorPosition(me) < me->text.length)
    {
        Text_CursorPosition(me)++;
        expose(w,NULL,(Region)NULL);
    }
}

static void forward_paragraph( Widget w,XEvent *ev,String *params,
                        Cardinal *num_params)
{
    XmTextWidget me=(XmTextWidget) w;
    while((Text_CursorPosition(me)<me->text.length)&&
	  (me->text.value[Text_CursorPosition(me)]!='\n')&&
	  (me->text.value[Text_CursorPosition(me)+1]!='\n'))
	  Text_CursorPosition(me)++;
    while((Text_CursorPosition(me)<me->text.length)&&
	  (me->text.value[Text_CursorPosition(me)]=='\n'))
	  Text_CursorPosition(me)++;
    expose(w,NULL,(Region)NULL);
}


static void forward_word(Widget w,XEvent *ev,String *params,
                        Cardinal *num_params)
{
    XmTextWidget me=(XmTextWidget) w;
    
    while((Text_CursorPosition(me) < me->text.length) &&
	 (_inword(me->text.value[Text_CursorPosition(me)])))
        Text_CursorPosition(me)++;
    while((Text_CursorPosition(me) < me->text.length) &&
	 !(_inword(me->text.value[Text_CursorPosition(me)])))
        Text_CursorPosition(me)++;
    expose(w,NULL,(Region)NULL);
}

static void grab_focus()
/* DO ME */{
}

static void Help()
/* DO ME */{
}

static void insert_string()
/* DO ME */{
}

static void key_select()
/* DO ME */{
}

static void kill_next_character(Widget w,XEvent *ev,String *params,
                        Cardinal *num_params)
{
    delete_next_character(w,ev,params,num_params);
}

static void kill_next_word(Widget w,XEvent *ev,String *params,
                        Cardinal *num_params)
{
    delete_next_word(w,ev,params,num_params);
}

static void kill_previous_character(Widget w,XEvent *ev,String *params,
				    Cardinal *num_params)
{
    delete_previous_character(w,ev,params,num_params);
}

static void kill_previous_word(Widget w,XEvent *ev,String *params,
				    Cardinal *num_params)
{
    delete_previous_word(w,ev,params,num_params);
}

static void kill_selection()
/* DO ME */{
}

static void kill_to_end_of_line(Widget w,XEvent *ev,String *params,
				Cardinal *num_params)
{
    delete_to_end_of_line(w,ev,params,num_params);
}

static void kill_to_start_of_line()
/* DO ME */{
}

static void move_destination()
/* DO ME */{
}

static void move_to()
/* DO ME */{
}

static void next_tab_group()
/* DO ME */{
}

static void next_line(Widget w,XEvent *ev,String *params,
				    Cardinal *num_params)
{
    XmTextWidget me=(XmTextWidget) w;
    int lcount=0;

    if(me->text.length==0)
	return;

    while((me->text.value[Text_CursorPosition(me)]!='\n')&&
	  (Text_CursorPosition(me)>0))
    {
        Text_CursorPosition(me)--;
	lcount++;
    }
    Text_CursorPosition(me)++;
    while((me->text.value[Text_CursorPosition(me)]!='\n')&&
	  (Text_CursorPosition(me)<me->text.length))
        Text_CursorPosition(me)++;
    while((lcount>0)&&(Text_CursorPosition(me)<me->text.length))
    {
	lcount--;
        Text_CursorPosition(me)++;
    }
    expose(w,NULL,(Region)NULL);
}

static void next_page()
/* DO ME */{
}

static void
newline(Widget w,XEvent *ev,String *params, Cardinal *num_params)
{
    XmTextWidget me=(XmTextWidget) w;
  
    if(!Text_Editable(me))
	return;   /* if we can't edit */
    else if (Text_EditMode(me) == XmSINGLE_LINE_EDIT) {
	activate(w, ev, params, num_params);
	return;
    }
      
   Text_CursorPosition(me)++;
   _XmTextInsert(w, Text_CursorPosition(me)-1,"\n", ev);
}

static void
newline_and_backup(Widget w,XEvent *ev,String *params, Cardinal *num_params)
{
    XmTextWidget me=(XmTextWidget) w;
    
    if ((!Text_Editable(me)) || (Text_EditMode(me) == XmSINGLE_LINE_EDIT))
	return;   /* if we can't edit, or we are in single line only mode.*/
   
    _XmTextInsert(w,Text_CursorPosition(me),"\n", ev);
}

static void newline_and_indent(Widget w,XEvent *ev,String *params,
				    Cardinal *num_params)
{
}

static void page_left()
/* DO ME */{
}

static void page_right()
/* DO ME */{
}

static void paste_clipboard()
/* DO ME */{
}

static void prev_tab_group()
/* DO ME */{
}


static void previous_line(Widget w,XEvent *ev,String *params,
				    Cardinal *num_params)
{
    XmTextWidget me=(XmTextWidget) w;
    int lcount=0;  /* count from left margin */  

    while((me->text.value[Text_CursorPosition(me)]!='\n')&&
	  (Text_CursorPosition(me)>0))
    {
        Text_CursorPosition(me)--;
	lcount++;
    }
    Text_CursorPosition(me)--;
    while((me->text.value[Text_CursorPosition(me)]!='\n')&&
	  (Text_CursorPosition(me)>0))
        Text_CursorPosition(me)--;
    while((lcount>0)&&(Text_CursorPosition(me)<me->text.length))
    {
	lcount--;
        Text_CursorPosition(me)++;
    }
    expose(w,NULL,(Region)NULL);
}

static void previous_page()
/* DO ME */{
}

static void process_bdrag()
/* DO ME */{
}

static void process_cancel()
/* DO ME */{
}

static void process_down(Widget w,XEvent *ev,String *params,
				    Cardinal *num_params)
{
    next_line(w,ev,params,num_params);
}

static void process_home(Widget w,XEvent *ev,String *params,
				    Cardinal *num_params)
{
    beginning_of_line(w,ev,params,num_params);
}

static void process_return(Widget w,XEvent *ev,String *params,
				    Cardinal *num_params)
{
    newline(w,ev,params,num_params);
}

static void process_shift_down()
/* DO ME */{
}

static void process_shift_up()
/* DO ME */{
}

static void process_tab()
/* DO ME */{
}

static void process_up()
/* DO ME */{
}

static void quick_copy_set()
/* DO ME */{
}

static void quick_cut_set()
/* DO ME */{
}

static void 
redraw_display(Widget w,
	       XEvent *ev,
	       String *params,
	       Cardinal *num_params)
{
    expose(w,NULL,(Region)NULL);
}

static void 
scroll_one_line_up()
/* DO ME */{
}

static void 
secondary_adjust()
/* DO ME */{
}

static void 
secondary_notify()
/* DO ME */{
}

static void 
secondary_start()
/* DO ME */{
}

static void 
select_adjust()
/* DO ME */{
}

static void 
select_all()
/* DO ME */{
}

static void 
select_start(Widget w,
	     XEvent *ev,
	     String *params,
	     Cardinal *num_params)
{
    XmTextWidget me=(XmTextWidget) w;
    
    if(ev->type!=ButtonPress)
	return;

    me->text.selectbegin=XmTextXYToPos(w,ev->xbutton.x,ev->xbutton.y);
}

static void 
select_end(Widget w,
	   XEvent *ev,
	   String *params,
	   Cardinal *num_params)
{
    XmTextWidget me=(XmTextWidget) w;

    if(ev->type!=ButtonRelease)
	return;

    Text_CursorPosition(me)=XmTextXYToPos(w,ev->xbutton.x,ev->xbutton.y);

    if(Text_CursorPosition(me)<me->text.selectbegin)
    {
	me->text.selectend=me->text.selectbegin;
	me->text.selectbegin=Text_CursorPosition(me);
    }
    else
	me->text.selectend=Text_CursorPosition(me);
    if(me->text.selectend>=me->text.length)
	me->text.selectend=me->text.length-1;
    expose(w, NULL, (Region) NULL);

}



static void 
self_insert(Widget w,
	    XEvent *ev,
	    String *params,
	    Cardinal *num_params)
{
#define	INSERT_LEN	10

    /* I hope to god XLookupString can't generate strings longer than 10
     * characters :) */

    char buffer[INSERT_LEN];
    XmTextWidget me=(XmTextWidget) w;
    KeySym keysym;
    int length;
#ifdef	I18N
    int	status;
#endif

#ifdef	I18N
    length = XmImMbLookupString(w, &ev->xkey, buffer, INSERT_LEN, &keysym, &status);
    if (length > 0 && (status == XLookupBoth || status == XLookupChars))
#else
    length = XLookupString(&ev->xkey, buffer, 10, &keysym, NULL);

    /* if we aren't a meta char.  must find some better way to do this */
    if (keysym < 0xFF00)
#endif
    {
	if (!Text_Editable(me) && Text_VerifyBell(me))
	    XBell(XtDisplay(w), 100);   /* Is the volume right ?? Danny */
	buffer[length] = '\0';
	XmTextInsert(w,Text_CursorPosition(me),buffer);        
	Text_CursorPosition(me)++;
    }

    expose(w, NULL, (Region)NULL);
}

static void 
set_anchor()
/* DO ME */{
}

static void 
set_insertion_point()
/* DO ME */{
}

static void 
set_selection_hint()
/* DO ME */{
}

static void 
scroll_one_line_down()
/* DO ME */{
}

static void 
toggle_add_mode()
/* DO ME */{
}

static void 
traverse_home()
/* DO ME */{
}

static void 
traverse_next()
/* DO ME */{
}

static void 
traverse_prev()
/* DO ME */{
}

static void 
unkill()
{


}


static Boolean 
_inword(char s)
{
    switch('s')
    {
	case ' ':
	case '\t':
	case '\n':
	case '!':
	case '?':
	case '.':
	case ',':
	    return False;
	default:
	    return True;
    }
}

static char * 
_delete(char *s,
	int pos, 
	int len)
{
    char	*new;
    int		l;
 
    XdbDebug(__FILE__, NULL, "S: %s pos: %d len: %d\n",s,pos,len);
 
    if (s == NULL || strlen(s) == 0)
 	return XtNewString("");
    
    l = strlen(s);
    
    if (pos < 0 || pos > l)
 	return s;
    if (len < 0 || len + pos > l)
 	return s;
    
    new = XtMalloc(l - len);
    
    strncpy(new, s, pos);
    new[pos] = '\0';	/* on Linux, strncpy() doesn't add a 0 byte */
    if (pos + len < l)
 	strcat(new, s+pos+len);
    XtFree(s);
    XdbDebug(__FILE__, NULL, "new is: %s\n",new);
    return new;
}

static void 
focus_in(Widget w, XEvent *evp, String *params, Cardinal *num_params)
{
   XdbDebug(__FILE__, NULL, "Text focus_in\n");

#ifdef	I18N
   if (Text_Editable(w))
      XmImSetFocusValues(w, NULL, 0);
#endif
}

static void 
focus_out(Widget w, XEvent *evp, String *params, Cardinal *num_params)
{
   XdbDebug(__FILE__, NULL, "Text focus_out\n");

#ifdef	I18N
   if (Text_Editable(w))
      XmImUnsetFocus(w);
#endif
}
