/**
 *
 * $Id: XmString.c,v 1.29 1996/04/30 23:59:36 toshok Exp $
 *
 * Copyright (C) 1995 Free Software Foundation, Inc.
 *
 * This file is part of the GNU LessTif Library.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **/

static char rcsid[] = "$Id: XmString.c,v 1.29 1996/04/30 23:59:36 toshok Exp $";

#include <LTconfig.h>
#include <Xm/XmP.h>
#include <Xm/XmI.h>
#include <Xm/DebugUtil.h>
#include <stdlib.h>
#include <stdio.h>
#if HAVE_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif

/* forward */
XmStringComponent	_XmStringPeekNextComponent(XmStringContext);

static XmString 
__XmAllocNewXmString(int number_of_segments)
{
    XmString newString = (XmString)XtMalloc(sizeof(XmStringRec));
    int i;

    newString->number_of_segments = number_of_segments;

    newString->segments = (XmStringComponent*)XtMalloc(sizeof(XmStringComponent)
							 * newString->number_of_segments);
    
    for (i=0; i<number_of_segments; i++)
	newString->segments[i] = (XmStringComponent)XtMalloc(sizeof(XmStringComponentRec));

    return newString;
}

/* just like XmStringPeekNextComponent, except it actually increments the
   current segment */
static XmStringComponent
__XmStringGetNextComponent(XmStringContext context)
{
    if (context == NULL)
	return NULL;

    context->current_segment ++;

    if (context->current_segment < context->string->number_of_segments)
	return context->string->segments[context->current_segment];
    else
	return NULL;
}

static void
__XmStringComponentCopy(XmStringComponent dest, XmStringComponent src)
{
    if (src->type == XmSTRING_COMPONENT_TEXT)
    {
	dest->contents = XtNewString(src->contents);
	dest->length = src->length;
	dest->tag = XtNewString(src->tag);
	dest->direction = src->direction;
    }
    dest->type = src->type;
}

#ifdef BUGGY_SPARC_GCC
_BuggySparcGccCall(int i)
{
     int   a = i + 1;
}
#endif

Dimension 
XmStringBaseline(XmFontList fontlist, 
		 XmString string)
{
    /* this code only handles fontlists with entries
       with type XmFONT_IS_FONT, for now... FIX ME */

    XmStringContext context = (XmStringContext)XtMalloc(sizeof(XmStringContextRec));
    XmStringComponent foo;
    Dimension current_baseline = (Dimension)0;

    if (!XmStringInitContext(&context, string))
	return (Dimension)0;

    if ((foo = __XmStringGetNextComponent(context)) != NULL)
    {
	XmFontListEntry entry;
	XFontStruct *fs;
	XmFontType idontcare;

	entry = (XmFontListEntry)_XmFontListEntryFromTag(fontlist,
					foo->tag);

	fs = (XFontStruct*)XmFontListEntryGetFont(entry, &idontcare);

	if (fs->max_bounds.ascent > current_baseline)
	    current_baseline = fs->max_bounds.ascent;
    }

    XdbDebug(__FILE__, NULL, "XmStringBaseline() = %d\n", current_baseline);
/* #define BUGGY_SPARC_GCC */
#ifdef BUGGY_SPARC_GCC
    /* without a caller-save here, gcc-2.7.0 generates bad code and
       corrupts the value in current_baseline when using -O2 (Solaris2.4) */
    _BuggySparcGccCall(1);
#endif

    XtFree((char*)context);

    return current_baseline;
}

Boolean 
XmStringByteCompare(XmString s1, 
		    XmString s2)
{
    /* don't really believe that this one is necessary, so we simply
       punt for now. */

       return XmStringCompare(s1, s2);
}

Boolean 
XmStringCompare(XmString s1, XmString s2)
{
    XmStringContext context1 = (XmStringContext)XtMalloc(sizeof(XmStringContextRec));
    XmStringComponent foo1;
    XmStringContext context2 = (XmStringContext)XtMalloc(sizeof(XmStringContextRec));
    XmStringComponent foo2;

    if (!XmStringInitContext(&context1, s1))
	return False;
    if (!XmStringInitContext(&context2, s2))
	return False;

    while ((foo1 = __XmStringGetNextComponent(context1)))
    {
	if (! (foo2 = __XmStringGetNextComponent(context2)))
	    return False;

	if (foo1->type == XmSTRING_COMPONENT_SEPARATOR) {
	    if (foo2->type == XmSTRING_COMPONENT_SEPARATOR)
		continue;
	    else
		return False;
	}
	else if (foo2->type == XmSTRING_COMPONENT_SEPARATOR)
	    return False;

	if (strcmp(foo1->contents, foo2->contents)
	    || strcmp(foo1->tag, foo2->tag))
	    return False;
    }

    XtFree((char*)context1);
    XtFree((char*)context2);

    return True;
}

XmString 
XmStringConcat(XmString s1, XmString s2)
{
    XmString newString;
    int i;

    newString = __XmAllocNewXmString((s1 ? s1->number_of_segments : 0 ) + s2->number_of_segments);

    if (s1)
	for (i=0; i<s1->number_of_segments; i++)
	    __XmStringComponentCopy(newString->segments[i], s1->segments[i]);

    for (i=0; i<s2->number_of_segments; i++)
	__XmStringComponentCopy(newString->segments[i+(s1 ? s1->number_of_segments : 0)], s2->segments[i]);

    return newString;
}

XmString 
XmStringSeparatorCreate(void)
{
   XmString newString = __XmAllocNewXmString(1);

   newString->segments[0]->type = XmSTRING_COMPONENT_SEPARATOR;
   newString->segments[0]->contents = NULL;

   return newString;
}

XmString 
XmStringCreate(char *text, char *tag)
{
    XmString newString = __XmAllocNewXmString(1);

    newString->segments[0]->contents = XtNewString(text);
    newString->segments[0]->length = strlen(text);
    newString->segments[0]->tag = XtNewString(tag);
    newString->segments[0]->type = XmSTRING_COMPONENT_TEXT;
    return newString;
}

XmString 
XmStringCreateLtoR(char *text, char *tag)
{
    XmString xmstr1;
    XmString xmstr2;
    XmString line;
    XmString sep = XmStringSeparatorCreate();

    char *t = XtNewString(text);
    char *p;

    p = strtok(t,"\n");

    if (p) /* if there was actually a return character in the string. */
    {
	xmstr1 = XmStringCreate(p, tag);

	while ((p = strtok(NULL, "\n")))
	{
	    line = XmStringCreate(p,tag);
	    xmstr2 = XmStringConcat(xmstr1, sep);
	    XmStringFree(xmstr1);
	    xmstr1 = XmStringConcat(xmstr2, line);
	    XmStringFree(xmstr2);
	    XmStringFree(line);
	}
    }
    else
	xmstr1 = XmStringCreate(text, tag);

    XmStringFree(sep);
    XtFree(t);

    return xmstr1;
}

XmString
XmStringLtoRCreate(char *text, char *tag)
{
    return XmStringCreateLtoR(text, tag);
}

XmString 
XmStringCreateLocalized(char *text)
{
	return XmStringCreate(text, XmFONTLIST_DEFAULT_TAG);
}

XmString 
XmStringCreateSimple(char *text)
{
    return XmStringCreate(text, XmFONTLIST_DEFAULT_TAG);
}

XmString 
XmStringDirectionCreate(XmStringDirection direction)
{
    return NULL;
}

void _XmStringLineDimension(XmFontList fontlist,
			    XmString string,
			    XmStringComponent starting,
			    Dimension *width,
			    Dimension *height,
			    Dimension *ascent,
			    Dimension *descent)
{
    XmStringContext context = (XmStringContext)XtMalloc(sizeof(XmStringContextRec));
    XmStringComponent foo;
    XFontStruct *fs = NULL;
    XtPointer font_return;
    XFontSet fset = NULL;
    int seen_starting = 0;

    XdbDebug(__FILE__, NULL, "_XmStringLineDimension()\n");

    *height = 0;
    *width = 0;
    *ascent = 0;
    *descent = 0;

    XmStringInitContext(&context, string);

    while ((foo = __XmStringGetNextComponent(context)) != NULL)
    {
	if (!seen_starting)
	{
	    if (foo == starting)
		seen_starting = 1;
	    else
		continue;
	}

	switch (foo->type)
	{
	case XmSTRING_COMPONENT_TEXT:
	    {
		XmFontType font_type;
		XmFontListEntry entry = (XmFontListEntry)_XmFontListEntryFromTag(fontlist,
										 foo->tag);

		font_return = XmFontListEntryGetFont(entry, &font_type);
		if (font_return == NULL) {
			XdbDebug(__FILE__, NULL, "_XmStringLineDimension: got NULL Font\n");
			return;
		}

		switch (font_type)
		{
		case XmFONT_IS_FONT:
		    fs = (XFontStruct*)font_return;
		    
		    XdbDebug(__FILE__, NULL, "Testing width of '%s' (width so far is %d)\n",
			foo->contents, *width);

		    *width += XTextWidth(fs, foo->contents, strlen(foo->contents));
		    
		    if (fs->max_bounds.ascent + fs->max_bounds.descent > *height)
			*height = fs->max_bounds.ascent + fs->max_bounds.descent;
		    
		    if (fs->max_bounds.ascent > *ascent)
			*ascent = fs->max_bounds.ascent;
		    
		    if (fs->max_bounds.descent > *descent)
			*descent = fs->max_bounds.descent;
		    
		    break;
		case XmFONT_IS_FONTSET:
		    fset = (XFontSet)font_return;
		    /* need to do something with the extents of the font set here. */
		    break;
		}
	    }
	    break;
	case XmSTRING_COMPONENT_SEPARATOR:
	    XdbDebug(__FILE__, NULL, "_XmStringLineDimension() case XmSTRING_COMPONENT_SEPARATOR not treated yet.\n");
	    return;
	}
    }
    XdbDebug(__FILE__, NULL, "_XmStringLineDimension() => w %d h %d a %d d %d\n",
	*width, *height, *ascent, *descent);
}


void 
XmStringDraw(Display *d, 
	     Window w,
	     XmFontList fontlist,
	     XmString string,
	     GC gc,
	     Position x,
	     Position y,
	     Dimension width,
	     unsigned char alignment,
	     unsigned char layout_direction,
	     XRectangle *clip)
{
    Position current_y;
    Position current_x = 0;
    XmStringContext context = (XmStringContext)XtMalloc(sizeof(XmStringContextRec));
    XmStringComponent foo;
    XFontStruct *fs = NULL;
    XtPointer font_return;
    XFontSet fset = NULL;
    int done_with_line;

    if (w == 0)
	return;

    XdbDebug(__FILE__, XtWindowToWidget(d, w), "XmStringDraw x %d y %d wid %d\n",
	x, y, width);

    current_y = y + XmStringBaseline(fontlist, string);

    XmStringInitContext(&context, string);

    /* repeat while there's at least one thing in the string */
    while ((foo = _XmStringPeekNextComponent(context)) != NULL)
    {
	/* the dimensions of the current line */
	Dimension line_height, line_width, line_ascent, line_descent;

	_XmStringLineDimension(fontlist, string, foo, &line_width, &line_height, &line_ascent, &line_descent);

	/* now we handle the starting x of this line based on the alignment */
	switch (alignment)
	{
	case XmALIGNMENT_CENTER:
	    current_x = x + (int)(width/2.0 - line_width/2.0);
	    break;
	case XmALIGNMENT_BEGINNING:
	    current_x = x;
	    break;
	case XmALIGNMENT_END:
	    current_x = x + width - line_width;
	    break;
	}

	/* initialize the important condition for the following loop */
	done_with_line = 0;

	/* now we loop through the lines on this line, drawing them. */
	while (!done_with_line 
	       && (foo = __XmStringGetNextComponent(context)))
	    switch (foo->type)
	    {
	    case XmSTRING_COMPONENT_TEXT:
		{
		    XmFontType font_type;
		    XmFontListEntry entry = (XmFontListEntry)_XmFontListEntryFromTag(fontlist,
										     foo->tag);
		    
		    font_return = XmFontListEntryGetFont(entry, &font_type);
		    
		    switch (font_type)
		    {
		    case XmFONT_IS_FONT:
			fs = (XFontStruct*)font_return;

			XSetFont(d, gc, fs->fid);

			XdbDebug(__FILE__, XtWindowToWidget(d, w), "XDrawString '%s' at %d %d\n",
				foo->contents, current_x, current_y);

			XDrawString(d, w, gc, current_x, current_y, foo->contents, strlen(foo->contents));

			current_x += XTextWidth(fs, foo->contents, strlen(foo->contents));

			break;
		    case XmFONT_IS_FONTSET:
			fset = (XFontSet)font_return;

			XmbDrawString(d, w, fset, gc, current_x, current_y, foo->contents, strlen(foo->contents)); /* strlen is wrong! */

			/* need to do something with the extents of the font set here. */
			break;
		    }
		}
		break;
	    case XmSTRING_COMPONENT_SEPARATOR:
		{
		    /* now we increment the current_y by this line's descent, and the _next_ line's ascent. */
		    Dimension next_line_height, next_line_width, next_line_ascent, next_line_descent;

		    _XmStringLineDimension(fontlist, string,
					   _XmStringPeekNextComponent(context),
					   &next_line_height,
					   &next_line_width,
					   &next_line_ascent,
					   &next_line_descent);
		    
		    current_y +=/* line_descent +*/ next_line_ascent;

		    /* we're done with this line.  The current_x will be updated above to reflect
		       how we should draw the next line.  The current_y was just updated */
		    done_with_line = 1;
		}
		break;
	    }
    }
}

void 
XmStringDrawImage(Display *d,
		  Window w,
		  XmFontList fontlist,
		  XmString string,
		  GC gc,
		  Position x,
		  Position y,
		  Dimension width,
		  unsigned char alignment,
		  unsigned char layout_direction,
		  XRectangle *clip)
{
    Position current_y;
    Position current_x = 0;
    XmStringContext context = (XmStringContext)XtMalloc(sizeof(XmStringContextRec));
    XmStringComponent foo;
    XFontStruct *fs = NULL;
    XtPointer font_return;
    XFontSet fset = NULL;
    int done_with_line;

    current_y = y + XmStringBaseline(fontlist, string);

    XmStringInitContext(&context, string);

    /* repeat while there's at least one thing in the string */
    while ((foo = _XmStringPeekNextComponent(context)) != NULL)
    {
	/* the dimensions of the current line */
	Dimension line_height, line_width, line_ascent, line_descent;

	_XmStringLineDimension(fontlist, string, foo, &line_width, &line_height, &line_ascent, &line_descent);

	/* now we handle the starting x of this line based on the alignment */
	switch (alignment)
	{
	case XmALIGNMENT_CENTER:
	    current_x = x + (int)(width/2.0 - line_width/2.0);
	    break;
	case XmALIGNMENT_BEGINNING:
	    current_x = x;
	    break;
	case XmALIGNMENT_END:
	    current_x = x + width - line_width;
	    break;
	}

	/* initialize the important condition for the following loop */
	done_with_line = 0;

	/* now we loop through the lines on this line, drawing them. */
	while (!done_with_line 
	       && (foo = __XmStringGetNextComponent(context)))
	    switch (foo->type)
	    {
	    case XmSTRING_COMPONENT_TEXT:
		{
		    XmFontType font_type;
		    XmFontListEntry entry = (XmFontListEntry)_XmFontListEntryFromTag(fontlist,
										     foo->tag);
		    
		    font_return = XmFontListEntryGetFont(entry, &font_type);
		    
		    switch (font_type)
		    {
		    case XmFONT_IS_FONT:
			fs = (XFontStruct*)font_return;

			XSetFont(d, gc, fs->fid);

			XDrawImageString(d, w, gc, current_x, current_y, foo->contents, strlen(foo->contents));

			current_x += XTextWidth(fs, foo->contents, strlen(foo->contents));

			break;
		    case XmFONT_IS_FONTSET:
			fset = (XFontSet)font_return;

			XmbDrawImageString(d, w, fset, gc, current_x, current_y, foo->contents, strlen(foo->contents)); /* strlen is wrong! */

			/* need to do something with the extents of the font set here. */
			break;
		    }
		}
		break;
	    case XmSTRING_COMPONENT_SEPARATOR:
		{
		    /* now we increment the current_y by this line's descent, and the _next_ line's ascent. */
		    Dimension next_line_height, next_line_width, next_line_ascent, next_line_descent;

		    _XmStringLineDimension(fontlist, string,
					   _XmStringPeekNextComponent(context),
					   &next_line_height,
					   &next_line_width,
					   &next_line_ascent,
					   &next_line_descent);
		    
		    current_y +=/* line_descent +*/ next_line_ascent;

		    /* we're done with this line.  The current_x will be updated above to reflect
		       how we should draw the next line.  The current_y was just updated */
		    done_with_line = 1;
		}
		break;
	    }
    }
}

/*
 * Similar to XmStringDraw, but with an additional parameter
 *
 * If the underline string is contained in string, then that part
 * of string is underlined.
 */
void 
XmStringDrawUnderline(Display *d,
		      Window w,
		      XmFontList fontlist,
		      XmString string,
		      GC gc,
		      Position x,
		      Position y,
		      Dimension width,
		      unsigned char alignment,
		      unsigned char layout_direction,
		      XRectangle *clip,
		      XmString underline)
{
    Position		current_y, current_x = 0;
    XmStringContext	context = (XmStringContext)XtMalloc(sizeof(XmStringContextRec)),
			und_c = (XmStringContext)XtMalloc(sizeof(XmStringContextRec));
    XmStringComponent	foo, und;
    XFontStruct		*fs = NULL;
    XtPointer		font_return;
    XFontSet		fset = NULL;
    int			done_with_line, xo;
    char		*p, *u;

    if (w == 0)
	return;

    XdbDebug(__FILE__, XtWindowToWidget(d, w), "XmStringDrawUnderline x %d y %d wid %d\n",
	x, y, width);

/* Currently only works if "underline" is a single-segment LtoR string */
    XmStringInitContext(&und_c, underline);
    und = _XmStringPeekNextComponent(und_c);
    if (und->type != XmSTRING_COMPONENT_TEXT)
	u = NULL;
    else
	u = und->contents;

/* Go on */
    current_y = y + XmStringBaseline(fontlist, string);

    XmStringInitContext(&context, string);

    /* repeat while there's at least one thing in the string */
    while ((foo = _XmStringPeekNextComponent(context)) != NULL)
    {
	/* the dimensions of the current line */
	Dimension line_height, line_width, line_ascent, line_descent;

	_XmStringLineDimension(fontlist, string, foo, &line_width, &line_height, &line_ascent, &line_descent);

	/* now we handle the starting x of this line based on the alignment */
	switch (alignment)
	{
	case XmALIGNMENT_CENTER:
	    current_x = x + (int)(width/2.0 - line_width/2.0);
	    break;
	case XmALIGNMENT_BEGINNING:
	    current_x = x;
	    break;
	case XmALIGNMENT_END:
	    current_x = x + width - line_width;
	    break;
	}

	/* initialize the important condition for the following loop */
	done_with_line = 0;

	/* now we loop through the lines on this line, drawing them. */
	while (!done_with_line 
	       && (foo = __XmStringGetNextComponent(context)))
	    switch (foo->type)
	    {
	    case XmSTRING_COMPONENT_TEXT:
		{
		    XmFontType font_type;
		    XmFontListEntry entry = (XmFontListEntry)_XmFontListEntryFromTag(fontlist,
										     foo->tag);
		    
		    font_return = XmFontListEntryGetFont(entry, &font_type);
		    
		    switch (font_type)
		    {
		    case XmFONT_IS_FONT:
			fs = (XFontStruct*)font_return;

			XSetFont(d, gc, fs->fid);

			XdbDebug(__FILE__, XtWindowToWidget(d, w), "XDrawString '%s' at %d %d\n",
				foo->contents, current_x, current_y);

			XDrawString(d, w, gc, current_x, current_y, foo->contents, strlen(foo->contents));

			if (u && (p = strstr(foo->contents, u))) {	/* Underline */
			    int		ww, l = p - foo->contents;
			    char *f = XtMalloc(l + 1);
			    strncpy(f, p, l);
			    f[l] = '\0';
			    xo = XTextWidth(fs, f, l);
			    XtFree(f);
			    ww = XTextWidth(fs, u, strlen(u));

/*
 * Subtracted 1 from the width to take the space between characters
 * into account. Is there a decent way to do this ??
 * Danny 17/4/96
 */
			    XDrawLine(d, w, gc,
				current_x + xo, current_y + 1,
				current_x + xo + ww - 1, current_y + 1);

			    XdbDebug(__FILE__, XtWindowToWidget(d, w), "Underlining '%s' in '%s', descent %d\n",
				u, foo->contents, fs->descent);
			    XdbDebug(__FILE__, XtWindowToWidget(d, w), "Drawing from [%d,%d] to [%d,%d]\n",
				current_x + xo, current_y + 1,
				current_x + xo + ww - 1, current_y + 1);
			    XdbDebug(__FILE__, XtWindowToWidget(d, w), "Widget size %d x %d\n",
				XtWidth(XtWindowToWidget(d, w)),
				XtHeight(XtWindowToWidget(d, w)));
			}

			current_x += XTextWidth(fs, foo->contents, strlen(foo->contents));

			break;
		    case XmFONT_IS_FONTSET:
			fset = (XFontSet)font_return;

			XmbDrawString(d, w, fset, gc, current_x, current_y, foo->contents, strlen(foo->contents)); /* strlen is wrong! */

			/* need to do something with the extents of the font set here. */
			break;
		    }
		}
		break;
	    case XmSTRING_COMPONENT_SEPARATOR:
		{
		    /* now we increment the current_y by this line's descent, and the _next_ line's ascent. */
		    Dimension next_line_height, next_line_width, next_line_ascent, next_line_descent;

		    _XmStringLineDimension(fontlist, string,
					   _XmStringPeekNextComponent(context),
					   &next_line_height,
					   &next_line_width,
					   &next_line_ascent,
					   &next_line_descent);
		    
		    current_y +=/* line_descent +*/ next_line_ascent;

		    /* we're done with this line.  The current_x will be updated above to reflect
		       how we should draw the next line.  The current_y was just updated */
		    done_with_line = 1;
		}
		break;
	    }
    }
}

void
_XmStringDrawMnemonic(Display *d,
		      Window w,
		      XmFontList fontlist,
		      XmString string,
		      GC gc,
		      Position x,
		      Position y,
		      Dimension width,
		      unsigned char alignment,
		      unsigned char layout_direction,
		      XRectangle *clip,
		      String mnemonic,
		      XmStringCharSet charset)
{
	XmString	und;

	if (mnemonic == NULL) {
		XmStringDraw(d, w, fontlist, string, gc, x, y, width, alignment,
			layout_direction, clip);
		return;
	}

	und = XmStringCreate(mnemonic, charset);
	XmStringDrawUnderline(d, w, fontlist, string, gc, x, y, width, alignment,
		layout_direction, clip, und);
	XmStringFree(und);
}

Boolean 
XmStringEmpty(XmString s1)
{
    if (s1 == NULL 
	|| s1->number_of_segments == 0)

	return True;
    return False;
}

void 
XmStringExtent(XmFontList fontlist, 
	       XmString string,
	       Dimension *width,
	       Dimension *height)
{
    /* I'm sure all this is wrong.... - toshok */
    XmStringContext context = (XmStringContext)XtMalloc(sizeof(XmStringContextRec));
    XmStringComponent foo;
    XmFontType idontcare;
    Dimension current_height = 0, current_width = 0;
    int last_was_sep = 0;
    XFontStruct *fs = NULL;
    Dimension line_height = 0;
    XmFontListEntry entry;

/* Danny 11/11/95 : should we continue with a NULL string ? */
    if (! string) 
    {
       *width = *height = 0;
       return;
    }

    XmStringInitContext(&context, string);

    *height 
	 = *width = 0;

    while ((foo = __XmStringGetNextComponent(context)) != NULL)
    {
	switch(foo->type)
	{
	case XmSTRING_COMPONENT_TEXT:
	    {
		entry = (XmFontListEntry)_XmFontListEntryFromTag(fontlist,
						foo->tag);

		fs = (XFontStruct*)XmFontListEntryGetFont(entry, &idontcare);

		if (fs) {
		    current_width += XTextWidth(fs,
					 	foo->contents,
						strlen(foo->contents));

		    if (fs->max_bounds.ascent + fs->max_bounds.descent >
			line_height)
			line_height = fs->max_bounds.ascent;
					/* + fs->max_bounds.descent*/;
		}
		else
			line_height = 0;
		last_was_sep = 0;
	    }
	    break;
	case XmSTRING_COMPONENT_SEPARATOR:
	    {
		if (*width < current_width)
		    *width = current_width;

		current_width = 0;
		current_height += line_height;

		last_was_sep = 1;
	    }
	    break;
	}
    }

    if (!last_was_sep)
    {
	if (fs)
	    current_height += fs->max_bounds.ascent + fs->max_bounds.descent;

	if (*height < current_height)
	    *height = current_height;

	if (*width < current_width)
	    *width = current_width;
    }

    XtFree((XtPointer)context);
}

void 
XmStringFree(XmString string)
{
}

void 
XmStringFreeContext(XmStringContext context)
{
}

Boolean 
XmStringGetLtoR(XmString string,
		XmStringCharSet tag,
		char **text)
{
    /* for now, we just convert the first one... */

    XmStringContext context = (XmStringContext)XtMalloc(sizeof(XmStringContextRec));
    XmStringComponent foo;

    XmStringInitContext(&context, string);
    
    if ((foo = __XmStringGetNextComponent(context)) != NULL)
    {
	text[0] = XtNewString(foo->contents);
	return True;
    }
    else
	return False;
}

XmStringComponentType 
XmStringGetNextComponent(XmStringContext context,
			 char **text,
			 XmStringCharSet *tag,
			 XmStringDirection *direction,
			 XmStringComponentType *unknown_tag,
			 unsigned short *unknown_length,
			 unsigned char **unknown_value)
{
    XmStringComponent r;
 
    if (context == NULL)
	return XmSTRING_COMPONENT_UNKNOWN;

    context->current_segment++;

    if (context->current_segment < context->string->number_of_segments)
    {
      r = context->string->segments[context->current_segment];

      switch (r->type)
      {
       case XmSTRING_COMPONENT_CHARSET:
		*tag = XtNewString(r->tag);
		break;
       case XmSTRING_COMPONENT_TEXT:
		*text = XtNewString(r->contents);
		break;
       /*case XmSTRING_COMPONENT_DIRECTION:
               break;
       case XmSTRING_COMPONENT_UNKNOWN:
       case XmSTRING_COMPONENT_LOCALE_TEXT:
       case XmSTRING_COMPONENT_LOCALE_TEXT:
       case XmSTRING_COMPONENT_END:
       case XmSTRING_COMPONENT_USER_BEGIN:
       case XmSTRING_COMPONENT_USER_END:
               break;*/
       default:
               _XmWarning(NULL,
			  "XmStringGetNextComponent: unknown type %d\n", r);
       }
       return r->type;
     }
     else
	return XmSTRING_COMPONENT_UNKNOWN;
}

Boolean 
XmStringGetNextSegment(XmStringContext context,
		       char **text,
		       XmStringCharSet *tag,
		       XmStringDirection *direction,
		       Boolean *separator)
{
    return False;
}

Boolean 
XmStringHasSubstring(XmString string,
		     XmString substring)
{
    return False;
}

Dimension 
XmStringHeight(XmFontList fontlist, XmString string)
{
    return 0;
}

Boolean 
XmStringInitContext(XmStringContext *context,
		    XmString string)
{
    XmStringContext	p;

    if (string && context) {
	p = (XmStringContext)XtMalloc(sizeof(XmStringContextRec));

	p->string = string;
	p->current_segment = -1;

	*context = p;
	return True;
    } else
	return False;
}

int 
XmStringLength(XmString s1)
{
    int total = 0;
    XmStringContext context = 
	(XmStringContext)XtMalloc(sizeof(XmStringContextRec));
    XmStringComponent foo;

    XmStringInitContext(&context, s1);
    
    while ((foo = __XmStringGetNextComponent(context)) != NULL)
    {
	switch (foo->type)
	{
	case XmSTRING_COMPONENT_TEXT:
	    total += sizeof(XmStringComponentRec) + foo->length;
	    break;
	case XmSTRING_COMPONENT_CHARSET:
	    total += sizeof(XmStringComponentRec) + strlen(foo->tag);
	    break;
	case XmSTRING_COMPONENT_DIRECTION:
	    total += sizeof(XmStringComponentRec);
	    break;
	case XmSTRING_COMPONENT_UNKNOWN:
	    /* ??? */
	    total += sizeof(XmStringComponentRec);
	    break;
	}
    }

    XtFree((XtPointer)context);
    return total;
}

int 
XmStringLineCount(XmString string)
{
    return 0;
}

XmString 
XmStringNConcat(XmString s1, XmString s2, int num_bytes)
{
    return NULL;
}

XmString 
XmStringNCopy(XmString s1, int num_bytes)
{
    return NULL;
}

XmString
XmStringCopy(XmString s)
{
    XmString newString;
    int i;

    if (s == NULL)
	return NULL;

    newString = __XmAllocNewXmString(s->number_of_segments);

    for (i=0; i<newString->number_of_segments; i++)
	__XmStringComponentCopy(newString->segments[i], s->segments[i]);

    return newString;
}

XmStringComponent 
_XmStringPeekNextComponent(XmStringContext context)
{
    if (context == NULL)
	return NULL;

    if (context->current_segment < context->string->number_of_segments)
	return context->string->segments[context->current_segment + 1];
    else
	return NULL;
}

XmStringComponentType 
XmStringPeekNextComponent(XmStringContext context)
{
    if (context == NULL)
	return XmSTRING_COMPONENT_UNKNOWN;

    if (context->current_segment < context->string->number_of_segments)
	return context->string->segments[context->current_segment + 1]->type;
    else
	return XmSTRING_COMPONENT_UNKNOWN;
}

XmString 
XmStringSegmentCreate(char *text, 
		      char *tag, 
		      XmStringDirection direction,
		      Boolean separator)
{
    return NULL;
}

Dimension 
XmStringWidth(XmFontList fontlist,
	      XmString string)
{
    XmStringContext context = (XmStringContext)XtMalloc(sizeof(XmStringContextRec));
    XmStringComponent foo;
    XmFontType idontcare;
    Dimension current_width = 0;
    XFontStruct *fs;
    int width = 0;

    XmStringInitContext(&context, string);

    while ((foo = __XmStringGetNextComponent(context)) != NULL)
    {
	switch(foo->type)
	{
	case XmSTRING_COMPONENT_TEXT:
	    {
		XmFontListEntry entry;

		entry = _XmFontListEntryFromTag(fontlist,
						foo->tag);

		fs = (XFontStruct*)XmFontListEntryGetFont(entry, &idontcare);

		current_width += XTextWidth(fs, foo->contents, strlen(foo->contents));
	    }
	    break;
	case XmSTRING_COMPONENT_SEPARATOR:
	    {
		if (width < current_width)
		    width = current_width;

		current_width = 0;
	    }
	    break;
	}
    }

    return width;
}

