/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@medizin.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: command.c,v 1.5 94/07/13 14:17:23 nau Exp $";

/* reads and executes commans from user
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/stat.h>

#include "global.h"

#include "command.h"
#include "control.h"
#include "create.h"
#include "data.h"
#include "dialog.h"
#include "draw.h"
#include "error.h"
#include "file.h"
#include "find.h"
#include "memory.h"
#include "misc.h"
#include "print.h"
#include "remove.h"
#include "search.h"

/* ---------------------------------------------------------------------------
 * some local types
 */
typedef int		(*FunctionTypePtr)(int);

typedef struct
{
	char			*Text;			/* commandstring */
	Boolean			Arg;			/* argument passed to function */
	FunctionTypePtr	Function;		/* function pointer */
} CommandType, *CommandTypePtr;

/* ---------------------------------------------------------------------------
 * local prototypes
 */
static	int		CommandLoadPCB(int);
static	int		CommandSavePCB(int);
static	int		CommandLoadElement(int);
static	int		CommandNew(int);
static	int		CommandQuit(int);
static	int		CommandSetLayername(int);
static	int		CommandSetPCBName(int);
static	int		CommandListElementConnections(int);
static	int		CommandListAllElementConnections(int);
static	void	PSUsage(void);
static	int		CommandPrint(int);
static	char	*ExpandFilename(char *, char *);
static	char	*SearchFile(char *, char *);
static	int		SetArgcArgv(char *);
static	char	*ConcatArgument(void);

/* ---------------------------------------------------------------------------
 * some local identifiers
 */
static	CommandType	Command[] = {
	{ "l",			0,	CommandLoadPCB },
	{ "l!",			1,	CommandLoadPCB },
	{ "s",			0,	CommandSavePCB },
	{ "s!",			1,	CommandSavePCB },
	{ "new",		0,	CommandNew },
	{ "new!",		1,	CommandNew },
	{ "q",			0,	CommandQuit },
	{ "q!",			1,	CommandQuit },
	{ "le",			0,	CommandLoadElement },
	{ "ec",			0,	CommandListElementConnections },
	{ "ec!",		1,	CommandListElementConnections },
	{ "aec",		0,	CommandListAllElementConnections },
	{ "aec!",		1,	CommandListAllElementConnections },
	{ "layername",	0,	CommandSetLayername },
	{ "pcbname",	0,	CommandSetPCBName },
	{ "ps",			0,	CommandPrint },
	{ "ps!",		1,	CommandPrint }};

static	char	**argv;					/* command-line of arguments */
static	int		argc;

/* ---------------------------------------------------------------------------
 * load a new layout
 * syntax: l|l! filename
 */
static int CommandLoadPCB(int Force)
{
	char	*filename;

	if (argc != 2)
	{
		MyWarningDialog("Usage: l[!] filename\n  loads PCB file\n");
		return(1);
	}

	if ((filename = SearchFile(Settings.FilePath, argv[1])) == NULL)
	{
		MyWarningDialog("can't find file '%s'", argv[1]);
		return(0);
	}

	if (PCB->Changed)
	{
		if (!Force)
		{
			WarningDialog(Output.Toplevel, "layout has changed !\n"
				"Save it first or use 'load!' to load\n"
				"the new data without saving the old one.\n");
			return(0);
		}
		if (Settings.SaveInTMP)
			EmergencySave();
	}
	LoadPCB(filename);
	return(0);
}

/* ---------------------------------------------------------------------------
 * save current layout
 * syntax: s|s! filename         or
 *         s|s!                  to use the original filename
 */
static int CommandSavePCB(int Force)
{
			char	*filename;
	struct	stat	buffer;

	if (argc > 2)
	{
		MyWarningDialog("Usage: s[!] [filename]\n  saves PCB file\n");
		return(1);
	}

	if (argc == 1)
	{
			/* no argument, try to use old name */
		if (!PCB->Filename)
			return(1);
		if ((filename = PCB->Filename) == NULL)
		{
			MyWarningDialog("the data is not yet related to a file\n"
				"use 's filename' instead");
			return(0);
		}
	}
	else
	{
			/* try to expand in current directory */
		if ((filename = ExpandFilename(".", argv[1])) == NULL)
			filename = argv[1];
	}

	if (!stat(filename, &buffer) && !Force)
		MyWarningDialog("file '%s' exists !\n"
			"Use another filename or 'save!'\n"
			"to override it.\n", filename);
	else
		SavePCB(filename);
	return(0);
}

/* ---------------------------------------------------------------------------
 * loads an element
 * syntax: le filename
 */
static int CommandLoadElement(int Dummy)
{
	char	*filename;

	if (argc != 2)
	{
		MyWarningDialog("Usage: le filename\n  loads element file\n");
		return(1);
	}

	if ((filename = SearchFile(Settings.ElementPath, argv[1])) == NULL)
	{
		MyWarningDialog("can't find file '%s'", argv[1]);
		return(0);
	}

	LoadMarkedElement(filename);
	return(0);
}

/* ---------------------------------------------------------------------------
 * creates a new layout
 * syntax: new|new! layoutname
 */
static int CommandNew(int Force)
{
	if (argc < 2)
	{
		MyWarningDialog("Usage: new[!] name\n  initializes new layout\n");
		return(1);
	}

	if (PCB->Changed)
	{
		if (!Force)
		{
			WarningDialog(Output.Toplevel, "layout has changed !\n"
				"Save it first or use 'new!' to\n"
				"delete all currently available data\n");
			return(0);
		}
		if (Settings.SaveInTMP)
			EmergencySave();
	}
	RemovePCB(PCB);
	PCB = CreateNewPCB();
	PCB->Name = ConcatArgument();
	CreateDefaultFont();
	UpdateSettingsOnScreen();
	RedrawOutput();
	return(0);
}

/* ---------------------------------------------------------------------------
 * quits program
 * syntax: q|q!
 */
static int CommandQuit(int Force)
{
	if (argc != 1)
	{
		MyWarningDialog("Usage: q[!] filename\n  quits program\n");
		return(1);
	}
	if (PCB->Changed && !Force)
	{
		WarningDialog(Output.Toplevel, "layout has changed !\n"
		"Save it first or use 'q!' to leave\n"
		"the application without saving the data\n");
		return(0);
	}
	QuitApplication();

		/* just to make 'gcc -Wall' happy, we wont come back */
	return(0);
}

/* ---------------------------------------------------------------------------
 * changes the name of the current layer
 * syntax: namelayer new_name
 */
static int CommandSetLayername(int Dummy)
{
	if (argc < 2)
	{
		MyWarningDialog("Usage: namelayer name\n  changes the layers name\n");
		return(1);
	}

	MyFree(&CURRENT->Name);
	CURRENT->Name = ConcatArgument();
	SetChangedFlag(True);
	UpdateControlPanel();
	return(0);
}

/* ---------------------------------------------------------------------------
 * changes the name of the current layout
 * syntax: namepcb new_name
 */
static int CommandSetPCBName(int Dummy)
{
	if (argc < 2)
	{
		MyWarningDialog("Usage: namepcb name\n  changes the layouts name\n");
		return(1);
	}

	MyFree(&PCB->Name);
	PCB->Name = ConcatArgument();
	SetChangedFlag(True);
	SetNameField();
	return(0);
}

/* ---------------------------------------------------------------------------
 * lists all connections to the pins of the element at the current
 * cursor position and writes them to the file passed as argument.
 * Syntax: ec[!] filename
 */
static int CommandListElementConnections(int Force)
{
	char			*filename;
	ElementTypePtr	element;
	struct	stat	buffer;
	FILE			*fp;

	if (argc != 2 ||
		(element = SearchElement(MyCursor.X, MyCursor.Y)) == NULL)
	{
		MyWarningDialog("Usage: ec[!] filename\n"
			"  lists all connections to element at cursor position\n");
		return(1);
	}

		/* try to expand in current directory */
	if ((filename = ExpandFilename(".", argv[1])) == NULL)
		filename = argv[1];

	if (!stat(filename, &buffer) && !Force)
		MyWarningDialog("file '%s' exists !\n"
			"Use another filename or 'ec!'\n"
			"to override it.\n", filename);
	else
	{
		if ((fp = fopen(filename, "w")) == NULL)
			MyWarningDialog("error during open of file '%s'");
		else
		{
			FindElementConnections(element, fp);
			fclose(fp);
		}
	}
	return(0);
}

/* ---------------------------------------------------------------------------
 * lists all connections to the pins of all elements
 * and writes them to the file passed as argument.
 * Syntax: aec[!] filename
 */
static int CommandListAllElementConnections(int Force)
{
	char			*filename;
	struct	stat	buffer;
	FILE			*fp;

	if (argc != 2 ||
		PCB->ElementN == 0)
	{
		MyWarningDialog("Usage: aec[!] filename\n"
			"  lists all connections to all elements\n");
		return(1);
	}

		/* try to expand in current directory */
	if ((filename = ExpandFilename(".", argv[1])) == NULL)
		filename = argv[1];

	if (!stat(filename, &buffer) && !Force)
		MyWarningDialog("file '%s' exists !\n"
			"Use another filename or 'aec!'\n"
			"to override it.\n", filename);
	else
	{
		if ((fp = fopen(filename, "w")) == NULL)
			MyWarningDialog("error during open of file '%s'");
		else
		{
			FindConnectionsToAllElements(fp);
			fclose(fp);
		}
	}
	return(0);
}

/* ----------------------------------------------------------------------
 * usage of ps command
 */
static void PSUsage(void)
{
	MyWarningDialog("Usage: ps[!] [options] filename\n"
		"   produces PostScript printout of visible layers\n"
		"   -m             mirror\n"
		"   -r             rotate\n"
		"   -ox value      x-offset in mil\n"
		"   -oy value      y-offset in mil\n"
		"   -scale value   scaling\n");
}

/* ---------------------------------------------------------------------------
 * generates a PostScript printout of all currently visible layers to a file
 * syntax: ps! [options] filename
 */
static int CommandPrint(int Force)
{
	char			*filename;
	FILE			*fp;
	struct	stat	buffer;

	if (argc < 2)
	{
		PSUsage();
		return(1);
	}

		/* try to expand in current directory */
	if ((filename = ExpandFilename(".", argv[argc-1])) == NULL)
		filename = argv[argc-1];

	if (!stat(filename, &buffer) && !Force)
		MyWarningDialog("file '%s' exists !\n"
			"Use another filename or 'ps!'\n"
			"to override it.\n", filename);
	else
		if ((fp = fopen(filename, "w")) == NULL)
			MyWarningDialog("error during fopen() of file '%s' !\n", filename);
		else
		{
				/* remove filename from argument list */
			if (PrintPS(--argc, argv, fp))
				PSUsage();
			fclose(fp);
		}
	return(0);
}

/* ---------------------------------------------------------------------------
 * concatenates directory and filename if directory != NULL,
 * expands them with a shell
 * and returns the found name(s) or NULL
 */
static char *ExpandFilename(char *Dirname, char *Filename)
{
	static	DynamicStringType	answer;
			char				*command;
			FILE				*pipe;
			int					c;

		/* clear old answer */
	if (answer.Data)
		answer.Data[0] = '\0';

		/* allocate memory for commandline and build it */
	if (Dirname)
	{
		command = MyCalloc(strlen(Filename) +strlen(Dirname) +7, sizeof(char), "Expand()");
		sprintf(command, "echo %s/%s", Dirname, Filename);
	}
	else
	{
		command = MyCalloc(strlen(Filename) +6, sizeof(char), "Expand()");
		sprintf(command, "echo %s", Filename);
	}

		/* execute it with shell */
	if ((pipe = popen(command, "r")) != NULL)
	{
			/* discard all but the first returned line */
		for(;;)
		{
			if ((c = fgetc(pipe)) == EOF || c == '\n' || c == '\r')
				break;
			else
				DSAddCharacter(&answer, c);
		}

		SaveFree(command);
		return (pclose(pipe) ? NULL : answer.Data);
	}
		/* couldn't be expanded by the shell */
	MyWarningDialog("can't execute '%s'", command);
	SaveFree(command);
	return(NULL);	
}

/* ---------------------------------------------------------------------------
 * expands the argument as a filename using /bin/sh
 * directories as specified path are scanned if the argument
 * doesn't contain a '/'
 * The existance of the file is checked as well as it's type.
 */
static char *SearchFile(char *SearchPath, char *Argument)
{
	char			*dir,
					*filename = NULL,	/* to make 'gcc -Wall' happy */
					*copy,
					*p;
	struct	stat	buffer;

		/* if filename has a path component expand it without
		 * scanning the search path
		 */
	for (p = Argument; *p; p++)
		if (*p == '/')
			return(ExpandFilename(NULL, Argument));

		/* save a copy of the path and try all directories in path */
	copy = MyStrdup(SearchPath, "SearchFile()");
	for (dir = strtok(copy, ":"); dir; dir = strtok(NULL, ":"))
		if ((filename = ExpandFilename(dir, Argument)) != NULL &&
			!stat(filename, &buffer) &&
			S_ISREG(buffer.st_mode))
			break;
	SaveFree(copy);
	return(dir ? filename : NULL);
}

/* ----------------------------------------------------------------------
 * split commandline and fill argv/argc
 */
static int SetArgcArgv(char *Line)
{
	static	int		maxcount = 0;
			char	*p;

	for(argc = 0, p = strtok(Line, " \t;"); p; p = strtok(NULL, " \t;"))
	{
			/* allocate more memory */
		if (argc >= maxcount)
		{
			maxcount += 20;
			argv = (char **) MyRealloc(argv, maxcount*sizeof(char *), "SetArgcArgv()");
		}
		argv[argc++] = p;
	}
	return(argc);
}

/* ---------------------------------------------------------------------------
 * concatenates all arguments starting with number 2
 * used for names which may include blanks
 */
static char *ConcatArgument(void)
{
	size_t		length = 1;
	int			i = argc-1,
				j;
	char		*s;

		/* calculate total length */
	for (j = 1; i; i--, j++)
		length += (strlen(argv[j]) +1);

		/* concatenate string */
	s = MyCalloc(length, sizeof(char), "ConcatArgument()");
	for (j = 1, i = argc-1; i; i--, j++)
	{
		strcat(s, argv[j]);
		if (i != 1)
			strcat(s, " ");
	}
	return(s);
}

/* ---------------------------------------------------------------------------
 * reads a new commandline, parses it and executes the appropriate
 * subroutines
 */
void ReadAndExecuteCommand(void)
{
			char	*commandline;		/* current command line */
			char	*buffer;			/* working area */
			int		i;
	static	char	*prevcommand = NULL;/* previous command line */

		/* read command, accept empty input to avoid warning */
	if ((commandline = GetUserInput(Output.Toplevel, True, "Command:", prevcommand)) == NULL ||
		! *commandline)
	{
		SaveFree(commandline);
		return;
	}

		/* copy new comand line to save buffer */
	if (Settings.SaveLastCommand)
	{
		SaveFree(prevcommand);
		prevcommand = commandline;
	}

		/* create a working copy and fetch command */
	buffer = MyStrdup(commandline, "ReadAndExecuteCommand()");
	if (SetArgcArgv(buffer))
	{
			/* scan command list */
		for (i = 0; i < ENTRIES(Command); i++)
			if (!strcmp(Command[i].Text, argv[0]))
				break;
		if (i == ENTRIES(Command))
		{
			MyWarningDialog("following commands are supported:\n"
				"  l[!]            load layout\n"
				"  s[!]            save layout\n"
				"  new[!]          create new layout\n"
				"  q[!]            quit program\n"
				"  le              load element\n"
				"  ec[!]           list element connections\n"
				"  aec[!]          list all element connections\n"
				"  layername       set name of current layer\n"
				"  pcbname         set name of current layout\n"
				"  ps[!]           create PostScript output\n");
		}
		else
			if (Command[i].Function(Command[i].Arg))
				XBell(Dpy, 100);
	}
	else
		XBell(Dpy, 100);
	SaveFree(buffer);

	if (!Settings.SaveLastCommand)
		SaveFree(commandline);
}

