/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@medizin.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: create.c,v 1.3 94/07/13 14:17:56 nau Exp $";

/* functions used to create vias, pins ...
 */

#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>

#include "global.h"

#include "create.h"
#include "cursor.h"
#include "data.h"
#include "error.h"
#include "memory.h"
#include "misc.h"
#include "parse_l.h"
#include "transform.h"

/* ---------------------------------------------------------------------------
 * local prototypes
 */
static	Boolean		TryFontDirectory(char *);

/* ---------------------------------------------------------------------------
 * creates a new PCB
 */
PCBTypePtr CreateNewPCB(void)
{
	PCBTypePtr	ptr;
	int			i;

		/* allocate memory, switch all layers on and copy resources */
	ptr = MyCalloc(1, sizeof(PCBType), "CreateNewPCB()");
	for (i = 0; i < MAX_LAYER; i++)
	{
		ptr->Layer[i].Color = Settings.LayerColor[i];
		ptr->Layer[i].On = True;
	}
	ptr->LayerGroups = Settings.LayerGroups;
	ptr->ElementOn = True;
	ptr->ElementColor = Settings.ElementColor;
	ptr->PinOn = True;
	ptr->PinColor = Settings.PinColor;
	ptr->ViaOn = True;
	ptr->ViaColor = Settings.ViaColor;
	ptr->ConnectedColor = Settings.ConnectedColor;
	if (Settings.UsePolygonLines)
		SET_FLAG(POLYGONFLAG, ptr);
	ptr->Zoom = Settings.Zoom;
	ptr->Grid = Settings.Grid;

		/* reset layer stack */
	for (i = 0; i < MAX_LAYER; i++)
		LayerStack[i] = i;

	return(ptr);
}

/* ---------------------------------------------------------------------------
 * creates a new via
 */
PinTypePtr CreateNewVia(PCBTypePtr OnPCB, Position X, Position Y, Dimension Thickness, char *Name, BYTE Flags)
{
	PinTypePtr via = OnPCB->Via;

		/* realloc new memory if necessary and clear it */
	if (OnPCB->ViaN >= OnPCB->ViaMax)
	{
		OnPCB->ViaMax += STEP_VIA;	
		via = MyRealloc(via, OnPCB->ViaMax *sizeof(PinType), "CreateNewVia()");
		OnPCB->Via = via;
		memset(via +OnPCB->ViaN, 0, STEP_VIA*sizeof(PinType));
	}

		/* copy values */
	via = via +OnPCB->ViaN;
	via->X = X;
	via->Y = Y;
	via->Thickness = Thickness;
	via->Name = Name ? MyStrdup(Name, "CreateNewVia()") : NULL;
	via->Flags = Flags;
	OnPCB->ViaN++;
	return(via);
}

/* ---------------------------------------------------------------------------
 * adds a new line to the linestack
 */
LineTypePtr AddLineToStack(LineTypePtr NewLine)
{
	LineTypePtr	line = MyCursor.LineStack.Line;

		/* realloc new memory if necessary and clear it */
	if (MyCursor.LineStack.LineN >= MyCursor.LineStack.LineMax)
	{
		MyCursor.LineStack.LineMax += STEP_LINE;	
		line = MyRealloc(line, MyCursor.LineStack.LineMax *sizeof(LineType),
			"AddLineToStack()");
		MyCursor.LineStack.Line = line;
		memset(line +MyCursor.LineStack.LineN, 0, STEP_LINE*sizeof(LineType));
	}

		/* copy values */
	line = line +MyCursor.LineStack.LineN;
	*line = *NewLine;
	MyCursor.LineStack.LineN++;
	return(line);
}

/* ---------------------------------------------------------------------------
 * creates a new line on a layer (one out of four possible directions)
 */
LineTypePtr CreateNewLineOnLayer(PCBTypePtr OnPCB, LayerTypePtr Layer, Position X1, Position Y1, Position X2, Position Y2, Dimension Thickness, BYTE Flags)
{
	LineTypePtr	line = Layer->Line;

		/* if the line has zero length, do nothing at all */
	if (X1 == X2 && Y1 == Y2)
		return(NULL);

		/* realloc new memory if necessary and clear it */
	if (Layer->LineN >= Layer->LineMax)
	{
		Layer->LineMax += STEP_LINE;	
		line = MyRealloc(line, Layer->LineMax *sizeof(LineType),
			"CreateNewLineOnLayer()");
		Layer->Line = line;
		memset(line +Layer->LineN, 0, STEP_LINE*sizeof(LineType));
	}

		/* copy values */
	line = line +Layer->LineN;

		/* only four directions are supported, so we may have to swap
		 * the coordinates, set direction too
		 */
	SETLINEDIRECTION(line,X1,Y1,X2,Y2);
	line->Thickness = Thickness;
	line->Flags = Flags;
	Layer->LineN++;
	return(line);
}

/* ---------------------------------------------------------------------------
 * creates a new rectangle
 */
RectTypePtr CreateNewRect(PCBTypePtr OnPCB, LayerTypePtr Layer, Position X, Position Y, Dimension Width, Dimension Height, BYTE Flags)
{
	RectTypePtr	rect = Layer->Rect;

		/* if the rectangle has zero width or height, do nothing at all */
	if (Width == 0 || Height == 0)
		return(NULL);

		/* realloc new memory if necessary and clear it */
	if (Layer->RectN >= Layer->RectMax)
	{
		Layer->RectMax += STEP_RECT;	
		rect = MyRealloc(rect, Layer->RectMax *sizeof(RectType),
			"CreateNewRect()");
		Layer->Rect = rect;
		memset(rect +Layer->RectN, 0, STEP_RECT*sizeof(RectType));
	}

		/* copy values */
	rect = rect +Layer->RectN;
	rect->X = X;
	rect->Y = Y;
	rect->Width = Width;
	rect->Height = Height;
	rect->Flags = Flags;
	Layer->RectN++;
	return(rect);
}

/* ---------------------------------------------------------------------------
 * creates the surrounding rectangle of a string
 */
void SetSurroundingTextRectangle(SymbolTypePtr Symbol, TextTypePtr Text)
{
	unsigned char	c;
	char			*string = Text->TextString;

		/* calculate size of surrounding rectangle */
	Text->Rect.Width = 0;	
	Text->Rect.Height = 0;	
	if (!string)
		return;
	while((c = *string++) != '\0')
	{
		Text->Rect.Width  += Symbol[c].Width +Symbol[c].Delta;
		Text->Rect.Height = MAX(Text->Rect.Height, Symbol[c].Height);
	}
}

/* ---------------------------------------------------------------------------
 * sets data of a text object a calculates surrounding rectangle of it
 * memory which is allocated for the text field is released and
 * the one for the new string is allocated
 */
void AddStringToText(PCBTypePtr OnPCB, TextTypePtr Text, char *TextString)
{
	SaveFree(Text->TextString);
	Text->TextString = TextString ? MyStrdup(TextString, "AddStringToText()") : NULL;

		/* calculate size of surrounding rectangle */
	SetSurroundingTextRectangle(OnPCB->Font.Symbol, Text);
	RotateRect(&Text->Rect, Text->Rect.X, Text->Rect.Y, Text->Direction);
}

/* ---------------------------------------------------------------------------
 * creates a new text on a layer
 */
TextTypePtr CreateNewText(PCBTypePtr OnPCB, LayerTypePtr Layer, Position X, Position Y, BYTE Direction, char *TextString, BYTE Flags)
{
	TextTypePtr		text = Layer->Text;

	if (!TextString || !*TextString)
		return(NULL);

		/* realloc new memory if necessary and clear it */
	if (Layer->TextN >= Layer->TextMax)
	{
		Layer->TextMax += STEP_TEXT;	
		text = MyRealloc(text, Layer->TextMax *sizeof(TextType),
			"CreateNewText()");
		Layer->Text = text;
		memset(text +Layer->TextN, 0, STEP_TEXT*sizeof(TextType));
	}

		/* copy values, width and height are set by drawing routine
		 * because at ths point we don't know which symbols are available
		 */
	text = text +Layer->TextN;
	text->X = text->Rect.X = X;
	text->Y = text->Rect.Y = Y;
	text->Direction = Direction;
	text->Flags = Flags;
	AddStringToText(OnPCB, text, TextString);
	Layer->TextN++;
	return(text);
}

/* ---------------------------------------------------------------------------
 * add data to a created element
 */
void CreateNewElementData(ElementTypePtr Element, char *CanonicalName, char *NameOnPCB, Position TextX, Position TextY, BYTE Direction)
{
	Element->NameOnPCB = NameOnPCB ? MyStrdup(NameOnPCB, "CreateNewElementData()") : NULL;
	Element->CanonicalName = CanonicalName ? MyStrdup(CanonicalName, "CreateNewElementData()") : NULL;
	Element->TextX = TextX;
	Element->TextY = TextY;
	Element->Direction = Direction;
}

/* ---------------------------------------------------------------------------
 * creates a new entry for an element
 */
ElementTypePtr CreateNewElement(PCBTypePtr OnPCB, char *CanonicalName, char *NameOnPCB, Position TextX, Position TextY, BYTE Direction)
{
	ElementTypePtr element = OnPCB->Element;

		/* realloc new memory if necessary and clear it */
	if (OnPCB->ElementN >= OnPCB->ElementMax)
	{
		OnPCB->ElementMax += STEP_ELEMENT;	
		element = MyRealloc(element, OnPCB->ElementMax *sizeof(ElementType),
			"CreateNewElement()");
		OnPCB->Element = element;
		memset(element +OnPCB->ElementN, 0, STEP_ELEMENT*sizeof(ElementType));
	}

		/* copy values and set additional information */
	element = element +OnPCB->ElementN;
	CreateNewElementData(element, CanonicalName, NameOnPCB, TextX, TextY, Direction);
	OnPCB->ElementN++;
	return(element);
}

/* ---------------------------------------------------------------------------
 * creates a new mark in an element
 */
MarkTypePtr CreateNewMarkInElement(ElementTypePtr Element, Position X, Position Y)
{
	MarkTypePtr	mark = Element->Mark;

		/* realloc new memory if necessary and clear it */
	if (Element->MarkN >= Element->MarkMax)
	{
		Element->MarkMax += STEP_MARK;	
		mark = MyRealloc(mark, Element->MarkMax *sizeof(MarkType),
			"CreateNewMarkInElement()");
		Element->Mark = mark;
		memset(mark +Element->MarkN, 0, STEP_MARK*sizeof(MarkType));
	}

		/* copy data */
	mark = mark + Element->MarkN;
	mark->X = X;
	mark->Y = Y;
	Element->MarkN++;
	return(mark);
}

/* ---------------------------------------------------------------------------
 * creates a new arc in an element
 */
ArcTypePtr CreateNewArcInElement(ElementTypePtr Element, Position X, Position Y, Dimension Width, Dimension Height, int Angle, int Delta, Dimension Thickness)
{
	ArcTypePtr	arc = Element->Arc;

		/* if the arc has zero width/height or the same angles 
		 * then do nothing */
	if (!Delta || !Width || !Height)
		return(NULL);

		/* realloc new memory if necessary and clear it */
	if (Element->ArcN >= Element->ArcMax)
	{
		Element->ArcMax += STEP_ELEMENTARC;	
		arc = MyRealloc(arc, Element->ArcMax *sizeof(ArcType),
			"CreateNewArcInElement()");
		Element->Arc = arc;
		memset(arc +Element->ArcN, 0, STEP_ELEMENTARC*sizeof(ArcType));
	}

		/* set Delta (0,360], StartAngle in [0,360) */
	if ((Delta = Delta % 360) == 0)
		Delta = 360;
	Angle = Angle % 360;

		/* copy values */
	arc = arc + Element->ArcN;
	arc->X = X;
	arc->Y = Y;
	arc->Width = Width;
	arc->Height = Height;
	arc->StartAngle = Angle;
	arc->Delta = Delta;
	arc->Thickness = Thickness;
	Element->ArcN++;
	return(arc);
}

/* ---------------------------------------------------------------------------
 * creates a new arc for an element
 */
LineTypePtr CreateNewLineInElement(ElementTypePtr Element, Position X1, Position Y1, Position X2, Position Y2, Dimension Thickness)
{
	LineTypePtr	line = Element->Line;

		/* if the line has zero length, do nothing at all */
	if (X1 == X2 && Y1 == Y2)
		return(NULL);

		/* realloc new memory if necessary and clear it */
	if (Element->LineN >= Element->LineMax)
	{
		Element->LineMax += STEP_ELEMENTLINE;	
		line = MyRealloc(line, Element->LineMax *sizeof(LineType),
			"CreateNewLineInElement()");
		Element->Line = line;
		memset(line +Element->LineN, 0, STEP_ELEMENTLINE*sizeof(LineType));
	}

		/* copy values */
	line = line +Element->LineN;

		/* only four directions are supported, so we may have to swap
		 * the coordinates, set direction too
		 */
	SETLINEDIRECTION(line,X1,Y1,X2,Y2);
	line->Thickness = Thickness;
	Element->LineN++;
	return(line);
}

/* ---------------------------------------------------------------------------
 * creates a new pin in an element
 */
PinTypePtr CreateNewPin(ElementTypePtr Element, Position X, Position Y, Dimension Thickness, char *Name, BYTE Flags)
{
	PinTypePtr pin = Element->Pin;

		/* realloc new memory if necessary and clear it */
	if (Element->PinN >= Element->PinMax)
	{
		Element->PinMax += STEP_PIN;	
		pin = MyRealloc(pin, Element->PinMax *sizeof(PinType),
			"CreateNewPin()");
		Element->Pin = pin;
		memset(pin +Element->PinN, 0, STEP_PIN*sizeof(PinType));
	}

		/* copy values */
	pin = pin +Element->PinN;
	pin->X = X;
	pin->Y = Y;
	pin->Thickness = Thickness;
	pin->Name = Name ? MyStrdup(Name, "CreateNewPin()") : NULL;
	pin->Flags = Flags;
	Element->PinN++;
	return(pin);
}

/* ---------------------------------------------------------------------------
 * copies data from one element to another and creates the destination 
 * if necessary
 */
ElementTypePtr CopyElement(PCBTypePtr OnPCB, ElementTypePtr From, ElementTypePtr To)
{
	LineTypePtr		line;
	PinTypePtr		pin;
	ArcTypePtr		arc;
	MarkTypePtr		mark;
	Cardinal		i;

		/* create a new element if necessary */
	if (!To)
	{
		To = CreateNewElement(OnPCB, From->CanonicalName, From->NameOnPCB,
			From->TextX, From->TextY, From->Direction);
	}
	else
	{
			/* copy some data anyway */
		FreeElementMemory(To);
		CreateNewElementData(To, From->CanonicalName, From->NameOnPCB,
			From->TextX, From->TextY, From->Direction);
	}

	for (line = From->Line, i = From->LineN; i; i--, line++)
		CreateNewLineInElement(To, line->X1, line->Y1,
			line->X2, line->Y2, line->Thickness);
	for (pin = From->Pin, i = From->PinN; i; i--, pin++)
		CreateNewPin(To, pin->X, pin->Y, pin->Thickness, pin->Name, pin->Flags);
	for (arc = From->Arc, i = From->ArcN; i; i--, arc++)
		CreateNewArcInElement(To, arc->X, arc->Y, arc->Width, arc->Height,
			arc->StartAngle, arc->Delta, arc->Thickness);
	for (mark = From->Mark, i = From->MarkN; i; i--, mark++)
		CreateNewMarkInElement(To, mark->X, mark->Y);

	return(To);
}

/* ---------------------------------------------------------------------------
 * creates a new line in a symbol (one out of four possible directions)
 */
LineTypePtr CreateNewLineInSymbol(SymbolTypePtr Symbol, Position X1, Position Y1, Position X2, Position Y2, Dimension Thickness)
{
	LineTypePtr	line = Symbol->Line;

		/* if the line has zero length, do nothing at all */
	if (X1 == X2 && Y1 == Y2)
		return(NULL);

		/* realloc new memory if necessary and clear it */
	if (Symbol->LineN >= Symbol->LineMax)
	{
		Symbol->LineMax += STEP_SYMBOLLINE;	
		line = MyRealloc(line, Symbol->LineMax *sizeof(LineType),
			"CreateNewLineInSymbol()");
		Symbol->Line = line;
		memset(line +Symbol->LineN, 0, STEP_SYMBOLLINE*sizeof(LineType));
	}

		/* copy values */
	line = line +Symbol->LineN;

		/* only four directions are supported, so we may have to swap
		 * the coordinates, set direction too
		 */
	SETLINEDIRECTION(line,X1,Y1,X2,Y2);
	line->Thickness = Thickness;
	Symbol->LineN++;
	return(line);
}

/* ---------------------------------------------------------------------------
 * tries to load font file from specified directory
 * returns True on success
 */
static Boolean TryFontDirectory(char *Dir)
{
	struct	stat	buffer;
	char			*filename;

		/* allocate memory adn add directory if != NULL */
	if (Dir)
	{
		filename = MyCalloc(strlen(Dir) +2 +strlen(Settings.FontFile), sizeof(char), "TryFontDirectory()");
		sprintf(filename, "%s/%s", Dir, Settings.FontFile);
	}
	else
		filename = MyStrdup(Settings.FontFile, "TryFontDirectory()");

		/* check if file exists */
	if (!stat(filename, &buffer))
	{
		ParseFont(&PCB->Font, filename);
		SaveFree(filename);
		return(True);
	}
	SaveFree(filename);
	return(False);
}

/* ---------------------------------------------------------------------------
 * parses a file with font information and installs it
 * checks directories given as colon seperated list by resource fontPath
 * if the fonts filename doesn't contain a directory component
 */
void CreateDefaultFont()
{
	char	*copy = NULL,
			*dir = NULL,
			*p;

	for (p = Settings.FontFile; *p; p++)
		if (*p == '/')
		{
			if (TryFontDirectory(NULL))
				return;
			MyWarningDialog("can't find default font-symbol-file '%s'",
				Settings.FontFile);
			return;
		}

		/* loop over all directories */
	copy = MyStrdup(Settings.FontPath, "CreateDefaultFont()");
	for (dir = strtok(copy, ":"); dir; dir = strtok(NULL, ":"))
		if (TryFontDirectory(dir))
		{
			SaveFree(copy);
			return;
		}
	SaveFree(copy);
	MyWarningDialog("can't find default font-symbol-file '%s'",
		Settings.FontFile);
}
