/* filename: rlpr.c
 * project: rlpr
 * author: meem  --  meem@sherilyn.wustl.edu
 * version: $Id: rlpr.c,v 1.18 1996/10/12 09:19:06 meem Exp $
 * contents: functions pertinent to the main flow of the program
 * 
 * Time-stamp: <1996/10/11 21:43 -- meem@sherilyn.wustl.edu>
 */

/* copyright (c) 1996 meem, meem@gnu.ai.mit.edu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 */

#include "config.h"

#ifndef STDC_HEADERS
#error currently there is no support for compiling on machines without ANSI headers. \
       write meem@gnu.ai.mit.edu if you are interested in adding it.
#endif

#include <stdio.h>
#include <string.h>			  /* for strcpy() strcat(), etc */
#include <stdlib.h>			  /* for getenv(), etc */
#include <unistd.h>			  /* for unlink(), getopt(), etc */
#include <fcntl.h>			  /* for O_RDWR, etc */
#include <pwd.h>			  /* for accessing the password file */
#include <stdarg.h>

#ifdef __GNU_LIBRARY__		          
#undef __GNU_LIBRARY__
#include "getopt.h"
#define __GNU_LIBRARY__
#else  /* not __GNU_LIBRARY__ */
#include "getopt.h"
#endif /* __GNU_LIBRARY__ */

#include "rlpr.h"
#include "rlpr-net.h"			  /* network rlpr commands are here */
#include "rlpr-dbfile.h"		  /* accessing the rlpr database file */
#include "rlpr-common.h"		  /* common to all rlpr c source files */

char * name;

int main(int argc, char *argv[]) {
  int cfd;				  /* control file descriptor */
  char *cfname;				  /* control file name */
  char *dfname;				  /* data file name */
  char *cffullname;			  /* full pathname to control file */
  int i;			          /* scrap variable */
  int nf;			          /* number of files spooled */
  name = *argv;			          /* program name */

  /* this sets the opts_ and net_ structures */
  parse_args(argc, argv);

  ++argv; --argc;			  /* skip name */
  while (argc && **argv == '-') {	  /* skip command-line options */
    argv++;
    argc--;
  }

  /* create name templates for temp files */
  
  cfname = rlpr_malloc(strlen(net_.localhost) + strlen("cfAxxx") + 1);
  dfname = rlpr_malloc(strlen(net_.localhost) + strlen("dfAxxx") + 1);
  sprintf(cfname, "cfA%03d%s", getpid() % 1000, net_.localhost);
  cffullname = rlpr_malloc(strlen(cfname) + strlen(opts_.tmpdir) + 2);
  sprintf(cffullname, "%s/%s", opts_.tmpdir, cfname);
  strcpy(dfname, cfname);      
  *dfname = 'd';

  if (!opts_.wflag) {		          /* if not connecting to a winNT lpd */
    open_connection();		          /* global sock is now open */
    send_recvj_req();			  /* tell lpd we have a job */
  }

  nf = argc ? argc : 1;			  /* number of files */

  /* open the cf once.  we're going to keep reusing the same one
   * on the client side because we only need one cf at a time
   */
  
  if ((cfd = open(cffullname, O_RDWR|O_TRUNC|O_CREAT, 0600)) < 0)
    rlpr_fatal("%s - cannot create: %s", cffullname, ERRNO);
  unlink(cffullname);
  
  do {
    if (opts_.wflag) {		          /* if connecting to a winNT lpd */
      open_connection();
      send_recvj_req();
    }

    lseek(cfd, 0, SEEK_SET);	          /* reposition at start of file */

    cf_header(cfd, argc ? *argv : NULL);  /* print static part of header */

    for (i = 0; i < opts_.copies; i++)	  /* print multiple copies */
      cf_add(opts_.filetype, dfname, cfd);
    cf_add('U', dfname, cfd);
    send_cf(cfd, cfname);

    send_df(argc ? *argv : NULL, dfname);
    if (opts_.rflag && argc)     	  /* make sure we're not stdin */
      if ((unlink(*argv)) < 0)
	rlpr_warn("unlink %s: %s", *argv, ERRNO);
    
    argv++;
    dfname[2]++;			  /* inc. letter (dfA --> dfB), etc */
    cfname[2]++;	

    if (opts_.wflag) close_connection();
  } while (--argc > 0);
  
  if (!opts_.wflag) close_connection();    /* global sock is now closed */
  
  if (!opts_.qflag) {
    printf("%i file%s spooled to %s:%s", nf, (nf > 1) ? "s" : "",
	    net_.printhost, opts_.printer);
    if (*net_.proxyhost) printf(" (proxy: %s)", net_.proxyhost);
    putchar('\n');
  }

  /* strictly, this is a waste of time, but it makes me feel good, okay? */
  free(cffullname);
  free(cfname);
  free(dfname);
  
  exit(EXIT_SUCCESS);
}

void init_options(void) {
  char *tmp;

  opts_.fflag 	 = 0;			  /* don't auto-form feed by default */
  opts_.bflag 	 = 1;			  /* burst page on by default */
  opts_.qflag 	 = 0;		          /* quiet off by default */
  opts_.mflag 	 = 0;			  /* don't mail after printing */
  opts_.rflag 	 = 0;			  /* don't remove after printing */
  opts_.wflag 	 = 0;			  /* windows braindead mode (off) */
  opts_.copies   = 1;			  /* number of copies to make */
  opts_.job[0]   = '\0';
  opts_.class[0] = '\0';
  opts_.title[0] = '\0';
  opts_.user[0]  = '\0';
  opts_.filetype = ' ';			  /* no file type yet known */
  opts_.width    = DEFAULT_WIDTH;
  net_.port      = DEFAULT_RLPRD_TO_PORT;
  opts_.indent_cols = NULL;		  /* number of columns to indent output */
  get_local_hostname(net_.localhost, sizeof(net_.localhost));

  if ((tmp = getenv("TMPDIR")) != NULL) {
    opts_.tmpdir = rlpr_malloc(strlen(tmp + 1));
    strcpy(opts_.tmpdir, tmp);
  } else opts_.tmpdir = DEFAULT_TMP_DIR;
  
  if ((tmp = getenv(PROXY)) != NULL)	  /* PROXY is #defined in rlpr-common.h */
    strncpy(net_.proxyhost, tmp, MAXHOSTNAMELEN);
  else net_.proxyhost[0] = '\0';

  if ((tmp = getenv(PRINTHOST)) != NULL)  /* PRINTHOST is #defined in rlpr-common.h */
    strncpy(net_.printhost, tmp, MAXHOSTNAMELEN);
  else net_.printhost[0] = '\0';

  /* first consult PRINTER, if it's NULL then LPDEST */
  if (!(tmp = getenv("PRINTER"))) tmp = getenv("LPDEST");
  if (tmp) strncpy(opts_.printer, tmp, MAX_QUEUE_LEN);
  else opts_.printer[0] = '\0';
}

void tidy_options(void) {
  struct passwd *pw;
  
  if (opts_.filetype == ' ')	             /* unleaded, please */
    opts_.filetype = 'f';

  /* set the username. this does not make me feel warm and fuzzy */
  
  if (!*opts_.user)
    if ((pw = getpwuid(getuid()))) 
      strncpy(opts_.user, pw->pw_name, MAX_USER_LEN); 
    else rlpr_fatal("who are you?");

  /* resolve remainder of printer/printhost information */
  
  if (!*net_.printhost && !*opts_.printer)
    rlpr_fatal("no host or queue specified -- consult man page");
  else if (!*net_.printhost)
    strncpy(net_.printhost, gethostfromq(opts_.printer), MAX_QUEUE_LEN);
  else if (!*opts_.printer)
    strncpy(opts_.printer, getqfromhost(net_.printhost), MAXHOSTNAMELEN);
}

/* add an entry to the control file */

void cf_add(char op, char *str, int cfd) { /* always terminated with LF */

  /* we could do an sprintf(), but we'd waste a lot of our time just
     malloc()'ing and free()'ing, so instead just employ brute force */

  safe_writen(cfd, &op, 1);
  safe_writen(cfd, str, strlen(str));
  safe_writen(cfd, "\n", 1);
}

/* uses options struct to put standard header on control file */

void cf_header(int cfd, char *filename) {

  if (filename == NULL)		            /* we're reading from stdin */
    filename = "stdin";
  
  cf_add('H', net_.localhost, cfd);        /* hostname */
  cf_add('P', opts_.user, cfd);	            /* printer name (printq name) */
  cf_add('W', opts_.width, cfd);            /* width for f, l and p files */

  if (opts_.indent_cols != NULL)	    /* indenting if appropriate */
    cf_add('I', opts_.indent_cols, cfd);

  if (opts_.mflag)			    /* mail user if appropriate */
    cf_add('M', opts_.user, cfd);	    

  if (*opts_.title)			    /* title for pr(1). rfc states */
    cf_add('T', opts_.title, cfd);	    /* it is ignored for other commands */

  if (opts_.bflag) {	                    /* print burst page */
    cf_add('L', opts_.user, cfd);	    /* rlpr never omits username */
    cf_add('C', *opts_.class ? opts_.class : "", cfd);
    cf_add('J', *opts_.job ? opts_.job : "", cfd);
  }

  if (opts_.filetype == 'p' && !*opts_.title) /* not sure if this is right */
    cf_add('N', filename, cfd);	              /* whine to me if you have an opinion */
}

void parse_args(int argc, char *argv[]) {
  int c;

  static struct option long_opts[] = {
    { "copies",    1, NULL, '#' },
    { "help",      0, NULL, -2  },
    { "indent",    1, NULL, 'i' },
    { "job",       1, NULL, 'J' },
    { "mail",      0, NULL, 'm' },
    { "no-burst",  0, NULL, 'h' },
    { "port",      1, NULL, -4  },
    { "printhost", 1, NULL, 'H' },
    { "printer",   1, NULL, 'P' },
    { "proxy",     1, NULL, 'X' },
    { "proxyhost", 1, NULL, 'X' },
    { "queue",     1, NULL, 'P' },
    { "quiet",     0, NULL, 'q' },
    { "remove",    0, NULL, 'r' },
    { "silent",    0, NULL, 'q' },
    { "tmpdir",    1, NULL, -3  },
    { "title",     1, NULL, 'T' },
    { "user",      1, NULL, 'U' },
    { "width",     1, NULL, 'w' },
    { "windows",   1, NULL, 'W' },
    { "version",   0, NULL, 'V' },
    { 0, 0, 0, 0 }
  };

  init_options();		          /* initialize options struct */
  
  while ((c = getopt_long(argc, 
			  argv, 
			  "1234cdfglnopqtv#:hi::mrsw::C:FH:J:P:Q:T:U:VWX:",
			  long_opts,
			  NULL)) != EOF)
    switch(c) {

      /* user is on best behavior here. that is, we assume the user
	 actually has a file of the proper type, without checking that it
	 is of that type. set the filetype characteristic accordingly */
      
    case '1':				  /* troff type R */
    case '2':				  /* troff type I */
    case '3':				  /* troff type B */
    case '4':				  /* troff type S */
    case 'c':				  /* cifplot */
    case 'd':				  /* TeX/DVI */
    case 'f':				  /* fortran */
    case 'g':				  /* graph */
    case 'l':				  /* leave control characters */
    case 'n':				  /* ditroff */
    case 'o':				  /* postscript (not in rlpr) */
    case 'p':				  /* pr */
    case 't':				  /* troff */
    case 'v':				  /* raster input */
      if (opts_.filetype != ' ')
	rlpr_fatal("cannot send as two different assumed file types!");
      opts_.filetype = (c == 'f') ? 'r' : c;
      break;
    case '#':				  /* number of copies to make */
      opts_.copies = atoi(optarg);
      break;
    case 'h':				  /* suppress printing of burst page */
      opts_.bflag  = 0;
      break;
    case 'i':				  /* indentation */
      opts_.indent_cols  = optarg ? optarg : DEFAULT_INDENT_NO_PARAM;
      break;
    case 'm':				  /* send mail after printing */
      opts_.mflag  = 1;
      break;
    case 'q':				  /* be quiet! */
      opts_.qflag  = 1;
      break;
    case 'r':				  /* remove file after printing */
      opts_.rflag  = 1;
      break;
    case 's':				  /* for compatibility with BSD lpr */
      rlpr_warn("symbolic link option not applicable (ignored)");
      break;
    case 'w':				  /* width for pr */
      opts_.width  = optarg ? optarg : DEFAULT_WIDTH_NO_PARAM;
      break;
    case 'C':				  /* class */
      strncpy(opts_.class, optarg, MAX_CLASS_LEN);
      break;
    case 'F':				  /* form feed after printing */
      opts_.fflag  = 1;
      break;
    case 'H':				  /* for overriding PRINTHOST */
      strncpy(net_.printhost, optarg, MAX_HOST_LEN);
      break;
    case 'J':				  /* job name on burst page */
      strncpy(opts_.job, optarg, MAX_JOB_LEN); 
      break;
    case 'P':				  /* printer queue name */
    case 'Q':				  /* i think this is a better name */
      strncpy(opts_.printer, optarg, MAX_QUEUE_LEN); 
      break;
    case 'T':				  /* title for pr (def: filename) */
      strncpy(opts_.title, optarg, MAX_TITLE_LEN); 
      break;
    case 'U':				  /* user (weary about this field) */
      strncpy(opts_.user, optarg, MAX_USER_LEN);
      break;
    case 'V':				  /* print version info */
      fprintf(stdout, "%s: version "VERSION" from "__DATE__" "__TIME__ 
              " -- meem@gnu.ai.mit.edu\n", name);
      exit(EXIT_SUCCESS);
    case 'W':
      opts_.wflag = 1;
      break;
    case 'X':			          /* for overriding PROXY  */
      strncpy(net_.proxyhost, optarg, MAXHOSTNAMELEN);
      break;
    case -2:
      printf("usage: rlpr [-Hprinthost]  [-Pprinter]  [-Xproxyhost] [OPTIONS] "
             "[file ...]\n\nplease see the manpage for detailed help.\n");
      exit(EXIT_SUCCESS);
    case -3:
      opts_.tmpdir = optarg;
      break;
    case -4:
      net_.port    = atoi(optarg);
      break;
    case '?':
      fprintf(stderr,"please see manpage for help\n");
      break;
    }
  tidy_options();
}

void rlpr_fatal(char *fmt, ...) {
  va_list ap;
  fprintf(stderr, "%s: fatal: ", name);
  va_start(ap, fmt);
  vfprintf(stderr, fmt, ap);
  fputc('\n', stderr);
  va_end(ap);
  exit(EXIT_FAILURE);
}

void rlpr_warn(char *fmt, ...) {
  va_list ap;
  if (opts_.qflag) return;
  fprintf(stderr, "%s: warning: ", name);
  va_start(ap, fmt);
  vfprintf(stderr, fmt, ap);
  fputc('\n', stderr);
  va_end(ap);
}

