# =============================================================================
#
# File:		dsk_FileList.tcl
# Project:	TkDesk
#
# Started:	09.10.94
# Changed:	09.10.94
# Author:	cb
#
# Description:	Implements a class that opens a file-list toplevel.
#
# Copyright (C) 1996  Christian Bolik
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
# See the file "COPYING" in the base directory of this distribution
# for more.
#
# =============================================================================

#
# =============================================================================
#
# Class:	dsk_FileList
# Desc:		Creates file-list toplevels.
#
# Methods:	
# Procs:	
# Publics:
#

itcl_class dsk_FileList {

    constructor {args} {
	global tkdesk env

	dsk_busy
	
	#
	# Create a toplevel with this object's name
	# (later accessible as $this-top):
	#
        set class [$this info class]
        ::rename $this $this-tmp-
        ::toplevel $this -class $class
	wm withdraw $this
        ::rename $this $this-top
        ::rename $this-tmp- $this

	frame $this.fMenu -bd 2 -relief raised
	pack $this.fMenu -fill x

	# ---- File Menu

	menubutton $this.fMenu.mbFile -text "File" -underline 0 \
		-menu $this.fMenu.mbFile.menu
	pack $this.fMenu.mbFile -side left

	menu [set m $this.fMenu.mbFile.menu]
	$m add command -label "Information" -underline 0 \
		-command "dsk_fileinfo" -accelerator "Ctrl-i"
	$m add command -label "New File" -underline 0 \
		-command "dsk_create file"
	$m add command -label "New Directory" -underline 4 \
		-command "dsk_create directory" -accelerator "Ctrl-d"
	$m add command -label "Copy, Move, Link... " -underline 0 \
		-command "dsk_copy" -accelerator "Ctrl-c"
	$m add command -label "Rename..." -underline 0 \
		-command "dsk_rename" -accelerator "Ctrl-r"
	$m add command -label "Delete..." -underline 1 \
		-command "dsk_delete" -accelerator "Del"
	$m add command -label "Print..." -underline 0 \
		-command "dsk_print" -accelerator "Ctrl-P"
	$m add separator
	$m add command -label "Find Files..." -underline 0 \
		-command "dsk_find_files" -accelerator "Ctrl-f"
	$m add command -label "Find Annotation..." -underline 5 \
		-command "dsk_find_annotation"
	$m add command -label "Copy to X selection" -underline 8 \
		 -command "$this select X"
	$m add command -label "Open selected files" -underline 0 \
		-command "dsk_openall"
	$m add command -label "Clear selection" -underline 6 \
		-command "$this select clear"
	$m add separator
	$m add command -label "Close Window" -underline 6 \
		-command "$this close"
	if !$tkdesk(xmaster) {
	    $m add command -label "Quit" -underline 0 -command "dsk_exit"
	} else {
	    $m add command -label "Quit X Windows" -underline 0 \
			-command "dsk_exit"
	}

	# ---- Directories Menu

	menubutton $this.fMenu.mbDirs -text "Directories" -underline 0 \
		-menu $this.fMenu.mbDirs.menu
	pack $this.fMenu.mbDirs -side left

	menu [set m $this.fMenu.mbDirs.menu]
	$m add command -label "Open..." -underline 0 \
		-command "dsk_ask_dir" -accelerator "Ctrl-o"
	$m add command -label "New..." -underline 0 \
		-command "dsk_create directory" -accelerator "Ctrl-d"
	$m add command -label "Home Directory " -underline 0 \
		-command "$this config -dir \$env(HOME)"
	$m add cascade -label "Trees" -menu ${m}.fs
	menu ${m}.fs
	menu ${m}.fs.home -postcommand "dsk_casdirs $env(HOME) ${m}.fs.home 1"
	${m}.fs add cascade -label "Home " -menu ${m}.fs.home
	menu ${m}.fs.root -postcommand "dsk_casdirs / ${m}.fs.root 1"
	${m}.fs add cascade -label "Root " -menu ${m}.fs.root
	$m add separator

	$m add command -label "Trash" -underline 0 \
		-command "dsk_FileList .dfl\[dsk_FileList :: id\] \
			-directory $tkdesk(configdir)/.trash"
	$m add command -label "Empty Trash" -underline 0 \
		-command "dsk_empty_trash"
	$m add separator

	if [info exists tkdesk(directories)] {
	    foreach mdir $tkdesk(directories) {
		if {$mdir == "-"} {
		    $m add separator
		} else {
		    _add_dir_to_menu $this $m $mdir
		}
	    }
	}

	bind $m <ButtonRelease-1> "
		set tkdesk(menu,control) 0
		[bind Menu <ButtonRelease-1>]"
	bind $m <Control-ButtonRelease-1> "
		set tkdesk(menu,control) 1
		[bind Menu <ButtonRelease-1>]"

	# ---- Others Menu
	menubutton $this.fMenu.mbOthers -text "Others" -underline 0 \
		-menu $this.fMenu.mbOthers.menu
	pack $this.fMenu.mbOthers -side left

	menu [set m $this.fMenu.mbOthers.menu]
	$m add command -label "Open Browser View" -underline 0 \
		-command "dsk_FileViewer .fv\[dsk_FileViewer :: id\] \
			-dir \[$this curdir\] -num_lbs \$tkdesk(num_lbs)"
	$m add cascade -label "Commands" -menu $this.fMenu.mbOthers.menu.cmd
	$m add cascade -label "Bookmarks" -menu $this.fMenu.mbOthers.menu.book
	$m add cascade -label "Options" -menu $this.fMenu.mbOthers.menu.opts
	$m add cascade -label "Help" -menu $this.fMenu.mbOthers.menu.help

	# ---- Commands Menu
	menu [set m $this.fMenu.mbOthers.menu.cmd]
	$m add command -label "Execute..." -underline 0 \
		-command "dsk_ask_exec" -accelerator "Ctrl-x"
	$m add command -label "Periodic Execution..." -underline 0 \
		-command "dsk_periodic"
	$m add command -label "Job Control" -underline 0 -command "dsk_jobs"
	$m add separator

	if [info exists tkdesk(commands)] {
	    foreach cmd $tkdesk(commands) {
		if {[llength $cmd] > 1} {
		    _add_cmd_to_menu $m $cmd
		} else {
		    $m add separator
		}
	    }
	}

	# ---- Bookmarks Menu
	menu [set m $this.fMenu.mbOthers.menu.book] \
		-postcommand "dsk_bookmark menu $m"
	# add dummy entry to work around bug in pre Tk 4.0p2:
	$m add command -label "dummy"
	bind $m <ButtonRelease-1> "
		set tkdesk(file_lb,control) 0
		[bind Menu <ButtonRelease-1>]"
	bind $m <Control-ButtonRelease-1> "
		set tkdesk(file_lb,control) 1
		[bind Menu <ButtonRelease-1>]"
	
	# ---- Options Menu
	menu [set m $this.fMenu.mbOthers.menu.opts]
	$m add checkbutton -label " Add Icons " -underline 1 \
	    -variable tkdesk(add_icons) \
	    -command "dsk_FileListbox :: addicons \$tkdesk(add_icons)
			$this refresh all"
	$m add checkbutton -label " Show All Files " -underline 1 \
	    -variable tkdesk(show_all_files) \
	    -command "dsk_FileListbox :: showall \$tkdesk(show_all_files)
			$this refresh all"
	$m add checkbutton -label " Folders On Top " -underline 1 \
	    -variable tkdesk(folders_on_top) \
	    -command "dsk_FileListbox :: topfolders \$tkdesk(folders_on_top)
			$this refresh all"
	$m add checkbutton -label " Append Type Char " -underline 2 \
	    -variable tkdesk(append_type_char) \
	    -command "dsk_FileListbox :: typechar \$tkdesk(append_type_char)
			$this refresh all"
	$m add checkbutton -label " Single Click (Dirs) " -underline 4 \
	    -variable tkdesk(single_click)
	$m add checkbutton -label " Always In Browser " -underline 11 \
	    -variable tkdesk(in_browser)
	$m add separator
	$m add checkbutton -label " Strip $env(HOME) " -underline 2 \
	    -variable tkdesk(strip_home) \
	    -command "$this refresh all"
	$m add checkbutton -label " Overwrite Always " -underline 1 \
	    -variable tkdesk(overwrite_always)
	$m add checkbutton -label " Really Delete " -underline 1 \
	    -variable tkdesk(really_delete)
	$m add checkbutton -label " Quick Drag'n'Drop " -underline 1 \
	    -variable tkdesk(quick_dragndrop)
	$m add checkbutton -label " Sort History " -underline 6 \
	    -variable tkdesk(sort_history)
	if [info exists tkdesk(soundcmd)] {
	    $m add checkbutton -label " Use Sound " -underline 8 \
		    -variable tkdesk(use_sound)
	}
	$m add checkbutton -label " Dialogs At Pointer " -underline 1 \
	    -variable tkdesk(at_pointer)
	
	# ---- Help Menu
	menu [set m $this.fMenu.mbOthers.menu.help]
	$m add command -label "User's Guide  " \
		-command "dsk_cbhelp $tkdesk(library)/doc/Guide howto" \
		-accelerator "F1"
	$m add command -label "FAQ" \
		-command "dsk_cbhelp $tkdesk(library)/doc/Guide#Frequently howto"
	$m add command -label "Changes" \
		-command "dsk_cbhelp $tkdesk(library)/doc/CHANGES"
	$m add command -label "License" \
		-command "dsk_cbhelp $tkdesk(library)/doc/License"
	$m add separator
	$m add checkbutton -label "Balloon Help" \
		-variable cb_tools(balloon_help)
	$m add command -label "About TkDesk..." \
		-command dsk_about

	# ---- Button Bar
	_button_bar

	# ---- Path Entry

	frame $this.fEntry -bd 1 -relief raised
	pack $this.fEntry -fill x

	entry $this.entry -bd 2 -relief sunken -width 5
	bindtags $this.entry "$this.entry Entry All"
	bind $this.entry <Return> "
	    $this config -dir \[%W get\]
	"
	bind $this.entry <Tab> "focus $this"
	bind $this.entry <3> "update idletasks; $this _path_popup %X %Y"
	cb_bindForCompletion $this.entry <Control-Tab>
    	blt_drag&drop target $this.entry \
			handler text "dd_handle_text $this.entry 1"

	pack $this.entry -in $this.fEntry -side left -fill x \
		-expand yes -padx $pad -pady $pad -ipady 2

	frame $this.fMenus
	pack $this.fMenus -in $this.fEntry -side left \
		-padx $tkdesk(pad) -pady $tkdesk(pad)

	menubutton $this.mbHist -bd 2 -relief raised \
		-bitmap @$tkdesk(library)/cb_tools/bitmaps/combo.xbm \
		-menu $this.mbHist.menu
	pack $this.mbHist -in $this.fMenus -side left -ipadx 2 -ipady 2

	menu $this.mbHist.menu \
		-postcommand "history buildmenu $this.mbHist.menu ; update"
	# add dummy entry to work around bug in pre Tk 4.0p2:
	$this.mbHist.menu add command -label "dummy"
	bind $this.mbHist.menu <ButtonRelease-1> "
		set tkdesk(menu,control) 0
		[bind Menu <ButtonRelease-1>]"
	bind $this.mbHist.menu <Control-ButtonRelease-1> "
		set tkdesk(menu,control) 1
		[bind Menu <ButtonRelease-1>]"

	dsk_FileListbox $this.flb -width $tkdesk(file_lb,minwidth) \
		-height $tkdesk(file_lb,minheight) \
		-pad $tkdesk(pad) -toplevel $this -notrivialdirs 0
	pack $this.flb -fill both -expand yes

	bind $this <Any-Enter> "set tkdesk(active_viewer) $this; break"
	bind $this <Tab> "focus $this.entry; break"

	bind $this <Control-i> "dsk_fileinfo; break"
	bind $this <Control-f> "dsk_find_files; break"
	bind $this <Control-d> "dsk_create directory; break"	
	bind $this <Control-c> "dsk_copy; break"
	bind $this <Control-r> "dsk_rename; break"
	bind $this <Delete> "dsk_delete; break"
	bind $this <Control-x> "dsk_ask_exec; break"
	bind $this <Control-o> "dsk_ask_dir; break"
	bind $this <Control-p> "dsk_print; break"
	bind $this <Return> "dsk_openall; break"
	bind $this <F1> "dsk_cbhelp $tkdesk(library)/doc/Guide howto"

	wm title $this "File List"
	wm minsize $this $tkdesk(file_lb,minwidth) $tkdesk(file_lb,minheight)
	#wm geometry $this $tkdesk(file_lb,width)x$tkdesk(file_lb,height)
	dsk_place_window $this flist \
		$tkdesk(file_lb,width)x$tkdesk(file_lb,height) 1
	wm protocol $this WM_DELETE_WINDOW "$this close"

	if $tkdesk(fvwm) {
	    # create the icon window
	    # (this code is based upon the code posted by:
	    # kennykb@dssv01.crd.ge.com (Kevin B. Kenny))
	    toplevel $this-icon -bg $tkdesk(color,icon_background)
	    wm withdraw $this-icon
	    label $this-icon.label \
		    -image [cb_image $tkdesk(icon,filelist)] \
		    -width 32 -height 32 -bd 0 \
		    -bg $tkdesk(color,icon_background)
	    pack $this-icon.label -ipadx 2 -ipady 2
	    blt_drag&drop target $this-icon.label handler \
		    file "$this _dd_drophandler"
	    update idletasks
	    wm geometry $this-icon \
		    [winfo reqwidth $this-icon]x[winfo reqheight $this-icon]
	    wm protocol $this-icon WM_DELETE_WINDOW "$this delete"
	    wm iconwindow $this $this-icon
	} else {
	    wm iconbitmap $this @$tkdesk(library)/images/filing_open.xbm
	}

	eval config $args
	dsk_sound dsk_new_filelist

	if !$dontmap {
	    wm deiconify $this
	    tkwait visibility $this
	    catch "lower $this .dsk_welcome"
	    update
	}
	update
	$this.entry icursor end
	$this.entry xview end

	dsk_lazy
    }

    destructor {
	global env tkdesk
	
	# add current directory to the path history:
	if {$last_directory != ""} {
	    if {[string first $env(HOME) $last_directory] == 0} {
		history add [string_replace $last_directory $env(HOME) ~]
	    } else {
		history add $last_directory
	    }
	}
	
	$this.flb delete
        #after 10 "rename $this-top {}"		;# delete this name
        catch {destroy $this}		;# destroy associated windows
	catch {destroy $this-icon}
	set tkdesk(active_viewer) ""
	foreach fv "[itcl_info objects -class dsk_FileViewer] \
		[itcl_info objects -class dsk_FileList]" {
	    if {"$fv" != "$this"} {
		set tkdesk(active_viewer) $fv
	    }
	}
    }

    #
    # ----- Methods and Procs -------------------------------------------------
    #

    method config {config} {
    }

    method cget {var} {
	return [set [string trimleft $var -]]
    }

    method close {} {
	global tkdesk env

	# add current directory to history before closing window:
	if {[string first $env(HOME) $directory] == 0} {
	    history add [string_replace $directory $env(HOME) ~]
	} else {
	    history add $directory
	}
	
	if [winfo exists .dsk_appbar] {
	    $this delete
	} elseif {[dsk_active viewer] == 1} {
	    # about to close last window
	    dsk_exit 1
	} else {
	    $this delete
	}
    }

    method curdir {} {
	return $directory
    }

    method refresh {{mode ""}} {
	if {$mode == "all"} {
	    foreach fv [itcl_info objects -class dsk_FileViewer] {
	    	$fv refresh
	    }
	}
    }

    method refreshdir {dir} {
	global tkdesk

	if {$dir == $directory} {
	    # $this.flb config -directory $directory
	    $this.flb refresh
	    if {$directory == "$tkdesk(configdir)/.trash/"} {
		if $tkdesk(fvwm) {
		    if {[llength [$this.flb.dlb get]] == 0} {
			$this-icon.label config \
				-image [cb_image $tkdesk(trash:empty)]
		    } else {
			$this-icon.label config \
				-image [cb_image $tkdesk(trash:full)]
		    }
		} else {
		    if {[llength [$this.flb.dlb get]] == 0} {
			wm iconbitmap $this \
				@$tkdesk(library)/images/trashcan.xbm
		    } else {
			wm iconbitmap $this \
				@$tkdesk(library)/images/trashcan_full.xbm
		    }
		}
	    }
	}
    }

    method select {cmd} {
	global tkdesk
	
	switch -glob -- $cmd {
	    get		{# return a list of all selected files
		    set sfl ""
		    set sl [$this.flb.dlb select get]
		    if {$sl != ""} {
		        set fl [$this.flb.dlb get]
			foreach s $sl {
			    set file [lindex [split [lindex $fl $s] \t] 0]
			    set file [string trimright $file " "]
			    if $tkdesk(append_type_char) {
				set file [dsk_striptc $file]
			    }
			    lappend sfl "$directory$file"
			}
		    }
		    return $sfl
			}
	    clear	{# clear selection in listbox
		    $this.flb.dlb select clear
			}
	    X		{# copy selected filenames to X selection
		    set sfl [$this select get] 
		    if {$sfl != ""} {
		        selection handle $this "$this _retrieve_X"
		        selection own $this
		    } else {
		        cb_info "Please select one or more files first."
		    }
			}
	    default	{
		    error "$this select: unknown option $cmd"
			}
	}
    }

    method _retrieve_X {offset maxBytes} {
	# return "offset: $offset, maxBytes: $maxBytes"

	if {$offset != 0} {
	    # this should never happen, so return
	    return ""
	}

	set sfl [$this select get]
	set rl ""
	if {$sfl != ""} {
	    if {[llength $sfl] > 1} {
	    	foreach file $sfl {
		    append rl "$file\n"
	    	}
	    } else {
		set rl $sfl
	    }
	}
	if {[string length $rl] > $maxBytes} {
	    return [string range $rl 0 [expr $maxBytes - 1]]
	} else {
	    return $rl
	}
    }

    method _dblclick {lb sel} {
	global tkdesk
	
	if {$sel == "" || $lb == ""} {
	    return
	}
	if {$tkdesk(single_click) && [llength $sel] > 1} {
	    return
	}

	set dir [string trimright [$lb info public directory -value] "/"]
	#set file [lindex [lindex [$lb.dlb get] [lindex $sel 0]] 0]
	set file [string trimright [lindex [split [lindex [$lb.dlb get] \
		[lindex $sel 0]] \t] 0] " "]
	if $tkdesk(append_type_char) {
	    set file [dsk_striptc $file]
	}
	if {[string first "/" $file] == -1} {
	    set file "$dir/$file"
	}
	if {!$tkdesk(single_click) || \
		($tkdesk(single_click) && [file isdirectory $file])} {
	    ::dsk_open $this "$file"
	}
    }

    method _popup {lb sel mx my} {
	if {$sel == "" || $lb == ""} {
	    return
	}
	set dir [string trimright [$lb info public directory -value] "/"]
	#set file [lindex [lindex [$lb.dlb get] [lindex $sel 0]] 0]
	set file [string trimright [lindex [split [lindex [$lb.dlb get] \
		[lindex $sel 0]] \t] 0] " "]
	::dsk_popup $lb "$dir/$file" $mx $my
    }

    method _dd_drophandler {} {
	global DragDrop tkdesk

	catch "wm withdraw $tkdesk(dd_token_window)"
	update
	set dest $directory

	if ![file writable $dest] {
	    dsk_errbell
	    if {$dest != ""} {
	    	cb_error "You don't have write permission for this directory!"
	    } else {
		cb_error "This listbox is not a valid target (since it's empty)."
	    }
	}

	#dsk_debug "Rec.: $DragDrop(file)"
	#dsk_debug "dest: $dest"
	if {[string first "$tkdesk(configdir)/.trash/" $dest] == -1} {
	    dsk_ddcopy $DragDrop(file) $dest
	} else {
	    if !$tkdesk(quick_dragndrop) {
		dsk_delete $DragDrop(file)
	    } else {
		if {!$tkdesk(file_lb,control) && !$tkdesk(really_delete)} {
		    dsk_ddcopy $DragDrop(file) $dest
		} else {
		    if {[cb_yesno "Really deleting! Are you sure that this is what you want?"] == 0} {
			dsk_sound dsk_really_deleting
			dsk_bgexec "$tkdesk(cmd,rm) $DragDrop(file)" \
				"Deleting [llength $DragDrop(file)] File(s)..."
			dsk_refresh "$DragDrop(file) $dest"
		    }
		}		    
	    }
	}
    }

    method _path_popup {x y} {
	global tkdesk
	
	set m $this-ppop
	catch {destroy $m}
	menu $m -tearoff 0
	bind $m <ButtonRelease-3> "
		set tkdesk(menu,control) 0
		[bind Menu <ButtonRelease-3>]"
	bind $m <Control-ButtonRelease-3> "
		set tkdesk(menu,control) 1
		[bind Menu <ButtonRelease-3>]"
	
	set p [cb_tilde [string trimright $directory "/"] collapse]
	if {$p != ""} {
	    while {[set p [file dirname $p]] != "/"} {
		$m add command -label $p -command "$this config -dir $p" \
			-font $tkdesk(font,entries)
	    }
	    $m add command -label "/" -command "$this config -dir /" \
			-font $tkdesk(font,entries)
	    tk_popup $m $x $y
	}
    }

    method _button_bar {} {
	global tkdesk
	
	if {[llength $tkdesk(small_button_bar)] > 0} {
	    catch {
		destroy $this.fBBo
		destroy $this.fBB
	    }
	    
	    frame $this.fBBo -bd 1 -relief raised
	    pack $this.fBBo -after $this.fMenu -fill x 

	    frame $this.fBB
	    pack $this.fBB -in $this.fBBo -fill x \
		    -padx $tkdesk(pad) -pady $tkdesk(pad)

	    set bcnt 0
	    foreach but $tkdesk(small_button_bar) {
		if {[llength $but] == 1} {
		    frame $this.fBB.f$bcnt -width $tkdesk(pad)
		    pack $this.fBB.f$bcnt -in $this.fBB -side left \
			-padx $tkdesk(pad) -pady $tkdesk(pad)
		    incr bcnt
		    continue
		}

		set bitmap [lindex $but 0]
		set bgcolor [. cget -background]
		set fgcolor black
		if {[llength $bitmap] > 1} {
		    if {[lindex $bitmap 1] != ""} {
			set fgcolor [lindex $bitmap 1]
		    }
		    if {[llength $bitmap] > 2} {
		    	if {[lindex $bitmap 2] != ""} {
			    set bgcolor [lindex $bitmap 2]
		    	}
		    }
		    set bitmap [lindex $bitmap 0]
		}
		if {[string index $bitmap 0] == "/" || \
		    [string index $bitmap 0] == "~"} {
		    set bitmap $bitmap
		} else {
		    foreach p [split $tkdesk(path,images) ":"] {
			if [file exists $p/$bitmap] {
			    set bitmap $p/$bitmap
			    break
			}
		    }
		}
		if ![file exists $bitmap] {
		    set bitmap $tkdesk(library)/images/xlogo16.xpm
		}

	    	set action [string_replace [lindex $but 1] \" \\\"]

		set desc ""
		if {[llength $action] > 1} {
		    set desc [lindex $action 1]
		    set action [lindex $action 0]
		}
		dsk_debug "BB$bcnt action: $action"

		button $this.fBB.b$bcnt \
			-image [cb_image $bitmap -background $bgcolor \
			-foreground $fgcolor] \
			-takefocus 0 -highlightthickness 0 \
			-activebackground $bgcolor -activeforeground $fgcolor \
			-command "cd \[dsk_active dir\] ;\
			eval \[_expand_pc \"$action\"\]; cd ~"
		pack $this.fBB.b$bcnt -in $this.fBB -side left \
			-padx 0 -pady 0 \
			-ipadx 2 -ipady 2

		if {$desc != ""} {
		    cb_balloonHelp $this.fBB.b$bcnt "$desc"
		}

		incr bcnt
	    }
	}
    }

    proc id {{cmd ""}} {
	if {$cmd == ""} {
	    set i $id
	    incr id
	    return $i
	} elseif {$cmd == "reset"} {
	    set id 0
	}
    }

    #
    # ----- Variables ---------------------------------------------------------
    #

    public dontmap 0

    public dir {} {
	global tkdesk

	if ![winfo exists $this] return

	set err [catch {set isdir [file isdirectory $dir]}]
	if !$err {
	    if !$isdir {
		catch {set dir [_make_path_valid $dir]}
		catch {dsk_bell}
		cb_alert "The path you specified is not completely valid."
	    } elseif ![file readable $dir] {
		dsk_errbell
		cb_error "[file tail $dir]: Permission denied."
		return
	    }
	} else {
	    cb_error "Path (or user?) does not exist. Opening home directory."
	    set dir ~
	}

	if $tkdesk(menu,control) {
	    dsk_FileList .dfl[dsk_FileList :: id] -directory $dir
    	    set tkdesk(menu,control) 0
	} else {
	    $this config -directory $dir
	}
    }

    public directory {} {
	global env tkdesk

	if ![string match {[~/]*} $directory] {
	    set directory [dsk_active dir]$directory
	}
	set directory "[string trimright [glob $directory] "/"]/"

	$this.entry delete 0 end
	$this.entry insert end [cb_tilde $directory collapse]

	# right-justify the text in the path entry:
	if {[wm state $this] != "withdrawn"} {
	    $this.entry icursor end
	    $this.entry xview end
	}

	$this.flb config -directory $directory

	# add this directory to the path history:
	if {$last_directory != ""} {
	    if {[string first $env(HOME) $last_directory] == 0} {
		history add [string_replace $last_directory $env(HOME) ~]
	    } else {
		history add $last_directory
	    }
	}
	set last_directory $directory

	if {[string first "$tkdesk(configdir)/.trash/" $directory] > -1} {
	    if {$directory == "$tkdesk(configdir)/.trash/"} {
		$this.flb config -notrivialdirs 1 -showall 1
	    } else {
		$this.flb config -notrivialdirs 0 -showall 1
	    }
	    pack forget $this.fEntry
	    wm title $this Trash
	    wm iconname $this Trash
	    if $tkdesk(fvwm) {
		if {[llength [$this.flb.dlb get]] == 0} {
		    $this-icon.label config \
			    -image [cb_image $tkdesk(trash:empty)]
		} else {
		    $this-icon.label config \
			    -image [cb_image $tkdesk(trash:full)]
		}
	    } else {
		if {[llength [$this.flb.dlb get]] == 0} {
		    wm iconbitmap $this @$tkdesk(library)/images/trashcan.xbm
		} else {
		    wm iconbitmap $this \
			    @$tkdesk(library)/images/trashcan_full.xbm
		}
	    }
	} else {
	    set wt [file tail [string trimright $directory "/"]]/
	    wm title $this $wt
	    wm iconname $this $wt
	    if $tkdesk(fvwm) {
		$this-icon.label config \
			-image [cb_image $tkdesk(library)/images/filing.xpm]
	    } else {
		wm iconbitmap $this @$tkdesk(library)/images/filing_open.xbm
	    }
	}

	if !$height_set {
	    set height_set 1
	    set h [cb_max [cb_min [expr [llength [$this.flb.dlb get]] + 1] \
		    $tkdesk(file_lb,height)] $tkdesk(file_lb,minheight)]
	    wm geometry $this $tkdesk(file_lb,width)x$h
	}
    }

    public pad {4} {
    }

    protected height_set 0
    protected last_directory ""

    common id 0
}


