# =============================================================================
#
# File:		appbar.tcl
# Project:	TkDesk
#
# Started:	13.11.94
# Changed:	13.11.94
# Author:	cb
#
# Description:	Implements an application bar. This features popup menus
#		and drag and drop targets.
#
# Copyright (C) 1996  Christian Bolik
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
# See the file "COPYING" in the base directory of this distribution
# for more.
#
# -----------------------------------------------------------------------------
#
# Sections:
#    proc dsk_appbar {args}
#    proc _appbar_create {}
#    proc _appbar_add_to_menu {menu cmd}
#    proc _appbar_unpost_parents {menu}
#    proc _appbar_dd_action {cmd}
#    proc _appbar_show_menu {butnum rootx rooty}
#    proc _appbar_close {}
#    proc _appbar_raise {}
#    proc _appbar_layout {orient}
#    proc _appbar_move {}
#    proc _appbar_date {}
#
# =============================================================================

#
# -----------------------------------------------------------------------------
#
# Proc:		dsk_appbar
# Args:		layout <horizontal|vertical>	change the layout
#		move		move the application bar around
# Returns: 	""
# Desc:		Meta proc for all things that concern the appbar.
# Side-FX:	none
#

proc dsk_appbar {args} {
    global tkdesk

    dsk_progress "Creating the application bar..."
    if ![info exists tkdesk(appbar)] {
	dsk_errbell
	cb_error "Couldn't read config file AppBar. Sorry, no application bar available."
	return
    }

    if {$args == ""} {
	_appbar_create
    } else {
	set cmd [lindex $args 0]
	set opts [lrange $args 1 [llength $args]]
	switch $cmd {
	    layout	{eval _appbar_layout $opts}
	    move	{eval _appbar_move}
	    raise	{eval _appbar_raise}
	    close	{eval _appbar_close}
	}
    }
}

#
# -----------------------------------------------------------------------------
#
# Proc:		_appbar_create
# Args:		none
# Returns: 	""
# Desc:		Builds and displays the application bar.
# Side-FX:	none
#

if ![info exists tkdesk(geometry,dsk_appbar)] {
    set tkdesk(geometry,dsk_appbar) ""
}

proc _appbar_create {} {
    global tkdesk dsk_appbar

    set t .dsk_appbar
    if [winfo exists $t] {
	cb_raise $t
	return
    }

    dsk_busy 

    toplevel $t
    wm withdraw $t

    set side top
    set fside left
    set dsk_appbar(layout) vertical
    if {$tkdesk(geometry,dsk_appbar) != ""} {
	set glist [split $tkdesk(geometry,dsk_appbar) x+]
	if {[lindex $glist 0] > [lindex $glist 1]} {
	    set side left
	    set fside top
	    set dsk_appbar(layout) horizontal
	}
    }
    if ![info exists tkdesk(appbar,max)] {set tkdesk(appbar,max) 100}

    set count 0
    set fcount 0
    foreach but $tkdesk(appbar) {
	if {[expr $count % $tkdesk(appbar,max)] == 0} {
	    incr fcount	    
	    frame $t.f$fcount -bg $tkdesk(color,icon_background)
	    pack $t.f$fcount -side $fside -fill both
	}

	if {[llength $but] > 1} {
	    set bitmap [lindex $but 0]
	    set bgcolor [. cget -background]
	    set fgcolor black
	    if {[llength $bitmap] > 1} {
		if {[lindex $bitmap 1] != ""} {
		    set fgcolor [lindex $bitmap 1]
		}
		if {[llength $bitmap] > 2} {
		    if {[lindex $bitmap 2] != ""} {
			set bgcolor [lindex $bitmap 2]
		    }
		}
		set bitmap [lindex $bitmap 0]
	    }
	    if {[string index $bitmap 0] == "/" || \
		    [string index $bitmap 0] == "~"} {
		set bitmap $bitmap
	    } else {
		foreach p [split $tkdesk(path,images) ":"] {
		    if [file exists $p/$bitmap] {
			set bitmap $p/$bitmap
			break
		    }
		}
	    }
	    if ![file exists $bitmap] {
		set bitmap $tkdesk(library)/images/xlogo.xbm
	    }

	    set appmenu [lindex $but 1]
	    menu [set m $t.m$count] -disabledforeground blue2
	    set dsk_appbar(num_cas) 0
	    set dsk_appbar(defaction) ""
	    set dsk_appbar(deflabel) ""
	    set dsk_appbar(ddaction) ""
	    if [$m cget -tearoff] {
		set inr 1
	    } else {
		set inr 0
	    }
	    foreach me $appmenu {
		if {[llength $me] == 1} {
		    if {$me == "-"} {
			$m add separator
		    } elseif {$me == "history:dirs"} {
			$m add cascade -label "Directories" -menu $m.mhd
			menu $m.mhd -postcommand \
				"history buildmenu $m.mhd open; update"
			# add dummy entry to work around bug in pre Tk 4.0p2:
			$m.mhd add command -label "dummy"
			bind $m.mhd <Visibility> "
			    if \{\[$m index active\] != $inr\} \{
			         %W unpost
			    \}"
		    } elseif {$me == "history:files"} {
			$m add cascade -label "Files" -menu $m.mhf
			menu $m.mhf -postcommand \
				"file_history buildmenu $m.mhf; update"
			# add dummy entry to work around bug in pre Tk 4.0p2:
			$m.mhf add command -label "dummy"
			bind $m.mhf <ButtonRelease-3> "
			      set tkdesk(file_lb,control) 0
			      [bind Menu <ButtonRelease-3>]"
			bind $m.mhf <Control-ButtonRelease-3> "
			      set tkdesk(file_lb,control) 1
			      [bind Menu <ButtonRelease-3>]"
			bind $m.mhf <Visibility> "
			    if \{\[$m index active\] != $inr\} \{
			         %W unpost
			    \}"
		    } elseif {$me == "history:execs"} {
			$m add cascade -label "Commands" -menu $m.mhe
			menu $m.mhe -postcommand \
				"exec_history buildmenu $m.mhe; update"
			# add dummy entry to work around bug in pre Tk 4.0p2:
			$m.mhe add command -label "dummy"
			bind $m.mhe <ButtonRelease-3> "
			      set tkdesk(file_lb,control) 0
			      [bind Menu <ButtonRelease-3>]"
			bind $m.mhe <Control-ButtonRelease-3> "
			      set tkdesk(file_lb,control) 1
			      [bind Menu <ButtonRelease-3>]"
			bind $m.mhe <Visibility> "
			if \{\[$m index active\] != $inr\} \{
				%W unpost
				\}"
		    } elseif {$me == "bookmarks"} {
			$m add cascade -label "Bookmarks" -menu $m.book
			menu $m.book -postcommand "dsk_bookmark menu $m.book"
			# add dummy entry to work around bug in pre Tk 4.0p2:
			$m.book add command -label "dummy"
			bind $m.book <ButtonRelease-3> "
			   set tkdesk(file_lb,control) 0
			   [bind Menu <ButtonRelease-3>]"
			bind $m.book <Control-ButtonRelease-3> "
			   set tkdesk(file_lb,control) 1
			   [bind Menu <ButtonRelease-3>]"
		    } elseif {$me == "config"} {
			$m add cascade -label "Configuration" -menu $m.cfg
			menu [set tm $m.cfg]
			#menu $m.cfg
			#$m.cfg add cascade -label "Edit Config Files" \
			#	 -menu $m.cfg.edmenu
			#$m.cfg add cascade -label "Reread Config Files" \
			#	 -menu $m.cfg.rdmenu
			#
			#menu [set tm $m.cfg.edmenu]
			$tm add command -label "All" \
				-command "dsk_edit_configs"
			$tm add separator
			foreach cf $tkdesk(configfiles) {
			    $tm add command -label $cf \
				    -command "dsk_edit_configs $cf"
			}

			#menu [set tm $m.cfg.rdmenu]
			#$tm add command -label "All" \
			#	 -command "dsk_reread_config"
			#$tm add separator
			#foreach cf $tkdesk(configfiles) {
			#    $tm add command -label $cf \
			#	     -command "dsk_reread_config $cf"
			#}
	            } else {
			$m add command -label [subst [lindex $me 0]] \
				-state disabled
		    }
		} else {
		    _appbar_add_to_menu $m $me
		}
		incr inr
	    }
	    
	    button $t.b$count -image [cb_image $bitmap -background $bgcolor \
		    -foreground $fgcolor] \
		    -activebackground $bgcolor -activeforeground $fgcolor \
		    -cursor top_left_arrow -command $dsk_appbar(defaction) \
		    -padx 0 -pady 0 -highlightthickness 0
	    pack $t.b$count -in $t.f$fcount -side $side -fill both \
		     -ipadx 2 -ipady 2
	    bind $t.b$count <3> "_appbar_show_menu $count %X %Y"
	    bind $t.m$count <B3-Motion> "_appbar_motion $count %X %Y"
	    cb_balloonHelp $t.b$count  $dsk_appbar(deflabel)

	    if {$dsk_appbar(ddaction) != ""} {
		blt_drag&drop target $t.b$count handler file \
			"_appbar_dd_action \"$dsk_appbar(ddaction)\""
	    }

	} else {
	    set special [lindex $but 0]
	    switch [lindex $special 0] {
		date {
		    _appbar_date $t.f$fcount $side
		    # the date occupies 2 buttons:
		    incr count
		}
	    }
	}

	incr count
    }


    wm title $t "TkDesk Application Bar"
    wm overrideredirect $t [expr !$tkdesk(appbar,wm_managed)]
    wm deiconify $t

    if {$tkdesk(geometry,dsk_appbar) == ""} {
	wm geometry $t +0+0
    } else {
	set glist [split $tkdesk(geometry,dsk_appbar) x+]
	wm geometry $t +[lindex $glist 2]+[lindex $glist 3]
    }

    dsk_lazy
}

proc _appbar_add_to_menu {menu cmd} {
	global tkdesk dsk_appbar

	if {[llength $cmd] == 2} {
	    set label [lindex $cmd 0]
	    set command [string_replace [lindex $cmd 1] \" \\\"]
	    if {$label != "dd" && $label != "DD"} {
	        $menu add command -label $label \
			-command "$menu unpost ;\
			_appbar_unpost_parents $menu ;\
			cd \[dsk_active dir\] ;\
			eval \[_expand_pc [list $command]\]; cd ~"
	        if {$dsk_appbar(defaction) == ""} {
		    set dsk_appbar(defaction) \
			    "cd \[dsk_active dir\] ;\
			    eval \[_expand_pc [list $command]\]; cd ~"
		    set dsk_appbar(deflabel) $label
	    	}
	    } else {
		set dsk_appbar(ddaction) $command
	    }
	} elseif {[llength $cmd] == 1} {
	    $menu add separator
	} else {
	    set m ${menu}.mc$dsk_appbar(num_cas)
	    incr dsk_appbar(num_cas)
	    $menu add cascade -label [lindex $cmd 0] -menu $m

	    menu $m
	    set cmd [lreplace $cmd 0 0]
	    foreach c $cmd {
	    	_appbar_add_to_menu $m $c
	    }
	}
}

proc _appbar_unpost_parents {menu} {

    set p [winfo parent $menu]
    while {$p != ""} {
	if {[winfo class $p] == "Menu"} {
	    catch "$p unpost"
	}
	set p [winfo parent $p]
    }
}

proc _appbar_dd_action {cmd} {
    global DragDrop tkdesk

    catch "wm withdraw $tkdesk(dd_token_window)"
    update
    
    if {[string first %A $cmd] > -1} {
	set cmd [string_replace $cmd %A $DragDrop(file)]
    } else {
	set cmd [_expand_pc $cmd]
    }
    cd [dsk_active dir]
    eval $cmd
    cd ~
}

proc _appbar_show_menu {butnum rootx rooty} {
    global dsk_appbar

    set t .dsk_appbar

    set geom [split [wm geometry $t] x+]
    set tw [lindex $geom 0]
    set th [lindex $geom 1]
#    set bx [lindex $geom 2]
#    set by [lindex $geom 3]
    set bgeom [split [winfo geometry $t.b$butnum] x+]
    set bw [lindex $bgeom 0]
    set bh [lindex $bgeom 1]
    set bx [winfo rootx $t.b$butnum]
    set by [winfo rooty $t.b$butnum]
    set sw [winfo screenwidth $t]
    set sh [winfo screenheight $t]
    set mw [winfo reqwidth $t.m$butnum]
    set mh [winfo reqheight $t.m$butnum]

    if {$tw > $th} {
	# horizontal layout
	set x [winfo rootx $t.b$butnum]
	if {$by > ($sh >> 1)} {
	    set y [expr $by - $mh]
	} else {
	    set y [expr $by + $bh]
	}
    } else {
	# vertical layout
	set y [winfo rooty $t.b$butnum]
	if {$bx > ($sw >> 1)} {
	    set x [expr $bx - $mw]
	} else {
	    set x [expr $bx + $bw]
	}
    }

    #cb_MenuPopupAdd $t.b$butnum 3 $t.m$butnum {} {} 1 $x $y 1
    update
    set dsk_appbar(lastbut) $t.b$butnum
    tk_popup $t.m$butnum $x $y
}

set dsk_appbar(motion) 0
proc _appbar_motion {lastnum x y} {
    global dsk_appbar

    if $dsk_appbar(motion) return
    set dsk_appbar(motion) 1
    set t .dsk_appbar
    set new [winfo containing $x $y]
    
    if {$new != $dsk_appbar(lastbut)} {
	if [string match $t.b* $new] {
	    $t.m$lastnum unpost
	    scan $new "$t.b%d" num
	    _appbar_show_menu $num $x $y
	}
    }
    set dsk_appbar(motion) 0
}


#
# -----------------------------------------------------------------------------
#
# Proc:		_appbar_close
# Args:		none
# Returns: 	""
# Desc:		Removes the application bar.
# Side-FX:	none
#

proc _appbar_close {} {
    global tkdesk

    if {[dsk_active viewer] != 0} {
	if [winfo exists .dsk_appbar] {
	    set tkdesk(geometry,dsk_appbar) [wm geometry .dsk_appbar]
	    destroy .dsk_appbar
	}
    } else {
	cb_info "The application bar cannot be closed because there is no file browser window on screen."
    }
}

#
# -----------------------------------------------------------------------------
#
# Proc:		_appbar_raise
# Args:		none
# Returns: 	""
# Desc:		Raises the application bar.
# Side-FX:	none
#

proc _appbar_raise {} {
    global tkdesk

    if [winfo exists .dsk_appbar] {
    	raise .dsk_appbar
    }
}

#
# -----------------------------------------------------------------------------
#
# Proc:		_appbar_layout
# Args:		orient		orientation: horizontal or vertical
# Returns: 	""
# Desc:		Repacks the buttons of the appbar accordingly to $orient.
# Side-FX:	none
#

proc _appbar_layout {orient} {
    global dsk_appbar

    if ![winfo exists .dsk_appbar] return

    if {$orient == "horizontal"} {
	set side left
	set fside top
	set dsk_appbar(layout) horizontal
    } else {
	set side top
	set fside left
	set dsk_appbar(layout) vertical
    }

    foreach obj [winfo children .dsk_appbar] {
	if {[winfo class $obj] == "Button" || \
		[winfo class $obj] == "AppDate"} {
	    if {[winfo class $obj] == "AppDate"} {
		_appbar_date "" $side
	    }
	    pack config $obj -side $side
	} elseif {[winfo class $obj] == "Frame"} {
	    pack config $obj -side $fside
	}
    }
}

#
# -----------------------------------------------------------------------------
#
# Proc:		_appbar_move
# Args:		none
# Returns: 	""
# Desc:		Displays a hand cursor to move the appbar around.
# Side-FX:	none
#

proc _appbar_move {} {
    global dsk_appbar tkdesk cb_tools

	if {[winfo depth .] != 1} {
	    set cc wheat
	} else {
	    set cc white
	}

    catch "unset dsk_appbar(released)"
	catch {destroy .dsk_appbar._Busy}
    foreach but [winfo children .dsk_appbar] {
	if {[winfo class $but] == "Frame"} continue
	$but config -cursor "@$tkdesk(library)/images/hand.xbm \
			$tkdesk(library)/images/hand.mask.xbm \
			black $cc"
	bind $but <B1-Motion> {
		wm geometry .dsk_appbar +[expr %X - 16]+[expr %Y - 16]; break}
	bind $but <ButtonRelease-1> {set dsk_appbar(released) 1; break}
	bind $but <Button-1> {set dummy 1; unset dummy; break}
    }

    set cbbh $cb_tools(balloon_help)
    set cb_tools(balloon_help) 0
    set gl [split [wm geometry .dsk_appbar] x+]
    ot_warp_pointer [expr [lindex $gl 2] + 16] [expr [lindex $gl 3] + 16]
    grab -global .dsk_appbar
    tkwait variable dsk_appbar(released)
    grab release .dsk_appbar
    set cb_tools(balloon_help) $cbbh

    catch "unset dsk_appbar(released)"
    foreach but [winfo children .dsk_appbar] {
	if {[winfo class $but] == "Frame"} continue
	$but config -cursor top_left_arrow
	bind $but <B1-Motion> {}
	bind $but <ButtonRelease-1> {}
	bind $but <Button-1> {}
    }
}

#
# -----------------------------------------------------------------------------
#
# Proc:		_appbar_date
# Args:		none
# Returns: 	""
# Desc:		Displays the time and date in the application bar.
# Side-FX:	none
#

proc _appbar_date {frame side} {
    global tkdesk dsk_appbar

    if {$frame != ""} {
	set dsk_appbar(date,frame) $frame
    } else {
	set frame $dsk_appbar(date,frame)
    }
    
    set ft .dsk_appbar.fDate
    if ![winfo exists $ft] {
	frame $ft -class "AppDate"
	pack $ft -fill both -expand yes -side $side -in $frame
    }
    set f .dsk_appbar.fDate.f
    catch {destroy $f}
    frame $f
    pack $f -fill both -expand yes -in $ft

    label $f.lTime -font $tkdesk(appbar,font,time) \
	    -pady 0 -bg black -fg green -cursor top_left_arrow \
	    -bd 2 -relief raised
    label $f.lWeekday -font $tkdesk(appbar,font,weekday) \
	    -pady 0 -cursor top_left_arrow
    label $f.lDay -font $tkdesk(appbar,font,day) \
	    -pady 0 -cursor top_left_arrow
    label $f.lMonth -font $tkdesk(appbar,font,month) \
	    -pady 0 -cursor top_left_arrow

    if {$dsk_appbar(layout) == "vertical"} {
	canvas $f.cDate -bd 2 -relief raised -width 36 -height 76 \
		-cursor top_left_arrow -highlightthickness 0

	$f.cDate create window 20 2 -window $f.lTime -anchor n
	$f.cDate create window 20 20 -window $f.lWeekday -anchor n
	$f.cDate create window 20 34 -window $f.lDay -anchor n
	$f.cDate create window 20 58 -window $f.lMonth -anchor n
    } else {
	canvas $f.cDate -bd 2 -relief raised -width 76 -height 36 \
		-cursor top_left_arrow -highlightthickness 0

	$f.cDate create window 20 2 -window $f.lTime -anchor n
	$f.cDate create window 20 17 -window $f.lMonth -anchor n
	$f.cDate create window 60 5 -window $f.lDay -anchor n -height 20
	$f.cDate create window 60 25 -window $f.lWeekday -anchor n -height 10
    }
    raise $f.lTime
    raise $f.lWeekday
    raise $f.lDay
    raise $f.lMonth

    pack $f.cDate -in $f -fill both -expand yes

    _appbar_get_date
}


proc _appbar_get_date {} {
    global dsk_appbar

    set f .dsk_appbar.fDate.f
    if ![winfo exists $f.lTime] {
	return
    }

    set date [exec date]
    set time [split [lindex $date 3] :]

    $f.lTime config -text "[lindex $time 0]:[lindex $time 1]"
    set wd [lindex $date 0]
    if {$wd == "Sun"} {
	set col red
    } else {
	set col black
    }
    $f.lWeekday config -text $wd -fg $col
    $f.lDay config -text [lindex $date 2]
    $f.lMonth config -text [lindex $date 1]

    if [info exists dsk_appbar(date,afterid)] {
	catch {after cancel $dsk_appbar(date,afterid)}
    }
    set dsk_appbar(date,afterid) [after 60000 _appbar_get_date]
}

