
/*
 * LIB/NEWSFEED.C
 *
 * (c)Copyright 1997, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution 
 *    for specific rights granted.
 *
 */

#include "defs.h"

Prototype int FeedAdd(const char *msgid, time_t t, History *h, const char *nglist, const char *npath);
Prototype int FeedWrite(int (*callback)(const char *hlabel, const char *msgid, const char *path, const char *offsize, int plfo), const char *msgid, const char *path, const char *offsize, const char *nglist, const char *npath);
Prototype void FeedFlush(void);
Prototype void LoadNewsFeed(time_t t, int force, const char *hlabel);
Prototype void TouchNewsFeed(void);
Prototype int FeedValid(const char *hlabel, int *pcount);
Prototype int IsFiltered(const char *hlabel, const char *nglist);
Prototype int FeedQuerySpam(const char *hlabel, const char *hostInfo);
Prototype int PathElmMatches(const char *hlabel, const char *p, int bytes, int *pidx);
Prototype int CommonElmMatches(const char *common, const char *p, int bytes);

Prototype FILE *FeedFo;

typedef struct GroupList {
    struct GroupList	*gl_Next;
    char		*gl_Label;
    struct Node		*gl_GroupAcceptBase;
    struct Node		*gl_SpamBase;
} GroupList;

typedef struct NewsFeed {
    struct NewsFeed	*nf_Next;
    char		*nf_Label;
    struct Node		*nf_PathAliasBase;
    struct Node		*nf_GroupAcceptBase;
    struct Node		*nf_FilterBase;
    struct Node		*nf_SpamBase;
    int			nf_MaxCrossPost;	/* 0=unlimited 		*/
    int			nf_MinCrossPost;	/* 0=unlimited 		*/
    int			nf_MaxArtSize;		/* 0=unlimited 		*/
    int			nf_MinArtSize;		/* 0=unlimited 		*/
    int			nf_MaxPathLen;		/* maximum outgoing path len */
    int			nf_MinPathLen;		/* minimum outgoing path len */
    int			nf_MaxConnect;		/* max connections	*/
    char		nf_PerLineFlushOpt;
    char		nf_NoMisMatch;
    char		nf_Reserved2;
    char		nf_Reserved3;
} NewsFeed;

#define RTF_ENABLED	0x01			/* enable realtime feed  */
#define RTF_NOBATCH	0x02			/* do not generate batch */

NewsFeed 	*NFBase;
GroupList	*GLBase;
NewsFeed 	*NFCache;
NewsFeed	*NFGlob;

char	NFBuf[16384];
int	NFIdx;
FILE	*FeedFo;
MemPool	*NFMemPool;
const char *SaveHLabel;

static int RecurCnt = 0;
static int RecurWarn = 0;


int feedQueryPaths(NewsFeed *feed, const char *npath, int size);
int feedQueryGroups(NewsFeed *feed, const char *nglist);
int feedGroupQuery(Node *list, const char *group, int accept);
int filterRequireGroups(Node *node, const char *nglist);
int filterQueryGroups(NewsFeed *feed, const char *nglist);
int filterGroupQuery(NewsFeed *feed, const char *group);
int feedPathQuery(NewsFeed *feed, const char *path);

void resolveGroupList(const char *label, Node *scan);
void AltFeed(NewsFeed *nf, FILE *fi);
int TooNear(time_t t);

/*
 * FEEDADD()  - given message-id, history file data, newsgroup list, pathlist,
 *		and article size, commit the article to outgoing feeds.
 *
 *		NOTE: none of the string objects is allowed to contain a tab,
 *		this is checked prior to the call.
 */

int 
FeedAdd(const char *msgid, time_t t, History *h, const char *nglist, const char *npath)
{
    int r = 0;

    LoadNewsFeed(t, 0, (char *)-1);

    if (FeedFo) {
	char path[256];

	ArticleFileName(path, h, 0);

	fprintf(FeedFo, "SOUT\t%s\t%ld,%ld\t%s\t%s\t%s\n",
	    path, (long)h->boffset, (long)h->bsize, msgid, nglist, npath
	);
	fflush(FeedFo);	/* simplify the parent reader on the pipe */
	if (ferror(FeedFo)) {
	    syslog(LOG_CRIT, "lost backchannel to master server");
	    r = -1;
	}
    }
    return(r);
}

int
FeedWrite(
    int (*callback)(const char *hlabel, const char *msgid, const char *path, const char *offsize, int plfo), 
    const char *msgid, 
    const char *path, 
    const char *offsize,
    const char *nglist,
    const char *npath
) {
    NewsFeed *nf;
    int r = 0;
    int bytes = 0;

    {
	char *p;
	if ((p = strchr(offsize, ',')) != NULL)
	    bytes = strtol(p + 1, NULL, 0);
    }

    for (nf = NFBase; nf; nf = nf->nf_Next) {
	if (feedQueryPaths(nf, npath, bytes) == 0 && 
	    feedQueryGroups(nf, nglist) == 0
	) {
	    r += callback(nf->nf_Label, msgid, path, offsize, nf->nf_PerLineFlushOpt);
	}
    }
    return(r);
}

#ifdef NOTDEF
	    {
		fprintf(FeedFo, "%s %s %s %ld,%ld %s\n",
		    ((nf->nf_PerLineFlushOpt) ? "REAL" : "FEED"),
		    nf->nf_Label,
		    path,
		    h->boffset,
		    h->bsize,
		    msgid
		);
	    }
	}
    }

    /*
     * dump final stats
     */

    if (FeedFo) {
	fprintf(FeedFo, "STATS c=%d s=%d m=%s\n",
	    cnt,
	    size,
	    msgid
	);
	fflush(FeedFo);	/* simplify the parent reader on the pipe */
	if (ferror(FeedFo)) {
	    syslog(LOG_CRIT, "lost backchannel to master server");
	    r = -1;
	}
    }
    return(r);
}
#endif

void
FeedFlush(void)
{
/*    NewsFeed *nf;*/

    /*
    if (FeedFo)
	fflush(FeedFo);
    */
}

int
FeedValid(const char *hlabel, int *pcount)
{
    NewsFeed *nf;

    if ((nf = NFCache) == NULL) {
	for (nf = NFBase; nf; nf = nf->nf_Next) {
	    if (strcmp(hlabel, nf->nf_Label) == 0)
		break;
	}
	NFCache = nf;
    }
    if (nf == NULL)
	return(FEED_MISSINGLABEL);
    if (nf->nf_MaxConnect && *pcount > nf->nf_MaxConnect) {
	*pcount = nf->nf_MaxConnect;
	return(FEED_MAXCONNECT);
    }
    return(0);
}

int 
IsFiltered(const char *hlabel, const char *nglist)
{
    NewsFeed *nf;
    int r = 0;

    if (hlabel[0] == 0)
	hlabel = "DEFAULT";

    if ((nf = NFCache) == NULL) {
	for (nf = NFBase; nf; nf = nf->nf_Next) {
	    if (strcmp(hlabel, nf->nf_Label) == 0)
		break;
	}
	NFCache = nf;
    }

    if (nf == NULL) {
	r = -1;
    } else {
	r = filterQueryGroups(nf, nglist);
    }
    if (DebugOpt > 1)
	printf("IsFiltered: %d (%s,%s)\n", r, hlabel, nglist);
    return(r);
}

/*
 * Return 0 if the feed does NOT have an alias for an element in the path
 *	AND the article is not too large, otherwise return -1.
 */

int
feedQueryPaths(NewsFeed *feed, const char *npath, int size)
{
    const char *p;
    int cnt = 0;

    if (feed->nf_MaxArtSize && size > feed->nf_MaxArtSize)
	return(-1);
    if (feed->nf_MinArtSize && size <= feed->nf_MinArtSize)
	return(-1);

    while (*npath == ' ' || *npath == '\t')
	++npath;
    for (p = npath; p; p = strchr(p, '!')) {
	char pat[MAXGNAME];
	int l;

	++cnt;

	if (*p == '!')
	    ++p;
	for (l = 0; l < MAXGNAME - 1 && p[l] && p[l] != '!' && p[l] != ' ' && p[l] != '\t' && p[l] != '\n' && p[l] != '\r'; ++l)
	    ;
	strncpy(pat, p, l);
	pat[l] = 0;
	if (feedPathQuery(feed, pat) == 0)
	    break;
    }
    if (p == NULL) {
	/*
	 * no path aliases matched, we are ok... unless MaxPathLen 
	 * or MinPathLen is set.
	 */
	if (feed->nf_MaxPathLen && feed->nf_MaxPathLen < cnt)
	    return(-1);
	if (feed->nf_MinPathLen && feed->nf_MinPathLen > cnt)
	    return(-1);
	return(0);
    }
    /*
     * path alias found, skip this feed
     */

    return(-1);
}

/*
 * Return 0 if the feed has a group match against a group in the group list
 * Return -1 if there is no match
 */

int
feedQueryGroups(NewsFeed *feed, const char *nglist)
{
    const char *p;
    int r = -1;
    int count = 0;

    while (*nglist == ' ' || *nglist == '\t')
	++nglist;

    for (p = nglist; p && r != -2; p = strchr(p, ',')) {
	int l;
	int nr;
	char group[MAXGNAME];

	if (*p == ',')
	    ++p;

	for (l = 0; l < MAXGNAME - 1 && p[l] && p[l] != ',' && p[l] != ' ' && p[l] != '\t' && p[l] != '\n' && p[l] != '\r'; ++l)
	    ;

	/*
	 * r:
	 *	0	feed based on group
	 *	-1	do not feed based on group
	 *	-2	do not feed based on group if group appears at all
	 */

	strncpy(group, p, l);
	group[l] = 0;
	nr = feedGroupQuery(feed->nf_GroupAcceptBase, group, -1);
	++count;
	if (nr != -1) {
	    r = nr;
	}
    }
    if (r >= 0 && feed->nf_MaxCrossPost && count > feed->nf_MaxCrossPost)
	r = -1;
    if (r >= 0 && feed->nf_MinCrossPost && count < feed->nf_MinCrossPost)
	r = -1;
    if (r >= 0 && filterRequireGroups(feed->nf_GroupAcceptBase, nglist) < 0)
	r = -2;
    return(r);
}

int
filterRequireGroups(Node *node, const char *nglist)
{
    int r = 0;

    while (*nglist == ' ' || *nglist == '\t')
	++nglist;

    for (; r == 0 && node; node = node->no_Next) {
	/*
	 * look for requiregroup
	 */
	if (node->no_Value == -3) {
	    const char *p;

	    /*
	     * requiregroup wildcard must match at least
	     * one group in the list.
	     */

	    r = -1;

	    for (p = nglist; p && r < 0; p = strchr(p, ',')) {
		int l;
		char group[MAXGNAME];

		if (*p == ',')
		    ++p;

		for (l = 0; l < MAXGNAME - 1 && p[l] && p[l] != ',' && p[l] != ' ' && p[l] != '\t' && p[l] != '\n' && p[l] != '\r'; ++l)
		    ;
		strncpy(group, p, l);
		group[l] = 0;
		if (WildCmp(node->no_Name, group) == 0) {
		    r = 0;
		    break;
		}
	    }
	}

	/*
	 * requiregroup in recursion
	 */

	if (node->no_Value == 2 && node->no_Data) {
	    /*
	     * GroupList recursion
	     */
	    ++RecurCnt;
	    if (RecurCnt < 10) {
		r = filterRequireGroups(((GroupList *)node->no_Data)->gl_GroupAcceptBase, nglist);
	    } else {
		if (RecurWarn == 0) {
		    syslog(LOG_EMERG, "DNewsfeeds file grouplist infinite recursion at %s!\n", ((GroupList *)node)->gl_Label);
		    RecurWarn = 1;
		}
	    }
	    --RecurCnt;
	}
    }
    return(r);
}

/*
 * FeedQuerySpam() - scan addspam/delspam filters on NNTP-Posting-Host:
 *
 *	0 - could be either, use normal spam filter
 *	-1 - definitely spam
 *      +1 - definitely not spam
 *
 * XXX hlabel, once found, may not be changed
 */

int
FeedQuerySpam(const char *hlabel, const char *hostInfo)
{
    Node *node;
    NewsFeed *feed;
    int accept = 0;

    if ((feed = NFCache) == NULL) {
	for (feed = NFBase; feed; feed = feed->nf_Next) {
	    if (strcmp(hlabel, feed->nf_Label) == 0)
		break;
	}
	NFCache = feed;
    }

    if (NFGlob != NULL) {
	for (node = NFGlob->nf_SpamBase; node; node = node->no_Next) {
	    if (WildCmp(node->no_Name, hostInfo) == 0) {
		if (node->no_Value < 0) {	/* delspam - not spam */
		    accept = 1;
		} else if (node->no_Value > 0) { /* addspam - spam  */
		    accept = -1;
		}
	    }
	}
    }
    if (feed) {
	for (node = feed->nf_SpamBase; node; node = node->no_Next) {
	    if (WildCmp(node->no_Name, hostInfo) == 0) {
		if (node->no_Value < 0) {
		    accept = 1;
		} else if (node->no_Value > 0) {	/* addspam - is spam*/
		    accept = -1;
		}
	    }
	}
    }
    return(accept);
}

/*
 * PathElmMatches() - match first path element against aliases.
 *		      Return -1 on failure, 0 on success.  Set
 *		      *pidx to the length of the first path element.
 */

int
PathElmMatches(const char *hlabel, const char *p, int bytes, int *pidx)
{
    NewsFeed *feed;
    int i;
    int r = 0;

    if ((feed = NFCache) == NULL) {
	for (feed = NFBase; feed; feed = feed->nf_Next) {
	    if (strcmp(hlabel, feed->nf_Label) == 0)
		break;
	}
	NFCache = feed;
    }
    for (i = 0; i < bytes && p[i] != '!'; ++i)
	;
    *pidx = i;

    if (feed && feed->nf_NoMisMatch == 0) {
	Node *node;
	char buf[256];

	if (i >= sizeof(buf))
	    i = sizeof(buf) - 1;
	bcopy(p, buf, i);
	buf[i] = 0;

	r = -1;
	for (node = feed->nf_PathAliasBase; node; node = node->no_Next) {
	    if (WildCaseCmp(node->no_Name, buf) == 0) {
		r = 0;
		break;
	    }
	}
    }
    return(r);
}

/*
 * Return 0 if the common path element exists in the given
 * path string.
 */

int 
CommonElmMatches(const char *common, const char *p, int bytes)
{
    int l = strlen(common);

    while (bytes >= l) {
	if (strncmp(common, p, l) == 0 &&
	    (bytes == l || p[l] == '!' || p[l] == '\n')
	) {
	    return(0);
	}
	while (bytes && *p != '!') {
	    --bytes;
	    ++p;
	}
	if (bytes && *p == '!') {
	    --bytes;
	    ++p;
	}
    }
    return(-1);
}

/*
 * Return 0 if a group in the group list has been
 * filtered from the particular incoming feed.
 */

int
filterQueryGroups(NewsFeed *feed, const char *nglist)
{
    const char *p;

    while (*nglist == ' ' || *nglist == '\t')
	++nglist;
    for (p = nglist; p; p = strchr(p, ',')) {
	int l;
	char group[MAXGNAME];

	if (*p == ',')
	    ++p;
	for (l = 0; l < MAXGNAME - 1 && p[l] && p[l] != ',' && p[l] != ' ' && p[l] != '\t' && p[l] != '\n' && p[l] != '\r'; ++l)
	    ;
	strncpy(group, p, l);
	group[l] = 0;
	if (filterGroupQuery(feed, group) == 0)
	    break;
    }
    if (p == NULL)
	return(-1);
    return(0);
}

/*
 * feedGroupQuery()
 *
 *	-2	do not feed ANY group if this group appears in the newsgroups
 *		line.
 *
 *	-1	do not feed this group
 *
 *	0	feed this group
 *
 */

int
feedGroupQuery(Node *node, const char *group, int accept)
{
    for (; node; node = node->no_Next) {
	if (node->no_Value == -3)
	    continue;
	if (node->no_Value == 2) {
	    if (node->no_Data) {
		/*
		 * GroupList recursion
		 */
		++RecurCnt;
		if (RecurCnt < 10) {
		    accept = feedGroupQuery(((GroupList *)node->no_Data)->gl_GroupAcceptBase, group, accept);
		} else {
		    if (RecurWarn == 0) {
			syslog(LOG_EMERG, "DNewsfeeds file grouplist infinite recursion at %s!\n", ((GroupList *)node)->gl_Label);
			RecurWarn = 1;
		    }
		}
		--RecurCnt;
	    }
	} else if (WildCmp(node->no_Name, group) == 0) {
	    if (node->no_Value < 0) {
		accept = node->no_Value;	/* -1 or -2 */
	    } else if (node->no_Value > 0) {
		accept = 0;			/* +1	    */
	    }
	}
    }
    return(accept);
}

int
filterGroupQuery(NewsFeed *feed, const char *group)
{
    Node *node;
    int accept = -1;

    if (NFGlob != NULL) {
	for (node = NFGlob->nf_FilterBase; node; node = node->no_Next) {
	    if (WildCmp(node->no_Name, group) == 0) {
		if (node->no_Value < 0) {
		    accept = -1;
		} else if (node->no_Value > 0) {
		    accept = 0;
		}
	    }
	}
    }
    if (feed) {
	for (node = feed->nf_FilterBase; node; node = node->no_Next) {
	    if (WildCmp(node->no_Name, group) == 0) {
		if (node->no_Value < 0) {
		    accept = -1;
		} else if (node->no_Value > 0) {
		    accept = 0;
		}
	    }
	}
    }
    return(accept);
}

int
feedPathQuery(NewsFeed *feed, const char *path)
{
    Node *node;
    int found = -1;

    if (NFGlob != NULL) {
	for (node = NFGlob->nf_PathAliasBase; node; node = node->no_Next) {
	    if (WildCaseCmp(node->no_Name, path) == 0) {
		found = 0;
		break;
	    }
	}
    }
    if (feed != NULL && found < 0) {
	for (node = feed->nf_PathAliasBase; node; node = node->no_Next) {
	    if (WildCaseCmp(node->no_Name, path) == 0) {
		found = 0;
		break;
	    }
	}
    }
    return(found);
}

time_t NFGmtMin = (time_t)-1;
time_t NFMTime = 0;

void
TouchNewsFeed(void)
{
    FILE *fi = xfopen("r+", "%s/dnewsfeeds", NewsHome);
    if (fi) {
	int c;
	if ((c = fgetc(fi)) != EOF) {
	    fseek(fi, 0L, 0);
	    fputc(c, fi);
	    fflush(fi);
	}
	fclose(fi);
    }
}

/*
 * LoadNewsFeed() - [re]load dnewsfeeds file
 */

void
LoadNewsFeed(time_t t, int force, const char *hlabel)
{
    int timeChanged = 0;

    /*
     * check for dnewsfeeds file modified once a minute
     */

    if (hlabel != (void *)-1)
	SaveHLabel = hlabel;

    if (force || t == 0 || t / 60 != NFGmtMin) {
	struct stat st = { 0 };
	FILE *fi;

	timeChanged = 1;

	if (t)
	    NFGmtMin = t / 60;

	errno = 0;

	fi = xfopen("r", "%s/dnewsfeeds", NewsHome);

	if (fi == NULL)
	    syslog(LOG_EMERG, "%s/dnewsfeeds: %s", NewsHome, strerror(errno));

	if (fi == NULL || 
	    (fstat(fileno(fi), &st) == 0 && st.st_mtime != NFMTime && !TooNear(st.st_mtime)) ||
	    force ||
	    NFMTime == 0
	) {
	    char buf[MAXGNAME+256];

	    NFMTime = st.st_mtime;	/* may be 0 if file failed to open */

	    /*
	     * flush existing feed information
	     */

	    if (DebugOpt) {
		printf("Reloading dnewsfeeds file hlabel=%s\n",
		    ((hlabel) ? hlabel : "?")
		);
	    }

	    FeedFlush();

	    /*
	     * free up feed structures
	     */

	    freePool(&NFMemPool);
	    NFCache = NULL;
	    NFGlob = NULL;
	    NFBase = NULL;
	    GLBase = NULL;

	    /*
	     * Reset MaxPerRemote if it wasn't specified on the command line.
	     *	0 = disabled
	     * -1 = disabled but there may be per-feed limits
	     * +N = some global limit
	     */

	    if (SaveHLabel == NULL && MaxPerRemote == -1)
		MaxPerRemote = 0;

	    /*
	     * load up new feed structures
	     */

	    {
		NewsFeed *nf = NULL;
		GroupList *gl = NULL;
		Node **paNode = NULL;
		Node **pgNode = NULL;
		Node **psNode = NULL;
		int lineNo = 0;
		FILE *altFi = NULL;

		while (fi && fgets(buf, sizeof(buf), fi) != NULL) {
		    char *s1 = strtok(buf, " \t\n");
		    char *s2 = (s1) ? strtok(NULL, " \t\n") : NULL;
		    int err = 1;

		    ++lineNo;

		    if (s1 == NULL || *s1 == '#')
			continue;

		    if (strcmp(s1, "label") == 0) {
			if (nf || gl) {
			    syslog(LOG_CRIT, 
				"Config line %d, no end before new label!\n",
				lineNo
			    );
			    if (paNode)
				*paNode = NULL;
			    if (pgNode)
				*pgNode = NULL;
			    if (psNode)
				*psNode = NULL;
			    if (nf) {
				nf->nf_Next = NFBase;
				NFBase = nf;
				if (altFi) {
				    AltFeed(nf, altFi);
				    fclose(altFi);
				    altFi = NULL;
				}
			    }
			    if (gl) {
				gl->gl_Next = GLBase;
				GLBase = gl;
			    }
			    nf = NULL;
			    gl = NULL;
			}

			/*
			 * If we are loading a particular label, it must exist.
			 */

			if (s2 && (SaveHLabel == NULL || strcmp(s2, SaveHLabel) == 0 ||
				strcmp(s2, "GLOBAL") == 0 || strcmp(s2, "DEFAULT") == 0
			)) {
			    char path[256];

			    nf = zalloc(&NFMemPool, sizeof(NewsFeed) + strlen(s2) + 1);
			    nf->nf_Label = (char *)(nf + 1);
			    strcpy(nf->nf_Label, s2);
			    paNode = &nf->nf_PathAliasBase;
			    pgNode = &nf->nf_GroupAcceptBase;
			    psNode = &nf->nf_SpamBase;

			    sprintf(path, "%s/feeds/%s", NewsHome, s2);
			    altFi = fcdopen(path, "r");
			    err = 0;
			    if (strcmp(nf->nf_Label, "GLOBAL") == 0)
				NFGlob = nf;
			}
		    } else if (strcmp(s1, "alias") == 0) {
			if (nf && s2) {
			    Node *node = MakeNode(&NFMemPool, s2, 0);
			    *paNode = node;
			    paNode = &node->no_Next;
			    err = 0;
			}
		    } else if (strcmp(s1, "rtflush") == 0) {
			if (nf) {
			    err = 0;
			    nf->nf_PerLineFlushOpt = 1;
			}
		    } else if (strcmp(s1, "nomismatch") == 0) {
			if (nf) {
			    nf->nf_NoMisMatch = 1;
			    err = 0;
			}
		    } else if (strcmp(s1, "filter") == 0) {
			if (nf && s2) {
			    Node *node = MakeNode(&NFMemPool, s2, 1);

			    node->no_Next = nf->nf_FilterBase;
			    nf->nf_FilterBase = node;
			    err = 0;
			}
		    } else if (strcmp(s1, "maxconnect") == 0) {
			if (nf && s2) {
			    nf->nf_MaxConnect = strtol(s2, NULL, 0);
			    err = 0;
			    if (nf->nf_MaxConnect && MaxPerRemote == 0)
				MaxPerRemote = -1;
			}
		    } else if (strcmp(s1, "mincross") == 0) {
			if (nf && s2) {
			    nf->nf_MinCrossPost = strtol(s2, NULL, 0);
			    err = 0;
			}
		    } else if (strcmp(s1, "maxcross") == 0) {
			if (nf && s2) {
			    nf->nf_MaxCrossPost = strtol(s2, NULL, 0);
			    err = 0;
			}
		    } else if (strcmp(s1, "minpath") == 0) {
			if (nf && s2) {
			    nf->nf_MinPathLen = strtol(s2, NULL, 0);
			    err = 0;
			}
		    } else if (strcmp(s1, "maxpath") == 0) {
			if (nf && s2) {
			    nf->nf_MaxPathLen = strtol(s2, NULL, 0);
			    err = 0;
			}
		    } else if (strcmp(s1, "minsize") == 0) {
			if (nf && s2) {
			    char *p;

			    nf->nf_MinArtSize = strtol(s2, &p, 0);

			    switch(*p) {
			    case 'g':
			    case 'G':
				nf->nf_MinArtSize *= 1024;
				/* fall through */
			    case 'm':
			    case 'M':
				nf->nf_MinArtSize *= 1024;
				/* fall through */
			    case 'k':
			    case 'K':
				nf->nf_MinArtSize *= 1024;
				break;
			    }
			    err = 0;
			}
		    } else if (strcmp(s1, "maxsize") == 0) {
			if (nf && s2) {
			    char *p;

			    nf->nf_MaxArtSize = strtol(s2, &p, 0);

			    switch(*p) {
			    case 'g':
			    case 'G':
				nf->nf_MaxArtSize *= 1024;
				/* fall through */
			    case 'm':
			    case 'M':
				nf->nf_MaxArtSize *= 1024;
				/* fall through */
			    case 'k':
			    case 'K':
				nf->nf_MaxArtSize *= 1024;
				break;
			    }
			    err = 0;
			}
		    } else if (strcmp(s1, "groupdef") == 0) {
			if (nf || gl) {
			    syslog(LOG_CRIT,
			       "Config line %d, no end before new groupdef!\n", 
				lineNo
			    );
			    if (paNode)
				*paNode = NULL;
			    if (pgNode)
				*pgNode = NULL;
			    if (psNode)
				*psNode = NULL;
			    if (nf) {
				nf->nf_Next = NFBase;
				NFBase = nf;
				if (altFi) {
				    AltFeed(nf, altFi);
				    fclose(altFi);
				    altFi = NULL;
				}
			    }
			    if (gl) {
				gl->gl_Next = GLBase;
				GLBase = gl;
			    }
			    nf = NULL;
			    gl = NULL;
			}
			if (s2) {
			    gl = zalloc(&NFMemPool, sizeof(GroupList) + strlen(s2) + 1);
			    gl->gl_Label = (char *)(gl + 1);
			    strcpy(gl->gl_Label, s2);
			    paNode = NULL;
			    pgNode = &gl->gl_GroupAcceptBase;
			    psNode = &gl->gl_SpamBase;
			    err = 0;
			}
		    } else if (strcmp(s1, "groupref") == 0) {
			if (pgNode && s2 && altFi == NULL) {
			    Node *node = MakeNode(&NFMemPool, s2, 2);
			    *pgNode = node;
			    pgNode = &node->no_Next;
			}
			if (pgNode && s2)
			    err = 0;
		    } else if (strcmp(s1, "delgroupany") == 0) {
			if (pgNode && s2 && altFi == NULL) {
			    Node *node = MakeNode(&NFMemPool, s2, -2);
			    *pgNode = node;
			    pgNode = &node->no_Next;
			}
			if (pgNode && s2)
			    err = 0;
		    } else if (strcmp(s1, "delgroup") == 0) {
			if (pgNode && s2 && altFi == NULL) {
			    Node *node = MakeNode(&NFMemPool, s2, -1);
			    *pgNode = node;
			    pgNode = &node->no_Next;
			}
			if (pgNode && s2)
			    err = 0;
		    } else if (strcmp(s1, "requiregroup") == 0) {
			if (pgNode && s2 && altFi == NULL) {
			    Node *node = MakeNode(&NFMemPool, s2, -3);
			    *pgNode = node;
			    pgNode = &node->no_Next;
			}
			if (pgNode && s2)
			    err = 0;
		    } else if (strcmp(s1, "addgroup") == 0) {
			if (pgNode && s2 && altFi == NULL) {
			    Node *node = MakeNode(&NFMemPool, s2, 1);
			    *pgNode = node;
			    pgNode = &node->no_Next;
			}
			if (pgNode && s2)
			    err = 0;
		    } else if (strcmp(s1, "delspam") == 0) {
			if (psNode && s2 && altFi == NULL) {
			    Node *node = MakeNode(&NFMemPool, s2, -1);
			    *psNode = node;
			    psNode = &node->no_Next;
			}
			if (psNode && s2)
			    err = 0;
		    } else if (strcmp(s1, "addspam") == 0) {
			if (psNode && s2 && altFi == NULL) {
			    Node *node = MakeNode(&NFMemPool, s2, 1);
			    *psNode = node;
			    psNode = &node->no_Next;
			}
			if (psNode && s2)
			    err = 0;
		    } else if (strcmp(s1, "end") == 0) {
			if (nf) {
			    *paNode = NULL;
			    *pgNode = NULL;
			    *psNode = NULL;
			    nf->nf_Next = NFBase;
			    NFBase = nf;
			    if (altFi) {
				AltFeed(nf, altFi);
				fclose(altFi);
				altFi = NULL;
			    }
			    nf = NULL;
			    err = 0;
			}
			if (gl) {
			    *pgNode = NULL;
			    *psNode = NULL;
			    gl->gl_Next = GLBase;
			    GLBase = gl;
			    gl = NULL;
			    err = 0;
			}
		    } else {
			syslog(LOG_CRIT, 
			    "Config line %d, unknown command\n", 
			    lineNo
			);
			err = 0;
		    }

		    /*
		     * deal with errors inside active labels (nf != NULL) or
		     * general errors when parsing the entire file (hlabel == NULL)
		     */

		    if (err && (nf != NULL || SaveHLabel == NULL)) {
			syslog(LOG_CRIT, "Config line %d, command in unexpected position or command requires argument\n", lineNo);
		    }
		}
		if (nf) {
		    *paNode = NULL;
		    *pgNode = NULL;
		    *psNode = NULL;
		    nf->nf_Next = NFBase;
		    NFBase = nf;
		    nf = NULL;
		}
		if (gl) {
		    *pgNode = NULL;
		    *psNode = NULL;
		    gl->gl_Next = GLBase;
		    GLBase = gl;
		    gl = NULL;
		}
		if (altFi) {
		    fclose(altFi);
		}
	    }
	}
	if (fi)
	    fclose(fi);

	/*
	 * Resolve GroupList references after the fact.  This allows
	 * grouplist definitions to put after newsfeed definitions.
	 */
	{
	    NewsFeed *nf;
	    GroupList *gl;

	    for (nf = NFBase; nf; nf = nf->nf_Next)
		resolveGroupList(nf->nf_Label, nf->nf_GroupAcceptBase);
	    for (gl = GLBase; gl; gl = gl->gl_Next)
		resolveGroupList(gl->gl_Label, gl->gl_GroupAcceptBase);
	}
    }
}

void
resolveGroupList(const char *label, Node *scan)
{
    for (; scan; scan = scan->no_Next) {
	if (scan->no_Value == 2) {
	    GroupList *gl;

	    for (gl = GLBase; gl; gl = gl->gl_Next) {
		if (strcmp(scan->no_Name, gl->gl_Label) == 0) {
		    scan->no_Data = gl;
		    break;
		}
	    }
	    if (gl == NULL) {
		syslog(LOG_CRIT, "Error: grouplist %s does not exist (from %s)\n", scan->no_Name, label);
	    }
	}
    }
}

void
AltFeed(NewsFeed *nf, FILE *fi)
{
    char buf[MAXGNAME+256];
    Node **pgNode = &nf->nf_GroupAcceptBase;
    Node **psNode = &nf->nf_SpamBase;

    while (*pgNode)
	pgNode = &(*pgNode)->no_Next;
    while (*psNode)
	psNode = &(*psNode)->no_Next;

    while (fgets(buf, sizeof(buf), fi) != NULL) {
	char *s1 = strtok(buf, " \t\n");
	char *s2 = strtok(NULL, " \t\n");

	if (s1 == NULL)
	    continue;
	if (s2 == NULL)
	    continue;
	if (strcmp(s1, "addgroup") == 0) {
	    Node *node = MakeNode(&NFMemPool, s2, 1);
	    *pgNode = node;
	    pgNode = &node->no_Next;
	} else if (strcmp(s1, "delgroup") == 0) {
	    Node *node = MakeNode(&NFMemPool, s2, -1);
	    *pgNode = node;
	    pgNode = &node->no_Next;
	} else if (strcmp(s1, "requiregroup") == 0) {
	    Node *node = MakeNode(&NFMemPool, s2, -3);
	    *pgNode = node;
	    pgNode = &node->no_Next;
	} else if (strcmp(s1, "addspam") == 0) {
	    Node *node = MakeNode(&NFMemPool, s2, 1);
	    *psNode = node;
	    psNode = &node->no_Next;
	} else if (strcmp(s1, "delspam") == 0) {
	    Node *node = MakeNode(&NFMemPool, s2, -1);
	    *psNode = node;
	    psNode = &node->no_Next;
	} else if (strcmp(s1, "delgroupany") == 0) {
	    Node *node = MakeNode(&NFMemPool, s2, -2);
	    *pgNode = node;
	    pgNode = &node->no_Next;
	}
    }
    *pgNode = NULL;
    *psNode = NULL;
}

int
TooNear(time_t t)
{
    time_t now = time(NULL);
    int32 dt = (int32)(now - t);

    if (dt > -10 && dt < 10)
	return(1);
    return(0);
}

