
/*
 * UTIL/DEXPIREOVER.C
 *
 * dexpireover [-a] [-w wildcard] [-f dactive.kp] [-NB]
 *
 * (1)
 * Adjust the beginning group in the active file XXX
 *
 * (2)
 * Scan /news/spool/group for over. and data. files, match against the active
 * file, and delete stale data.* files (including those related to deleted
 * newsgroups).
 */

#include "defs.h"

typedef struct Group {
    struct Group *gr_Next;
    int		gr_State;
    int		gr_StartNo;
    int		gr_EndNo;
    char	*gr_GroupName;
    char	*gr_Flags;
    hash_t	gr_Hv;
    int		gr_ScanArtBeg;
    int		gr_ScanArtEnd;
} Group;

#define GRF_DESCRIPTION 0x0001
#define GRF_STARTNO     0x0002
#define GRF_ENDNO       0x0004
#define GRF_FLAGS       0x0008
#define GRF_FROMLOCAL   0x0800
#define GRF_NEW         0x1000
#define GRF_FROMREMOTE  0x2000
#define GRF_MODIFIED    0x8000

#define GHSIZE		1024
#define GHMASK		(GHSIZE-1)

KPDB  *KDBActive;
Group *GHash[GHSIZE];

void ProcessOverviewFile(const char *dirPath, const char *name);
char *allocTmpCopy(const char *buf, int bufLen);
Group *EnterGroup(const char *groupName, int begNo, int endNo, const char *flags);
Group *FindGroupByHash(hash_t *hv);
int SetField(char **pptr, const char *str);

int UpdateBegArtNoOpt = 0;
int BadGroups = 0;
int GoodGroups = 0;

int
main(int ac, char **av)
{
    int i;
    char *dbfile = NULL;
    char *wild   = NULL;

    LoadDiabloConfig();

    for (i = 1; i < ac; ++i) {
	char *ptr = av[i];
	if (*ptr != '-') {
	    fprintf(stderr, "Unexpected argument: %s\n", ptr);
	    exit(1);
	}
	ptr += 2;
	switch(ptr[-1]) {
	case 'a':
	    UpdateBegArtNoOpt = 1;
	    break;
	case 'w':
	    wild = (*ptr) ? ptr : av[++i];
	    break;
	case 'f':
	    dbfile = (*ptr) ? ptr : av[++i];
	    break;
	case 'N':
	    while (*ptr) {
		switch(*ptr) {
		case 'B':
		    UpdateBegArtNoOpt = 1;
		    break;
		default:
		    break;
		}
		++ptr;
	    }
	    break;
	default:
	    fprintf(stderr, "Unknown option: %s\n", ptr - 2);
	    exit(1);
	}
    }

    /*
     * Open active file database
     */

    if (dbfile) {
	KDBActive = XKPDBOpen(O_RDWR, "%s", dbfile);
    } else {
	KDBActive = XKPDBOpen(O_RDWR, "%s/dactive.kp", NewsHome);
    }
    if (KDBActive == NULL) {
	printf("Unable to open dactive.kp\n");
	exit(1);
    }

    /*
     * scan dactive.kp
     */

    {
	int recLen;
	const char *rec;

	for (rec = KPDBScanFirst(KDBActive, 0, &recLen);
	     rec;
	     rec = KPDBScanNext(KDBActive, rec, 0, &recLen)
	) {
	    int groupLen;
	    int flagsLen;
	    const char *group = KPDBGetField(rec, recLen, NULL, &groupLen, NULL);
	    const char *flags = KPDBGetField(rec, recLen, "S", &flagsLen, "y");
	    int begNo = strtol(KPDBGetField(rec, recLen, "NB", NULL, "-1"), NULL, 10);
	    int endNo = strtol(KPDBGetField(rec, recLen, "NE", NULL, "-1"), NULL, 10);
	    Group *grp;

	    if (group)
		group = allocTmpCopy(group, groupLen);
	    if (flags)
		flags = allocTmpCopy(flags, flagsLen);

	    /*
	     * ignore bad group or group that does not match the wildcard
	     */

	    if (group == NULL)
		continue;
	    if (wild && WildCmp(wild, group) != 0)
		continue;

	    grp = EnterGroup(
		group,
		begNo,
		endNo,
		flags
	    );
	}
    }

    /*
     * scan /news/spool/group/
     */

    {
	DIR *dir;

	if ((dir = opendir(OverviewHome)) != NULL) {
	    den_t *den;
	    char *tmp = malloc(strlen(OverviewHome) + 32);

	    while ((den = readdir(dir)) != NULL) {
		if (strlen(den->d_name) == 2 && 
		    isalnum(den->d_name[0]) &&
		    isalnum(den->d_name[1])
		) {
		    DIR *dir2;

		    sprintf(tmp, "%s/%s", OverviewHome, den->d_name);
		    if ((dir2 = opendir(tmp)) != NULL) {
			den_t *den2;
			while ((den2 = readdir(dir2)) != NULL) {
			    ProcessOverviewFile(tmp, den2->d_name);
			}
			closedir(dir2);
		    }
		}
	    }
	    closedir(dir);
	    free(tmp);
	}
    }

    printf("Scanned %d groups %d were bad\n",
	GoodGroups + BadGroups, BadGroups
    );

	exit(99);
#ifdef NOTDEF

    /*
     * Writeback
     */

    {
	int i;

	for (i = 0; i < GHSIZE; ++i) {
	    Group *group;

	    for (group = GHash[i]; group; group = group->gr_Next) {
		/*
		 * If we have a new group not previously in the database,
		 * we only add it if SyncGroupsOpt is set.
		 */
		if (group->gr_State & GRF_NEW) {
		    if (SyncGroupsOpt == 0)
			continue;
		}

		/*
		 * If group did not come from remote and SyncDeleteOpt is set,
		 * delete the group.
		 */
		if ((group->gr_State & GRF_FROMREMOTE) &&
		    !(group->gr_State & GRF_FROMLOCAL) &&
		    SyncDeleteOpt
		) {
		    printf("Deleting %s\n", group->gr_GroupName);
		    KPDBDelete(KDBActive, group->gr_GroupName);
		} else if (group->gr_State & GRF_MODIFIED) {
		    if (group->gr_State & GRF_NEW) {
			printf("Creating %s %s\n", group->gr_GroupName, (group->gr_Flags ? group->gr_Flags : "y"));
		    } else {
			printf("Updating %s %s\n", group->gr_GroupName, (group->gr_Flags ? group->gr_Flags : "y"));
		    }
		    if (group->gr_State & GRF_DESCRIPTION)
			KPDBWriteEncode(KDBActive, group->gr_GroupName, "GD", group->gr_Description, 0);
		    if (group->gr_State & GRF_STARTNO) {
			char startBuf[16];
			sprintf(startBuf, "%010d", group->gr_StartNo);
			KPDBWriteEncode(KDBActive, group->gr_GroupName, "NB", startBuf, 0);
		    }
		    if (group->gr_State & GRF_ENDNO) {
			char endBuf[16];
			sprintf(endBuf, "%010d", group->gr_EndNo);
			KPDBWriteEncode(KDBActive, group->gr_GroupName, "NE", endBuf, 0);
		    }
		    if (group->gr_State & GRF_FLAGS)
			KPDBWriteEncode(KDBActive, group->gr_GroupName, "S", group->gr_Flags, 0);
		}
	    }
	}
    }
#endif

    return(0);
}

void
ProcessOverviewFile(const char *dirPath, const char *name)
{
    long h1;
    long h2;
    long artBase;
    int type = 0;
    Group *group;
    hash_t hv;
    char path[1024];

    snprintf(path, sizeof(path), "%s/%s", dirPath, name);

    if (strncmp(name, "over.", 5) == 0)
	type = 1;
    if (strncmp(name, "data.", 5) == 0)
	type = 2;

    if (type == 0)
	return;

    if (sscanf(name + 5, "%ld.%lx.%lx", &artBase, &h1, &h2) != 3)
	return;
    hv.h1 = h1;
    hv.h2 = h2;

    if ((group = FindGroupByHash(&hv)) == NULL) {
	++BadGroups;
	printf("Group not found, removing %s\n", path);
	remove(path);
	return;
    }
    ++GoodGroups;

    if (type == 1) {
	/*
	 * over. file	(fixed length file)
	 */
	/* XXX */
	
    } else {
	/*
	 * data. file, modulo OD_HARTS.  OD_HARTS constant in second
	 * part of conditional is a fudge to make 100% sure we do not
	 * delete a brand new data file.
	 */
	if (artBase + OD_HARTS <= group->gr_StartNo ||
	    artBase >= group->gr_EndNo + OD_HARTS
	) {
	    printf("Deleting stale overview data %s: %s\n",
		group->gr_GroupName,
		path
	    );
	    remove(path);
	}
    }
}

char *
allocTmpCopy(const char *buf, int bufLen)
{
    static char *SaveAry[8];
    static int SaveCnt;
    char **pptr;

    SaveCnt = (SaveCnt + 1) % arysize(SaveAry);
    pptr = &SaveAry[SaveCnt];
    if (*pptr)
	free(*pptr);
    *pptr = malloc(bufLen + 1);
    memcpy(*pptr, buf, bufLen);
    (*pptr)[bufLen] = 0;
    return(*pptr);
}

Group *
EnterGroup(const char *groupName, int begNo, int endNo, const char *flags)
{
    hash_t hv = hhash(groupName);
    Group **pgroup = &GHash[hv.h1 & GHMASK];
    Group *group;

    while ((group = *pgroup) != NULL) {
	if (strcmp(groupName, group->gr_GroupName) == 0)
	    break;
	pgroup = &group->gr_Next;
    }
    if (group == NULL) {
	*pgroup = group = calloc(sizeof(Group) + strlen(groupName) + 1, 1);
	group->gr_State = GRF_NEW;
	group->gr_GroupName = (char *)(group + 1);
	group->gr_Hv = hv;
	strcpy(group->gr_GroupName, groupName);
    }

    /*
     * update fields
     */
    if (begNo >= 0) {
	group->gr_State |= GRF_STARTNO;
	if (group->gr_StartNo != begNo) {
	    group->gr_State |= GRF_MODIFIED;
	    group->gr_StartNo = begNo;
	}
    }
    if (endNo >= 0) {
	group->gr_State |= GRF_ENDNO;
	if (group->gr_EndNo != endNo) {
	    group->gr_EndNo = endNo;
	    group->gr_State |= GRF_MODIFIED;
	}
    }

    if (flags) {
	group->gr_State |= GRF_FLAGS;
	if (SetField(&group->gr_Flags, flags))
	    group->gr_State |= GRF_MODIFIED;
    }
    return(group);
}

Group *
FindGroupByHash(hash_t *hv)
{
    Group *group;

    for (group = GHash[hv->h1 & GHMASK]; group; group = group->gr_Next) {
	if (group->gr_Hv.h1 == hv->h1 &&
	    group->gr_Hv.h2 == hv->h2
	) {
	    break;
	}
    }
    return(group);
}

int
SetField(char **pptr, const char *str)
{
    if (*pptr && strcmp(*pptr, str) == 0)
	return(0);
    if (*pptr)
	free(*pptr);
    *pptr = strcpy(malloc(strlen(str) + 1), str);
    return(1);
}

