/*********************************************************************
 *
 *	Copyright (C) 1999-2002 Nathan Fiedler
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * PROJECT:     Abstract Data Types
 * MODULE:      Disjoint Set Test
 * FILE:        TestDisjointSet.java
 *
 * AUTHOR:      Nathan Fiedler
 *
 * REVISION HISTORY:
 *      Name    Date            Description
 *      ----    ----            -----------
 *      nf      01/09/99        Initial version
 *      nf      01/05/02        Greatly improved
 *
 * DESCRIPTION:
 *      Implements test-wrapper for the DisjointSet class.
 *
 * $Id: TestDisjointSet.java,v 1.4 2002/01/06 01:12:21 nfiedler Exp $
 *
 ********************************************************************/

package com.bluemarsh.adt.test;

import com.bluemarsh.adt.DisjointSet;
import java.util.*;

/**
 * Test-wrapper for the disjoint set class.
 *
 * @author  Nathan Fiedler
 */
public class TestDisjointSet {

    /**
     * Test wrapper for DisjointSet class. This is used to test the
     * class to make sure it works properly.
     *
     * @param  args  array of string arguments
     */
    public static void main(String[] args) {
        System.out.println();
        System.out.println("  + = good");
        System.out.println("  - = bad");
        System.out.println("  ! = information");
        System.out.println("  ? = verify manually");
        System.out.println();

        DisjointSet djs = new DisjointSet(1000);
        if (djs.isEmpty() && djs.size() == 0 &&
            djs.numberOfElements() == 0) {
            System.out.println("+ set is empty");
        } else {
            System.out.println("- set is not empty, what's up with that?");
            return;
        }

        // Populate the disjoint set.
        System.out.println("! populating set...");
        for (int ii = 0; ii < 1000; ii++) {
            djs.set(ii, new Integer(ii));
        }

        if (!djs.isEmpty()) {
            System.out.println("+ set is not empty");
        } else {
            System.out.println("- set is empty, what's up with that?");
            return;
        }

        System.out.println("? set hash = " + djs.hashCode());
        if (djs.equals(djs)) {
            System.out.println("+ set equals itself");
        } else {
            System.out.println("- set does not equal itself, eh?");
            return;
        }

        if (djs.size() == 1000 && djs.numberOfElements() == 1000) {
            System.out.println("+ set has 1000 elements");
        } else {
            System.out.println("- set count is not 1000");
            return;
        }

        if (djs.numberOfTrees() == 1000) {
            System.out.println("+ set has 1000 trees");
        } else {
            System.out.println("- set tree count is not 1000");
            return;
        }

        boolean success = true;
        System.out.println("! searching for all 1000 elements...");
        for (int ii = 0; ii < 1000; ii++) {
            Integer igr = (Integer) djs.get(ii);
            if (igr == null || igr.intValue() != ii) {
                System.out.println("- could not find element " + ii);
                success = false;
                break;
            }
        }
        if (success) {
            System.out.println("+ found all 1000 elements");
        } else {
            return;
        }

        System.out.println("! joining the set into two trees (0-500, 501-1000)...");
        for (int ii = 1; ii < 500; ii++) {
            djs.union(0, ii);
        }
        for (int ii = 501; ii < 1000; ii++) {
            djs.union(500, ii);
        }
        if (djs.size() == 1000 && djs.numberOfElements() == 1000) {
            System.out.println("+ set has 1000 elements");
        } else {
            System.out.println("- set count is not 1000");
            return;
        }
        if (djs.numberOfTrees() == 2) {
            System.out.println("+ set has 2 trees");
        } else {
            System.out.println("- set tree count is not 2");
            return;
        }
        System.out.println("! verifying roots of all elements...");
        for (int ii = 1; ii < 500; ii++) {
            if (djs.find(ii) != 0) {
                System.out.println("- element " + ii + " root is not 0");
                success = false;
                break;
            }
        }
        if (!success) {
            System.out.println("- find() failed");
            return;
        }
        for (int ii = 501; ii < 1000; ii++) {
            if (djs.find(ii) != 500) {
                System.out.println("- element " + ii + " root is not 500");
                success = false;
                break;
            }
        }
        if (!success) {
            System.out.println("- find() failed");
            return;
        } else {
            System.out.println("+ find() seems to work");
        }

        System.out.println("! testing contains()...");
        for (int ii = 0; ii < 1000; ii++) {
            if (!djs.contains(new Integer(ii))) {
                System.out.println("- contains() failed for element " + ii);
                success = false;
            }
        }
        if (success) {
            System.out.println("+ contains() seems to work");
        } else {
            return;
        }

        System.out.println("! testing containsAll()...");
        List contents = new ArrayList(100);
        for (int ii = 0; ii < 1000; ii += 10) {
            contents.add(new Integer(ii));
        }
        if (djs.containsAll(contents)) {
            System.out.println("+ containsAll() seems to work");
        } else {
            System.out.println("- containsAll() failed");
            return;
        }

        System.out.println("! testing indexOf(Object)...");
        for (int ii = 0; ii < 1000; ii++) {
            if (djs.indexOf(new Integer(ii)) != ii) {
                System.out.println("- indexOf(Object) failed for element " + ii);
                success = false;
            }
        }
        if (success) {
            System.out.println("+ indexOf(Object) seems to work");
        } else {
            return;
        }

        System.out.println("! testing indexOf(Object, int)...");
        for (int ii = 500; ii < 1000; ii++) {
            if (djs.indexOf(new Integer(ii), 100) != ii) {
                System.out.println("- indexOf(Object, int) failed for element " + ii);
                success = false;
            }
        }
        if (success) {
            System.out.println("+ indexOf(Object, int) seems to work");
        } else {
            return;
        }

        System.out.println("! testing lastIndexOf()...");
        for (int ii = 0; ii < 1000; ii++) {
            if (djs.lastIndexOf(new Integer(ii)) != ii) {
                System.out.println("- lastIndexOf() failed for element " + ii);
                success = false;
            }
        }
        if (success) {
            System.out.println("+ lastIndexOf() seems to work");
        } else {
            return;
        }

        List sub = djs.subList(300, 800);
        if (sub.isEmpty() || sub.size() != 500) {
            System.out.println("- subList size is not 500");
        }
        for (int ii = sub.size() - 1, jj = 799; ii >= 0; ii--, jj--) {
            Integer igr = (Integer) sub.get(ii);
            if (igr == null || igr.intValue() != jj) {
                System.out.println("- subList element incorrect " + jj +
                                   " at " + ii);
                success = false;
                break;
            }
        }
        if (success) {
            System.out.println("+ subList() seems to work");
        } else {
            System.out.println("- subList() failed");
            return;
        }

        System.out.println("! testing toArray()...");
        Object[] arr = djs.toArray();
        for (int ii = 0; ii < arr.length; ii++) {
            if (((Integer) arr[ii]).intValue() != ii) {
                System.out.println("- array element incorrect " + arr[ii] +
                                   " at " + ii);
                success = false;
                break;
            }
        }
        if (success) {
            System.out.println("+ toArray() seems to work");
        } else {
            System.out.println("- toArray() failed");
            return;
        }

        System.out.println("! testing toArray(Object[])...");
        Integer[] irr = (Integer[]) djs.toArray(new Integer[1000]);
        for (int ii = 0; ii < irr.length; ii++) {
            if (irr[ii].intValue() != ii) {
                System.out.println("- irray element incorrect " + irr[ii] +
                                   " at " + ii);
                success = false;
                break;
            }
        }
        if (success) {
            System.out.println("+ toArray(Object[]) seems to work");
        } else {
            System.out.println("- toArray(Object[]) failed");
            return;
        }

        System.out.println("! testing clone()");
        DisjointSet clone = (DisjointSet) djs.clone();
        if (!clone.isEmpty()) {
            System.out.println("+ clone is not empty");
        } else {
            System.out.println("- clone is empty, what's up with that?");
            return;
        }

        System.out.println("? clone hash = " + clone.hashCode());
        if (clone.equals(djs)) {
            System.out.println("- clone equals original, eh?");
            return;
        } else {
            System.out.println("+ clone does not equal original");
        }

        if (clone.size() == 1000 && clone.numberOfElements() == 1000) {
            System.out.println("+ clone has 1000 elements");
        } else {
            System.out.println("- clone count is not 1000");
            return;
        }

        if (clone.numberOfTrees() == 2) {
            System.out.println("+ clone has 2 trees");
        } else {
            System.out.println("- clone tree count is not 2");
            return;
        }

        djs = new DisjointSet(5);
        for (int ii = 0; ii < 5; ii++) {
            djs.set(ii, new Integer(ii));
        }
        System.out.println("? toString() of 5 elements: " + djs);

        djs = new DisjointSet(20);
        for (int ii = 0; ii < 20; ii++) {
            djs.set(ii, new Integer(ii));
        }
        Iterator iter = djs.iterator();
        int jj = 0;
        while (iter.hasNext()) {
            Integer igr = (Integer) iter.next();
            if (igr.intValue() != jj) {
                System.out.println("- iterator element incorrect at " + jj);
                success = false;
                break;
            }
            jj++;
        }
        if (success) {
            System.out.println("+ iterator() seems to work");
        } else {
            return;
        }

        ListIterator liter = djs.listIterator();
        jj = 0;
        while (liter.hasNext()) {
            if (liter.nextIndex() != jj) {
                System.out.println("- listIterator nextIndex incorrect at " + jj);
                success = false;
                break;
            }
            Integer igr = (Integer) liter.next();
            if (igr.intValue() != jj) {
                System.out.println("- listIterator next element incorrect at " + jj);
                success = false;
                break;
            }
            jj++;
        }
        if (success) {
            System.out.println("+ listIterator() next seems to work");
        } else {
            return;
        }

        while (liter.hasPrevious()) {
            jj--;
            if (liter.previousIndex() != jj) {
                System.out.println("- listIterator previousIndex incorrect at " + jj);
                success = false;
                break;
            }
            Integer igr = (Integer) liter.previous();
            if (igr.intValue() != jj) {
                System.out.println("- listIterator previous element incorrect at " + jj);
                success = false;
                break;
            }
        }
        if (success) {
            System.out.println("+ listIterator() previous seems to work");
        } else {
            return;
        }

        liter = djs.listIterator(10);
        if (liter.nextIndex() == 10) {
            System.out.println("+ listIterator(int) seems to work");
        } else {
            System.out.println("- listIterator(int) failed");
            return;
        }
        liter.set(new Integer(12345));
        if (((Integer) djs.get(10)).intValue() == 12345) {
            System.out.println("+ ListIterator.set() seems to work");
        } else {
            System.out.println("+ ListIterator.set() failed");
            return;
        }

        //
        //----------  CLEARING TESTS ----------
        //

        System.out.println("! clearing the set (of 20 elements)");
        djs.clear();
        if (djs.isEmpty() && djs.size() == 0 &&
            djs.numberOfElements() == 0) {
            System.out.println("+ set is empty");
        } else {
            System.out.println("- set is not empty, what's up with that?");
            return;
        }
        if (djs.numberOfTrees() == 20) {
            System.out.println("+ set has 20 trees");
        } else {
            System.out.println("- set tree count is not 20");
            return;
        }

        System.out.println();
        System.out.println("+ Tests completed successfully.");
    } // main
} // TestDisjointSet
