/*********************************************************************
 *
 *	Copyright (C) 1999-2001 Nathan Fiedler
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * PROJECT:     Abstract Data Types
 * MODULE:      Pancake Test
 * FILE:        TestPancakeStack.java
 *
 * AUTHOR:      Nathan Fiedler
 *
 * REVISION HISTORY:
 *      Name    Date            Description
 *      ----    ----            -----------
 *      nf      01/09/99        Initial version
 *      nf      01/21/01        Added clone test
 *
 * DESCRIPTION:
 *      Implements test-wrapper for the PancakeStack class.
 *
 * $Id: TestPancakeStack.java,v 1.1 2001/01/22 06:18:40 nfiedler Exp $
 *
 ********************************************************************/

package com.bluemarsh.adt.test;

import com.bluemarsh.adt.PancakeStack;
import java.util.Iterator;
import java.util.ConcurrentModificationException;

/**
 * Test-wrapper for the PancakeStack class.
 *
 * @author  Nathan Fiedler
 * @version 1.1  1/21/01
 */
public class TestPancakeStack {

    /**
     * Test the iterator method.
     */
    protected static void testIterator() {
        System.out.println("Testing iterator method...");
	PancakeStack stack = new PancakeStack();
	stack.push(new Integer(12));
	stack.add(new Integer(34));
	stack.push(new Integer(56));
	stack.add(new Integer(78));
	stack.push(new Integer(90));
	stack.add(new Integer(100));
        stack.push(null);

        Iterator iter = stack.iterator();
        while (iter.hasNext()) {
            System.out.print(iter.next() + " ");
        }
        System.out.println();

        System.out.println("Testing concurrent modification detection...");
        iter = stack.iterator();
        while (iter.hasNext()) {
            try {
                iter.next();
                stack.push(new Integer(5));
            } catch (ConcurrentModificationException cme) {
                System.out.println("Concurrent modification detection works");
                break;
            }
        }

	PancakeStack copy = new PancakeStack();
	copy.push(new Integer(12));
	copy.add(new Integer(34));
	copy.push(new Integer(56));
        System.out.println("Testing containsAll()...");
        if (!stack.containsAll(copy)) {
            System.out.println("ERROR: containsAll() failed");
        }

        System.out.println("Testing retainAll()...");
        if (!stack.retainAll(copy)) {
            System.out.println("ERROR: retainAll() failed");
        }
        System.out.println("The following should be 56, 34, 12");
        iter = stack.iterator();
        while (iter.hasNext()) {
            System.out.print(iter.next() + " ");
        }
        System.out.println();
    } // testIterator

    /**
     * Test remove and removeAll methods.
     */
    public static void testRemove() {
        System.out.println("Testing remove and removeAll...");
        PancakeStack stack = new PancakeStack();
	stack.push(new Integer(12));
	stack.add(new Integer(34));
	stack.push(new Integer(56));
        stack.push(null);
	stack.add(new Integer(78));
	stack.push(new Integer(90));
	stack.add(new Integer(100));
        if (!stack.remove(new Integer(56))) {
	    System.out.println("ERROR: remove returned false");
        }

        PancakeStack other = new PancakeStack();
	other.add(new Integer(78));
	other.push(new Integer(90));
        other.push(new Integer(12));
        if (!stack.removeAll(other)) {
	    System.out.println("ERROR: removeAll returned false");
        }
        System.out.println("The following should be 100, null, 34");
        Iterator iter = stack.iterator();
        while (iter.hasNext()) {
            System.out.print(iter.next() + " ");
        }
        System.out.println();
    } // testRemove

    /**
     * Test-wrapper for the Pancake stack class.
     *
     * @param  args  array of string arguments
     */
    public static void main(String args[]) {
	PancakeStack stack = new PancakeStack();

	// Test emptiness of stack - should be empty.
	if (!stack.isEmpty()) {
	    System.out.println("ERROR: Stack is not empty");
	}

	// Push some objects onto the stack.
        System.out.println("Adding seven items to stack...");
	stack.push(new Integer(12));
	stack.add(new Integer(34));
	stack.push(new Integer(56));
	stack.add(new Integer(78));
	stack.push(new Integer(90));
	stack.add(new Integer(100));
        stack.push(null);

	// Test emptiness of stack - should not be empty.
	if (stack.isEmpty()) {
	    System.out.println("ERROR: Stack is empty");
	}

        System.out.println("Clearing stack...");
        stack.clear();
	if (!stack.isEmpty()) {
	    System.out.println("ERROR: Stack is not empty");
	}

	// Push some objects onto the stack.
        System.out.println("Adding seven items to stack...");
	stack.push(new Integer(12));
	stack.add(new Integer(34));
	stack.push(new Integer(56));
        stack.push(null);
	stack.add(new Integer(78));
	stack.push(new Integer(90));
	stack.add(new Integer(100));

	// Test the search feature.
	if (stack.indexOf(new Integer(56)) == -1) {
	    System.out.println("ERROR: Did not find item 56");
	}
	if (!stack.contains(new Integer(56))) {
	    System.out.println("ERROR: Did not find item 56");
	}
	if (stack.indexOf(null) == -1) {
	    System.out.println("ERROR: Did not find null item");
	}

	// Test the peek method.
	System.out.println("Top element: " + stack.peek());

	// Test the size feature.
	if (stack.size() != 7) {
	    System.out.println("ERROR: Size is " + stack.size() +
                               " instead of 7");
	}

        PancakeStack copy = null;
        try {
            System.out.println("Cloning stack...");
            copy = (PancakeStack) stack.clone();
        } catch (CloneNotSupportedException cnse) {
            cnse.printStackTrace();
        }
        if (copy.equals(stack)) {
            System.out.println("ERROR: stack equals clone");
        }
        if (copy.hashCode() == stack.hashCode()) {
            System.out.println("ERROR: stack hash code equals clone's");
        }
        if (!stack.equals(stack)) {
            System.out.println("ERROR: stack does not equal itself");
        }

	// Test the flip facility.
        System.out.println("Testing flip() with -1, 0, 2, and 20");
	stack.flip(-1);
	stack.flip(0);
	stack.flip(2);
	stack.flip(20);

	// Pop off all the elements in order.
        System.out.println("Stack elements after flip:");
	while (!stack.isEmpty()) {
	    System.out.print(stack.pop() + " ");
	}
	System.out.println();

	// Test emptiness of stack - should be empty.
	if (!stack.isEmpty()) {
	    System.out.println("ERROR: Stack is not empty");
	}

        System.out.println("Adding cloned elements to original stack...");
        if (!stack.addAll(copy)) {
	    System.out.println("ERROR: allAll returned false");
        }
	if (stack.isEmpty()) {
	    System.out.println("ERROR: Stack is empty");
	}
        System.out.println("Stack elements after addAll (will be backwards):");
	while (!stack.isEmpty()) {
	    System.out.print(stack.pop() + " ");
	}
	System.out.println();

        System.out.println("Testing toArray()...");
        Object[] array = copy.toArray();
        if ((array == null) || (array.length != 7)) {
            System.out.println("ERROR: toArray() failed");
        }
        for (int i = 0; i < array.length; i++) {
            System.out.print(array[i] + " ");
        }
	System.out.println();

        // Pop off the clone's elements.
        System.out.println("Cloned elements from before the flip:");
	while (!copy.isEmpty()) {
	    System.out.print(copy.pop() + " ");
	}
	System.out.println();

        testRemove();

        testIterator();
    } // main
} // TestPancakeStack
