/*********************************************************************
 *
 *      Copyright (C) 2001-2002 Nathan Fiedler
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * PROJECT:     JSwat
 * MODULE:      JSwat
 * FILE:        PathManager.java
 *
 * AUTHOR:      Nathan Fiedler
 *
 * REVISION HISTORY:
 *      Name    Date            Description
 *      ----    ----            -----------
 *      nf      08/06/01        Initial version
 *      nf      10/21/01        Fixed bug 259
 *      nf      11/25/01        Fixed bug 321
 *      nf      01/03/02        Fixed bug 375
 *      nf      02/20/02        Fixed bug 52
 *      nf      02/21/02        Fixed bug 306
 *      MvD     03/10/02        Use Session classpath and sourcepath
 *      nf      03/18/02        Fixed bug 411
 *
 * DESCRIPTION:
 *      Defines the class responsible for managing classpath and
 *      sourcepath.
 *
 * $Id: PathManager.java,v 1.17 2002/04/08 23:56:20 nfiedler Exp $
 *
 ********************************************************************/

package com.bluemarsh.jswat;

import com.bluemarsh.config.JConfigure;
import com.sun.jdi.AbsentInformationException;
import com.sun.jdi.PathSearchingVirtualMachine;
import com.sun.jdi.ReferenceType;
import com.sun.jdi.VirtualMachine;
import java.io.*;
import java.util.*;
import java.util.zip.*;

/**
 * Class PathManager is responsible for managing the classpath and
 * sourcepath. It uses a given classpath and sourcepath, along with
 * class names and package names to find source files for requested
 * classes. If both a classpath and sourcepath are provided, the path
 * manager will search both to find a source file, searching the
 * sourcepath first.
 *
 * @author  Nathan Fiedler
 * @author  Marko van Dooren
 */
public class PathManager extends DefaultManager {
    /** Session that owns us. */
    protected Session owningSession;
    /** Table of SourceSource objects, keyed by the classname. */
    protected Hashtable classnameSources;

    /**
     * Constructs a PathManager object.
     */
    public PathManager() {
        classnameSources = new Hashtable();
    } // PathManager

    /**
     * Called when the Session is about to begin an active debugging
     * session. That is, JSwat is about to debug a debuggee VM.
     *
     * @param  session  Session being activated.
     */
    public void activate(Session session) {
        // Force the classpath to be loaded from the debuggee VM
        // since that is more accurate than anything else.
        VMConnection vmConnection = owningSession.getConnection();
        if (vmConnection != null) {
            VirtualMachine vm = vmConnection.getVM();
            if (vm instanceof PathSearchingVirtualMachine) {
                // Use the classpath from the VM, since it supports it.
                PathSearchingVirtualMachine psvm =
                    (PathSearchingVirtualMachine) vm;
                List paths = psvm.classPath();
                if (paths != null) {
                    setClassPath(paths);
                }
            }
        }
    } // activate

    /**
     * Turn the package name into a file path using simple character
     * subsitution.
     *
     * @param  classname  fully-qualified name of the class, possibly
     *                    including an inner-class specification.
     * @return  path and filename of source file.
     */
    protected String classnameToFilename(String classname) {
        int dollar = classname.indexOf('$');
        if (dollar > 0) {
            // Drop the inner-class specifier.
            classname = classname.substring(0, dollar);
        }
        classname = classname.replace('.', File.separatorChar);
        JConfigure config = JSwat.instanceOf().getJConfigure();
        classname += config.getProperty("files.defaultExtension");
        return classname;
    } // classnameToFilename

    /**
     * Returns the array of classpath directories, if any.
     *
     * @return  Array of Strings containing classpath directories;
     *          may be empty.
     */
    public String[] getClassPath() {
        StringTokenizer st = new StringTokenizer(getClassPathAsString(),
                                                 File.pathSeparator);
        int size = st.countTokens();
        String[] result = new String[size];
        for (int ii = 0; ii < size; ii++) {
            File f = new File(st.nextToken());
            try {
                // Use canonical path for better matching.
                result[ii] = f.getCanonicalPath();
            } catch (IOException ioe) {
                ioe.printStackTrace();
            }
        }
        return result;
    } // getClassPath

    /**
     * Returns the classpath as a String.
     *
     * @return  String of classpath, or empty string if not set.
     */
    public String getClassPathAsString() {
        String classpath = owningSession.getProperty("classpath");
        if (classpath == null) {
            // No session property, then get the classpath from
            // the VM running JSwat.
            classpath = System.getProperty("java.class.path");
            if (classpath == null) {
                // Default to current working directory.
                classpath = System.getProperty("user.dir");
                if (classpath == null) {
                    // Well, return empty string then.
                    classpath = "";
                }
            }
        }
        return classpath;
    } // getClassPathAsString

    /**
     * Returns the array of sourcepath directories, if any.
     *
     * @return  Array of Strings containing sourcepath directories;
     *          may be empty.
     */
    public String[] getSourcePath() {
        StringTokenizer st = new StringTokenizer(getSourcePathAsString(),
                                                 File.pathSeparator);
        int size = st.countTokens();
        String[] result = new String[size];
        for (int ii = 0; ii < size; ii++) {
            File f = new File(st.nextToken());
            try {
                // Use canonical path for better matching.
                result[ii] = f.getCanonicalPath();
            } catch (IOException ioe) {
                ioe.printStackTrace();
            }
        }
        return result;
    } // getSourcePath

    /**
     * Returns the sourcepath as a String.
     *
     * @return  String of sourcepath, or empty string if not set.
     */
    public String getSourcePathAsString() {
        // It is generally assumed that if java.source.path is set,
        // the user explicitly set it to override the session value.
        String sourcepath = System.getProperty("java.source.path");
        if ((sourcepath == null) || (sourcepath.length() == 0)) {
            sourcepath = owningSession.getProperty("sourcepath");
            if (sourcepath == null) {
                sourcepath = "";
            }
        }
        return sourcepath;
    } // getSourcePathAsString

    /**
     * Called after the Session has instantiated this mananger.
     * To avoid problems with circular dependencies between managers,
     * iniitialize data members before calling
     * <code>Session.getManager()</code>.
     *
     * @param  session  Session initializing this manager.
     */
    public void init(Session session) {
        owningSession = session;
    } // init

//      /**
//       * Return the fully-qualified classname corresponding to the
//       * given source file. Returns null if file has an error.
//       *
//       * @param  source   source file containing class.
//       * @return  Classname, or null if error.
//       */
//      public String mapClassName(File source) {
//          String fpath;
//          try {
//              fpath = source.getCanonicalPath();
//          } catch (IOException ioe) {
//              ioe.printStackTrace();
//              return null;
//          }
//          // Don't need the filename extension.
//          int idx = fpath.lastIndexOf(".java");
//          if (idx > 0) {
//              fpath = fpath.substring(0, idx);
//          }

//          try {
//              // Use canonical path for better matching.
//              String canon = new File(fpath).getCanonicalPath();
//              fpath = null;
//              // Scan through the source path list, if available.
//              if (sourcepathArray != null) {
//                  fpath = mapClassNameLow(sourcepathArray, canon);
//              }
//              if (fpath == null) {
//                  // Scan through the classpath list, if available.
//                  if (classpathArray != null) {
//                      fpath = mapClassNameLow(classpathArray, canon);
//                  }
//              }
//          } catch (IOException ioe) {
//              ioe.printStackTrace();
//              return null;
//          }
//          return fpath;
//      } // mapClassName

//      /**
//       * Looks through the given path list, comparing each entry with
//       * the given fpath. For the best matching entry, this method
//       * returns the fpath without that matching prefix.
//       *
//       * @param  pathlist  List of paths.
//       * @param  fpath     File path to look for.
//       * @return  Name of class without the extra prefix, or null if no match.
//       *          (using File.fileSeparatorChar to separate elements)
//       * @exception  IOException
//       *             Thrown if File.getCanonicalPath() has a problem.
//       */
//      protected String mapClassNameLow(String[] pathlist, String fpath)
//          throws IOException {
//          int longest = -1;
//          for (int i = 0; i < pathlist.length; i++) {
//              if (fpath.startsWith(pathlist[i])) {
//                  int len = pathlist[i].length();
//                  if (pathlist[i].charAt(len - 1) != File.separatorChar) {
//                      // Path list entry does not end with a separator so
//                      // we must add one to remove it from fpath.
//                      len++;
//                  }
//                  if (len > longest) {
//                      // Save the length for the possible best match.
//                      // This may change as we go through the list.
//                      longest = len;
//                  }
//              }
//          }

//          if (longest > -1) {
//              // Convert the file separators to dots.
//              fpath = fpath.replace(File.separatorChar, '.');
//              fpath = fpath.substring(longest);
//              return fpath;
//          } else {
//              return null;
//          }
//      } // mapClassNameLow

    /**
     * Return a SourceSource corresponding to the fully-qualified class
     * name. Return null if the source was not found.
     *
     * @param  classname  fully-qualified class name.
     * @return  source containing the desired location.
     * @exception  IOException
     *             Thrown if an I/O error occurred.
     */
    public SourceSource mapSource(String classname) throws IOException {
        // Check in the classname/File look-up table to see if
        // we've already mapped this classname to a file.
        SourceSource source = (SourceSource) classnameSources.get(classname);
        if (source != null) {
            return source;
        }

        // See if we can use the better method.
        // This may be too slow.
//          VirtualMachine vm = owningSession.getVM();
//          if (vm != null) {
//              try {
//                  ReferenceType clazz = ClassUtils.getReferenceFromName(
//                      classname, vm);
//                  if (clazz != null) {
//                      // Use the better method.
//                      return mapSource(clazz);
//                  }
//              } catch (AmbiguousClassSpecException acse) {
//                  // this is unlikely
//              }
//          }

        // Use the primitive means of finding the source.
        String filename = classnameToFilename(classname);
        return mapSourceLow(filename, classname);
    } // mapSource

    /**
     * Return a SourceSource corresponding to the given class.
     * Return null if the source was not found.
     *
     * @param  clazz    class for which to find source file.
     * @return  source containing the desired location.
     * @exception  IOException
     *             Thrown if an I/O error occurred.
     */
    public SourceSource mapSource(ReferenceType clazz) throws IOException {
        // Check in the classname/SourceSource look-up table to see if
        // we have already mapped this classname to a source.
        String classname = clazz.name();
        SourceSource source = (SourceSource) classnameSources.get(classname);
        if (source != null) {
            return source;
        }

        String filename = classnameToFilename(classname);

        // Try to use the source filename as given by the class.
        try {
            // Get the source name first so the exception gets thrown
            // now rather than after we modify the filename variable.
            String srcname = clazz.sourceName();
            int bsi = srcname.lastIndexOf('\\');
            int fsi = srcname.lastIndexOf('/');
            // Work-around for bug 4404985 where SourceFile has path.
            if (bsi > -1) {
                srcname = srcname.substring(bsi + 1);
            } else if (fsi > -1) {
                srcname = srcname.substring(fsi + 1);
            }
            int lastbit = filename.lastIndexOf(File.separatorChar);
            if (lastbit > -1) {
                filename = filename.substring(0, lastbit);
                filename = filename + File.separator + srcname;
            } else {
                // Class without a path, just use the source name.
                filename = srcname;
            }
        } catch (AbsentInformationException aie) {
            // If this happens, this method ends up being the same
            // as the version that takes a String argument.
        }

        return mapSourceLow(filename, classname);
    } // mapSource

    /**
     * Looks for a matching entry in either the class path or source path.
     *
     * @param  filename   name of file to look for.
     * @param  classname  name of class for caching result.
     * @return  matching source, if found.
     */
    protected SourceSource mapSourceLow(String filename, String classname) {
        String[] sourcepathArray = getSourcePath();
        // Scan through the source path list, if available.
        if (sourcepathArray != null) {
            for (int i = 0; i < sourcepathArray.length; i++) {
                SourceSource src = mapSource0(sourcepathArray[i], filename,
                                              classname);
                if (src != null) {
                    return src;
                }
            }
        }

        String[] classpathArray = getClassPath();
        // Scan through the class path list, if available.
        if (classpathArray != null) {
            for (int i = 0; i < classpathArray.length; i++) {
                SourceSource src = mapSource0(classpathArray[i], filename,
                                              classname);
                if (src != null) {
                    return src;
                }
            }
        }

        // Did not find a matching source file.
        return null;
    } // mapSourceLow

    /**
     * Look for the file in the given class or source path entry.
     * This method deals with zip and jar archives, as well as the
     * usual directory paths.
     *
     * @param  path       source or class path entry.
     * @param  filename   name of file to look for.
     * @param  classname  name of class for caching result.
     */
    protected SourceSource mapSource0(String path,
                                      String filename,
                                      String classname) {

        if (path.endsWith(".zip") || path.endsWith(".jar")) {
            // Zip/jar path entry.
            ZipFile zipFile = null;
            try {
                zipFile = new ZipFile(path);
            } catch (IOException ioe) {
                return null;
            }
            Enumeration enum = zipFile.entries();
            while (enum.hasMoreElements()) {
                ZipEntry zipEntry = (ZipEntry) enum.nextElement();
                String entryName = zipEntry.getName();
                // Convert the name to the local file system form so we
                // can compare it to the filename argument.
                entryName = new File(entryName).getPath();
                if (entryName.equals(filename)) {
                    SourceSource src = new ZipSource(zipFile, zipEntry);
                    // We found it, cache it and return the source.
                    classnameSources.put(classname, src);
                    return src;
                }
            }

        } else {
            // Directory path entry.
            File file = new File(path, filename);
            if (file.exists()) {
                FileSource src = new FileSource(file);
                // We found it, cache it and return the source.
                classnameSources.put(classname, src);
                return src;
            }
        }

        return null;
    } // mapSource0

    /**
     * Sets the classpath this source manager uses. The previous
     * class path is discarded in favor of the new one. The class
     * path is used together with the source path, if any.
     *
     * @param  classpath  Classpath for VM.
     */
    public void setClassPath(String classpath) {
        owningSession.setProperty("classpath", classpath);
    } // setClassPath

    /**
     * Sets the classpath this source manager uses. The previous
     * class path is discarded in favor of the new one. The class
     * path is used together with the source path, if any.
     *
     * @param  list  List of String classpath entries.
     */
    public void setClassPath(List list) {
        int size = list.size();
        if (size == 0) {
            // Special case of empty list.
            setClassPath("");
        } else {
            // Turn the list into a String so we can save it.
            StringBuffer sb = new StringBuffer(80);
            sb.append((String) list.get(0));
            for (int i = 1; i < size; i++) {
                sb.append(File.pathSeparator);
                sb.append((String) list.get(i));
            }
            // Call the other method.
            setClassPath(sb.toString());
        }
    } // setClassPath

    /**
     * Sets the sourcepath this source manager uses. The previous
     * source path is discarded in favor of the new one. The source
     * path is used together with the class path, if any.
     *
     * @param  sourcepath  Sourcepath for VM.
     */
    public void setSourcePath(String sourcepath) {
        owningSession.setProperty("sourcepath", sourcepath);
    } // setSourcePath
} // PathManager
