/*********************************************************************
 *
 *      Copyright (C) 1999-2001 Nathan Fiedler
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * PROJECT:     JSwat
 * MODULE:      JSwat Events
 * FILE:        ContextChangeEvent.java
 *
 * AUTHOR:      Nathan Fiedler
 *
 * REVISION HISTORY:
 *      Name    Date            Description
 *      ----    ----            -----------
 *      nf      10/16/99        Initial version
 *      nf      03/18/01        Changed to support multiple types.
 *
 * DESCRIPTION:
 *      Defines the ContextChangeEvent class for context change
 *      notification.
 *
 * $Id: ContextChangeEvent.java,v 1.5 2001/05/14 01:51:55 nfiedler Exp $
 *
 ********************************************************************/

package com.bluemarsh.jswat.event;

import java.util.EventObject;

/**
 * An event which indicates that the debugger context has changed.
 * This includes the current thread, current stack frame, and
 * current stepping location.
 *
 * @author  Nathan Fiedler
 * @version 1.1  3/18/01
 */
public class ContextChangeEvent extends EventObject {
    /** The current thread changed event type. */
    public static final int TYPE_THREAD = 0x01;
    /** The current frame changed event type. */
    public static final int TYPE_FRAME = 0x02;
    /** The current location changed event type. */
    public static final int TYPE_LOCATION = 0x04;
    /** The type flags of this context change.
     * @serial */
    protected int types;

    /**
     * Constructs a new ContextChangeEvent.
     *
     * @param  source  Source of this event.
     * @param  types   A set of type flags.
     */
    public ContextChangeEvent(Object source, int types) {
        super(source);
        this.types = types;
    } // ContextChangeEvent

    /**
     * Compares the type of this event to the given argument and
     * returns true if they match. This event may match more than
     * one type of event. For instance, if a thread change occurs,
     * the frame and location will also change at the same time.
     *
     * @param  type  One of <code>THREAD</code>,
     *                      <code>FRAME</code>, or
     *                      <code>LOCATION</code>.
     * @return  true if this event is of the given type
     */
    public boolean isType(int type) {
        return (this.types & type) > 0;
    } // isType

    /**
     * Returns a String representation of this ContextChangeEvent.
     *
     * @return  A String representation of this ContextChangeEvent.
     */
    public String toString() {
        StringBuffer buf = new StringBuffer("ContextChange=[source=");
        buf.append(getSource());
        buf.append(", types=");
        if ((types & TYPE_THREAD) > 0) {
            buf.append("thread,");
        }
        if ((types & TYPE_FRAME) > 0) {
            buf.append("frame,");
        }
        if ((types & TYPE_LOCATION) > 0) {
            buf.append("location");
        }
        buf.append(']');
        return buf.toString();
    } // toString
} // ContextChangeEvent
