#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include "../../src/acm/astro.h"
#include "../../src/dis/dis/earth.h"
#include "../../src/util/units.h"

// Max err on Sun lat and lon (DEG):
#define LAT_ERR_MAX 0.1
#define LON_ERR_MAX 0.2

static int err;

/**
 * Evaluates Sun ephemeris.
 * @param line Caller line number for error reporting.
 * @param date Ephemeris date (ISO 8601 format).
 * @param exp_latlon Expected latitude and longitude (usual ACM formats).
 */
static void test_getSun(int line, char *date, char *exp_latlon)
{
	// Parse expected values:
	zulu_Date date_zulu;
	if( ! zulu_dateParse(date, &date_zulu) ){
		fprintf(stderr, "%d: invalid date: %s\n", line, date);
		err++;
		return;
	}
	earth_LatLonAlt exp_sun_w;
	if( ! earth_parseLatLon(exp_latlon, &exp_sun_w) ){
		fprintf(stderr, "%d: invalid latitude or longitude: %s\n", line, exp_latlon);
		err++;
		return;
	}
	
	// Init astro module:
	astro_init(0, &date_zulu);
	VPoint got_sun_xyz;
	
	// Calculate Sun ephemeris:
	astro_getSun(0, &got_sun_xyz);
	earth_LatLonAlt got_sun_w;
	
	// FIXME: earth_XYZToLatLonAlt() returns NaN for large dists!
	// Workaround with a bit of precision loss:
/*
	double scale = 3*earth_MAJOR / VMagnitude(&got_sun_xyz);
	VPoint scaled = got_sun_xyz;
	scaled.x *= scale;
	scaled.y *= scale;
	scaled.z *= scale;
	earth_XYZToLatLonAlt(&scaled, &got_sun_w);
*/
	/*
	 * Workaround: for very distant points, bare atan() work well enough:
	 */
	got_sun_w.latitude = earth_normalizeLatitude( atan(got_sun_xyz.z
		/ sqrt(got_sun_xyz.x * got_sun_xyz.x + got_sun_xyz.y * got_sun_xyz.y)) );
	got_sun_w.longitude = earth_normalizeLongitude(
		atan2(got_sun_xyz.y, got_sun_xyz.x) );
	got_sun_w.z = 0;
	
	// Compare vs. expected Sun ephemeris:
	double lat_err_deg = units_RADtoDEG(got_sun_w.latitude - exp_sun_w.latitude);
	double lon_err_deg = units_RADtoDEG(got_sun_w.longitude - exp_sun_w.longitude);
	if( lon_err_deg > 180 )
		lon_err_deg = 360 - lon_err_deg;
	if( lon_err_deg < -180 )
		lon_err_deg = 360 + lon_err_deg;
	if( fabs(lat_err_deg) > LAT_ERR_MAX || fabs(lon_err_deg) > LON_ERR_MAX ){
		err++;
		char date_zulu_s[20];
		zulu_dateFormat(&date_zulu, date_zulu_s, sizeof(date_zulu_s));
		fprintf(stderr, "%d: Sun ephemeris for %s:\n", line, date_zulu_s);
		char s[99], t[99];
		fprintf(stderr, "\texp %s %s\n",
			earth_latitudeToString(s, sizeof(s), exp_sun_w.latitude, earth_LLM_DMS),
			earth_longitudeToString(t, sizeof(t), exp_sun_w.longitude, earth_LLM_DMS));
		
		//fprintf(stderr, "\texp %g %g\n", exp_sun_w.latitude, exp_sun_w.longitude);
		
		fprintf(stderr, "\tgot %s %s  (err %.02g DEG, %.02g DEG)\n",
			earth_latitudeToString(s, sizeof(s), got_sun_w.latitude, earth_LLM_DMS),
			earth_longitudeToString(t, sizeof(t), got_sun_w.longitude, earth_LLM_DMS),
			lat_err_deg, lon_err_deg);
		//fprintf(stderr, "\t    cartesian: %g, %g, %g\n",
		//	got_sun_xyz.x, got_sun_xyz.y, got_sun_xyz.z);
	}
}


int main(int argc, char** argv)
{
	// Source: ALMANACCO NAUTICO - ITALIAN EPHEMERIDES AND NAUTICAL YEARBOOK
	// 2010 - Ulisse Quadri, Bassano Bresciano Astronomical Observatory
	test_getSun(__LINE__, "2010-01-01T00:00", "23-01-36S 179-10-30W");
	test_getSun(__LINE__, "2010-01-01T06:00", "23-00-24S 090-51-18E");
	test_getSun(__LINE__, "2010-01-01T12:00", "22-59-12S 000-53-01E");
	test_getSun(__LINE__, "2010-01-01T18:00", "22-57-54S 089-05-12W");
	test_getSun(__LINE__, "2010-04-01T00:00", "04-24-54N 178-59-30W");
	test_getSun(__LINE__, "2010-04-01T06:00", "04-30-42N 090-59-24E");
	test_getSun(__LINE__, "2010-04-01T12:00", "04-36-30N 000-58-18E");
	test_getSun(__LINE__, "2010-04-01T18:00", "04-42-18N 089-02-48W");
	test_getSun(__LINE__, "2010-07-01T00:00", "23-07-42N 179-03-48W");
	test_getSun(__LINE__, "2010-07-01T06:00", "23-06-42N 090-56-54E");
	test_getSun(__LINE__, "2010-07-01T12:00", "23-05-42N 000-57-36E");
	test_getSun(__LINE__, "2010-07-01T18:00", "23-04-36N 089-01-42W");
	test_getSun(__LINE__, "2010-10-01T00:00", "03-03-42S 177-27-42E");
	test_getSun(__LINE__, "2010-10-01T06:00", "03-09-30S 087-26-30E");
	test_getSun(__LINE__, "2010-10-01T12:00", "03-15-18S 002-34-42W");
	test_getSun(__LINE__, "2010-10-01T18:00", "03-21-06S 092-35-54W");
	test_getSun(__LINE__, "2010-12-31T00:00", "23-07-12S 179-19-30W");
	test_getSun(__LINE__, "2010-12-31T06:00", "23-06-06S 090-42-18E");
	test_getSun(__LINE__, "2010-12-31T12:00", "23-05-00S 000-44-06E");
	test_getSun(__LINE__, "2010-12-31T18:00", "23-03-54S 089-14-06W");
	
	
	// Source: https://www.thenauticalalmanac.com/ from here on:
	test_getSun(__LINE__, "2017-01-01T00:00", "22-59-54S 179-08-24W");
	test_getSun(__LINE__, "2017-01-01T06:00", "22-58-42S 090-53-24E");
	test_getSun(__LINE__, "2017-01-01T12:00", "22-57-24S 000-55-12E");
	test_getSun(__LINE__, "2017-02-01T00:00", "17-06-12S 176-37-00W");
	test_getSun(__LINE__, "2017-02-01T06:00", "17-01-54S 093-23-30E");
	test_getSun(__LINE__, "2017-02-01T12:00", "16-57-36S 003-24-00E");
	test_getSun(__LINE__, "2017-03-01T00:00", "07-35-42S 176-54-18W");
	test_getSun(__LINE__, "2017-03-01T06:00", "07-30-00S 093-05-00E");
	test_getSun(__LINE__, "2017-03-20T10:00", "00-00-30S 031-51-18E");
	test_getSun(__LINE__, "2017-03-21T12:00", "00-25-12N 001-46-30E");
	test_getSun(__LINE__, "2017-04-01T00:00", "04-31-48N 179-00-42W");
	test_getSun(__LINE__, "2017-04-01T06:00", "04-37-36N 090-58-12E");
	test_getSun(__LINE__, "2017-04-01T12:00", "04-43-24N 000-52-36E");
	test_getSun(__LINE__, "2017-05-01T00:00", "15-04-00N 179-16-54E");
	test_getSun(__LINE__, "2017-05-01T06:00", "15-08-36N 089-10-24E");
	test_getSun(__LINE__, "2017-05-01T12:00", "15-13-06N 000-44-00W");
	test_getSun(__LINE__, "2017-06-01T00:00", "22-02-48N 179-26-48E");
	test_getSun(__LINE__, "2017-06-01T06:00", "22-04-48N 089-27-24E");
	test_getSun(__LINE__, "2017-06-21T04:24", "23-26-06N 114-26-08E");
	test_getSun(__LINE__, "2017-06-21T12:00", "23-26-00N 000-27-18E");
	test_getSun(__LINE__, "2017-07-01T00:00", "23-06-12N 179-02-48W");
	test_getSun(__LINE__, "2017-07-01T06:00", "23-05-12N 090-57-54E");
	test_getSun(__LINE__, "2017-07-01T12:00", "23-04-12N 000-58-36E");
	test_getSun(__LINE__, "2017-08-01T00:00", "18-01-00N 178-24-42W");
	test_getSun(__LINE__, "2017-08-01T06:00", "17-57-12N 091-35-06E");
	test_getSun(__LINE__, "2017-08-01T12:00", "17-53-24N 001-34-48E");
	test_getSun(__LINE__, "2017-09-01T00:00", "08-17-06N 179-58-36W");
	test_getSun(__LINE__, "2017-09-01T06:00", "08-11-42N 090-00-12E");
	test_getSun(__LINE__, "2017-09-01T12:00", "08-06-12N 000-01-00W");
	test_getSun(__LINE__, "2017-09-21T12:00", "00-31-12N 001-45-06W");
	test_getSun(__LINE__, "2017-09-22T20:00", "00-00-00N 121-52-06W");
	test_getSun(__LINE__, "2017-10-01T00:00", "03-10-36S 177-26-24E");
	test_getSun(__LINE__, "2017-10-01T06:00", "03-16-24S 087-25-12E");
	test_getSun(__LINE__, "2017-10-01T12:00", "03-22-12S 002-36-00W");
	
	// Whole 2017-10-23 day:
	test_getSun(__LINE__, "2017-10-23T00:00", "11-23-24S 176-05-42E");
	test_getSun(__LINE__, "2017-10-23T01:00", "11-24-18S 161-05-36E");
	test_getSun(__LINE__, "2017-10-23T02:00", "11-25-12S 146-05-30E");
	test_getSun(__LINE__, "2017-10-23T03:00", "11-26-00S 131-05-24E");
	test_getSun(__LINE__, "2017-10-23T04:00", "11-26-54S 116-05-36E");
	test_getSun(__LINE__, "2017-10-23T05:00", "11-27-48S 101-05-18E");
	test_getSun(__LINE__, "2017-10-23T06:00", "11-28-42S 086-05-12E");
	test_getSun(__LINE__, "2017-10-23T07:00", "11-29-30S 071-05-06E");
	test_getSun(__LINE__, "2017-10-23T08:00", "11-30-24S 056-05-00E");
	test_getSun(__LINE__, "2017-10-23T09:00", "11-31-18S 041-04-54E");
	test_getSun(__LINE__, "2017-10-23T10:00", "11-32-12S 026-04-48E");
	test_getSun(__LINE__, "2017-10-23T11:00", "11-33-00S 011-04-48E");
	test_getSun(__LINE__, "2017-10-23T12:00", "11-33-54S 003-55-18W");
	test_getSun(__LINE__, "2017-10-23T13:00", "11-34-48S 018-55-24W");
	test_getSun(__LINE__, "2017-10-23T14:00", "11-35-42S 033-55-30W");
	test_getSun(__LINE__, "2017-10-23T15:00", "11-36-30S 048-55-36W");
	test_getSun(__LINE__, "2017-10-23T16:00", "11-37-24S 063-55-42W");
	test_getSun(__LINE__, "2017-10-23T17:00", "11-38-18S 078-55-42W");
	test_getSun(__LINE__, "2017-10-23T18:00", "11-39-06S 093-55-48W");
	test_getSun(__LINE__, "2017-10-23T19:00", "11-40-00S 108-55-54W");
	test_getSun(__LINE__, "2017-10-23T20:00", "11-40-54S 123-56-00W");
	test_getSun(__LINE__, "2017-10-23T21:00", "11-41-48S 138-56-06W");
	test_getSun(__LINE__, "2017-10-23T22:00", "11-42-36S 153-56-12W");
	test_getSun(__LINE__, "2017-10-23T23:00", "11-43-30S 168-56-12W");
	
	test_getSun(__LINE__, "2017-11-01T00:00", "14-25-00S 175-54-06E");
	test_getSun(__LINE__, "2017-11-01T06:00", "14-29-48S 085-54-00E");
	test_getSun(__LINE__, "2017-11-01T12:00", "14-34-36S 004-06-06W");
	test_getSun(__LINE__, "2017-12-01T00:00", "21-47-30S 177-13-54E");
	test_getSun(__LINE__, "2017-12-01T06:00", "21-49-48S 087-15-18E");
	test_getSun(__LINE__, "2017-12-01T12:00", "21-52-06S 002-43-18W");
	test_getSun(__LINE__, "2017-12-21T12:00", "23-26-06S 000-27-12W");
	test_getSun(__LINE__, "2017-12-21T16:00", "23-26-06S 060-26-00W");

	return err==0? 0 : 1;
}

