# Code structure information.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

module AdLint #:nodoc:

  # == DESCRIPTION
  # Base class of code structure information.
  class CodeStructure
    # === DESCRIPTION
    # Converts this code structure information into array representation.
    #
    # Subclasses must implement this method.
    #
    # === RETURN VALUE
    # Array< Object > -- Array representation.
    def to_a
      subclass_responsibility
    end

    # === DESCRIPTION
    # Converts this code structure information into string representation.
    #
    # === RETURN VALUE
    # String -- String representation.
    def to_s
      delim = ",".to_default_external
      to_a.map { |obj| obj.to_s.to_default_external }.join(delim)
    end

    def to_csv
      to_a.map { |obj| obj ? obj.to_s.to_default_external : nil }.to_csv
    end
  end

  # == DESCRIPTION
  # Type declaration information.
  class TypeDcl < CodeStructure
    # === DESCRIPTION
    # Constructs the type declaration information.
    #
    # === PARAMETER
    # _location_:: Location -- Location where the declaration appears.
    # _typedcl_type_:: String -- Type string of the type declaration.
    # _type_name_:: String -- Type name.
    # _type_rep_:: String -- Type representation.
    def initialize(location, typedcl_type, type_name, type_rep)
      @location = location
      @typedcl_type = typedcl_type
      @type_name = type_name
      @type_rep = type_rep
    end

    # === DESCRIPTION
    # Converts this code structure information into array representation.
    #
    # === RETURN VALUE
    # Array< Object > -- Array representation.
    def to_a
      ["DCL", @location.fpath, @location.line_no, @location.column_no, "T",
        @typedcl_type, @type_name, @type_rep]
    end
  end

  # == DESCRIPTION
  # Global variable declaration information.
  class GVarDcl < CodeStructure
    # === DESCRIPTION
    # Constructs the global variable declaration information.
    #
    # === PARAMETER
    # _location_:: Location -- Location where the declaration appears.
    # _variable_name_:: String -- Global variable name.
    # _type_rep_:: String -- Type of the global variable.
    def initialize(location, variable_name, type_rep)
      @location = location
      @variable_name = variable_name
      @type_rep = type_rep
    end

    # === DESCRIPTION
    # Converts this code structure information into array representation.
    #
    # === RETURN VALUE
    # Array< Object > -- Array representation.
    def to_a
      ["DCL", @location.fpath, @location.line_no, @location.column_no, "V",
        @variable_name, @type_rep]
    end
  end

  # == DESCRIPTION
  # Function declaration information.
  class FuncDcl < CodeStructure
    # === DESCRIPTION
    # Constructs the function declaration information.
    #
    # === PARAMETER
    # _location_:: Location -- Location where the declaration appears.
    # _function_linkage_type_:: String -- Functionn linkage type string.
    # _function_scope_type_:: String -- Declaration scope type string.
    # _function_identifier_:: String -- Function identifier.
    def initialize(location, function_linkage_type, function_scope_type,
                   function_identifier)
      @location = location
      @function_linkage_type = function_linkage_type
      @function_scope_type = function_scope_type
      @function_identifier = function_identifier
    end

    # === DESCRIPTION
    # Converts this code structure information into array representation.
    #
    # === RETURN VALUE
    # Array< Object > -- Array representation.
    def to_a
      ["DCL", @location.fpath, @location.line_no, @location.column_no, "F",
        @function_linkage_type, @function_scope_type,
        @function_identifier.name, @function_identifier.signature]
    end
  end

  # == DESCRIPTION
  # Variable definition information.
  class VarDef < CodeStructure
    # === DESCRIPTION
    # Constructs the variable definition information.
    #
    # === PARAMETER
    # _location_:: Location -- Location where the definition appears.
    # _var_linkage_type_:: String -- Variable linkage type string.
    # _var_scope_type_:: String -- Variable scope type string.
    # _storage_class_type_:: String -- Variable storage class type.
    # _variable_name_:: String -- Variable name.
    # _type_rep_:: String -- Variable type representation string.
    def initialize(location, var_linkage_type, var_scope_type,
                   storage_class_type, variable_name, type_rep)
      @location = location
      @var_linkage_type = var_linkage_type
      @var_scope_type = var_scope_type
      @storage_class_type = storage_class_type
      @variable_name = variable_name
      @type_rep = type_rep
    end

    # === DESCRIPTION
    # Converts this code structure information into array representation.
    #
    # === RETURN VALUE
    # Array< Object > -- Array representation.
    def to_a
      ["DEF", @location.fpath, @location.line_no, @location.column_no, "V",
        @var_linkage_type, @var_scope_type, @storage_class_type,
        @variable_name, @type_rep]
    end
  end

  # == DESCRIPTION
  # Function definition information.
  class FuncDef < CodeStructure
    # === DESCRIPTION
    # Constructs the function definition information.
    #
    # === PARAMETER
    # _location_:: Location -- Location where the definition appears.
    # _function_linkage_type_:: String -- Function linkage type string.
    # _function_scope_type_:: String -- Definition scope type string.
    # _function_identifier_:: String -- Function identifier.
    # _lines_:: Integer -- Physical lines.
    def initialize(location, function_linkage_type, function_scope_type,
                   function_identifier, lines)
      @location = location
      @function_linkage_type = function_linkage_type
      @function_scope_type = function_scope_type
      @function_identifier = function_identifier
      @lines = lines
    end

    # === DESCRIPTION
    # Converts this code structure information into array representation.
    #
    # === RETURN VALUE
    # Array< Object > -- Array representation.
    def to_a
      ["DEF", @location.fpath, @location.line_no, @location.column_no, "F",
        @function_linkage_type, @function_scope_type,
        @function_identifier.name, @function_identifier.signature, @lines]
    end
  end

  # == DESCRIPTION
  # Macro definition information.
  class MacroDef < CodeStructure
    # === DESCRIPTION
    # Constructs the macro definition information.
    #
    # === PARAMETER
    # _location_:: Location -- Location where the definition appears.
    # _macro_name_:: String -- Macro name.
    # _macro_type_:: String -- Macro type string.
    def initialize(location, macro_name, macro_type)
      @location = location
      @macro_name = macro_name
      @macro_type = macro_type
    end

    # === DESCRIPTION
    # Converts this code structure information into array representation.
    #
    # === RETURN VALUE
    # Array< Object > -- Array representation.
    def to_a
      ["DEF", @location.fpath, @location.line_no, @location.column_no, "M",
        @macro_name, @macro_type]
    end
  end

  # == DESCRIPTION
  # Label definition information.
  class LabelDef < CodeStructure
    # === DESCRIPTION
    # Constructs the label definition information.
    #
    # === PARAMETER
    # _location_:: Location -- Location where the definition appears.
    # _label_name_:: String -- Label name.
    def initialize(location, label_name)
      @location = location
      @label_name = label_name
    end

    # === DESCRIPTION
    # Converts this code structure information into array representation.
    #
    # === RETURN VALUE
    # Array< Object > -- Array representation.
    def to_a
      ["DEF", @location.fpath, @location.line_no, @location.column_no, "L",
        @label_name]
    end
  end

  # == DESCRIPTION
  # Initialization information.
  class Initialization < CodeStructure
    # === DESCRIPTION
    # Constructs the initialization information.
    #
    # === PARAMETER
    # _location_:: Location -- Location where the variable appears.
    # _variable_name_:: String -- Initialized variable name.
    # _initializer_rep_:: String -- Initializer representation.
    def initialize(location, variable_name, initializer_rep)
      @location = location
      @variable_name = variable_name
      @initializer_rep = initializer_rep
    end

    # === DESCRIPTION
    # Converts this code structure information into array representation.
    #
    # === RETURN VALUE
    # Array< Object > -- Array representation.
    def to_a
      ["INI", @location.fpath, @location.line_no, @location.column_no,
        @variable_name, @initializer_rep]
    end
  end

  # == DESCRIPTION
  # Assignment information.
  class Assignment < CodeStructure
    # === DESCRIPTION
    # Constructs the assignment information.
    #
    # === PARAMETER
    # _location_:: Location -- Location where the variable appears.
    # _variable_name_:: String -- Assigned variable name.
    # _assignment_rep_:: String -- Assignment expression representation.
    def initialize(location, variable_name, assignment_rep)
      @location = location
      @variable_name = variable_name
      @assignment_rep = assignment_rep
    end

    # === DESCRIPTION
    # Converts this code structure information into array representation.
    #
    # === RETURN VALUE
    # Array< Object > -- Array representation.
    def to_a
      ["ASN", @location.fpath, @location.line_no, @location.column_no,
        @variable_name, @assignment_rep]
    end
  end

  # == DESCRIPTION
  # Header include information.
  class Include < CodeStructure
    # === DESCRIPTION
    # Constructs the header include information.
    #
    # === PARAMETER
    # _location_:: Location -- Location where the directive appears.
    # _included_fpath_:: Pathname -- Path name of the included file.
    def initialize(location, included_fpath)
      @location = location
      @included_fpath = included_fpath
    end

    # === DESCRIPTION
    # Converts this code structure information into array representation.
    #
    # === RETURN VALUE
    # Array< Object > -- Array representation.
    def to_a
      ["DEP", @location.fpath, @location.line_no, @location.column_no, "I",
        @included_fpath]
    end
  end

  # == DESCRIPTION
  # Function call information.
  class Call < CodeStructure
    # === DESCRIPTION
    # Constructs the function call informatin.
    #
    # === PARAMETER
    # _location_:: Location -- Location where the function call appears.
    # _caller_function_:: FunctionIdentifier -- Calling function identifier.
    # _callee_function_:: FunctionIdentifier -- Called function identifier.
    def initialize(location, caller_function, callee_function)
      @location = location
      @caller_function = caller_function
      @callee_function = callee_function
    end

    # === DESCRIPTION
    # Converts this code structure information into array representation.
    #
    # === RETURN VALUE
    # Array< Object > -- Array representation.
    def to_a
      ["DEP", @location.fpath, @location.line_no, @location.column_no, "C",
        @caller_function.name, @caller_function.signature,
        @callee_function.name, @callee_function.signature]
    end
  end

  # == DESCRIPTION
  # Variable cross reference information.
  class XRefVar < CodeStructure
    # === DESCRIPTION
    # Constructs the cross reference information.
    #
    # === PARAMETER
    # _location_:: Location -- Location where the cross-ref appears.
    # _accessor_function_:: FunctionIdentifier -- Accessing function
    #                       identifier.
    # _access_type_:: String -- Access type string.
    # _accessee_variable_:: String -- Accessed variable name.
    def initialize(location, accessor_function, access_type, accessee_variable)
      @location = location
      @accessor_function = accessor_function
      @access_type = access_type
      @accessee_variable = accessee_variable
    end

    # === DESCRIPTION
    # Converts this code structure information into array representation.
    #
    # === RETURN VALUE
    # Array< Object > -- Array representation.
    def to_a
      ["DEP", @location.fpath, @location.line_no, @location.column_no, "X",
        "V", @accessor_function.name, @accessor_function.signature,
        @access_type, @accessee_variable]
    end
  end

  class XRefFunc < CodeStructure
    # === DESCRIPTION
    # Constructs the cross reference information.
    #
    # === PARAMETER
    # _location_:: Location -- Location where the cross-ref appears.
    # _accessor_function_:: FunctionIdentifier -- Accessing function
    #                       identifier.
    # _access_type_:: String -- Access type string.
    # _accessee_function_:: FunctionIdentifier -- Accessed function identifier.
    def initialize(location, accessor_function, access_type, accessee_function)
      @location = location
      @accessor_function = accessor_function
      @access_type = access_type
      @accessee_function = accessee_function
    end

    # === DESCRIPTION
    # Converts this code structure information into array representation.
    #
    # === RETURN VALUE
    # Array< Object > -- Array representation.
    def to_a
      ["DEP", @location.fpath, @location.line_no, @location.column_no, "X",
        "F", @accessor_function.name, @accessor_function.signature,
        @access_type, @accessee_function.name, @accessee_function.signature]
    end
  end

  class Literal < CodeStructure
    def initialize(location, literal_type, literal_prefix, literal_suffix,
                   literal_value)
      @location = location
      @literal_type = literal_type
      @literal_prefix = literal_prefix
      @literal_suffix = literal_suffix
      @literal_value = literal_value
    end

    def to_a
      ["LIT", @location.fpath, @location.line_no, @location.column_no,
        @literal_type, @literal_prefix, @literal_suffix, @literal_value]
    end
  end

  class PPDirective < CodeStructure
    def initialize(location, pp_directive, pp_tokens)
      @location = location
      @pp_directive = pp_directive
      @pp_tokens = pp_tokens
    end

    def to_a
      ["PRE", @location.fpath, @location.line_no, @location.column_no,
        @pp_directive, @pp_tokens]
    end
  end

  class FunctionIdentifier
    def initialize(name, signature)
      @name = name
      @signature = signature
    end

    attr_reader :name
    attr_reader :signature
  end

  class CodeExtraction
    def initialize(context)
      @context = context
    end

    def execute
      do_prepare(@context)
      do_execute(@context)
    end

    private
    def do_prepare(context)
      subclass_responsibility
    end

    def do_execute(context)
      subclass_responsibility
    end

    def report
      @context.report
    end
  end

end
