# Base classes of the warning and the error messages.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/traits"
require "adlint/token"
require "adlint/error"
require "adlint/version"

module AdLint #:nodoc:

  # == DESCRIPTION
  # Base class of all messages.
  class Message
    # === DESCRIPTION
    # Creates a fatal error message from exception object.
    #
    # === PARAMETER
    # _exception_:: Exception -- Cause exception.
    def self.from_exception(exception)
      FatalErrorMessage.new(exception)
    end

    # === DESCRIPTION
    # Constructs a message.
    #
    # === PARAMETER
    # _id_:: Symbol -- Message ID symbol.
    # _location_:: Location -- Location where the message detected.
    # _parts_:: Array< Object > -- Message formatting parts.
    def initialize(id, location, *parts)
      @id, @location = id, location
      @parts = parts
      validate_id(@id, @location)
    end

    attr_reader :id

    # === DESCRIPTION
    # Reads the type of this message.
    #
    # Subclasses must implement this method.
    #
    # === RETURN VALUE
    # Symbol -- Message type symbol.
    def type
      subclass_responsibility
    end

    # === DESCRIPTION
    # Reads the type string of this message.
    #
    # Subclasses must implement this method.
    #
    # === RETURN VALUE
    # String -- Message type string.
    def type_str
      subclass_responsibility
    end

    # === DESCRIPTION
    # Converts this message into an array of message elements.
    #
    # === RETURN VALUE
    # Array< Object > -- Array of message elements.
    def to_a
      if templ = MessageCatalog.instance[@id]
        begin
          mesg = templ.to_default_external %
            @parts.map { |obj| obj.to_s.to_default_external }
          [type.to_s, @location.fpath, @location.line_no, @location.column_no,
            @id, mesg]
        rescue
          raise InvalidMessageFormatError.new(@id, @location)
        end
      else
        raise InvalidMessageIdError.new(@id, @location)
      end
    end

    # === DESCRIPTION
    # Converts this message into a string for human readable output.
    #
    # === RETURN VALUE
    # String -- String representation of this message.
    def to_s
      @location.to_s.to_default_external +
        ":#{type_str.to_default_external}:%5$s:%6$s".to_default_external %
          to_a.map { |obj| obj.to_s.to_default_external }
    end

    def to_csv
      to_a.map { |obj| obj ? obj.to_s.to_default_external : nil }.to_csv
    end

    def eql?(rhs)
      @id == rhs.id && @location == rhs.location && @parts == rhs.parts
    end

    def hash
      [@id, @location, @parts].hash
    end

    protected
    attr_reader :location
    attr_reader :parts

    private
    # === DESCRIPTION
    # Validates the message ID.
    #
    # === PARAMETER
    # _id_:: String -- Message ID string.
    # _location_:: Location -- Location where the message detected.
    #
    # === RETURN VALUE
    # None.
    def validate_id(id, location)
      MessageCatalog.instance[id] or
        raise InvalidMessageIdError.new(id, location)
      nil
    end
  end

  # == DESCRIPTION
  # Syntactical error message.
  class ErrorMessage < Message
    # === DESCRIPTION
    # Reads the type of this message.
    #
    # === RETURN VALUE
    # Symbol -- Message type symbol.
    def type
      :E
    end

    # === DESCRIPTION
    # Reads the type string of this message.
    #
    # === RETURN VALUE
    # String -- Message type string.
    def type_str
      "error"
    end
  end

  # == DESCRIPTION
  # AdLint specific internal fatal error message.
  class FatalErrorMessage < Message
    # === DESCRIPTION
    # Constructs a AdLint specific internal fatal error message.
    #
    # === PARAMETER
    # _exception_:: Exception -- Cause exception.
    def initialize(exception)
      super(exception.id,
            exception.location ? exception.location : Location.new,
            *exception.parts)
      @exception = exception
    end

    # === DESCRIPTION
    # Reads the type of this message.
    #
    # === RETURN VALUE
    # Symbol -- Message type symbol.
    def type
      :X
    end

    # === DESCRIPTION
    # Reads the type string of this message.
    #
    # === RETURN VALUE
    # String -- Message type string.
    def type_str
      "fatal error"
    end
  end

  # == DESCRIPTION
  # Semantical warning message.
  class WarningMessage < Message
    # === DESCRIPTION
    # Reads the type of this message.
    #
    # === RETURN VALUE
    # Symbol -- Message type symbol.
    def type
      :W
    end

    # === DESCRIPTION
    # Reads the type string of this message.
    #
    # === RETURN VALUE
    # String -- Message type string.
    def type_str
      "warning"
    end
  end

  # == DESCRIPTION
  # Message complements other messages.
  class ContextMessage < Message
    # === DESCRIPTION
    # Reads the type of this message.
    #
    # === RETURN VALUE
    # Symbol -- Message type symbol.
    def type
      :C
    end

    # === DESCRIPTION
    # Reads the type string of this message.
    #
    # === RETURN VALUE
    # String -- Message type string.
    def type_str
      "context"
    end
  end

  # == DESCRIPTION
  # Message catalog entry.
  class MessageTemplate < String; end

  # == DESCRIPTION
  # Message catalog.
  class MessageCatalog < Hash
    include Singleton

    DEFAULT_DNAME = Pathname.new("mesg.d")
    private_constant :DEFAULT_DNAME

    DEFAULT_FNAME = Pathname.new("messages.yml")
    private_constant :DEFAULT_FNAME

    def load(fpath = nil)
      unless fpath
        mesg_dpath = DEFAULT_DNAME.expand_path($etcdir)
        lang_dpath = Pathname.new(Traits.instance.of_message.language)
        fpath = mesg_dpath.join(lang_dpath).join(DEFAULT_FNAME)
      end
      File.open(fpath, "r:utf-8") { |io| read_from(io) }
    end

    # === DESCRIPTION
    # Loads message templates from the message definition file.
    #
    # === PARAMETER
    # _io_:: IO -- I/O object for the message definition file.
    #
    # === RETURN VALUE
    # MessageCatalog -- Self.
    def read_from(io)
      case array_or_stream = YAML.load_stream(io)
      when Array
        # NOTE: YAML.load_stream returns Array in Ruby 1.9.3-preview1.
        doc = array_or_stream.first
      when YAML::Stream
        doc = array_or_stream.documents.first
      end
      version = doc["version"]
      validate_version(version)
      if message_definition = doc["message_definition"]
        message_definition.each do |id, text|
          if changed = Traits.instance.of_message.change_list[id.intern]
            self[id.intern] = MessageTemplate.new(changed)
          else
            self[id.intern] = MessageTemplate.new(text)
          end
        end
      else
        raise "invalid form of the message definition file."
      end
      self
    end

    private
    # === DESCRIPTION
    # Validates the version of the message definition file.
    #
    # === RETURN VALUE
    # None.
    def validate_version(version)
      # NOTE: Version field of the message catalog does not mean the schema
      #       version of the catalog file, but a revision number of the catalog
      #       contents.
      #       When AdLint is installed normally, the schema version of the
      #       catalog is always valid.  So, schema version validation is
      #       unnecessary.
    end
  end

  # == DESCRIPTION
  # Base of message detection classes.
  class MessageDetection
    # === DESCRIPTION
    # Constructs a message detecting object.
    #
    # === PARAMETER
    # _context_:: PhaseContext -- Analysis context.
    def initialize(context)
      @context = context
    end

    # === DESCRIPTION
    # Executes a message detecting.
    def execute
      do_prepare(@context)
      do_execute(@context)
    end

    private
    # === DESCRIPTION
    # Initializes a message.
    #
    # Subclasses must implement this method.
    def do_prepare(context)
      subclass_responsibility
    end

    # === DESCRIPTION
    # Detects a message.
    #
    # Subclasses must implement this method.
    def do_execute(context)
      subclass_responsibility
    end

    def report
      @context.report
    end
  end

  class PassiveMessageDetection < MessageDetection
    private
    def do_prepare(context) end
    def do_execute(context) end
  end

end
