# AdLint project configuration.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/version"
require "adlint/util"

module AdLint #:nodoc:

  # == DESCRIPTION
  # Analysis configuration information.
  class Traits
    include Singleton
    include Validation

    # === DESCRIPTION
    # Constructs an empty configuration information.
    def initialize
      @project_traits = nil
      @compiler_traits = nil
      @linker_traits = nil
      @message_traits = nil
    end

    def name
      ""
    end

    ensure_validity_of :project_traits
    ensure_validity_of :compiler_traits
    ensure_validity_of :linker_traits
    ensure_validity_of :message_traits

    def of_project
      @project_traits
    end

    def of_compiler
      @compiler_traits
    end

    def of_linker
      @linker_traits
    end

    def of_message
      @message_traits
    end

    def read_from(traits_fpath)
      File.open(traits_fpath, "r:utf-8") do |io|
        case array_or_stream = YAML.load_stream(io)
        when Array
          # NOTE: YAML.load_stream returns Array in Ruby 1.9.3-preview1.
          doc = array_or_stream.first
        when YAML::Stream
          doc = array_or_stream.documents.first
        end
        validate_version(doc["version"])

        @project_traits = ProjectTraits.new(doc["project_traits"]).freeze
        @compiler_traits = CompilerTraits.new(doc["compiler_traits"]).freeze
        @linker_traits = LinkerTraits.new(doc["linker_traits"]).freeze
        @message_traits = MessageTraits.new(doc["message_traits"]).freeze
      end
    end

    private
    def validate_version(version)
      version == TRAITS_SCHEMA_VERSION or
        raise "invalid version of the trailts file."
    end
  end

  class ProjectTraits
    include Validation

    def initialize(doc)
      @project_name = doc["project_name"]

      if path_strs = doc["include_path"]
        @include_path = path_strs.map { |str| Pathname.new(str) }
      else
        @include_path = []
      end

      @initial_header = doc["initial_header"]
      @coding_style = CodingStyle.new(doc["coding_style"])
      @file_encoding = doc["file_encoding"]
    end

    def name
      "project_traits"
    end

    attr_reader :project_name
    ensure_presence_of :project_name

    attr_reader :include_path
    ensure_dirs_presence_of :include_path

    attr_reader :initial_header
    ensure_file_presence_of :initial_header, :allow_nil => true

    attr_reader :coding_style
    ensure_validity_of :coding_style

    attr_reader :file_encoding
    ensure_with :file_encoding, :message => "is not a valid encoding name.",
                :validator => Encoding.method(:include_name?)

    class CodingStyle
      include Validation

      K_AND_R = "K&R".freeze
      ALLMAN = "Allman".freeze
      GNU = "GNU".freeze

      def initialize(doc)
        @indent_style = doc["indent_style"]
        @tab_width = doc["tab_width"]
        @indent_width = doc["indent_width"]
      end

      def name
        "project_traits:coding_style"
      end

      attr_reader :indent_style
      ensure_inclusion_of :indent_style, :values => [K_AND_R, ALLMAN, GNU]

      attr_reader :tab_width
      ensure_numericality_of :tab_width, :only_integer => true, :min => 1

      attr_reader :indent_width
      ensure_numericality_of :indent_width, :only_integer => true, :min => 1

      def freeze
        @indent_style.freeze
        @tab_width.freeze
        @indent_width.freeze
        super
      end
    end

    def freeze
      @project_name.freeze
      @include_path.freeze
      @initial_header.freeze
      @coding_style.freeze
      @file_encoding.freeze
      super
    end
  end

  # == DESCRIPTION
  # Traits information of the compiler used in the project.
  class CompilerTraits
    include Validation

    def initialize(doc)
      @standard_type = StandardType.new(doc["standard_type"])

      if path_strs = doc["include_path"]
        @include_path = path_strs.map { |str| Pathname.new(str) }
      else
        @include_path = []
      end

      @initial_header = doc["initial_header"]
      @arithmetic = Arithmetic.new(doc["arithmetic"])
      @extension_substitution = doc["extension_substitution"] || {}
      @arbitrary_substitution = doc["arbitrary_substitution"] || {}
      @identifier_max = doc["identifier_max"]
    end

    def name
      "compiler_traits"
    end

    # === VALUE
    # CompilerTraits::StandardType -- The standard type traits information.
    attr_reader :standard_type
    ensure_validity_of :standard_type

    # === VALUE
    # Array< String > -- System include paths.
    attr_reader :include_path
    ensure_dirs_presence_of :include_path

    # === VALUE
    # String -- The file path of the initial source.
    attr_reader :initial_header
    ensure_file_presence_of :initial_header, :allow_nil => true

    # === VALUE
    # CompilerTraits::Arithmetic -- The arithmetic traits information.
    attr_reader :arithmetic
    ensure_validity_of :arithmetic

    # === VALUE
    # Hash<String, String > -- The compiler-extension code substitution
    # setting.
    attr_reader :extension_substitution

    # === VALUE
    # Hash<String, String > -- The arbitrary code substitution setting.
    attr_reader :arbitrary_substitution

    # === VALUE
    # Integer -- Max length of all symbols identified by the compiler.
    attr_reader :identifier_max
    ensure_numericality_of :identifier_max, :only_integer => true, :min => 1

    # == DESCRIPTION
    # Traits information of standard types.
    class StandardType
      include Validation

      def initialize(doc)
        @char_size = doc["char_size"]
        @char_alignment = doc["char_alignment"]

        @short_size = doc["short_size"]
        @short_alignment = doc["short_alignment"]

        @int_size = doc["int_size"]
        @int_alignment = doc["int_alignment"]

        @long_size = doc["long_size"]
        @long_alignment = doc["long_alignment"]

        @long_long_size = doc["long_long_size"]
        @long_long_alignment = doc["long_long_alignment"]

        @float_size = doc["float_size"]
        @float_alignment = doc["float_alignment"]

        @double_size = doc["double_size"]
        @double_alignment = doc["double_alignment"]

        @long_double_size = doc["long_double_size"]
        @long_double_alignment = doc["long_double_alignment"]

        @code_ptr_size = doc["code_ptr_size"]
        @code_ptr_alignment = doc["code_ptr_alignment"]

        @data_ptr_size = doc["data_ptr_size"]
        @data_ptr_alignment = doc["data_ptr_alignment"]

        @char_as_unsigned_char = doc["char_as_unsigned_char"]
      end

      def name
        "compiler_traits:standard_type"
      end

      # === VALUE
      # Integer -- Bit length of the `char' type.
      attr_reader :char_size
      ensure_numericality_of :char_size, :only_integer => true,
                             :min => 1, :max => 256

      attr_reader :char_alignment
      ensure_numericality_of :char_alignment, :only_integer => true, :min => 1

      # === VALUE
      # Integer -- Bit length of the `short' type.
      attr_reader :short_size
      ensure_numericality_of :short_size, :only_integer => true,
                             :min => 1, :max => 256

      attr_reader :short_alignment
      ensure_numericality_of :short_alignment, :only_integer => true, :min => 1

      # === VALUE
      # Integer -- Bit length of the `int' type.
      attr_reader :int_size
      ensure_numericality_of :int_size, :only_integer => true,
                             :min => 1, :max => 256

      attr_reader :int_alignment
      ensure_numericality_of :int_alignment, :only_integer => true, :min => 1

      # === VALUE
      # Integer -- Bit length of the `long' type.
      attr_reader :long_size
      ensure_numericality_of :long_size, :only_integer => true,
                             :min => 1, :max => 256

      attr_reader :long_alignment
      ensure_numericality_of :long_alignment, :only_integer => true, :min => 1

      # === VALUE
      # Integer -- Bit length of the `long long' type.
      attr_reader :long_long_size
      ensure_numericality_of :long_long_size, :only_integer => true,
                             :min => 1, :max => 256

      attr_reader :long_long_alignment
      ensure_numericality_of :long_long_alignment, :only_integer => true,
                             :min => 1

      # === VALUE
      # Integer -- Bit length of the `float' type.
      attr_reader :float_size
      ensure_numericality_of :float_size, :only_integer => true,
                             :min => 1, :max => 256

      attr_reader :float_alignment
      ensure_numericality_of :float_alignment, :only_integer => true, :min => 1

      # === VALUE
      # Integer -- Bit length of the `double' type.
      attr_reader :double_size
      ensure_numericality_of :double_size, :only_integer => true,
                             :min => 1, :max => 256

      attr_reader :double_alignment
      ensure_numericality_of :double_alignment, :only_integer => true,
                             :min => 1

      # === VALUE
      # Integer -- Bit length of the `long double' type.
      attr_reader :long_double_size
      ensure_numericality_of :long_double_size, :only_integer => true,
                             :min => 1, :max => 256

      attr_reader :long_double_alignment
      ensure_numericality_of :long_double_alignment, :only_integer => true,
                             :min => 1

      attr_reader :code_ptr_size
      ensure_numericality_of :code_ptr_size, :only_integer => true,
                             :min => 1, :max => 256

      attr_reader :code_ptr_alignment
      ensure_numericality_of :code_ptr_alignment, :only_integer => true,
                             :min => 1

      attr_reader :data_ptr_size
      ensure_numericality_of :data_ptr_size, :only_integer => true,
                             :min => 1, :max => 256

      attr_reader :data_ptr_alignment
      ensure_numericality_of :data_ptr_alignment, :only_integer => true,
                             :min => 1

      # === VALUE
      # Boolean -- The flag indicates `char' is `unsigned char'.
      attr_reader :char_as_unsigned_char
      ensure_true_or_false_of :char_as_unsigned_char

      def freeze
        @char_size.freeze
        @char_alignment.freeze
        @short_size.freeze
        @short_alignment.freeze
        @int_size.freeze
        @int_alignment.freeze
        @long_size.freeze
        @long_alignment.freeze
        @long_long_size.freeze
        @long_long_alignment.freeze
        @float_size.freeze
        @float_alignment.freeze
        @double_size.freeze
        @double_alignment.freeze
        @long_double_size.freeze
        @long_double_alignment.freeze
        @code_ptr_size.freeze
        @code_ptr_alignment.freeze
        @data_ptr_size.freeze
        @data_ptr_alignment.freeze
        @char_as_unsigned_char.freeze
        super
      end
    end
    private_constant :StandardType

    # == DESCRIPTION
    # Traits information of arithmetic process.
    class Arithmetic
      include Validation

      def initialize(doc)
        @logical_right_shift = doc["logical_right_shift"]
      end

      def name
        "compiler_traits:arithmetic"
      end

      # === VALUE
      # Boolean -- The flag value indicates the right shift operation is
      # logical.
      attr_reader :logical_right_shift
      ensure_true_or_false_of :logical_right_shift

      def freeze
        @logical_right_shift.freeze
        super
      end
    end
    private_constant :Arithmetic

    def freeze
      @standard_type.freeze
      @include_path.freeze
      @initial_header.freeze
      @arithmetic.freeze
      @extension_substitution.freeze
      @arbitrary_substitution.freeze
      @identifier_max.freeze
      super
    end
  end

  # == DESCRIPTION
  # Traits information of the linker used in the project.
  class LinkerTraits
    include Validation

    def initialize(doc)
      @identifier_max = doc["identifier_max"]
      @identifier_ignore_case = doc["identifier_ignore_case"]
    end

    def name
      "linker_traits"
    end

    # === VALUE
    # Integer -- Max length of all external symbols identified by the linker.
    attr_reader :identifier_max
    ensure_numericality_of :identifier_max, :only_integer => true, :min => 1

    # === VALUE
    # Boolean -- The flag indicates that external symbols are identified
    # without case by the linker.
    attr_reader :identifier_ignore_case
    ensure_true_or_false_of :identifier_ignore_case

    def freeze
      @identifier_max.freeze
      @identifier_ignore_case.freeze
      super
    end
  end

  class MessageTraits
    include Validation

    def initialize(doc)
      @language = doc["language"]
      str_str_hash = doc["change_list"] || {}
      @change_list = str_str_hash.each_with_object({}) { |(id, text), hash|
        hash[id.intern] = text
      }
    end

    def name
      "message_traits"
    end

    attr_reader :language
    ensure_inclusion_of :language, :values => %w(ja_JP en_US)

    attr_reader :change_list

    def freeze
      @language.freeze
      @change_list.freeze
      super
    end
  end

  module StandardTypeAccessor
    def standard_type
      Traits.instance.of_compiler.standard_type
    end
    module_function :standard_type

    def char_size
      standard_type.char_size
    end
    module_function :char_size

    def char_alignment
      standard_type.char_alignment
    end
    module_function :char_alignment

    def short_size
      standard_type.short_size
    end
    module_function :short_size

    def short_alignment
      standard_type.short_alignment
    end
    module_function :short_alignment

    def int_size
      standard_type.int_size
    end
    module_function :int_size

    def int_alignment
      standard_type.int_alignment
    end
    module_function :int_alignment

    def long_size
      standard_type.long_size
    end
    module_function :long_size

    def long_alignment
      standard_type.long_alignment
    end
    module_function :long_alignment

    def long_long_size
      standard_type.long_long_size
    end
    module_function :long_long_size

    def long_long_alignment
      standard_type.long_long_alignment
    end
    module_function :long_long_alignment

    def float_size
      standard_type.float_size
    end
    module_function :float_size

    def float_alignment
      standard_type.float_alignment
    end
    module_function :float_alignment

    def double_size
      standard_type.double_size
    end
    module_function :double_size

    def double_alignment
      standard_type.double_alignment
    end
    module_function :double_alignment

    def long_double_size
      standard_type.long_double_size
    end
    module_function :long_double_size

    def long_double_alignment
      standard_type.long_double_alignment
    end
    module_function :long_double_alignment

    def code_ptr_size
      standard_type.code_ptr_size
    end
    module_function :code_ptr_size

    def code_ptr_alignment
      standard_type.code_ptr_alignment
    end
    module_function :code_ptr_alignment

    def data_ptr_size
      standard_type.data_ptr_size
    end
    module_function :data_ptr_size

    def data_ptr_alignment
      standard_type.data_ptr_alignment
    end
    module_function :data_ptr_alignment

    def char_as_unsigned_char?
      standard_type.char_as_unsigned_char
    end
    module_function :char_as_unsigned_char?
  end

  module CodingStyleAccessor
    def coding_style
      Traits.instance.of_project.coding_style
    end
    module_function :coding_style

    def indent_style
      coding_style.indent_style
    end
    module_function :indent_style

    def tab_width
      coding_style.tab_width
    end
    module_function :tab_width

    def indent_width
      coding_style.indent_width
    end
    module_function :indent_width

    INDENT_STYLE_K_AND_R = ProjectTraits::CodingStyle::K_AND_R
    INDENT_STYLE_ALLMAN = ProjectTraits::CodingStyle::ALLMAN
    INDENT_STYLE_GNU = ProjectTraits::CodingStyle::GNU
  end

end
