# Pre-analysis code substitution mechanism.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/report"
require "adlint/util"
require "adlint/cpp/lexer"

module AdLint #:nodoc:
module Cpp #:nodoc:

  class CodeSubstitution
    def initialize(pattern_str, replacement_str)
      @pattern = StringToPPTokensLexer.new(pattern_str).execute.to_a
      @replacement = StringToPPTokensLexer.new(replacement_str).execute.to_a
    end

    extend Pluggable

    def_plugin :on_substitution

    def execute(tokens)
      result = []
      index = 0
      while first_token = tokens[index]
        matcher = Matcher.new(@pattern)
        matched_length = matcher.match(tokens, index)
        if matcher.accepted? || index + matched_length == tokens.size
          notify_substitution(tokens, index, matched_length)
          result.concat(@replacement.map { |t|
            Token.new(t.type, t.value, first_token.location, t.type_hint)
          })
          index += matched_length
        else
          result.push(first_token)
          index += 1
        end
      end
      result
    end

    private
    def notify_substitution(tokens, index, length)
      matched_tokens = tokens[index, length]
      on_substitution.invoke(matched_tokens) unless matched_tokens.empty?
    end
  end

  class Matcher
    def initialize(pattern_tokens)
      @state = OuterTokenMatching.new(self)
      @pattern_tokens = pattern_tokens
      @pattern_index = 0
    end

    def rest_pattern_tokens
      @pattern_tokens[@pattern_index..-1]
    end

    def next_pattern_token
      pattern_token = @pattern_tokens[@pattern_index]
      @pattern_index += 1
      pattern_token
    end

    def match(tokens, index)
      return 0 if head = tokens[index] and head.type == :NEW_LINE

      match_length = 0
      while token = tokens[index]
        unless token.type == :NEW_LINE
          @state = @state.process(token)
          break unless @state.matching?
        end
        match_length += 1
        index += 1
      end
      match_length
    end

    def accepted?
      @state.accepted?
    end

    def matching?
      @state.matching?
    end

    def rejected?
      @state.rejected?
    end

    class State
      def initialize(matcher)
        @matcher = matcher
      end

      attr_reader :matcher

      def process(token)
        subclass_responsibility
      end

      def accepted?
        subclass_responsibility
      end

      def rejected?
        subclass_responsibility
      end

      def matching?
        !accepted? && !rejected?
      end

      private
      def next_pattern_token
        @matcher.next_pattern_token
      end

      def rest_pattern_tokens
        @matcher.rest_pattern_tokens
      end
    end
    private_constant :State

    class Accepted < State
      def accepted?
        true
      end

      def rejected?
        false
      end
    end
    private_constant :Accepted

    class Rejected < State
      def accepted?
        false
      end

      def rejected?
        true
      end
    end
    private_constant :Rejected

    class Matching < State
      def accepted?
        false
      end

      def rejected?
        false
      end
    end
    private_constant :Matching

    class OuterTokenMatching < Matching
      def process(token)
        if pattern_token = next_pattern_token
          if token.value == pattern_token.value
            case token.value
            when "(", "[", "{"
              InnerTokenMatching.new(matcher, self)
            else
              self
            end
          else
            if pattern_token.value == "__adlint__any"
              if sentry_token = rest_pattern_tokens.first and
                  token.value == sentry_token.value
                case token.value
                when "(", "[", "{"
                  InnerTokenMatching.new(matcher, self).process(token)
                else
                  self
                end
              else
                OuterAnyMatching.new(matcher)
              end
            else
              Rejected.new(matcher)
            end
          end
        else
          Accepted.new(matcher)
        end
      end
    end
    private_constant :OuterTokenMatching

    class OuterAnyMatching < Matching
      def process(token)
        if sentry_token = rest_pattern_tokens.first
          if token.value == sentry_token.value
            return OuterTokenMatching.new(@matcher).process(token)
          end
        end
        self
      end
    end
    private_constant :OuterAnyMatching

    class InnerTokenMatching < Matching
      def initialize(matcher, prev_state)
        super(matcher)
        @prev_state = prev_state
      end

      def process(token)
        if pattern_token = next_pattern_token
          if token.value == pattern_token.value
            case token.value
            when "(", "[", "{"
              InnerTokenMatching.new(matcher, self)
            when ")", "]", "}"
              @prev_state
            else
              self
            end
          else
            if pattern_token.value == "__adlint__any"
              case token.value
              when "(", "[", "{"
                InnerAnyMatching.new(matcher, self, 1)
              else
                InnerAnyMatching.new(matcher, self, 0)
              end
            else
              Rejected.new(matcher)
            end
          end
        else
          Accepted.new(matcher)
        end
      end
    end
    private_constant :InnerTokenMatching

    class InnerAnyMatching < Matching
      def initialize(matcher, prev_state, depth)
        super(matcher)
        @prev_state = prev_state
        @depth = depth
      end

      def process(token)
        case token.value
        when "(", "[", "{"
          @depth += 1
        when ")", "]", "}"
          @depth -= 1
        end

        if sentry_token = rest_pattern_tokens.first
          if token.value == sentry_token.value
            if @depth < 0
              return @prev_state.process(token)
            end
          end
        end
        self
      end
    end
    private_constant :InnerAnyMatching
  end

end
end
