// ---------------------------------------------------------------------------
// - Rco.hpp                                                                 -
// - afnix:csm service - registered cloud object class definition            -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2016 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_RCO_HPP
#define  AFNIX_RCO_HPP

#ifndef  AFNIX_SERIAL_HPP
#include "Serial.hpp"
#endif

#ifndef  AFNIX_NAMEABLE_HPP
#include "Nameable.hpp"
#endif

namespace afnix {

  /// The Rco class is a registered cloud object. An object is registed as
  /// soon as its registration id is set. If the registration id is unset
  /// the object is unregistered or anonymous. The registration id can be
  /// anything as long as as it is understood by the implementation that
  /// such registration is to be interpreted somewhere else.
  /// @author amaury darsch

  class Rco : public Nameable, public Serial {
  protected:
    /// the key id
    t_long  d_kid;
    /// the registration id
    String  d_rid;
    /// the registration name
    String  d_name;
    /// the registration name
    String  d_info;
    /// the registered object
    Object* p_rco;

  public:
    /// create nil object
    Rco (void);

    /// create an anonymous object
    /// @param rco the object to regsiter 
    Rco (Object* rco);

    /// create a registered object by name
    /// @param rco  the registrered object
    /// @param name the registration name
    Rco (Object* rco, const String& name);

    /// create a registered object by name and info
    /// @param rco  the registrered object
    /// @param name the registration name
    /// @param info the registration info
    Rco (Object* rco, const String& name, const String& info);

    /// create a registered object by rid, name and info
    /// @param rid  the registration id
    /// @param rco  the registrered object
    /// @param name the registration name
    /// @param info the registration info
    Rco (const String& rid, 
	 Object* rco, const String& name, const String& info);

    /// copy construct this registered object
    /// @param that the object to copy
    Rco (const Rco& that);

    /// destroy this registered object
    ~Rco (void);

    /// @return the class name
    String repr (void) const;

    /// @return a clone of this object
    Object* clone (void) const;

    /// @return the rco serial id
    t_byte serialid (void) const;

    /// serialize this object
    /// @param os the output stream
    void wrstream (OutputStream& os) const;

    /// deserialize this object
    /// @param is the input stream
    void rdstream (InputStream& os);

    /// @return the registration name
    String getname (void) const;

    /// assign a rco to this one
    /// @param that the object to assign
    Rco& operator = (const Rco& that);

    /// validate a roc key id
    /// @param kid the key id to validate
    virtual bool iskid (const t_long kid) const;

    /// set the key id
    /// @param kid the kid to set
    virtual void setkid (const t_long kid);

    /// @return the key id
    virtual t_long getkid (void) const;

    /// validate a rco registration id
    /// @param rid the rid to validate
    virtual bool isrid (const String& rid) const;

    /// set the registration id
    /// @param rid the rid to set
    virtual void setrid (const String& rid);

    /// @return the registration id
    virtual String getrid (void) const;

    /// @return the registration info
    virtual String getinfo (void) const;

    /// @return the registered object
    virtual Object* getobj (void) const;

  public:
    /// create a object in a generic way
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// apply this object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);
  };
}

#endif
