// ---------------------------------------------------------------------------
// - Condvar.cpp                                                             -
// - standard object library - condition variable class implementation       -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "Condvar.hpp"
#include "Exception.hpp"
#include "cthr.hpp"

namespace aleph {

  // the supported condition variable quarks
  static const long QUARK_LOCK     = String::intern ("lock");
  static const long QUARK_WAIT     = String::intern ("wait");
  static const long QUARK_MARK     = String::intern ("mark");
  static const long QUARK_RESET    = String::intern ("reset");
  static const long QUARK_UNLOCK   = String::intern ("unlock");
  static const long QUARK_WAITUNLK = String::intern ("wait-unlock");

  // create a new condition variable

  Condvar::Condvar (void) {
    d_cond = false;
    p_mtx  = c_mtxcreate ();
    if (p_mtx == nilp) 
      throw Exception ("condition-error", "cannot create condition variable");
    p_tcv = c_tcvcreate ();
    if (p_tcv == nilp) {
      c_mtxdestroy (p_mtx);
      throw Exception ("condition-error", "cannot create condition variable");
    }
  }

  // destroy this condition variable

  Condvar::~Condvar (void) {
    c_tcvdestroy (p_tcv);
    c_mtxdestroy (p_mtx);
  }

  // return the class name

  String Condvar::repr (void) const {
    return "Condvar";
  }

  // lock this condition variable

  void Condvar::lock (void) const {
    c_mtxlock (p_mtx);
  }

  // unlock this condition variable

  void Condvar::unlock (void) const {
    c_mtxunlock (p_mtx);
  }

  // wait on this condition variable

  void Condvar::wait (void) const {
    // take the lock
    c_mtxlock (p_mtx);
    // now loop until the condition is true
    while (d_cond == false) c_tcvwait (p_tcv, p_mtx);
  }

  // wait on this condition variable and unlock

  void Condvar::waitunlk (void) {
    // take the lock
    c_mtxlock (p_mtx);
    // now loop until the condition is true
    while (d_cond == false) c_tcvwait (p_tcv, p_mtx);
    d_cond = false;
    c_mtxunlock (p_mtx);
  }

  // mark this condition variable and broadcast

  void Condvar::mark (void) {
    c_mtxlock (p_mtx);
    d_cond = true;
    c_tcvbdcast (p_tcv);
    c_mtxunlock (p_mtx);
  }
  
  // reset this condition variable

  void Condvar::reset (void) {
    c_mtxlock (p_mtx);
    d_cond = false;
    c_mtxunlock (p_mtx);
  }
  
  // create a new condition variable in a generic way

  Object* Condvar::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    if (argc == 0) return new Condvar;
    throw Exception ("argument-error", "too many argument with condvar");
  }

  // apply a condition variable method with a set of arguments and a quark

  Object* Condvar::apply (Runnable* robj, Nameset* nset, const long quark,
			  Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_LOCK) {
	lock ();
	return nilp;
      }
      if (quark == QUARK_UNLOCK) {
	unlock ();
	return nilp;
      }
      if (quark == QUARK_WAIT) {
	wait ();
	return nilp;
      }
      if (quark == QUARK_MARK) {
	mark ();
	return nilp;
      }
      if (quark == QUARK_WAITUNLK) {
	waitunlk ();
	return nilp;
      }
      if (quark == QUARK_RESET) {
	reset ();
	return nilp;
      }
    }

    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }

}
