// ---------------------------------------------------------------------------
// - Constant.cpp                                                            -
// - aleph engine - constant object class implementation                     -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Output.hpp"
#include "Engsid.hpp"
#include "Constant.hpp"
#include "Exception.hpp"

namespace aleph {
  // this procedure returns a new constant object for deserialization
  static Serial* mksob (void) {
    return new Constant;
  }
  // register this lexical serial id
  static const t_byte SERIAL_ID = Serial::setsid (SERIAL_CNST_ID, mksob);

  // create an empty constant

  Constant::Constant (void) {
    p_lobj = nilp;
  }

  // create a new constant
 
  Constant::Constant (Literal* lobj) {
    Object::iref (p_lobj = lobj);
  }

  // copy construct this constant

  Constant::Constant (const Constant& that) {
    Object::iref (p_lobj = that.p_lobj);
  }

  // destroy this constant
  
  Constant::~Constant (void) {
    Object::dref (p_lobj);
  }

  // return the class name

  String Constant::repr (void) const {
    return "Constant";
  }

  // make this constant shared

  void Constant::mksho (void) {
    if (p_shared != nilp) return;
    Object::mksho ();
    if (p_lobj != nilp) p_lobj->mksho ();
  }

  // return a literal representation of this constant
  String Constant::toliteral (void) const {
    return (p_lobj == nilp) ? "" : p_lobj->toliteral ();
  }

  // return a string representation of this constant

  String Constant::tostring (void) const {
    return (p_lobj == nilp) ? "" : p_lobj->tostring ();
  }

  // clone this constant

  Object* Constant::clone (void) const {
    return new Constant (*this);
  }

  // return the constant serial code

  t_byte Constant::serialid (void) const {
    return SERIAL_CNST_ID;
  }

  // serialize this constant

  void Constant::wrstream (Output& os) const {
    rdlock ();
    if (p_lobj == nilp)
      os.write (nilc);
    else
      p_lobj->serialize (os);
    unlock ();
  }

  // deserialize this constant

  void Constant::rdstream (Input& is) {
    wrlock ();
    Object*  sobj = Serial::deserialize (is);
    Literal* lobj = dynamic_cast <Literal*> (sobj);
    if (lobj == nilp) {
      String what = Object::repr (sobj);
      Object::cref (sobj);
      unlock ();
      throw Exception ("serial-error", "invalid object for constant", what);
    }
    Object::iref (p_lobj = lobj);
    unlock ();
  }

  // evaluate this constant in the current environment

  Object* Constant::eval (Runnable* robj, Nameset* nset) {
    return (p_lobj == nilp) ? nilp : p_lobj->clone ();
  }
}
