// ---------------------------------------------------------------------------
// - Directory.cpp                                                           -
// - aleph:sio library - directory class implementation                      -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Interp.hpp"
#include "System.hpp"
#include "Vector.hpp"
#include "Directory.hpp"
#include "Exception.hpp"
#include "csio.hpp"
#include "cdir.hpp"

namespace aleph {

  // the directory supported quarks
  static const long QUARK_MKDIR      = String::intern ("mkdir");
  static const long QUARK_RMDIR      = String::intern ("rmdir");
  static const long QUARK_RMFILE     = String::intern ("rmdir");
  static const long QUARK_GETNAME    = String::intern ("get-name");
  static const long QUARK_GETLIST    = String::intern ("get-list");
  static const long QUARK_GETFILES   = String::intern ("get-files");
  static const long QUARK_GETSUBDIRS = String::intern ("get-subdirs");

  // open the default directory

  Directory::Directory (void) {
    char* buffer = c_getcwd ();
    d_name       = buffer;
    p_handle = c_opendir (buffer);
    delete [] buffer;
    if (p_handle == nilp)
      throw Exception ("dir-error", "cannot open directory", d_name);
  }

  // open a directory by name

  Directory::Directory (const String& name) {
    d_name = name;
    char* buffer = name.tochar ();
    p_handle = c_opendir (buffer);
    delete [] buffer;
    if (p_handle == nilp) 
      throw Exception ("dir-error", "cannot open directory", name);
  }

  // destroy this directory 

  Directory::~Directory (void) {
    c_closedir (p_handle);
  }

  // return the class name

  String Directory::repr (void) const {
    return "Directory";
  }

  // return a list of this directory names

  List* Directory::getlist (void) {
    List* result = new List;
    while (true) {
      char* name = c_readdir (p_handle);
      if (name == nilp) break;
      result->insert (new String (name));
      delete [] name;
    }
    return result;
  }

  // return a list of this directory files

  List* Directory::getfiles (void) {
    List* result = new List;
    while (true) {
      char* name = c_readdir (p_handle);
      if (name == nilp) break;
      if (c_isfile (name) == true) result->insert (new String (name));
      delete [] name;
    }
    return result;
  }

  // return a list of the sub directories

  List* Directory::getdirs (void) {
    List* result = new List;
    while (true) {
      char* name = c_readdir (p_handle);
      if (name == nilp) break;
      if (c_isdir (name) == true) result->insert (new String (name));
      delete [] name;
    }
    return result;
  }

  // create a new directory by name

  void Directory::mkdir (const String& name) {
    // join the name
    String path = System::join (d_name, name);
    char*  buf  = path.tochar ();
    // try to create the directory
    if (c_mkdir (buf) == false) {
      delete [] buf;
      throw Exception ("directory-error", "cannot create directory", path);
    }
    delete [] buf;
  }

  // remove an empty directory by name

  void Directory::rmdir (const String& name) {
    String path = System::join (d_name, name);
    if (System::rmdir (path) == false)
      throw Exception ("directory-error", "cannot remove directory", path);
  }

  // remove a file in a directory

  void Directory::rmfile (const String& name) {
    String path = System::join (d_name, name);
    if (System::rmfile (path) == false)
      throw Exception ("directory-error", "cannot remove file", path);
  }

  // create a new directory in a generic way

  Object* Directory::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    // open a default directory
    if (argc == 0) return new Directory;
    // open a directory by name
    if (argc == 1) {
      String val = argv->getstring (0);
      return new Directory (val);
    }
    throw Exception ("argument-error",
		     "too many argument with directory open");
  }

  // apply a directory member with a set of arguments and a quark

  Object* Directory::apply (Runnable* robj, Nameset* nset, const long quark,
			    Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
 
    // check for 0 arguments
    if (argc == 0) {
      if (quark == QUARK_GETNAME)    return new String (d_name);
      if (quark == QUARK_GETLIST)    return getlist  ();
      if (quark == QUARK_GETFILES)   return getfiles ();
      if (quark == QUARK_GETSUBDIRS) return getdirs ();
    }

    // check for one argument
    if (argc == 1) {
      if (quark == QUARK_MKDIR) {
	String val = argv->getstring (0);
	mkdir (val);
	return new Directory (System::join (d_name, val));
      }
      if (quark == QUARK_RMDIR) {
	String val = argv->getstring (0);
	rmdir (val);
	return nilp;
      }
      if (quark == QUARK_RMFILE) {
	String val = argv->getstring (0);
	rmdir (val);
	return nilp;
      }
    }

    // call the object method 
    return Object::apply (robj, nset, quark, argv);
  }
}
