// ---------------------------------------------------------------------------
// - Buffer.hpp                                                              -
// - standard object library - character buffer class definition             -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_BUFFER_HPP
#define  ALEPH_BUFFER_HPP

#ifndef  ALEPH_OUTPUT_HPP
#include "Output.hpp"
#endif

namespace aleph {

  /// The Buffer class is a character buffer which can be used to accumulate
  /// character. The class automatically resize itself when full.
  /// Standard methods to read / write or extract data is provided.
  /// @author amaury darsch

  class Buffer : public virtual Object {
  private:
    /// the character buffer
    char* p_buffer;
    /// the buffer size
    long d_size;
    /// the buffer length
    long d_length;

  public:
    /// create a new buffer class. The buffer is created with a default
    /// size of 1024 characters.
    Buffer (void);

    /// create a new buffer class with a default size.
    /// @param size the buffer default size
    Buffer (const long size);
    
    /// create a new buffer and initialize it with a c string
    /// @param value the c string to initialize
    Buffer (const char* value);

    /// create a new buffer and initialize it with a string
    /// @param value the string to initialize
    Buffer (const String& value);

    /// copy construct a buffer
    Buffer (const Buffer& that);

    /// destroy this buffer
    ~Buffer (void);

    /// @return the class name
    String repr (void) const;

    /// add a character to this buffer
    /// @param value the character to add to this buffer
    void add (const char value);

    /// add a character buffer in this buffer
    /// @param buffer the buffer to add
    /// @param size   the buffer size
    void add (const char* buffer, const long size);

    /// add a string to this buffer
    /// @param value the string to add to this buffer
    void add (const String& value);

    /// add a buffer object to this buffer
    /// @param buffer the buffer to add
    void add (const Buffer& buffer);
    
    /// @return the next character but do not remove it
    char get (void) const;

    /// @return the character at the current index position
    char read (void);

    /// pushback a character in this buffer
    /// @param value the character to push back
    void pushback (const char value);

    /// pushback a character buffer in this buffer
    /// @param buffer the buffer to push back
    /// @param size   the buffer size
    void pushback (const char* buffer, const long size);

    /// pushback a string in this buffer
    /// @param value the string to push back
    void pushback (const String& value);

    /// @return the length of this buffer
    long length (void) const;

    /// @return the buffer contents as a string
    String tostring (void) const;

    /// reset this buffer 
    void reset (void);

    /// map this buffer to an anonymous data structure
    /// @param data the data structure to map
    long map (void* data, const long size);

    /// @return a word (in big endian format) from this buffer 
    t_word getword (void);

    /// @return a quad (in big endian format) from this buffer 
    t_quad getquad (void);

    /// @return a octa (in big endian format) from this buffer 
    t_octa getocta (void);

    /// write the buffer content to an output stream
    /// @param os the output stream
    void write (Output& os) const;

    /// generate a new buffer
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// apply this buffer with a set of arguments and a quark
    /// @param robj   robj the current runnable
    /// @param nset   the current nameset    
    /// @param quark  the quark to apply these arguments
    /// @param argv   the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);

  private:
    // make the assignment operator private
    Buffer& operator = (const Buffer&);
  };
}

#endif
