// ---------------------------------------------------------------------------
// - Output.cpp                                                              -
// - standard object library - output stream class implementation            -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Output.hpp"
#include "Vector.hpp"
#include "Runnable.hpp"
#include "Character.hpp"
#include "Exception.hpp"
#include "ccnv.hpp"

namespace aleph {

  // the output supported quarks
  static const long QUARK_WRITE   = String::intern ("write");
  static const long QUARK_WRITELN = String::intern ("writeln");
  static const long QUARK_ERRORLN = String::intern ("errorln");
  static const long QUARK_NEWLINE = String::intern ("newline");

  // return the stream descriptor

  int Output::getsid (void) const {
    return -1;
  }

  // return true if we have a tty - by default is false

  bool Output::istty (void) const {
    return false;
  }

  // write an array of character to the output stream

  void Output::write (const char* value, const long size) {
    wrlock ();
    try {
      for (long i = 0; i < size; i++)
	write (value[i]);
    } catch (...) {
      unlock ();
      throw;
    }
    unlock ();
  }

  // write a string to the output stream

  void Output::write (const String& value) {
    wrlock ();
    try {
      // get a c-string from this string
      char* cstrg = value.tochar ();
      if (cstrg == nilp) {
	unlock ();
	return;
      }
      // write string content
      write (cstrg);
      delete cstrg;
    } catch (...) {
      unlock ();
      throw;
    }
    unlock ();
  }

  // write a string as an error string

  void Output::error (const String& value) {
    write (value);
  }

  // write a newline to this output stream

  void Output::newline (void) {
    write (eolc);
  }

  // write a string followed by a new line. This is a convenient function

  void Output::writeln (const String& line) {
    write (line + eolc);
  }

  // write a string followed by a new line in error mode

  void Output::errorln (const String& line) {
    error (line + eolc);
  }

  // write an exception string on this output stream

  void Output::errorln (const Exception& e) {
    wrlock ();
    try {
      String what = "exception : ";
      String file = "in file   : "; 
      String resm = "reason    : ";
      String resv = e.getval ();
      if (e.getnlf () == true) newline ();
      error (what + e.geteid ());
      newline ();
      long lnum    = e.getlnum  ();
      String fname = e.getfname ();
      if ((lnum != 0) && (fname.length () != 0)) {
	error (file + fname + " at or around line " + lnum);
	newline ();
      }
      if (resv.length () > 0) {
	error (resm + resv);
	newline ();
      }
    } catch (...) {
      unlock ();
      throw;
    }
    unlock ();
  }

  // write a character on the output stream - no exception is thrown

  Output& Output::operator << (const char value) {
    try {
      this->write (value);
    } catch (const Exception& e) {};
    return *this;
  }

  // write a string on the output stream - no exception is thrown
  
  Output& Output::operator << (const String& value) {
    try {
      this->write (value);
    } catch (const Exception& e) {};
    return *this;
  }

  // write an integer on the output stream - no exception is thrown

  Output& Output::operator << (const long value) {
    char* data = c_ltoa (value);
    try {
      this->write (data);
    } catch (const Exception& e) {
      delete [] data;
    };
    delete [] data;
    return *this;
  }

  // apply this output class with a set of arguments and a quark

  Object* Output::apply (Runnable* robj, Nameset* nset, const long quark,
			 Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch variable length arguments
    if ((quark == QUARK_WRITE) || (quark == QUARK_WRITELN)) {
      String result;
      for (long i = 0; i < argc; i++) {
	Object*   obj = argv->get (i);
	Literal* lobj = dynamic_cast <Literal*> (obj);
	if (lobj == nilp) 
	  throw Exception ("type-error", "invalid object to write", 
			   Object::repr (obj));
	result = result + lobj->tostring ();
      }
      if (quark == QUARK_WRITELN)  result = result + eolc;
      write (result);
      return nilp;
    }

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_NEWLINE) {
	newline ();
	return nilp;
      }
    }

    // dispatch one argument
    if (argc == 1) {
      if (quark == QUARK_ERRORLN) {
	String val = argv->getstring (0);
	errorln (val);
	return nilp;
      }
    }
    
    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }
}
