// ---------------------------------------------------------------------------
// - Vector.hpp                                                              -
// - standard object library - dynamic vector class definition               -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_VECTOR_HPP
#define  ALEPH_VECTOR_HPP

#ifndef  ALEPH_ITERABLE_HPP
#include "Iterable.hpp"
#endif

#ifndef  ALEPH_SERIAL_HPP
#include "Serial.hpp"
#endif

namespace aleph {

  /// The Vector class implementes a dynamic array. It is by far less efficent
  /// than an array where the size of the array is known in advance. This
  /// class should be used when an indexed container is required.
  /// @author amaury darsch

  class Vector : public Iterable, public Serial {
  private:
    /// the current size of this vector
    long d_length;
    /// the size of the allocated vector
    long d_size;
    /// the allocated vector
    Object** p_vector;

  public:
    /// create an empty vector.
    Vector (void);

    /// create a vector with a predefined allocated size
    /// @param size the requested size
    Vector (const long size);

    /// copy constructor for this vector
    /// @param that the vector to copy 
    Vector (const Vector& that);

    /// destroy this vector
    ~Vector (void);

    /// @return the class name
    String repr (void) const;

    /// assign a vector to this one
    /// @param that the vector to assign
    Vector& operator = (const Vector& that);

    /// make this vector a shared object
    void mksho (void);

    /// @return the vector serial code
    t_byte serialid (void) const;

    /// serialize this vector to an output stream
    /// @param os the output stream to write
    void wrstream (class Output& os) const;

    /// deserialize a vector from an input stream
    /// @param is the input steam to read in
    void rdstream (class Input& is);

    /// append an element in this vector
    /// @param object the object to append
    void append (Object* object);

    /// set an object at a given position
    /// @param index the vector index
    /// @param object the object to set
    void set (const long index, Object* object);

    /// get an object at a certain index
    /// @param index the vector index
    /// @return the object at this position
    Object* get (const long index) const;

    /// backtrack one position in this vector
    void back (void);

    /// @return the first object in this vector
    Object* first (void) const {
      return get (0);
    }

    /// @return the last object in this vector
    Object* last (void) const {
      return get (d_length-1);
    }

    /// @return the number of objects in this vector
    long length (void) const;

    /// @return true if the object exists
    bool exists (Object* object) const;

    /// find and object in this vector
    /// @param object the object to find
    long find (Object* object);

    /// remove and object by index and repack the vector
    /// @param index the object index to remove
    void remove (const long index);

    /// remove and object and repack the vector
    /// @param object the object to remove
    void remove (Object* object);

    /// reset this vector
    void reset (void);

    /// @return a new iterator for this vector
    Iterator* makeit (void);

    /// return a long integer value by object index
    t_long getint (const long index) const;

    /// return a real value by object index
    t_real getreal (const long index) const;

    /// return a real value by object index, even from an integer
    t_real getireal (const long index) const;

    /// return a boolean value by object index
    bool getbool (const long index) const;

    /// return a character value by object index
    char getchar (const long index) const;

    /// return a string value by object index
    String getstring (const long index) const;

    /// generate a vector of evaluated arguments
    /// @param robj   the current runnable
    /// @param nset   the current nameset
    /// @param args   the arguments to evaluate
    /// @return a vector of evaluated argument
    static Vector* eval (Runnable*robj, Nameset* nset, Cons* args);

    /// generate a new vector
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// apply this vector with a set of arguments and a quark
    /// @param robj   robj the current runnable
    /// @param nset   the current nameset    
    /// @param quark  the quark to apply these arguments
    /// @param argv   the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);

  private:
    // make the vector iterator a friend
    friend class Vectorit;
  };

  /// The Vectorit class is the iterator for the vector class. Such 
  /// iterator is constructed with the "makeit" vector method. The iterator
  /// is reset to the beginning of the vector.
  /// @author amaury darsch

  class Vectorit : public Iterator {
  private:
    /// the vector to iterate
    Vector* p_vector;
    /// the current vector index
    long d_index;

  public:
    /// create a new iterator from a vector
    /// @param vec the vector to iterate
    Vectorit (Vector* vec);

    /// destroy this vector iterator
    ~Vectorit (void);

    /// @return the class name
    String repr (void) const;

    /// make this iterator a shared object
    void mksho (void);

    /// reset the iterator to the begining
    void begin (void);

    /// reset the iterator to the end
    void end (void);

    /// move the vector iterator to the next position
    void next (void);

    /// move the vector iterator to the previous position
    void prev (void);

    /// @return the object at the current position
    Object* getobj (void);

    /// @return true if the iterator is at the end
    bool isend (void);

  private:
    // make the copy constructor private
    Vectorit (const Vectorit&);
    // make the assignment operator private
    Vectorit& operator = (const Vectorit&);
  };
}

#endif
