// ---------------------------------------------------------------------------
// - Reserved.cpp                                                            -
// - aleph engine - reserved name class implementation                       -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Engsid.hpp"
#include "Nameset.hpp"
#include "Integer.hpp"
#include "Reserved.hpp"

namespace aleph {
  // this procedure returns a new reserved object for deserialization
  static Serial* mksob (void) {
    return new Reserved;
  }
  // register this lexical serial id
  static const t_byte SERIAL_ID = Serial::setsid (SERIAL_RESV_ID, mksob);

  // create a default reserved name
  
  Reserved::Reserved (void) {
    d_quark  = 0;
    d_lnum   = 0;
    p_object = nilp;
  }

  // create a reserved with a name and a line number
  
  Reserved::Reserved (const String& name, const long lnum) {
    d_name   = name;
    d_quark  = name.toquark ();
    p_object = nilp;
    d_lnum   = lnum;
  }

  // copy construct this reserved keyword

  Reserved::Reserved (const Reserved& that) {
    d_name   = that.d_name;
    d_quark  = that.d_quark;
    p_object = nilp;
    d_lnum   = that.d_lnum;
  }

  // destroy this reserved keyword

  Reserved::~Reserved (void) {
    Object::dref (p_object);
  }

  // return the class name

  String Reserved::repr (void) const {
    return "Reserved";
  }

  // make this reserved name a shared object

  void Reserved::mksho (void) {
    if (p_shared != nilp) return;
    Object::mksho ();
    if (p_object != nilp) p_object->mksho ();
  }

  // return a clone of this reserved keyword

  Object* Reserved::clone (void) const {
    return new Reserved (*this);
  }

  // return the reserved serial code

  t_byte Reserved::serialid (void) const {
    return SERIAL_RESV_ID;
  }

  // serialize this reserved name

  void Reserved::wrstream (Output& os) const {
    rdlock ();
    Integer lnum (d_lnum);
    d_name.wrstream (os);
    lnum.wrstream   (os);
    unlock ();
  }

  // deserialize this reserved name

  void Reserved::rdstream (Input& is) {
    wrlock ();
    String  sval;
    Integer ival;
    sval.rdstream (is);
    ival.rdstream (is);
    d_name  = sval;
    d_lnum  = ival.tointeger ();
    d_quark = sval.toquark ();
    unlock ();
  }

  // evaluate this object in the current nameset

  Object* Reserved::eval (Runnable* robj, Nameset* nset) {
    // look for the cached object
    if (p_object != nilp) return p_object;
    // normal evaluation
    p_object = Object::iref (nset->eval (robj, nset, d_quark));
    return p_object;
  }
}
