/* apt-spy (c) Steven Holmes, 2003. This software is licensed as detailed in the LICENSE file. */

/* This file is a bit icky */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#include "include/global.h"
#include "include/parse.h" 
#include "include/file.h"

int build_area_file(FILE *config_p, FILE *infile_p, FILE *mirror_list, char *area)
{
	char *line;		/* Where we read the lines into */
	char *tmp;		/* Temp. pointer */
	char *inputline;	/* The line that will be written to config_p */
	char *country_code;	/* where we put the country code */

	/* EWWWWWWW */
	while((line = next_entry(config_p)) != NULL) {
		/* test for area string */
		if ((tmp = strstr(line, area)) != NULL) {
		
			if ((strchr(tmp,':')) == NULL) { 	/* And for trailing colon.. */
				free(line);
				continue;			/* .. it's not there */
			}
			break;
		}
		free(line);

		if (ferror(config_p) != 0)			/* Check for File error */
			return 1;
	}

	/* No match. Return. */
	if (line == NULL) {
		fprintf(stderr, "Could not find area named %s\n", area);
		return 1;
	}
	
	free(line);
		
	/* We now have the "label" line. The country list begins on the next line. */
	while ((line = next_entry(config_p)) != NULL) {

		if (ferror(config_p) != 0) {	/* Check for file error */
			free(line);
			return 1;
		}

		/* Skip blank lines */
		country_code = line;

		/* We check for either a non-space or a '\n'. Has the useful side-effect of */
		/* fast-forwarding country_code past any preceding whitespace.*/
		while ((*country_code != '\n') && (isspace(*country_code) != 0))
			++country_code;
		
		/* If country_code points to a '\n', there were no other characters. It was a blank line. */
		/* If it points to a '#', there is a comment. We skip it too. */
		if ((*country_code == '\n') || (*country_code == '#'))
			continue;

		if ((strchr(line, ':')) != NULL)
			return 0;			/* End of list. Return. */

		/* We do a little fiddling to get the country code down to 2 letters and a space */
		*(country_code + 2) = ' ';
		*(country_code + 3) = '\0';

		/* Sigh. The country code is in "country_code". We now parse the mirrors file for this */
		/* and set the file position so that the next read will return the first */
		/* mirror. */
		
		if (find_country(mirror_list, country_code) == 1) {
			fprintf(stderr, "Couldn't find country %s. Skipping.\n", country_code);
			free(line);
			continue;
		}

		/* The next read of infile_p will return the first mirror entry. We parse */
		/* this and build a line to put into the temporary file. */

		while ((inputline = get_mirrors(mirror_list)) != NULL) {

			/* We now write the line to the temporary file */	
			fputs(inputline, infile_p);
			free(inputline);

			if ((ferror(infile_p)) != 0) { 	/* Check for file error */
				free(line);
				return 1;
			}
		}

		free(line);
	}
	return 0;
}

int build_country_file(FILE *config_p, FILE *infile_p, FILE *mirror_list, char *country_list)
{
	char *country_code;
	char *p, *q; 
	char *inputline;

	int found = 0;

	/* A cheap way to make sure we have enough space */
	country_code = (char *) malloc(strlen(country_code));

	p = country_list;

	while (*p != '\0') {
		/* Reset country code pointer */
		q = country_code;
	
		/* Skip white space */
		while (isspace(*p))
			++p;
			
		/* Copy up until end or comma */
		while ((*p != '\0') && (*p != ',') && (isspace(*p) == 0))
			*q++ = *p++;
			
		/* Skip more white space. *sigh* */
		while (isspace(*p))
			++p;
	
		/* skip past comma */
		if (*p != '\0')
			++p;
		
		/* String-ify */
		*q++ = ' ';
		*q = '\0';

		/* And find the country/build the file */
		if (find_country(mirror_list, country_code) == 1) {
			fprintf(stderr, "Couldn't find country %s. Skipping.\n", country_code);
			continue;
		}

		found = 1;

		while ((inputline = get_mirrors(mirror_list)) != NULL) {
			fputs(inputline, infile_p);
			free(inputline);
			
			if (ferror(infile_p)) {
				free(country_code);
				return 1;
			}
		}
	}
	free(country_code);
	
	/* Check we have found at least one country */
	if (found == 0)
		return 1;
	else
		return 0;
}

int find_country(FILE *mirror_list, char *country_code)
{
	char *line, *cc;

	/* Make sure we're at beginning of file */
	rewind(mirror_list);

	/* This is a hack to allow users to specify "UK" instead of "GB" */
	if (!strcmp(country_code, "UK "))
		strncpy(country_code, "GB ", 4);

	/* Read until we find the country code */
	while ((line = next_entry(mirror_list)) != NULL) {

		cc = strstr(line, country_code);
		if (cc == NULL)
			continue;
			
		/* Skip white space */
		while (isspace(*line))
			++line;
		
		/* Country code should be first two characters on line. */
		if (cc == line)		
			break;
	}

	if (line == NULL)
		return 1;
	
	next_entry(mirror_list);		/* Skip a line */
	
	if (ferror(mirror_list)) {
		free(line);
		return 1;
	}
	return 0;		/* We're positioned nicely for the next read */
}

/* line format is "server:ftp-path:http-path". */

char *get_mirrors(FILE *mirror_list)
{
	char *line, *save_line;
	char *creation, *save_creation;
	int counter = 0;

	/* Allocate space for creation */
	save_creation = creation = (char *)malloc(100);

	if (creation == NULL) {
		perror("malloc");
		exit(1);
	}


	/* First, we read in a line from the file */
	save_line = line = next_entry(mirror_list);

	/* test for file error */
	if (ferror(mirror_list)) {
		perror("fopen");
		return NULL;
	}
	
	/* If the line begins with a space, we assume it is empty and the list is exhausted. */
	if (isspace(*line) != 0)
		return NULL;

	/* We now read the server name into "creation" */
	while (isspace(*line) == 0)
		*creation++ = *line++;

	/* And add a colon, which is the field seperator */
	*creation++ = ':';
	
	/* We skip over whitespace. If there is a lot of whitespace, we assume there is no FTP entry. */
	while ((isspace(*line) != 0) && (counter < 30)) {
		++line;
		++counter;
	}

	/* Check if there is an entry or just more space */	
	while (isspace(*line) == 0) 
		*creation++ = *line++;

	*creation++ = ':';

	/* We now check for an HTTP entry */
	while (*line != '\n') {
		if (isspace(*line) == 0)
			break;
		line++;
	}

	while (isspace(*line) == 0)
		*creation++ = *line++;

	*creation++ = ':';
	*creation++ = '\n';
	*creation++ = '\0';

	free(save_line);
	
	return save_creation;
}

void tokenise(server_t *current, char *cur_entry)
{
	char *temp;	/* We use this for temporary string-pointing :P */

	/* We initialise the structure to 0 */
	memset(current, 0, sizeof(*current));

	/* First, we copy the server name into the struct. */
	temp = current->hostname;
	while (*cur_entry != ':')
		*temp++ = *cur_entry++;

	*temp++ = '\0';		/* Turn into string */

	/* We now check for an ftp entry and copy it in */
	if (*(++cur_entry) != ':') {
		temp = current->path[FTP];		
		while (*cur_entry != ':')
			*temp++ = *cur_entry++;
			
		*temp++ = '\0';
	}

	/* And now check for HTTP entry */
	if (*(++cur_entry) != ':') {
		temp = current->path[HTTP];
		while (*cur_entry != ':')
			*temp++ = *cur_entry++;
			
		*temp++ = '\0';
	}

	/* We're done for now */
}

int write_list(FILE *outfile_p, server_t *best, char *dist)
{
	char url[100];

	/* Make our mark ;) */
	fprintf(outfile_p, "# sources.list generated by apt-spy %s\n", apt_spy_v);

	/* Copy URL information */
	if (best[0].stats.protocol == FTP)
		strncpy(url, best[0].url[FTP], 100);
	else
		strncpy(url, best[0].url[HTTP], 100);

	/* And write the line */
	fprintf(outfile_p, "deb %s %s main\n", url, dist);

	/* We also write a deb-src line */
	fprintf(outfile_p, "deb-src %s %s main\n", url, dist);

	/* And if we are using "stable", add a security line. Otherwise comment it out.*/
	if (strcmp(dist, "stable") == 0)
		fprintf(outfile_p, "deb http://security.debian.org/ stable/updates main\n");
	else
		fprintf(outfile_p, "#deb http://security.debian.org/ stable/updates main\n");

	if (ferror(outfile_p) != 0) {
		perror("fprintf");
		return 1;
	}
	return 0;
}

int write_top(FILE *infile_p, FILE *outfile_p, server_t *best)
{
	int i = 0;
	char *line;
	
	while (i < BESTNUMBER) {
	
		/* Make sure we're at the beginning */
		rewind(infile_p);
		
		/* Read in a line... */
		while ((line = next_entry(infile_p)) != NULL)
			if (strstr(line, best[i].hostname) != NULL)	/* Check for hostname */
				fputs(line, outfile_p);		/* if it's there, write to file */
		
		if ((ferror(infile_p) != 0) || (ferror(outfile_p) != 0))
			return 1;
		++i;
	}
	return 0;
}
