# 
# make-version.sh - create a new version within an archive
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "create a new version within an archive\\n"
		printf "usage: make-version [options] version\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -R --root root                specify the local archive root\\n"
		printf " -A --archive archive          specify the archive name\\n"
		printf "\\n"
		printf " -r --readme file              save FILE as the README\\n"
	        printf "                               for this category\\n"
		printf "\\n"
		printf "Create VERSION as a version in the indicated archive.\\n"
		printf "The branch for this version must not already exist (see\\n"
		printf "\"larch make-branch --help\").\\n"
		printf "\\n"
		printf "For information about how the archive to operate on is selected,\\n"
		printf "try \"larch my-default-archive --help\".\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=
readme=
lock=

while test $# -ne 0 ; do

  case "$1" in 

    --lock)		shift
    			lock=--lock
			;;

    -R|--root)		shift
    			if test $# -eq 0 ; then
			  printf "make-version: -R and --root require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archroot="$1"
			shift
			;;

    -A|--archive)	shift
    			if test $# -eq 0 ; then
			  printf "make-version: -A and --archive require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archive="$1"
			shift
			;;


    -r|--readme)	shift
			if test $# -eq 0 ; then
			  printf "make-version: -R and --readme require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			readme="$1"
			shift
			;;

    -*)			printf "make-version: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 1 ; then
    printf "usage: make-version [options] category\\n" 1>&2
    printf "try --help\\n" 1>&2
    exit 1
fi

version="$1"

################################################################
# Sanity Check 
# 

larch valid-package-name -e make-version -v -- "$version"

archive="`larch parse-package-name -R \"$archroot\" -A \"$archive\" --arch \"$version\"`"
category=`larch parse-package-name -b $version`
package=`larch parse-package-name $version`
version=`larch parse-package-name -v $version`

idstr="`larch my-id -e make-version`"

################################################################
# Get a Temp Dir
# 

tmpdir="`larch make-cache-temp-dir new-category-$$`"


################################################################
# Create the Version Directory (in the Temp Dir)
# 

tmpvsn="$tmpdir/$package--$version"
mkdir "$tmpvsn"


################################################################
# Make the =README file
# 

printf "Creator: %s\\nDate: %s\\n\\n" "$idstr" "`date`" > "$tmpvsn/=README" 


if test "x$readme" != x ; then

  if ! cat "$readme" >> "$tmpvsn/=README" ; then
     rm -r "$tmpdir"
  fi

fi

################################################################
# Create the Revision Lock
# 

cd "$tmpvsn"
larch make-lock revision-lock


################################################################
# Create the Next Version Lock
# 

cd "$tmpvsn"
larch make-lock version-lock

################################################################
# Try to install it in the archive
# 

cd "$tmpdir"
status=0

if ! larch with-archive -R "$archroot" -A "$archive" \
 	larch putlast -e make-version -R "$archroot" -A "$archive"  \
 		lock-branch "$package--$version" "$package--$version" "$category/$package/$package--$version" noop ; then
  printf "make-version: unable to create version\\n" 1>&2
  status=1
elif larch my-notifier > /dev/null 2>&1 ; then
  notifier="`larch my-notifier`"
  ( cd "$notifier" ; \
    larch without-archive as-daemon $ARCH_LOG_OPTION --null-output larch notify "$notifier" "$archive/$package--$version" )
fi



################################################################
# Remove the Cache Temp Directory
# 
cd ..
rm -rf "$tmpdir"

exit "$status"

