# 
# log-header-field.sh: filter a header field from a log entry
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#       --version | -V
#       --help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
                printf "filter a header field from a log entry\\n"
                printf "usage: log-header-field [options] field-name \\n"
                printf "\\n"
                printf " -V --version                  print version info\\n"
                printf " -h --help                     display help\\n"
                printf "\\n"
                printf " --literal                     don't remove newlines,\\n"
                printf " --list                        split field into items,\\n"
                printf "                                one per line\\n"
                printf "\\n"
                printf "Print the contents of the header field name FIELD-NAME in \\n"
                printf "the log entry on standard input.\\n"
                printf "\\n"
                exit 0
                ;;

      *)
                ;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

list=
literal=

while test $# -ne 0 ; do

  case "$1" in 

    --list)		shift
			list=--list
			;;

    --literal)		shift
			literal=--literal
			;;

    --)			shift
    			break
			;;
			
    -*)                 printf "rfc822-header-field: unrecognized option (%s)\\n" "$1" 1>&2
                        printf "try --help\\n" 1>&2
                        exit 1
                        ;;

    *)                  break
                        ;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 1 ; then
  printf "usage: log-header-field [options] field-name\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

field_name="$1"
shift

################################################################
# Do It
# 

awk -v field_name="$field_name" \
    -v list="$list" \
    -v literal="$literal" \
    'function io_error()
     {
        printf ("\\n") | "cat 1>&2";
	printf ("log-header-field: I/O error reading log file\\n") | "cat 1>&2";
	printf ("\\n") | "cat 1>&2";
	exit (1);
     }

     function print_value(value)
     {
       if (list == "")
         print value;
       else
         {
	   n_items = split (value, items, "[ \t]+");
	   for (x = 1; x <= n_items; ++x)
	     {
	       print items[x];
	     }
	 }
     }

     tolower($1) == tolower(field_name) ":" {
                                               value = $0;
					       if (literal != "")
					         {
						   spaces = field_name ":";
						   gsub(".", " ", spaces);
					           sub("^[^:]*:", spaces, value);
						 }
					       else
					         {
                                                   sub ("^[^:]*:[ \t]*", "", value);
						 }
                                               while (1)
                                               {
                                                 status = getline cont;
                                                 if (status < 0)
						   {
                                                     io_error();
						   }
                                                 else if ((status == 0) || (!match (cont, "^[ \t]")))
                                                   {
                                                     print_value(value);
                                                     exit(0);
                                                   }
						 else if (literal != "")
						   {
						     print_value(value); # print the previous line
						     value = cont;
						   }
                                                 else
                                                   {
                                                     sub("^[ \t]*", " ", cont);
                                                     value = value cont;
                                                   }
                                               }
                                            }'
 
# tag: Tom Lord Tue Dec 18 05:00:28 2001 (patch-logs/rfc822-header-field.sh)
#
