# 
# init-tree.sh - initialize a new project tree
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

###############################################################
# Format Version Identifier
# 
# These values are used to identify versions of the archive
# project tree format.  Please do not change them.
# 
fmtvsn=1
fmtstr="Hackerlab arch project directory, format version $fmtvsn."

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "initialize a new project tree\\n"
		printf "usage: init-tree [options] [[archive/]version]\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -R --root root                specify the local archive root\\n"
		printf " -A --archive archive          specify the archive name\\n"
		printf "\\n"
                printf " -d --dir DIR                  cd to DIR first\\n"
		printf "\\n"
		printf " --nested                      init a nested project tree\\n"
		printf "\\n"
		printf "Initialize DIR as a new project tree.\\n"
		printf "\\n"
		printf "An empty patch-log for VERSION will be created, and VERSION\\n"
		printf "will be made the default location for check-ins.\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=
nested=
dir=.

while test $# -ne 0 ; do

  case "$1" in 

    -d|--dir)		shift
			if test $# -eq 0 ; then
			  printf "init-tree: -d and --dir require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			dir="$1"
			shift
			;;

    --nested)		shift
    			nested=--nested
			;;


    -R|--root)		shift
    			if test $# -eq 0 ; then
			  printf "init-tree: -R and --root require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archroot="$1"
			shift
			;;

    -A|--archive)	shift
    			if test $# -eq 0 ; then
			  printf "init-tree: -A and --archive require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archive="$1"
			shift
			;;


    -*)			printf "init-tree: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -gt 1 ; then
  printf "usage: init-tree [options] [[archive/]version]\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

if test $# -gt 0 ; then
  arch_version="$1"
  shift
else
  arch_version=
fi

################################################################
# Sanity Check
# 

cd "$dir"
dir="`pwd`"

if test ! -z "$arch_version" ; then

  larch valid-package-name -e "init-tree" --vsn -- "$arch_version"

  archive=`larch parse-package-name -R "$archroot" -A "$archive" --arch "$arch_version"`

fi

cd "$dir"

if test -z "$nested" && larch tree-root -s > /dev/null ; then
  printf "init-tree: already a project tree\\n" 1>&2
  printf "  directory %s\\n" "$dir" 1>&2
  printf "  is already part of a project tree rooted at\\n" 1>&2
  printf "  %s\\n" "`larch tree-root`" 1>&2
  printf "\\n" 1>&2
  exit 1
fi


################################################################
# Make the {arch} Directory
# 

cd "$dir"
mkdir {arch}
cd {arch}
printf "%s\\n" "$fmtstr" > .arch-project-tree


if ! test -z "$arch_version" ; then
  cd "$dir"
  larch add-log -R "$archroot" -A "$archive" "$arch_version" 
  larch set-tree-version  -R "$archroot" -A "$archive" "$arch_version" 
fi


