/*
 * Copyright (c) 1995 - 2001 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <nnpfs/nnpfs_locl.h>

RCSID("$Id: nnpfs_vfsops-osf.c,v 1.28 2003/06/02 18:26:53 lha Exp $");

/*
 * NNPFS vfs operations.
 */

#include <nnpfs/nnpfs_common.h>
#include <nnpfs/nnpfs_message.h>
#include <nnpfs/nnpfs_fs.h>
#include <nnpfs/nnpfs_dev.h>
#include <nnpfs/nnpfs_deb.h>
#include <nnpfs/nnpfs_vnodeops.h>
#include <nnpfs/nnpfs_syscalls.h>

static int
nnpfs_mount(struct mount * mp,
	  char *user_path,
	  caddr_t user_data,
	  struct nameidata * ndp)
{
    return nnpfs_mount_common(mp, user_path, user_data, ndp, NULL);
}

static int
nnpfs_start(struct mount * mp, int flags)
{
    NNPFSDEB(XDEBVFOPS, ("nnpfs_start mp = %lx, flags = %d\n",
		       (unsigned long)mp, flags));
    return 0;
}

static int
nnpfs_unmount(struct mount * mp, int mntflags)
{
    NNPFSDEB(XDEBVFOPS, ("nnpfs_umount: mp = %lx, mntflags = %d\n",
		       (unsigned long)mp, mntflags));
    return nnpfs_unmount_common(mp, mntflags);
}

static int
nnpfs_root(struct mount * mp, struct vnode ** vpp)
{
    NNPFSDEB(XDEBVFOPS, ("nnpfs_root mp = %lx vnode = %lx\n", 
		       (unsigned long)mp,(unsigned long)vpp));

    return nnpfs_root_common (mp, vpp, u.u_procp, u.u_procp->p_rcred);
}

static int
nnpfs_quotactl(struct mount *mp, int cmd, uid_t uid, caddr_t arg)
{
    NNPFSDEB(XDEBVFOPS, ("nnpfs_quotactl: mp = %lx, cmd = %d, uid = %u, "
		       "arg = %lx\n", 
		       (unsigned long)mp, cmd, uid,
		       (unsigned long)arg));
    return (EOPNOTSUPP);
}

static int
nnpfs_statfs(struct mount *mp)
{
    NNPFSDEB(XDEBVFOPS, ("nnpfs_statfs: mp = %lx\n",
		       (unsigned long)mp));
    return 0;
}

static int
nnpfs_sync(struct mount *mp, int waitfor)
{
    NNPFSDEB(XDEBVFOPS, ("nnpfs_sync: mp = %lx, waitfor = %d\n",
		       (unsigned long)mp, waitfor));
    return 0;
}

static int
nnpfs_vget(struct mount * mp,
	 ino_t ino,
	 struct vnode ** vpp)
{
    NNPFSDEB(XDEBVFOPS, ("nnpfs_vget\n"));
    return EOPNOTSUPP;
}

static int
nnpfs_fhtovp(struct mount *mp,
	   struct fid *fhp,
	   struct vnode **vpp)
{
    NNPFSDEB(XDEBVFOPS, ("nnpfs_fhtovp\n"));
    return EOPNOTSUPP;
}

static int
nnpfs_vptofh(struct vnode * vp,
	   struct fid * fhp)
{
    NNPFSDEB(XDEBVFOPS, ("nnpfs_vptofh\n"));
    return EOPNOTSUPP;
}

#ifdef HAVE_STRUCT_VFSOPS_VFS_MOUNTROOT
static int
nnpfs_mountroot(struct mount * mp,
          struct vnode ** vp)
{
    NNPFSDEB(XDEBVFOPS, ("nnpfs_mountroot\n"));
    return EOPNOTSUPP;
}
#endif /* HAVE_STRUCT_VFSOPS_VFS_MOUNTROOT */

#ifdef HAVE_STRUCT_VFSOPS_VFS_SWAPVP
static int
nnpfs_swapvp(void)
{
    NNPFSDEB(XDEBVFOPS, ("nnpfs_swapvp\n"));
    return EOPNOTSUPP;
}
#endif /* HAVE_STRUCT_VFSOPS_VFS_SWAPVP */

#ifdef HAVE_STRUCT_VFSOPS_VFS_SMOOTHSYNC
static int
nnpfs_smoothsync(struct mount * mp,
          u_int int1, u_int int2)
{
    return EOPNOTSUPP;
}
#endif /* HAVE_STRUCT_VFSOPS_VFS_SMOOTHSYNC */

/*
 * dead nnpfs vnodes
 */

static int
nnpfs_dead_lookup(struct vnode *dvp, struct nameidata *ndp)
{
    ndp->ni_dvp = dvp;
    ndp->ni_vp  = NULL;
    return ENOTDIR;
}

static int
nnpfs_dead_create(struct nameidata *ndp, struct vattr *vap)
{
    return ENODEV;
}

static int
nnpfs_dead_mknod(struct nameidata *ndp, struct vattr *vap, struct ucred *cred)
{
    return ENODEV;
}

static int
nnpfs_dead_open(struct vnode **vpp, int mode, struct ucred *cred)
{
    return ENODEV;
}

static int
nnpfs_dead_close(struct vnode *vp, int mode, struct ucred *cred)
{
    return ENODEV;
}

static int
nnpfs_dead_access(struct vnode *vp, int mode, struct ucred *cred)
{
    return ENODEV;
}

static int
nnpfs_dead_getattr(struct vnode *vp, struct vattr *vap, struct ucred *cred)
{
    return ENODEV;
}

static int
nnpfs_dead_setattr(struct vnode *vp, struct vattr *vap, struct ucred *cred)
{
    return ENODEV;
}

static int
nnpfs_dead_read(struct vnode *vp, struct uio *uiop, int ioflag,
	      struct ucred *cred)
{
    return ENODEV;
}

static int
nnpfs_dead_write(struct vnode *vp, struct uio *uiop, int ioflag,
	      struct ucred *cred)
{
    return ENODEV;
}

static int
nnpfs_dead_ioctl(struct vnode *vp, int com, caddr_t data, int fflag,
	       struct ucred *cred)
{
    return ENODEV;
}

static int
nnpfs_dead_select(struct vnode *vp, int *events, int *revents, int scanning,
		struct ucred *cred)
{
    return 0;
}

static int
nnpfs_dead_mmap(struct vnode *vp, vm_offset_t off,
	      struct vm_map *map, vm_offset_t *addrp,
	      vm_size_t len, vm_prot_t prot, vm_prot_t maxprot,
	      int flags, struct ucred *cred)
{
    return ENODEV;
}

static int
nnpfs_dead_fsync(struct vnode *vp, int fflags, struct ucred *cred, int waitfor)
{
    return 0;
}

static int
nnpfs_dead_seek(struct vnode *vp, off_t o1, off_t o2, struct ucred *cred)
{
    return 0;
}

static int
nnpfs_dead_remove(struct nameidata *ndp)
{
    return ENODEV;
}

static int
nnpfs_dead_link(struct vnode *vp, struct nameidata *ndp)
{
    return ENODEV;
}

static int
nnpfs_dead_rename(struct nameidata *ndp1, struct nameidata *ndp2)
{
    return ENODEV;
}

static int
nnpfs_dead_mkdir(struct nameidata *ndp, struct vattr *vap)
{
    return ENODEV;
}

static int
nnpfs_dead_rmdir(struct nameidata *ndp)
{
    return ENODEV;
}

static int
nnpfs_dead_symlink(struct nameidata *ndp, struct vattr *vap, char *target)
{
    return ENODEV;
}

static int
nnpfs_dead_readdir(struct vnode *vp, struct uio *uiop, struct ucred *cred,
		 int *eofflagp)
{
    return ENODEV;
}

static int
nnpfs_dead_readlink(struct vnode *vp, struct uio *uiop, struct ucred *cred)
{
    return ENODEV;
}

static int
nnpfs_dead_abortop(struct nameidata *ndp)
{
    return ENODEV;
}

static int
nnpfs_dead_inactive(struct vnode *vp)
{
    return 0;
}

static int
nnpfs_dead_reclaim(struct vnode *vp)
{
    return 0;
}

static int
nnpfs_dead_bmap()
{
    return EIO;
}

static int
nnpfs_dead_strategy(struct buf *bp)
{
    return EIO;
}

static int
nnpfs_dead_print(struct vnode *vp)
{
    return 0;
}

static int
nnpfs_dead_pgrd(struct vnode *vp, struct uio *uiop, struct ucred *cred)
{
    return ENODEV;
}

static int
nnpfs_dead_pgwr()
{
    return ENODEV;
}

static int
nnpfs_dead_swap(struct vnode *vp, vp_swap_op_t swap, vm_offset_t off,
	      struct ucred *cred)
{
    return ENODEV;
}

static int
nnpfs_dead_bread(struct vnode *vp, off_t off, struct buf **buf,
	       struct ucred *cred)
{
    return ENODEV;
}

static int
nnpfs_dead_brelse(struct vnode *vp, struct buf *buf)
{
    return ENODEV;
}

static int
nnpfs_dead_lockctl(struct vnode *vp, struct eflock *ef, int i,
		 struct ucred *cred, pid_t pid, off_t off)
{
    return ENODEV;
}

static int
nnpfs_dead_syncdata(struct vnode *vp, int i, vm_offset_t off, vm_size_t size,
		  struct ucred *cred)
{
    return 0;
}

static int
nnpfs_dead_lock(struct vnode *vp, int flag, struct ucred *cred)
{
    return ENODEV;
}

static int
nnpfs_dead_unlock(struct vnode *vp, int flag, struct ucred *cred)
{
    return ENODEV;
}

static int
nnpfs_dead_getproplist(struct vnode *vp, char **names, struct uio *uiop,
		     int *min_buf_size, int mask, struct ucred *cred)
{
    return ENODEV;
}

static int
nnpfs_dead_setproplist(struct vnode *vp, struct uio *uiop, struct ucred *cred)
{
    return ENODEV;
}

static int
nnpfs_dead_delproplist(struct vnode *vp, char **names, int mask,
		     struct ucred *cred, long xid)
{
    return ENODEV;
}

static int
nnpfs_dead_pathconf(struct vnode *vp, int name, long *retval)
{
    return ENODEV;
}

static struct vnodeops nnpfs_dead_vnodeops = {
	nnpfs_dead_lookup,
	nnpfs_dead_create,
	nnpfs_dead_mknod,
	nnpfs_dead_open,
	nnpfs_dead_close,
	nnpfs_dead_access,
	nnpfs_dead_getattr,
	nnpfs_dead_setattr,
	nnpfs_dead_read,
	nnpfs_dead_write,
	nnpfs_dead_ioctl,
	nnpfs_dead_select,
	nnpfs_dead_mmap,
	nnpfs_dead_fsync,
	nnpfs_dead_seek,
	nnpfs_dead_remove,
	nnpfs_dead_link,
	nnpfs_dead_rename,
	nnpfs_dead_mkdir,
	nnpfs_dead_rmdir,
	nnpfs_dead_symlink,
	nnpfs_dead_readdir,
	nnpfs_dead_readlink,
	nnpfs_dead_abortop,
	nnpfs_dead_inactive,
	nnpfs_dead_reclaim,
	nnpfs_dead_bmap,
	nnpfs_dead_strategy,
	nnpfs_dead_print,
	nnpfs_dead_pgrd,
	nnpfs_dead_pgwr,
	nnpfs_dead_swap,
	nnpfs_dead_bread,
	nnpfs_dead_brelse,
	nnpfs_dead_lockctl,
	nnpfs_dead_syncdata,
	nnpfs_dead_lock,
	nnpfs_dead_unlock,
	nnpfs_dead_getproplist,
	nnpfs_dead_setproplist,
	nnpfs_dead_delproplist,
	nnpfs_dead_pathconf
};

int
nnpfs_make_dead_vnode(struct mount *mp, struct vnode **vpp)
{
    int error;

    NNPFSDEB(XDEBNODE, ("make_dead_vnode mp = %lx\n",
		      (unsigned long)mp));

    error = getnewvnode(VT_NON, &nnpfs_dead_vnodeops, vpp);
    if (error) {
	return error;
    }
    insmntque(*vpp, mp, NULL);

    return 0;
}

static int
nnpfs_init(void)
{
    NNPFSDEB(XDEBVFOPS, ("nnpfs_init\n"));
    return 0;
}

struct vfsops nnpfs_vfsops = {
        nnpfs_mount,
        nnpfs_start,
        nnpfs_unmount,
        nnpfs_root,
        nnpfs_quotactl,
        nnpfs_statfs,
        nnpfs_sync,
        nnpfs_fhtovp,
        nnpfs_vptofh,
        nnpfs_init,
#ifdef HAVE_STRUCT_VFSOPS_VFS_MOUNTROOT
	nnpfs_mountroot,
#endif
#ifdef HAVE_STRUCT_VFSOPS_VFS_SWAPVP
	nnpfs_swapvp,
#endif
#ifdef HAVE_STRUCT_VFSOPS_VFS_SMOOTHSYNC
	nnpfs_smoothsync,
#endif
};


extern struct vfsops nnpfs_vfsops;

int
nnpfs_install_fs(void)
{
    if (vn_maxprivate < sizeof(struct nnpfs_node))
	return ENOMEM;
    return vfssw_add_fsname(MOUNT_NNPFS, &nnpfs_vfsops, "nnpfs");
}

int
nnpfs_uninstall_fs(void)
{
    return vfssw_del(MOUNT_NNPFS);
}

int
nnpfs_fhlookup (struct proc *proc,
	      struct nnpfs_fhandle_t *fhp,
	      struct vnode **vpp)
{
    int error;
    struct ucred *cred = proc->p_rcred; 
    struct mount *mp;
    fid_t fid;
    fhandle_t *fh = (fhandle_t *)fhp;

    NNPFSDEB(XDEBVFOPS, ("nnpfs_fhlookup\n"));

    error = nnpfs_suser (proc);
    if (error)
	return EPERM;

    if (fhp->len < sizeof (*fh))
	return EINVAL;

    mp = getvfs(&fh->fh_fsid);
    if(mp == NULL) {
	NNPFSDEB(XDEBVFOPS, ("getvfs(%d) = NULL\n", fh->fh_fsid));
	return ESTALE;
    }

    VFS_FHTOVP(mp, &fh->fh_fid, vpp, error);
    NNPFSDEB(XDEBVFOPS, ("VFS_FHTOVP() = %d\n", error));
    /* getvfs returned the mount structure locked */
    UNMOUNT_READ_UNLOCK(mp);
    if (error)
	return (ESTALE);
    VREF(*vpp);
    return (0);

#if 0 
	bzero(&fh);
	fh.fh_fsid = fsid;
	fh.fh_fid.fid_len = sizeof(fh.fh_fid);
	bcopy(&fileid, fh.fh_fid.fid_data, sizeof(fileid));
    
    error = vn_fhtovp(&fh, 0, vpp);
    NNPFSDEB(NNPFSDEBVFSOPS, ("vn_fhtovp() = %d\n", error));
    return error;
#endif
}

int
nnpfs_fhopen (struct proc *proc,
	    struct nnpfs_fhandle_t *fhp,
	    int user_flags,
	    register_t *retval)
{
    int error;
    struct vnode *vp;
    struct ucred *cred = proc->p_rcred;
    int flags = FFLAGS(user_flags);
    int index;
    struct file *fp;
    extern struct fileops vnops;
    struct ufile_state *ufp;
#ifndef U_FD_SET
    struct ufile_entry *fe;
#endif

    NNPFSDEB(XDEBVFOPS, ("nnpfs_fhopen: flags = %x\n", user_flags));

    error = nnpfs_fhlookup (proc, fhp, &vp);
    if (error)
	return error;

    VOP_OPEN(&vp, flags, cred, error);
    if (error)
	goto out;

    error = vn_writechk(vp, 1);
    if(error)
	goto out;

#ifdef U_FD_SET
    error = falloc(&fp, &index);
#else
    error = falloc(&fp, &index, &fe);
#endif
    if (error)
	goto out;
    ufp = &u.u_file_state;
    FP_UNREF_MT(fp,ufp);
#ifdef U_FD_SET
    U_FD_SET(index, fp, ufp);
#else
    u_set_fe(index, fe, fp, ufp);
#endif

    /* ??? */
    if (flags & FREAD) {
	VN_LOCK(vp);
        vp->v_rdcnt++;
	VN_UNLOCK(vp);
    }

    fp->f_flag = flags & FMASK;
    fp->f_type = DTYPE_VNODE;
    fp->f_ops  = &vnops;
    fp->f_data = (caddr_t)vp;
    nnpfs_vfs_unlock(vp, proc);

    NNPFSDEB(XDEBVFOPS, ("nnpfs_fhopen: fileno = %d\n", index));
    *retval = index;
    return 0;
out:
    NNPFSDEB(XDEBVFOPS, ("nnpfs_fhopen: errno = %d\n", error));
    vput(vp);
    return error;
}
