/*
 * Copyright (c) 1995 - 2000 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * NNPFS operations.
 */

#include <nnpfs/nnpfs_locl.h>
#include <vm/vm_mmap.h>
#include <nnpfs/nnpfs_message.h>
#include <nnpfs/nnpfs_common.h>
#include <nnpfs/nnpfs_fs.h>
#include <nnpfs/nnpfs_dev.h>
#include <nnpfs/nnpfs_deb.h>
#include <nnpfs/nnpfs_syscalls.h>
#include <nnpfs/nnpfs_vnodeops.h>

#include <sys/buf.h>

RCSID("$Id: nnpfs_vnodeops-osf.c,v 1.33 2002/09/07 10:45:38 lha Exp $");

/*
 * vnode functions
 */

static int
nnpfs_open(struct vnode **vpp, int mode, struct ucred *cred)
{
    return nnpfs_open_common (*vpp, mode, cred, nnpfs_curproc());
}

static int
nnpfs_fsync(struct vnode *vp, int fflags, struct ucred *cred, int waitfor)
{
    int error=0;
    if (vp->v_type == VREG) {
	error = ubc_flush_dirty(vp->v_object, 
				waitfor & MNT_WAIT ? B_DONE : B_ASYNC);
	if(error)
	    return error;
    }

    error = nnpfs_fsync_common(vp, cred, waitfor, nnpfs_curproc());

    return error;
}

static int
nnpfs_close(struct vnode *vp, int fflags, struct ucred *cred)
{
    return nnpfs_close_common(vp, fflags, nnpfs_curproc(), cred);
}

static int
nnpfs_read(struct vnode *vp, struct uio *uio, int ioflag, struct ucred *cred)
{
    return nnpfs_read_common(vp, uio, ioflag, cred);
}

static int
nnpfs_write(struct vnode *vp, struct uio *uiop, int ioflag, struct ucred *cred)
{
    return nnpfs_write_common(vp, uiop, ioflag, cred);
}

static int
nnpfs_ioctl(struct vnode *vp, 
	  int com, 
	  caddr_t data, 
	  int fflag, 
	  struct ucred *cred)
{
    NNPFSDEB(XDEBVNOPS, ("nnpfs_ioctl\n"));
    return EOPNOTSUPP;
}

static int
nnpfs_getattr(struct vnode *vp, struct vattr *vap, struct ucred *cred)
{
    return nnpfs_getattr_common(vp, vap, cred, nnpfs_curproc ());
}

static int
nnpfs_setattr(struct vnode *vp, struct vattr *vap, struct ucred *cred)
{
    return nnpfs_setattr_common (vp, vap, cred, nnpfs_curproc ());
}

static int
nnpfs_access(struct vnode *vp, int mode, struct ucred *cred)
{
    return nnpfs_access_common(vp, mode, cred, nnpfs_curproc());
}

static int
nnpfs_lookup(struct vnode *dvp, struct nameidata *ndp)
{
    ndp->ni_dvp = dvp;
    return nnpfs_lookup_common(dvp, ndp, &ndp->ni_vp);
}

static int
nnpfs_create(struct nameidata *ndp, struct vattr *vap)
{
    struct vnode *dvp  = ndp->ni_dvp;
    const char *name   = ndp->ni_dent.d_name;
    struct ucred *cred = ndp->ni_cred;
    int error;

    NNPFSDEB(XDEBVNOPS, ("nnpfs_create: vap->va_mode = %o\n",
		       vap->va_mode));

    error = nnpfs_create_common(dvp,
			      name,
			      vap,
			      cred,
			      nnpfs_curproc());
    if(error == 0) {
	ndp->cn_nameptr = ndp->ni_dent.d_name;
	error = nnpfs_lookup_common (dvp, ndp, &ndp->ni_vp);
    }
    vrele(dvp);
    NNPFSDEB(XDEBVNOPS, ("nnpfs_create: error = %d\n", error));
    
    return error;
}

static int
nnpfs_remove(struct nameidata *ndp)
{
    int ret = nnpfs_remove_common(ndp->ni_dvp, 
			     ndp->ni_vp, 
			     ndp->ni_dent.d_name, 
			     ndp->ni_cred,
			     nnpfs_curproc());
    vrele(ndp->ni_dvp);
    vrele(ndp->ni_vp);
    return ret;
}

static int
nnpfs_rename(struct nameidata *from, struct nameidata *to)
{
    int error =  nnpfs_rename_common(from->ni_dvp, 
				   from->ni_vp, 
				   from->ni_dent.d_name, 
				   to->ni_dvp, 
				   to->ni_vp, 
				   to->ni_dent.d_name, 
				   from->ni_cred,
				   nnpfs_curproc());
    
    vrele(to->ni_dvp);
    if (to->ni_vp)
	vrele(to->ni_vp);
    vrele(from->ni_dvp);
    vrele(from->ni_vp);
    
    return error;
}

static int
nnpfs_mkdir(struct nameidata *ndp,
	  struct vattr *vap)
{
    struct vnode *dvp  = ndp->ni_dvp;
    const char *name   = ndp->ni_dent.d_name;
    struct ucred *cred = ndp->ni_cred;
    int error;

    error = nnpfs_mkdir_common(dvp,
			     name,
			     vap,
			     cred,
			     nnpfs_curproc());

    if(error == 0) {
	ndp->cn_nameptr = ndp->ni_dent.d_name;
	error = nnpfs_lookup_common (dvp, ndp, &ndp->ni_vp);
    }
    vrele(dvp);
    NNPFSDEB(XDEBVNOPS, ("nnpfs_mkdir: error = %d\n", error));
    
    return error;
}

static int
nnpfs_rmdir(struct nameidata *ndp)
{
    int ret = nnpfs_rmdir_common(ndp->ni_dvp, 
			    ndp->ni_vp, 
			    ndp->ni_dent.d_name, 
			    ndp->ni_cred,
			    nnpfs_curproc());
    vrele(ndp->ni_dvp);
    vrele(ndp->ni_vp);
    return ret;
}

static int
nnpfs_readdir(struct vnode *vp, 
	    struct uio *uio, 
	    struct ucred *cred, 
	    int *eofflagp)
{
    return nnpfs_readdir_common(vp, uio, cred, nnpfs_curproc(), eofflagp);
}

static int
nnpfs_link(struct vnode* vp, struct nameidata *ndp)
{
    int ret = nnpfs_link_common(ndp->ni_dvp, vp, ndp->ni_dent.d_name,
			   ndp->ni_cred, nnpfs_curproc());
    vrele(ndp->ni_dvp);
    return ret;
}

static int
nnpfs_symlink(struct nameidata *ndp, struct vattr *vap, char *target)
{
    int ret;

    ndp->cn_nameptr = ndp->ni_dent.d_name;
    ret = nnpfs_symlink_common(ndp->ni_dvp, &ndp->ni_vp,
			      ndp, vap, target);
    vrele(ndp->ni_dvp);
    return ret;
}

static int
nnpfs_readlink(struct vnode *vp, struct uio *uiop, struct ucred *cred)
{
    return nnpfs_readlink_common (vp, uiop, cred);
}

static int
nnpfs_inactive(struct vnode *vp)
{
    return nnpfs_inactive_common(vp, nnpfs_curproc());
}

static int
nnpfs_reclaim(struct vnode *vp)
{
    return nnpfs_reclaim_common (vp);
}

static int
nnpfs_lock(struct vnode *vp, int flag, struct ucred *cred)
{
    NNPFSDEB(XDEBVNOPS, ("nnpfs_lock(%lx, %d, %lx)\n",
		       (unsigned long)vp, flag,
		       (unsigned long)cred));
    return EOPNOTSUPP;
}

static int
nnpfs_unlock(struct vnode *vp, int flag, struct ucred *cred)
{
    NNPFSDEB(XDEBVNOPS, ("nnpfs_unlock(%lx, %d, %lx)\n",
		       (unsigned long)vp, flag,
		       (unsigned long)cred));
    return EOPNOTSUPP;
}

static int
nnpfs_abortop(struct nameidata *ndp)
{
    NNPFSDEB(XDEBVNOPS, ("nnpfs_abortop(%s)\n", ndp->ni_dirp));
    return 0;
}

static int
nnpfs_mmap(struct vnode *vp,
	 vm_offset_t offset,
	 vm_map_t map,
	 vm_offset_t *addrp,
	 vm_size_t len,
	 vm_prot_t prot,
	 vm_prot_t maxprot,
	 int flags,
	 struct ucred *cred)
{
    struct vp_mmap_args args;
    extern kern_return_t u_vp_create(vm_map_t, 
				     struct vm_ubc_object *, 
				     vm_offset_t);
    
    NNPFSDEB(XDEBVNOPS, ("nnpfs_mmap (%lx, %lx, %lx, %lx, %lx, %x, %x, %d, %lx)\n",
		       (unsigned long)vp, offset, map, addrp,
		       len, prot, maxprot, flags,
		       (unsigned long)cred));
    
    args.a_offset = offset;
    args.a_vaddr = addrp;
    args.a_size = len;
    args.a_prot = prot;
    args.a_maxprot = maxprot;
    args.a_flags = flags;
    return u_vp_create(map, vp->v_object, (vm_offset_t) &args);
}

static int
nnpfs_bmap(struct vnode *vp, 
	 daddr_t bn, 
	 struct vnode **vpp, 
	 daddr_t *bnp)
{
    NNPFSDEB(XDEBVNOPS, ("nnpfs_bmap\n"));
    return EOPNOTSUPP;
}

/*
 *      common routine for unsupported vnode ops 
 */
int
nnpfs_unsupp()
{
    return ENOSYS;
}

int
nnpfs_getproplist (struct vnode *vp, char **names, struct uio *uio, 
		 int *min_buf_size, int mask, struct ucred *cred)
{
    NNPFSDEB(XDEBVNOPS, ("nnpfs_getproplist(%lx, %lx, %lx, %lx, %d, %lx)\n",
		       (unsigned long)vp,
		       (unsigned long)names,
		       (unsigned long)uio, min_buf_size, mask,
		       (unsigned long)cred));
    return EOPNOTSUPP;
}

int
nnpfs_setproplist (struct vnode *vp, struct uio *uio, struct ucred *cred)
{
    NNPFSDEB(XDEBVNOPS, ("nnpfs_setproplist(%lx, %lx, %lx)\n",
		       (unsigned long)vp,
		       (unsigned long)uio,
		       (unsigned long)cred));
    return EOPNOTSUPP;
}

int
nnpfs_delproplist (struct vnode *vp, char **names, int mask, struct ucred *cred)
{
    NNPFSDEB(XDEBVNOPS, ("nnpfs_delproplist(%lx, %lx, %d, %lx)\n",
		       (unsigned long)vp,
		       (unsigned long)names, mask,
		       (unsigned long)cred));
    return EOPNOTSUPP;
}


int
nnpfs_ok()
{
    return 0;
}

struct vnodeops nnpfs_vnodeops = {
    nnpfs_lookup,		/* lookup */
    nnpfs_create,		/* create */
    nnpfs_unsupp,		/* mknod */
    nnpfs_open,		/* open */
    nnpfs_close,		/* close */
    nnpfs_access,		/* access */
    nnpfs_getattr,	/* getattr */
    nnpfs_setattr,	/* setattr */
    nnpfs_read,		/* read */
    nnpfs_write,		/* write */
    nnpfs_ioctl,		/* ioctl */
    nnpfs_unsupp,		/* select */
    nnpfs_mmap,		/* mmap */
    nnpfs_fsync,		/* fsync */
    nnpfs_ok,		/* seek */
    nnpfs_remove,		/* remove */
    nnpfs_link,		/* link */
    nnpfs_rename,		/* rename */
    nnpfs_mkdir,		/* mkdir */
    nnpfs_rmdir,		/* rmdir */
    nnpfs_symlink,	/* symlink */
    nnpfs_readdir,	/* readdir */
    nnpfs_readlink,	/* readlink */
    nnpfs_ok,		/* abortop */
    nnpfs_inactive,	/* inactive */
    nnpfs_reclaim,	/* reclaim */
    nnpfs_unsupp,		/* bmap */
    nnpfs_unsupp,		/* strategy */
    nnpfs_unsupp,		/* print */
    nnpfs_unsupp,		/* page_read */
    nnpfs_unsupp,		/* page_write */
    nnpfs_unsupp,		/* swap handler */
    nnpfs_unsupp,		/* buffer read */
    nnpfs_unsupp,		/* buffer release */
    nnpfs_unsupp,		/* file locking */
    nnpfs_unsupp,		/* fsync byte range */
    nnpfs_lock,		/* Lock a node */
    nnpfs_unlock,		/* Unlock a node */
    nnpfs_getproplist,	/* Get extended attributes */
    nnpfs_setproplist,	/* Set extended attributes */
    nnpfs_delproplist,	/* Delete extended attributes */
    vn_pathconf,		/* pathconf */		
};

int
nnpfs_refer(vm_ubc_object_t vop)
{
    NNPFSDEB(XDEBVNOPS, ("nnpfs_refer: vop = %lx, vp = %lx\n",
		       (unsigned long)vop,
		       (unsigned long)vop->vu_vp));
    VREF(vop->vu_vp);
    return 0;
}

int
nnpfs_release(vm_ubc_object_t vop)
{
    NNPFSDEB(XDEBVNOPS, ("nnpfs_release: vop = %lx, vp = %lx\n",
		       (unsigned long)vop,
		       (unsigned long)vop->vu_vp));
    vrele(vop->vu_vp);
    return 0;
}

#ifdef HAVE_SIX_ARGUMENT_UBC_LOOKUP
int
nnpfs_getpage(vm_ubc_object_t vop,
	    vm_offset_t offset,
	    vm_size_t len,
	    vm_prot_t *protp,
	    vm_page_t *pl,
	    int plsz,
	    vm_map_entry_t ep,
	    vm_offset_t addr,
	    int rwflg,
	    struct ucred *cred)
#else
/* OSF5 */
int
nnpfs_getpage(vm_ubc_object_t vop,
	    vm_offset_t offset,
	    vm_size_t len,
	    vm_prot_t *protp,
	    vm_page_t *pl,
	    int plsz,
	    struct vm_policy *vmp,
	    int rwflg,
	    struct ucred *cred)
#endif
{
    int error;
    struct vnode *vp = vop->vu_vp;
    struct vnode *t = DATA_FROM_VNODE(vp);

    NNPFSDEB(XDEBVNOPS, ("nnpfs_getpage: enter\n"));

    error = nnpfs_data_valid(vp, cred, nnpfs_curproc(), NNPFS_DATA_R,offset+len);

    NNPFSDEB(XDEBVNOPS, ("nnpfs_getpage = %d\n", error));

    if (error == 0) {
#ifdef HAVE_SIX_ARGUMENT_UBC_LOOKUP
	FSOP_GETPAGE(t->v_object,offset,len,protp,pl,plsz,ep,
		     addr,rwflg,cred,error);
#else
	FSOP_GETPAGE(t->v_object,offset,len,protp,pl,plsz,
		     vmp,rwflg,cred,error);
#endif
    }

    NNPFSDEB(XDEBVNOPS, ("nnpfs_getpage = %d\n", error));
    return error;
}

int
nnpfs_putpage(vm_ubc_object_t vop,
	    vm_page_t *pl,
	    int pcnt,
	    int flags,
	    struct ucred *cred)
{
    int error;
    struct vnode *vp = vop->vu_vp;
    struct vnode *t = DATA_FROM_VNODE(vp);

    NNPFSDEB(XDEBVNOPS, ("nnpfs_putpage: enter\n"));

    error = nnpfs_data_valid(vp, cred, nnpfs_curproc(), NNPFS_DATA_R,
			   VNODE_TO_XNODE(vp)->attr.va_size);

    NNPFSDEB(XDEBVNOPS, ("nnpfs_putpage = %d\n", error));

    if (error == 0)
	FSOP_PUTPAGE(t->v_object,pl,pcnt,flags,cred,error);

    NNPFSDEB(XDEBVNOPS, ("nnpfs_putpage = %d\n", error));
    return error;
}

int
nnpfs_write_check(vm_ubc_object_t vpp, vm_page_t pp)
{
    NNPFSDEB(XDEBVNOPS, ("nnpfs_write_check: vpp = %lx, vp = %lx, pp = %lx\n", 
		       (unsigned long)vpp,
		       (unsigned long)vpp->vu_vp,
		       (unsigned long)pp));
    return TRUE;
}

struct vfs_ubcops nnpfs_ubcops = {
    nnpfs_refer,              /* refer vnode */
    nnpfs_release,            /* release vnode */
    nnpfs_getpage,            /* get page */
    nnpfs_putpage,            /* put page */
    nnpfs_write_check         /* check writability */
};
